/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
// ******************************************************************
// InitAFEPowerUp.c
//
// History
//
// 23/03.2011 Vinjam: ARX208 support in VR9 ADSL Firmware
//            Grep for XDSLRTFW-278: Feature_AB_ALL_ALL_ARX208_Support_DSL_CMV
//
// 24/03/2011 Vinjam: Add "enable FIFOs" in FIFO init function and
//            renamed "VR9_A2x_FIFO_FIX( )" to "VR9_Init_AFE_FIFOs( )"
//            Grep for XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO
//
// 26/11/2012 Kannan: VR9 & VRX318 AFE Powerup init status
//            Grep for XDSLRTFW-541: Platform_VR9_VRX318
//
// 15/01/2013 Kannan: 3GLP exception code variables are initialized with
//            3GLP exception code, if there is any error in 3GLP initialization.
//            Grep for XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication
// ******************************************************************
/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/
#include "common.h"
#include "typedef.h"
#include "gdata.h"       //XDSLRTFW-541: Platform_VR9_VRX318 (START_END)
#include "AfeRdWrite.h"
#include "VR9_Afedata.h"
#include "vr9_afe_analog_reg.h"
#include "vr9_afe_dig_global_reg.h"
#include "vr9_afe_dig_ch0_reg.h"
#include "vr9_afe_dig_filt_ch0_reg.h"
#include "vr9_afe_glp_slave_reg.h"
#include "LL_IOf.h"     //ReadCoreReg(), WriteCoreReg() defined here
#include "dsp_regs_62.h"
#include "glptop_reg.h" //For GLP_TOP_STATUS
#include "glpm_reg.h"   //For GLPM_PLL_STATUS
#include "VR9_AfeFunc.h"
#include "VR9_Afe_const.h"
#include "VRX_AfeCommonData.h"
/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/
//#define GLP_TOP_STATUS                  (0x20CE04)  //TBD, use correct include file
//#define GLPM_PLL_STATUS                 (0x20CCB0)  //TBD, use correct include file

//  288Mhz DSP clock;1MS delay; the while calls VR9_AfeRead(); which consumes
// ~150 cycles.
//#define GLP_PORT_STAT_TIMEOUT           (1 << 11)    //2048 TBD
#define GLP_PORT_STAT_TIMEOUT             (1 << 16)    //65536 TBD

#define GLP_MACRO_STARTUP_NOT_READY       (0x80)    //TBD
#define GLP_FRAMER_STATE_MACHINE_ERROR    (0x81)    //TBD
#define GLPM_PLL_STATUS_NOT_READY         (0x82)    //TBD

#define GLPM_PLL_STATUS_READY           (0x00000010)
#define GLP_MACRO_0_RDY                 (0x00000001)
#define GLP_MACRO_1_RDY                 (0x00000100)

#define GLPM_FR_STATUS_0_STATE_AND_RMT_FM_AL_MASK  \
   (GLPM_GLP_FR_STATUS_0_STATE_MASK | GLPM_GLP_FR_STATUS_0_RMT_FM_AL_MASK)

#define STATE_AND_RMT_FM_AL_READY      (GLPM_FR_STATUS_0_STATE_AND_RMT_FM_AL_MASK)

#define  PORT_1_AFE_1   (0x00000001)  //DSP_PORT_SEL <Bit0> LDST_PORT_SEL>
#define  PORT_0            (0)
#define  PORT_1            (1)

#define  READY             (1)
#define  ERROR             (0)


extern PortModeControl_t gt_PortModeControl;
extern void GlpEn_0();
/*
** =============================================================================
**                           LOCAL TYPES
** =============================================================================
*/


/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/
/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/


/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_InitAFEPowerUp ()
**
** DESCRIPTION:    Initialize the AFE registers during power up condition.
**                 This function is common for single mode as well as bonded mode.
**
** PARAMETERS:     Port Number, Power up status, Bonded mode, AEC status (ON/OFF)
**
** RETURN VALUE:   If GLP is ready, return READY, else ERROR
**
**
** NOTES:          Reference: (1) VR9_3GLP_IF.doc;
**                            (2) VR9 Arete doc
**                            (3) VR9 Afe Spec
**                            SMS00946055
** =============================================================================
*/


uint16 VR9_AFEWarmInit(void);
uint16 VR9_PowerUpAFE_Port0(FlagT ft_AEC_On);
uint16 VR9_PowerUpAFE_Port1(FlagT ft_AEC_On);
uint16 VR9_AFE_3GLP_Statuscheck();
void  VR9_WaitFor_3GLP_Ready();
extern void Init_3GLP_clk_data_int();
extern void GlpFifoCfg_0();
extern void ErrCorrBypass_0();

//
// Changes were made to previous version of VR9_InitAFEPowerUp() mainly due
// to the fact that DSP_PORT_SEL can't be changed within a function
// that uses stack. At the instance modem DSP_PORT_SEL was changed, the stack space
// will be switched/mapped to the physical local ldst space of the other port
// and the function will crashed.
//
// To accomplish the same powerup sequence, the original function has been
// splitted up into three other functions:
//
//  uint16 VR9_AFEWarmInit(void)        can be run under port0 or port1 context
//                                      there's no synchronizations required between
//                                      the two ports.
//
//  uint16 VR9_PowerUpAFE_Port0(FlagT ft_AEC_On)        only run under port0 context
//
//  uint16 VR9_PowerUpAFE_Port1(FlagT ft_AEC_On)        only run under port1 context
//
// the last two functions will synchronize their processing by calling the helper
// function:
//
//  void SetAndWaitForNextAfeEvent(uint8 uc_CurState, uint8 uc_NextState)
//
// This help function will allow the two port to coordinate their initialization
// accordingly.
//
// In addition, the gft_PowerUpLink meaning has changed from just indicating COLD
// or WARM start; it now indicates a COLD start, or 1PORT_STARETD or 2PORTS_STARTED.
// This will allow the FW to force a COLD start if only port0 AFE was initialized in
// previous run, and the system is set to run in dual-port on a warm restart.
//

extern FlagT gft_ByPassAfe;

uint16 VR9_InitAFEPowerUp(void)
{
    uint16 us_status;

    // this check is for VR9 bring-up only
    if (gft_ByPassAfe)
    {
        return(0);
    }

#if 1
    // for now, force a COLD_START always.
    //XDSLRTFW-541: Platform_VR9_VRX318 (START_END)
    gt_PortModeControl.uc_AfePowerUp = (gt_PortModeControl.uc_AfePowerUp & (~AFE_POWERUP_MASK)) | AFE_COLD_START;
#endif



#ifdef ADSL_62
    //Possible MIN PGA would be -12dB, with this we get 6dB less DS PCB in shorter loops as compared
    //to Danube, if less DSPCB on shorter loops, CO sends the PAR tones. Hence min PGA was set to -6dB
    //to match Danube DSPCB.
    gs_min_pga_setting = -12 << 8; // ==> AGC1(-6) + AGC2 Fixed(-12)+AGC2 var(6), -12dB
    gs_max_pga_setting = 42 << 8; // => AGC1(27) + AGC2 Fixed(-12)+AGC2 var(27) + AGC3(0)
#endif
    //XDSLRTFW-541: Platform_VR9_VRX318 (START_END)
    if ((gt_PortModeControl.uc_AfePowerUp & AFE_POWERUP_MASK) == AFE_COLD_START)
    {
        // for now AEC is off be default
        // later on, the power-up process might have to change if AEC was turned from 0 to 1, etc
        gft_AEC_On = 0;
    }

    gft_BondedMode = (gt_PortModeControl.uc_DualPortModeCurrent == DUAL_PORT_MODE);

    //XDSLRTFW-541: Platform_VR9_VRX318 (START_END)
    if (((gt_PortModeControl.uc_AfePowerUp & AFE_POWERUP_MASK)  == AFE_COLD_START) ||
        (((gt_PortModeControl.uc_AfePowerUp & AFE_POWERUP_MASK) == AFE_1PORT_STARTED) && gft_BondedMode))
    {
        gft_PowerUpLink = TRUE;

        if (guc_ch_id == 0)
        {
            us_status = VR9_PowerUpAFE_Port0(gft_AEC_On);
        }
        else
        {
            us_status = VR9_PowerUpAFE_Port1(gft_AEC_On);
        }
    }
    else
    {
        gft_PowerUpLink = FALSE;

        // AFE_2PORTS_STARTED or (AFE_1PORT_STARTED && single-port-mode)
        us_status = VR9_AFEWarmInit();
    }

    return(us_status);
}


uint16 VR9_AFEWarmInit(void)
{
    uint16 us_status;
    VR9_AfeInitAnalogRegistersReset();
    VR9_AfeInitDigitalRegistersReset();
#if 1
    VR9_3GLP_Init_DFE(gft_BondedMode, gft_AEC_On);
    VR9_3GLP_Init_AFE(gft_BondedMode); // AFE_0 initialization
    us_status = 1;
#endif
#if 1
    VR9_WaitFor_3GLP_Ready();
#endif

    return us_status;
}


void SetAndWaitForNextAfeEvent(uint8 uc_CurState, uint8 uc_NextState)
{
    if (gft_BondedMode)
    {
        if (uc_CurState != NO_AFE_EVENT)
        {
            gt_PortModeControl.uc_AfeInitState = uc_CurState;
        }

        while(gt_PortModeControl.uc_AfeInitState != uc_NextState)
        {
            // will be optimize to render background time to the other port
        }
    }
    else
    {
        // for debug in 1-port mode; this allows us to know the last
        // sub-state/event before the AFE-init process fails.
        gt_PortModeControl.uc_AfeInitState = uc_CurState;

        // no synchronization needed
    }
}


uint16 VR9_PowerUpAFE_Port0(FlagT ft_AEC_On)
{
    uint32 ul_temp = 0;

    VR9_SSC_init(); //Initialize the SSC interface.

    //Make sure that we are not writing into the registers which
    //are the result of EfuseCopy(), MeasureRC().
    VR9_AfeInitAnalogRegistersReset();
    VR9_AfeInitDigitalRegistersReset();

    // synchronize with port1
    SetAndWaitForNextAfeEvent(AFE_PORT1_INIT_REGS, AFE_PORT1_INIT_REGS_DONE);

    // 3GLP reset, config and enabling of PORT_0 and PORT_1 based on bondedmode.
    VR9_3GLP_Init_DFE(gft_BondedMode, ft_AEC_On);

    // synchronize with port1
    SetAndWaitForNextAfeEvent(AFE_PORT1_INIT_3GLP, AFE_PORT1_INIT_3GLP_DONE);

    VR9_3GLP_Init_AFE(gft_BondedMode); // AFE_0 initialization

    //Wait till 3GLP is ready for PORT_0.
    //Log an exception code on timeout if the status is still not ready
    //and return from this function with an error.

      GlpEn_0();

    //There might be some delay needed here.
    wait_ms(1);


#if 1
    VR9_WaitFor_3GLP_Ready();
#endif

    VR9_EfuseCopy(); // Copying the fuse information

    VR9_MeasureRC(); // RC tuning

    //XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO (Start)
    //Initialize AFE FIFOs for all chips. Older chips don't have FIFOs. Still It won't cause any problem
    //as writing to those memory locations is void - Comment from Stefan
    VR9_Init_AFE_FIFOs();
    //XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO (End)

    //ADC trimming has been tested and found working.
    VR9_ADC_Trim();  // ADC trimming

    //Workaround: If any ADC Trim value is "0", this value should be replaced with 0xF,
    //This issue was observed in both Firmware as well as in Villah. This workaround is
    //done as per the I/P from villah. It may be removed, once after getting the proper fix.
    VR9_ADC_Trim_Workaround();

    // synchronize with port1
    SetAndWaitForNextAfeEvent(AFE_PORT1_FUSE_TUNE_TRIM, AFE_PORT1_FUSE_TUNE_TRIM_DONE);

    // successfully powered up AFE. Next time, it would be a WARM_START
    if (gft_BondedMode)
    {
      //XDSLRTFW-541: Platform_VR9_VRX318 (START_END)
      gt_PortModeControl.uc_AfePowerUp = (gt_PortModeControl.uc_AfePowerUp & (~AFE_POWERUP_MASK)) | AFE_2PORTS_STARTED;
    }
    else
    {
      //XDSLRTFW-541: Platform_VR9_VRX318 (START_END)
      gt_PortModeControl.uc_AfePowerUp = (gt_PortModeControl.uc_AfePowerUp & (~AFE_POWERUP_MASK)) | AFE_1PORT_STARTED;
    }
    gt_PortModeControl.uc_AfeInitState = AFE_INIT_DONE;

    return(READY);
}


uint16 VR9_PowerUpAFE_Port1(FlagT ft_AEC_On)
{
    uint32 ul_temp = 0;
    uint16 us_status;

    SetAndWaitForNextAfeEvent(NO_AFE_EVENT, AFE_PORT1_INIT_REGS);

    //Make sure that we are not writing into the registers which
    //are the result of EfuseCopy(), MeasureRC().
    VR9_AfeInitAnalogRegistersReset();
    VR9_AfeInitDigitalRegistersReset();

    // synchronize with port0
    SetAndWaitForNextAfeEvent(AFE_PORT1_INIT_REGS_DONE, AFE_PORT1_INIT_3GLP);

    VR9_3GLP_Init_AFE(gft_BondedMode); // AFE_0 initialization

    // synchronize with port0
    SetAndWaitForNextAfeEvent(AFE_PORT1_INIT_3GLP_DONE, AFE_PORT1_FUSE_TUNE_TRIM);


    //Wait till 3GLP is ready for PORT_1.
    //Log an exception code on timeout if the status is still not ready
    //and return from this function with an error.
    us_status = VR9_AFE_IF_Statuscheck(PORT_1); //for all the links.
    if (us_status != READY)
    {
        return us_status;
    }

   VR9_EfuseCopy(); // Copying the fuse information

    VR9_MeasureRC(); // RC tuning

    //XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO (Start)
    //Initialize AFE FIFOs for all chips. Older chips don't have FIFOs. Still It won't cause any problem
    //as writing to those memory locations is void - Comment from Stefan
    VR9_Init_AFE_FIFOs();
    //XDSLRTFW-271: Fix_AB_ALL_ALL_Init_FIFO (End)

#if 0 //VR9_BRINGUP
    //It has been commented since, it has the wait time of 500ms for every stage iteration for
    //4 modes, it leads to Not getting modem ready message. It needs to confirmed with AFE team
    //At the moment this function is not required.
    VR9_ADC_Trim();  // ADC trimming

    //Workaround: If any ADC Trim value is "0", this value should be replaced with 0xF,
    //This issue was observed in both Firmware as well as in Villah. This workaround is
    //done as per the I/P from villah. It may be removed, once after getting the proper fix.
    VR9_ADC_Trim_Workaround();
#endif

    // synchronize with port0
    SetAndWaitForNextAfeEvent(AFE_PORT1_FUSE_TUNE_TRIM_DONE, AFE_INIT_DONE);

    return(READY);
}

extern uint16 gus_3GLP_timeout_count;

void  VR9_WaitFor_3GLP_Ready()
{
   uint16 us_status;
   uint32 data;

   us_status = VR9_AFE_3GLP_Statuscheck(); //for all the links.


   // Wait for the GLP Ready
//   while ( us_status == 1 )
   //25/11/2009: Kannan: Added timeout to know the 3GLP status.
   //and exit the indefinite loop.
   while (( us_status == 1 ) && (gus_3GLP_timeout_count < 100))
   {
      gus_3GLP_timeout_count++;
     // GlPDis_0() - Start
     // Enable the GLP for Slice0 (single channel)
     ReadCoreReg(GLP_TOP_CFG, &data);
     data = data & (0xFFFFFFFE);
     WriteCoreReg(GLP_TOP_CFG, data);
     // GlPDis_0() - End
     wait_ms(1);
     GlpEn_0();
     wait_ms(1);
     us_status = VR9_AFE_3GLP_Statuscheck(); //for all the links.
   }

   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
   // us_status = 1 (GLP error), us_status = 0 (GLP Ready)
   if ((us_status == 1) && (gus_GLP_ExceptionCode != E_CODE_NO_ERROR))
   {
      if (gus_ExceptionCode == E_CODE_NO_ERROR)
         gus_ExceptionCode = gus_GLP_ExceptionCode;
   }
   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)

   // Finally disable ready force
   VR9_AfeWrite(GLP_FR_CTRL, 0x0, 0x4040);  //ready = 0 override

   // Configure the GLP FIFOs
   GlpFifoCfg_0();

   // Error correction in Bypass
   ErrCorrBypass_0();

   //READ FR (CRC) Errors to clear the counter
   ReadCoreReg(GLPM_GLP_FR_ERR_0, &data); //GLP Master FR Errors Read
   VR9_AfeRead(GLP_FR_ERR);               //GLP Slave FR Errors Read
   //XDSLRTFW-167: Reset registers for 3GLP sync monitoring (start)
   ReadCoreReg(GLP_TOP_STATUS, &data); //Reference: VR9ssc.pdf
   if( (data & GLP_TOP_STATUS_RDY0_MASK) == RDY0_GLP_RDY_1)
      SetCoreReg(GLP_TOP_STATUS,GLP_TOP_STATUS_NRDY0_MASK); // Reset Sticky bit
   //XDSLRTFW-167:Reset registers for 3GLP sync monitoring(end)
}


uint16 VR9_AFE_3GLP_Statuscheck()
{
   uint32 ul_timeout = 0;
   uint32 ul_temp = 0;
   uint16 us_fr_status;
   uint8 vr9a_framer_status, vr9a_remote_align;

   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
   gus_GLP_ExceptionCode = E_CODE_NO_ERROR; //initialize

  // Finally disable ready force
   VR9_AfeWrite(GLP_FR_CTRL, 0x0, 0x4040);  //ready = 0 override

   wait_ms(1);

   //1 MS timeout is OK, since this routine will be called during powerup
   //initialization
   do
   {
      ReadCoreReg(GLPM_GLP_FR_STATUS_0, &ul_temp); //Reference: VR9ssc.pdf
      ul_temp = (ul_temp & GLPM_FR_STATUS_0_STATE_AND_RMT_FM_AL_MASK);
      ul_timeout++;
   } while((ul_temp != STATE_AND_RMT_FM_AL_READY) && (ul_timeout < GLP_PORT_STAT_TIMEOUT));

   if (ul_timeout >= GLP_PORT_STAT_TIMEOUT)
   {
      //Log Exception code in gul_ExceptionCode
      //Port number in which Error occured <16>
      //3GLP startup procedure is not successful <15:0>
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
      gus_GLP_ExceptionCode = E_CODE_GLP_MASTER_DFE_NOT_READY;
      return 1; //Error
   }

  // check for the Remote frame alignment
  // ReadSerialIf(0,0x0101, &data);
  // vr9a_framer_status     = (UINT8) (data >> 13) & 0x7;
  // vr9a_remote_align      = (UINT8) (data >> 5) & 0x1;
  // check for vr9a_framer_status == 0x7 and vr9a_remote_align == 0x1
   ul_timeout = 0;
   do
   {
      //us_fr_status = GLP_FR_STATUS_STATE_AFE_GET; //Refer VR9Afe spec
      us_fr_status = GLP_FR_STATUS_AFE_GET;
      vr9a_framer_status = (uint8) (us_fr_status >> 13) & 0x7;
      vr9a_remote_align = (uint8) (us_fr_status >> 5) & 0x1;
      ul_timeout++;
   } while ((vr9a_framer_status != 0x7) && (vr9a_remote_align != 0x1) && (ul_timeout < GLP_PORT_STAT_TIMEOUT));

   if (ul_timeout >= GLP_PORT_STAT_TIMEOUT)
   {
      //Log Exception code in gul_ExceptionCode
      //Reason code <19:17>; 3 bits ; Refer "GLP_FR_STATUS<15:13>"
      //Port number in which Error occured <16>
      //Framer State Machine is not ready <15:0>
      //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
      gus_GLP_ExceptionCode = E_CODE_GLP_SLAVE_AFE_NOT_READY;
      return 1; //Error
   }

   return 0; //Ready
}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  VR9_AFE_IF_Statuscheck ()
**
** DESCRIPTION:    Generic routine which shall be called to read out the
**                 port status of 0 & 1.
**                 Check the GLPStatus(GLP_TOP_STATUS register) of DFE and AFE
**                   status in GLP_FR_STATUS register & PLL_STATUS register.
** PARAMETERS:     Port Number
**
** RETURN VALUE:   If GLP is ready, return READY, else ERROR
**
**
** NOTES:          Reference: (1) VR9_3GLP_IF.doc;
**                            (2) VR9ssc.pdf
**                            (3) VR9 Afe Spec
**                            SMS00946055
** =============================================================================
*/
uint16 VR9_AFE_IF_Statuscheck (uint8 port_num)
{
   uint32 ul_timeout = 0;
   uint32 ul_temp;
   uint16 us_fr_status;
   uint32 ul_glp_mask = GLP_MACRO_0_RDY; //Default PORT_0 Mask

   if (port_num != PORT_0)
      ul_glp_mask = GLP_MACRO_1_RDY; //Use PORT 1 Mask

   //1 MS timeout is OK, since this routine will be called during powerup
   //initialization
   do
   {
      ReadCoreReg(GLP_TOP_STATUS, &ul_temp); //Reference: VR9ssc.pdf
      ul_timeout++;
   } while((!(ul_temp & ul_glp_mask)) && (ul_timeout < GLP_PORT_STAT_TIMEOUT));

   if (ul_timeout >= GLP_PORT_STAT_TIMEOUT)
   {
      //Log Exception code in gul_ExceptionCode
      //Port number in which Error occured <16>
      //3GLP startup procedure is not successful <15:0>
      gul_ExceptionCode = (port_num << 16) | GLP_MACRO_STARTUP_NOT_READY;
      return ERROR;
   }

   ul_timeout = 0;
   do
   {
      us_fr_status = GLP_FR_STATUS_STATE_AFE_GET; //Refer VR9Afe spec
      ul_timeout++;
   } while ((us_fr_status != STATE_READY) && (ul_timeout < GLP_PORT_STAT_TIMEOUT));

   if (ul_timeout >= GLP_PORT_STAT_TIMEOUT)
   {
      //Log Exception code in gul_ExceptionCode
      //Reason code <19:17>; 3 bits ; Refer "GLP_FR_STATUS<15:13>"
      //Port number in which Error occured <16>
      //Framer State Machine is not ready <15:0>
      gul_ExceptionCode = (us_fr_status << 17) | (port_num << 16) | GLP_FRAMER_STATE_MACHINE_ERROR;
      return ERROR;
   }


   ul_timeout = 0;
   //Wait till the READY bit <bit4> is set or Timeout occurs
   do
   {
      ReadCoreReg(GLPM_PLL_STATUS, &ul_temp); //Reference: VR9ssc.pdf
      ul_timeout++;
   } while((!(ul_temp & GLPM_PLL_STATUS_READY)) && (ul_timeout < GLP_PORT_STAT_TIMEOUT));

   if (ul_timeout >= GLP_PORT_STAT_TIMEOUT)
   {
      //Log Exception code in gul_ExceptionCode
      //Port number in which Error occured <16>
      //PLL status not ready <15:0>
      gul_ExceptionCode = (port_num << 16) | GLPM_PLL_STATUS_NOT_READY;
      return ERROR;
   }

   return READY;
} //VR9_AFE_IF_Statuscheck ()

#ifdef VRX318
//XDSLRTFW-541: Platform_VRX318 (START)
void PowerBoost3GLP(void)
{
    // Only the following 4 calls inside VR9_3GLP_Init_AFE()
    // are retained for proper operation.
    // HSFPGA_SSC(device,r.PAD_CTL,'0010');
    PAD_CTL_SCDO_PPEN_AFE_SET(SCDO_PPEN_SCDO_PP);     // set SSC.SCDO push pull
    // HSFPGA_SSC(device,r.BIAS_1,'0000','000B');
    VR9_AfeWrite(BIAS_1,0x0, 0xB);      // Power up BIAS block
    //Increase data amplitude to maximum
    VR9_AfeWrite(TX_A_CTRL, 0x800F, 0x800F);
    //Increase transmitter termination to increase amplitude some more
    VR9_AfeWrite(TX_RTERM, 0x0010, 0xFFFF);
}
//XDSLRTFW-541: Platform_VRX318 (END)
#endif //VRX318
