/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 2009 - Infineon Technologies
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
// ******************************************************************
// RC_Measure.c
//
// History
//
// 15/01/2013 Kannan: AFE init fail, RC measurement exception code are initialized
//            Grep for XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication
//
// ******************************************************************

/***C***************************************************************************
** The 'CCASE' lines will be filled by ClearCase upon checkin,
** please do not change this information manually!
**
** =============================================================================
**
** All rights reserved.
** =============================================================================
**
** =============================================================================
**
** This document contains proprietary information belonging to Infineon
** Technologies. Passing on and copying of this document, and communication of
** its contents is not permitted without prior written authorisation.
**
** =============================================================================
**
**
** =============================================================================
**
** Revision Information :
**
** CCASE_FILE = ""
** CCASE_DATE = ""
**
** =============================================================================
**
** References: XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**             - See chapter 2.5 RC- Tuning for details of
**               programming the RC-Tuning sequence.
**
** =============================================================================
** History:
**
** Date        Author            Comment
**
** =============================================================================
**
Additional version control information
********************************************************************************
 CCASE_USER    = ""
 CCASE_VERSION = ""
***C*E**************************************************************************


** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/

#include "common.h"

#ifdef ADSL_62
    #include "const.h"
#else
    #include "vdsl_xception.h"
#endif

#include "typedef.h"
#include "vr9_afe_analog_reg.h"
#include "vr9_afe_dig_global_reg.h"
#include "AfeRdWrite.h"
#include "VR9_Afedata.h"
#include "VR9_AfeFunc.h"
#include "gdata.h"
#include "VRX_AfeCommonData.h"
/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/

// Maximum No. of iterations to check for RC measure status(ie.,DONE or ERROR)
// Each iteration will be waiting for certain delay (MEAS_DELAY_THRESHOLD)
#define RC_MEAS_NUM_RETRIES      (3)
//Input from Villach is Max delay can't be more than 1ms Final value TBD
#define MEAS_DELAY_THRESHOLD     (0x1234) //tbd

//Defines for mode of operations (ie.,RC Measure for Calib or Measurement)
#define CALIBRATION              (0)
#define MEASUREMENT              (1)

//Return status from RC Measurement function (ie., MeasureRC() )
#ifndef  TRUE
   #define TRUE                  (1)
#endif

#ifndef FALSE
   #define FALSE                 (0)
#endif

//Initialisations defined for RC Measurement status
#ifndef  RC_FAIL
   #define RC_FAIL               (1)
#endif
#ifndef RC_SUCCESS
   #define RC_SUCCESS            (0)
#endif

// Nominal count value to be returned if RC measure for any(CALIB & MEAS) or both
// the measurements fail. This value is taken as per page number 91 of
// "XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf"
// This value can be changed later
#define CNT_NOM                  (479)

/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/
/*
** =============================================================================
*/

/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  MeasureRC()
**
** DESCRIPTION:    MeasuresRC internally calls "ReadRCCnt" which calculates the
**                 the RC time constant value as described in sec 2.5 of AFEspec
**
**                 This function does the RC  measurement first in Calibration
**                 mode and then in Measurement mode.
**
**      Computation of the RC time constant detailed procedure is as given below.
**
**     1. Power Up RC Measurement Block
**     2. Set the calibration mode for the RC measurement in the analog part
**          to calibration.
**     3. Enable RC-measurement
**     4. Wait time of 1 ms
**     5. Starts the RC-measurement
**     6. Wait until the Measurement is completed
**       7. Read out the calibration value CNTcal
**     8. Stop Caliberation Measurement (close the switches)
**     9. Disable RC-measurement
**    10. Switch from Calib to Normal mode
**    11. Power Down RC Measuremnt Block
**    12. Wait time of 1ms
**    13. Power Up RC Measurement Block
**    14. Enable RC-measurement
**    15. Wait for 1ms
**    16. Start RC Measurement (open the switch and trigger HW counter)
**    17. Wait until the Measurement is completed
**    18. Read out the Measurement value CNTMeas
**    19. Calculate the Calibrated Value (EQ(1) in Chapte 2.5)
**        CNT = CNTMeas - CNTCal;
**    20. Stop Calibaration Measurement (close the switches)
**    21. Switch off and power down the RC Tuning Block
**        Return the value CNT
**
** PARAMETERS:     NONE
**
** RETURN VALUE:Flag Status TRUE for Success FAIL for failure
**
** NOTES:Reference:Sec2.5 of XWAY_VRX208_V1.0_Design_Spec_Rev1.0_20090417.pdf
**                :Vinax implementation in source\Vdsl_IFX\REV1.4\RC_Measure.c
**                 SMS00904137
**
**       Registers used :
**          RCMEAS_CTL(AFE Digital, REG_BANK_B: Global Registers 3, Offset 20H)
**          RCMEAS_STAT(AFE Digital, REG_BANK_B: Global Registers 3, Offset 21H)
**          TBUF_RCTUNING(AFE Analog register,Base 0x0300, Offset 10H )
**
** =============================================================================
*/

uint16 gus_CntCalib = 0x1234;
uint16 gus_CntMeas = 0xDEAD;

FlagT VR9_MeasureRC(void)
{
   uint16 us_CntCalib =0;
   uint16 us_CntMeas =0;
   uint16 us_data;
   FlagT gft_retVal;


   //Initialise Calibration/Measurement mode result to RC_FAIL
   guc_CalibResult= RC_FAIL;
   guc_MeasResult = RC_FAIL;

   //Call for calculating the RC CNT for measurement(normal) mode
   us_CntMeas  = ReadRCCnt(MEASUREMENT);
   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
   if ((gus_ExceptionCode == E_CODE_NO_ERROR) && (gus_RC_Measure_ExceptionCode != E_CODE_NO_ERROR))
      gus_ExceptionCode = gus_RC_Measure_ExceptionCode;
   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)

   //Call for calculating the RC CNT for calibration mode
   us_CntCalib = ReadRCCnt(CALIBRATION);
   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START)
   if ((gus_ExceptionCode == E_CODE_NO_ERROR ) && (gus_RC_Calib_ExceptionCode != E_CODE_NO_ERROR))
      gus_ExceptionCode = gus_RC_Calib_ExceptionCode;
   //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (END)


#if 1 //Debug
   gus_CntCalib = us_CntCalib;
   gus_CntMeas = us_CntMeas;
#endif

   //check for if both measurements successful or not
   if ((guc_CalibResult | guc_MeasResult)== RC_SUCCESS)
   {
      //If the RC measurement works properly, us_CntMeas > us_CntCalib
      us_data = (us_CntMeas - us_CntCalib);
      gus_RCtimeConstCNT = us_data;
      gft_retVal = TRUE;
   }
   //set to nominal count if any or both the measurements fail
   else
   {
      gus_RCtimeConstCNT = CNT_NOM;
      gft_retVal = FALSE;
   }

   //RC start measurement is disabled
   RCMEAS_CTL_RC_START_MEAS_N_AFE_SET(RC_START_MEAS_N_RC_START_MEAS_OFF);

   //RC tuning is disabled
   RCMEAS_CTL_RC_TUNING_EN_AFE_SET(RC_TUNING_EN_RC_TUN_OFF);

   //Power down RC tuning
   TBUF_RCTUNING_RCTUNING_PD_AFE_SET(RCTUNING_PD_RCTUNING_PD);

   // Return the status (TRUE/FALSE)
   return(gft_retVal);
} //end of MeasureRC()



/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  InitRCMeasMode(void)
**
** DESCRIPTION:   This function initializes for RC tuning in normal mode
**                and statrs RC measurement.Sequence is as follows
**                a)Power Up RC Meas Block,b)Set to normal mode,
**                c)Enable RC-meas,d)Wait for 1ms and e)Start the RC-measurement.
**
** PARAMETERS: none
**
** RETURN VALUE: none
**
**
** NOTES:   Reference:sequence followed as per Vinax referernce
**          Vinax implementation is in "\source\Vdsl_IFX\REV1.4\RC_Measure.c"
**          SMS00904137
**
**          Registers used
**            Control Register  TBUF_RCTUNING (AFE_BASE + 0x10)
**            RCMEAS_CTL (VINAX_AFE_DIG_GLOBAL_BASE + 0x20)
**            RCMEAS_CTL(AFE Digital, REG_BANK_B: Global Registers 3, Offset 20H)
**            RCMEAS_STAT (VINAX_AFE_DIG_GLOBAL_BASE + 0x21)
**            RCMEAS_STAT(AFE Digital, REG_BANK_B: Global Registers 3, Offset 21H)
**
** =============================================================================
*/

void InitRCMeasMode(void)
{

   // Power Up RC Meas Block (value=0000h)
   TBUF_RCTUNING_RCTUNING_PD_AFE_SET(RCTUNING_PD_RCTUNING_PU);

   //Set the calibration mode for the RC--meas in the analog part to calibration bit 12==1
   TBUF_RCTUNING_RCTUNING_CALIB_AFE_SET(RCTUNING_CALIB_RC_CALIB_OFF);

   // Enable RC tuning (value=0001h)
   RCMEAS_CTL_RC_TUNING_EN_AFE_SET(RC_TUNING_EN_RC_TUN_ON);

   // Wait time of 1ms
   wait_ms(1);

   // RC start measurement is enabled (open the switches)
   RCMEAS_CTL_RC_START_MEAS_N_AFE_SET(RC_START_MEAS_N_RC_START_MEAS_ON);

}

/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  InitRCCalibMode(void)
**
** DESCRIPTION:This function initializes RC tuning in Calibration mode
**             and starts RC measurement.As this is called after the normal
**             mode measurement,this function starts stop calibaration disable
**             RC measurement tuning and switch from Normal to Calib mode.
**
** PARAMETERS: none
**
** RETURN VALUE:
**
**
** NOTES:   Reference:Sequence followed as per Vinax referernce
**          Vinax implementation is in "\source\Vdsl_IFX\REV1.4\RC_Measure.c"
**          SMS00904137
**
**          Registers used
**            Control Register  TBUF_RCTUNING (AFE_BASE + 0x10)
**            RCMEAS_CTL (VINAX_AFE_DIG_GLOBAL_BASE + 0x20)
**            RCMEAS_CTL(AFE Digital, REG_BANK_B: Global Registers 3, Offset 20H)
**            RCMEAS_STAT (VINAX_AFE_DIG_GLOBAL_BASE + 0x21)
**            RCMEAS_STAT(AFE Digital, REG_BANK_B: Global Registers 3, Offset 21H)
**
**
** =============================================================================
*/

void InitRCCalibMode(void)
{

   //RC start measurement is disabled(ie.,Stop Calibaration Measurement)
   RCMEAS_CTL_RC_START_MEAS_N_AFE_SET(RC_START_MEAS_N_RC_START_MEAS_OFF);

   // Calibration mode OFF(ie.,switch from Calib to Normal (value=0000h))
   TBUF_RCTUNING_RCTUNING_CALIB_AFE_SET(RCTUNING_CALIB_RC_CALIB_ON);

   //now start the measurement in normal mode
   RCMEAS_CTL_RC_START_MEAS_N_AFE_SET(RC_START_MEAS_N_RC_START_MEAS_ON);

}

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  ReadRCCnt()
**
** DESCRIPTION: Measures the RC time constant value as described in chapter 2.5
**             of AFE spec. The RC measurement based on the input arguement
**             ie., either calibration or measurement mode.
**             Try to read the RC time constant within certain period
**             MEAS_DELAY_THRESHOLD(the value TBD,not to exceed 1ms)
**             If response is not recieved repeat the procedure again.
**             The no.of repetitions should be RC_MEAS_NUM_RETRIES(Value TBD)
**
** PARAMETERS:    Input mode(CALIBRATION/MEASUREMENT)
**
** RETURN VALUE:   CNT, which is the RC time constatnt value.
**
** =============================================================================
*/

uint16 ReadRCCnt(uint16 us_mode)
{
   uint16 us_Wait_MeasCnt=0;
   uint16 us_Result = RC_FAIL; // Local initialissation of result to RC_FAIL
   uint16 us_MeasStatus =0;
   uint16 us_RetryCnt =0;
   uint16 us_Cnt =0;

   // Loop for number of iterations
   while((us_Result != RC_SUCCESS) && (us_RetryCnt < RC_MEAS_NUM_RETRIES))
   {
      if(us_mode == CALIBRATION)
      {
         InitRCCalibMode(); // Initialisation for calibration mode
         //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
         gus_RC_Calib_ExceptionCode = E_CODE_NO_ERROR;
      }
      else
      {
         InitRCMeasMode(); // Initialisation for Measurement(normal)mode
         //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
         gus_RC_Measure_ExceptionCode = E_CODE_NO_ERROR;
      }

      // Read the  RC Measurement Status
      us_MeasStatus = RCMEAS_STAT_AFE_GET;
      // Loop for Status check for meas done or error and max delay
      while(
         ((us_MeasStatus & (RCMEAS_STAT_RC_ERROR_MASK|RCMEAS_STAT_RC_MEAS_DONE_MASK))==0)
         &&
         (us_Wait_MeasCnt < MEAS_DELAY_THRESHOLD)
         )
      {
         // Read the  RC Measurement Status
         us_MeasStatus = RCMEAS_STAT_AFE_GET;
         us_Wait_MeasCnt++;
      }

      //Check for Measurement done
      if (us_MeasStatus & RCMEAS_STAT_RC_MEAS_DONE_MASK)
      {
         us_Result = RC_SUCCESS;//Set result to RC_SUCCESS
         if(us_mode == CALIBRATION)
            guc_CalibResult = RC_SUCCESS;//Set global calib result to RC_SUCCESS
         else
            guc_MeasResult = RC_SUCCESS;//Set global meas result to RC_SUCCESS

         us_Wait_MeasCnt =0;
         us_Cnt = (us_MeasStatus & RCMEAS_STAT_RC_TUNE_COUNT_MASK);
      }
      else
      {
         //Check for max number of retries
         if (us_RetryCnt == (RC_MEAS_NUM_RETRIES-1))
         {
            // Check if Error Status is Set
            if (us_MeasStatus & RCMEAS_STAT_RC_ERROR_MASK)
            {
               if (us_mode == CALIBRATION)
               {
                  //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
                  gus_RC_Calib_ExceptionCode = E_CODE_AFE_RC_CALIB_ERROR;
               }
               else
               {
                  //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
                  gus_RC_Measure_ExceptionCode = E_CODE_AFE_RC_MEAS_ERROR;
               }
               guc_ErrorType = (us_MeasStatus & RCMEAS_STAT_RC_ERROR_TYPE_MASK)>> 12;

            }
            // else Timeout occured, return timeout exception
            else
            {
               if (us_mode == CALIBRATION)
               {
                  //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
                  gus_RC_Calib_ExceptionCode = E_CODE_AFE_RC_CALIB_TIMEOUT;
               }
               else
               {
                  //XDSLRTFW-181 VR9_VRX318_AFE_Init_Fail_Indication (START_END)
                  gus_RC_Measure_ExceptionCode = E_CODE_AFE_RC_MEAS_TIMEOUT;
               }
            }
         }  // end of if (us_RetryCnt == (RC_MEAS_NUM_RETRIES-1))
         us_Wait_MeasCnt = 0;
         us_RetryCnt++;
         // Power Down RC Meas Block (value=8000h)
         TBUF_RCTUNING_RCTUNING_PD_AFE_SET(RCTUNING_PD_RCTUNING_PD);
         // wait for 1 msec
         wait_ms(1);
      }
   } //end of while

   return(us_Cnt);
} //end of ReadRCCnt()


