/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*
*   file name: AFED_FlterTune.c
*
*   This file contains VRX518 AFE Filter tuning formula implementation routines
*
*-------------------------------------------------------------------------------
*/

// **********************************************************************************************//
// History
// 10/05/2016 Palaksha: Added VRX518 Filter tune routines
//
// **********************************************************************************************//

//===========================INCLUDE FILES ========================================
#include "typedef.h"
#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Constants.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"
#include "AFED_ReadWriteModify.h"
#include "Decimalgain.h"
#include "mul.h"

#include "vrx5afe_mode_defines.h"
#include "vrx5afe_central_addrmap.h"
#include "vrx5afe_central_fcsi_regpkg.h"     // register for CENTRAL
#include "vrx5afe_dsl_addrmap.h"
#include "vrx5afe_dsl_fcsi_regpkg.h"         // register for DSL

////////////////// VRX518 DSL FW Routines  //////////////


//-------------------------------
// float2int subroutines
// Fixed point implementation FW
//-------------------------------

/*^^^
*-------------------------------------------------------------------
*
*  Function Name: void Calc_PgaClp1PrefiClp2PocoC2
*
*   Description:
*      This function Implements below formula:
*
*     Rpga1 = pow(10.0, gain_pga_dB/20.0) * Rsa;
*     Clp1 = 1.0 / (2.0 * 3.1415 * fc_pga_lp_kHz*1e3 * Rpga1);               // in [F]
*     Clp1_pF_weighted = Clp1*1e12 / weight_clp1 + 0.5;                      // 0.5*LSB due to round
*
*   Input Arguments:
*     int16_t Rsa                //Q0
*     int16_t weight_clp1           //Q13
*     int16_t gain_pga_dB           //Q8
*     int16_t fc_pga_lp_MHz         //Q0
*     uint32_t Value1e9By2piFcKhz      //Q20
*
*   Ouput Arguments:
*      uint16_t C_Bin  Q0
*
*   Return:
*
*-------------------------------------------------------------------
*^^^
*/

uint16_t Calc_PgaClp1PrefiClp2PocoC2(int16_t Rsa, int16_t weight_clp1, int16_t  gain_pga_dB, uint32_t Value1e9By2piFcKhz, uint8_t ChooseFilter)
{
   int16_t mant1, exp1, temp1;
   uint16_t Cxp_bin;
   int32_t temp2, temp3;
   int32_t Clp1_pF_weighted, Cx1e12;

   // get RC-counter value from global variable
   uint16_t RC_count = vrx5afe_get_rc_count();

   //Rpga1 = pow(10.0, gain_pga_dB/20.0) * Rsa;
   //Clp1 = 1.0 / (2.0 * 3.1415 * fc_pga_lp_kHz*1e3 * Rpga1);               // in [F]
   //Clp1_pF_weighted = Clp1*1e12 / weight_clp1 + 0.5;                      // 0.5*LSB due to round

   //Function db_to_linear calculates 10^(x/20), Convert to 8.8 dB format for input to db_to_linear.
   //mantissa (Q15 format) value; exponent (# of left shifts)
   //pow(10.0, gain_pga_dB/20.0);dB can be from -20 to +21 means values from 0.2 to 12, Needs max 4 bits
   db_to_linear(gain_pga_dB, &mant1, &exp1);//pow(10.0, gain_pga_dB / 20.0);

   //Clp1*1e12
   //(1e6)/(2.0 * 3.1415 * fc_pga_lp_MHz * Rpga1) = (1e6)/(2.0 * 3.1415 * fc_pga_lp_MHz * Rsa  pow(10.0, gain_pga_dB/20.0) ;
   //Translate into -
   //(1e6)/(2.0 * 3.1415 * fc_pga_lp_MHz * Rpga1) = (1e9)/(2.0 * 3.1415* fc_pga_lp_kHz  * Rsa *   pow(10.0, gain_pga_dB/20.0) ;
   //Value1e9By2piFcKhz = (1e9)/(2.0 * 3.1415* fc_pga_lp_kHz) is pre calculated
   //Value1e9By2piFcKhz / Rsa *   pow(10.0, gain_pga_dB/20.0) is pre calculated
   //Rsa can be 400 to 5300 means (8 bits to 13 bits):9 bits can be used for fractional part
   temp2 = Value1e9By2piFcKhz / Rsa; //Q20/Q0=Q20 ; //V_1e9By2piFcKhzByR
   temp2 = (temp2 << 9); //Q29   //Value1e9By2piFcKhz / Rsa  compare with float V_1e9By2piFcKhzByR

   if (exp1 < 0)
   {
      temp1 = -exp1;
      //pow(10.0, gain_pga_dB / 20.0);
      temp3 = mant1 / (1 << temp1); //Q15 format
   }
   else
   {
      temp1 = exp1;
      //pow(10.0, gain_pga_dB / 20.0);
      temp3 = mant1 * (1 << temp1); //Q15 format
   }
   //Value1e9By2piFcKhz /   (Rsa *   pow(10.0, gain_pga_dB/20.0))

   Cx1e12 = temp2 / (temp3 >> 3); //Q29/Q12 =    Q17

    if (ChooseFilter == 2)   // PREFI: To support RC-tuning of Prefi Clp2 = Clp2 * RC_nom / RC_count;
    {
        temp3 = (weight_clp1 * RC_count) / RC_nom; //(Q13*Q0/Q0=Q13)
        weight_clp1 = (int16_t)temp3;
    }

   if (ChooseFilter == 1)  // POFI
   {
      //if (C2*1e12 < weight_c2) //pF
      // C2 = weight_c2 / 1e12; //F
      //C2_pF_weighted = C2*1e12 / weight_c2;         // linear spaced C2-values
      //C2_bin = ( (uint16_t) C2_pF_weighted ) - 1;
      //weight_c2 = 1.55;
      //WeightCap = (int16_t) (weight_c2 * 8192);  //Q13
      if ((Cx1e12 << 5) < weight_clp1) // C2*1e12 < weight_c2  9-(17-13)= 5
      {
         //C2 = weight_c2 / 1e12;
         //C2_pF_weighted = C2*1e12 / weight_c2;
         //C2_pF_weighted = (weight_c2 / 1e12)*1e12 / weight_c2; = 1
         //C2_bin = ( (uint16_t) C2_pF_weighted ) - 1; = 1-1=0
         Cxp_bin = 0;
      }
      else
      {
         //C2_pF_weighted = C2*1e12 / weight_c2;                            // linear spaced C2-values
         //C2_bin = ((uint16_t)C2_pF_weighted) - 1;                         // truncate

         Clp1_pF_weighted = (Cx1e12 << (2 + SCALE_1E9BY2PIX512X256BY9)) / weight_clp1; // (17+2=19), 9 is scaled one  Q19/Q13 =Q6
         Cxp_bin = ((uint16_t)(Clp1_pF_weighted >> 6)) - 1;
      }
   }
   else // PGA
   {
        Clp1_pF_weighted = (Cx1e12 << (2 + SCALE_1E9BY2PIX512X256BY9)) / weight_clp1 + (1 << 5); // (17+2=19), 9 is scaled one  Q19/Q13 =Q6
      Cxp_bin = (uint16_t)(Clp1_pF_weighted >> 6);
   }
    gul_Afe_flowtest |=0x8000000;

   return (Cxp_bin);
}



/*^^^
*-------------------------------------------------------------------
*
*  void db_to_linear(int16 s_dB_input, int16 *ps_lin_mant, int16 *ps_lin_exp);
*
*   Description:
*      Computes the decimal equivalent of the fine gain in 1.15 format,
*      Assumes the the input gain_db is in 8.8 format and is within the
*      range [-128.0 dB, +127.0 dB].
*      Returns 10^(gain_db/20) as mantissa (Q1.15 format) and exponent (# left shifts)
*
*      Algorithm:
*         1)  Map input x to range [-6.0 dB, 0.0 dB])
*
*         2)   Polynomial Expansion Details:
*            The original coeffficients are : 0.0002    0.0064    0.1148    1.0000
*            To fit the 16 bit coef size they are stored as
*            c1*8*8*8  c2*8*8  c3*8  c4
*            0.0929    0.4078    0.9187    1.0000
*            0.0002    0.0064    0.1148    1.0000
*
*   Input Arguments:
*      s_dB_input -- input value in dB (Q8.8 format)
*
*   Ouput Arguments:
*      *ps_lin_mant -- ouput linear value mantissa (Q1.15 format)
*      *ps_lin_exp  -- ouput linear value exponent (# of left shifts)
*
*   Return:
*
*-------------------------------------------------------------------
*^^^
*/

//  note this is not 6*256 but round(20*log10(2)*256)
#define   LOG2x20_DB 1541   //   in Q8.8 format

// coefs for power seies approx of 10^(x/20)
int16 gsa_logexpansion_coefs[4]= { 3045    ,   13364  ,     30104    ,   32767};

void db_to_linear(int16 s_dB_input, int16 *ps_lin_mant, int16 *ps_lin_exp)
{
   int16 norm_dB_input, exp, n; // by keeping it as an int avoid sex instructions.
   int32 l_acc;
   int16 s_acc, *ps_coef;

   // Prescale input within range
   // Map input to range [-6,0], where the approximation is valid
   norm_dB_input = s_dB_input;   // by keeping it as an int avoid sex instructions.
   exp = 0;
   while (norm_dB_input < -LOG2x20_DB)
   {
      norm_dB_input += LOG2x20_DB;
      exp--;
   }
   while (norm_dB_input > 0)
   {
      norm_dB_input -= LOG2x20_DB;
      exp++;
   }

   // 3rd order power series expansion from -6 to 0
   // Calculate  ((((c3*x)+c2)*x)+c1)*x+c0      (ideal)
   // or as      ((((c3*x)/8 +c2)*x)/8 +c1)*x/8 +c0   (coef precision enhancement)
   // Each iteration is
   //  (s_acc[1.15] * norm_dB_input[8.8] )/256  + coef(i)
   // While it may appear that the product is 9.23 and will overflow when you take the
   // the bottom 16 bits -> it won't because the worst case s_db is -6 and the coefs are small
   ps_coef= gsa_logexpansion_coefs;
   s_acc = *ps_coef++;
   n=3;
   do
   {
      MULS16(l_acc, s_acc,norm_dB_input) ;
      l_acc= ((l_acc>>8)>>3);  // /256 to convert a into a 1.15 quantity.
      // /8 is to compensate for the coef scaling.
      s_acc = (int16) (l_acc + *ps_coef++ );
   }
   while(--n);

   // return
   *ps_lin_mant = s_acc;
   *ps_lin_exp = (int16)exp;
}

