/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*
*   file name: AFED_ModeConfig.c
*
*   This file contains VRX518 AFE Mode specific routines
*
*-------------------------------------------------------------------------------
*/

// **********************************************************************************************//
// History
// 10/05/2016 Palaksha: Added few VRX518 Mode specific routines
//
// **********************************************************************************************//

/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/
#include "common.h"
#include "typedef.h"
#include "str_memmap.h"
#include "LL_IOf.h"
#include "delay.h"
#include "Const.h"
#include "Decimalgain.h"
#include "cmv.h"
#include "gdata.h"

#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Constants.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"
#include "AFED_ReadWriteModify.h"

#include "vrx5afe_mode_defines.h"
#include "vrx5afe_central_addrmap.h"
#include "vrx5afe_central_fcsi_regpkg.h"     // register for CENTRAL
#include "vrx5afe_dsl_addrmap.h"
#include "vrx5afe_dsl_fcsi_regpkg.h"         // register for DSL


/*
** =============================================================================
**
** FUNCTION-NAME: AFED_ModeConfig()
**
**
** DESCRIPTION:   This function shall be called before handshake
**                This function will configure the AFE Tx/Rx paths
**                Default configuration for VDSL mode is VDSL_8
**                selection is done
**                This function will configure the AFE Tx/Rx paths
**
** PARAMETERS:    uint16 DslMode
**
** RETURN VALUE:   None
**
** NOTES:
**
** =============================================================================
*/

void AFED_ModeConfig(void)
{
  //AFE Tx and Rx Path confi based on AFE mode
   AFED_ConfigAfeTxPath(gus_AFE_TxMode);

   AFED_ConfigAfeRxPath(gus_AFE_RxMode);

   if(gft_AFE_TxMode_Overwrite == TRUE)
   {
      AFED_ConfigAfeTxPathForced();
   }

   if(gft_AFE_RxMode_Overwrite == TRUE)
   {
      AFED_ConfigAfeRxPathForced();
   }

}

/*
** =============================================================================
** FUNCTION-NAME:   AFED_ConfigAfeTxPath
**
** DESCRIPTION:   configure TxPath  Analog and Digital registers based Mode
**                This function shall be called from ConfigEngineForLinkStart(),
**                ConfigEngineForPostGhs() and from the state where HW reconfig-
**                uration is done based on the messages exchanged in CD phase
**                for the all the VDSL modes.
**
** PARAMETERS:     uint16 us_xDSLMode -> VDSL modes
**
** RETURN VALUE:   none
**
** NOTES:Reference:
**
** =============================================================================
*/

void AFED_ConfigAfeTxPath(uint16 AfeTxMode)
{
   AFED_ConfigAnalogTxPath(AfeTxMode);
   AFED_ConfigDigitalTxPath(AfeTxMode);
}

/*
** =============================================================================
** FUNCTION-NAME:   AFED_ConfigAnalogTxPath
**
** DESCRIPTION:     configure TxPath  AFE analog registers based on VDSL Mode
**
** PARAMETERS:     uint16 us_xDSLMode -> VDSL modes
**
** RETURN VALUE:   none
**
** NOTES:Reference:
**
** =============================================================================
*/
void AFED_ConfigAnalogTxPath(uint16 mode_set)
{
   //VRX518_BringUp
   //gain_pofi_dB 1 done here
   //POCO-setting:
   //POCO: bit12..8  =1BH
   //POCO: bit1:0    = 3H


      // Setting PLL from 288MHz to 144MHz via direct-control
      // means VDSL-mode to ADSL-mode
      // nothing to configure in DAC and SYNC

      // POFI gain + fc_lp
      //                         BYPASSED   VDSL35b  VDSL17   VDSL8     ADSL2
      //int16_t gain_pofi_dB[]      = {   0,    7,       7,    -3,      -3 };
      //int32_t fc_pofi_lp_kHz[]     = { 80000,    80000,      30000,   15000,    320 };

      //DSL Mode                         VDSL17    VDSL8  ADSL2               VDSL35b    BYP
      //Index                       0     1     2     3        4     5     6     7     8
      //int16_t gain_pofi_dB[]   = { 0,     4,     00000,  4,     -3,   00000, 00000,   4,         0    };
      //int32_t fc_pofi_lp_kHz[] = { 0,   30000, 00000, 15000,  320,   00000, 00000,  80000,    80000};


        // this check is for VRx518 bring-up only
        if (gus_SkipAfeFlow & 0x0010)
        {
           return;
        }

      vrx5afe_dsl_dac_conf(mode_set);

      vrx5afe_dsl_pofi_gainfc_set(mode_set, gsa_gain_pofi_dB[mode_set], gla_fc_pofi_lp_kHz[mode_set]);

       if (mode_set == ADSL2_CPE)
       {
          gus_LD_Mode = LD_M_ADSL;
       }

      // Setting LD mode
      // Note: The Line Driver mode gets only be changed, if it differs from
      // the previous mode read out of the HW!
      AFED_SetLdMode(gus_LD_Mode);

}

/*
** =============================================================================
** FUNCTION-NAME:   AFED_ConfigDigitalTxPath
**
** DESCRIPTION:   configure TxPath  DFI registers based on VDSL Mode
**
** PARAMETERS:     uint16 us_xDSLMode -> VDSL modes
**
** RETURN VALUE:   none
**
** NOTES:Reference:
**
** =============================================================================
*/
void AFED_ConfigDigitalTxPath(uint16 AfeTxMode)
{
   // ---------- Configure DFI to start with Handshake -----------------------
   //CHECK_VRX518
   //Initialize DFI with 17a configuration since,
   //both FFT & IFFT will be configured with 4K tones before G.hs.
   //Do the reconfig once more after the profile is selected.
   VRX518_DfiConfig_Common(AfeTxMode);
   VRX518_DfiConfig_TxPath(AfeTxMode);

}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   AFED_ConfigAfeRxPath()
**
** DESCRIPTION:   This function shall be called from ConfigEngineForLinkStart(),
**                ConfigEngineForPostGhs() and from the state where HW reconfig-
**                uration is done based on the messages exchanged in CD phase
**                for the all the VDSL modes.
**
** PARAMETERS:     1. us_xDSLMode is the mode for which AFE regsiters needs to be
**                    programmed.
**
** RETURN VALUE:   NONE
**
** NOTES:Reference:
**
** =============================================================================
*/

void AFED_ConfigAfeRxPath(uint16 AfeRxMode)
{
   AFED_ConfigAnalogRxPath(AfeRxMode);
   AFED_ConfigDigitalRxPath(AfeRxMode);
}


/*
** =============================================================================
** FUNCTION-NAME:   AFED_ConfigAnalogRxPath
**
** DESCRIPTION:   configure RxPath  Analog AFE registers based on VDSL Mode
**
** PARAMETERS:     uint16 us_xDSLMode -> VDSL modes
**
** RETURN VALUE:   none
**
** NOTES:Reference:
**
** =============================================================================
*/

void AFED_ConfigAnalogRxPath(uint16 mode_set)
{

   //VRX518_BringUp
   //required setting for HS (17m-setting) and Showtime for 50m loop in VDSL35B derived from Franks settings.
   //It is one common setting for the moment
   //
   //  PGA0=  0253H
   //  PGA1=   043FH
   //  PGA2=   0133H
   //gain_pga_dB   -9 Done here
   //gain_prefi_dB -5  Done here
   //PGA6=0;ACE=0ff

   //#define  VDSL_17  VDSL17_CPE   (1)
   //#define  VDSL_8   VDSL8_CPE    (3)
   //#define  ADSL_A   ADSL2_CPE    (4)
   //#define  VDSL_35  VDSL35b_CPE  (7)
   //#define BYPASSED               (8)

   // PGA gain + fc_lp
   //                         BYPASSED   VDSL35b   VDSL17   VDSL8    ADSL2
   //int16_t gain_pga_dB[]   = {   0,        -9,       -9,     -9,      -9 };
   //int32_t fc_pga_lp_kHz[] = { 50000,     50000,    25000,  12500,   10000 };

   //DSL Mode                        VDSL17        VDSL8  ADSL2                VDSL35b   BYP
   //Index                      0    1      2      3      4      5      6      7         8
    int16_t gain_pga_dB[]   = { 0,   -9,    00000, -9,    -9,    00000, 00000, -9,       0    };
    //int32_t fc_pga_lp_kHz[] = { 0,   25000, 00000, 12500, 10000, 00000, 00000, 50000,    50000};


   // PREFI gain + fc_lp
   //                               BYPASSED   VDSL35b   VDSL17   VDSL8    ADSL2
   //int16_t gain_prefi_dB[]   = {   0,        -5,       -5,      -5,      -5 };
   //int32_t fc_prefi_lp_kHz[] = { 50000,     50000,    25000,   12500,   10000 };

    //DSL Mode                            VDSL17      VDSL8  ADSL2                VDSL35b   BYP
   //Index                          0     1     2     3     4     5     6     7     8
//    int16_t gain_prefi_dB[]     = { 0,   -5,  00000, -5,    -5,    00000, 00000, -5,    0    };
   int16_t gain_prefi_dB[]    = { 0,   -8,   00000, -8,     -8,   00000, 00000, -8,    0     };

   //int32_t fc_prefi_lp_kHz[]   = { 0,   25000, 00000, 12500, 10000, 00000, 00000, 50000,   50000};

   int16_t GainPrefiModeSpecific;  //VRX518_DEBUG_CHECK

   // this check is for VRx518 bring-up only
   if (gus_SkipAfeFlow & 0x0020)
   {
      return;
   }

   // SDADC configures SYNC - clk/2 (internal clk-divider for ADSL/VDSL)
   // gets always 1.152GHz clock from VLCPLL
   // SYNC from ADC -> in vrx5afe_dsl_init()
    vrx5afe_dsl_adc_conf(mode_set);


   // LA/Hybrid
    //vrx5afe_dsl_pga_hybrid_set(mode_set);

    // PGA gain + fc_lp
    vrx5afe_dsl_pga_gainfc_set(mode_set, gain_pga_dB[mode_set], gla_fc_pga_lp_kHz[mode_set]);


   // PREFI ACE
   vrx5afe_dsl_prefi_ace_set(ACE_OFF);

   GainPrefiModeSpecific = gain_prefi_dB[mode_set];

   //VRX518_DEBUG_CHECK Start
   if (mode_set == VDSL35b_CPE)
    {
      GainPrefiModeSpecific = -12;
    }

   // PREFI gain + fc_lp
    vrx5afe_dsl_prefi_gainfc_set(mode_set, GainPrefiModeSpecific, gla_fc_prefi_lp_kHz[mode_set]);
}

/*
** =============================================================================
** FUNCTION-NAME:   AFED_ConfigDigitalRxPath
**
** DESCRIPTION:   configure RxPath  DFI registers based on VDSL Mode
**
** PARAMETERS:     uint16 us_xDSLMode -> VDSL modes
**
** RETURN VALUE:   none
**
** NOTES:Reference:
**
** =============================================================================
*/

void AFED_ConfigDigitalRxPath(uint16 AfeRxMode)
{

    // DFI Rx Path config mode
   VRX518_DfiConfig_RxPath(AfeRxMode);
}


void vrx5afe_dsl_pga_hybrid_set(uint16_t mode_set)
{
   // LA - line adaptation (Hybrid)

   // Rsa->400Ohm, Rpa->inf
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA0),
         VRX5AFE_DSL_FCSI_PGA0_PGA35RPA__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA0_PGA35RSA__ISMSK,
         VRX5AFE_DSL_FCSI_PGA0_PGA35RPA_INF__SVAL | \
         VRX5AFE_DSL_FCSI_PGA0_PGA35RSA_RSA400__SVAL);

   // Rsb->inf, Rpb->inf, Cpb->inf
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA1),
         VRX5AFE_DSL_FCSI_PGA1_PGA35RSB__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA1_PGA35RPB__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA1_PGA35CPB__ISMSK,
         VRX5AFE_DSL_FCSI_PGA1_PGA35RSB_INF__SVAL | \
         VRX5AFE_DSL_FCSI_PGA1_PGA35RPB_INF__SVAL | \
         VRX5AFE_DSL_FCSI_PGA1_PGA35CPB_CPB0PF__SVAL);

   // Rsc->inf, Rpc->inf, Cpc->inf
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA2),
         VRX5AFE_DSL_FCSI_PGA2_PGA35RSC__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA2_PGA35RPC__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA2_PGA35CPC__ISMSK,
         VRX5AFE_DSL_FCSI_PGA2_PGA35RSC_INF__SVAL | \
         VRX5AFE_DSL_FCSI_PGA2_PGA35RPC_INF__SVAL | \
         VRX5AFE_DSL_FCSI_PGA2_PGA35CPC_CPC0PF__SVAL);


   if (mode_set == BYPASSED) // 0dB gain to be programmed
                           // covered in vrx5afe_dsl_pga_gainfc_set()
   {

   }

   // Hybrid/LA settings to be defined by Fleischhacker Chr.
   if ( (mode_set == VDSL35b_CPE) | (mode_set == VDSL17_CPE) | (mode_set == VDSL8_CPE) )
   {

   }

   if (mode_set == ADSL2_CPE)
   {

   }
}


void vrx5afe_dsl_pga_gainfc_set(uint16_t mode_set, int16_t gain_pga_dB, int32_t fc_pga_lp_kHz)
{
   //--------------------------------------------------------------------------
   // e.g. for passing parameters
   // vrx5afe_dsl_pga_gainfc_set(VDSL2_CPE, -9, 50000);
   // mode, gain [dB], fc_lp [kHz]
   //
   // HYBRID/LA -> vrx5afe_dsl_pga_hybrid_set(mode_set)
   //
   // OA-CTRL mode dependent
   //--------------------------------------------------------------------------
   gt_vrx5afe_dsl_config.us_dsl_pga_gainfc_mode_set        = mode_set;
   gt_vrx5afe_dsl_config.s_dsl_pga_gainfc_gain_pga_dB      = gain_pga_dB;
   gt_vrx5afe_dsl_config.l_dsl_pga_gainfc_fc_pga_lp_kHz    = fc_pga_lp_kHz;


#ifdef TEST_FLOAT_VRX518
   double weight_clp1 = 1.0;   // 'LSB' in [pF]

   double Rpga1, Clp1, Clp1_pF_weighted;
   double Clp1e12, V_1e9By2piFcKhzByR;
   uint16_t Clp1_bin_Float;
#endif
   int16_t Rsa=400;
   //int16_t Rsa=0;

   int16_t gain_bin, gain_bin_shift;
   //int16_t Rpga1;

   int16_t oactrl_oa_enable, oactrl_c_tune = 0;
   int16_t oactrl_oa_enable_shift, oactrl_c_tune_shift;
   uint16_t Clp1_bin, Clp1_bin_shift;

    // Fixed point function related variables
   int16_t Rohm;           // Q0
   int16_t WeightCap;         // Q13
   int16_t GaindB;            // Q8
   uint8_t ChooseFilter = 0;   // Q0 - 0=PGA, 1=POFI, 2=PREFI
   int32_t Const1e9By2PiX;
   int32_t Value1e9By2piFcKhz; // Q20


   //*********(1e9/ (2.0 * 3.1415* fc_pga_lp_kHz * 512) ) // Scale by value 512 (9 bits) to reduce the range of number *********//
   // rearrange constants and do off-line float computation
   //1e9/ (2.0 * 3.1415 * 512 * fc_pga_lp_kHz) )  = (1e9/ (2.0 * 3.1415 * 512)) / fc_pga_lp_kHz
   //Const1e9By2pix = ((1e9 / (2.0 * 3.1415 * 512 * 256)) * 2 ^ 20) = 0x4BE4AAA9  Q20

   // Here scale factor 256(8bits) is used later, 512(9 bits) is  used later
    Const1e9By2PiX     = ((int32_t) NUM_Q20_1E9BY2PIX512X256) / fc_pga_lp_kHz;    // Q20/Q0 = Q0

    // assuming fc_pofi_lp_kHz in value of 256, which means division will give 8 more bits
    Value1e9By2piFcKhz = (int32_t) (Const1e9By2PiX << SCALE_1E9BY2PIX512X256BY8); // Q20  Scale back again with scale factor 256(8bits). 9 bit scale factor will be taken later


    // get Rsa-value from global variable RSA
    //Rsa = vrx5afe_get_hybrid_rsa();


   //---------
   // OA-CTRL
   //---------

   // PUP of OpAmp
   oactrl_oa_enable = 1; // always 1

   // comp. cap. tuning
   if (mode_set == VDSL35b_CPE)
   {
      if (gain_pga_dB >= -9 && gain_pga_dB <=0 )
      {
         oactrl_c_tune = 2;
      }
      else if (gain_pga_dB >= 1 && gain_pga_dB <=10 )
      {
         oactrl_c_tune = 2;
      }
      else if (gain_pga_dB >= 11 && gain_pga_dB <=15 )
      {
         oactrl_c_tune = 7; // Clp1: tiny | small | large = 7 | 6 | NA
      }
   }

   if ((mode_set == VDSL17_CPE) | (mode_set == VDSL8_CPE) | (mode_set == ADSL2_CPE))
   {
      if (gain_pga_dB >= -9 && gain_pga_dB <=0 )
      {
         oactrl_c_tune = 2;
      }
      else if (gain_pga_dB >= 1 && gain_pga_dB <=10 )
      {
         oactrl_c_tune = 2;
      }
      else if (gain_pga_dB >= 11 && gain_pga_dB <=21 )
      {
         oactrl_c_tune = 2;
      }
   }

   if (mode_set == BYPASSED)
   {
      oactrl_c_tune = 2; // program 0dB for bypass-mode
   }

   // PUP of OpAmp
    oactrl_oa_enable_shift = oactrl_oa_enable & VRX5AFE_DSL_FCSI_PGA4_PGA35ENPGA__MSK;
    oactrl_oa_enable_shift = oactrl_oa_enable_shift << VRX5AFE_DSL_FCSI_PGA4_PGA35ENPGA__POS;

   // comp. cap. tuned
    oactrl_c_tune_shift = oactrl_c_tune & VRX5AFE_DSL_FCSI_PGA4_PGA35COMPSEL__MSK;
    oactrl_c_tune_shift = oactrl_c_tune_shift << VRX5AFE_DSL_FCSI_PGA4_PGA35COMPSEL__POS;


    // Register PGA4
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA4),
         VRX5AFE_DSL_FCSI_PGA4_PGA35ENPGA__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA4_PGA35COMPSEL__ISMSK,
         oactrl_oa_enable_shift | \
         oactrl_c_tune_shift);


   //------
   // gain
   //------
   if (mode_set == BYPASSED)
   {
      gain_pga_dB = 0;
   }

   if (gain_pga_dB == -9)
   {
      gain_bin = 0;
   }
   else if ( (gain_pga_dB >= -6) && (gain_pga_dB <= 21) )
   {
      gain_bin = gain_pga_dB + 7; // -6dB ... 21dB, 1dB step
   }
   else if  (gain_pga_dB > 21)
   {
      gain_bin = 28; // 21dB
   }
   else
   {
      gain_bin = 0;               // set to -9dB
   }

   gain_bin_shift = gain_bin & VRX5AFE_DSL_FCSI_PGA3_PGA35GAIN__MSK;         // guarantee data bit-range
   gain_bin_shift = gain_bin_shift << VRX5AFE_DSL_FCSI_PGA3_PGA35GAIN__POS;  // shift to correct position for "write"

   Rohm = (int16_t)Rsa;           // Q0

#ifdef TEST_FLOAT_VRX518
   //------
   // Clp1
   //------
   Rpga1 = pow(10.0, gain_pga_dB/20.0) * Rsa;

   Clp1 = 1.0 / (2.0 * 3.1415 * fc_pga_lp_kHz * 1e3 * Rpga1);             // in [F]

   Clp1_pF_weighted = Clp1*1e12 / weight_clp1 + 0.5;                      // 0.5*LSB due to round
   Clp1_bin_Float = (uint16_t)Clp1_pF_weighted;                           // round(x) = truncate(x+0.5)

   if (Clp1_bin_Float > VRX5AFE_DSL_FCSI_PGA3_PGA35LP__MSK)               // saturate to max. value
   {
      Clp1_bin_Float = VRX5AFE_DSL_FCSI_PGA3_PGA35LP__MSK;
   }

   // Needed for Debugging
   Clp1e12 = Clp1*1e12;
   V_1e9By2piFcKhzByR = (1e9) / (2.0 * 3.1415* fc_pga_lp_kHz * Rohm);
#endif
   //int16_t Rohm             // Q0
   //int16_t WeightCap        // Q13
   //int16_t GaindB        // Q8
   //uint32_t Value1e9By2piFcKhz // Q20
   //uint8 ChooseFilter          // Q0 - 0=PGA, 1=POFI, 2=PREFI

   WeightCap = 8192;             // weight_clp1 = 1.0 in Q13 = 1 *2^13 = 8192
   GaindB = gain_pga_dB << 8;    // Q8
   // around 12 bits for integer representation of Value1e9By2piFcKhz: 32-12=20 bits for fraction
   // Value1e9By2piFcKhz         // Q20
   // Value1e9By2piFcKhz = (int32_t)((double)(1e9 * (1 << 20)) / (2.0 * 3.1415* fc_pga_lp_kHz * 1000 * 512)); // Scale by value 512 (9 bits)

   Clp1_bin  = Calc_PgaClp1PrefiClp2PocoC2(Rohm, WeightCap, GaindB, Value1e9By2piFcKhz, ChooseFilter);

   if (Clp1_bin > VRX5AFE_DSL_FCSI_PGA3_PGA35LP__MSK)               // saturate to max. value
   {
      Clp1_bin = VRX5AFE_DSL_FCSI_PGA3_PGA35LP__MSK;
   }

#ifdef TEST_FLOAT_VRX518
   debug_printf("\nPgaGain=%3d fc_pga_lp_kHz=%5d V1e9By2piFcKhz=%8d Rohm=%4d ", gain_pga_dB, fc_pga_lp_kHz, Value1e9By2piFcKhz, Rohm);
   debug_printf("\n >>Clp1*e12 =%f Clp1_Wt=%f ######Clp1_bin=%4d %4d Diff=%4d", Clp1e12, Clp1_pF_weighted, Clp1_bin_Float, Clp1_bin, Clp1_bin_Float - Clp1_bin);
#endif

   Clp1_bin_shift = Clp1_bin & VRX5AFE_DSL_FCSI_PGA3_PGA35LP__MSK;        // guarante's data bit-range
   Clp1_bin_shift = Clp1_bin_shift << VRX5AFE_DSL_FCSI_PGA3_PGA35LP__POS; // shift to correct position for "write"

   // Register PGA3
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA3),
         VRX5AFE_DSL_FCSI_PGA3_PGA35GAIN__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA3_PGA35LP__ISMSK,
         gain_bin_shift | \
         Clp1_bin_shift);
}


void vrx5afe_dsl_prefi_ace_set(uint16_t ace_setting)
{
   // double R21=100.0;         // not modeled so far
   // double R22=200.0;
   // double R23=400.0;

   uint16_t ctune_ace_bin,   ctune_ace1_shift, ctune_ace2_shift, ctune_ace3_shift;

   int32_t temp1, RC_nomxK, ConstValue;

   uint16_t RC_count = 0;
   uint16_t DSL_version;
   DSL_version = vrx5afe_get_dsl_version_init(0); //copy information from static area

   gt_vrx5afe_dsl_config.us_dsl_prefi_ace_ace_setting        = ace_setting;

#ifdef TEST_FLOAT_VRX518
   uint16_t ctune_ace_bin_Float;
   float temp2;
#endif


   // get RC-counter value from global variable
   RC_count = vrx5afe_get_rc_count();


#ifdef TEST_FLOAT_VRX518
   //ctune_ace_bin = (uint16_t) ( (14.8 * RC_nom / RC_count + 9.3) + 0.5 );    // round(x) = truncate(x+0.5)
   temp2 = (14.8 * RC_nom / RC_count + 9.3) ;
   if (temp2 < 0)
   {
      ctune_ace_bin_Float = 0;
   }
   else
   {
      ctune_ace_bin_Float = (uint16_t)(temp2 + 0.5);                          // round(x) = truncate(x+0.5)
   }
#endif

    // float to fixed point of ctune_poco_bin
   RC_nomxK   = 0x9389999;      // Q15  // = 14.8 * RC_nom = 14.8*319 in Q15 = 0x9389999
   ConstValue = 0x4A666;        // Q15  // =             9.3     in Q15 = 0x4A666 (=9.3 *(1<<15))

   temp1 = RC_nomxK / RC_count; // Result in Q15 format
   temp1 = temp1 + ConstValue;  // Result in Q15 format

   if ((temp1 >> 15) < 0)
   {
      ctune_ace_bin = 0;
   }
   else
   {
      ctune_ace_bin = (uint16_t)((temp1 + (1 << 14)) >> 15);  // Convert to integer; round(x) = truncate(x+0.5)
   }

   ctune_ace_bin = ctune_ace_bin | 0x10;

   if (ctune_ace_bin > VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__MSK)// saturate to max. value
   {
            ctune_ace_bin = VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__MSK;
      }

   ctune_ace1_shift = ctune_ace_bin & VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__MSK;      // guarante's data bit-range
   ctune_ace1_shift = ctune_ace1_shift << VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__POS;  // shift to correct position for "write"

   ctune_ace2_shift = ctune_ace_bin & VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__MSK;      // guarante's data bit-range
   ctune_ace2_shift = ctune_ace2_shift << VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__POS;  // shift to correct position for "write"

   ctune_ace3_shift = ctune_ace_bin & VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__MSK;      // guarante's data bit-range
   ctune_ace3_shift = ctune_ace3_shift << VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__POS;  // shift to correct position for "write"


   if (ace_setting == ACE_OFF)
   {
      // ACE always OFF in ADSL-mode (per default OFF)
      // C21(30pF) | C22(15pF) | C23(7.5pF)
      //     0           0           0
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA6),
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__ISMSK,
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1_OFF__SVAL | \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2_OFF__SVAL | \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3_OFF__SVAL);
        if (DSL_version == 0)
        {   //A11
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5), \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__ISMSK, \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT_2__SVAL);
        }
#if 0 //Not required for A21
        else
        {   //A21
            // disable workaround for A11 ACE stability, to be checked if it can be removed
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5), \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__ISMSK, \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT_2__SVAL);
        }
#endif
   }
    else
    {
        if (DSL_version == 0)
        {   //A11
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5), \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__ISMSK, \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT_0__SVAL);
        }
#if 0  //Not required for A21
        else
        {   //A21
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5), \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__ISMSK, \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT_0__SVAL);
        }
#endif

   if (ace_setting == ACE_52p5F)
   {
      // C21(30pF) | C22(15pF) | C23(7.5pF)
      //     1           1           1
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA6),
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__ISMSK,
            ctune_ace1_shift | \
            ctune_ace2_shift | \
            ctune_ace3_shift);
   }

   if (ace_setting == ACE_30pF)
   {
      // C21(30pF) | C22(15pF) | C23(7.5pF)
      //     1           0           0
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA6),
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__ISMSK,
            ctune_ace3_shift | \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2_OFF__SVAL | \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1_OFF__SVAL);
   }

   if (ace_setting == ACE_15pF)
   {
      // C21(30pF) | C22(15pF) | C23(7.5pF)
      //     0           1           0
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA6),
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__ISMSK,
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1_OFF__SVAL | \
            ctune_ace2_shift | \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3_OFF__SVAL);
   }

   if (ace_setting == ACE_7p5F)
   {
      // C21(30pF) | C22(15pF) | C23(7.5pF)
      //     0           0           1
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA6),
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC1__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2__ISMSK & \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3__ISMSK,
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC3_OFF__SVAL | \
            VRX5AFE_DSL_FCSI_PGA6_PREFI35ACERC2_OFF__SVAL | \
            ctune_ace1_shift);
   }
    }

#ifdef TEST_FLOAT_VRX518
   debug_printf("\nACE with modes: \n");
   debug_printf("\nRCcount=%5d::ace_bin =%4d %4d Diff=%2d", RC_count, ctune_ace_bin_Float, ctune_ace_bin-32, ctune_ace_bin_Float - ctune_ace_bin+32);
#endif
}


void vrx5afe_dsl_prefi_gainfc_set(uint16_t mode_set, int16_t gain_prefi_dB, int32_t fc_prefi_lp_kHz)
{
   //--------------------------------------------------------------------------
   // e.g. for passing parameters
   // vrx5afe_dsl_prefi_gainfc_set(VDSL2_CPE, -8, 50000);
   // mode, gain [dB], fc_lp [kHz]
   //
   // ACE OFF -> vrx5afe_dsl_prefi_ace_set(mode_set)
   //
   // OA-CTRL mode dependent
   //--------------------------------------------------------------------------
   gt_vrx5afe_dsl_config.us_dsl_prefi_gainfc_mode_set        = mode_set;
   gt_vrx5afe_dsl_config.s_dsl_prefi_gainfc_gain_prefi_dB    = gain_prefi_dB;
   gt_vrx5afe_dsl_config.l_dsl_prefi_gainfc_fc_prefi_lp_kHz  = fc_prefi_lp_kHz;

//------------------------------------------------------------------------------
// //
// // uint16_t x_val_disc; // return value (bits)
// //
//    // calculates binary weighted oactrl_pga_bin with 7bit
//    uint16_t dec2binweighted(uint8_t n_bits, double x_val, double weight)
//    {
//       // x_val ... to be discretized
//       // y ... discretized value
//       // n_bits ... number of bits used to discretize
//       // weight ... weighting of bits
//       //
//       // bit weighted discretization
//
//
//       double x_pow_weighted;
//       double c;
//
//       int bit_powered;
//
//       uint16_t bits; // returned discretized x_val
//
//       debug_printf("\n");
//       debug_printf("b=%4.15f\n", weight);
//
//       bits = 0;
//       for (int ii=n_bits; ii>0; ii--)
//       {
//          bit_powered = 2>>(ii-1); // pow
//
//          x_pow_weighted = bit_powered * weight;
//
//          if ((x_val-x_pow_weighted)>=0)
//          {
//                debug_printf("Bit: %d\n", ii-1);
//                bits |= 1 << (ii-1);
//
//                x_val = ( (1e18*x_val) - (1e18*x_pow_weighted) ) / 1e18;
//          }
//
//       }
//       debug_printf("\n");
//       debug_printf("%4.12f,  %d\n", x_val, bits);
//
//       return bits;
//    }
   //------------------------------------------------------------------------------


#ifdef TEST_FLOAT_VRX518
   double weight_clp2 = 0.1;   // 'LSB' in [pF]
   double Rpga2, Clp2, Clp2_pF_weighted;
   double Clp2e12, V_1e9By2piFcKhzByR;
   uint16_t Clp2_bin_Float;
   //uint8_t n_bits = 7;
#endif

   int16_t R4 = 100;          // for ACE-HP
   int16_t R1=0;           // in ACE for prefi-gain
   int16_t R3=0;                // used in bypass-mode

   int16_t gain_bin=0, gain_bin_shift, gain_prefi_dB_int = 0;
   //int16_t Rpga2;
   int16_t oactrl_oa_enable, oactrl_c_tune=0, oactrl_i_tune=0;
   int16_t oactrl_oa_enable_shift, oactrl_c_tune_shift, oactrl_i_tune_shift;

   uint16_t Clp2_bin=0, Clp2_bin_shift;
   uint16_t error_out=0;

    // Fixed point function related variables
   int16_t Rohm =0;     //Q0
   int16_t WeightCap;         // Q13
   int16_t GaindB;            // Q8
   uint8_t ChooseFilter = 2;   // Q0 - 0=PGA, 1=POFI, 2=PREFI
   int32_t Const1e9By2PiX;
   int32_t Value1e9By2piFcKhz; // Q20
    uint16_t RC_count;
    uint16_t DSL_version;
    DSL_version = vrx5afe_get_dsl_version_init(0); //copy information from static area

    //((1e9 / (2.0 * 3.1415 * 512 * 256)) * 2 ^ 20) = 0x4BE4AAA9  Q20
    // Here scale factor 256(8bits) is used later, 512(9 bits) is  used later
    Const1e9By2PiX     = ((int32_t) NUM_Q20_1E9BY2PIX512X256) / fc_prefi_lp_kHz;  // Q20/Q0 = Q0
    Value1e9By2piFcKhz = (int32_t) (Const1e9By2PiX << SCALE_1E9BY2PIX512X256BY8); // Q20  Scale back again with scale factor 256(8bits). 9 bit scale factor will be taken later


   // get RC-counter value from global variable
   RC_count = vrx5afe_get_rc_count();


   //---------
   // OA-CTRL
   //---------

   // PUP of OpAmp
   oactrl_oa_enable = 1; // always 1

   // comp. cap. tuning
   if ((mode_set == VDSL35b_CPE) | (mode_set == VDSL17_CPE))
   {
      oactrl_c_tune = 6; //changed on 13.12.2016
//    if (gain_prefi_dB >= -16 && gain_prefi_dB <=-12 )
//    {
//       oactrl_c_tune = 2;
//    }
//    else if (gain_prefi_dB >= -11 && gain_prefi_dB <=-6 )
//    {
//       oactrl_c_tune = 1;
//    }
//    else if (gain_prefi_dB >= -5 && gain_prefi_dB <=-0 )
//    {
//       oactrl_c_tune = 0;
//    }
   }

   if (mode_set == VDSL8_CPE)
   {
      oactrl_c_tune = 6; //changed on 13.12.2016
//    if (gain_prefi_dB >= -16 && gain_prefi_dB <=-12 )
//    {
//       oactrl_c_tune = 3;
//    }
//    else if (gain_prefi_dB >= -11 && gain_prefi_dB <=-6 )
//    {
//       oactrl_c_tune = 2;
//    }
//    else if (gain_prefi_dB >= -5 && gain_prefi_dB <=-0 )
//    {
//       oactrl_c_tune = 1;
//    }
   }

   if ((mode_set == ADSL2_CPE) || (mode_set == ADSL2_CPE_LL))
   {
      oactrl_c_tune = 7; //changed on 13.12.2016
   }

   if (mode_set == BYPASSED)
   {
      oactrl_c_tune = 6; //changed on 13.12.2016
   }

   // current tuning
   if ((mode_set == ADSL2_CPE) || (mode_set == ADSL2_CPE_LL))
   {
      oactrl_i_tune = 0;
   }
// else           //changed on 14.12.2016
// {
//    oactrl_i_tune = 2;
// }

   // PUP of OpAmp
    oactrl_oa_enable_shift = oactrl_oa_enable & VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__MSK;
    oactrl_oa_enable_shift = oactrl_oa_enable_shift << VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__POS;

    if (DSL_version == 0)
    {   //A11
      // comp. cap. tuned
        oactrl_c_tune_shift = oactrl_c_tune & VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ENCAP__MSK;
        oactrl_c_tune_shift = oactrl_c_tune_shift << VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ENCAP__POS;
        // current tuned
        oactrl_i_tune_shift = oactrl_i_tune & VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__MSK;
        oactrl_i_tune_shift = oactrl_i_tune_shift << VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__POS;
    }
    else
    {  //A21
       // comp. cap. tuned
       oactrl_c_tune_shift = oactrl_c_tune & VRX5AFE_DSL_FCSI_PGA5_PREFI35ENCAP__MSK;
       oactrl_c_tune_shift = oactrl_c_tune_shift << VRX5AFE_DSL_FCSI_PGA5_PREFI35ENCAP__POS;

      // current tuned
       oactrl_i_tune_shift = oactrl_i_tune & VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__MSK;
       oactrl_i_tune_shift = oactrl_i_tune_shift << VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__POS;
    }

   //------
   // gain
   //------

   if (mode_set == BYPASSED)
   {
      // In Prefi35 BYPASSED mode, Gain is fixed @-8dB
      gain_prefi_dB_int = -8;
      gain_bin = gain_prefi_dB_int + 16;

      //R1 = 1e18;
      R3 = 2000;

        if (DSL_version == 0)
        {   //A11
            // Register PGA5
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_1__SVAL);
        }
        else
        {   //A21
            // Register PGA5
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_1__SVAL);
        }
   }

   if ( (mode_set == VDSL35b_CPE) | (mode_set == VDSL17_CPE) | (mode_set == VDSL8_CPE) ) // saturation added 30.9.2016
   {
      if (gain_prefi_dB >= -16 && gain_prefi_dB <= 0)
      {
      gain_prefi_dB_int = gain_prefi_dB;
      gain_bin = gain_prefi_dB + 16; // -16dB ... 0dB, 1dB step
      error_out = 0;
      }
      else if (gain_prefi_dB < -16)
      {
         gain_prefi_dB_int = -16;
         gain_bin = 0;
         error_out = 1;
      }
      else if (gain_prefi_dB > 0)
      {
         gain_prefi_dB_int = 0;
         gain_bin = 16;
         error_out = 1;
      }
      R1 = 1900;

        if (DSL_version == 0)
        {   //A11
            // Register PGA5
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ACESRES__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_0__SVAL | \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ACESRES_R1900__SVAL);
        }
        else
        {   //A21
            // Register PGA5
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ACESRES__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_0__SVAL | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ACESRES_R1900__SVAL);
        }
   }

   if (mode_set == ADSL2_CPE)
   {
      if (gain_prefi_dB >= -8 && gain_prefi_dB <= 12) // saturation added 30.9.2016
      {
         gain_prefi_dB_int = gain_prefi_dB;
         gain_bin = gain_prefi_dB + 24; // -8dB ... 0dB ... 12dB, 1dB step
         error_out =0;
      }
      else if (gain_prefi_dB < -8)
      {
         gain_prefi_dB_int = -8;
         gain_bin = 16;
         error_out = 1;
      }
      else if (gain_prefi_dB > 12)
      {
         gain_prefi_dB_int = 12;
         gain_bin = 36;
         error_out = 1;
      }
      R1 = 4900;

        if (DSL_version == 0)
        {   //A11
            // Register PGA5
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK & \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ACESRES__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_0__SVAL | \
                    VRX5AFE_DSLA11_FCSI_PGA5_PREFI35ACESRES_R4900__SVAL);
        }
        else
        {   //A21
            // Register PGA5
            VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ACESRES__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_0__SVAL | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ACESRES_R4900__SVAL);
        }
   }

   //XDSLRTFW-3674 New_LLAPI_changes (start)
   if (mode_set == ADSL2_CPE_LL)
   {
      if (gain_prefi_dB >= -8 && gain_prefi_dB <= 18)
      {
         gain_prefi_dB_int = gain_prefi_dB;
         gain_bin = gain_prefi_dB + 18; // -8dB ... 0dB ... 18dB, 1dB step
         error_out =0;
      }
      else if (gain_prefi_dB < -8)
      {
         gain_prefi_dB_int = -8;
         gain_bin = 10;
         error_out = 1;
      }else if (gain_prefi_dB > 18)
      {
         gain_prefi_dB_int = 18;
         gain_bin = 36;
         error_out = 1;
      }
      R1 = 2400;

                // Register PGA5
                VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA5),
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35PUP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ENCAP__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35IOUT__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS__ISMSK & \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ACESRES__ISMSK,
                    oactrl_oa_enable_shift | \
                    oactrl_c_tune_shift | \
                    oactrl_i_tune_shift | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35BYPASS_0__SVAL | \
                    VRX5AFE_DSL_FCSI_PGA5_PREFI35ACESRES_R2400__SVAL);
   }
   //XDSLRTFW-3674 New_LLAPI_changes (end)
   gain_bin_shift = gain_bin & VRX5AFE_DSL_FCSI_PGA7_PREFI35FBRES__MSK;         // guarantee data bit-range
   gain_bin_shift = gain_bin_shift << VRX5AFE_DSL_FCSI_PGA7_PREFI35FBRES__POS;  // shift to correct position for "write"

   //debug_printf("%s %i %s %i %s %i %s %i\n", "MODE = ", mode_set, ", gain_prefi_dB = ", gain_prefi_dB, ", gain_bin =  ",  gain_bin,  ", gain_bin_shift = ",  gain_bin_shift);
   ////debug_printf("%s 0x%x\n", "gain_bin_shift = ", gain_bin_shift);


   //------
   // Clp2
   //------

   if (mode_set == BYPASSED)
   {
      #ifdef TEST_FLOAT_VRX518
      Rpga2 = pow(10.0, gain_prefi_dB_int/20.0) * R3;
   #endif
      Rohm = R3;
   }

   if (mode_set != BYPASSED)
   {
   #ifdef TEST_FLOAT_VRX518
      Rpga2 = pow(10.0, gain_prefi_dB_int/20.0) * (R1 + R4);       // in bypass R3=2kOhm (the same as 1900+100)
      #endif
      Rohm = (uint16_t) (R1 + R4);
   }

 #ifdef TEST_FLOAT_VRX518
   Clp2 = 1.0 / (2.0 * 3.1415 * fc_prefi_lp_kHz * 1e3 * Rpga2); // in [F]
   //Clp2_bin = dec2binweighted(n_bits, Clp2, weight_clp2);     // 7bit binary weighted C-array

   Clp2 = Clp2 * RC_nom / RC_count;                             // RC-tuning

   Clp2_pF_weighted = Clp2*1e12 / weight_clp2 + 0.5;            // 0.5*LSB due to round
   Clp2_bin_Float   = (uint16_t)Clp2_pF_weighted;               // round(x) = truncate(x+0.5)

   if (Clp2_bin_Float > VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__MSK)// saturate to max. value
   {
      Clp2_bin_Float = VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__MSK;
   }
   //debug_printf("---> Clp2_pF_weighted = %f,    Clp2_bin_Float = %d \n",Clp2_pF_weighted, Clp2_bin_Float);

   //May be needed for Debugging
   Clp2e12 = Clp2*1e12;
   V_1e9By2piFcKhzByR = (1e9) / (2.0 * 3.1415 * fc_prefi_lp_kHz * Rohm); // Rpga2???
 #endif
   //int16_t Rohm             // Q0
   //int16_t WeightCap        // Q13
   //int16_t GaindB        // Q8
   //uint32_t Value1e9By2piFcKhz // Q20
   //uint8 ChooseFilter          // 2 here
   //Rohm = ; done above
   WeightCap = 819;           // Q13  0.1*2^13 = 819  double weight_clp2 = 0.1;  // 'LSB'
   GaindB = gain_prefi_dB_int << 8;  // Q8

   //Value1e9By2piFcKhz          // Q20
   //Value1e9By2piFcKhz = (int32_t)((double)(1e9 * (1 << 20)) / (2.0 * 3.1415* fc_pga_lp_MHz * 1000 * 512)); //Scale by value 512 (9 bits)
   //ChooseFilter = 2;
   Clp2_bin = Calc_PgaClp1PrefiClp2PocoC2(Rohm, WeightCap, GaindB, Value1e9By2piFcKhz, ChooseFilter);

   if (Clp2_bin > VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__MSK)// saturate to max. value
   {
      Clp2_bin = VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__MSK;
   }

#ifdef TEST_FLOAT_VRX518
   debug_printf("\nPrefiGain=%3d fc_prefi_lp_kHz=%5d V1e9By2piFcKhz=%8d Rohm=%4d ", gain_prefi_dB, fc_prefi_lp_kHz, Value1e9By2piFcKhz, Rohm);
   debug_printf("\n >>Clp2*e12 =%f Clp2w=%f   ######Clp2_bin=%4d %4d Diff=%4d ", Clp2e12, Clp2_pF_weighted, Clp2_bin_Float, Clp2_bin, Clp2_bin_Float - Clp2_bin);
#endif

   Clp2_bin_shift = Clp2_bin & VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__MSK;        // guarante's data bit-range
   Clp2_bin_shift = Clp2_bin_shift << VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__POS; // shift to correct position for "write"

// debug_printf("%s %i %s %f %s %f %s %f\n", "MODE = ", mode_set, ", fc_prefi_lp_MHz [MHz] = ", fc_prefi_lp_MHz, ", Rpga2 = ", Rpga2, ", Clp2 [pF] = ", Clp2*1e12);
// debug_printf("%s %i %s %f %s %i %s %i\n", "MODE = ", mode_set, ", Clp2_weighted [pF] = ", Clp2_pF_weighted, ", Clp2_bin = ", Clp2_bin, ", Clp2_bin_shifted = ", Clp2_bin_shift);


   // Register PGA7
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA7),
         VRX5AFE_DSL_FCSI_PGA7_PREFI35FBRES__ISMSK & \
         VRX5AFE_DSL_FCSI_PGA7_PREFI35FBCAP__ISMSK,
         gain_bin_shift | \
         Clp2_bin_shift);
   //debug_printf("%s 0x%x\n", "gain_bin_shift = ", gain_bin_shift);
   //debug_printf("%s 0x%x\n", "Clp2_bin_shift = ", Clp2_bin_shift);
}


void vrx5afe_dsl_adc_conf(uint16_t mode_set)
{
   uint16_t DSL_version = 0;
   DSL_version = vrx5afe_get_dsl_version_init(0);
   if ( (mode_set == VDSL35b_CPE) | (mode_set == VDSL17_CPE) | (mode_set == BYPASSED) )
   {
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC0), \
         VRX5AFE_DSL_FCSI_SDADC0_ENADSL__ISMSK & \
      VRX5AFE_DSL_FCSI_SDADC0_ENADSLPATH__ISMSK, \
      VRX5AFE_DSL_FCSI_SDADC0_ENADSL_0__SVAL | \
      VRX5AFE_DSL_FCSI_SDADC0_ENADSLPATH_OFF__SVAL);

      //-------------------------------------------------------
      // Type 1 - improve performance VDSL
#if 0  //Vdsl specific under VDSL mode
      VRX5AFE_FCSI_WRITE(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC2), 0x0008);
#endif

      VRX5AFE_FCSI_WRITE(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC3), 0xE30F); // max IBIAS (former C72C)

      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC1), \
      VRX5AFE_DSL_FCSI_SDADC1_ENFORCEEXT__ISMSK, \
      VRX5AFE_DSL_FCSI_SDADC1_ENFORCEEXT_ON__SVAL);
      //-------------------------------------------------------
   }

   if ((mode_set == ADSL2_CPE)  | (mode_set == VDSL8_CPE))
   {
        if (DSL_version == 0)
        {
         VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC0), \
            VRX5AFE_DSL_FCSI_SDADC0_ENADSL__ISMSK & \
         VRX5AFE_DSL_FCSI_SDADC0_ENCLA__ISMSK & \
         VRX5AFE_DSL_FCSI_SDADC0_ENADSLPATH__ISMSK, \
         VRX5AFE_DSL_FCSI_SDADC0_ENADSL_1__SVAL | \
         VRX5AFE_DSL_FCSI_SDADC0_ENCLA_ON__SVAL | \
         VRX5AFE_DSL_FCSI_SDADC0_ENADSLPATH_ON__SVAL );
        }
#if 1  //The following is not taken in A11, A11 tested without this working fine.
        if (DSL_version != 0)
        {

         VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC0), \
               VRX5AFE_DSL_FCSI_SDADC0_ENADSL__ISMSK & \
               VRX5AFE_DSL_FCSI_SDADC0_ENCLA__ISMSK & \
               VRX5AFE_DSL_FCSI_SDADC0_ENADSLPATH__ISMSK, \
               VRX5AFE_DSL_FCSI_SDADC0_ENADSL_1__SVAL | \
               VRX5AFE_DSL_FCSI_SDADC0_ENCLA_OFF__SVAL | \
               VRX5AFE_DSL_FCSI_SDADC0_ENADSLPATH_ON__SVAL );


         //added on 11th January 2017
         VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_SDADC1), \
         VRX5AFE_DSL_FCSI_SDADC1_ENFORCEEXT__ISMSK, \
         VRX5AFE_DSL_FCSI_SDADC1_ENFORCEEXT_OFF__SVAL);
        }
#endif
   }
}


void vrx5afe_dsl_adc_reset(uint16_t mode_set)
{
   if (mode_set == BYPASSED)
   {

   }

   if ( (mode_set == VDSL35b_CPE) | (mode_set == VDSL17_CPE) | (mode_set == VDSL8_CPE) )
   {

   }

   if (mode_set == ADSL2_CPE)
   {

   }
}

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME: AFED_ConfigAfeRxPathForced()
**
** DESCRIPTION:   Debug function to overwrite the default Rx config settings in the firmware.
**
** PARAMETERS:    NONE
**
**
** RETURN VALUE:  NONE
**
** NOTES:Reference:
**
**
**
** =============================================================================
*/

void AFED_ConfigAfeRxPathForced(void)
{
   //VRX518_BringUp
   uint16 HybIndex =0;
   //Add the reg setting to forced values
   // LA - line adaptation (Hybrid)


//required setting for HS (17m-setting) and Showtime for 50m loop in VDSL35B derived from Franks settings.
//It is one common setting for the moment
//
// PGA0=  0253H
// PGA1= 043FH
// PGA2= 0133H Done here
//gain_pga_dB  -9
//gain_prefi_dB -5
//PGA6=0;ACE=0ff   Done here

   // Rsa->400Ohm, Rpa->inf
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA0), 0x0000, gusa_SpecialHybSetting[HybIndex+0]);

   // Rsb->inf, Rpb->inf, Cpb->inf
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA1), 0x0000, gusa_SpecialHybSetting[HybIndex+1]);

   // Rsc->inf, Rpc->inf, Cpc->inf
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA2), 0x0000, gusa_SpecialHybSetting[HybIndex+2]);

//The ACE isn't set by the API yet. This means that the default register setting is used (i.e. ACE=0ff).
// PGA6=0;ACE=0ff
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA6),0x0000,0x0000);
#if 0
   //VRX518_DEBUG_CHECK Start
   if (gus_AFE_RxMode == VDSL35b_CPE)
   {
      // Rsa->400Ohm, Rpa->inf
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA0), 0x0000, 0x0210);

      // Rsb->inf, Rpb->inf, Cpb->inf
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA1), 0x0000, 0x0000);

      // Rsc->inf, Rpc->inf, Cpc->inf
      VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA2), 0x0000, 0x0000);

   }

#endif

}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME: AFED_ConfigAfeTxPathForced()
**
** DESCRIPTION:   Debug function to overwrite the default Tx AFE config settings in the firmware.
**
** PARAMETERS:    NONE
**
**
** RETURN VALUE:  NONE
**
** NOTES:Reference:
**
**
**
** =============================================================================
*/
void AFED_ConfigAfeTxPathForced(void)
{
   //VRX518_BringUp

   //VRX518_BringUp
   //gain_pofi_dB 1
   //POCO-setting:
   //POCO: bit12..8  =1BH  Done here
   //POCO: bit1:0    = 3H  Done here
      // Register POCO0
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_POCO0),
         VRX5AFE_DSL_FCSI_POCO0_FC2VDSL__ISMSK & \
         VRX5AFE_DSL_FCSI_POCO0_FC1VDSL__ISMSK,
         VRX5AFE_DSL_FCSI_POCO0_FC2VDSL_6P2__SVAL | \
         VRX5AFE_DSL_FCSI_POCO0_FC1VDSL_10P__SVAL);
}

#if 0
void AFED_GetProfileSelected(uint16 us_ProfileSelected)
{
   switch (us_ProfileSelected)
   {

   case CNFG_V2_PROFILE_35B_MASK:
     gus_AFE_RxMode = VDSL_35;
     gus_AFE_TxMode = VDSL_35;
   break;

   case CNFG_V2_PROFILE_30A_MASK:
      gus_AFE_RxMode = VDSL_35; //VRX518AFE_OPEN
      gus_AFE_TxMode = VDSL_35;//VRX518AFE_OPEN
      break;

   case CNFG_V2_PROFILE_8A_MASK:
   case CNFG_V2_PROFILE_8B_MASK:
   case CNFG_V2_PROFILE_8C_MASK:
   case CNFG_V2_PROFILE_8D_MASK:
      gus_AFE_RxMode = VDSL_8;
      gus_AFE_TxMode = VDSL_8;
    //  gs_ADCGain = (int16)0xff34;            // ADC gain for 8Mhz profiles is -0.8dB (minus since the gain gets subtracted)
      break;

   case CNFG_V2_PROFILE_12A_MASK:
   case CNFG_V2_PROFILE_12B_MASK:
      gus_AFE_RxMode = VDSL_17;
      gus_AFE_TxMode = VDSL_17;
      break;

   case CNFG_V2_PROFILE_17A_MASK:
     gus_AFE_RxMode = VDSL_17;
     gus_AFE_TxMode = VDSL_17;
      break;

   default:
      break;
   }
}
#endif
////////////////////////////////////////////////////////////////////////////////////////////////
//Function Name:
//Description: This function Initialiseds the DFI register for starting VDSL mode Handshake
// Prototype:
//
//
// Input Arguments:
//             None
//  Output Arguments:
//             None
// Return:
//                None
// Global Variables Used:
//
// Notes
//
////////////////////////////////////////////////////////////////////////////////////////////////

void VRX518_DfiConfig_Common(uint16 AfeTxMode)
{

   // this check is for VRx518 bring-up only
   if (gus_SkipAfeFlow & 0x0001)
   {
      return;
   }

   //Address   Register Name           Short Description          35b_Mode 30a_Mode 17a_Mode 8a_Mode

   //0x30C780  V_LBK_35B               Strymon-VFDF Loopback Control    0x00000060  0x00000000  0x00000000  0x00000100
    if (AfeTxMode == VDSL35b_CPE) //35 MHZ
      WriteCoreReg((uint32)(V_LBK_35B_ADDR), 0x00000060);
   else if (AfeTxMode == VDSL8_CPE)//8 MHz
      WriteCoreReg((uint32)(V_LBK_35B_ADDR), 0x00000100);
   else if (AfeTxMode == ADSL2_CPE)//8 MHz
      WriteCoreReg((uint32)(V_LBK_35B_ADDR), 0x00000180);
   else
      WriteCoreReg((uint32)(V_LBK_35B_ADDR), 0x00000000);//17MHz 30MHz 12 MHz
    //Reset DFI BLOCK should be before AFE init. The reason is after AFE Config we are not suppose reset this
    //only RX.FIFO_RSTN should be done here
   //0x30C78C  V_DFI_BLOCK_RSTN_35B       DFI Block Reset Register      0x00000000  0x00000000  0x00000000  0x00000000
   //0x30C78C  V_DFI_BLOCK_RSTN_35B       DFI Block Reset Register      0x00070007  0x00070007  0x00070007  0x00070007

    //The following RX/TX FIFO reset is happening eventhough there is reconfiguration scenarios also.
    if(AfeTxMode != ADSL2_CPE)
    {
        WriteCoreReg((uint32)(V_DFI_BLK_RSTN_35B_ADDR), 0x00050005);
        WriteCoreReg((uint32)(V_DFI_BLK_RSTN_35B_ADDR), 0x00070007);
    }
}

void VRX518_DfiConfig_TxPath(uint16 AfeTxMode)
{
   // this check is for VRx518 bring-up only
   if (gus_SkipAfeFlow & 0x0002)
   {
      return;
   }

   //Address   Register Name           Short Description          35b_Mode 30a_Mode 17a_Mode 8a_Mode
   //0x30B000  V_TX_WDF_INT1_CTRL_35B     Tx WDF INT1 control register  0x00000002  0x00000000  0x00000000  0x00000000
   //0x30B004  V_TX_WDF_INT1_COEF_35B     Tx WDF INT1 coefficient reg      0xcb78390f  0x00000000  0x00000000  0x00000000
   //0x30B000  V_TX_WDF_INT1_CTRL_35B     Tx WDF INT1 control register  0x00000009
   if (AfeTxMode == VDSL35b_CPE)
   {
      WriteCoreReg((uint32)(V_TX_WDF_INT1_CTRL_35B_ADDR), 0x00000002);
      WriteCoreReg((uint32)(V_TX_WDF_INT1_COEF_35B_ADDR), 0xcb78390f);
      //SoC team done SRC performance test including the WDF for profile 35B and found that we do not need the additional scaling
      //So the configuration V_TX_WDF_INT1_CTRL_35B(0x30B000)=0x00000001 instead of 0x00000009
      WriteCoreReg((uint32)(V_TX_WDF_INT1_CTRL_35B_ADDR), 0x00000001);//
   }
   else
   {
      WriteCoreReg((uint32)(V_TX_WDF_INT1_CTRL_35B_ADDR), 0x00000000);
      WriteCoreReg((uint32)(V_TX_WDF_INT1_COEF_35B_ADDR), 0x00000000);
   }
   //0x30B400  DFI_TX_WDF_INT2_CTRL_35B   Tx WDF INT2 control register  0x00000002  0x00000002  0x00000002  0x00000002
   //0x30B404  DFI_TX_WDF_INT2_COEF_35B   Tx WDF INT2 coefficient reg      0x00ae4711  0x00ae4711  0x00ae4711  0x00ae4711
   //0x30B400  DFI_TX_WDF_INT2_CTRL_35B   Tx WDF INT2 control register  0x00000001  0x00000001  0x00000001  0x00000001
   if (AfeTxMode == ADSL2_CPE)//8 MHz
   {
      WriteCoreReg((uint32)(V_DFI_TX_WDF_INT2_CTRL_35B_ADDR), 0x00000000);
      WriteCoreReg((uint32)(V_DFI_TX_WDF_INT2_COEF_35B_ADDR), 0x00000000);
   }
   else
   {
      WriteCoreReg((uint32)(V_DFI_TX_WDF_INT2_CTRL_35B_ADDR), 0x00000002);
      WriteCoreReg((uint32)(V_DFI_TX_WDF_INT2_COEF_35B_ADDR), 0x00ae4711);
      WriteCoreReg((uint32)(V_DFI_TX_WDF_INT2_CTRL_35B_ADDR), 0x00000001);
   }
   //0x30B700  DFI_TX_NS_CTRL_35B         Tx NS control register        0x00000002  0x00000002  0x00000002  0x00000002
   //0x30B704  DFI_TX_NS_COEF_35B         Tx NS coefficient register    0x00000199  0x00000000  0x00000000  0x00000000
   //0x30B700  DFI_TX_NS_CTRL_35B         Tx NS control register        0x00000021  0x00000011  0x00000011  0x00000011

   if (AfeTxMode == VDSL35b_CPE)
   {
      WriteCoreReg((uint32)(V_DFI_TX_NS_CTRL_35B_ADDR), 0x00000002);
      WriteCoreReg((uint32)(V_DFI_TX_NS_COEF_35B_ADDR), 0x00000199);
      WriteCoreReg((uint32)(V_DFI_TX_NS_CTRL_35B_ADDR), 0x00000021);
   }
   else if (AfeTxMode == ADSL2_CPE)//8 MHz
   {
      WriteCoreReg((uint32)(V_DFI_TX_NS_CTRL_35B_ADDR), 0x0000003E); //3rd order filter, ADSL clock, Quatize ADSLmode, INIT
      WriteCoreReg((uint32)(V_DFI_TX_NS_COEF_35B_ADDR), 0x000002FF);
      WriteCoreReg((uint32)(V_DFI_TX_NS_CTRL_35B_ADDR), 0x0000003D);
   }
   else
   {
        WriteCoreReg((uint32)(V_DFI_TX_NS_CTRL_35B_ADDR), 0x00000002);
      WriteCoreReg((uint32)(V_DFI_TX_NS_COEF_35B_ADDR), 0x00000000);
      WriteCoreReg((uint32)(V_DFI_TX_NS_CTRL_35B_ADDR), 0x00000011);
   }
   //0x30C784  V_DFI_TX_FIFO_35B       DFI TX FIFO Control/status Reg   0x00000033  0x00000033  0x00000033  0x00000033
   WriteCoreReg((uint32)(V_DFI_TX_FIFO_35B_ADDR), 0x00000033);
}

void VRX518_DfiConfig_RxPath(uint16 AfeRxMode)
{
   // this check is for VRx518 bring-up only
   if (gus_SkipAfeFlow & 0x0004)
   {
      return;
   }

   //Address   Register Name           Short Description          35b_Mode 30a_Mode 17a_Mode 8a_Mode
   //0x30B100  V_RX_WDF_DEC1_CTRL_35B     Rx WDF DEC1 control register  0x00000002  0x00000000  0x00000000  0x00000000
   //0x30B104  V_RX_WDF_DEC1_COEF_35B     Rx WDF DEC1 coefficient reg      0xcb78390f  0x00000000  0x00000000  0x00000000
   //0x30B100  V_RX_WDF_DEC1_CTRL_35B     Rx WDF DEC1 control register  0x00000001
   if (AfeRxMode == VDSL35b_CPE)
   {
        WriteCoreReg((uint32)(V_RX_WDF_DEC1_CTRL_35B_ADDR), 0x00000002);
        WriteCoreReg((uint32)(V_RX_WDF_DEC1_COEF_35B_ADDR), 0xcb78390f);

      if (gus_SkipAfeFlow & 0x1000)
         {
         WriteCoreReg((uint32)(V_RX_WDF_DEC1_CTRL_35B_ADDR), 0x00000009);
      }
      else
         {
         WriteCoreReg((uint32)(V_RX_WDF_DEC1_CTRL_35B_ADDR), 0x00000001);
         }
   }
   else
   {
        WriteCoreReg((uint32)(V_RX_WDF_DEC1_CTRL_35B_ADDR), 0x00000000);
        WriteCoreReg((uint32)(V_RX_WDF_DEC1_COEF_35B_ADDR), 0x00000000);
    }
   //0x30B500  DFI_RX_WDF_DEC2_CTRL_35B   Rx WDF DEC2 control register  0x00000002  0x00000002  0x00000002  0x00000000
   //0x30B504  DFI_RX_WDF_DEC2_COEF_35B   Rx WDF DEC2 coefficient reg      0x00ae4711  0x00ae4711  0x00ae4711  0x00000000
   //0x30B500  DFI_RX_WDF_DEC2_CTRL_35B   WDF DEC2 control register     0x00000001  0x00000001  0x00000001
   //0x30B600  DFI_RX_CIC_DEC_CTRL_35B    Rx CIC Decimator control reg  0x00000002  0x00000002  0x00000002  0x0000000A
   //0x30B600  DFI_RX_CIC_DEC_CTRL_35B    Rx CIC Decimator control reg  0x00000001  0x00000001  0x00000001  0x00000009


   if (AfeRxMode == VDSL8_CPE)
   {
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_CTRL_35B_ADDR), 0x00000000);
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_COEF_35B_ADDR), 0x00000000);

      WriteCoreReg((uint32)(V_DFI_RX_CIC_DEC_CTRL_35B_ADDR), 0x0000000A);
      WriteCoreReg((uint32)(V_DFI_RX_CIC_DEC_CTRL_35B_ADDR), 0x00000009);
   }
   else if (AfeRxMode == ADSL2_CPE)
   {
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_CTRL_35B_ADDR), 0x00000000);
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_COEF_35B_ADDR), 0x00000000);

      WriteCoreReg((uint32)(V_DFI_RX_CIC_DEC_CTRL_35B_ADDR), 0x0000000A); //ADSL clock mode, Init
      WriteCoreReg((uint32)(V_DFI_RX_CIC_DEC_CTRL_35B_ADDR), 0x00000009);
   }
   else  //35 MHz 30MHz 17MHz 12 MHz
   {
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_CTRL_35B_ADDR), 0x00000002);
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_COEF_35B_ADDR), 0x00ae4711);
      WriteCoreReg((uint32)(V_DFI_RX_WDF_DEC2_CTRL_35B_ADDR), 0x00000001);
      WriteCoreReg((uint32)(V_DFI_RX_CIC_DEC_CTRL_35B_ADDR), 0x00000002);
      WriteCoreReg((uint32)(V_DFI_RX_CIC_DEC_CTRL_35B_ADDR), 0x00000001);
   }

   //0x30C788  V_DFI_RX_FIFO_35B       DFI RX FIFO Control/status Reg   0x00000033  0x00000033  0x00000033  0x00000033
   WriteCoreReg((uint32)(V_DFI_RX_FIFO_35B_ADDR), 0x00000033);
}

