/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;  40 Middlesex Turnpike, Bedford, MA 01730-1413
;  Phone (781) 276 - 4000
;  Fax   (781) 276 - 4001
;
;  File Name: LL_IOf.c
;
;  Low-level core interface layer.
;
*****************************************************************************/

#include "common.h"
#include "ll_iof.h"
#include "dsp_regs_62.h"
//XDSLRTFW-3432 (Start_End)
#include "cri_iof.h"
#include "vdsl_xception.h"
#include "gdata.h"

#define OFFSET 4;


#define VDSL_62_LOCAL_LDST_START    0x10000
#define VDSL_62_LOCAL_LDST_END      (VDSL_62_LOCAL_LDST_START + 16*1024)
#define VDSL_62_IIBRAM_START        0x40000

//XDSLRTFW-3432 (Start_End)
#define ADMA_TIMEOUT_PERIOD         (16000)

extern int16 gs_FGActive;
extern uint8 guc_ch_id;
extern uint8 guc_adma;
extern uint32 gula_adma_status[3];

uint8 GetFreeAdmaQueue(void);
uint16 CheckAdmaCompletion(void);

//XDSLRTFW-1383 (Start)
/*
 *------------------------------------------------------------------------
 *
 *  Name:
 *    SetUpDMATransfer
 *
 * Prototype:
 *    void SetUpDMATransfer(uint32 SourceAddr, uint32 SourceEnd,
 *                          uint32 DestAddr, uint32 ul_DspAdmaCtrl_FillMask,
 *                          int32 ul_FillValue, uint16 us_WaitForDMAComplete)
 *
 *  Abstract:
 *    SetUpDMATransfer() -
 *
 *    Sets up a DMA transfer from the SourceStart to DestStart until the SourceEnd address
 *
 *  Returns:
 *    None
 *
 *------------------------------------------------------------------------
 */
void SetUpDMATransfer(uint32 SourceAddr, uint32 SourceEnd, uint32 DestAddr, uint32 ul_DspAdmaCtrl_FillMask, int32 ul_FillValue, uint16 us_WaitForDMAComplete)
{

   uint32 ul_adma_status = DSP_ADMA_STAT;
   uint32 ul_adma_status_value;
   uint32 ul_adma_ctrl = DSP_ADMA_CTRL;
   uint32 ul_adma_ctrl_value;

   //XDSLRTFW-3432 (Start)
   //This function uses only ADMA0
   guc_adma = 0;

   //Go ahead if ERROR bit is NOT SET due to previous ADMA operation
   ul_adma_status_value = GetDspReg(ul_adma_status);
   if ((ul_adma_status_value & ARC_ADMA_STAT_ERR_MASK) == 0)
   {

      // If a DMA is currently ready for transfer, do not proceed.
      do {
         ul_adma_status_value = GetDspReg(ul_adma_status);
         ul_adma_ctrl_value = GetDspReg(ul_adma_ctrl);
      } while((ul_adma_status_value & ARC_ADMA_STAT_PROC_MASK) || (ul_adma_ctrl_value & ARC_ADMA_CTRL_START_MASK));

      // setup the source start and end address, destination address, and start
      // the transfer.
      SetDspReg(DSP_ADMA_SA_S, SourceAddr);
      SetDspReg(DSP_ADMA_SA_E, SourceEnd);
      SetDspReg(DSP_ADMA_DA_S, DestAddr);
      SetDspReg(DSP_ADMA_FILL_VAL, ul_FillValue);

      // clear DSP_INT_STAT before starting ADMA
      SetDspReg(DSP_INT_STAT, DSP_INT_STAT_ADMA0_DONE_MASK);
      SetDspReg(ul_adma_ctrl, ARC_ADMA_CTRL_START_MASK | ul_DspAdmaCtrl_FillMask);

      //Wait for DMA to complete (only if requested)
      if (us_WaitForDMAComplete == 1) {
         // wait for the transfer to complete.
         do {
            ul_adma_status_value = GetDspReg(DSP_INT_STAT);
         } while((ul_adma_status_value & DSP_INT_STAT_ADMA0_DONE_MASK) == 0);
      }

      // Enter fail states if error detected
      ul_adma_status_value = GetDspReg(DSP_ADMA_STAT);

      //uc_error_code = 2;
   }
   //XDSLRTFW-3432 (End)

   if (ul_adma_status_value & ARC_ADMA_STAT_ERR_MASK)
   {
      EnterFailStates(E_CODE_ADMA_ERROR);
   }

}
//XDSLRTFW-1383 (End)



void SetUpDMATransfer1(uint32 SourceAddr, uint32 SourceEnd, uint32 DestAddr, uint32 ul_DspAdmaCtrl_FillMask, int32 ul_FillValue, uint16 us_WaitForDMAComplete)
{
   uint32 ul_adma_status = DSP_ADMA_STAT;
   uint32 ul_adma_status_value;
   uint32 ul_adma_ctrl = DSP_ADMA_CTRL;
   uint32 ul_adma_ctrl_value;
   uint8 uc_adma;

   //XDSLRTFW-3432 (Start)
   //Select the next ADMA descriptor. VRx518 supports three ADMA descriptors, HW processes these
   //descriptors in a circular way.
   uc_adma = (guc_adma + 1);

   //Handle wraparound
   uc_adma = (uc_adma < 3)? uc_adma : 0;

   guc_adma = uc_adma;

   if (uc_adma == 1)
   {
      ul_adma_status = DSP_ADMA1_STAT;
      ul_adma_ctrl = DSP_ADMA1_CTRL;
   }
   else if (uc_adma == 2)
   {
      ul_adma_status = DSP_ADMA2_STAT;
      ul_adma_ctrl = DSP_ADMA2_CTRL;
   }

   //Go ahead if ERROR bit is NOT SET due to previous ADMA operation
   ul_adma_status_value = GetDspReg(ul_adma_status);
   if ((ul_adma_status_value & ARC_ADMA_STAT_ERR_MASK) == 0)
   {
      // If a DMA is currently ready for transfer, do not proceed.
      do {
         ul_adma_status_value = GetDspReg(ul_adma_status);
         ul_adma_ctrl_value = GetDspReg(ul_adma_ctrl);
      } while((ul_adma_status_value & ARC_ADMA_STAT_PROC_MASK) || (ul_adma_ctrl_value & ARC_ADMA_CTRL_START_MASK));

      // setup the source start and end address, destination address, and start
      // the transfer.
      if (uc_adma == 0)
      {
         SetDspReg(DSP_ADMA_SA_S, SourceAddr);
         SetDspReg(DSP_ADMA_SA_E, SourceEnd);
         SetDspReg(DSP_ADMA_DA_S, DestAddr);
         SetDspReg(DSP_ADMA_FILL_VAL, ul_FillValue);
      }
      else if (uc_adma == 1)
      {
         SetDspReg(DSP_ADMA1_SA_S, SourceAddr);
         SetDspReg(DSP_ADMA1_SA_E, SourceEnd);
         SetDspReg(DSP_ADMA1_DA_S, DestAddr);
         SetDspReg(DSP_ADMA1_FILL_VAL, ul_FillValue);
      }
      else if (uc_adma == 2)
      {
         SetDspReg(DSP_ADMA2_SA_S, SourceAddr);
         SetDspReg(DSP_ADMA2_SA_E, SourceEnd);
         SetDspReg(DSP_ADMA2_DA_S, DestAddr);
         SetDspReg(DSP_ADMA2_FILL_VAL, ul_FillValue);
      }
      // clear DSP_INT_STAT before starting ADMA
      if (uc_adma == 0)
      {
         SetDspReg(DSP_INT_STAT, DSP_INT_STAT_ADMA0_DONE_MASK);
         SetDspReg(ul_adma_ctrl, ARC_ADMA_CTRL_START_MASK | ul_DspAdmaCtrl_FillMask);
      }
      else if (uc_adma == 1)
      {
         SetDspReg(DSP_INT_STAT, DSP_INT_STAT_ADMA1_DONE_MASK);
         SetDspReg(ul_adma_ctrl, ARC_ADMA_CTRL_START_MASK | ul_DspAdmaCtrl_FillMask);
      }
      else if (uc_adma == 2)
      {
         SetDspReg(DSP_INT_STAT_EXT, DSP_INT_STAT_ADMA2_DONE_MASK);
         SetDspReg(ul_adma_ctrl, ARC_ADMA_CTRL_START_MASK | ul_DspAdmaCtrl_FillMask);
      }

      //Wait for DMA to complete (only if requested)
      if ((us_WaitForDMAComplete == 1) && (uc_adma == 0)) {
         // wait for the transfer to complete.
         do {
            ul_adma_status_value = GetDspReg(DSP_INT_STAT);
         } while((ul_adma_status_value & DSP_INT_STAT_ADMA0_DONE_MASK) == 0);
      }
      else if ((us_WaitForDMAComplete == 1) && (uc_adma == 1)) {
         // wait for the transfer to complete.
         do {
            ul_adma_status_value = GetDspReg(DSP_INT_STAT);
         } while((ul_adma_status_value & DSP_INT_STAT_ADMA1_DONE_MASK) == 0);
      }
      else if ((us_WaitForDMAComplete == 1) && (uc_adma == 2)) {
         // wait for the transfer to complete.
         do {
            ul_adma_status_value = GetDspReg(DSP_INT_STAT_EXT);
         } while((ul_adma_status_value & DSP_INT_STAT_ADMA2_DONE_MASK) == 0);
      }

      // Enter fail states if error detected
      ul_adma_status_value = GetDspReg(DSP_ADMA_STAT);
      if (uc_adma == 1)
      {
         ul_adma_status_value = GetDspReg(DSP_ADMA1_STAT);
      }
      else if (uc_adma == 2)
      {
         ul_adma_status_value = GetDspReg(DSP_ADMA2_STAT);
      }

      //uc_error_code = 4;
   }
   //XDSLRTFW-3432 (End)

   if (ul_adma_status_value & ARC_ADMA_STAT_ERR_MASK)
   {
      EnterFailStates(E_CODE_ADMA_ERROR);
   }
}



/****************************************************************************
; Name: ReadCoreBuf32NoDMA
;
; Prototype:
;
; Description:
;  Function for 32-bit read of ADSL core buffer
;
; Arguments:
;  ul_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;   us_num      (I)     number of 32-bit words to be read
;
;  Return Value:
;  N/A
*****************************************************************************/
void ReadCoreBuf32NoDMA(uint32 ul_addr, int16 *psa_words, uint16 us_num)
{
   uint16 us_idx;
   uint32 ul_data;
   uint32 *pul_LongBuf;

   if ((int32)psa_words & 0x2)
   {
      // it is a 16-bit aligned buffer
      for(us_idx=0; us_idx<us_num; us_idx++)
      {
         ReadCoreReg(ul_addr, &ul_data);
         ul_addr += OFFSET;
         *psa_words++ = ((int16 *)(void *)&ul_data)[0];
         *psa_words++ = ((int16 *)(void *)&ul_data)[1];
      }
   }
   else
   {
      // it is a 32-bit aligned buffer
      pul_LongBuf = (uint32 *)((void *)psa_words);
      for(us_idx=0; us_idx<us_num; us_idx++)
      {
         ReadCoreReg(ul_addr, pul_LongBuf++);
         ul_addr += OFFSET;
      }

   }
}

/****************************************************************************
; Name: WriteCoreBuf32NoDMA
;
; Prototype:
;  void WriteCoreBuf32(uint32 ul_addr, void *psa_words, uint16 us_num)
;
; Description:
;  Function for writing 32-bit words to a 32-bit buffer in the ADSL cores.
;
; Arguments:
;  ul_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 32-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
;
;  Notes: Initially the second parameter was of type int16 but to suppress/remove a warning
;         it has been changed to type void. And suitable changes are made in the function
;         and wherever it is being employed.
*****************************************************************************/

void WriteCoreBuf32NoDMA(uint32 ul_addr, int16 *psa_words, uint16 us_num)
{
   uint16 us_idx;
   uint32 ul_data;
   uint16 *pus_TwoWords;
   uint32 *pul_LongBuf;
   if ((int32)psa_words & 0x2)
   {
      // it is a 16-bit aligned buffer
      pus_TwoWords = (uint16*)(void *)&ul_data;
      for(us_idx=0; us_idx<us_num; us_idx++)
      {

         ((uint16 *)(void *)&ul_data)[0] = *psa_words++;
         ((uint16 *)(void *)&ul_data)[1] = *psa_words++;
         WriteCoreReg(ul_addr, ul_data);
         ul_addr += OFFSET;
      }
   }
   else
   {
      // it is a 32-bit aligned buffer
      pul_LongBuf = (uint32 *)(void *)psa_words;
      for(us_idx=0; us_idx<us_num; us_idx++)
      {
         WriteCoreReg(ul_addr, *pul_LongBuf++);
         ul_addr += OFFSET;
      }

   }
}

/****************************************************************************
; Name: ReadCoreBuf32
;
; Prototype:
;
; Description:
;  Function for 32-bit read of ADSL core buffer
;
; Arguments:
;  ul_addr     (I)      physical address at which to start reading
;  psa_words   (O)      array containing data read from the buffer
;   us_num      (I)     number of 32-bit words to be read
;
;  Return Value:
;  N/A
;
; Note: The second parameter has been changed from type int16 to void to suppress
;       or remove the warning. Suitable corrections are carried out.
*****************************************************************************/
void ReadCoreBuf32(uint32 ul_addr, int16 *psa_words, uint16 us_num)
{

   uint32 l_NumberOfBytes;
   // Exit gracefully if numwords to read is zero.
   // Without this check it otherwise, causes kernel exception/ crash
   if (!us_num)
      return;

   if (( (!(gul_35bLiteConfig & EN_ADMA_COPYBETWEEN_XRAM_RTV) ) && (gs_FGActive == 0)) ||
         ((VDSL_62_LOCAL_LDST_START <= ul_addr) && (ul_addr < VDSL_62_LOCAL_LDST_END)) ||
         ((VDSL_62_LOCAL_LDST_START <= (int32)psa_words) && ((int32)psa_words < VDSL_62_LOCAL_LDST_END)))
   {
      ReadCoreBuf32NoDMA(ul_addr, psa_words, us_num);
   }
   else
   {
      l_NumberOfBytes = 4*us_num-2;
      if (gul_35bLiteConfig & EN_ADMA_COPYBETWEEN_XRAM_RTV)
      {
         SetUpDMATransfer1(ul_addr, (ul_addr + l_NumberOfBytes),(uint32) psa_words, 0, 0, 0);
      }
      else
      {
         //XDSLRTFW-1383 (Start_End)
         SetUpDMATransfer(ul_addr, (ul_addr + l_NumberOfBytes),(uint32) psa_words, 0, 0, 1);
      }
   }

}

/****************************************************************************
; Name: WriteCoreBuf32
;
; Prototype:
;  void WriteCoreBuf32(uint32 ul_addr, int16 *psa_words, uint16 us_num)
;
; Description:
;  Function for writing 32-bit words to a 32-bit buffer in the ADSL cores.
;
; Arguments:
;  ul_addr     (I)      physical address at which to start writing
;  psa_words   (I)      array containing data to be written to the buffer
;  us_num      (I)      number of 32-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void WriteCoreBuf32(uint32 ul_addr, int16 *psa_words, uint16 us_num)
{
   // Exit gracefully if numwords to read is zero.
   // Without this check it otherwise, causes kernel exception/ crash
   if (!us_num)
      return;

   if ( ( (!(gul_35bLiteConfig & EN_ADMA_COPYBETWEEN_XRAM_RTV) ) && (gs_FGActive == 0)) ||
         ((VDSL_62_LOCAL_LDST_START <= ul_addr) && (ul_addr < VDSL_62_LOCAL_LDST_END)) ||
         ((VDSL_62_LOCAL_LDST_START <= (int32)psa_words) && ((int32)psa_words < VDSL_62_LOCAL_LDST_END)))
   {
      WriteCoreBuf32NoDMA(ul_addr, psa_words, us_num);
   }
   else
   {

      if (gul_35bLiteConfig & EN_ADMA_COPYBETWEEN_XRAM_RTV)
      {
         SetUpDMATransfer1((uint32) psa_words, (uint32) &psa_words[2*us_num-1],ul_addr, 0, 0, 0);
      }
      else
      {
         //XDSLRTFW-1383 (Start_End)
         SetUpDMATransfer((uint32) psa_words, (uint32) &psa_words[2*us_num-1],ul_addr, 0, 0, 1);
      }
   }
}



/****************************************************************************
; Name: FillCoreBuf32
;
; Prototype:
;  void FillCoreBuf32(uint32 ul_addr, int32 s_value, uint16 us_num)
;
; Description:
;
; Arguments:
;  ul_addr     (I)      physical address at which to start writing
;  s_value     (I)      data to be filled in to the buffer
;  us_num      (I)      number of 16-bit words to write to the buffer
;
;
;  Return Value:
;  N/A
*****************************************************************************/
void FillCoreBuf32(uint32 ul_addr, int32 l_value, uint16 us_num)
{
   int32 i;
   if (( (!(gul_35bLiteConfig & EN_ADMA_COPYBETWEEN_XRAM_RTV) ) && (gs_FGActive == 0)) ||
         ((VDSL_62_LOCAL_LDST_START <= ul_addr) && (ul_addr < VDSL_62_LOCAL_LDST_END)))
   {
      for (i=0; i<us_num; i++) {
         WriteCoreReg(ul_addr, (uint32)l_value);
         ul_addr += OFFSET;
      }
   }
   else
   {
      i = VDSL_62_IIBRAM_START + 4*us_num - 2;

      if (gul_35bLiteConfig & EN_ADMA_COPYBETWEEN_XRAM_RTV)
      {
         SetUpDMATransfer1(VDSL_62_IIBRAM_START, i, ul_addr, ARC_ADMA_CTRL_FILL_MASK, l_value, 0);
      }
      else
      {
         //XDSLRTFW-1383 (Start_End)
         SetUpDMATransfer(VDSL_62_IIBRAM_START, i, ul_addr, ARC_ADMA_CTRL_FILL_MASK, l_value, 1);
      }
   }

}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: void ReadCoreBuf8(uint32 ul_addr, int8 *pca_word, uint16 us_num)
*
*  This function reads a 32-bit word from a core memory.
*
*  Input Arguments:
*     ul_addr: start address
*     *pca_word: pointer to 8-bit data array to be read from the buffer
*     us_num: # of data to read
*
*  Output Arguments:
*
*  Returns:
*
*  Global Variables:
*
*  Notes: Initially the second parameter was of type void but to suppress/remove a warning
*         it has been changed to type void. And suitable changes are made in the function
*         and wherever it is being employed.
*-------------------------------------------------------------------------------
*/

void ReadCoreBuf8(uint32 ul_addr, int8 *pca_word, uint16 us_num)
{
   int32 i;
   uint32 ul_data;
   for (i=0; i<us_num; i++) {
      ReadCoreReg(ul_addr, &ul_data);
      pca_word[i] = (int8)ul_data;
      ul_addr+= OFFSET;
   }
}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: void WriteCoreBuf8(uint32 ul_addr, void *pca_word, uint16 us_num)
*
*  This function writes a 32-bit word to a core memory.
*
*  Input Arguments:
*     ul_addr: start address
*     *pca_word: pointer to 8-bit data array to be written to the buffer
*     us_num: # of data to write
*
*  Output Arguments:
*
*  Returns:
*
*  Global Variables:
*
*  Notes: Initially the second parameter was of type int8 but to suppress/remove a warning
*         it has been changed to type void. And suitable changes are made in the function
*         and wherever it is being employed.
*-------------------------------------------------------------------------------
*/

void WriteCoreBuf8(uint32 ul_addr, int8 *pca_word, uint16 us_num)
{
   int32 i;
   uint32 ul_data;
   for (i=0; i<us_num; i++) {
      ul_data = (uint32)pca_word[i];
      WriteCoreReg(ul_addr, ul_data);
      ul_addr+=OFFSET;
   }
}

//Return Free ADMA Queue
//Note: if none of the ADMA queue is free, use adma queue 0
//While starting ADMA, check for the previous ADMA engine complettion status.
uint8 GetFreeAdmaQueue(void)
{
   uint32 ul_adma0_stat = 0, ul_adma1_stat = 0, ul_adma2_stat = 0;
   uint8 uc_adma = 0;

   ul_adma0_stat = GetDspReg(DSP_ADMA_STAT);
   ul_adma1_stat = GetDspReg(DSP_ADMA1_STAT);
   ul_adma2_stat = GetDspReg(DSP_ADMA2_STAT);

   if ((ul_adma0_stat & ARC_ADMA_STAT_PROC_MASK) == 0)
   {
      uc_adma = 0;
   }
   else if ((ul_adma1_stat & ARC_ADMA_STAT_PROC_MASK) == 0)
   {
      uc_adma = 1;
   }
   else if ((ul_adma2_stat & ARC_ADMA_STAT_PROC_MASK) == 0)
   {
      uc_adma = 2;
   }

   return uc_adma;

}

//XDSLRTFW-3432 (Start)
//Returns ADMAn task error status; (us_error_status = 0, No error);  (us_error_status != 0, ADMAn task error)
uint16 CheckAdmaCompletion(void)
{
   uint32 ula_addr_stat[3] = {DSP_ADMA_STAT, DSP_ADMA1_STAT, DSP_ADMA2_STAT};
   uint32 ula_addr_ctrl[3] = {DSP_ADMA_CTRL, DSP_ADMA1_CTRL, DSP_ADMA2_CTRL};
   uint32 ul_adma_status_value, ul_adma_ctrl_value;
   uint32 ul_CycleStart, ul_CycleEnd;
   int32  temp;
   int16 i;

   //Bit0: ADMA 0 Error status;
   //Bit1: ADMA 1 Error status;
   //Bit2: ADMA 2 Error status;
   //Bit15: ADMA Timeout (ERROR condition)
   uint16 us_error_status = 0;

   ReadRxTimer(&ul_CycleStart);

   for (i = 0; i < 3; i++)
   {

      //Refer to section 2.1.3 (and sub sections), 2.3.2.11, 2.3.2.12 of VRx518 Arete document.
      //D_ADMAn_STAT.PROC     D_ADMAn_CTRL.READY               Status
      //     0                      0                 ADMAn task processing is complete and is available for next DMA operation
      //     0                      1                 ADMAn task is READY for processing i.e. Data is queued and ready for DMA operation
      //     1                      0                 ADMAn task is under processing i.e. DMA operation is in progress
      //
      // Note: ADMA engine first SETS PROC bit followed by CLEARING of READY bit.
      // Hence D_ADMAn_CTRL register should be ***READ FIRST**** to check that READY bit is ZERO, followed by
      // read of D_ADMAn_STAT register to check that PROC bit is ZERO.
      // This register read order should ***NOT BE REVERSED***, else there is chance (corner case) that both PROC bit
      // and READY bit is read as ZERO even though ADMAn task is not yet completed. This can happen when ADMA engine
      // changes the status of "PROC bit, READY bit" from "0, 1" to "1, 0" in between reading of D_ADMAn_STAT and
      // D_ADMAn_CTRL register
      //
      //Wait until ADMAn task execution is complete i.e. both PROC bit and READY bit are ZERO
      do {
         //ALWAYS read D_ADMAn_CTRL register FIRST, followed by read of D_ADMAn_STAT register
         //DO NOT REVERSE THIS ORDER here!!
         ul_adma_ctrl_value = GetDspReg(ula_addr_ctrl[i]);
         ul_adma_status_value = GetDspReg(ula_addr_stat[i]);

         //Timeout if ADMA is taking lot of time (e.g. 16000 cycles)
         ReadRxTimer(&ul_CycleEnd);
         temp = (int32)(ul_CycleEnd - ul_CycleStart);
         if (temp < 0)
         {
            temp += gl_MaxRxTimerCnt;
         }
         if (temp > ADMA_TIMEOUT_PERIOD)
         {
            //Error: Looks like ADMA is taking too long!
            us_error_status |= (1 << 15);
            return (us_error_status);
         }
      } while ((ul_adma_status_value & ARC_ADMA_STAT_PROC_MASK) || (ul_adma_ctrl_value & ARC_ADMA_CTRL_START_MASK));

      gula_adma_status[i] = ul_adma_status_value;
      us_error_status |= (((ul_adma_status_value & ARC_ADMA_STAT_ERR_MASK) >> 1) << i);
   }

   return (us_error_status);
}
//XDSLRTFW-3432 (End)
