/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;  Aware DMT Technology. Proprietary and Confidential.
;
;  40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;  Phone (781) 276 - 4000
;  Fax   (781) 276 - 4001
;
;  File Name: dsp_op.c
;
;  This file contains functions emulating DSP operations.
;
;***************************************************************************/

#include "common.h"
#include "dsp_op.h"

FlagT gs_Overflow;         /*  indicates that overflow has occurred */

/*****************************************************************************
 ;
 ;  Subroutine Name: norm_l(L_var1)
 ;
 ;  Description:
 ;  Produces the number of left shift needed to normalize the 32 bit varia-
 ;  ble l_var1 for positive values on the interval with minimum of
 ;  1073741824 and maximum of 2147483647, and for negative values on the in-
 ;  terval with minimum of -2147483648 and maximum of -1073741824; in order
 ;  to normalize the result, the following operation must be done :
 ;                   norm_L_var1 = L_shl(L_var1,norm_l(L_var1)).
 ;
 ;  Prototype:
 ;    int16 norm_l(int32 L_var1)
 ;
 ;  Arguments:
 ;    L_var1 -- 32 bit long signed integer (int32) whose value falls in the
 ;              range : 0x8000 0000 <= var1 <= 0x7fff ffff.
 ;
 ;  Outputs:
 ;    none
 ;
 ;  Return Value:
 ;    var_out-- Left shift value.
 ;
 ******************************************************************************/
int16 norm_l(int32 L_var1)
{
   int16 s_var_out;

   if (L_var1 == 0L) {
      s_var_out = (int16)0;  /* crs test */
   }
   else {
      if (L_var1 == (int32)0xffffffffL) {
         s_var_out = (int16)31;
      }
      else {
         if (L_var1 < 0L) {
            L_var1 = ~L_var1;
         }

         for(s_var_out = (int16)0; L_var1 < (int32)0x40000000L; s_var_out++) {
            L_var1 <<= 1L;
         }
      }
   }

   return(s_var_out);
}

/****************************************************************************
 ;
 ; Subroutine Name : round(L_var, s_bit_position)
 ;
 ;  This function performs conventional rounding at bit
 ; indicated by "bit_position" and then shift the result
 ; right by "bit_position"
 ;
 ; Prototype:
 ;   int32 round(int32 L_var, int s_bit_position)
 ;
 ;  Input Arguments:
 ;
 ;    L_var - 32 bit long signed integer whose value falls in the
 ;             range : 0x8000 0000 <= L_var1 <= 0x7fff ffff
 ;   s_bit_position - the bit position rounding bit 1 is added
 ;
 ;   Output Arguments:
 ;
 ;    none
 ;
 ;  Return Value:
 ;
 ;    l_var_out - 32 bit long signed integer whose value falls in the
 ;              range : 0x8000 0000 <= L_var1 <= 0x7fff ffff
 *****************************************************************************/

/****************************************************************************
 ;
 ; Subroutine Name: l_add(L_var1, L_var2)
 ;
 ;  Description:
 ;  Adds two 32bit values and tests for overflow/underflow.  If overflow/
 ;  underflow occurs then result if saturated to MAX_32/MIN_32
 ;
 ; Prototype:
 ;   int32 l_add(int32 L_var1, int32 L_var2)
 ;
 ;  Input Arguments:
 ;
 ;    L_var1, L_var2 - 32 bit long signed integers to be summed
 ;
 ;   Output Arguments:
 ;
 ;  Return Value:
 ;
 ;    l_var_out - 32 bit long signed integer whose value falls in the
 ;              range : 0x8000 0000 <= L_var1 <= 0x7fff ffff
 *****************************************************************************/

/*****************************************************************************
 ;
 ;   Subroutine Name: Divide_16bit( s_numer_mantissa, s_numer_exp,
 ;                         s_denom_mantissa, s_denom_exp,
 ;                         ps_quot_mantissa, ps_quot_exp)
 ;
 ;   Description:
 ;
 ;    Divides 16-bit numerator by 16-bit denominator to get 16-bit quotient.
 ;
 ;  {s_numer_mantissa,s_numer_exponent} and {s_denom_mantissa,s_denom_exponent}
 ; must both represent *normalized* values where
 ; s_numer and s_denom are mantissas in signed format Q1.15, and s_numer_exponent and
 ; s_denom_exponent are signed 16-bit exponents.
 ;
 ; The output quotient is represented by {s_quot_mantissa,s_quot_exponent}
 ;
 ;   s_numer_mantissa and s_denom_mantissa must both be positive.
 ;
 ;   Prototype:
 ;    int16 Divide_16bit(int16 s_numer_mantissa, int16 s_numer_exponent,
 ;             int16 s_denom_mantissa, int16 s_denom_exponent,
 ;             int16 *ps_quot_mantissa, int16 *ps_quot_exponent
 ;             )
 ;
 ;  Input Arguments:
 ;    int16 s_numer_mantissa -- 16-bit normalized numerator mantissa. Must be positive.
 ;    int16 s_numer_exponent -- 16-bit numerator exponent (negative means right shift)
 ;    int16 s_denom_mantissa -- 16-bit normalized denominator mantissa. Must be positive.
 ;    int16 s_denom_exponent -- 16-bit denominator exponent (negative means right shift)
 ;
 ;   Output Arguments:
 ;    int16 *ps_quot_mantissa -- 16-bit normalized quotient mantissa.
 ;    int16 *ps_quot_exponent -- 16-bit quotient exponent.
 ;
 ;   Return Value:
 ;    0 if no error
 ;    -1 if error (division by zero or negative arguments)
 ;
 ; NOTE: Does not check for overflow of exponents.
 ;
 ******************************************************************************/

int16 Divide_16bit(int16 s_numer_mantissa, int16 s_numer_exponent,
                   int16 s_denom_mantissa, int16 s_denom_exponent,
                   int16 *ps_quot_mantissa, int16 *ps_quot_exponent
                  )
{
   // Convert 16-bit numerator to 32 bit numerator and call other division routine.

   return(Divide_32by16bit((int32)s_numer_mantissa << 16, (int16)(s_numer_exponent-16),
                           s_denom_mantissa,  s_denom_exponent,
                           ps_quot_mantissa, ps_quot_exponent));
}

int16 Divide_32by16bit(int32 l_numer_mantissa, int16 s_numer_exponent,
                       int16 s_denom_mantissa, int16 s_denom_exponent,
                       int16 *ps_quot_mantissa, int16 *ps_quot_exponent
                      )
{
   int16 s_quot_mantissa;
   int16  iteration, s_First_1bit_position;
   int32 L_numer;
   int32 L_denom, L_NumerPartial, L_Numer_LowWord;
   uint16 us_bitmask;

   if ((l_numer_mantissa < 0) || (s_denom_mantissa <= (int16)0)) {
#if DSP_OP_PRN
      if ((s_denom_mantissa == (int16)0)
            fprintf(stderr, "Division by 0, Fatal error \n");
            else
               fprintf(stderr, "Input arguments for Division don't meet the requirement\n");
#endif
               return(-1);
      }

      L_numer = (int32) l_numer_mantissa;
   L_denom = (int32) s_denom_mantissa;

   L_NumerPartial = L_numer >> 16;

   if (L_NumerPartial >= L_denom) {
      // Leftshift denominator by one so that it's strictly greater than numerator.
      L_denom <<= 1;
      s_denom_exponent--;
   }

   // Do long division loop, calculating one bit of quotient per loop.
   // The quotient mantissa has an implicit exponent of +1.
   // Also make note of position of first '1' bit in quotient for purposes of normalizing
   // quotient.

   L_Numer_LowWord = L_numer & 0xffff;
   s_quot_mantissa = 0;
   us_bitmask = 0xffff;
   s_First_1bit_position = 0;

   for(iteration=(int16)0; iteration< (int16)15; iteration++) {
      s_quot_mantissa <<=1;
      // Add the next bit of the full numerator to L_NumerPartial.
      L_NumerPartial = (L_NumerPartial << 1) | (L_Numer_LowWord >> 15);
      L_Numer_LowWord <<= 1;

      if (L_NumerPartial >= L_denom) {
         L_NumerPartial = L_NumerPartial - L_denom;
         s_quot_mantissa += (int16)1;
         // These two lines save position of first '1' bit for
         // normalization purposes, and they do it without a
         // cycle-consuming conditional statement.
         s_First_1bit_position |= iteration & us_bitmask;
         us_bitmask = 0;
      }
   }

   // Normalize quotient

   *ps_quot_mantissa = (s_quot_mantissa <<= s_First_1bit_position);

   if (s_quot_mantissa == 0)
      *ps_quot_exponent = 0;     // Force exponent to zero if quotient is zero.
   else
      // Take into account normalization of quotient when calculating quotient exponent.
      *ps_quot_exponent = 1 + s_numer_exponent - s_denom_exponent - s_First_1bit_position;

   return(0);  // No error.
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: RightShiftAndRound32to16
 *
 *  Description: Right-shift and truncate values in a buffer and copy the results
 *           to another buffer.
 *
 *  Prototype: void RightShiftAndRound32to16(int16 *psa_dest, int16 s_doffset,
 *                                  int32 *pla_src, int16 s_soffset,
 *                                  int16 s_nitems, int16 s_rshift)
 *
 *  Input Arguments: pla_src     - pointer to the source buffer (int32)
 *              s_soffset     - offset for pl_src
 *              s_nitems      - number of items to copy
 *              s_rshift      - number of bits to shift
 *
 *  Output Arguments: psa_dest      - pointer to the destination buffer (int16)
 *               s_doffset    - offset for ps_dest
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */

void RightShiftAndRound32to16(int16 *psa_dest, int16 s_doffset,
                              int32 *pla_src, int16 s_soffset,
                              int16 s_nitems, int16 s_rshift)
{
   int16 i;

   psa_dest += s_doffset;
   pla_src += s_soffset;

   for (i = 0; i < s_nitems; i++)
      psa_dest[i] = sature16(round(pla_src[i], s_rshift));
}


/****************************************************************************
 ;
 ; Subroutine Name : sature16(L_var1)
 ;
 ;  This function limits the 32 bit input to the range of a 16 bit word
 ; with saturation control.
 ;
 ; Prototype:
 ;   int16 sature16(int32 L_var1)
 ;
 ;  Input Arguments:
 ;
 ;    L_var1 - 32 bit long signed integer whose value falls in the
 ;             range : 0x8000 0000 <= L_var1 <= 0x7fff ffff.
 ;
 ;   Output Arguments:
 ;
 ;    none
 ;
 ;  Return Value:
 ;
 ;    s_var_out - 16 bit short signed integer (int16) whose value falls in the
 ;              range : 0xffff 8000 <= var_out <= 0x0000 7fff.
 *****************************************************************************/
int16 sature16(int32 L_var1)
{
   int16 s_var_out;

   if (L_var1 > (int32)0X00007fffL) {
      gs_Overflow = TRUE;
      s_var_out = (int16)MAX_16;
   }
   else {
      if (L_var1 < (int32)0xffff8000L) {
         gs_Overflow = TRUE;
         s_var_out = (int16)MIN_16;
      }
      else {
         gs_Overflow = FALSE;
         s_var_out = (int16)(L_var1);
      }
   }
   return(s_var_out);
}

/*****************************************************************************
 ;
 ;  Subroutine Name: norm_16bit(s_var1)
 ;
 ;  Description:
 ;  Returns the number of left shifts needed to normalize the 16 bit
 ; signed variable s_var1 to Q1.15 format.
 ;
 ; In order
 ;  to normalize the result, the following operation must be done :
 ;                   norm_s_var1 = s_var1 << norm_16bit(s_var1).
 ;
 ;  Prototype:
 ;    int16 norm_16bit(int16 s_var1)
 ;
 ;  Arguments:
 ;    s_var1 -- 16 bit signed integer
 ;
 ;  Outputs:
 ;    none
 ;
 ;  Return Value:
 ;    var_out-- Left shift value.
 ;
 ******************************************************************************/
int16 norm_16bit(int16 s_var1)
{
   int16 s_LeftShift;

   if (s_var1 == 0) {
      s_LeftShift = (int16)0;
   }
   else if (s_var1 == (int16)0xffff) {
      s_LeftShift = (int16)15;
   }
   else {
      if (s_var1 < 0)
         s_var1 = ~s_var1;

      for(s_LeftShift = (int16)0; s_var1 < (int16)0x4000; s_LeftShift++) {
         s_var1 <<= 1;
      }
   }

   return(s_LeftShift);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : NormAndDivideMantExp_32by16bit
 *
 *  Description:  This function performs the division of a 32-bit numerator by a 16-bit denominator,
 *  and the result is represented by a mantissa, s_mant, and exponent, s_exp.
 * Both numerator and denominator are first normalized before division.
 * In equation:
 *    s_mant*(2^s_exp) = (l_numer/s_denom)
 *
 *  Prototype:
 *     void NormAndDivideMantExp_32by16bit(int32 l_numer, int16 s_denom, int16 *ps_mant, int16 *ps_exp)
 *
 *  Input Arguments:
 *      l_numer -- 32-bit numerator
 *    s_denom -- 16-bit denominator
 *
 *  Output Arguments:
 *    ps_mant -- pointer to the mantissa of quotient
 *    ps_exp -- pointer to the exponent of quotient
 *
 * Return:
 *
 *
 *  Global Variables Used:
 *
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void NormAndDivideMantExp_32by16bit(int32 l_numer, int16 s_denom, int16 *ps_mant, int16 *ps_exp)
{
   int16 s_num_exp, s_den_exp, s_quot_exp;
   int16 s_quot;

   s_num_exp = norm_l(l_numer);
   l_numer <<= s_num_exp;
   s_num_exp = -s_num_exp;

   s_den_exp = norm_l((int32)s_denom);
   s_den_exp -= 16;
   s_denom <<= s_den_exp;
   s_den_exp = -s_den_exp;

   Divide_32by16bit(l_numer, s_num_exp, s_denom, s_den_exp, &s_quot, &s_quot_exp);

   *ps_mant = s_quot;
   *ps_exp = s_quot_exp;

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : NormAndDivide_32by16bit
 *
 *  Description:  This function performs the division of a 32-bit numerator by a 16-bit denominator,
 *  and then truncate the result to 16-bit interger.
 * Both numerator and denominator are first normalized before division.
 * In equation:
 *    return = (int32)(l_numer/s_denom)
 *
 *  Prototype:
 *     int32 NormAndDivide_32by16bit(int32 l_numer, int16 s_denom)
 *
 *  Input Arguments:
 *      l_numer -- 32-bit numerator
 *    s_denom -- 16-bit denominator
 *
 *
 *  Output Arguments:
 *
 *
 * Return:
 *    16-bit division result
 *
 *  Global Variables Used:
 *
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int32 NormAndDivide_32by16bit(int32 l_numer, int16 s_denom)
{
   int16 s_quot, s_quot_exp;
   int32 l_quot;

   NormAndDivideMantExp_32by16bit(l_numer, s_denom, &s_quot, &s_quot_exp);

   if(s_quot_exp < 0)
      l_quot = (int32)s_quot >> (-s_quot_exp);
   else
      l_quot = (int32)s_quot << s_quot_exp;

   return(l_quot);
}

int32 NormAndDivide_32by16bit_with_rnd(int32 l_numer, int16 s_denom)
{
   int16 s_quot, s_quot_exp;
   int32 l_quot;

   NormAndDivideMantExp_32by16bit(l_numer, s_denom, &s_quot, &s_quot_exp);

   if(s_quot_exp < 0)
      l_quot = round((int32) s_quot, (int16) (-s_quot_exp));
   else
      l_quot = (int32)s_quot << s_quot_exp;

   return(l_quot);
}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: int16 floor16(int16 s_num, int16 s_den)
*
*  This function returns floor(s_num/s_den).
*
*  Input Arguments:
*     s_num: 16-bit numerator
*     s_den: 16-bit denominator
*
*  Output Arguments:
*
*  Returns:
*     floor(s_num/s_den)
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 floor16(int16 s_num, int16 s_den)
{
   int16 s_ans = 0;

   if (s_den == 1)
      s_ans = s_num;
   else if (s_den == 0)
      s_ans = (int16)0x7FFF;
   else {
      while (s_num >= s_den) {
         s_num -= s_den;
         s_ans++;
      }
   }

   return (int16)s_ans;
}

int32 floor32(int32 l_num, int16 s_den)
{
   int32 l_ans = 0;

   if (s_den == 1)
      l_ans = l_num;
   else if (s_den == 0)
      l_ans = (int32)0x7FFFFFFF;
   else {
      while (l_num >= s_den) {
         l_num -= s_den;
         l_ans++;
      }
   }

   return l_ans;
}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: int16 ceil16(int16 s_num, int16 s_den)
*
*  This function returns ceil(s_num/s_den).
*
*  Input Arguments:
*     s_num: 16-bit numerator
*     s_den: 16-bit denominator
*
*  Output Arguments:
*
*  Returns:
*     ceil(s_num/s_den)
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 ceil16(int16 s_num, int16 s_den)
{
   int16 s_ans = 0;

   if (s_den == 1)
      s_ans = s_num;
   else if (s_den == 0)
      s_ans = (int16)0x7FFF;
   else {
      while (s_num >= s_den) {
         s_num -= s_den;
         s_ans++;
      }
      if (s_num != 0)
         s_ans++;
   }

   return (int16)s_ans;
}


int32 ceil32(int32 l_num, int16 s_den)
{
   int32 l_ans = 0;

   if (s_den == 1)
      l_ans = l_num;
   else if (s_den == 0)
      l_ans = (int32)0x7FFFFFFF;
   else {
      while (l_num >= s_den) {
         l_num -= s_den;
         l_ans++;
      }
      if (l_num != 0)
         l_ans++;
   }

   return l_ans;
}

uint32 ceil32u(uint32 Num, uint16 Den)
{
   uint32 Result;

   if (Den == 0)
   {
      Result = 1;
   }
   else
   {
      Result = Num/Den;

      if((Result*Den) != Num)
      {
         Result++;
      }
   }

   return Result;
}

/*
*-------------------------------------------------------------------------------
*
*  Prototype: int16 mod16(int16 s_num, int16 s_den)
*
*  This function returns s_num mod s_den. Inputs are assumed to be positive,
*  and s_den should not be zero.
*
*  Input Arguments:
*     s_num: 16-bit number
*     s_den: 16-bit number
*
*  Output Arguments:
*
*  Returns:
*     s_num mod s_den
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 mod16(int16 s_num, int16 s_den)
{
   if (s_den == 1)
      return (int16)0;

   if (s_den == 0) // error
      return (int16)(0x7FFF);

   while (s_num >= s_den)
      s_num -= s_den;

   return (int16)s_num;
}

/***************************************************************************************
;  Subroutine Name: sqrt32
;
;  Description:
;     This function returns the square root of absolute value of the 32-bit
;     signed input l_in.
;
;  Prototype:
;     int16 sqrt32(int32 l_in)
;
;  Input Arguments:
;     l_in
;
;  Output Arguments:
;     sqrt of abs(l_in)
;
;**********************************************************************************************/
int16 sqrt32(int32 l_in)
{
   int16 s_exp;
   int16 s_sqrt, s_test;
   int32 l_temp;

   if(l_in < 0) {
      l_in = -l_in;
   }

   //saturate sqrt(l_in) at 0x7fff
   if(l_in >= (int32)(0x7FFF*0x7FFF))
      return(0x7FFF);

   s_sqrt = 0;
   for(s_exp=14; s_exp>=0; s_exp--) {

      s_test = s_sqrt + (1<<s_exp);
      l_temp = (int32)s_test*s_test;
      if(l_in >= l_temp) {
         s_sqrt = s_test;
      }
   }
   l_temp = (int32)s_sqrt*s_sqrt;
   if(l_in-l_temp > (int32) s_sqrt)
      s_sqrt++;
   return(s_sqrt);
}

/*******************************************************************************
*
*  Prototype: uint32 BitRev(uint32 ul_In, int16 s_Nbits)
*
*  This function reverses the bits in ul_In
*
*  Input Arguments:
*     uint32 ul_In    - long word whose bits are to be reversed.
*       int16  s_Nbits  - num of bits in the word.
*
*  Output Arguments:
*
*  Returns:
*
*  Global Variables:
*
*******************************************************************************/
uint32 BitRev(uint32 ul_In, int16 s_Nbits)
{
   int16 i;
   uint32 ul_BitReversed=0;

   for (i=0 ; i<s_Nbits ; i++) {
      ul_BitReversed |= (ul_In & 1) << (s_Nbits-1-i);
      ul_In >>= 1;
   }

   return (ul_BitReversed);
}
