/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 1998-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *----------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Condfidential
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276-4000
 *   FAX   (781) 276-4001
 *
 *   InsertCE.c
 *
 *   Iridia interface layer functions used to insert the cyclic extension.
 *
 *
 *----------------------------------------------------------------------------
 */
#include "common.h"
#include "gdata.h"
#include "str_memmap.h"
#include "LL_Iof.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : DisableTxCE
 *
 *  Description:   Function to disable Tx Cyclic Extension
 *
 *  Prototype:  void DisableTxCE(void)
 *
 *
 *  Input Arguments: None.
 *
 *
 *  Output Arguments: None.
 *
 *  Return: None
 *
 *  Global Variables Used:  None
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void DisableTxCE(void)
{
   WriteCoreReg((uint32)V_TX_EXTEN_ADDR, (uint32)0);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ChangeTxCE_ForFirstSym
 *
 *  Description:    Time Critical Function to change TX_EXTEN register for
 *  generating the first symbol of O/R-P-Periodic state with the following
 *  setting:
 *      CP length: beta
 *      CS length: 0
 *      CP strobe: enable
 *      window strobe: enable
 *      CS strobe: disable
 *      CS capture: disable
 *
 *
 *  Prototype:  void ChangeTxCE_ForFirstSym(void)
 *
 *
 *  Input Arguments: None.
 *
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:  gs_TxBetaLength
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void ChangeTxCE_ForFirstSym(void)
{
   uint32 ul_tmpWord;

   // Set Tx extension register
   ul_tmpWord = (gs_TxBetaLength<<CP_SIZE_LSB);   // Tx CP length = beta, CS length = 0

   if(gs_TxBetaLength > 0)
   {
      ul_tmpWord |= (CP_EN | WIN_EN);         // Tx CP strobe enable, Tx window strobe enable                                    // Tx CS strobe disabled
      ul_tmpWord |= TX_WIN_COEFF_MODE_VDSL;   // the TX window coefficient are stored in the VDSL pattern
   }

   WriteCoreReg((uint32)V_TX_EXTEN_ADDR, ul_tmpWord);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ChangeTxCE_ForLastSym
 *
 *  Description:    Time Critical Function to change TX_EXTEN register for
 *  generating the first symbol of the non periodic state following the periodic state.
 *  It is noted that we choose to use the same function name as the one used in the 5.0
 *  to minimize the code change, although this name is somewhat misleading.
 *  It is also noted that this function does not do as what the VDSL2 specified
 *  This function symply shorten the CP length by beta so the total number of samples
 *  in the periodic state should be correct.
 *  In any case, the HW will not be able to do what the VDSL2 spec. required.
 *
 *  setting:
 *      CP length: CPLength - beta
 *      CS length: CSLength - beta
 *      window length : beta
 *      CP strobe: enabled
 *      window strobe: disabled
 *      CS strobe: enable
 *      CS capture: enable
 *
 *
 *  Prototype:  void ChangeTxCE_ForLastSym(void)
 *
 *
 *  Input Arguments: None.
 *
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:  gs_TxIfftLength, gs_TxBetaLength
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void ChangeTxCE_ForLastSym(void)
{
   uint32 ul_tmpWord;

   //Note!!! In the 6.2, this funciton is actually to configure
   //the TX extention for the first symbol following the periodic state

   // Set Tx extension register
   ul_tmpWord = (gs_TxCPLength - gs_TxBetaLength)<<CP_SIZE_LSB;   // Tx CP length
   ul_tmpWord |= (gs_TxCSLength<<CS_SIZE_LSB);            // Tx CS length

   if(gs_TxCPLength > gs_TxBetaLength)
   {
      ul_tmpWord |= CP_EN;   // Tx CP strobe enable
   }

   if(gs_TxCSLength > 0)
   {
      ul_tmpWord |= CS_EN;   // Tx CS strobe enable
   }
   // Tx window is disabled for the first symbol

   ul_tmpWord |= TX_WIN_COEFF_MODE_VDSL;   // the TX window coefficient are stored in the VDSL pattern

   WriteCoreReg((uint32)V_TX_EXTEN_ADDR, ul_tmpWord);

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ConfigTxFrameSize
 *
 *  Description:    Time Critical Function to set the strymon TX IFFT size
 *  to the normal FFT size.
 *
 *  Prototype:  void ConfigTxFrameSize(void)
 *
 *  Input Arguments: None.
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *      gs_TxIfftLength -- (I) normal TX IFFT size
 *      gul_TxFftRate -- (I) IFFT processing sampling rate
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void ConfigTxFrameSize(void)
{
   uint32 ul_TxFrameSize;

   ul_TxFrameSize = gs_TxIfftLength|gul_TxIfftRate;
   WriteCoreReg((uint32)V_TX_FSIZE_ADDR, ul_TxFrameSize);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : DisableRxCE
 *
 *  Description:    Function to disable Rx Cyclic Extension
 *
 *  Prototype:  void DisableRxCE(void)
 *
 *
 *  Input Arguments:none
 *
 *
 *  Output Arguments: None
 *
 *  Return: None
 *
 *
 *  Global Variables Used:   None
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void DisableRxCE(void)
{
   WriteCoreReg((uint32)V_RX_EXTEN_ADDR, (uint32)0);
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ChangeRxCE_PreFirstSym
 *
 *  Description:    Time Critical Function to change RX_EXTEN register for
 *  receiving the symbol prior to the first symbol of O/R-P-Periodic state.
 *  This is necessary only if the remote modem TX and local modem RX use
 *  different CS length. This function adjust the CS lenght of the last symbol
 *  to compensate for this difference.
 *
 *
 *  Prototype:  void ChangeRxCE_PreFirstSym(void)
 *
 *
 *  Input Arguments: None.
 *
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *      gs_EnablePeriodicFrameRealign -- (I) indicator is the adjustment is needed or not
 *      gs_RxTxCsDiff -- (I) local RX CS Length - remote TX CS length
 *      gft_RxTxCsDiff_err -- error indicator: 0: no error, 1 or 2: cannot make adjustment
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

#define MAX_CS_LENGTH (CS_SIZE_MASK>>CS_SIZE_LSB)

C_SCOPE void ChangeRxCE_PreFirstSym(void)
{
   uint32 ul_tmpWord;

   if((gs_EnablePeriodicFrameRealign == 0)||(gs_RxTxCsDiff == 0))
   {
      return;
   }

   gft_RxTxCsDiff_err = 0;

   //Read current RX_EXTEN register
   ReadCoreReg((uint32)V_RX_EXTEN_ADDR, &ul_tmpWord);
   gs_CsLength = (ul_tmpWord & CS_SIZE_MASK)>>CS_SIZE_LSB;

   gs_CsLength -= gs_RxTxCsDiff;

   if(gs_CsLength > MAX_CS_LENGTH)
   {
      gft_RxTxCsDiff_err = 1;
   }
   //Cannot compensate for such large gs_RxTxCsDiff
   else if(gs_CsLength < 0)
   {
      gft_RxTxCsDiff_err = 2;
   }

   if(gft_RxTxCsDiff_err == 0)
   {
      //Replace CS length by the new value
      ul_tmpWord &= ~((uint32)(CS_SIZE_MASK));
      ul_tmpWord |= (gs_CsLength<<CS_SIZE_LSB);

      //Write back to RX_EXTEN register
      WriteCoreReg((uint32)V_RX_EXTEN_ADDR, ul_tmpWord);
   }

} //ChangeRxCE_ForCsDiff


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ChangeRxCE_ForFirstSym
 *
 *  Description:    Time Critical Function to change RX_EXTEN register for
 *  receiving the first symbol of O/R-P-Periodic state with the following
 *  setting:
 *      CP length: beta
 *      CS length: 0
 *      CP strobe: enable
 *      window strobe: disable
 *      CS strobe: disable
 *
 *
 *  Prototype:  void ChangeRxCE_ForFirstSym(void)
 *
 *
 *  Input Arguments: None.
 *
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *      gs_RxBetaLength - (I) the RX beta length
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void ChangeRxCE_ForFirstSym(void)
{
   uint32 ul_tmpWord;

   //Set RX CP length to gs_RxBetaLength (according to the VDSL2 spec)
   ul_tmpWord = (gs_RxBetaLength<<CP_SIZE_LSB);

   if(ul_tmpWord > 0)
   {
      ul_tmpWord |= CP_EN;   // Rx CP strobe enable
   }
   // Rx CS strobe disable, Rx Window strobe disable

   WriteCoreReg((uint32)V_RX_EXTEN_ADDR, ul_tmpWord);

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ChangeRxFftLength_PreLastSym
 *
 *  Description:    Time Critical Function to change RX FFT length to achieve
 *  the RX frame alignment after coming out of the periodic state.
 *  This function should be called at the 2nd to the last frame before coming out of
 *  the periodic state.
 *
 *   To align the next frame properly after coming out the periodic state,
 *  sometimes it is needed to reduce the RX FFT size of the last frame.
 *  However, in the 6.2 HW, one strymon frame interval (between two buffer swaps),
 *  is not allowed to be less than the normal FFT size (which is used to drive the time sloted
 *  operation). Thus, when the last frame needs to be less than one normal FFT size,
 *  we combined the last frame and its pervious frame into one long frame.
 *  That is why this function needs to be called at the 2nd to the last frame of the periodic state.
 *
 *  Prototype:  void ChangeRxFftLength_PreLastSym(void)
 *
 *  Input Arguments: None.
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *      gs_RxFftLength -- (I) normal RX FFT size
 *      gs_RxBetaLength -- (I) RX beta length
 *      gs_EnablePeriodicFrameRealign -- (I) flag to indicate whether or not to align the RX frame
 *                                       due to the difference between remote TX CS and local RX CS length
 *      gs_RxTxCsDiff -- (I) equal to the local modem RX CS length subtracted by the remote modem TX CS length.
 *      gul_RxFftRate -- (I) FFT processing sampling rate
 *
 *      gft_SkipSymbol -- (O) set to 1 if two last frames are combined into one frame
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

C_SCOPE void ChangeRxFftLength_PreLastSym(void)
{
   uint32 ul_tmpWord;

   gft_SkipSymbol = 0;

   // Set Rx frame size in TD samples, not including the cyclic extension, to be 2N-beta
   ul_tmpWord = gs_RxFftLength - gs_RxBetaLength;

   //Perform the frame realignment to compensate for Tx/RX CS different
   if(gs_EnablePeriodicFrameRealign)
   {
      ul_tmpWord += gs_RxTxCsDiff;
   }

   //If the frame length is less than the minmum frame length  (for time slot frame)
   //combine  the next two frames into one frame
   if(ul_tmpWord < (uint32)gs_RxFftLength)
   {
      ul_tmpWord += gs_RxFftLength;
      gft_SkipSymbol = 1;
   }

   if(gft_SkipSymbol == 1)
   {
      ul_tmpWord |= gul_RxFftRate;
      WriteCoreReg(V_RX_FSIZE_ADDR, ul_tmpWord);
   }
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ChangeRxFftLength_ForLastSym
 *
 *  Description:    Time Critical Function to change RX FFT length for
 *  receiving the last symbol of O/R-P-Periodic state with the following
 *  setting:
 *
 *
 *  Prototype:  void ChangeRxFftLength_ForLastSym(void)
 *
 *
 *  Input Arguments: None.
 *
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *      gs_RxFftLength -- (I) normal RX FFT size
 *      gs_RxBetaLength -- (I) RX beta length
 *      gs_EnablePeriodicFrameRealign -- (I) flag to indicate whether or not to align the RX frame
 *                                       due to the difference between remote TX CS and local RX CS length
 *      gs_RxTxCsDiff -- (I) equal to the local modem RX CS length subtracted by the remote modem TX CS length.
 *      gul_RxFftRate -- (I) FFT processing sampling rate
 *
 *      gft_SkipSymbol -- (O) always set to 0
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
C_SCOPE void ChangeRxFftLength_ForLastSym(void)
{
   uint32 ul_tmpWord;

   gft_SkipSymbol = 0;

   // Set Rx frame size in TD samples, not including the cyclic extension, to be 2N-beta
   ul_tmpWord = gs_RxFftLength - gs_RxBetaLength;

   //Perform the frame realignment to compensate for Tx/RX CS different
   if(gs_EnablePeriodicFrameRealign)
   {
      ul_tmpWord += gs_RxTxCsDiff;
   }

   ul_tmpWord |= gul_RxFftRate;
   WriteCoreReg(V_RX_FSIZE_ADDR, ul_tmpWord);

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ConfigRxFrameSize
 *
 *  Description:    Time Critical Function to set the strymon FFT size
 *  to the normal FFT size.
 *
 *
 *  Prototype:  void ConfigRxFrameSize(void)
 *
 *  Input Arguments: None.
 *
 *  Output Arguments: None.
 *
 *  Return:
 *          None
 *
 *  Global Variables Used:
 *      gs_RxFftLength -- (I) normal RX FFT size
 *      gul_RxFftRate -- (I) FFT processing sampling rate
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

C_SCOPE void ConfigRxFrameSize(void)
{
   uint32 ul_data;

   ul_data = gs_RxFftLength | gul_RxFftRate;
   WriteCoreReg(V_RX_FSIZE_ADDR, ul_data);
}
