/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   TimingRecov.c
 *
 *   Interface functions for timing recovery hardware.
 *
 *------------------------------------------------------------------------
 */

#include "gdata.h"

#include "str_memmap.h"

#include "LL_IOf.h"
#include "dsp_op.h"
#include "TimingRecov.h"
#include "cmv.h"
#include "mul.h"


/*^^^
*------------------------------------------------------------------------
*
*  Name: UpdateTimingRecoveryHW(int32 l_in);
*
*  Description: Updates the hardware register for timing recovery.
*
*------------------------------------------------------------------------
*^^^
*/
int32 gl_upperlimit = 0x4F000;
int32 gl_lowerlimit = 0x47000;

int32 CalcPhaseAccumIncr(int32 l_SRCFR);

void UpdateTimingRecoveryHW(int32 l_in)
{
   int32 l_in_quan;
   int32 l_PhaseAccIncr;

   if (gft_EnableNoiseShaping)
   {
      l_in += gl_noise_shaping_quan_err;
   }

   if ((gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)||(gft_Debug35bHwConfig17aModeEnable == TRUE) )
   {
      //Algorithm to avoid Corner Case in Phase Accumulator
      //In the HW there is a problem if the phase accumulator becomes exactly 2^24.
      //In this case the SRC event comes one cycle too late, which means that 1 output sample is wrong.
      //This leads to a noise spike, i.e. one TX symbol becomes noisy.
      //A simple methiod to overcome this problem is available and has also been successfully tested in RTL and the Cocomo simulation platform:
      //- Apply a phase jump of 1 shortly after the SRC has been enabled
      //- Only allow even values of SRCFR to be programmed
      //This ensures that the phase accumulator value is always odd and can therefore never become exactly 2^24.

      //if the SRCFR value to be programmed is odd, then Floor to the nearest even number to avoid SRCFR value as odd. This is required for 35B mode only.
      //Either Round or Floor permitted is according to Roland, Bit Floor is selected to work with COCOMO having upper limit of 0x7FFFF
      //SRCFR <= (SRCFR) & 0xFFFFFFFE;
      l_in_quan   = l_in + ((int32)0x1 << (15-1)); // add rounding bit
      l_in_quan >>= 15;
      gl_noise_shaping_quan_err = l_in - (l_in_quan << 15);
      l_in_quan <<= 1;                             // retain upper 18 bits
   }
   else
   {
      l_in_quan   = l_in + ((int32)0x1 << (14-1)); // add rounding bit
      l_in_quan >>= 14;                            // retain upper 18 bits
      gl_noise_shaping_quan_err = l_in - (l_in_quan << 14);
   }
   l_in_quan   = -l_in_quan; // Hercules SRC FR has opposite sense compared to Socrates SRC FR

   // Add constant SRC FR offset. CO crytal @ 35.328 MHz; CPE @ 36 MHz.
   l_in_quan  += gl_constant_SRCFR_offset;

   if (l_in_quan > gl_upperlimit)
   {
      l_in_quan = gl_upperlimit;   // make sure it's not too big or too small;
   }
   if (l_in_quan < gl_lowerlimit)
   {
      l_in_quan = gl_lowerlimit;   // otherwise PLL not able to pull in
   }

   // Update sampling frequency ratio register
   if (gft_UpdateTimingRecoveryHWEnable == 1)
   {
      if (!gft_DisableSRCBugWorkAround)
      {
         // SRC wrinkle bug FW workaround:
         l_PhaseAccIncr = CalcPhaseAccumIncr(l_in_quan);

         // Check to see whether phase accum increment is even...
         if (l_PhaseAccIncr & 0x1)
         {
            l_in_quan += 1; // ... if not, try next SRCFR offset value...
            gl_noise_shaping_quan_err -= (1 << 14); // Hercules SRC FR has opposite sense compared to Socrates SRC FR

            l_PhaseAccIncr = CalcPhaseAccumIncr(l_in_quan);
            // ... if still not even, use original (SRCFR offset value - 1) because
            // there is guaranteed 1 even phase accum increment corresponding to 3 consecutive
            // SRCFR offset values.
            if (l_PhaseAccIncr & 0x1)
            {
               l_in_quan -= 2;
               gl_noise_shaping_quan_err += (2 << 14); // Hercules SRC FR has opposite sense compared to Socrates SRC FR
            }
         }
      }

      // 20bit - OFFSET : SRC frequency ratio for both Rx and Tx directions.
      WriteCoreReg((uint32)V_SRCFR_ADDR, (uint32)(l_in_quan));
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Prototype: int32 Mul32x32_sh24(int32 x, int32 y);
 *
 *  Description: Multiplies x by y and round 64 bits result by shifting
 *               down 24 bits. Adds 1.0 in 1.24 format at the end.
 *
 *------------------------------------------------------------------------
 *^^^
 */
int32 Mul32x32_sh24(int32 x, int32 y)
{
   uint16 x1, x2, y1, y2;
   uint32 l_p1, l_p2, l_p3, l_p4, l_ps;
   int32 l_p;

   // let  x   = x1*2^m + x2
   // let  y   = y1*2^m + y2
   // then x*y = x1*y1*2^(2*m) + (x1*y2 + x2*y1)*2^m + x2*y2

   x1 = (uint16)(x >> 16);
   x2 = (uint16)(x & 0xFFFF);
   y1 = (uint16)(y >> 16);
   y2 = (uint16)(y & 0xFFFF);

   // result is downshifted by 16 first
   MULU16(l_p1, x2, y2);
   l_p1 >>= 16;
   MULU16(l_p2, x1, y2);
   MULU16(l_p3, x2, y1);
   MULU16(l_p4, x1, y1);
   l_p4 <<= 16;

   l_ps = (l_p1+l_p2+l_p3+l_p4);
   l_ps >>= 7; // downshifted by another 7 to make it 23

   l_p  = (int32)l_ps;

   // Round result to 4.24
   l_p += 1;
   l_p >>= 1;

   // Add 1.0 in 1.24
   l_p += (1<<24);

   return l_p;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Prototype: int32 CalcPhaseAccumIncr(int32 l_SRCFR);
 *
 *  Description: Compute reciprocal of Rx SRC frequency ratio using
 *               4th order binomial approximation of 1/(1-x)
 *               where 0 <= x <= (2^20-1)/2^24.
 *
 *               1/(1-x) = 1.0 + x + x^2 + x^3 + x^4
 *                       = 1.0 + x *(1.0 + x*(1.0 + x*(1.0 + x)))
 *------------------------------------------------------------------------
 *^^^
 */
int32 CalcPhaseAccumIncr(int32 l_SRCFR)
{

   int16 i;
   int32 acc_tmp1;
   int32 acc_tmp2;

   acc_tmp1 = l_SRCFR;              // x
   acc_tmp2 = (1 << 24) + acc_tmp1; // 1.0 (1.24 format) + x

   for (i=0; i<3; i++)
   {
      acc_tmp2 = Mul32x32_sh24(acc_tmp1, acc_tmp2);
   }

   // Phase accumulator increment = (1/(1-x)) - 1
   acc_tmp2 -= (1<<24);

   return (int32)acc_tmp2;
}

