/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 1998-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *----------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Condfidential
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276-4000
 *   FAX   (781) 276-4001
 *
 *   cri_iof.c
 *
 *   Clock, Reset, Interrupt (CRI) block core interface functions
 *
 *
 *----------------------------------------------------------------------------
 */
// ***********************************************************************************************************
//  cri_iof.c
//
// History
//
// 01/02/2012 Kannan: Added function to clear the BC, Rx data status & Link status bits.
//                    Added another function to poll S_44K owner bit.
//                    Grep for "XDSLRTFW-710 VR9_VRX318_PDBRAM_Handover_OrderlyShutDown"
//
// ************************************************************************************************************

#include "common.h"
#include "cri_cocomo_interface.h"
#include "LL_IOf.h"
#include "gdata.h"
#include "cri_memmap.h"
#include "mtkernel.h"
#include "dsp_op.h"
#include "str_memmap.h"
#include "dsp_regs_62.h"
#include "delay.h"
#include "cmv.h"

#ifdef PPE_ENGINE
#include "ppe_memmap.h"
#endif //PPE_ENGINE


/*^^^
*------------------------------------------------------------------------
*
*  Name : ModifyTxInterruptReg
*
*  Description:    Enable or disable Tx Interrupt.
*
*  Prototype:  ModifyTxInterruptReg(int16 s_enable, uint32 ul_bitmask)
*
*
*  Input Arguments: s_enable -- 0 : disable interrupt, 1 : enable interrupt
*                   ul_bitmask -- interrupt source mask
*
*  Return: None
*
*  Global Variables Used: None
*
*------------------------------------------------------------------------
*^^^
*/

void ModifyTxInterruptReg(int16 s_enable, uint32 ul_bitmask)
{
   if (s_enable)
   {
      SetCoreReg(CRI_MASK0_TX_ADDR,  ul_bitmask );
   }
   else
   {
      ResetCoreReg(CRI_MASK0_TX_ADDR,  ul_bitmask );
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ModifyRxInterruptReg
 *
 *  Description:    Enable or disable Rx Interrupt.
 *
 *  Prototype:  ModifyRxInterruptReg(int16 s_enable, uint32 ul_bitmask)
 *
 *
 *  Input Arguments: s_enable -- 0 : disable interrupt, 1 : enable interrupt
 *                   ul_bitmask -- interrupt source mask
 *
 *  Return: None
 *
 *  Global Variables Used: None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ModifyRxInterruptReg(int16 s_enable, uint32 ul_bitmask)
{
   if (s_enable)
   {
      SetCoreReg(CRI_MASK0_RX_ADDR,  ul_bitmask );
   }
   else
   {
      ResetCoreReg(CRI_MASK0_RX_ADDR,  ul_bitmask );
   }
}

void DisableAllInterrupt(void)
{
   //Disable TSC interrupt
   WriteCoreReg(CRI_MASK0_RX_ADDR, 0);

   //No need to do this since the TX interrupt has never been enabled
   WriteCoreReg(CRI_MASK0_TX_ADDR, 0);
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EnableTxPms/DisableTxPms
 *
 *  Description: Enable/Disable TX PMS core
 *
 *  Prototype:  void EnableTxPms(void) or void DisableTxPms(void)
 *
 *
 *  Input Arguments: None
 *
 *  Return: None
 *
 *  Global Variables Used: None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EnableTxPms(void)
{
   //XDSLRTFW-2392 (Cascaded scheduling - Start)
   uint32 ul_data;
   ul_data = (gl_TxFcTimerCount | TX_TSC_START_TX_FC_SEL);
   if (guc_CasMode_enable)
   {
      ul_data = (gl_TxFcTimerCount | TX_TIMER_START_TX_FC_SEL);
   }
   WriteCoreReg(CRI_TX_PMS_CTL_ADDR, ul_data);
   //XDSLRTFW-2392 (Cascaded scheduling - End)
}


void DisableTxPms(void)
{
   //Disable TX PMS
   WriteCoreReg(CRI_TX_PMS_CTL_ADDR, 0x0);
}

//XDSLRTFW-2392 (Cascaded scheduling - Start)
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : StartRxPmsRegStart
 *
 *  Description: Start Rx PMS using Register start approach
 *
 *  Prototype:  void StartRxPmsRegStart(void)
 *
 *
 *  Input Arguments: None
 *
 *  Return: None
 *
 *  Global Variables Used: None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void StartRxPmsRegStart(void)
{
   SetCoreReg(CRI_RX_PMS_CTL_ADDR,  REGISTER_START_RX_FC_START );  //Give Go signal for Rx PMS processing
}
//XDSLRTFW-2392 (Cascaded scheduling - End)

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EnableRxPms / DisableRxPms
 *
 *  Description: Enable/Disable RX PMS core
 *
 *  Prototype:  void EnableRxPms(void) or void EnableRxPms(void)
 *
 *
 *  Input Arguments: None
 *
 *  Return: None
 *
 *  Global Variables Used: None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EnableRxPms(void)
{
   //XDSLRTFW-2392 (Cascaded scheduling - Start)
   uint32 ul_data = (gs_RxFcTimerCount | RX_TSC_START_RX_FC_SEL);
   if (guc_CasMode_enable)
   {
      //For cascaded interrupt approach select register start trigger
      ul_data = REGISTER_START_RX_FC_SEL;
   }
   WriteCoreReg(CRI_RX_PMS_CTL_ADDR, ul_data);
   //XDSLRTFW-2392 (Cascaded scheduling - End)
}


void DisableRxPms(void)
{
   //Disable RX PMS
   WriteCoreReg(CRI_RX_PMS_CTL_ADDR, 0x0);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EnableTxQtp /DisableTxQtp
 *
 *  Description: Enable/Disable TX Qtp core
 *
 *  Prototype:  void EnableTxPms(void) or void DisableTxPms(void)
 *
 *
 *  Input Arguments: None
 *
 *  Return: None
 *
 *  Global Variables Used: None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EnableTxQtp(void)
{
   //XDSLRTFW-2392 (Cascaded scheduling - Start)
   uint32 ul_data = (gl_TxQtpTimerCount | TX_TSC_START_TX_QTP_SEL);
   //VRX518_OPEN (check for VRX318 configuration;
   if (guc_CasMode_enable)
   {
      //For cascaded interrupt approach use the Tx timer to trigger Tx QTP after buffer swap occurs.
      ul_data = (gl_TxQtpTimerCount | TX_TIMER_START_TX_QTP_SEL);
   }
   //Enable TX QTP
   WriteCoreReg(CRI_TX_QTP_CTL_ADDR, ul_data);

}

void DisableTxQtp(void)
{
   //Disable TX QTP
   WriteCoreReg(CRI_TX_QTP_CTL_ADDR, 0x0);
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EnableRxQtp / DisableRxQtp
 *
 *  Description: Enable/Disable RX QTP core
 *
 *  Prototype:  void EnableRxQtp(void) / void DisableRxQtp(void)
 *
 *
 *  Input Arguments: None
 *
 *  Return: None
 *
 *  Global Variables Used: None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EnableRxQtp(void)
{
   uint32 ul_data = (gs_RxQtpTimerCount | RX_TSC_START_RX_QTP_SEL);
   if (guc_CasMode_enable)
   {
      //XDSLRTFW-2618 (Start)
      if (gul_35bLiteConfig & EN_HRT_5_0_SCHEDULING_FOR_CASCADED_MODE)
      {
         //For cascaded interrupt approach with HRT 5,0 use the Rx Timer to start Rx QTP
         //XDSLRTFW-3901 (START)
         //Issue
         //(1) Some of the HW resources like Inactive BAT and fine gain tables are
         //shared between DSP and QTP. When HW is accessing the tables, DSP is not
         //allowed to access those tables. Rx QTP was starting at 46K cycles from
         //Rx buffer swap event and is expected to run 17K cycles more, i.e Rx QTP
         //shall run up to 63K cycles (46 + 17 = 63K).
         //
         //(2) DSP was configured to use Tx Timer done interrupt to run Tx TC task
         //and is configured to raise the interrupt at 43K cycles from Tx buffer
         //swap event. Tx TC task is expected to run 7K cycles more, i.e Tx TC task
         //may run up to 50K cycles (43 + 7 = 50K).
         //
         //(3) Starting of Tx TC task may be delayed in case if Rx NTC is not completed
         //within the scheduled time and possibility that Rx QTP and Tx TC task
         //may access the Inactive tables at the same time.
         //Solution
         //(1) Starting of Rx QTP is changed from 46K to 36K and and Tx Timer done interrupt
         //is changed from 43K to 58K to avoid possible conflicts between Rx QTP and
         //DSP (Tx TC task)
         ul_data = (RX_TIMER_START_RX_QTP_SEL | 36000); //XDSLRTFW-3901 (START_END) //46000
         //XDSLRTFW-3901 (END)
      }
      else
      {
         //For cascaded interrupt approach use the Buffer swap as a trigger to start the Rx QTP for HRT != 5,0
         ul_data = RX_BUFFER_SWAP_START_RX_QTP_SEL;
      }
      //XDSLRTFW-2618 (END)

   }

   //Enable RX QTP
   WriteCoreReg(CRI_RX_QTP_CTL_ADDR, ul_data);
}

void DisableRxQtp(void)
{
   //Disable RX QTP
   WriteCoreReg(CRI_RX_QTP_CTL_ADDR, 0x0);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void StopTxDataPath(void)
*
*   This function disables TX data path.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void StopTxDataPath(void)
{
   uint32 ul_addr, ul_data;

   DisableTxPms();

   // XDSLRTFW-2778 (Start): Teardown procedure in VRX518
   if ((guc_Showtime_Entered & (MASK_BIT0 | MASK_BIT1)) != 0)  // XDSLRTFW-3591 (Start_End)
   {
      // DSL FW needs to set Bit 0 of DREG7 to inform PPE that Framer Request is turned off. This bit is cleared by PPE FW.
      // Please refer the document "PPE DSL Notifications" for more information.
      // Bit definition of 0x7DC7 (DREG_MISCRAM7_ADDR)
      //  Bit 0     --> Framer Request Stopped (Set by DSL FW and cleared by PPE FW)
      ul_addr = DREG_MISCRAM7_ADDR;
      ReadPpeReg(ul_addr, &ul_data);
      ul_data |= MASK_BIT0;
      WritePpeReg(ul_addr,ul_data);
      guc_Showtime_Entered = 0;
   }

   ul_addr = DREG_MISCRAM6_ADDR;
   ReadPpeReg(ul_addr, &ul_data);
   if (ul_data & (MASK_BIT0 | MASK_BIT1 | MASK_BIT2))
   {
      // Clear Bit 0 of DREG6 to indicate PPE that Framer Request is Off.
      // Bit 0 is set by the PPE driver indiacting that the Framer request can be turned on.
      // Please refer the document "PPE DSL Notifications" for more information.
      // Additionally Bit 1 is set by the PPE driver to indicate the that it supports PPE
      // frequency scaling and the frequency is configured as per the content of 0x7DCE (DREG_MISCRAM14_ADDR).
      // This bit (Bit 1 of 0x7DC6) is cleared when the Bit0 of 0x7DC6 (DREG_MISCRAM6_ADDR) is cleared.
      // Bit definition of 0X7DC6 (DREG_MISCRAM6_ADDR):
      //  Bit 0      --> Framer Request Enable (Set by the PPE driver and cleared by DSL FW)
      //  Bit 1      --> Dynamic Frequency supported (Set by the PPE driver and cleared by DSL FW)
      //  Bit 2      --> To indicate whether ERB descriptors are reset (Set by the PPE driver and cleared by DSL FW)
      ul_data &= ~(MASK_BIT0 | MASK_BIT1 | MASK_BIT2);      // XDSLRTFW-3711 (Start_End)
      WritePpeReg(ul_addr,ul_data);
   }
   // XDSLRTFW-2778 (End): Teardown procedure in VRX518
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void StopRxDataPath(void)
*
*   This function disables RX data path.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void StopRxDataPath(void)
{
   //Disable RX PMS
   DisableRxPms();

}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : SoftResetCores()
 *
 *  Description:   Reset all the cores
 *
 *  Prototype:  void ResetCores(void)
 *
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
void SoftResetCores(void)
{
   // have to manually disable each core for the port

   uint32 ul_RegAddr;

   for (ul_RegAddr = CRI_RXGP0_CTL_ADDR; ul_RegAddr <= CRI_RX_PMS_CTL_ADDR; ul_RegAddr += 4)
   {
      WriteCoreReg(ul_RegAddr, 0);
   }

   if(guc_ch_id == 0)
   {

#define RX_DSP_TS_PORT0_4K   (3)
      WriteCoreReg(CRI_RX_TSLOTC_A_ADDR, (RX_DSP_TS_PORT0_4K<<25));
   }
   else if(guc_ch_id == 1)
   {
#define RX_DSP_TS_PORT1_4K   (1)
      WriteCoreReg(CRI_RX_TSLOTC_A_ADDR, (RX_DSP_TS_PORT1_4K<<25));
   }

   //Enable DSP interrupt to TSC interrupt
   //TSC_RX_INT = 1, all the others are 0's
   WriteCoreReg(CRI_MASK0_RX_ADDR, CRI_MASK_RX_TSC_INT);

#define DIS_RXTSC_UPD   (0) // (do not disable TSC start assertion)
#define FRC_RXTSC_UPD   (1) // (constantly update)
#define DIS_TXTSC_UPD   (0) // (do not disable TSC start assertion)
#define FRC_TXTSC_UPD   (1) // (constantly update)
#define MASK1_TX_UPD    (3)   // (continuous update)
#define MASK0_TX_UPD   (3)   // (continuous update)
#define MASK1_RX_UPD    (3)   // (continuous update)
#define MASK0_RX_UPD   (3)   // (continuous update)

   WriteCoreReg(CRI_UPDCTL_ADDR, (DIS_RXTSC_UPD<<15)|(FRC_RXTSC_UPD<<14)|(DIS_TXTSC_UPD<<13)|(FRC_TXTSC_UPD<<12)|
                (MASK1_TX_UPD<<7)|(MASK0_TX_UPD<<5)|(MASK1_RX_UPD<<3)|(MASK0_RX_UPD<<1));

   //Disable CRI_TSC to clear CRI_TSC_STATUS register
   //(note: we cannot do this here since this may affect another port)
   //ResetCoreReg(CRI_TSC_CTRL_ADDR, (CRI_TSC_CTRL_ENABLE<<31));

   // enable and reconfig CRI_TSC
   WriteCoreReg(CRI_TSC_CTRL_ADDR, (CRI_TSC_CTRL_NUM_TS_4K | ((CRI_TSC_CTRL_NUM_CLKS-1)<<5) |
                                    (CRI_TSC_CTRL_CLKS_LAST<<22) | (1<<31)));
} //SoftResetCores()


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ReadTxTimer
 *
 *  Description:    Read the TX timer count
 *
 *  Prototype:  ReadTxTimer(uint32 *pl_timer)
 *
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *      pl_timer -- pointer to the TX timer count read from CRI_TX_TIMER register
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ReadTxTimer(uint32 *pl_timer)
{
   ReadCoreReg(CRI_TX_TIMER_ADDR, pl_timer);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ReadRxTimer
 *
 *  Description:    Read the RX timer count
 *
 *  Prototype:  ReadRxTimer(uint32 *pl_timer)
 *
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *      pl_timer -- pointer to the RX timer count read from CRI_RX_TIMER register
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *
 *  Notes: Initially the parameter was of type uint32 but to suppress/remove a warning
 *         it has been changed to type void. And suitable changes are made in the function
 *         and wherever it is being employed.
 *------------------------------------------------------------------------
 *^^^
 */
void ReadRxTimer(uint32 *pl_timer)
{
   ReadCoreReg(CRI_RX_TIMER_ADDR, pl_timer);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : GetTxRxCounters
 *
 *  Description:    Back to back reads of the Tx and Rx timer counts in order
 *               to measure the Tx/Rx frame alignment.  Assumes the Tx and Rx
 *               timers are set to rollover at the buffer swap boundary.
 *
 *               Note that this function should be called from a TC task to
 *               insure that no interrupt occurs between reads.
 *
 *  Prototype:  void GetTxRxCounters(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments (global variables):
 *      gs_TxTimerVal -- value read from Tx timer.
 *      gs_RxTimerVal -- value read from Rx timer.
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */

#if 1  //New code to get the TX/RX frame differece from Strymon register V_FRAME_SKEW_ADDR
void GetTxRxCounters(void)
{
   int32 l_temp1, l_temp2;

   ReadCoreReg(V_FRAME_SKEW_ADDR, &gl_txRxFrameOffset);
   if (gl_txRxFrameOffset & (int32)(1 << 24))
   {
      gs_DbgFrameSkewError = (gs_DbgFrameSkewError | (1<<gs_DbgFrameSkewErrorIndex));
      SetCoreReg(V_FRAME_SKEW_ADDR, (1 << 24));
   }
   gs_DbgFrameSkewErrorIndex++;

   // The rest of this function is not necessary.
   // But left in temporarily since they are logged and are useful for off-line verification.

   //Read TX timer
   ReadCoreReg(CRI_TX_TIMER_ADDR, &l_temp1);

   //Read RX timer
   ReadCoreReg(CRI_RX_TIMER_ADDR, &l_temp2);

   //Adjust the RX timer by the offset due two different readings of TX/RX register
   //(the number 5 is obtained by the measurment at 4 kHz frame rate)
   l_temp2 -= 5;

   gs_TxTimerVal = l_temp1;
   gs_RxTimerVal = l_temp2;
}
#else //Old code to get the TX/RX frame differece from TX and RX timer read
void GetTxRxCounters(void)
{
   int32 l_temp1, l_temp2, l_Acc;
   int16 s_exp, i;

   //Read TX timer
   ReadCoreReg(CRI_TX_TIMER_ADDR, &l_temp1);

   //Read RX timer
   ReadCoreReg(CRI_RX_TIMER_ADDR, &l_temp2);

   //Adjust the RX timer by the offset due two different readings of TX/RX register
   //(the number 5 is obtained by the measurment at 4 kHz frame rate)
   l_temp2 -= 5;

   //In the 6.2 HW, since the high speed clock rate is 2*AFE_CLK_FREQ, where AFE_CLK_FREQ 144000 KHz,
   //and the highest sampling rate is 35328 KHz (corresponding to 4096 tones),
   //thus the ratio of the high speed clock rate to the highest sampling rate is (2*144000/35328) = 8.152
   //In the 5.0 HW, this ratio is 3.
   //To avoid change the other parts of the code, we scale the gs_TxTimerVal and gs_RxTimerVal by (3/8.152)
   //so that their values also reflect the ratio of 3 between the clock rate and the sampling rate.

   //In the following, we multiply each timer value by (3/8.152) or divide them by (8.152/3)
   //represent (8.152/3) by 22261/(2^13)

#define SCALE_MANT   (22261)
#define SCALE_EXP   (13)

   i = 0;
   l_Acc = l_temp1;

_compute:
   //Normalize l_temp
   s_exp = norm_l(l_Acc);
   l_Acc = (l_Acc << s_exp);
   l_Acc = l_Acc/SCALE_MANT;

   //Compute the number of left shift
   s_exp = SCALE_EXP - s_exp;

   if(s_exp >= 0)
   {
      l_Acc <<= s_exp;
   }
   else
   {
      l_Acc >>= (-s_exp);
   }

   if(i == 0)
   {
      //Note the maximum timer value should be less or equal to 3*(8192+640) = 0x6780 (representable by 16-bit)
      gs_TxTimerVal = (int16)l_Acc;
      i = 1;
      l_Acc = l_temp2;
      goto _compute;
   }
   else
   {
      gs_RxTimerVal = (int16)l_Acc;
   }


}
#endif  //#if 1

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ClearTxStall
 *
 *  Description: Clear bit TX_QT_STALL_DROP in reg CRI_TX_QT_STALL
 *
 *  Prototype: void ClearTxStall(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments{
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *      None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ClearTxStall(void)
{
   SetCoreReg(CRI_TX_QT_STALL_ADDR, (1<<17));
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ClearRxStall
 *
 *  Description: Clear bit RX_QT_STALL_DROP in reg CRI_RX_QT_STALL
 *
 *  Prototype: void ClearRxStall(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments{
 *
 *  Return:
 *      None
 *
 *  Global Variables Used:
 *      None
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ClearRxStall(void)
{
   SetCoreReg(CRI_RX_QT_STALL_ADDR, (1<<17));
}


// this function needs to be in IRAM because by the time it is called by LinkStart,
// INIT_2 page is no longer in the memory.

//Mei debug
FlagT gft_IfftTimerStart = 1;

void InitCRI_ForLinkStart(void)
{
   //Configure TX QTP to tx time-slot start
   //TX_QTP_REG_START = 0
   //TX_QTP_START_SEL = 4
   //TX_QTP_GO_CNT = 0
   //WriteCoreReg(CRI_TX_QTP_CTL_ADDR, TX_TSC_START_TX_QTP_SEL);
   WriteCoreReg(CRI_TX_QTP_CTL_ADDR, 0);

   //Configure TX QT to tx time-slot start
   //TX_QT_REG_START = 0
   //TX_QT_START_SEL = 4
   //TX_QT_GO_CNT = 0
   WriteCoreReg(CRI_TX_QT_CTL_ADDR, TX_TSC_START_TX_QT_SEL);

   //Configure IFFT core
   if(gs_CRI_DbgCntl & CRI_DBG_CNTL_DISABLE_IFFT)
   {
      WriteCoreReg(CRI_TX_IFFT_CTL_ADDR, 0);
   }
   else
   {
      //Mei debug
      if(gft_IfftTimerStart == 0)
      {
         //Program IFFT to be triggered by TX QT done signal
         //TX_IFFT_REG_START = 0
         //TX_IFFT_START_SEL = 6 (tx_qt_done)
         //TX_IFFT_GO_CNT = 0
         WriteCoreReg(CRI_TX_IFFT_CTL_ADDR, TX_QT_DONE_START_TX_IFFT_SEL);
      }
      else
      {
         //Program IFFT to be triggered by timer-based method
         //TX_IFFT_REG_START = 0
         //TX_IFFT_START_SEL = 2 (timer-based)
         //TX_IFFT_GO_CNT = Predefined timer
         WriteCoreReg(CRI_TX_IFFT_CTL_ADDR, (TX_TIMER_START_TX_IFFT_SEL|TX_IFFT_START_TIMER_4K));
      }
   }

   if(gs_CRI_DbgCntl & CRI_DBG_CNTL_DISABLE_FFT)
   {
      //Configure RX FFT to disabled
      WriteCoreReg(CRI_RX_FFT_CTL_ADDR, 0);
   }
   else
   {
      //Configure RX FFT to rx buffer swap
      WriteCoreReg(CRI_RX_FFT_CTL_ADDR, RX_BUFFER_SWAP_START_RX_FFT_SEL);
   }

   if((gs_CRI_DbgCntl & CRI_DBG_CNTL_DISABLE_RXQT) == 0)
   {
      //Configure RX QT to rx time-slot start
      //RX_QT_REG_START = 0
      //RX_QT_START_SEL = 4
      //RX_QT_GO_CNT = 0
      WriteCoreReg(CRI_RX_QT_CTL_ADDR, RX_TSC_START_RX_QT_SEL);
   }
   else
   {
      WriteCoreReg(CRI_RX_QT_CTL_ADDR, 0);
   }

   //Configure RX QTP to rx time-slot start
   //RX_QTP_REG_START = 0
   //RX_QTP_START_SEL = 4
   //RX_QTP_GO_CNT = 0
   WriteCoreReg(CRI_RX_QTP_CTL_ADDR, RX_TSC_START_RX_QTP_SEL);

   //According to Strymon Spec (Initial FIFO Condition)
   //we need to follow the following procedure to enable QT stall and strymon
   //(Note: At this point, Strymon is held at reset mode)

   //First disable QT stall
   ResetCoreReg(CRI_TX_QT_STALL_ADDR, (1<<16));
   ResetCoreReg(CRI_RX_QT_STALL_ADDR, (1<<16));

   //Then enable QT stall if desired
   if(gft_EnableQtStall == TRUE)
   {
      SetCoreReg(CRI_TX_QT_STALL_ADDR, (1<<16));
      SetCoreReg(CRI_RX_QT_STALL_ADDR, (1<<16));
   }

} //void InitCRI_ForLinkStart(void)


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DFE_PPE_LinkStatus_TxRxLeaveShow()
*
*   This function notifies PPE that both TX and RX directions leave
*   showtime. (BIT0)
*
*   Input Arguments:
*
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

#ifdef PPE_ENGINE
void DFE_PPE_LinkStatus_TxRxLeaveShow(void)
{
   uint32 ul_addr, ul_data;

   // Clear Bit 0 of DREG0 to inform PPE that both Tx and Rx has left showtime.
   // Please refer the document "PPE DSL Notifications" for more information.
   // Bit defintion of DREG registers:
   // 0x7DC0 (DREG_MISCRAM0_ADDR)
   //  Bit 0      --> Set by DSL FW when both Tx and Rx enter showtime and cleared by DSL FW when showtime is left
   //  Bit 1      --> Set by DSL FW on first Rx showtime
   //  Bit 2      --> To indicate whether BEARER_CHANNEL is ON or OFF
   ul_addr = DREG_MISCRAM0_ADDR;
   ReadPpeReg(ul_addr, &ul_data);
   ul_data &=~MASK_BIT0;
   WritePpeReg(ul_addr, ul_data);
}
#endif


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DFE_PPE_RxDataStatus_LeaveShow(void)
*
*   This function notifies PPE that Zephyr Rx is disabled (bytes stop going to PPE).
*
*
*   Input Arguments:
*
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
#ifdef PPE_ENGINE
void DFE_PPE_RxDataStatus_LeaveShow(void)
{
   uint32 ul_addr, ul_data;

   // XDSLRTFW-2774 (Start)
   // Clear Bit 1 of DREG0 to inform PPE that both Tx and Rx has left showtime.
   // Please refer the document "PPE DSL Notifications" for more information.
   // Bit defintion of DREG registers:
   // 0x7DC0 (DREG_MISCRAM0_ADDR)
   //  Bit 0      --> Set by DSL FW when both Tx and Rx enter showtime and cleared by DSL FW when showtime is left
   //  Bit 1      --> Set by DSL FW on first Rx showtime
   //  Bit 2      --> To indicate whether BEARER_CHANNEL is ON or OFF
   ul_addr = DREG_MISCRAM0_ADDR ;
   ReadPpeReg(ul_addr, &ul_data);
   ul_data &=~MASK_BIT1;
   WritePpeReg(ul_addr, ul_data);
   // XDSLRTFW-2774 (End)
}
#endif

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DFE_PPE_Shared_PPE_Reg_Init(void)
*
*   This function initializes the DSL FW & PPE FW shared regsisters for both bearer channel.
*
*
*   Input Arguments:
*
*
*       Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void DFE_PPE_Shared_PPE_Reg_Init(void)
{
   uint32 ul_addr, ul_data;

   //BC0 related registers
   //Clear B0 ==> Tx & Rx showtime status
   //clear B1 ==> Rx Data status
   //Clear B2 ==> Selected BC0 indication
   ul_addr = DREG_MISCRAM0_ADDR;   //0x7DC0
   ReadPpeReg(ul_addr, &ul_data);
   ul_data &= (~(MASK_BIT2 | MASK_BIT1 | MASK_BIT0));
   WritePpeReg(ul_addr, ul_data);


   //BC1 related registers
   //Clear B0 ==> Tx & Rx showtime status
   //clear B1 ==> Rx Data status
   //Clear B2 ==> Selected BC0 indication
   ul_addr = DREG_MISCRAM16_ADDR;   //0x7DD0
   ReadPpeReg(ul_addr, &ul_data);
   ul_data &= (~(MASK_BIT2 | MASK_BIT1 | MASK_BIT0));
   WritePpeReg(ul_addr, ul_data);


}
//XDSLRTFW-710 VR9_VRX318_PDBRAM_Handover_OrderlyShutDown (END)


/*
** =============================================================================
** LOCAL-FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  Ppe_clockChange
**
** DESCRIPTION:   Set the clock bits of the PPE's CGU_FSR_OFFSET register.
**                In mode A (of DSL 34), We set the PPE32 clock frequnecy based on the
**                profile selected i.e, for 35 MHz line, we need a PPE32 clock
**                frequency of 576 MHz.And if the CPE is bonded one, we configure it
**                with a frequency of 576 MHZ. And for the legacy profiles (8,12 and 17),
**                we need the PPE32 clock frequency of 288 MHz.

**                In TEST Mode (DSL 15), if the trigger bit (Bit 4) is set, the
**                frequency configured in Bit 2-0 of DSL 15 is configured as PPE32
**                clock frequency. The mode selected in DSL 34 is ignored. The trigger
**                bit (Bit 4) is cleared at the end of GHs.
**
** PARAMETERS:    A 3 bit field for the clock and 1 Bit as a trigger.
**
** RETURN VALUE:  None
**
** NOTES:          Reference:
**
** =============================================================================
*/
// XDSLRTFW-3565 (Start)
void Ppe_clockChange(uint16 us_TC_Mode, uint16 us_ProfileSelected)
{
   uint32 ul_addr, ul_data = 0;
   ul_addr = DREG_MISCRAM14_ADDR;

   // Bit defintion of DREG register 0x7DCE (DREG_MISCRAM14_ADDR):
   //       Value (Bit1 - Bit0)                 PPE Frequency
   //             0                                576 MHz
   //             1                                494 MHz
   //             2                                432 MHz
   //             3                                288 MHz

   // Please refer MCAT 3.1 for the field definitions.
   // Please refer the concept document attached in the JIRA-3711 to find the frequencies to be used
   // across different profile, bonding capability and TC mode chosen.
   // If the trigger bit is set (in DSL 15), force the configured PPE32 frequency (Bit 2-Bit0).
   if (gus_PpeClockControl & TRIGGER_PPE_WRITE)
   {
      gus_PpeClockControl &= ~TRIGGER_PPE_WRITE;
      ul_data = (gus_PpeClockControl & PPE_CLOCK_FREQ_MASK) - 1;
   }
   else
   {
      // The Mode (DSL 34) is configured with value 1 by default.
      if (gus_PPEClockConfigure == FSCALE_AUTO_A)
      {
         if ((gt_Bonding_DiscAggr_Control.s_PCS_control & CNFG_PAF_ENABLE_MASK) ||
               (us_ProfileSelected == CNFG_V2_PROFILE_35B_MASK) ||
               (us_ProfileSelected == CNFG_V2_PROFILE_30A_MASK) ||
               (us_TC_Mode == TPS_TC_TYPE_ATM))
         {
            ul_data = (PPE_CLOCK_576MHz - 1);
         }
         else
         {
            ul_data = (PPE_CLOCK_288MHz - 1);
         }
      }
      else
      {
         // If the PPE Clock Frequency scaling feature is disabled, we need use the PPE clock frequency as
         // 576 MHz. Becuase, if we run at a lower clock frequency (say 288 or 432 MHz), we might see issues
         // in 35B or bonding mode of operation.
         ul_data = (PPE_CLOCK_576MHz - 1);
      }
   }
   WritePpeReg(ul_addr,ul_data);
}
// XDSLRTFW-3711 (End)
// XDSLRTFW-3565 (End)
