/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: show_iof.c
*
*   This file contains engine interface functions to used during showtime.
*
*-------------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// show_iof.c
//
// History
//
// 24/12/2012 Ram: Redefining FECS,ES,SES based on new definiton of CRC in ReTx mode of operation
//                 Refer Sec 11.3.1, 11.3.3 of G.998.4 (G.Inp/ReTx spec)
//                 Grep for 'XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode'
//
// 07/08/2015  ChihWen:
//             Move the Tx timer setup for QTP, and QT (30a RTX) from "Reconfig_CRI" to "StartTxDataPath"
//             because the setup will cause Strymon Tx overflows during training and lower DS rate.
//             Grep for XDSLRTFW-2307_Move_TxTimerSetup_30a_RTX
//
// 31/01/2018  Abu Rahman
//             XDSLRTFW-3688:Control of GPIO14 to indicate dying gasp for VDSL
//
//             In case of Dying Gasp set VRX518 GPIO pin 14 from high to low.
//             This feature is CMV controlled
//             CMV CNFG 58 bit CMV CNFG 58 Bit 8 (LSB = Bit0) = 1 Enable this feature
//             CMV CNFG 58 bit CMV CNFG 58 Bit 8 (LSB = Bit0) = 0 Disable this feature (default configuration)
//             Preferable configuration method for Enabling this feature: "dmms 3A48 0 1 0100 0100" at "wait for link activation" state
//
//             To control GPIO 14 first below mentioned registers need to be programmed
//             Register GPIO PAD PORTMUXC14(VRX518 address 0x24038) = 0
//             Register GPIO_FUNC DIRSET_0(VRX518 address 0x20048) bit14 value = 1
//             Register GPIO_FUNC OUTSET_0(VRX518 address 0x20040) bit14 value = 1
//
//             After detecting Dying Gasp
//             Register GPIO_FUNC OUTCLR_0(VRX518 address 0x20044) bit14 value = 1 will set
//             GPIO pin 14 voltage level low.
//             Check Jira for more information about register spec
//
//             Note: BAR registers can access only up to 64k from it's base address so that special care needed
//             to be taken while using BAR register to access data.
//
//             Grep pattern:  XDSLRTFW-3688
// ************************************************************************************************************

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "fifo.h"
#include "IRI_Iof.h"
#include "LL_IOf.h"
#include "cri_iof.h"
#include "alp_memmap.h"
#include "cri_memmap.h"
#include "ft_memmap.h"
#include "qt_memmap.h"
#include "qtp_memmap.h"
#include "zep_memmap.h"
#include "zep_memmap_cnfg.h"
#include "show_iof.h"
#include "OvhdMsg_IOf.h"
#include "cmv.h"
#include "profile.h"
#include "showtime.h"
#include "CustomerTasks.h"
#include "mul.h"
#include "vdsl_state.h"
#include "DshInterface.h"

#include "dsp_regs_62.h"
#include "codeswap.h"

#ifdef PPE_ENGINE
#include "ppe_memmap.h"
#endif //PPE_ENGINE

#ifdef ZEP_DEBUG_BLD
extern FlagT gft_EnableFifo3Capture;
#endif

extern uint8 guc_ReTX_CRC1_ReportTS;
extern uint16 gus_ReTX_crc1;

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DisablePrbsGen(void)
*
*   This function disables the PRBS generator.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void DisablePrbsGen(void)
{
   ResetCoreReg(IRI_QT_REG_TX_MISC_ADDR, MASK_BIT5);
   ResetCoreReg(IRI_QTP_REG_TX_CTRL_ADDR, MASK_BIT1);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void StartTxDataPath(void)
*
*   This function initiates TX data pump for Zephyr-FC TX path processing.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void StartTxDataPath(void)
{
   uint32 ul_data;

   //XDSLRTFW-2307_Move_TxTimerSetup_30a_RTX (START)
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_30A_MASK)
   {
      gl_TxFcTimerCount = CRI_TX_PMS_GO_CNT_8K_ST;
      gl_TxQtpTimerCount = CRI_TX_QTP_GO_CNT_8K_ST;
      gus_TxQtTimerCount = CRI_TX_QT_GO_CNT_8K_ST;
   }
   //XDSLRTFW-2307_Move_TxTimerSetup_30a_RTX (END)
   //XDSLRTFW-3901 (START)
   if (guc_CasMode_enable)
   {
      int32 l_cycles_offset;
      int32 l_cycles_1;
      int32 l_cycles_2;
      int32 l_TxFcTimerCount;
      int32 l_TxQtpTimerCount;

      //Read Rx Timer and Tx Timer and find out the offset between them and
      //add the offset to the initial start values Tx QTP, Tx PMS
      //to have fixed offset between Tx & Rx to start the HW cores.
      ReadRxTimer((uint32 *)(void *)&l_cycles_1);
      ReadTxTimer((uint32 *)(void *)&l_cycles_2);
      l_cycles_offset = l_cycles_2 - l_cycles_1;

      l_TxFcTimerCount = gl_TxFcTimerCount + l_cycles_offset;
      if (l_TxFcTimerCount < 0) l_TxFcTimerCount += gl_MaxRxTimerCnt;

      l_TxQtpTimerCount = gl_TxQtpTimerCount + l_cycles_offset;
      if (l_TxQtpTimerCount < 0) l_TxQtpTimerCount += gl_MaxRxTimerCnt;

      gl_TxFcTimerCount  = l_TxFcTimerCount ;
      gl_TxQtpTimerCount = l_TxQtpTimerCount;

      //Reconfig TX PMS with Tx & Rx buffer offset.
      WriteCoreReg(CRI_TX_PMS_CTL_ADDR, (gl_TxFcTimerCount | TX_TIMER_START_TX_FC_SEL));
      //Reconfig TX QTP with Tx & Rx buffer offset.
      WriteCoreReg(CRI_TX_QTP_CTL_ADDR, (gl_TxQtpTimerCount | TX_TIMER_START_TX_QTP_SEL));
   }
   else
   {
      //EnableTX PMS
      EnableTxPms();
   }
   //XDSLRTFW-3901 (END)

   //XDSLRTFW-2307_Move_TxTimerSetup_30a_RTX (START)
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_30A_MASK)
   {
      WriteCoreReg(CRI_TX_QTP_CTL_ADDR, (gl_TxQtpTimerCount | TX_TSC_START_TX_QTP_SEL));
      WriteCoreReg(CRI_TX_QT_CTL_ADDR, (gus_TxQtTimerCount | TX_TSC_START_TX_QT_SEL));
   }
   //XDSLRTFW-2307_Move_TxTimerSetup_30a_RTX (END)

   // enable TCM
   if (gft_TxTcmFlag)
   {
      //IT_TCM_CTRL
      SetCoreReg(IRI_QT_REG_TX_TCM_CTRL_ADDR, MASK_BIT25);
   }

   //Do not use referece gain, bit4 = 0
   //Disable PRBS Gen, bit5 = 0
   //Set INIT to 0 for showtime
   ul_data = MASK_BIT4 | MASK_BIT5 | MASK_BIT6;

   //IT_MISC
   ResetCoreReg(IRI_QT_REG_TX_MISC_ADDR, ul_data);

   // switch TX ping-pong table
   SwitchTxPingPongTable();

   DisablePrbsGen();

   //XDSLRTFW -1420
   //Dont reset PRBS at the end of every frame for monitor tones support
   //ResetCoreReg(IRI_QT_REG_TX_PRBS_MISC_ADDR,(uint16)MASK_BIT11);

#ifdef ZEP_DEBUG_BLD
   //Use the FIFO3 to capture the TX scrambler input data
   if((gft_EnableFifo3Capture) && (guc_PortMode == SINGLE_PORT_MODE))
   {
      EnableTxFifo3Capture();
   }
#endif

   // init gt_RrcStat
   //XDSLRTFW-2392 (Cascaded scheduling - Start)
   if ((gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED) && (guc_CasMode_enable == 0))
   {
      extern void InsertDsRRCToDTB(void);
      InsertDsRRCToDTB();
   }
   //XDSLRTFW-2392 (Cascaded scheduling - End)
}

#ifdef QT_LATCH_LP_BITS
void EnableTxQtLpBitsLatch(void)
{
   SetCoreReg(ZEP_REG_ZT_QTDTB_CONFIG_LP0_ADDR, (3<<30));
   SetCoreReg(ZEP_REG_ZT_QTDTB_CONFIG_LP1_ADDR, (3<<30));
}

void DisableTxQtLpBitsLatch(void)
{
   ResetCoreReg(ZEP_REG_ZT_QTDTB_CONFIG_LP0_ADDR, (3<<30));
   ResetCoreReg(ZEP_REG_ZT_QTDTB_CONFIG_LP1_ADDR, (3<<30));
}
#endif //#ifdef QT_LATCH_LP_BITS

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void StartRxDataPath(void)
*
*   This function initiates RX data pump for Zephyr-FC RX path processing.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void StartRxDataPath(void)
{
   uint32 ul_data;

   //============== Configure QT ===========================
   // enable TCM
   if(gft_RxTcmFlag)
   {
      //IR_TCM_CTRL
      SetCoreReg(IRI_QT_REG_RX_TCM_CTRL_ADDR, MASK_BIT25);

   }

   //IR_MISC
   ul_data = MASK_BIT5      //set to 0 to choose showtime sense RZ QAM decoder
             | MASK_BIT6      //enable TOT
             | MASK_BIT12   //enable gain scale caluclation
             | MASK_BIT14   //enable SFDQ calculation
             | MASK_BIT15;   //enable bit stuffer

   ResetCoreReg(IRI_QT_REG_RX_MISC_ADDR, ul_data);

#ifdef ENABLE_SHOW_LPBK
   // disable bit stuffer for DTB loopback test
   if (TESTArray[TEST_LoopbackMode] == TEST_LpbkAtDTB)
   {
      SetCoreReg(IRI_QT_REG_RX_MISC_ADDR, MASK_BIT15);
   }
#endif //#ifdef ENABLE_SHOW_LPBK

   // disable QTP's PRBS
   DisableRxPRBS();

   // switch RX ping-pong table
   SwitchRxPingPongTable();

#ifdef ZEP_DEBUG_BLD
   //Use the FIFO3 to capture the RX DeScrambler output data
   if((gft_EnableFifo3Capture) && (guc_PortMode == SINGLE_PORT_MODE))
   {
      EnableRxFifo3Capture();
   }
#endif

}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ReadCRCErrors(int16 s_lp)
*
*   This function reads CRC error counters from the Zephyr registers, and
*   increment error counters.
*
*   Input Arguments:
*      s_lp: 0 = INLV/LP0, 1 = FAST/LP1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void ReadCRCErrors(int16 s_lp)
{
   uint32 ul_data, ul_addr;
   uint16 us_err_cnt;

   // read CRC errors
   ul_addr = ZEP_PRAM_ZR_CRC_CTRL0_LP0_ADDR;
   if (s_lp != 0)
   {
      ul_addr = ZEP_PRAM_ZR_CRC_CTRL0_LP1_ADDR;
   }
   ReadCoreReg(ul_addr, &ul_data);


   us_err_cnt = (uint16)(ul_data >> 16) & 0xFFFF;
   ul_data &= (uint32)0x0000FFFF; // Clear CRC count

   if (s_lp == 1)
   {
      //CV counter has to be build as mentioned in chapter 11.3.1/G.998.4,
      //one or more uncorrected DTUs within a time window of 17 ms shall generate one CRC.
      if ( (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
         //&& (gft_DsUsReTxDiagEnter == FALSE) //XDSLRTFW-4044: expected "increase of ES, SES" when noise applied in ReTx TEST MODE also
         )//XDSLRTFW-1854 Feature_VDSL2_ALL_RTX_TESTMODE (Start_End)
      {
         us_err_cnt = gus_ReTX_crc1;
         gus_ReTX_crc1 =0;
      }
   }

   // accumulate counters
   {
      int16 s_transition;

      s_transition = 0;

      if (us_err_cnt > 0)
      {
         // reset counters
         WriteCoreReg(ul_addr, ul_data);

         if(!guc_EnableInhibition)
         {
            gt_g997_ChPerf_CRC_TR1_Curr_Prev_NE.t_Current.la_CV_C_count[s_lp] += (int16)us_err_cnt;
            gt_g997_ChPerf_CRC_Running_NE.la_CV_C_count[s_lp] += (int16)us_err_cnt;
         }
         else
         {
            gt_CRC_Curr_Reading.la_CV_C_count[s_lp] += (int16)us_err_cnt;
         }

         //This counter is always non-inhibiting
         gusa_NonInhibit_CRC_Cntr_NE[s_lp] += us_err_cnt;

         //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (Start)
         //ES counter has to be build as mentioned in chapter 11.3.1/G.998.4,
         //No defect, anamoly and failure are defined for LP0 (carrying ovhd chnl)
         if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
         {
            // Consider only for LP1
            us_err_cnt *= s_lp; //Works since s_lp is either 0 or 1;
         }
         //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (End)

         gt_g997_LinePerf_Sec_NE.l_ES_L_count += (int16)us_err_cnt;

         //gt_g997_LinePerf_help.l_1Sec_CRC8_L_count += (int16)us_err_cnt;
         MULU16(ul_data, us_err_cnt, gusa_DeltaCRCsecp_NormFactor[s_lp]);
         gt_g997_LinePerf_help.l_1Sec_CRC8_L_count += ul_data;

         // NE CRC debug trail code
         // Note: By adding an additional check with variable "gt_g997_LinePerf_Sec_NE.l_ES_L_count"
         //       it is possible to get a logging only once per second!
         {
            if (gt_TxIbData.uc_be_anom[s_lp] == TERMINATED)
            {
               // CRC on
               s_transition = ((DEBUG_TRAIL_SHOW_EVENTS_NE_CRC + s_lp) | 0x08);
            }
         }

         gt_TxIbData.uc_be_anom[s_lp] = PRESENT;
         guc_CrcOffHistory = 0;
      }
      else
      {
         // if no error
         guc_CrcOffHistory = ((guc_CrcOffHistory << 1) | 0x01);
         if (guc_CrcOffHistory >= 0x7)
         {
            // NE CRC debug trail code
            // Note: By adding an additional check with variable "gt_g997_LinePerf_Sec_NE.l_ES_L_count"
            //       it is possible to get a logging only once per second!
            {
               if (gt_TxIbData.uc_be_anom[s_lp] == PRESENT)
               {
                  // CRC off
                  s_transition = (DEBUG_TRAIL_SHOW_EVENTS_NE_CRC + s_lp);
               }
            }

            gt_TxIbData.uc_be_anom[s_lp] = TERMINATED;
         }
      }
#ifdef DEBUG_TRAIL
      // If NE CRC event logging enabled
      // Note: The function is called per OH period "PERp". This is done in fct. RxSuperFrameProcess()
      //       by checking the zephyr HW - ReadCoreReg( ul_zepPramZrR0, &ul_data) -> RX_SEQP_DONE0_BIT!

      if (gt_debugTrailControl.s_ShowtimeEventControl & DEBUG_TRAIL_SHOW_EVENTS_NE_CRC_ENABLE)
      {
         if(s_transition != 0)
         {
            // Log CRC events into ShowTime Debug Trail
            DebugTrail1(4,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
                        (int16)s_transition,
                        (int16)gl_RxSymbolCount,
                        (int16)(gl_RxSymbolCount>>16),
                        (int16)gusa_NonInhibit_CRC_Cntr_NE[s_lp]);
#ifdef DEBUG_TRACES
// #include "DebugBuffer.h"
//            {
//               int16 s_MsgRaw[2];
//               s_MsgRaw[0] = (int16) s_lp;
//               s_MsgRaw[1] = (int16) gusa_NonInhibit_CRC_Cntr_NE[s_lp];
//
//               DebugBuffWriteEv(DEBUG_EVT_CRC_NE_FE, sizeof(s_MsgRaw), s_MsgRaw);
//            }
#endif // DEBUG_TRACES
//         }
//         else
//         {
//            DebugTrail1(4,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
//                        (int16)s_transition,
//                        (int16)gl_RxSymbolCount,
//                        (int16)(gl_RxSymbolCount>>16),
//                        (int16)us_err_cnt,
//                        (int16)gusa_NonInhibit_CRC_Cntr_NE[s_lp]);
         }
      }
#endif // DEBUG_TRAIL
   }
}

/*
 *-------------------------------------------------------------------------------
 *
 *   Prototype: void ReadRSCorrections(int16 s_lp)
 *
 *   This function reads RS Corrections counters from the Zephyr registers, and
 *   increment correction counters.
 *
 *   Input Arguments:
 *      s_lp: 0 = INLV/LP0, 1 = FAST/LP1
 *
 *   Output Arguments:
 *
 *   Returns:
 *
 *   Global Variables:
 *
 *-------------------------------------------------------------------------------
 */
void ReadRSCorrections(int16 s_lp)
{
   uint32 ul_HwCntRd;
   uint32 ul_addr_CW, ul_addr;
   G997_NE_ChMonFEC_t *pFEC_TR1_Curr_Prev_NE_Cur = &gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current;

   if (s_lp == 0)
   {
      ul_addr = ZEP_PRAM_ZR_FEC_CNT_LP0_ADDR;
      ul_addr_CW = ZEP_PRAM_ZR_CW_COUNT_LP0_ADDR;
   }
   else
   {
      ul_addr = ZEP_PRAM_ZR_FEC_CNT_LP1_ADDR;
      ul_addr_CW = ZEP_PRAM_ZR_CW_COUNT_LP1_ADDR;
   }
   ReadCoreReg(ul_addr, &ul_HwCntRd);

   // accumulate corrected CW counters
   {
      int16 s_transition;
      uint32 ul_corr_cnt;

      ul_corr_cnt = (ul_HwCntRd >> 16) & (uint32)0x0000FFFF; // Number of corrected CW since last read

      s_transition = 0;
      if (ul_corr_cnt > 0)
      {
         if(!guc_EnableInhibition)
         {
            pFEC_TR1_Curr_Prev_NE_Cur->la_FEC_C_count[s_lp] += ul_corr_cnt;
//            gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FEC_C_count[s_lp] += ul_corr_cnt;
            gt_g997_ChPerf_FEC_Running_NE.la_FEC_C_count[s_lp] += ul_corr_cnt;

            //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (Start)
            //FECS counter has to be build as mentioned in chapter 11.3.1/G.998.4,
            //No defect, anamoly and failure are defined for LP0 (carrying ovhd chnl)
            if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
            {
               // Increment only for Lp1
               gt_g997_LinePerf_Sec_NE.l_FECS_L_count += s_lp; //Works since s_lp is either 0 or 1
            }
            else
            {
               gt_g997_LinePerf_Sec_NE.l_FECS_L_count++; // Independent of Latency Path
            }
            //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (End)
         }
         else
         {
            gul_FEC_Curr_Reading[s_lp] += ul_corr_cnt;
            gul_FECS_Curr_Reading ++;
         }

         //This counter is always non-inhibiting
         gusa_NonInhibit_FEC_Cntr_NE[s_lp] += ul_corr_cnt;

         // NE FEC debug trail code
         // Note: By adding an additional check with variable "gt_g997_LinePerf_Sec_NE.l_FECS_L_count"
         //       it is possible to get a logging only once per second!
         {
            if (gt_TxIbData.uc_fec_anom[s_lp] == TERMINATED)
            {
               // FEC on
               s_transition = ((DEBUG_TRAIL_SHOW_EVENTS_NE_FEC + s_lp) | 0x08);
            }
         }
         // Set up the indicator bit to show RS corrections (FEC)
         gt_TxIbData.uc_fec_anom[s_lp] = PRESENT;
      }
      else
      {
         // if no error

         // NE FEC debug trail code
         // Note: By adding an additional check with variable "gt_g997_LinePerf_Sec_NE.l_FECS_L_count"
         //       it is possible to get a logging only once per second!
         {
            if (gt_TxIbData.uc_fec_anom[s_lp] == PRESENT)
            {
               // FEC off
               s_transition = (DEBUG_TRAIL_SHOW_EVENTS_NE_FEC + s_lp);
            }
         }
         // Clear the indicator bit corresponding to RS corrections (FEC)
         gt_TxIbData.uc_fec_anom[s_lp] = TERMINATED;
      }
#ifdef DEBUG_TRAIL
      // If NE FEC event logging enabled
      // Note: The function is called per OH period "PERp". This is done in fct. RxSuperFrameProcess()
      //       by checking the zephyr HW - ReadCoreReg( ul_zepPramZrR0, &ul_data) -> RX_SEQP_DONE0_BIT!
      if (gt_debugTrailControl.s_ShowtimeEventControl & DEBUG_TRAIL_SHOW_EVENTS_NE_FEC_ENABLE)
      {
         if(s_transition != 0)
         {
            // Log CRC events into ShowTime Debug Trail
            DebugTrail1(4,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
                        (int16)s_transition,
                        (int16)gl_RxSymbolCount,
                        (int16)(gl_RxSymbolCount>>16),
                        (int16)gusa_NonInhibit_FEC_Cntr_NE[s_lp]);
         }
      }
#endif // DEBUG_TRAIL
   }

   // accumulate uncorrected CW counters
   {
      uint32 ul_uncorr_cnt;
      ul_uncorr_cnt = (ul_HwCntRd & 0x0000FFFF);               // Number of uncorrected CW since last read

      if (ul_uncorr_cnt > 0)
      {
         if(!guc_EnableInhibition)
         {
            pFEC_TR1_Curr_Prev_NE_Cur->la_FECV_C_count[s_lp] += ul_uncorr_cnt;
//            gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_FECV_C_count[s_lp] += ul_uncorr_cnt;
            gt_g997_ChPerf_FEC_Running_NE.la_FECV_C_count[s_lp] += ul_uncorr_cnt;
         }
         else
         {
            gul_FEC_Uncorr_Curr_Reading[s_lp] += ul_uncorr_cnt;
         }
      }
   }

   // reset counters
   WriteCoreReg(ul_addr, 0);

   // accumulate total CW count and reset counter
   {
      uint32 ul_total_cnt;

      ReadCoreReg(ul_addr_CW, &ul_total_cnt);
      WriteCoreReg(ul_addr_CW, 0);
      ul_total_cnt = ul_total_cnt & 0x0000FFFF; // Number of CodeWord since last read

      if(!guc_EnableInhibition)
      {
         pFEC_TR1_Curr_Prev_NE_Cur->la_CW_C_count[s_lp] += ul_total_cnt;
//         gt_g997_ChPerf_FEC_TR1_Curr_Prev_NE.t_Current.la_CW_C_count[s_lp] += ul_total_cnt;
         gt_g997_ChPerf_FEC_Running_NE.la_CW_C_count[s_lp] += ul_total_cnt;
      }
      else
      {
         gul_RS_CW_Curr_Reading[s_lp] += ul_total_cnt;
      }
   }
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetTxATMCellCnts(int16 s_abc)
*
*   This function reads TX ATM counters.
*
*   Input Arguments:
*      s_abc: ATM BC #, 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void GetPPETxAtmCnts(int16 s_abc, uint32 *pula_CntBuf);
void GetPPERxAtmCnts(int16 s_abc, uint32 *pula_CntBuf);

void GetTxATMCellCnts(int16 s_abc)
{
   uint32 ula_data[NUM_READ_TX_ATM_REGS];

#ifdef PPE_ENGINE
   if (gft_AlphaeusSel != TRUE)
   {
      //Get PPE TX count values
      GetPPETxAtmCnts(s_abc, ula_data);
   }
#endif //#ifdef PPE_ENGINE

   //Update Tx ATM Counters
   gt_g997_AtmPerf_UCELL_Running_NE.la_UCELL_P_count[s_abc] += ula_data[TX_CELL_CNT];
   gt_g997_AtmPerf_ICELL_Running_NE.la_ICELL_P_count[s_abc] += ula_data[TX_IDLE_CNT];

   //populate to new cmvs
   gt_g997_AtmPerf_TX_UICELL[s_abc].ul_UCELL_P_count += ula_data[TX_CELL_CNT];
   gt_g997_AtmPerf_TX_UICELL[s_abc].ul_ICELL_P_count += ula_data[TX_IDLE_CNT];

}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetRxATMCellCnts(int16 s_lp, int16 s_abc)
*
*   This function reads RX ATM counters.
*
*   Input Arguments:
*      s_lp: 0 = INLV/LP0, 1 = FAST/LP1
*      s_abc: ATM BC #, 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void GetRxATMCellCnts(int16 s_lp, int16 s_abc)
{
   uint32  ula_data[NUM_READ_RX_ATM_REGS];
   int32 l_temp_data;



#ifdef PPE_ENGINE
   if (gft_AlphaeusSel != TRUE)
   {
      //Get PPE RX count values
      GetPPERxAtmCnts(s_abc, ula_data);
   }
#endif //#ifdef PPE_ENGINE

   // Test code to create fake HEC errors
   if (gs_ALF_HEC_override_cnt[s_abc] > 0)
   {
      ula_data[RX_HEC_CNT] += 1;
      if (gs_ALF_HEC_override_cnt[s_abc] != 0x7fff)
      {
         gs_ALF_HEC_override_cnt[s_abc]--;
      }
   }

   //Update Rx ATM Counters
   //CU_P:      User Cells or all cells passed to the ATM layer.
   //         Therefore, CU_P = Register (RX_CELL_CNT) + Register (RX_AIDLE_CNT)
   //         Usually the RXKEEPIDLE flag =0 then Idle cells are dropped and ATM_IDLE_CNT = 0
   //CD_P:      Total Delineated Cells - are all cells passed through Alphaeus ie. all user cells
   //         plus all cells dropped due to Cell Buffer overflow or because RXKEEPIDLE flag = 0
   //         Therefore, CD_P = CU_P + Register(RX_CD_CNT)
   //HEC_P:   Cells with HEC errors and is found in Register(RX_HEC_CNT)
   //IBE_P:   Idle Cells with Bit Errors. If the flag RXKEEPIDLE=0 then all idle cells are dropped
   //         IBE_P = Register(RX_BE_CNT)
   //ICELL_NE:   Idle Cells dropped or passed to ATM layer depending on flag RXKEEPIDLE
   //         ICELL_NE = Register(RX_IDLE_CNT)
   //DCELL_NE:   Dropped cells = Register(Rx_CD_CNT) - Register(RX_IDLE_CNT) - Register(RX_HEC_CNT)
   //         Subtract Rx_IDLE_CNT only if RXKEEPIDLE flag != 0

   //HEC_P counter computation
   if (ula_data[RX_HEC_CNT] > 0)
   {
      gt_g997_AtmPerf_HEC_Running_NE.la_HEC_P_count[s_abc] += ula_data[RX_HEC_CNT];
      gt_g997_AtmPerf_HEC_Curr_Prev_TR1_NE.t_Current.la_HEC_P_count[s_abc] += ula_data[RX_HEC_CNT];
      gt_g997_AtmPerf_NE[s_abc].ul_HEC_P_count += ula_data[RX_HEC_CNT]; //near end HEC error count
   }

   //CU_P counter computation
   l_temp_data = ula_data[RX_CELL_CNT] + ula_data[RX_AIDLE_CNT];
   gt_g997_AtmPerf_CU_Running_NE.la_CU_P_count[s_abc] +=l_temp_data; //CU_P count
   gt_g997_AtmPerf_CU_Curr_Prev_TR1_NE.t_Current.la_CU_P_count[s_abc] += l_temp_data;
   gt_g997_AtmPerf_NE[s_abc].ul_CU_P_count += l_temp_data;;  //near end CU_P count

   //CD_P counter computation
   gt_g997_AtmPerf_CD_Running_NE.la_CD_P_count[s_abc] += l_temp_data + ula_data[RX_CD_CNT];
   gt_g997_AtmPerf_CD_Curr_Prev_TR1_NE.t_Current.la_CD_P_count[s_abc] += l_temp_data + ula_data[RX_CD_CNT];
   gt_g997_AtmPerf_NE[s_abc].ul_CD_P_count += l_temp_data + ula_data[RX_CD_CNT];   //near end CD_P count

   //IBE_P counter computation
   if (ula_data[RX_BE_CNT] > 0)
   {
      gt_g997_AtmPerf_IBE_Running_NE.la_IBE_P_count[s_abc] += ula_data[RX_BE_CNT];
      gt_g997_AtmPerf_IBE_Curr_Prev_TR1_NE.t_Current.la_IBE_P_count[s_abc] += ula_data[RX_BE_CNT];
      gt_g997_AtmPerf_NE[s_abc].ul_IBE_P_count += ula_data[RX_BE_CNT];  //near end idle cell w/ bit error count
   }

   //ICELL counter computation
   gt_g997_AtmPerf.la_ICELL_count[s_abc] += ula_data[RX_IDLE_CNT];

   //DCELL counter computation
   l_temp_data = ula_data[RX_CD_CNT] - ula_data[RX_IDLE_CNT] - ula_data[RX_HEC_CNT];
   gt_g997_AtmPerf.la_DCELL_count[s_abc] += l_temp_data;

   if (ula_data[RX_HEC_CNT] > 0)
   {
      gt_TxIbData.uc_hec_anom[s_lp] = PRESENT;
   }
   else
   {
      gt_TxIbData.uc_hec_anom[s_lp] = TERMINATED;
   }

   //XDSLRTFW-521 NewFeatures_ALL_ALL_ALL_ATMVDSL2 (Start)
   //DREG_AR_STAT0 :
   //Bit 1 :
   //SYNC Loss Indicator
   //Indicator of ATM or EFM delineation status. The DSP should read this bit,
   //then write a '0' to restart the accumulation record.
   //0 Bit - delineation kept cell/codeword synchronization was maintained.
   //1 Bit - delineation lost cell/codeword synchronization was lost.
   if (ula_data[RX_ATM_STAT] & MASK_BIT1)
   {
      gt_TxIbData.uc_cd_flag[s_lp] = FALSE;
   }
   else
   {
      gt_TxIbData.uc_cd_flag[s_lp] = TRUE;
   }
   //XDSLRTFW-521 NewFeatures_ALL_ALL_ALL_ATMVDSL2 (End)

}


#ifdef PPE_ENGINE

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void CompWrapAroundCounterDiff(uint32 *pul_pre_data, uint32 *pul_curr_data)
*
*   Compute the difference between two wrap around counters and copy
*   the current counter to the previous count
*   ul_diff = (MAX_32BITS_COUNTER + (ul_curr_data - ul_pre_data)) mod MAX_32BITS_COUNTER
*
*   Input Arguments:
*      *pul_pre_data -- the previous count value
*      *pul_curr_data -- the current count value
*
*   Output Arguments:
*
*   Returns:
*      the difference between two counters
*
*-------------------------------------------------------------------------------
*/

#define MAX_32BITS_COUNTER   (0xFFFFFFFF)

uint32 CompWrapAroundCounterDiff(uint32 *pul_pre_data, uint32 *pul_curr_data)
{
   int32 ul_diff;
   uint32 ul_pre_data, ul_curr_data;

   ul_pre_data = *pul_pre_data;
   ul_curr_data = *pul_curr_data;

   if(ul_curr_data >= ul_pre_data)
   {
      ul_diff = ul_curr_data - ul_pre_data;
   }
   else
      //Taking care of wrap around by computing ul_data-ul_pre_data + 0xFFFFFFFF
   {
      ul_diff = MAX_32BITS_COUNTER - ul_pre_data + ul_curr_data;
   }

   *pul_pre_data = ul_curr_data;

   return(ul_diff);
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetPPETxAtmCnts(int16 s_abc, uint32 *pula_CntBuf)
*
*   This function reads PPE TX ATM counters
*
*   Input Arguments:
*      s_abc: the bearer channel number: 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*      pula_CntBuf -- pointer to the array containing the register values.
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void GetPPETxAtmCnts(int16 s_abc, uint32 *pula_CntBuf)
{
   uint8 uc_offset;
   uint32 ul_addr, ul_data;

   //Compute the register address offset
   uc_offset = (uint8)((s_abc + +guc_ch_id)<<2);

   //Compute the number of non-idle cells -----------------------
   ul_addr = DREG_AT_CELL0_ADDR + uc_offset;
   ReadCoreReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[TX_CELL_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AT_CELL_Shadow[s_abc], &ul_data);

   //Compute the number of idle cells -----------------------
   ul_addr = DREG_AT_IDLE_CNT0_ADDR + uc_offset;
   ReadCoreReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[TX_IDLE_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AT_IDLE_CNT_Shadow[s_abc], &ul_data);

} //GetPPETxAtmCnts()


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetPPERxAtmCnts(int16 s_abc, uint32 *pula_CntBuf)
*
*   This function reads PPE RX ATM counters
*
*   Input Arguments:
*      s_abc: the bearer channel number: 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*      pula_CntBuf -- pointer to the array containing the register values.
*
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void GetPPERxAtmCnts(int16 s_abc, uint32 *pula_CntBuf)
{
   uint8 uc_offset;
   uint32 ul_addr, ul_data;

   //Compute the register address offset
   uc_offset = (uint8)((s_abc +guc_ch_id)<<2);

   // Check PPE RX Status Register ----------------------------
   ul_addr = DREG_AR_STAT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   pula_CntBuf[RX_ATM_STAT] = ul_data;

   //Clear the status register
   WritePpeReg(ul_addr, 0);

   //Compute the number of user cells -------------------
   ul_addr = DREG_AR_CELL0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_CELL_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CELL_Shadow[s_abc], &ul_data);

   //Compute the number of idle cells ---------------------
   ul_addr = DREG_AR_IDLE_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_IDLE_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_IDLE_CNT_Shadow[s_abc], &ul_data);

   //Compute the passed idle ATM cells -------------------
   ul_addr = DREG_AR_AIIDLE_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_AIDLE_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_AIIDLE_CNT_Shadow[s_abc], &ul_data);

   //Compute the number of cells with bit error ------------
   ul_addr = DREG_AR_BE_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_BE_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_BE_CNT_Shadow[s_abc], &ul_data);

   //Compute the number of cells with HEC error -------------
   ul_addr = DREG_AR_HEC_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_HEC_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_HEC_CNT_Shadow[s_abc], &ul_data);

   //Compute the number of dropped cells --------------------
   ul_addr = DREG_AR_CD_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_CD_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CD_CNT_Shadow[s_abc], &ul_data);

} //void GetPPERxAtmCnts(int16 s_lp, int16 s_abc)


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetPPETxPtmCnts(int16 s_abc, uint32 *pula_CntBuf)
*
*   This function reads PPE TX PTM counters
*
*   Input Arguments:
*      s_abc: the bearer channel number: 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*      pula_CntBuf -- pointer to the array containing the register values.
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void GetPPETxPtmCnts(int16 s_abc, uint32 *pula_CntBuf)
{
   uint8 uc_offset;
   uint32 ul_addr, ul_data;

   //Compute the register address offset
   uc_offset = (uint8)((s_abc + guc_ch_id) <<2);

   //Compute the number of non-idle cells -----------------------
   ul_addr = DREG_AT_CELL0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[TX_CW_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AT_CELL_Shadow[s_abc], &ul_data);

} //GetPPETxAtmCellCnts()

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetPPERxPtmCnts(int16 s_abc, uint32 *pula_CntBuf)
*
*   This function reads PPE RX PTM counters
*
*   Input Arguments:
*      s_abc: the bearer channel number: 0 = ABC0, 1 = ABC1
*
*   Output Arguments:
*      pula_CntBuf -- pointer to the array containing the register values.
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void GetPPERxPtmCnts(int16 s_abc, uint32 *pula_CntBuf)
{
   uint8 uc_offset;
   uint32 ul_addr, ul_data;

   //Compute the register address offset
   uc_offset = (uint8)((s_abc +guc_ch_id)<<2);

   // Check PPE RX Status Register ----------------------------
   ul_addr = DREG_AR_STAT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   pula_CntBuf[RX_PTM_STAT] = ul_data;

   //Clear the status register
   WritePpeReg(ul_addr, 0);

   //Compute the number of non-idle cells -----------------------
   ul_addr = DREG_AR_CELL0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Compute the counter difference between the previous read and current read
   pula_CntBuf[RX_CW_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CELL_Shadow[s_abc], &ul_data);

   //Compute the number of CRC errors for normal flow -----------------------
   ul_addr = DREG_AR_CERRN_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Clear this count if it satures
   if(ul_data == MAX_32BITS_COUNTER)
   {
      WritePpeReg(ul_addr, 0);
      gula_DREG_AR_CERRN_CNT_Shadow[s_abc] = 0;
      pula_CntBuf[RX_CERRN_CNT] = 0;
   }
   else
   {
      //Compute the counter difference between the previous read and current read
      pula_CntBuf[RX_CERRN_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CERRN_CNT_Shadow[s_abc], &ul_data);
   }


   //Compute the number of CRC errors for preemption flow -----------------------
   ul_addr = DREG_AR_CERRNP_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Clear this count if it satures
   if(ul_data == MAX_32BITS_COUNTER)
   {
      WritePpeReg(ul_addr, 0);
      gula_DREG_AR_CERRNP_CNT_Shadow[s_abc] = 0;
      pula_CntBuf[RX_CERRNP_CNT] = 0;
   }
   else
   {
      //Compute the counter difference between the previous read and current read
      pula_CntBuf[RX_CERRNP_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CERRNP_CNT_Shadow[s_abc], &ul_data);
   }

   //Compute the number of code violations for normal flow -----------------------
   ul_addr = DREG_AR_CVN_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Clear this count if it satures
   if(ul_data == MAX_32BITS_COUNTER)
   {
      WritePpeReg(ul_addr, 0);
      gula_DREG_AR_CVN_CNT_Shadow[s_abc] = 0;
      pula_CntBuf[RX_CVN_CNT] = 0;
   }
   else
   {
      //Compute the counter difference between the previous read and current read
      pula_CntBuf[RX_CVN_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CVN_CNT_Shadow[s_abc], &ul_data);
   }

   //Compute the number of code violations for preemption flow -----------------------
   ul_addr = DREG_AR_CVNP_CNT0_ADDR + uc_offset;
   ReadPpeReg(ul_addr, &ul_data);

   //Clear this count if it satures
   if(ul_data == MAX_32BITS_COUNTER)
   {
      WritePpeReg(ul_addr, 0);
      gula_DREG_AR_CVNP_CNT_Shadow[s_abc] = 0;
      pula_CntBuf[RX_CVNP_CNT] = 0;
   }
   else
   {
      //Compute the counter difference between the previous read and current read
      pula_CntBuf[RX_CVNP_CNT] = CompWrapAroundCounterDiff(&gula_DREG_AR_CVNP_CNT_Shadow[s_abc], &ul_data);
   }

} //GetPPERxPtmCnts()

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetTxPTMCnts(int16 s_abc)
*
*   This function reads Tx PTM counters.
*
*   Input Arguments:
*
*      s_abc: PTM BC #, 0 = ABC0, 1 =ABC1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/


void GetTxPTMCnts(int16 s_abc)
{
   uint32 ula_data[NUM_READ_TX_PTM_REGS];

   GetPPETxPtmCnts(s_abc, ula_data);

//   gt_g997_PtmPerf_Running_NE.ula_tx_pkt_cnt[s_abc] += ula_data[TX_CW_CNT];
//   gt_g997_PtmPerf_Cur_Prev_NE.t_Current.ula_tx_pkt_cnt[s_abc] += ula_data[TX_CW_CNT];

}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void GetRxPTMCnts(int16 s_abc)
*
*   This function reads Rx PTM counters.
*
*   Input Arguments:
*
*      s_abc: PTM BC #, 0 = ABC0, 1 =ABC1
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void GetRxPTMCnts(int16 s_lp, int16 s_abc)
{

   uint32 ula_data[NUM_READ_RX_PTM_REGS];


   //Get PPE RX count values
   GetPPERxPtmCnts(s_abc, ula_data);

   //Update the PTM counter CMVs (CRC and CVC)
   gt_g997_PtmPerf_NE[s_abc].ul_CRC_P_cnt += ula_data[RX_CERRN_CNT];   //CRC_P
   gt_g997_PtmPerf_NE[s_abc].ul_CVC_P_cnt += ula_data[RX_CVN_CNT];     //CVC_P
   gt_g997_PtmPerf_NE[s_abc].ul_CRC_NP_cnt += ula_data[RX_CERRNP_CNT]; //CRC_NP
   gt_g997_PtmPerf_NE[s_abc].ul_CVC_NP_cnt += ula_data[RX_CVNP_CNT];   //CVC_NP

   //Set the SYNC Loss Indicator
   //Debug Trigger for TC SYNC LOSS: XDSLRTFW-2094/2050
   if (ula_data[RX_PTM_STAT] & MASK_BIT1)
   {
      gt_TxIbData.uc_cd_flag[s_lp] = FALSE;
      gusa_nSyncLoss[s_lp] = gusa_nSyncLoss[s_lp] + 1;   // XDSLRTFW-2050: DEBUG_TC_SYNC_LOSS_COUNTE
   }
   else
   {
      gt_TxIbData.uc_cd_flag[s_lp] = TRUE;
   }
   //Test TC No Sync
   //XDSLRTFW-2050
   if (TESTArray[TEST_TCNOSYNC] == 1)
   {
      gt_TxIbData.uc_cd_flag[s_lp] = FALSE;
   }
   //Test TC No Sync

}

#endif //PPE_ENGINE

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void TxSuperFrameProcess(void)
*
*   This function processes a Tx super frame.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void TxSuperFrameProcess(void)
{
   uint32 ul_data;

   // check if this frame is the end of superframe for INLV path
   ReadCoreReg(ZEP_PRAM_ZT_R0_LP0_ADDR, &ul_data);

   if (ul_data & TX_SEQP_DONE0_BIT)
   {
      // perform superframe task for a given path
      TxSuperFrameTaskPerPath(LP0);

      // clear superframe done bit
      ResetCoreReg(ZEP_PRAM_ZT_R0_LP0_ADDR, (uint32)TX_SEQP_DONE0_BIT);
   }

   // check if this frame is the end of superframe for FAST path
   ReadCoreReg(ZEP_PRAM_ZT_R0_LP1_ADDR, &ul_data);
   if (ul_data & TX_SEQP_DONE1_BIT)
   {
      // perform superframe task for a given path
      TxSuperFrameTaskPerPath(LP1);

      // clear superframe done bit
      ResetCoreReg(ZEP_PRAM_ZT_R0_LP1_ADDR, (uint32)TX_SEQP_DONE1_BIT);
   }

} //void TxSuperFrameProcess(void)


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void TxSuperFrameTaskPerPath(int16 s_lp)
*
*   This function performs TX superframe tasks for a given latency path.
*
*   Input Arguments:
*      s_lp: latency path #
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void TxSuperFrameTaskPerPath(int16 s_lp)
{
   int16 s_abc;

   // obtain ATM BC # assigned to a given path
   s_abc = gt_tx_TPS_Map.s_LPtoABC[s_lp];

   if (s_abc != DISABLED_ABC)
   {
      if(gft_TcTypeSelected == TPS_TC_TYPE_ATM)
         // get ATM statistics
      {
         GetTxATMCellCnts(s_abc);
      }
#ifdef PPE_ENGINE
      else
         // get PTM statistics
      {
         GetTxPTMCnts(s_abc);
      }
#endif //#ifdef PPE_ENGINE

   } //if (s_abc != DISABLED_ABC)

} //TxSuperFrameTaskPerPath()

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void RxSuperFrameProcess(void)
*
*   This function processes RX super frame.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void RxSuperFrameProcess(void)
{
   uint32 ul_data;

   //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (Start)
   int s_lp, s_lp_temp;
   //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (End)

   uint32 ul_zepPramLPOffset = ZEP_PRAM_ZR_R0_LP1_ADDR - ZEP_PRAM_ZR_R0_LP0_ADDR;
   uint32 ul_zepPramZrR0 = ZEP_PRAM_ZR_R0_LP0_ADDR;


   // check if this frame is the end of superframe for INLV path
   for (s_lp = 0; s_lp <= 1 ; s_lp++)
   {
      ReadCoreReg( ul_zepPramZrR0, &ul_data);
      if (ul_data & RX_SEQP_DONE0_BIT)  // note it is bit 0 for both Lp.
      {
         //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (Start)
         // perform superframe task for a given path
         s_lp_temp = s_lp;

         // for retx, LP1 link status will be collected besides those of LP0
         // this block of code will have to be revisited when true dual latency is implemented
         if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
         {
            //First LP1 and then LP0
            s_lp_temp = !s_lp_temp;
         }
         RxSuperFrameTaskPerPath((int16)s_lp_temp);
         //XDSLRTFW-419: BugFix_DS_VDSL2_ALL_Redfine_FECS_ES_SES_In_ReTx_Mode (End)

         // clear superframe done bit
         ResetCoreReg(ul_zepPramZrR0, (uint32)RX_SEQP_DONE0_BIT);
      }
      ul_zepPramZrR0 += ul_zepPramLPOffset;
   }

   //CV counter has to be build as mentioned in chapter 11.3.1/G.998.4,
   //one or more uncorrected DTUs within a time window of 17 ms shall generate one CRC.
   if (guc_ReTX_CRC1_ReportTS == 0)
   {
      if (gt_ReTXStats.ul_UncorrectedDtuCnt != gt_ReTXStats.ul_UncorrectedDtuCnt_Prev)
      {
         gt_ReTXStats.ul_UncorrectedDtuCnt_Prev = gt_ReTXStats.ul_UncorrectedDtuCnt;
         gus_ReTX_crc1 ++;
      }
   }
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void RxSuperFrameTaskPerPath(int16 s_lp)
*
*   This function performs RX superframe tasks for a given latency path.
*
*   Input Arguments:
*      s_lp: latency path #
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void RxSuperFrameTaskPerPath(int16 s_lp)
{
   int16 s_abc;

//#ifdef RX_CRC_SCRAMBLE_DISABLE
   if (!(gul_dbgShowtimeControl & RX_SET_FC_REGISTER_START))
//#endif
   {
      // read & accumulate CRC error counts
      ReadCRCErrors(s_lp);

      // read & accumulate RS correction counts
      ReadRSCorrections(s_lp);
   }

   s_abc = gt_rx_TPS_Map.s_LPtoABC[s_lp];

   if (s_abc != DISABLED_ABC)
   {
      if(gft_TcTypeSelected == TPS_TC_TYPE_ATM)
      {
         // get ATM statistics
         GetRxATMCellCnts(s_lp, s_abc);
      }

#ifdef PPE_ENGINE
      else
      {
         // get PTM statistics
         GetRxPTMCnts(s_lp, s_abc);
      }
#endif //#ifdef PPE_ENGINE

   } //if (s_abc != DISABLED_ABC)

} //RxSuperFrameTaskPerPath()

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DisableRxPRBS(void)
*
*   This function disables the Rx PRBS state generator
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:

*-------------------------------------------------------------------------------
*/
void DisableRxPRBS(void)
{
   // disable PRBS bit in QTP
   ResetCoreReg(IRI_QTP_REG_RX_CTRL_ADDR, MASK_BIT1);
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: int16 CheckForLowPower(void)
*
*      Poll HW voltage monitor to detect LPR condition.  This is a
*   dummy function in Cocomo, though the gft_ForceLPR flag can be
*   used to force the LPR condition.
*
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*         0: Power OK
*         1: Low Power Indicator
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 CheckForLowPower(void)
{

   uint16 us_lowpower;
   uint32 ul_data;

   volatile uint32 ul_RegAddress;
   uint32 ul_AddressPointedByBAR18;

   // Bit 15 of GPIO is a Low Power (LPR) Indicator (active low).
   // convert indicator to active high and return value.
   // Active high indicator is returned if LPR is present or,
   // for testing purposes, if ForceLPR flag is set.
   ul_data =GetDspReg(DSP_GPIO_CTRL);

   // Bit15 Controls direction of each individual GPIO pin. When a bit is '0', the corresponding GPIO pin is configured as an input
   ul_data &= ~(1<<15);
   //program Bit15 in the ctrl register
   SetDspReg(DSP_GPIO_CTRL, ul_data);

   //returns the state of the  GPIO pin 15 for low power mode.
   ul_data =GetDspReg(DSP_GPIO_DATA);
   us_lowpower = (ul_data &(1<<15))>>15;

   if ((us_lowpower == gus_LowPowerSetting ) || (gus_ForceLPR > 0) || (gus_SetLPR == 1) )
   {
      if (gus_ForceLPR > 0)
      {
         gus_ForceLPR--;
      }

      // XDSLRTFW-3688 :Control of GPIO14 to indicate dying gasp (Start)
      if (gsa_Cnfg_Misc[0] & CNFG_MISC_VRX518_ENABLE_GPIO14_FOR_DYING_GASP)
      {
         // VRX518 CHIPID_EFUSE (relative) base address which is stored in first 20 bits of DSP_XMEM_BAR18
         // address location. 12 MS bits are used for PCIe_BaseAddress
         // all the address access via BAR18 register must be located within 64K byte address range from it's base address
         ul_AddressPointedByBAR18 = (*((volatile uint32 *)DSP_XMEM_BAR18)) & 0xFFFFF;

         // note1: memory locations below is located outside ARC memory region. According to coding guideline
         // these memory locations must be accessed via DMA!
         // This memory is needed to be accessed during Dying Gasp. Before power goes down a very small time
         // will be available so the this memory is accessed via direct access.
         // !! These memory access is an exception!!

         // Note2: address offset range is checked intentionally. If there is any problem in address offset renge
         // then it would be caught in InitVRX518Regs_for_Gpio14() function.


          // Set VRx518 GPIO14 pin voltage from high to low
         ul_RegAddress = (uint32)(&__StartOfSramBAR18) + (VRX518_GPIO_FUNC_OUTCLR_0_ADDRESS - ul_AddressPointedByBAR18);
         ul_data = *(volatile uint32 *)ul_RegAddress;
         ul_data |= 0x4000; // set bit 14 to 1
         *(volatile uint32 *)ul_RegAddress = ul_data;
      }
      // XDSLRTFW-3688 :Control of GPIO14 to indicate dying gasp (End)


      return(1);
   }
   else
   {
      return(0);
   }

}


#ifdef DEBUG_SHOWTIME

#define NUM_SHOW_FRAME_TO_LOG (10)
#define NUM_HEADER_SIZE_PER_FRAME (5)
#define NUM_32BIT_WORDS_PER_FRAME (20) // based on 32-bit word

int16 gs_LogShowBuf = 0; // 0: disabled, 1: TxDTB/RxDTB, 2: IFFT/FFT
int16 gs_debug_tx_cnt = 0;
int16 gs_debug_rx_cnt = 0;
int32 gla_tx_buf[(NUM_32BIT_WORDS_PER_FRAME+NUM_HEADER_SIZE_PER_FRAME)*NUM_SHOW_FRAME_TO_LOG];
int32 gla_rx_buf[(NUM_32BIT_WORDS_PER_FRAME+NUM_HEADER_SIZE_PER_FRAME)*NUM_SHOW_FRAME_TO_LOG];

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void LogTxBufAtTxQtTc(void)
*
*   This function logs TxDTB buffer or IFFT buffer with various symbol counters
*   at TX-QT TC task. It may log small part of buffer with a few consecutive
*   frames so that it does not interfere normal showtime processing.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void LogTxBufAtTxQtTc(void)
{
   int16 s_offset;
   int32 ul_IFFTInputBuffer, ul_IFFTOutputBuffer;
   uint32 ul_tx_addr;

   if (gs_LogShowBuf == 0)
   {
      return;
   }

   if (gs_debug_tx_cnt < NUM_SHOW_FRAME_TO_LOG)
   {
      s_offset = gs_debug_tx_cnt*(NUM_32BIT_WORDS_PER_FRAME+NUM_HEADER_SIZE_PER_FRAME);
      gla_tx_buf[s_offset++] = (int32)0xCCCCCCCC;
      gla_tx_buf[s_offset++] = gl_TxTcCount;
      gla_tx_buf[s_offset++] = gl_RxTcCount;
      gla_tx_buf[s_offset++] = gl_TxSymbolCount;
      gla_tx_buf[s_offset++] = gl_RxSymbolCount;
      if (gs_LogShowBuf == 1)
      {
         ReadCoreReg(ZEP_REG_ZT_DTB_OFFSET_ADDR, &ul_tx_addr); //Per port register
         ul_tx_addr = ul_tx_addr*4 + ZEP_ILV_RAM_DTB_ADDR;
         ReadCoreBuf32(ul_tx_addr, &gla_tx_buf[s_offset], NUM_32BIT_WORDS_PER_FRAME);
      }
      else if (gs_LogShowBuf == 2)
      {
         WhichIFFTBuffers(&ul_IFFTInputBuffer, &ul_IFFTOutputBuffer);
         ul_IFFTInputBuffer += (uint32)gsa_TxBandLeftChannel[0];
         ReadCoreBuf32(ul_IFFTInputBuffer, &gla_tx_buf[s_offset], NUM_32BIT_WORDS_PER_FRAME);
      }
      gs_debug_tx_cnt++;
   }
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void LogRxBufAtRxFcTc(void)
*
*   This function logs RxDTB buffer or FFT buffer with various symbol counters
*   at RX-FC TC task. It may log small part of buffer with a few consecutive
*   frames so that it does not interfere normal showtime processing.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void LogRxBufAtRxFcTc(void)
{
   int16 s_offset;
   int32 ul_FFTInputBuffer, ul_FFTOutputBuffer;
   uint32 ul_rx_addr;

   if (gs_LogShowBuf == 0)
   {
      return;
   }

   if (gs_debug_rx_cnt < NUM_SHOW_FRAME_TO_LOG)
   {
      s_offset = gs_debug_rx_cnt*(NUM_32BIT_WORDS_PER_FRAME+NUM_HEADER_SIZE_PER_FRAME);
      gla_rx_buf[s_offset++] = (int32)0xCCCCCCCC;
      gla_rx_buf[s_offset++] = gl_TxTcCount;
      gla_rx_buf[s_offset++] = gl_RxTcCount;
      gla_rx_buf[s_offset++] = gl_TxSymbolCount;
      gla_rx_buf[s_offset++] = gl_RxSymbolCount;
      if (gs_LogShowBuf == 1)
      {
         ReadCoreReg(ZEP_REG_ZR_DTB_OFFSET_ADDR, &ul_rx_addr); //Per port register
         ul_rx_addr = ul_rx_addr*4 + ZEP_ILV_RAM_DTB_ADDR;
         ReadCoreBuf32(ul_rx_addr, &gla_rx_buf[s_offset], NUM_32BIT_WORDS_PER_FRAME);

      }
      else if (gs_LogShowBuf == 2)
      {
         WhichFFTBuffers(&ul_FFTInputBuffer, &ul_FFTOutputBuffer);
         ul_FFTInputBuffer += (uint32)gsa_RxBandLeftChannel[0];
         ReadCoreBuf32(ul_FFTInputBuffer, (void*)&gla_rx_buf[s_offset], NUM_32BIT_WORDS_PER_FRAME);
      }
      gs_debug_rx_cnt++;
   }
}

#endif // DEBUG_SHOWTIME


#ifdef CHECK_ILV_MEM_ERROR
void CheckIlvMemErr(void)
{

   uint32 ul_addr;
   uint32 ul_data;

   //Since there should be some gap between TX and RX ILV memory,
   //here we want to readout the memory in this gay to make sure
   //it is not overwritten by TX interleaver
   ul_addr = ZEP_RAM_IIBRAM_ADDR + gula_rxILVBaseAddr[LP0] - 4;

   //truncate the address in word boundary
   ul_addr &= 0xFFFFFFFC;

   if(gl_IlvMemErr_Cnt == -1)
   {
      //First zero out the memory location
      WriteCoreReg(ul_addr, 0);

      gl_IlvMemErr_Cnt = 0;
   }
   else
   {
      ReadCoreReg(ul_addr, &ul_data);

      if(ul_data != 0)
      {
         gl_IlvMemErr_Cnt++;

         if(gl_IlvMemErr_Cnt == 0x7FFFFFFF)
         {
            gl_IlvMemErr_Cnt = 0;
         }
      }

   }

} //CheckIlvMemErr()
#endif //CHECK_ILV_MEM_ERROR

//Zephyr debug code
#ifdef ZEP_DEBUG_BLD

//Mei debug
#ifdef VR9_BRINGUP_DBG
//CHECH_DTB
extern FlagT gft_CapDtbFlag;
extern uint16 *gpusa_DbgDtb;
extern int16 gs_DbgDtbSize;
extern int16 gs_DbgDtbCnt;
extern int16 gs_NumBytesInDtb;
extern int16 gs_NumBytesInDtbCheck;

extern int16 gs_DtbPtrBufCnt;
extern int16 gs_DtbPtrBufSize;   //1024 frames
extern uint16 *gusa_ZEP_DTBPTR_DbgBuf;
extern uint16 *gusa_QT_DTBPTR_DbgBuf;

#endif //#ifdef VR9_BRINGUP_DBG

void EnableTxFifo3Capture(void)
{
   uint32 ul_data;

#ifdef VR9_BRINGUP_DBG
//CHECH_DTB
   if(gft_CapDtbFlag == 1)
   {
      gs_DbgDtbCnt = 0;
   }
#endif

   // Use FIFO3 to capture the input data to the scrambler
   ul_data = ZEP_RAM_TX_OHFIFO3_DBG0_OFFSET;
   ul_data |= (ZEP_RAM_TX_OHFIFO3_DBG0_SIZE-1) << 18;
   WriteCoreReg(ZEP_PRAM_ZT_FIFO_BASE3_LP0_ADDR, ul_data);

   WriteCoreReg(ZEP_PRAM_ZT_FIFO_PTRS3_LP0_ADDR, 0);
   WriteCoreReg(ZEP_PRAM_ZT_FIFO_CNST3_LP0_ADDR, 0);

   //Compute the maximum number of bytes per symbol to transmit
   //To be conservative, we use this number to estimate the number
   //of symbols that can be stored in FIFO3
   //(the actual number of scrambler input bytes could be less than this number
   // if we count the RS overhead)
   gs_TxScamNumBytesPerSym = (int16)gt_tx_config_v2.ul_Lp[LP0]>>3;

   //Double this size to prevent the variation between frames, causing buffer wrap around
   gs_TxScamNumBytesPerSym <<= 1;

   gs_TxPreWRPTR = 0;
   gs_CapturedTxScamSyms = 0;
   gs_TxCapBufFullness = ZEP_RAM_TX_OHFIFO3_DBG0_SIZE;

   if(gs_TxScamNumBytesPerSym <= gs_TxCapBufFullness)
   {
      //Enable capture
      gft_TxFifo3Capture = FIFO3_CAPTURE_ENABLE;
   }
   else
   {
      //Indicate the buffer is not big enough to store one symbol of bytes
      gft_TxFifo3Capture = FIFO3_CAPTURE_ERROR;
   }

} //void EnableTxFifo3Capture(void)

void DisableTxFifo3Capture(void)
{
   uint32 ul_data;

   //Move the FIFO2 point to a dummy buffer
   ul_data = ZEP_RAM_TX_OHFIFO3_DBG1_OFFSET;
   ul_data |= (ZEP_RAM_TX_OHFIFO3_DBG1_SIZE-1) << 18;
   WriteCoreReg(ZEP_PRAM_ZT_FIFO_BASE3_LP0_ADDR, ul_data);

   WriteCoreReg(ZEP_PRAM_ZT_FIFO_PTRS3_LP0_ADDR, 0);
   WriteCoreReg(ZEP_PRAM_ZT_FIFO_CNST3_LP0_ADDR, 0);

} //void DisableTxFifo3Capture(void)



void UpdateTxFifo3Stat(void)
{
   uint32 ul_data;
   int16 s_WritePtr, s_NumBytes;

   if(gft_TxFifo3Capture == FIFO3_CAPTURE_ENABLE)
   {
      ReadCoreReg(ZEP_PRAM_ZT_FIFO_PTRS3_LP0_ADDR, &ul_data);

      //Get the write pointer value
      s_WritePtr = ul_data & 0xFFF;

      if(s_WritePtr > gs_TxPreWRPTR)
      {
         //Updata symbol count
         gs_CapturedTxScamSyms++;

         //Update the buffer fullness
         s_NumBytes = s_WritePtr - gs_TxPreWRPTR;
         gs_TxCapBufFullness -= s_NumBytes;

         //Update the previous write pointer
         gs_TxPreWRPTR = s_WritePtr;

         if(gs_TxCapBufFullness < gs_TxScamNumBytesPerSym)
         {
            gft_TxFifo3Capture = FIFO3_CAPTURE_GETDATA;
         }

//Mei debug
#ifdef VR9_BRINGUP_DBG
//CHECH_DTB
//capture the DTB
         if(gft_CapDtbFlag == 1)
         {
            int16 i;

            if(s_NumBytes != gs_NumBytesInDtb)
            {
               gs_NumBytesInDtbCheck = s_NumBytes;
            }

            i = gs_DbgDtbCnt+gs_NumBytesInDtb;

            //Note gs_NumBytesInDtb must be multiple of 4 bytes
            if( (i<gs_DbgDtbSize) && (i<(gs_ZEP_ILV_RAM_TXDTB_LP0_SIZE*4)))
            {

               //read out DTB buffer
               ReadCoreBuf32((uint32)(ZEP_ILV_RAM_DTB_ADDR+(gs_ZEP_ILV_RAM_TXDTB_OFFSET*4)+gs_DbgDtbCnt),
                             &gpusa_DbgDtb[gs_DbgDtbCnt>>1], (int16)(gs_NumBytesInDtb>>1));

               //update count
               gs_DbgDtbCnt += gs_NumBytesInDtb;
            }
         }
#endif //#ifdef VR9_BRINGUP_DBG

      }
      else if(s_WritePtr < gs_TxPreWRPTR)
      {
         //Indicate that wrap around occurs
         gft_TxFifo3Capture = FIFO3_CAPTURE_ERROR;
      }
      //else if(s_WritePtr == gs_TxPreWRPTR)
      //no capture has happened

      if((gft_TxFifo3Capture == FIFO3_CAPTURE_GETDATA) || (gft_TxFifo3Capture == FIFO3_CAPTURE_ERROR))
      {
         //Disable TX FIFO3 capture
         DisableTxFifo3Capture();
      }


   } //if(gft_TxFifo3Capture == FIFO3_CAPTURE_ENABLE)

} //void UpdateTxFifo3Stat(void)

void EnableRxFifo3Capture(void)
{
   uint32 ul_data;

#ifdef VR9_BRINGUP_DBG
//CHECH_DTB
   if(gft_CapDtbFlag == 2)
   {
      gs_DbgDtbCnt = 0;
      gs_DtbPtrBufCnt = 0;
   }
#endif //#ifdef VR9_BRINGUP_DBG

   // Use FIFO3 to capture the output data from the DeScrambler
   ul_data = ZEP_RAM_RX_OHFIFO3_DBG0_OFFSET;
   ul_data |= (ZEP_RAM_RX_OHFIFO3_DBG0_SIZE-1) << 18;

   WriteCoreReg(ZEP_PRAM_ZR_FIFO_BASE3_LP0_ADDR, ul_data);

   WriteCoreReg(ZEP_PRAM_ZR_FIFO_PTRS3_LP0_ADDR, 0);
   WriteCoreReg(ZEP_PRAM_ZR_FIFO_CNST3_LP0_ADDR, 0);

   //Compute the maximum number of bytes per symbol to transmit
   //To be conservative, we use this number to estimate the number
   //of symbols that can be stored in FIFO3
   //(the actual number of scrambler input bytes could be less than this number
   // if we count the RS overhead)
   gs_RxScamNumBytesPerSym = (int16)gt_rx_config_v2.ul_Lp[LP0]>>3;

   //Double this size to prevent the variation between frames, causing buffer wrap around
   gs_RxScamNumBytesPerSym <<= 1;

   gs_RxPreWRPTR = 0;
   gs_CapturedRxScamSyms = 0;
   gs_RxCapBufFullness = ZEP_RAM_RX_OHFIFO3_DBG0_SIZE;

   if(gs_RxScamNumBytesPerSym <= gs_RxCapBufFullness)
   {
      //Enable capture
      gft_RxFifo3Capture = FIFO3_CAPTURE_ENABLE;
   }
   else
   {
      //Indicate the buffer is not big enough to store one symbol of bytes
      gft_RxFifo3Capture = FIFO3_CAPTURE_ERROR;
   }

} //void EnableRxFifo3Capture(void)


void DisableRxFifo3Capture(void)
{
   uint32 ul_data;

   // Use FIFO3 to capture the output data from the DeScrambler
   ul_data = ZEP_RAM_RX_OHFIFO3_DBG1_OFFSET;
   ul_data |= (ZEP_RAM_RX_OHFIFO3_DBG1_SIZE-1) << 18;

   WriteCoreReg(ZEP_PRAM_ZR_FIFO_BASE3_LP0_ADDR, ul_data);

   WriteCoreReg(ZEP_PRAM_ZR_FIFO_PTRS3_LP0_ADDR, 0);
   WriteCoreReg(ZEP_PRAM_ZR_FIFO_CNST3_LP0_ADDR, 0);

} //void DisableRxFifo3Capture(void)

void UpdateRxFifo3Stat(void)
{
   uint32 ul_data;
   int16 s_WritePtr, s_NumBytes;

   if(gft_RxFifo3Capture == FIFO3_CAPTURE_ENABLE)
   {
      ReadCoreReg(ZEP_PRAM_ZR_FIFO_PTRS3_LP0_ADDR, &ul_data);
      //Get the write pointer value
      s_WritePtr = ul_data & 0xFFF;

      if(s_WritePtr > gs_RxPreWRPTR)
      {

         //Updata symbol count
         gs_CapturedRxScamSyms++;

         //Update the buffer fullness
         s_NumBytes = s_WritePtr - gs_RxPreWRPTR;
         gs_RxCapBufFullness -= s_NumBytes;

         //Update the previous write pointer
         gs_RxPreWRPTR = s_WritePtr;

         if(gs_RxCapBufFullness < gs_RxScamNumBytesPerSym)
         {
            gft_RxFifo3Capture = FIFO3_CAPTURE_GETDATA;
         }

//Mei debug
#ifdef VR9_BRINGUP_DBG
//CHECH_DTB
//capture the DTB
         if(gft_CapDtbFlag == 2)
         {
            int16 i;

            if(s_NumBytes != gs_NumBytesInDtb)
            {
               gs_NumBytesInDtbCheck = s_NumBytes;
            }

            i = gs_DbgDtbCnt+gs_NumBytesInDtb;

            //Note gs_NumBytesInDtb must be multiple of 4 bytes
            if( (i<gs_DbgDtbSize) && (i<(gs_ZEP_ILV_RAM_RXDTB_LP0_SIZE*4)))
            {

               //read out DTB buffer
               ReadCoreBuf32((uint32)(ZEP_ILV_RAM_DTB_ADDR+(gs_ZEP_ILV_RAM_RXDTB_OFFSET*4)+gs_DbgDtbCnt),
                             &gpusa_DbgDtb[gs_DbgDtbCnt>>1], (int16)(gs_NumBytesInDtb>>1));

               //update count
               gs_DbgDtbCnt += gs_NumBytesInDtb;
            }

            //Check DTB read and write pointer
            if(gs_DtbPtrBufCnt < gs_DtbPtrBufSize)
            {
               ReadCoreReg(ZEP_PRAM_ZR_DTB_PTRS_LP0_ADDR, &ul_data);
               gusa_ZEP_DTBPTR_DbgBuf[gs_DtbPtrBufCnt] = (uint16)(ul_data & 0xFFFF);

               ReadCoreReg(ZEP_REG_ZR_QTDTB_ADDR_LP01_ADDR, &ul_data);
               gusa_QT_DTBPTR_DbgBuf[gs_DtbPtrBufCnt] = (uint16)(ul_data & 0x1FFF); //<12:0> RXDTB_WR_ADDR_0
               gs_DtbPtrBufCnt++;
            }

         }
#endif //#ifdef VR9_BRINGUP_DBG

      }
      else if(s_WritePtr < gs_RxPreWRPTR)
      {
         //Indicate that wrap around occurs
         gft_RxFifo3Capture = FIFO3_CAPTURE_ERROR;
      }
      //else if(s_WritePtr == gs_RxPreWRPTR)
      //no capture has happened

      if((gft_RxFifo3Capture == FIFO3_CAPTURE_GETDATA) || (gft_RxFifo3Capture == FIFO3_CAPTURE_ERROR))
      {
         //Disable RX FIFO3 capture
         DisableRxFifo3Capture();
      }


   } //if(gft_RxFifo3Capture == FIFO3_CAPTURE_ENABLE)

} //void UpdateRxFifo3Stat(void)



#endif //#ifdef ZEP_DEBUG_BLD
