/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 2005 - Infineon Technologies
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/***C***************************************************************************
** The 'CCASE' lines will be filled by ClearCase upon checkin,
** please do not change this information manually!
**
** =============================================================================
**
** All rights reserved.
** =============================================================================
**
** =============================================================================
**
** This document contains proprietary information belonging to Infineon
** Technologies. Passing on and copying of this document, and communication of
** its contents is not permitted without prior written authorisation.
**
** =============================================================================
**
**
** =============================================================================
**
** Revision Information :
**
** CCASE_FILE = "xTC_DebugMsgHandler.c"
** CCASE_DATE = "Wed Jul 26 09:02:59 2006"
**
** =============================================================================
**
** Contents:   Inside this file the xTC_DebugMsgHandler for incoming debug
**             messages is implemented.
**
** Functions:  DebugMsgHandler()
**             - This function sends the incoming message to the specific
**               function, where the message will be executed.
**
**             xTC_DebugMemMapRead()
**             - Read from a address
**
**             xTC_DebugMemMapWrite()
**             - Write a value under a address
**
**             xTC_DebugAuxRegRead()
**             - Read from an auxiliary register
**
**             xTC_DebugAuxRegWrite()
**             - Write a value in an auxiliary register
**
**             xTC_ReadMemoryBlock()
**             - Read a block from the counter memory
**
**             xTC_ReadVDSL_AFE()
**             - Write data in VDSL-AFE (Analog Front End)
**
**             xTC_WriteVDSL_AFE()
**             - Write data in VDSL-AFE (Analog Front End)
**
**             xTC_TcBlockSelection()
**             - Make TC (xTC or Alpheaus) selection for active TC
**
**             xTC_MultiFuncSwitch()
**             - Switch On/Off various firmware functionalities
**
** =============================================================================
**
** References: <List of related design documents>
**
** =============================================================================
** History:
**
** Date        Author        Comment
** 2005-04-22  Wehmann       Creation
** =============================================================================
**
Additional version control information
********************************************************************************
 CCASE_USER    = "Bschor"
 CCASE_VERSION = "\main\dev_hh_2\xtc_codeswap\Rev_1.4\1"
***C*E*************************************************************************/

/*
** =============================================================================
**                           INCLUDE FILES
** =============================================================================
*/

#include "BaseTypes.h"        /* Type definition IFX */

#include "typedef.h"             /* Type definition Aware */

#include "nmp.h"
#include "cmv.h"

#include "cmv_Data.h"

#include "xTC_CmvDefinitions.h"
#include "xTC_mapping.h"
#include "xTC_CounterMemory_hw.h"

#include "afe_if_dd.h"

#include "SubSysHW.h"
#include "xTC_Modules.h"
#include "xTC_ModemMonitorInit.h"
#include "Status_LED.h"
#include "CustomerTasks.h"

/*
** =============================================================================
**                           LOCAL DEFINITIONS
** =============================================================================
*/

#define RESERVED_VALUE              0x0000

/* Definitions for CMD_DBG_COUNTER_BLOCK_READ */
#define NUM_OF_CNTR_MEM_1           61
#define NUM_OF_CNTR_MEM_2           116

/* Definitions for CMD_DEVICE_RESET */
#define VINAX_HARDRESET             0x0
#define RESET_TIMER_VALUE           0x1
#define WATCHDOG_DISABLE_FLAG       0x0


/*
** =============================================================================
**                           LOCAL DATA
** =============================================================================
*/

/*
** =============================================================================
**                           LOCAL FUNCTION PROTOTYPES
** =============================================================================
*/

uint_8 xTC_DebugMemMapRead(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_DebugMemMapWrite(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_DebugAuxRegRead(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_DebugAuxRegWrite(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_ReadVDSL_AFE(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_WriteVDSL_AFE(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_PsiResetCheck(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_VinaxDeviceReset(uint_16 *PayLoadSize, uint_16 PayLoad[]);
uint_8 xTC_ReadMemoryBlock(uint_16 *PayLoadSize, HerculesMPStruct_t *HMP_BufferPointer);
uint_8 xTC_HW_WatchdogControl(uint_16 *PayLoadSize, uint_16 PayLoad[]);


/*
** =============================================================================
**                           IMPLEMENTATION
** =============================================================================
*/

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  DebugMsgHandler()
**
** DESCRIPTION:    This function executes the incoming debug message.
**
** PARAMETERS:     MsgID (uint_16 / unsigned short)
**                 - MsgID from actual message.
**
**                 *PayLoadSize (uint_16 / unsgned short)
**                 - Pointer to payload size.
**
**                 *HMP_BufferPointer (HerculesMPStruct_t)
**                 - Currently used HMP_Buffer.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsgned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 DebugMsgHandler(uint_16 MsgID, uint_16 *PayLoadSize, HerculesMPStruct_t *HMP_BufferPointer)
{

   uint_8 xTCMsgExecuted = CMV_MSG_ID_UNKNOWN;

   /*
   ** Switch between different debug message events
   */
   switch(MsgID)
   {

      /*
      ** Read from memory / mapped devices
      */
   case CMD_DBG_MEM_MAP_READ:

      xTCMsgExecuted = xTC_DebugMemMapRead(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Write in memory / mapped devices
      */
   case CMD_DBG_MEM_MAP_WRITE:

      xTCMsgExecuted = xTC_DebugMemMapWrite(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Read from auxiliary register
      */
   case CMD_DBG_AUX_REG_READ:

      xTCMsgExecuted = xTC_DebugAuxRegRead(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Write in auxiliary register
      */
   case CMD_DBG_AUX_REG_WRITE:

      xTCMsgExecuted = xTC_DebugAuxRegWrite(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Read counters from memory block
      */
   case CMD_DBG_COUNTER_BLOCK_READ:

      xTCMsgExecuted = xTC_ReadMemoryBlock(&(*PayLoadSize), &(*HMP_BufferPointer));
      break;

      /*
      ** Read from VDSL-AFE
      */
   case CMD_DBG_READ_VDSL_AFE:

      xTCMsgExecuted = xTC_ReadVDSL_AFE(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Write in VDSL-AFE
      */
   case CMD_DBG_WRITE_VDSL_AFE:

      xTCMsgExecuted = xTC_WriteVDSL_AFE(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Enable / Disable PSI-Reset check
      */
   case CMD_DBG_UTOPIA_WATCHDOG_CONTROL:

      xTCMsgExecuted = xTC_PsiResetCheck(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

      /*
      ** Make a device reset to the Vinax
      */
   case CMD_DEVICE_RESET:

      xTCMsgExecuted = xTC_VinaxDeviceReset(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;

   case CMD_DBG_HW_WATCHDOG_CONTROL:

      xTCMsgExecuted = xTC_HW_WatchdogControl(&(*PayLoadSize), HMP_BufferPointer->usa_HmpPayLoad);
      break;


   }

   return xTCMsgExecuted;
}

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTCDebugMemMapRead()
**
** DESCRIPTION:    Read from memory / mapped devices
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_DebugMemMapRead(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_32 DbgAddress = 0, DbgValue = 0;
   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from buffer  */
   DbgAddress = PayLoad[1];
   DbgAddress = DbgAddress << 16;
   DbgAddress += PayLoad[0];

   /* Execute read access */
   DbgValue = *((uint_32*) DbgAddress);

   /* Write value in buffer */
   PayLoad[2] = DbgValue;
   DbgValue = DbgValue >> 16;
   PayLoad[3] = DbgValue;

   /* The PayLoadSize has 1 element more */
   *PayLoadSize += 2;

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_DebugMemMapWrite()
**
** DESCRIPTION:    Write in memory / mapped devices
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_DebugMemMapWrite(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_32 DbgAddress = 0, DbgValue = 0;
   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from the xTC-Buffer  */
   DbgAddress = PayLoad[1];
   DbgAddress = DbgAddress << 16;
   DbgAddress += PayLoad[0];

   /* Create value for write access */
   DbgValue = PayLoad[3];
   DbgValue = DbgValue << 16;
   DbgValue += PayLoad[2];

   /* Execute write access */
   *((uint_32*) DbgAddress) = DbgValue;

   /*
   ** Check written data
   */
   if (DbgValue == *((uint_32*) DbgAddress))
   {
      /* The PayLoadSize has 1 element less */
      *PayLoadSize -= 2;
   }
   else
   {
      xTCMsgExecuted = CMV_ERR_DURING_EXEC;
   }

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_DebugAuxRegRead()
**
** DESCRIPTION:    Read from auxiliary register
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_DebugAuxRegRead(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_32 DbgAddress, DbgValue;
   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from the xTC-Buffer  */
   DbgAddress = PayLoad[1];
   DbgAddress = DbgAddress << 16;
   DbgAddress += PayLoad[0];

   /* Execute read access */
   DbgValue = _lr(DbgAddress);

   /* Write value in xTC-Buffer */
   PayLoad[2] = DbgValue;
   DbgValue = DbgValue >> 16;
   PayLoad[3] = DbgValue;

   /* The PayLoadSize has 1 element more */
   *PayLoadSize += 2;

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_DebugAuxRegWrite()
**
** DESCRIPTION:    Write in auxiliary register
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_DebugAuxRegWrite(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_32 DbgAddress, DbgValue;
   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read the address from the xTC-Buffer  */
   DbgAddress = PayLoad[1];
   DbgAddress = DbgAddress << 16;
   DbgAddress += PayLoad[0];

   /* Create value for write access */
   DbgValue = PayLoad[3];
   DbgValue = DbgValue << 16;
   DbgValue += PayLoad[2];

   /* Execute write access */
   _sr(DbgValue, DbgAddress);

   /*
   ** Check written data
   */
   if (DbgValue == _lr (DbgAddress))
   {
      /* The PayLoadSize has 1 element less */
      *PayLoadSize -= 2;
   }
   else
   {
      xTCMsgExecuted = CMV_ERR_DURING_EXEC;
   }

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_ReadMemoryBlock()
**
** DESCRIPTION:    Reading the counter values (max. 62) from counter block.
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 *HMP_BufferPointer (HerculesMPStruct_t)
**                 - Currently used HMP_Buffer.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_ReadMemoryBlock(uint_16 *PayLoadSize, HerculesMPStruct_t *HMP_BufferPointer)
{

   uint_32  TempValue = 0;
   uint_16  Count = 0, PayloadID = 4;
   uint_8   Offset = 0, bearerChannel = 0, CntrBlock = 0;
   uint_8   CountID = 0, xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   /* Get the Offset of the counters */
   Offset =  (uint_8)HMP_BufferPointer->t_MsgHeader.us_HmpMsgWord3;
   /* Get the Count of the counters */
   Count = HMP_BufferPointer->t_MsgHeader.us_HmpMsgWord4;
   /* Get the CntrBlock of the counters */
   CntrBlock = (uint_8)HMP_BufferPointer->usa_HmpPayLoad[0];
   /* Get the number of the slice/bearer channel   */
   bearerChannel = (uint_8)HMP_BufferPointer->usa_HmpPayLoad[2];

   /*
   ** Check size of parameter Count
   */
   if((Count > ((HMP_MBOX_PAYLOAD_SIZE - PayloadID) / 2)) ||
         ((CntrBlock == 1) && ((Count + Offset) > NUM_OF_CNTR_MEM_1)) ||
         ((CntrBlock == 2) && ((Count + Offset) > NUM_OF_CNTR_MEM_2)) ||
         ((CntrBlock != 1) && (CntrBlock != 2)))
   {
      /* Count is too big or CntrBlock is wrong*/
      xTCMsgExecuted = CMV_WRONG_PARAMETER;
   }
   else
   {
      /*
      ** Get address of counter memory
      */
      if(CntrBlock == 1)
      {
         CntrBlock = MAP_MODULE_CNT_MEM1;
      }
      else
      {
         CntrBlock = MAP_MODULE_CNT_MEM2;
      }

      /*
      ** Fill payload with counter values
      */
      for(CountID = 1; CountID <= Count; CountID++)
      {
         /* ============ Only for Rev 1.x ====================== */

         /* Read counter value from counter memory */
         TempValue = CM_ReadCounter(bearerChannel, CntrBlock, Offset + (CountID - 1),
                                    CNTR_READ_WITHOUT_RESET);
         /* ========== End of #ifndef IFX_REV2 ================= */

         /* Insert counter value into payload */
         HMP_BufferPointer->usa_HmpPayLoad[PayloadID] = TempValue;
         TempValue = TempValue >> 16;
         HMP_BufferPointer->usa_HmpPayLoad[(PayloadID + 1)] = TempValue;

         PayloadID += 2;
      }

      /* Set PayLoadSize depends from the number of counters */
      *PayLoadSize = (Count + 2) * 2;
   }

   return xTCMsgExecuted;
}

/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_ReadVDSL_AFE()
**
** DESCRIPTION:    Read data from VDSL-AFE (Analog Front End)
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_ReadVDSL_AFE(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   /* Read data from VDSL-AFE */

   PayLoad[2] = ReadVDSLAfe((uint16)PayLoad[0]);


   /* Clean first address and payload word */
   PayLoad[1] = RESERVED_VALUE;
   PayLoad[3] = RESERVED_VALUE;

   /* The PayLoadSize has 1 element more */
   *PayLoadSize += 2;

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_WriteVDSL_AFE()
**
** DESCRIPTION:    Write data in VDSL-AFE (Analog Front End)
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_WriteVDSL_AFE(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;


   /* Write data in VDSL-AFE */
   WriteVDSLAfe((uint16)PayLoad[0], (uint16)PayLoad[2]);


   /* Clean first address and payload word */
   PayLoad[1] = RESERVED_VALUE;
   PayLoad[3] = RESERVED_VALUE;


   /*
   ** Check written data
   */
   if (PayLoad[2] == ReadVDSLAfe((uint16)PayLoad[0]))
   {


      /* The PayLoadSize has 1 element less */
      *PayLoadSize -= 2;
   }
   else
   {
      xTCMsgExecuted = CMV_ERR_DURING_EXEC;
   }

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_VinaxDeviceReset()
**
** DESCRIPTION:    Make a device reset to this Vinax. This reset is realized
**                 with the Watchdog timer (HW-Timer 2 in Rev 1.x, Watchdog
**                 Timer in Rev 2).
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad[] in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_VinaxDeviceReset(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   if(PayLoad[0] == VINAX_HARDRESET)
   {

      /* ======================= Only for Rev 1.x ======================= */

      /* Write the timer value */
      TIME_TIMVAL2_RegWrite(RESET_TIMER_VALUE);
      /* Enable the soft reset upon expiration */
      TIME_TIMCFG_RegWrite(TIMER2_PER_CLOCK_128_ENABLE | TIMER2_SOFT_RESET);

      /* ===== End of Rev 1.x, begin of Rev 2 =============================== */

      /* The PayLoadSize is zero */
      *PayLoadSize = 0;
   }
   else
   {
      xTCMsgExecuted = CMV_WRONG_PARAMETER;
   }

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_PsiResetCheck()
**
** DESCRIPTION:    Enable / Disable check of CRC-Counter to reset PSI-Block
**                 in case of an error.
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad[] in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_PsiResetCheck(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   if(PayLoad[0] == 0x1)
   {
      /* Activate PSI-Reset check */
      PsiResetEnabled = TRUE;
   }
   else
   {
      /* Deactivate PSI-Reset check */
      PsiResetEnabled = FALSE;
   }

   /* The PayLoadSize is zero */
   *PayLoadSize = 0;

   return xTCMsgExecuted;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:  xTC_WatchdogControl()
**
** DESCRIPTION:    Disable/Enable the watchdog timer
**
**
** PARAMETERS:     *PayLoadSize (uint_16 / unsigned short)
**                 - Pointer to payload size.
**
**                 PayLoad[] (uint_16 / unsigned short)
**                 - PayLoad[] in actual message.
**
** RETURN VALUE:   xTCMsgExecuted (uint_8 / unsigned char)
**                 - Value for function code creation.
**
** NOTES:          -
** =============================================================================
*/
uint_8 xTC_HW_WatchdogControl(uint_16 *PayLoadSize, uint_16 PayLoad[])
{

   uint_8 xTCMsgExecuted = CMV_WITHOUT_ERRORS;

   if(PayLoad[0] == WATCHDOG_DISABLE_FLAG)
   {
      /* Watchdog is disabled */
      gt_WatchdogTimerEnable = 0;

      /* Soft reset on watchdog expiration is disabled */
      WatchdogTimerStop();
   }
   else
   {
      /* Watchdogtimer flag is enabled. This is also done by default */
      gt_WatchdogTimerEnable = 1;
   }

   /* The PayLoadSize is zero */
   *PayLoadSize = 0;

   return xTCMsgExecuted;
}

/* ======================= Only for Rev 2 ======================= */
