/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 2016 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*
*   file name: AFED_HybTrainUtilities.c
*
*   This file contains VRX518 Hybrid training related functions
*
*-------------------------------------------------------------------------------
*/

// **********************************************************************************************//
// History
// 10/05/2016 Palaksha: Added VRX518 Hybrid training functions
//
// **********************************************************************************************//

#include <stdio.h>
#include <string.h>
#include "common.h"
#include "typedef.h"
#include "str_memmap.h"
#include "LL_IOf.h"
#include "vdsl_xception.h"
#include "gdata.h"

#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Constants.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"
#include "AFED_ReadWriteModify.h"

#include "vrx5afe_mode_defines.h"
#include "vrx5afe_central_addrmap.h"
#include "vrx5afe_central_fcsi_regpkg.h"     // register for CENTRAL
#include "vrx5afe_dsl_addrmap.h"
#include "vrx5afe_dsl_fcsi_regpkg.h"         // register for DSL

/*******************************************************************************
*
*   Prototype: void AFED_BgSetHybrid(void)
*
*      This function loads AFED_BgSetHybrid() address into the function pointer
*     to execute them in the TC task.
*   Input Arguments:
*   Global Variables:
*  Pointer to hybrid 2D array: gpus_HybridsForProfileSelected[I]
*   Hybrid array index corresponding to the row: gs_HybTrIndex
*
*
*   Output Arguments:
*
*   Returns:
*
*******************************************************************************/
void AFED_BgSetHybrid(void)
{

   guc_PgaState = TRAINING_IN_PROGRESS;
   //Set the Hybrid Registers through TC task
   gpF_AfeLoadFunc = (PtrToFunc)AFED_SetHybrid;
   gft_LoadAfeFunctions = 1;

}

/*******************************************************************************
*
*   Prototype: void AFED_SetHybrid(void)
*
*      This function shall make use of the global variable: gs_HybTrIndex and
*   global pointer to Hybrid array gpus_HybridsForProfileSelected to program the AFE
*   registers corresponding to that Hybrid index for that mode of operation.
*   The initialization of the Hybrid pointer based on appropriate mode and profile
*   has been considered to be handled outside this function and in function
*   ConfigTaskLayerForPostGhs(void) above.
*   This function will only update the pointer to the corresponding updated
*   Hybrid index meant for that mode and profile so that the same can be used
*   for programming the AFE registers.
*   This function restores the PGA to its default and calls BgSetPGA()
*   after setting up the hybrid
*
*   Input Arguments:
*   Global Variables:
*  Pointer to hybrid 2D array: gpus_HybridsForProfileSelected[I]
*   Hybrid array index corresponding to the row: gs_HybTrIndex
*
*
*   Output Arguments:
*
*   Returns:
*
*******************************************************************************/
void AFED_SetHybrid(void)
{
   uint16 *pHybRegSetting;
   if (gus_SkipAfeFlow & 0x800)
   {
      guc_PgaState = TRAINING_DONE;
      return;
   }

   if (gs_HybTrFinishUp == 1)
   {
      gs_HybTrIndex = gs_HybTrMaxIndex; // if complete Hybrid array is completed
   }// Make the Hybrid index point to the best Hybrid selected and progarm the same

   pHybRegSetting = gpus_HybridsForProfileSelected + (gs_HybTrIndex * VRX_HYB_NUM_SETTINGS);

   // Note: This line must be placed here, because otherwise the pointer is not correct anymore!
   gs_HybridGain = (pHybRegSetting[VRX_HYBGAIN_IDX]);

   //Hybrid Coeff 0
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA0), 0x0000, *pHybRegSetting++);
   //Hybrid Coeff 1
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA1), 0x0000, *pHybRegSetting++);
   //Hybrid Coeff 2
   VRX5AFE_FCSI_RMW(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA2), 0x0000, *pHybRegSetting);

   guc_PgaState = TRAINING_DONE;
}


/*******************************************************************************
*
*   Prototype: void AFED_UpdateHybridBuffer(void)
*
*    This function shall make use of the global variable: gs_HybTrIndex and
*  global pointer to Hybrid array gpus_HybridsForProfileSelected to update the hybrid
*   buffer with the special Hybrid settings to provide -6dB gain as input to the AGC1.
*   The initialization of the Hybrid pointer based on appropriate mode and profile
*   has been considered to be handled outside this function and in function
*   ConfigTaskLayerForPostGhs(void) above.
*
*  Input Arguments:
*   Global Variables:
*  Pointer to hybrid 2D array: gpus_HybridsForProfileSelected[I]
*  Hybrid array index corresponding to the row: gs_HybTrIndex
*
*
*   Output Arguments:
*
*   Returns:
*
*******************************************************************************/
void AFED_UpdateHybridBuffer(void)
{
   uint16 *pHybRegSetting;
   uint32 i;
   pHybRegSetting = gpus_HybridsForProfileSelected + (gs_HybTrIndex * VRX_HYB_NUM_SETTINGS);
   for (i = 0; i < VRX_HYB_NUM_SETTINGS; i++)
   {
      pHybRegSetting[i] = gusa_SpecialHybSetting[i];
   }
}

void AFED_ToggleHybridCap(int16 s_StopIdx, int16 s_WrapAdder, int16 *ps_cap_delta,
                          uint16 *pHybRegSrc, int16 s_HybridTableIdx, int16 s_DestOffset)
{
   int16  s_CPx, s_CPxOriginalValue;
   uint16 *pHybRegDest;
   uint16 us_PGAxRegValue;
   signed int i;

   us_PGAxRegValue = pHybRegSrc[s_HybridTableIdx];                // get PGAx register value from Hybrid array
   if (s_HybridTableIdx == VRX518_PGA2_IDX)
   {
      s_CPxOriginalValue = (((us_PGAxRegValue & VRX5AFE_DSL_FCSI_PGA2_PGA35CPC__SMSK) >> VRX5AFE_DSL_FCSI_PGA2_PGA35CPC__POS)
                            & VRX5AFE_DSL_FCSI_PGA2_PGA35CPC__MSK);
      us_PGAxRegValue   &= (VRX5AFE_DSL_FCSI_PGA2_PGA35CPC__ISMSK);
   }
   else
   {  // (s_HybridTableIdx == VRX518_PGA1_IDX)
      s_CPxOriginalValue = (((us_PGAxRegValue & VRX5AFE_DSL_FCSI_PGA1_PGA35CPB__SMSK) >> VRX5AFE_DSL_FCSI_PGA1_PGA35CPB__POS)
                            & VRX5AFE_DSL_FCSI_PGA1_PGA35CPB__MSK);
      us_PGAxRegValue   &= (VRX5AFE_DSL_FCSI_PGA1_PGA35CPB__ISMSK);
   }

   for (i = 0; i < s_StopIdx; i++)
   {
      // Copy best hybrid.
      // Note: Always the original CAP value must be used!
      //       If all values should be copied then it must be taken care that
      //       the "gs_HybTrIndex" doesn't fall into the copy range!
      pHybRegDest = gpus_HybridsForProfileSelected + (i * VRX_HYB_NUM_SETTINGS) + s_DestOffset;
//      if (i != gs_HybTrIndex)
      {
         memcpy(pHybRegDest, pHybRegSrc, sizeof(uint16)*VRX_HYB_NUM_SETTINGS);
      }

      //Add the delta with original cap value
      s_CPx = s_CPxOriginalValue + ps_cap_delta[i];
      if (s_CPx < 0)
      {
         // Min Cap value 0 (Open)
         // Extend the range to higher CAP values. This is done to test not always with "Open",
         // which would be a wast.
         s_CPx += (s_WrapAdder);
      }
      else if (s_CPx > 15)
      {
         // Max Cap Value 15 PF
         // Extend the range to lower CAP values. This is done to test not always with "15pF",
         // which would be a wast.
         s_CPx -= (s_WrapAdder);
      }

      pHybRegDest[s_HybridTableIdx] = (us_PGAxRegValue | (s_CPx << VRX5AFE_DSL_FCSI_PGA1_PGA35CPB__POS));
   }
}

void AFED_SearchGoodHybridCap(void)
{
   uint16 *pHybRegSrc;
   int16  s_DestOffset, s_HybMaxMetricIdx;


   // Initialization of local variables for every function call.
   s_DestOffset = (VRX_NUM_CAP_START_SEARCH * VRX_HYB_NUM_SETTINGS);       // Best Hybrid setting used as default!

   s_HybMaxMetricIdx = gs_HybTrMaxIndex;
   // Note: The if-case is needed, because otherwise the SNR- and Gain-Algo
   //       are not working!
   if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
   {
      s_HybMaxMetricIdx = gs_HybTrMaxIndexEcho;
   }

   gs_HybTrMaxIndex_CapSearch = s_HybMaxMetricIdx;


   // For cap search set x hybrid settings to the settings of the best hybrid!
   // Note: x = number of cap search = VR9_NUM_CAP_SEARCH
   //----------------------------
   // Address of the Best Metric. Note: Idx to Best metric is stored in "gs_HybTrIndex".
   pHybRegSrc = gpus_HybridsForProfileSelected + (s_HybMaxMetricIdx * VRX_HYB_NUM_SETTINGS);

   // Initialization of local variables only once.
   if((guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_START) == 0)
   {
      // Copy Best Metric to "gusa_HybridsForProfileSelected" entry zero!
      if (s_HybMaxMetricIdx != 0)
      {
         memcpy(gpus_HybridsForProfileSelected, pHybRegSrc, sizeof(uint16)*VRX_HYB_NUM_SETTINGS);
         // Set pHybRegSrc to best metric entry, which is now on idx zero. This is needed for the
         // memcpy in fct. AFED_ToggleHybridCap().
         pHybRegSrc = gpus_HybridsForProfileSelected;
      }

      // This global variable is used in fct. "AFED_ToggleHybridCap()".
      // It is like old code to avoid an additional fct. parameter!
      gs_HybTrIndex = s_HybMaxMetricIdx;


      // Best Hybrid moved to idx 0, i.e. clear index variable and
      // save best metric in the array!
      // Variables are used for SNR-Algo, Gain-Algo and Echo-Algo
      {
         gs_HybTrMaxIndex = 0;

         // Variables are used for Echo-Algo only
         // This code can run always. When no additional MIPS should be added
         // for the old SNR-Algo then the if-case should be commented in!
         // Note: Gain- and Echo-algo are running parallel, therefore the if-case is checking
         //       bit "HYB_ALGO_GAIN_BASED" and not "HYB_ALGO_ECHO_BASED".
         //      if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
         {
            gs_HybTrMaxIndexEcho = 0;

            gsa_HybTrMetricEcho[gs_HybTrMaxIndexEcho] = gsa_HybTrMetricEcho[s_HybMaxMetricIdx];
            gsa_HybTrRxPathGain[gs_HybTrMaxIndexEcho] = gsa_HybTrRxPathGain[s_HybMaxMetricIdx];
         }

         gl_HybTrMetric[gs_HybTrMaxIndex] = gl_HybTrMetric[s_HybMaxMetricIdx];
      }
   }

   // Toggle best metric capacitor "CPx" of PGAx register
   if (guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_CPB)
   {
      // Toggle best metric capacitor "CPB" of PGA1 register
      // 15  14    13 12 11 10   9 8 7 6 5      4 3 2 1 0
      //             PGA35CPB     PGA35RPB       PGA35RSB
      AFED_ToggleHybridCap(gsa_CapCpb_delta_size, VRX_NUM_CAP_CPB_ZEROLOOP_WRAP_ADD,
                           &gsa_CapCpb_delta[0], pHybRegSrc, VRX518_PGA1_IDX, s_DestOffset);

      // Clear control variable, because it is not allowed to execute this code part again!
      guc_EnableHybCapacitorSearch &= ~(CHANGE_CAP_SEARCH_CPB);
   }
   else if (guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_CPC)
   {
      // Due to the order it is known that CPB cap-search was performed first!
      if (guc_EnableHybCapacitorSearch & ENABLE_CAP_SEARCH_CPB)
      {
         // Start with capacitor "CPC" toggling values behind the already performed "CPB" toggling.
         s_DestOffset += (gsa_CapCpb_delta_size*VRX_HYB_NUM_SETTINGS);
      }

      // Toggle best metric capacitor "CPC" of PGA2 register
      // 15  14    13 12 11 10   9 8 7 6 5      4 3 2 1 0
      //             PGA35CPC    PGA35RPC       PGA35RSC
      AFED_ToggleHybridCap(gsa_CapCpc_delta_size, VRX_NUM_CAP_CPC_ZEROLOOP_WRAP_ADD,
                           &gsa_CapCpc_delta[0], pHybRegSrc, VRX518_PGA2_IDX, s_DestOffset);

      // Clear control variable, because it is not allowed to execute this code part again!
      guc_EnableHybCapacitorSearch &= ~(CHANGE_CAP_SEARCH_CPC);
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Description: Background function to calculate metric for hybrid training
 *
 *  Prototype: void AFED_BgHybTrCalcMetric(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */
void AFED_BgHybTrCalcMetric(void)
{
   unsigned int i, j;
   int32 l_HybTrMaxMetric;
   int16 us_SNRref, s_Snr;

   us_SNRref = 10*256;         // 10dB
   if (gs_NumOfRxBands == 1)
   {
      us_SNRref = 1*256;       //  1dB
   }

   l_HybTrMaxMetric = 0;
   for (j=0; j<gs_NumOfRxBands; j++)
   {
      for (i=gsa_RxBandLeftChannel[j]; i<=gsa_RxBandRightChannel[j]; i++)
      {
         s_Snr = gpsa_MeasuredSnrBuf[i];

         // For the metric add-up only tones with SNRs >= 10dB for short loops or
         //                                                1dB for long loops
         if (s_Snr >= us_SNRref)
         {
            // limit use of SNRs to 60dB [Q8.8]
            if (s_Snr >= 60*256)
            {
               l_HybTrMaxMetric += (int32)(60*256);
            }
            else
            {
//               l_HybTrMaxMetric += s_Snr;
               l_HybTrMaxMetric += (int32)(s_Snr);
               l_HybTrMaxMetric -= (int32)(us_SNRref);
            }
         }
      }
   }

   //SMS01357115 - VDSL_62 HybridTraining - Start
   //Save the Hybrid Train Metric for Each Hybrid settings
   gl_HybTrMetric[gs_HybTrIndex] = l_HybTrMaxMetric;
   //SMS01357115 - VDSL_62 HybridTraining - End
   if(l_HybTrMaxMetric > gl_HybTrMaxMetric)
   {
      // Save best Hybrid index and metric value.
      gs_HybTrMaxIndex = gs_HybTrIndex;
      gl_HybTrMaxMetric = l_HybTrMaxMetric;
   }
   gs_RxBkgdProcessFlag = TRAINING_DONE;
}


