/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*
*   file name: AFED_PgaHandlerUtilities.c
*
*   This file contains VRX518 AFE Filter tuning formula implementation routines
*
*-------------------------------------------------------------------------------
*/

// **********************************************************************************************//
// History
// 10/05/2016 Palaksha: Added VRX518 Filter tune routines
//
// **********************************************************************************************//

//===========================INCLUDE FILES ========================================
#include "typedef.h"
#include "typedef.h"
#include "gdata.h"
#include "cmv.h"


#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Constants.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"
#include "AFED_ReadWriteModify.h"

#include "vrx5afe_mode_defines.h"
#include "vrx5afe_central_addrmap.h"
#include "vrx5afe_central_fcsi_regpkg.h"     // register for CENTRAL
#include "vrx5afe_dsl_addrmap.h"
#include "vrx5afe_dsl_fcsi_regpkg.h"         // register for DSL


// XDSLRTFW-4236 VRX518: CTLK: Modify/Improve Single Stage PGA Training algorithm (Start)
/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   void AFED_BgSingleStagePGASetting(void)
**
** DESCRIPTION:   This function calculates the AGC1 & AGC2 gain setting
**                from gs_PGA_required and program them accordingly
**                This function follows the below mentioned table to program PGA gains
**                ==========================================================
**                | gs_PGA_required [dB] |    PGA1 [dB]   |    PGA2 [dB]   |
**                ==========================================================
**                |          21          |       21       |       0        |
**                |          20          |       21       |      -1        |
**                |          19          |       21       |      -2        |
**                |          18          |       21       |      -3        |
**                |          17          |       21       |      -4        |
**                |          16          |       21       |      -5        |
**                |          15          |       21       |      -6        |
**                |          14          |       21       |      -7        |
**                |     13 : -1 : -14    |  21 : -1 : -6  |      -8        |
**                |         -15          |       -9       |      -6        |
**                |         -16          |       -9       |      -7        |
**                |         -17          |       -9       |      -8        |
**                |         -18          |       -9       |      -9        |
**                |         -19          |       -9       |      -10       |
**                |         -20          |       -9       |      -11       |
**                |         -21          |       -9       |      -12       |
**                |         -22          |       -9       |      -13       |
**                |         -23          |       -9       |      -14       |
**                |         -24          |       -9       |      -15       |
**                |         -25          |       -9       |      -16       |
**                ==========================================================
**
**
** PARAMETERS:       gs_PGA_required      (I): Required gain (dB in 8.8 format)
**                   guc_PgaTrainingState (O): Status indicator
**                   gs_PGA_reg_setting   (O): Register value for PGA gain (RX_GAIN)
**                   gs_PGA_set           (O): Actual gain (dB in 8.8 format)
**                   gs_AGC1_Gain_Set     (O): Actual gain of PGA1 in dB format
**                   gs_AGC2_Gain_Set     (O): Actual gain of PGA2 in dB format
**
** RETURN VALUE:   NONE
**
** NOTES: AFED_CalcAGC1() & AFED_CalcAGC2() functions which are used by this host function
**        can be replaced by a lookup table (see above). Since these two functions are already existed,
**        they are used by the host function with input parameter manipulation.
**
** Registers used :
**
** =============================================================================
*/

void AFED_BgSingleStagePGASetting(void)
{
   int16  s_Gain;

   //set this flag to false so we can program all AGCs (AGC1, AGC2)
   gft_EnableDoublePGATrain = FALSE;

   // AGC1 gain (variable part)
   // input parameter is modified in a way that output satisfy the table above
   AFED_CalcAGC1(gs_PGA_required + 0x800 - gs_AGC1_margin);

   // Remaining gain program to AGC2
   s_Gain = gs_PGA_required - (gs_AGC1_Gain_Set << 8);

   AFED_CalcAGC2(s_Gain);

   gpF_AfeLoadFunc = (PtrToFunc)AFED_SetPga;
   gft_LoadAfeFunctions = 1;
}

    // XDSLRTFW-4236 VRX518: CTLK: Modify/Improve Single Stage PGA Training algorithm (End)
/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   void AFED_SetPga1(void)
**
** DESCRIPTION:      This TimeCritical task function is calling the LL-API function to set PGA1.
**                   When the PGA1 gain is violating the HW constrains then a exception is triggered.
**
** PARAMETERS:       gs_AGC1_Gain_Set (I): required gain (dB in 16.0 format)
**                   gus_AFE_RxMode   (I): Rx mode, i.e. highest frequency or in other words supported profiles
**                                         BYPASSED
**                                         VDSL35b_CPE
**                                         VDSL17_CPE
**                                         VDSL8_CPE
**                                         ADSL2_CPE
**
** RETURN VALUE:   NONE
**
** NOTES: It is expected that the "AFED_CalcAGC1()" function is used before, which takes care of the constrains.
**
** =============================================================================
*/
void AFED_SetPga1()
{
   uint16 mode_set;
   uint16 us_ErrorCodeAfe;

   mode_set = gus_AFE_RxMode;

   // PGA gain + fc_lp
   us_ErrorCodeAfe = vrx5afe_dsl_pga_gainfc_set(mode_set, gs_AGC1_Gain_Set, gla_fc_pga_lp_kHz[mode_set]);
   if ( us_ErrorCodeAfe != 0 ) //VRX518 AFE Error code
   {
      EnterFailStates(E_CODE_AFE_EXCEPTION);   // XDSLRTFW-1628
   }

   // Debug PGA info for Tracee!
   {
      DshHwConfEventPga_t tmpDshConf;

      tmpDshConf.s_PgaHandlerRun          = gs_PgaHandlerRun;
      tmpDshConf.ft_StartHybTrn           = gft_StartHybTrn;
      tmpDshConf.uc_HybTrIndex            = (uint8) gs_HybTrIndex;
      tmpDshConf.s_AvRxTimePwr            = gs_AvRxTimePwr;
      tmpDshConf.s_MaxRxTonePwr           = gs_MaxRxTonePwr;
      tmpDshConf.s_PGA_required_Pre       = gs_PGA_required_Pre;
      tmpDshConf.s_PGA_required           = gs_PGA_required;
      tmpDshConf.s_AGC1_Gain_Set          = gs_AGC1_Gain_Set;
      tmpDshConf.s_AGC2_Gain_Set          = gs_AGC2_Gain_Set;
      tmpDshConf.s_RxVarGainDB            = gt_DfeAfeGainSettings.s_Rx_VGWin_Gain_dB;
      tmpDshConf.s_PrefiAceMode_Selected  = gs_PrefiAceMode_Selected;

      DSH_SendEvent(DSH_EVT_PGA_RECONFIGURE_VRX518,sizeof(DshHwConfEventPga_t),&tmpDshConf);
   }

   // Update the PGA state (If this function called in Bg Task)
   guc_PgaState = TRAINING_DONE;
}

void AFED_SetPga()
{
   uint16 mode_set;
   uint16 us_ErrorCodeAfe;
   int16 s_RxVarGainDB;

   mode_set = gus_AFE_RxMode;

   // PGA gain + fc_lp
   if(((gs_PgaHandlerRun == 0)&& (gft_EnableDoublePGATrain == TRUE)) || (gft_EnableDoublePGATrain == FALSE))
   {
      // program AGC1 settings
      // PGA gain + fc_lp
      if (gus_SkipAfeFlow & 0x200)
      {
         guc_PgaState = TRAINING_DONE;
         return;
      }
      us_ErrorCodeAfe = vrx5afe_dsl_pga_gainfc_set(mode_set, gs_AGC1_Gain_Set, gla_fc_pga_lp_kHz[mode_set]);
      if ( us_ErrorCodeAfe != 0 ) //VRX518 AFE Error code
      {
         EnterFailStates(E_CODE_AFE_EXCEPTION);   // XDSLRTFW-1628
      }
    }

   if(((gs_PgaHandlerRun == 1) && (gft_EnableDoublePGATrain == TRUE)) || (gft_EnableDoublePGATrain == FALSE))
   {
      if (gus_SkipAfeFlow & 0x100)
      {
      guc_PgaState = TRAINING_DONE;
      return;
      }

      //AGC2gain2(Variable Gain)\
      // PREFI gain + fc_lp
      us_ErrorCodeAfe = vrx5afe_dsl_prefi_gainfc_set(mode_set, gs_AGC2_Gain_Set, gla_fc_prefi_lp_kHz[mode_set]);
      if ( us_ErrorCodeAfe != 0 ) //VRX518 AFE Error code
      {
         EnterFailStates(E_CODE_AFE_EXCEPTION);   // XDSLRTFW-1628
      }
   }


   if ((gft_EnableDoublePGATrain == FALSE) && (gft_EnaPrefiAceOffOneStagePga == TRUE) ) //XDSLRTFW-3613
   {
      vrx5afe_dsl_prefi_ace_set(ACE_OFF); // PREFI ACE   Should be OFF in case of Single Stage PGA
   }

   // XDSLRTFW-3897(Start)
   // Following code has been added for debugging purpose only.
   // Previously the RxVarGain setting was not correct (it was reporting the old, not the new one)
   // To ensure that that reported and used RxVarGain settings are in sync we call here the same
   // code once more which is run in function PGAHandler(void) - sub-state PGA_WAIT
   s_RxVarGainDB = gs_RxVarGainDB_Dbg;
   if (gs_PgaHandlerRun == 1)
   {
      if(gt_Reconf_Hybrid_Coeff.s_control & USE_RX_POWER_INSTEAD_OF_RX_AMPL_FOR_RXVARGAINCONFIG)
         s_RxVarGainDB  = (gs_PGA_required - (gs_AGC2_Gain_Set << 8));
      else
         s_RxVarGainDB  = (gs_AvRxTimePwr - gs_MaxRxTonePwr) + (gs_PGA_required - (gs_AGC2_Gain_Set << 8));
   }
   else if (gft_EnableDoublePGATrain == FALSE)
   {
      s_RxVarGainDB = (gs_AvRxTimePwr - gs_MaxRxTonePwr) + (gs_PGA_required - gs_PGA_set);
   }
   // XDSLRTFW-3897(End)
   // Debug PGA info for Tracee!
   {
      DshHwConfEventPga_t tmpDshConf;

      tmpDshConf.s_PgaHandlerRun          = gs_PgaHandlerRun;
      tmpDshConf.ft_StartHybTrn           = gft_StartHybTrn;
      tmpDshConf.uc_HybTrIndex            = (uint8) gs_HybTrIndex;
      tmpDshConf.s_AvRxTimePwr            = gs_AvRxTimePwr;
      tmpDshConf.s_MaxRxTonePwr           = gs_MaxRxTonePwr;
      tmpDshConf.s_PGA_required_Pre       = gs_PGA_required_Pre;
      tmpDshConf.s_PGA_required           = gs_PGA_required;
      tmpDshConf.s_AGC1_Gain_Set          = gs_AGC1_Gain_Set;
      tmpDshConf.s_AGC2_Gain_Set          = gs_AGC2_Gain_Set;
      tmpDshConf.s_RxVarGainDB            = s_RxVarGainDB;     // XDSLRTFW-3897(Start_End)
      tmpDshConf.s_PrefiAceMode_Selected  = gs_PrefiAceMode_Selected;

      DSH_SendEvent(DSH_EVT_PGA_RECONFIGURE_VRX518,sizeof(DshHwConfEventPga_t),&tmpDshConf);
   }

   // Update the PGA state, This function called in TC Task
   guc_PgaState = TRAINING_DONE;
}

void AFED_BgGetPga(void)
{
   gpF_AfeLoadFunc = (PtrToFunc)AFED_GetPgaGains;
   gft_LoadAfeFunctions = 1;
}

void AFED_GetPgaGains(void)
{
   int16 s_AGC1Code;
   int16 s_AGC2Code;
   int16  s_AGC1Gain=0, s_AGC2Gain=0;
   uint16 ReadDataDummy, mode_set;

   mode_set = gus_AFE_RxMode;
   // PGA1:     -9dB and  -6dB to +15dB in 1dB steps (VDSL35B)
   // PGA1:     -9dB and  -6dB to +21dB in 1dB steps (VDSL and ADSL)
   // PGA2:              -16dB to   0dB in 1dB steps (VDSL)
   //                     -8dB to +12dB in 1dB steps (ADSL)
   //
   s_AGC1Code = VRX5AFE_FCSI_READ(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA3), &ReadDataDummy);
   s_AGC1Code = s_AGC1Code & VRX5AFE_DSL_FCSI_PGA3_PGA35GAIN__MSK;
   // gain
   if (s_AGC1Code == 0)
   {
      s_AGC1Gain = -9 ;
   }
   else
   {
      s_AGC1Gain = s_AGC1Code - 7;// From 00001b to 11100b
   }
   s_AGC1Gain = s_AGC1Gain << 8;
   gs_AGC1_Gain_Get = s_AGC1Gain;

   // AGC2 Gain2 (Variable Gain) in dB (Q8.8 format)
   s_AGC2Code = VRX5AFE_FCSI_READ(VRX5AFE_DSL, DSL_FCSI(VRX5AFE_DSL_FCSI_PGA7), &ReadDataDummy);
   s_AGC2Code = s_AGC2Code & VRX5AFE_DSL_FCSI_PGA7_PREFI35FBRES__MSK;
   s_AGC2Gain = s_AGC2Code - 16;
   s_AGC2Gain = s_AGC2Gain << 8;
   gs_AGC2_Gain_Get = s_AGC2Gain;


   //Max AGCGain = 21dB +0dB for 35B mode
   //else
   //Max AGCGain = 15dB + 0dB
   gs_AGC_Gain_Get = s_AGC1Gain + s_AGC2Gain;

   // Update the PGA state, This function called in TC Task
   guc_PgaState = TRAINING_DONE;
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   int16 AFED_CalcAGC1(int16 s_PGA_required)
**
** DESCRIPTION:   This function calculates the Gain setting on AGC1
**
** PARAMETERS:
**
** RETURN VALUE:   AGC1 reg setting
**
** NOTES:
** =============================================================================
*/
void AFED_CalcAGC1(int16 s_PGA_required)
{
   // Gain of total rx-path is set -> splitted in PGA1 and PGA2 (PREFI)
   //
   // Rx-gain: -25dB to +21dB / +15dB in 1dB steps (VDSL / VDSL 35b)
   //          -17dB to +33dB         in 1dB steps (ADSL)
   //
   // PGA1:     -9dB to -6dB to +21dB in 3/1dB steps (VDSL and ADSL)
   //           -9dB to -6dB to +15dB in 3/1dB steps (VDSL 35b)
   // PREFI:   -16dB to   0dB         in 1dB steps   (VDSL)
   //           -8dB to +12dB         in 1dB steps   (ADSL)
   //
   // Rule: always program PGA1 max. and if needed PGA2 (PREFI)

   int16_t rxgain_dB =0;
   int16_t gain_pga_dB =0;
   int16_t pga_gain_min = AFED_AGC1_PGA_MINUS_6DB;  //(-6dB)
   int16_t pga_gain_max;
   uint16  mode_set;

   mode_set = gus_AFE_RxMode;


   if (mode_set == VDSL35b_CPE)
   {
      pga_gain_max =  AFED_AGC1_MAX_PGA_8K; //(15)  //in dB
   }
   else
   {
      pga_gain_max =  AFED_AGC1_MAX_PGA; //(21)  //in dB
   }

   if (gs_Pga1MaxGainForce != -1 )
   {
      pga_gain_max = gs_Pga1MaxGainForce; //Force PGA1 Max gain in dB
   }

    rxgain_dB = s_PGA_required >> 8; //Convert to dB Format
   //----------------
   // gain splitting
   //----------------
   if ( (rxgain_dB >= pga_gain_min) && (rxgain_dB <= pga_gain_max) ) // PGA1 only
   {
      gain_pga_dB = rxgain_dB;
   }
   else if ( rxgain_dB < pga_gain_min)
   {
      gain_pga_dB = AFED_AGC1_MIN_PGA; //-9dB
   }
   else if (rxgain_dB > pga_gain_max)
   {
      gain_pga_dB = pga_gain_max;
   }

    gs_AGC1_Gain_Set = gain_pga_dB;

   if (gs_ForcePga1Gain != -32)
   {
      gs_AGC1_Gain_Set = gs_ForcePga1Gain;
   }

   if (gft_EnableDoublePGATrain == TRUE)
   {
      gs_PGA_set = ((gs_AGC1_Gain_Set + gs_AGC2_Gain_Set) << 8); //8.8 format
   }
}


/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   int16 AFED_CalcAGC2(int16 PGA_required)
**
** DESCRIPTION:   This function calculates the Gain setting off Prefi
**
** PARAMETERS:
**
** RETURN VALUE:   Prefi gain setting
**
** NOTES:
** =============================================================================
*/
void AFED_CalcAGC2(int16 s_PGA_required)
{
   // Gain of total rx-path is set -> splitted in PGA1 and PGA2 (PREFI)
   //
   // Rx-gain: -25dB to +21dB / +15dB in 1dB steps (VDSL / VDSL 35b)
   //          -17dB to +33dB         in 1dB steps (ADSL)
   //
   // PGA1:     -9dB to -6dB to +21dB in 3/1dB steps (VDSL and ADSL)
   //           -9dB to -6dB to +15dB in 3/1dB steps (VDSL 35b)
   // PREFI:   -16dB to   0dB         in 1dB steps   (VDSL)
   //           -8dB to +12dB         in 1dB steps   (ADSL)
   //
   // Rule: always program PGA1 max. and if needed PGA2 (PREFI)

   int16_t gain_prefi_dB;
   int16_t prefi_gain_min;
   int16_t prefi_gain_max;
   uint16  mode_set;

   // Initialization of local valiables
   mode_set = gus_AFE_RxMode;

   prefi_gain_min = AFED_AGC2_MIN_PGA;               //  (-16) in dB
   prefi_gain_max = AFED_AGC2_MAX_PGA;               //    (0) in dB
   if (mode_set == ADSL2_CPE)
   {
      prefi_gain_min = AFED_AGC2_MIN_PGA_ADSL;       //  (-8) in dB
      prefi_gain_max = AFED_AGC2_MAX_PGA_ADSL;       //  (12) in dB
   }

   if (gs_Pga2MaxGainForce != -20)
   {
      prefi_gain_max = gs_Pga2MaxGainForce;         // Force Prefi Max gain in dB
   }

   gain_prefi_dB = s_PGA_required >> 8;              // Convert to dB Format

   // Evaluate and limit the required gain to the Prefi gain constrains.
   if (gain_prefi_dB > prefi_gain_max)
   {
      gain_prefi_dB = prefi_gain_max;
   }
   else if (gain_prefi_dB < prefi_gain_min)
   {
      int16 s_Force_PrefiAceMode;

      s_Force_PrefiAceMode = TESTArray[TEST_DS_ACE];

      // CMV bit "TEST_DS_PREFI_CAP_DIS" to enable exception also for ACE off mode.
      // This is to check, if the normal FW has also coner cases (clipping) like for ACE on.
      // Details: fct. vrx5afe_dsl_prefi_gainfc_set() will return an error and fct. AFED_SetPga()
      //          will throw the exception "E_CODE_AFE_EXCEPTION".
      if (!(s_Force_PrefiAceMode & TEST_DS_PREFI_CAP_DIS))
      {
         // Old ACE off implementation : ACE mode = 4 (OFF) the hiding of clipping is active (It was not allowed to change this implementation)!
         // For ACE on implementation: ACE mode < 4 (ON) the clipping info is needed to switch down the ACE mode!
         // To test ACE ON with clipping and to get the exeption, i.e. "hiding on" the CMV bit "TEST_DS_PREFI_CAP_EN" was added.
         if((!((gft_EnableDoublePGATrain == TRUE) && (gs_PrefiAceMode_Selected < ACE_OFF))) ||
            (s_Force_PrefiAceMode & TEST_DS_PREFI_CAP_EN))
         {
            gain_prefi_dB = prefi_gain_min;
         }
      }
   }
   else
   {
      if (mode_set == BYPASSED)
      {
         // In Prefi35 BYPASSED mode, Gain is fixed @-8dB
         gain_prefi_dB = gsa_gain_prefi_dB[BYPASSED];
      }
   }

   // Set output
   gs_AGC2_Gain_Set = gain_prefi_dB;
   if (gs_ForcePga2Gain != -32)
   {
      gs_AGC2_Gain_Set = gs_ForcePga2Gain;
   }
   gs_PGA_set = ((gs_AGC1_Gain_Set + gs_AGC2_Gain_Set) << 8); //8.8 format
}


void AFED_InitAFE_RegsPGA(void)
{
   uint16 us_PgaTrainingDbg;

   uint16 us_AFE_RxMode, us_ErrorCodeAfe;
   int16 s_AGC1_Gain_Set, s_AGC2_Gain_Set, s_PrefiAceMode;


   us_PgaTrainingDbg = 0;
   if ( (gs_PgaTrainingDbg !=0) && (gl_HybTrSymcnt_start > 0) )
   {
      us_PgaTrainingDbg = gs_PgaTrainingDbg;
   }

   //LA: Hybridsetting
   //PGA1: gain = gain from HS??
   //LP1: on
   //ACE: bypass mode
   //PREFI: gain = -8dB
   //LP2: on

   //%%%%%%% SET PGA gains to start values for PGA-Training %%%%%%%%%%%%%%%%%%%%
   // program AGC1 settings: PGA gain + fc_lp
   s_AGC2_Gain_Set = gsa_gain_prefi_dB[BYPASSED];    // Set PREFI to bypass mode, -8dB
   s_PrefiAceMode = ACE_OFF;                         // Set ACE to bypass

   us_AFE_RxMode = gus_AFE_RxMode;
   s_AGC1_Gain_Set = gsa_gain_pga_dB[us_AFE_RxMode]; // Set to lowest mode value to avoid clipping

   us_ErrorCodeAfe = vrx5afe_dsl_pga_gainfc_set(us_AFE_RxMode, s_AGC1_Gain_Set, gla_fc_pga_lp_kHz[us_AFE_RxMode]);
   if (us_ErrorCodeAfe != 0)
   {
      EnterFailStates(E_CODE_AFE_EXCEPTION);   // XDSLRTFW-1628
   }


   if ((gft_EnableDoublePGATrain == TRUE) || (gft_BypassRxAFE_HPF == TRUE))
   {
      us_AFE_RxMode = BYPASSED;
   }
   else
   {
      // Train both PGA1 & 2 together

      // PREFI ACE
      s_PrefiAceMode = gs_PrefiAceMode_Selected;

      // PREFI gain + fc_lp
//CHECK_VRX518AFE  is it 0 or even low
//      s_AGC2_Gain_Set = -8;
   }

   // PREFI gain + fc_lp
   us_ErrorCodeAfe = vrx5afe_dsl_prefi_gainfc_set(us_AFE_RxMode, s_AGC2_Gain_Set, gla_fc_prefi_lp_kHz[us_AFE_RxMode]);
   if (us_ErrorCodeAfe != 0)
   {
      EnterFailStates(E_CODE_AFE_EXCEPTION);   // XDSLRTFW-1628
   }

   // ACE
   // Note: When training PGA1, ACE should OFF thats why gs_PrefiAceMode_Selected is not added here
   // C21(30pF) | C22(15pF) | C23(7.5pF)
   //     0           0           0
   // PREFI ACE
   vrx5afe_dsl_prefi_ace_set(s_PrefiAceMode);

   // gs_PGA_set = PGA1 + PGA2
   gs_AGC1_Gain_Set = s_AGC1_Gain_Set;
   gs_AGC2_Gain_Set = s_AGC2_Gain_Set;
   gs_PGA_set = (s_AGC1_Gain_Set + s_AGC2_Gain_Set) << 8;

   // Update the PGA state (If this function called in Bg Task)
   guc_PgaState = TRAINING_DONE;
}



void AFED_RestoreAFE_RegsforAGC2(void)
{
   uint16 mode_set;
   uint16 us_ErrorCodeAfe;
   int32  fc_prefi_lp_kHz;

   mode_set = gus_AFE_RxMode;
   fc_prefi_lp_kHz = gla_fc_prefi_lp_kHz[mode_set];

   if (gs_PgaHandlerRun ==1)
   {
      // PREFI ACE
      vrx5afe_dsl_prefi_ace_set(gs_PrefiAceMode_Selected); //CHECK_VRX518AFE

      // PREFI gain + fc_lp
      gs_AGC2_Gain_Set = AFED_AGC2_MIN_PGA;
      us_ErrorCodeAfe = vrx5afe_dsl_prefi_gainfc_set(mode_set, gs_AGC2_Gain_Set, fc_prefi_lp_kHz);
      if ( us_ErrorCodeAfe != 0 ) //VRX518 AFE Error code
      {
         EnterFailStates(E_CODE_AFE_EXCEPTION);   // XDSLRTFW-1628
      }
   }

   // Update the PGA state (If this function called in Bg Task)
   guc_PgaState = TRAINING_DONE;
}


void AFED_BgSetNmsPGAMedley(void)
{
   AFED_CalcAGC1(gs_PGA_required);

   // XDSLRTFW-4216 :CTL, VDSL: Downstream low perf in mid range loops from 2000ft to 3000ft (NMS) (Start_End)
   gpF_AfeLoadFunc = (PtrToFunc)AFED_SetPga1;
   gft_LoadAfeFunctions = 1;
}

//XDSLRTFW-3613 Start
/*
** =============================================================================
** FUNCTION-DESCRIPTION
**
** FUNCTION-NAME:   int16 AFED_ReUsePgaPrefiGains(int16 PGA_required)
**
** DESCRIPTION:   This function takes Hybrid Gain Difference as input and add this to PGA1 and spillover(+ve as well -ve)from PGA1
**                and distribute to PGA2 gain. Hybird gain difference is needed to be distributed PGA1/PGA2 inorder to make sure that
**                total AFE gain is same.
**                This function is specifically to detect ChannelDiscovery Vector1 signal and to detect Training Vector11 signal
**
** PARAMETERS: Hybrid Gain difference
**
** RETURN VALUE:   None
**
** NOTES:
** =============================================================================
*/

void AFED_ReUsePgaPrefiGains(int16 s_HybGainDiff)
{
   int16    s_Pga1_Required,s_Pga2_Required,s_GainDeltaToPga2;

   //set this flag to false so we can program PGA/PREFI
   gft_EnableDoublePGATrain = FALSE;

   // PGA Gain calculation
   s_Pga1_Required = (gs_Stored_AGC1_Gain_Set << 8)+s_HybGainDiff; //PGA gain+PGA gain
   // PGA gain (variable part)
   AFED_CalcAGC1(s_Pga1_Required);

   // PREFI Gain calculation
   //If s_Pga1_Required > MaxPga1 then, s_GainDeltaToPga2 = s_Pga1_Required - s_pga_gain_max should be added to PGA2
   //if (s_Pga1_Required < s_pga_gain_min), then s_GainDeltaToPga2 = s_pga_gain_min - s_Pga1_Required should be added to PGA2
   s_GainDeltaToPga2 = s_Pga1_Required - (gs_AGC1_Gain_Set << 8);//Take the diff RequiredPga1 - ProgrammedPga1
   s_Pga2_Required = (gs_Stored_AGC2_Gain_Set << 8)+s_GainDeltaToPga2; //Add to the  PGA2 additional  s_GainDeltaToPga2(-ve or +ve)
   AFED_CalcAGC2(s_Pga2_Required);

   gpF_AfeLoadFunc = (PtrToFunc)AFED_SetPga;
   gft_LoadAfeFunctions = 1;

}
//XDSLRTFW-3613 End

