/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C) 2016 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   HybridTrainingHandler.c
 *
 *   This file contains the routine that handles the hybrid training.
 *
 *------------------------------------------------------------------------
 */

// ***********************************************************************************************************
// HybridTrainingHandler.c
//
// History
// 16/05/2012 Vinjam: Code Pointers connected to Downstream Virtual Noise feature
//            Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
// 19/09/2013 Fuss: Addded code for vectoring to use only two different hybrid settings.
//            Grep for XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation
// 11/08/2014 Fuss: Hybrid setting Clean-up and integrated HybridGain to the Hybrid setting tables
//            Grep for XDSLRTFW-1488
// ************************************************************************************************************

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "fifo.h"
#include "noiseacc.h"
#include "IRI_Iof.h"
#include "accum32.h"
#include "Snr_b.h"
#include "HybridTrainingHandler.h"
#include "PGAHandler.h"
#include "FdqHandler.h"
#include "SnrHandler.h"
#include "cmv_Data.h"
#include "cmv.h"
#include "afe.h"
#include "ConvertToDB.h"
#include "VRX_AfeCommonConst.h"
#include "VRX_AfeCommonData.h"
#include "AFED_Data.h"
#include "AFED_Functions.h"

#ifdef VR9_DBG_HYB
extern int16 gsa_dbg_hyb[32];
extern int16 gs_dbgTestIndex;
#endif

#define HYBRID_PLL_REFTONE_AVG_SYMBOLS  4

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : HybridTrainingHandler
 *
 *  Description:  This function is used to train the hybrid
 *
 *  Prototype:  void HybridTrainingHandler(void)
 *
 *------------------------------------------------------------------------
 *^^^
 */
void HybridTrainingHandler(void)
{
   signed int i;

   switch (gs_HybrTrHandlerState)
   {
      //==============================================================================
      // initialize variables for Hybrid Training
      //==============================================================================
   case HYBRID_TRAINING_INIT:

      //If the hybrid training is not enabled
      if (gft_HybridTrainingEnable == 0)
      {
         gs_HybrTrHandlerState = HYBRID_TRAINING_DONE;
      }
      else
      {
         gl_HybTrSymcnt_start = gl_RxSymbolCount;
         // Secure code that gl_HybTrSymcnt_start is for sure != 0.
         if (gl_HybTrSymcnt_start == 0)
         {
            gl_HybTrSymcnt_start = 1;
         }

         // Disable Virtual Noise in SNR measurements during hybrid training:
         // we train the hybrid based on the actual SNR, instead of virtual SNR.
         // Besides, on CO, at this point we don't know yet the CPE TX PSD,
         // so we cannot compute the virtual SNR anyway.
         //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start)
         gft_VirtualNoiseStatus_save = gft_DSVirtualNoiseStatus;
         gft_DSVirtualNoiseStatus = 0;
         //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (End)
         // save original AlgLog2NumFramesToAccum
         gs_FdqAlgLog2NumFramesToAccum_Saved = gt_FdqConfig.s_AlgLog2NumFramesToAccum;
         gs_SnrAlgLog2NumFramesToAccum_Saved = gt_SnrConfig.s_AlgLog2NumFramesToAccum;

         //Using the smaller number of symbols to reduce the training time
         gt_FdqConfig.s_AlgLog2NumFramesToAccum = 3;
         gt_SnrConfig.s_AlgLog2NumFramesToAccum = 3;

         //   XDSLRTFW-477 PERF_DS_ALL_ALL_TX_BAND_SWITCH (Start_End)
         {
            //Save the RX Band Info for further use after Hybrid Training
            gs_SaveRxNumBands = gs_NumOfRxBands;

            //Check if the Highest Allowed RX Tone is lower than the First Tone of US Bands
            //870 First Bin of US1
            //1971 First bin od US2
            //This change only for Hybrid Training, later this will be changed back to original
            //value
            if(gs_HighestAllowedRxTone < LAST_BIN_DS1)
            {
               gs_NumOfRxBands = 1;
            }
            else if(gs_HighestAllowedRxTone < LAST_BIN_DS2)
            {
               gs_NumOfRxBands = 2;
            }
         }

         // Prepare ACE mode array for Hybrid selection
         for (i=0; i < (VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH); i++)
         {
            gsa_AceModeforHybridSelected[i] = gs_PrefiAceMode_Selected;
         }

         //Initialize the hybrid search parameters
         gs_MaxNumHybrSettings = TESTArray[TEST_MaxNumHybrSettings];
         gl_HybTrMaxMetric = 0;
         gs_HybTrIndex = 0;
         gs_HybTrMaxIndex = 0;
         gs_HybTrFinishUp = 0;
         gs_MaxNumHyb_save = 0;
#ifdef VR9_DBG_HYB
         gs_dbgTestIndex = 0;
#endif  // VR9_DBG_HYB

         // XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation (Start)
         if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
         {
            gl_HybTrMaxMetric = ((-256)<<8);     // Set to -256, because the gain is 8.8 format
            // and therfore no smaller value can exist!
            gs_HybTrMaxMetricEcho = 0x7fff;      // Set to max value
         }
         // XDSLRTFW-1181: BugFix_DS_VDSL2_All_VecDSRateRunToRunVariation (End)

         // Force a specific hybrid index for debug purposes
         if(gs_HybTrIndexDbg >= 0)
         {
            uint16 *pHybRegSrc, *pHybRegDest;

            // Stay in range of the selected profile hybrid table, i.e. if the
            // configuration is to big always the last entry of the table will be used!
            if (gs_HybTrIndexDbg >= gs_MaxNumHybrSettings)
            {
               gs_HybTrIndexDbg = (gs_MaxNumHybrSettings - 1);
            }

            // Copy hybrid setting to local buffer
            pHybRegSrc = gpus_HybridsForProfileSelected + (gs_HybTrIndexDbg * VRX_HYB_NUM_SETTINGS);

            for (i = 0; i < gs_MaxNumHybrSettings; i++)
            {
               pHybRegDest = gpus_HybridsForProfileSelected + (i * VRX_HYB_NUM_SETTINGS);
               if (i != gs_HybTrIndexDbg)
               {
                  memcpy(pHybRegDest, pHybRegSrc, sizeof(uint16)*VRX_HYB_NUM_SETTINGS);
               }
            }
         }

         // Debug code to trace all gains for every hybrid and PGA run0/run1 (AGC1/AGC2+AGC3) training.
//         AddFunctionToFifo(gp_RxLoadingFunctionFifo, GetPgaGains_VR9);

         //go to the next substate
         gs_HybrTrHandlerState = HYBRID_TRAINING_SET_HYBRID;
      }

      break;

      //==============================================================================
      // Set hybrid
      //==============================================================================
   case HYBRID_TRAINING_SET_HYBRID:

      // Now write hybrid setting associated with gs_HybTrIndex
      // BgPGA is called at the end of BgSetHybrid, so this is the function we will monitor
      // for completion.
      gft_StartHybTrn=1;

      // ACE mode must be considered during Hybrid and PGA training.
      // Background: It could be that the PREFI gain range is not sufficient,
      //             i.e. gain below -16dB is calculated, then the ACE level gets reduced.
      //             Restore the original ACE level for the next hybrid.
      // It should not be executed for the CAP search!
      {
        signed int s_Flag;

        s_Flag = TRUE;
        if ((guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_START) && (TESTArray[TEST_DS_ACE] & TEST_DS_ACE_HYB_ONLY_EN))
        {
           s_Flag = FALSE;
        }

        if (s_Flag)
        {
           gs_PrefiAceMode_Selected = gsa_AceModeforHybridSelected[gs_HybTrIndex];
        }
      }

      guc_PgaState = TRAINING_IN_PROGRESS;
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, AFED_SetHybrid);

      gft_EnablePLL = 0; // Turn off PLL

      //go to the next substate
      gs_HybrTrHandlerCount = 0;
      gs_HybrTrHandlerState = HYBRID_TRAINING_WAIT_FOR_HYBRID_SET;

      break;

      //==============================================================================
      // Wait for hybrid settings to take effect
      //==============================================================================
   case HYBRID_TRAINING_WAIT_FOR_HYBRID_SET:

      if (guc_PgaState == TRAINING_DONE)
      {
         if (gs_HybrTrHandlerCount < gs_HwSettleTime)
         {
            gs_HybrTrHandlerCount++;
         }
         else
         {
            gs_PgaHandlerState = PGA_INIT;
            gs_HybrTrHandlerState = HYBRID_TRAINING_PGA;
            gs_HybrTrHandlerCount = 0;
         }
      }// if (guc_HybrTrState == TRAINING_DONE)

      break;

      //==============================================================================
      // Train PGA
      //==============================================================================
   case HYBRID_TRAINING_PGA:

      if (gs_PgaHandlerState != PGA_DONE)
      {
         PGAHandler();
      }
      else
      {
         // Tracee debug info
         // Note: The stream must be located after PGA handler, because the PGA handler can decide
         //       to switch to the special hybrid!
         {
            uint16 *pHybRegSetting;

            pHybRegSetting = gpus_HybridsForProfileSelected + (gs_HybTrIndex * VRX_HYB_NUM_SETTINGS);
            DSH_SendEvent(DSH_EVT_HYBRID_SETTING,VRX_HYB_NUM_SETTINGS*sizeof(uint16),pHybRegSetting);
         }

         // XDSLRTFW-3280/XDSLRTFW-3835 - Start - PLL improvement / pilot tone selection improvement
         // XDSLRTFW-3280/XDSLRTFW-3835 - Start - PLL improvement / pilot tone selection improvement
         // Note: Normally the code is conflicting with FG fct. MultiPilotUpdate().
         //       But it is working, because the Pilot tone is not changing during the Pga- and HybridTrainingHandler() and
         //       the fct. MultiPilotUpdate() contains the following if-case in the state "PLL_AVG_IDLE":
         //       -  if (gt_PilotConfig.te_UsedPTArrayIdx != gt_PilotConfig.te_BestPTArrayIdx)
         //       Therefore it must be taken care that MultiPilotUpdate() stays in the IDLE state, i.e. in a kind of bypass.
         //       The only possible is to set the variable "gs_PLLAvgHandlerState" back to "PLL_AVG_IDLE" at the end of operation,
         //       because fct. ComputePLLRefAvg() is changing the variable "gs_PLLAvgHandlerState" to "PLL_AVG_DONE".
         //
         //       Due to gt_PilotConfig.ta_PilotTones[PilotIdx].ft_DeRotatePilot the gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Re
         //       and gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Imag cannot be used directly, i.e. always the fct. ComputePLLRefAvg()
         //       must be called.
         //       Also no "Data On pilot" is available the flag "ft_DeRotatePilot" can stay set, because constellation is
         //       is not changing during ChannelDiscovery and TransceiverTraining.
         {
            PT_ArrIdx_te UsedArrayIdx = gt_PilotConfig.te_UsedPTArrayIdx;

            gs_PLLAvgHandlerCnt = 0;
            gla_AvgRefPilotToneReal[0] = 0;
            gla_AvgRefPilotToneImag[0] = 0;
            gla_AvgRefPilotToneReal[1] = 0;
            gla_AvgRefPilotToneImag[1] = 0;
            gla_AvgRefPilotToneReal[2] = 0;
            gla_AvgRefPilotToneImag[2] = 0;
            // Note: Due to train time requirements, it is not possible to update/handle all pilot tones!
            ComputePLLRefAvg(UsedArrayIdx, &gsa_AvgRefPilotToneReal[UsedArrayIdx], &gsa_AvgRefPilotToneImag[UsedArrayIdx],HYBRID_PLL_REFTONE_AVG_SYMBOLS,2);
            gs_PLLAvgHandlerState = PLL_AVG_IDLE;  // This code is needed due to FG fct. MultiPilotUpdate(). See Note above!
         }

         gs_AlgHandlerCount = 0;
         gs_HybrTrHandlerState = HYBRID_TRAINING_PGA_PLLREFAVG;
      }
      break;

   case HYBRID_TRAINING_PGA_PLLREFAVG:
   {
      PT_ArrIdx_te UsedArrayIdx = gt_PilotConfig.te_UsedPTArrayIdx;

      if (gs_PLLAvgHandlerCnt < (HYBRID_PLL_REFTONE_AVG_SYMBOLS+1))
      {
         // compute average for reference, as sometimes reference could be noisy
         // Note: Due to train time requirements, it is not possible to update/handle all pilot tones!
         ComputePLLRefAvg(UsedArrayIdx, &gsa_AvgRefPilotToneReal[UsedArrayIdx], &gsa_AvgRefPilotToneImag[UsedArrayIdx],HYBRID_PLL_REFTONE_AVG_SYMBOLS,2);
         gs_PLLAvgHandlerState = PLL_AVG_IDLE; // This code is needed due to FG fct. MultiPilotUpdate(). See Note above, at switch-case "HYBRID_TRAINING_PGA"!
      }
      else
      {
         // Set new Pll pilot tone reference for the used pilot tones.
         ResetPllRefTone(UsedArrayIdx, NULL, &gsa_AvgRefPilotToneReal[0], &gsa_AvgRefPilotToneImag[0], (int16)1);
         gft_EnablePLL = 1; // Turn on PLL
      // XDSLRTFW-3280/XDSLRTFW-3835 - End - PLL improvement / pilot tone selection improvement

         // If the best hybrid has been found, and its PGA setting programmed,
         // we are done.  Otherwise go on to FDQ training, etc.
         if (gs_HybTrFinishUp == 1)
         {
            gs_HybrTrHandlerState = HYBRID_TRAINING_RESTORE;
         }
         else
         {
            if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
            {
               guc_PgaState = TRAINING_IN_PROGRESS;
               AddFunctionToBkgdFifo((PtrToBkgdFunc)AFED_BgGetPga);

               gs_HybrTrHandlerState = HYBRID_TRAINING_METRIC;
//               if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
               {
                  {
                     int16 s_NumTxTones, s_NumTxTonesMax;

                     // Determinate the max num of tones for US echo accumulation
                     s_NumTxTonesMax = 0;
                     for (i=0; i<gs_NumOfTxBands; i++)
                     {
                        s_NumTxTonesMax += (gsa_TxBandRightChannel[i] - gsa_TxBandLeftChannel[i])+1;
                     }
                     // Account total # of tones in a frame/symbol
                     guc_VectorPowerRightShiftsMax = 0;
                     s_NumTxTones = 1;
                     while ((s_NumTxTonesMax > s_NumTxTones) && (guc_VectorPowerRightShiftsMax < 15))
                     {
                        s_NumTxTones = (s_NumTxTones << 1);
                        guc_VectorPowerRightShiftsMax++;
                     }
                  }

                  // Clear the HW power register
                  AddFunctionToFifo(gp_RxLoadingFunctionFifo, ClearVectorPower);

                  gs_RtvSelect = FFT_OUTPUT ;
                  gs_CurrentTxBand = 0;
                  gl_Pa = 0;
                  gs_HybrTrHandlerCount = 0;
                  gs_HybrTrHandlerState = HYBRID_TRAINING_ECHOPOW;
               }
            }
            else
            {
               //Set up for the FDQ training
//               gs_HybrTrHandlerCount = 0;
               gs_AlgHandlerState = FDQ_TRAIN_INIT;
               gs_RxNumBytesPerSymbol = 1;         //Assume Hybrid training occurs during Reverb
               gs_HybrTrHandlerState = HYBRID_TRAINING_FDQ;
            }
         }
      }
   }
      break;

      //==============================================================================
      // Train FDQ
      //==============================================================================
   case HYBRID_TRAINING_FDQ:
//      if (gs_HybrTrHandlerCount < 10)
//      {
//         gs_HybrTrHandlerCount++;
//      }
//      else
      {
         if (gs_AlgHandlerState != FDQ_TRAIN_DONE)
            FdqHandler();
         else
         {
            gs_AlgHandlerState = SNR_INIT;
            gpsa_MeasuredSnrBuf = gsa_SnrBuf;
            gs_HybrTrHandlerState = HYBRID_TRAINING_SNR;
         }
      }
      break;

      //==============================================================================
      // Compute SNRs
      //==============================================================================
   case HYBRID_TRAINING_SNR:

      if (gs_AlgHandlerState != SNR_CALC_DONE)
      {
         SnrHandler();
      }
      else
      {
         //Compute the Hybrid selection metric
         gs_RxBkgdProcessFlag = TRAINING_IN_PROGRESS;

         AddFunctionToBkgdFifo((PtrToBkgdFunc)AFED_BgHybTrCalcMetric);

         gs_HybrTrHandlerState = HYBRID_TRAINING_METRIC;
      }
      break;

      //==============================================================================
      // Compute Echo power
      //==============================================================================
   case HYBRID_TRAINING_ECHOPOW:

      if(gs_HybrTrHandlerCount == 0)
      {
// Debug code start
//         if((gs_Debug3) < 4096-2)
//         {
//            gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)0xAAAA;
//            gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)0xAAAA;
//         }
// Debug code end

         if(guc_PgaState == TRAINING_DONE)
         {
            int16 s_CurTxBand;

            s_CurTxBand = gs_CurrentTxBand;
            gs_LeftChannel = gsa_TxBandLeftChannel[s_CurTxBand];
            gs_AlgNumTonesToProcess = (gsa_TxBandRightChannel[s_CurTxBand] - gsa_TxBandLeftChannel[s_CurTxBand])+1;

            // Account total # of tones in a frame and # of accumulation frames, i.e. for fct. ReadVectorPower().
            // Note: determine k of 2^k
            {
               int16 s_NumTxTones, s_NumTxTonesMax;

               guc_VectorPowerRightShifts = 0;

               // Account total # of tones in a frame
               s_NumTxTonesMax = gs_AlgNumTonesToProcess;
               s_NumTxTones = 1;
               while ((s_NumTxTonesMax > s_NumTxTones) && (guc_VectorPowerRightShifts < 15))
               {
                  s_NumTxTones = (s_NumTxTones << 1);
                  guc_VectorPowerRightShifts++;
               }

               // Consider # of accumulation frames, i.e. accumulation over x symbols
               s_NumTxTonesMax = (guc_HybridEchoAccu-1);
               s_NumTxTones = 1;
               while ((s_NumTxTonesMax > s_NumTxTones) && (guc_VectorPowerRightShifts < 15))
               {
                  s_NumTxTones = (s_NumTxTones << 1);
                  guc_VectorPowerRightShifts++;
               }
               guc_VectorPowerRightShifts = guc_VectorPowerRightShiftsMax;
// Debug code start
//               if((gs_Debug3) < 4096-1)
//               {
//                  gsa_TrnVectoringBuffer[gs_Debug3++] = guc_VectorPowerRightShifts;
//               }
// Debug code end
            }

            if(s_CurTxBand != 0)
            {
               gft_ExecuteShadowing = FALSE;
            }
            // setup QT for vector power accumulation
            // Note: The function contains the clearing of the vector accumulation register
            //       WriteCoreReg(IRI_QT_REG_RX_VECPOW_L_ADDR,0);
            //       WriteCoreReg(IRI_QT_REG_RX_VECPOW_H_ADDR,0);
            //       But for Hybrid training this must be skippt, because the HW is used to hold
            //       the accumulation over all tx echo bands!
            AddFunctionToFifo(gp_RxLoadingFunctionFifo, TriggerVectorPower);

            gs_HybrTrHandlerCount++;
         }
      }
      else
      {
         // Accumulation is done over guc_HybridEchoAccu, i.e. directly in HW!
         if(gs_HybrTrHandlerCount == (guc_HybridEchoAccu-1))
         {
            AddFunctionToFifo(gp_RxLoadingFunctionFifo, StopVectorPower);
//            // Trigger TC tasks which are executed in the next symbol!
//            // Echo power read out (accumulated value)
//            AddFunctionToFifo(gp_RxLoadingFunctionFifo, ReadVectorPower);  // Out: gl_Pa
//            // Clear the HW power register
//            AddFunctionToFifo(gp_RxLoadingFunctionFifo, ClearVectorPower);
         }

         // Note: TC task is executed before foreground and no HW is running in between.
         //       Therefore the first read can/must be skipped!
         //------------------------------------------------------------------
//         if(gs_HybrTrHandlerCount > 1)
         {
            // Echo power is available!
// Debug code start
//            if((gs_Debug3) < 4096-3)
//            {
//               gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)(gl_Pa & 0xFFFF);
//               gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)((gl_Pa >> 16) & 0xFFFF);
//               gsa_TrnVectoringBuffer[gs_Debug3++] = gs_HybrTrHandlerCount;
////               gl_Pa = 0;
//            }
// Debug code end

            // Average over X symbols. The value must be X+1, e.g. X=64 then the value must be 65!
            if(gs_HybrTrHandlerCount >= guc_HybridEchoAccu)
            {
               // Note: Since gl_Pa gets not cleared, it contains the summed power over
               //       gs_NumOfTxBands!
// Debug code start
//               if((gs_Debug3) < 4096-1)
//               {
//                  gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)(0xBBBB);
//               }
// Debug code end
               gs_CurrentTxBand++;
               {
                  int16 s_NumOfTxBands;

                  s_NumOfTxBands = gs_NumOfTxBands;
                  if (gs_HybNumUsBand > 0)
                  {
                     s_NumOfTxBands = gs_HybNumUsBand;
                  }

                  if(s_NumOfTxBands <= gs_CurrentTxBand)
                  {
                     AddFunctionToFifo(gp_RxLoadingFunctionFifo, ReadVectorPower);  // Out: gl_Pa

                     // Note: gl_Pa contains the accumulated value of the US bands according to gs_NumOfTxBands.
                     gs_HybrTrHandlerState = HYBRID_TRAINING_METRIC;
                  }
// Debug code start
// Note: When this debug code is enabled the decision is done on the last US band.
//       Further the measurement for every band can be seen in "gsa_TrnVectoringBuffer"!
//                  else
//                  {
//                     gl_Pa = 0;
//                  }
// Debug code end

               }
               // Set to zero.
               gs_HybrTrHandlerCount = -1;
            }
         }

         // Note: Since gs_HybrTrHandlerCount gets always incremented the
         //       value is one increment to high, when the variable gets checked!
         gs_HybrTrHandlerCount++;
      }
      break;

      //==============================================================================
      // Compute metric
      //==============================================================================
   case HYBRID_TRAINING_METRIC:

      if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
      {
         if(guc_PgaState == TRAINING_DONE)
         {
// Debug code start
//            if((gs_Debug3) < 4096-3)
//            {
//               gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)(gl_Pa & 0xFFFF);
//               gsa_TrnVectoringBuffer[gs_Debug3++] = (int16)((gl_Pa >> 16) & 0xFFFF);
//               gsa_TrnVectoringBuffer[gs_Debug3++] = gs_HybrTrHandlerCount;
//            }
// Debug code end

            {
               int16 s_HybGain;

               // Get the hybrid gain from a look-up table, since the fromula can
               // not be calculated in firmware!
               // Note: The hybrid gain gets calculated out of the hybrid settings:
               //       [RLa, RQa]=convert_AGC1_regs2hyb_R_values(aregs.AGC1_RIN1, aregs.AGC1_RSHUNT1);
               //       Ra = 390;
               //       hyb_gain=20*log10(400/ RLa*( RLa.* RQa/2/(RLa+RQa/2)/(Ra + RLa.* RQa/2/( RLa+ RQa/2))));
               //       For more details please conntact Kai Drebes.
               // XDSLRTFW-1488 (Start_End)
               {
                  uint16 *pHybRegSetting;

                  pHybRegSetting = gpus_HybridsForProfileSelected + (gs_HybTrIndex * VRX_HYB_NUM_SETTINGS);
                  gs_Hyb_Gain_Get = (int16)pHybRegSetting[VRX_HYBGAIN_IDX];
               }

               //Save the Hybrid Train Metric for Each Hybrid settings
               s_HybGain = (gs_AGC1_Gain_Get + gs_Hyb_Gain_Get);
               gl_HybTrMetric[gs_HybTrIndex] = s_HybGain;

               if(s_HybGain > gl_HybTrMaxMetric)
               {
                  // In case of (s_HybGain == gl_HybTrMaxMetric) the hybrid with the higher
                  // AGC1 value should be taken!

                  // save Best Hybrid to best hybrid Structure.
                  gs_HybTrMaxIndex = gs_HybTrIndex;
                  gl_HybTrMaxMetric = s_HybGain;
               }
            }

//            if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
            {
               int16 s_RxPathGain, s_EchoPow;

               // Restore QT MISC and QT_RNG0 Register to their original value
               gft_ExecuteShadowing = TRUE;
               AddFunctionToFifo(gp_RxLoadingFunctionFifo, DeTriggerVectorPower);

               // Used Rx path gain (Q8.8)
//               s_RxPathGain = (gs_AGC_Gain_Get + gs_Hyb_Gain_Get + gs_RxVarGainDB);
               s_RxPathGain = (gs_AGC_Gain_Get + gs_Hyb_Gain_Get);
               gsa_HybTrRxPathGain[gs_HybTrIndex] = s_RxPathGain;

               // Echo power (Q8.8)
               s_EchoPow = ConvertToDB(gl_Pa);
               // Consideration of Rx path gain
               s_EchoPow = (s_EchoPow - s_RxPathGain);
               if (s_EchoPow < 0)
               {
                  s_EchoPow = 0;
               }

               // Question: Should the analog be considered for all Rx signals?
               //           Or is a differentiation needed dependent of signal that controls the ADC?
               //           What should be done with "gs_RxVarGainDB"?
//               s_EchoPow = (s_EchoPow - (s_RxPathGain - gs_Hyb_Gain_Get));
               gsa_HybTrMetricEcho[gs_HybTrIndex] = s_EchoPow;

               if(s_EchoPow < gs_HybTrMaxMetricEcho)
               {
                  // save Best Hybrid to best hybrid Structure.
                  gs_HybTrMaxIndexEcho = gs_HybTrIndex;
                  gs_HybTrMaxMetricEcho = s_EchoPow;
               }
// Debug code start
//               //DSM_Vectoring_Debug:
//               if ( gs_PauseControl == 0x94)
//                  Pause(gs_PauseControl);
// Debug code end
            }

            gs_HybrTrHandlerState = HYBRID_TRAINING_NEXT_HYBRID;
         }
      }
      else
      {
         if (gs_RxBkgdProcessFlag == TRAINING_DONE)
         {
            gs_HybrTrHandlerState = HYBRID_TRAINING_NEXT_HYBRID;
         }
      }
      break;

      //==============================================================================
      // Try next hybrid setting
      //==============================================================================
   case HYBRID_TRAINING_NEXT_HYBRID:


#ifdef VR9_DBG_HYB
      if((gs_HybTrIndex < gs_MaxNumHybrSettings)&& (guc_EnableHybCapacitorSearch & (ENABLE_CAP_SEARCH_CPC|ENABLE_CAP_SEARCH_CPB)))
      {
         gsa_dbg_hyb[gs_dbgTestIndex++] = gs_PGA_set;
      }
#endif  //#ifdef VR9_DBG_HYB

      // Write the ACE mode back.
      // Note: The ACE mode can be changed in PgaHandler()!
      //       It should not be executed for the CAP search!
      //       But the handling is as for the hybrid + cap search and therefore
      //       it is done always!
//      if ((guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_START) == 0)
      {
         gsa_AceModeforHybridSelected[gs_HybTrIndex] = gs_PrefiAceMode_Selected;
      }

      if(gs_HybTrIndex < gs_MaxNumHybrSettings)
      {
         gs_HybTrIndex += gs_HybTrStepSize;
      }

      // Fct. AFED_SearchGoodHybridCap() must be called a second time if both cap's of Pag1 and Pga2
      // gets be toggled.
      if (guc_EnableHybCapacitorSearch & (CHANGE_CAP_SEARCH_CPC|CHANGE_CAP_SEARCH_CPB))
      {
         // Due to the order it is known that CPB cap-search was performed first!
         // XDSLRTFW-3649 Start_End
         if (gs_HybTrIndex >= (gs_MaxNumHybrSettings - gsa_CapCpc_delta_size))
         {
            int16  s_HybMaxMetricIdx;

            s_HybMaxMetricIdx = gs_HybTrMaxIndex;
            if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
            {
               s_HybMaxMetricIdx = gs_HybTrMaxIndexEcho;
            }

            for (i = gs_HybTrIndex; i < gs_MaxNumHybrSettings; i++)
            {
               gsa_AceModeforHybridSelected[i] = gsa_AceModeforHybridSelected[s_HybMaxMetricIdx];
            }
            AFED_SearchGoodHybridCap();
         }
      }

      if(gs_HybTrIndex >= gs_MaxNumHybrSettings)
      {
         if((guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_START) == 0)
         {
            // Save always hybrid indexes
            gs_HybTrIterat_save = gs_HybTrIndex;
            gs_HybTrMaxIndex_save = gs_HybTrMaxIndex;
            gs_MaxNumHyb_save = gs_MaxNumHybrSettings;


            // ACE mode must be considered during Hybrid and PGA training.
            // The ACE mode selected to the best hybrid has to be used in the afterwards coming training phases!
            // XDSLRTFW-3649 Start_End
            {
               int16  s_HybMaxMetricIdx;

               s_HybMaxMetricIdx = gs_HybTrMaxIndex;
               if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
               {
                  s_HybMaxMetricIdx = gs_HybTrMaxIndexEcho;
               }
               gs_PrefiAceMode_Selected = gsa_AceModeforHybridSelected[s_HybMaxMetricIdx];
            }
         }

         if(((guc_EnableHybCapacitorSearch & CHANGE_CAP_SEARCH_START) == 0) &&
             (guc_EnableHybCapacitorSearch & (ENABLE_CAP_SEARCH_CPC|ENABLE_CAP_SEARCH_CPB)))
         {

            // Copy x (= VR9_NUM_CAP_SEARCH) hybrid metrics to end, because
            // cap search will be done starting from index 0. So the previous
            // handlerstates can be reused!
            // Used for SNR-Algo VDSL2, Gain-Algo and Echo-Algo
            for (i = 0; i < VRX_NUM_CAP_SEARCH; i++)
            {
               gl_HybTrMetric[i+VRX_MAX_NUM_ADAP_HYB_SETTINGS] = gl_HybTrMetric[i];
               gsa_AceModeforHybridSelected[i+VRX_MAX_NUM_ADAP_HYB_SETTINGS] = gsa_AceModeforHybridSelected[i];
            }
            // Used for Echo-Algo only
            {
               // This code can run always. When no additional MIPS should be added
               // for the old SNR-Algo then the if-case should be commented in!
               // Note: Gain- and Echo-algo are running parallel, therefore the if-case is checking
               //       bit "HYB_ALGO_GAIN_BASED" and not "HYB_ALGO_ECHO_BASED".
//               if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
               {
                  gs_HybTrMaxIndexEcho_save = gs_HybTrMaxIndexEcho;

                  for (i = 0; i < VRX_NUM_CAP_SEARCH; i++)
                  {
                     gsa_HybTrMetricEcho[i+VRX_MAX_NUM_ADAP_HYB_SETTINGS] = gsa_HybTrMetricEcho[i];
                     gsa_HybTrRxPathGain[i+VRX_MAX_NUM_ADAP_HYB_SETTINGS] = gsa_HybTrRxPathGain[i];
                  }
               }
            }

            // Modify Hybrid input capacitor (Cqb) to get the best hybrid &
            // other parameters remains unchanged.
            //---------------------------------------------------------------
            // Set the control which cap's should be changed.
            guc_EnableHybCapacitorSearch |= ((guc_EnableHybCapacitorSearch & (ENABLE_CAP_SEARCH_CPC|ENABLE_CAP_SEARCH_CPB)) << CHANGE_CAP_SEARCH_CPB_POS);
            gs_MaxNumHybrSettings = VRX_NUM_CAP_START_SEARCH;
            if ((gus_SwitchingCriterionVal > gusa_SwitchingThreshRxHyb[SWITCH_TO_PROFILE_X_HYB]) ||
                (guc_HybridAlgoCtrl == HYB_ALGO_SNR_BASED))
            {
               gsa_CapCpb_delta_size = VRX_NUM_CAP_CPB_SEARCH_NOZEROLOOP;
//               memcpy(&gsa_CapCpb_delta[0], &gsa_CapCpb_delta[1], sizeof(int16)*VRX_NUM_CAP_CPB_SEARCH_NOZEROLOOP);
               gsa_CapCpc_delta_size = VRX_NUM_CAP_CPC_SEARCH_NOZEROLOOP;
//               memcpy(&gsa_CapCpc_delta[0], &gsa_CapCpc_delta[1], sizeof(int16)*VRX_NUM_CAP_CPC_SEARCH_NOZEROLOOP);
            }
            if (guc_EnableHybCapacitorSearch & ENABLE_CAP_SEARCH_CPB)
            {
               gs_MaxNumHybrSettings += gsa_CapCpb_delta_size;
            }
            if (guc_EnableHybCapacitorSearch & ENABLE_CAP_SEARCH_CPC)
            {
               gs_MaxNumHybrSettings += gsa_CapCpc_delta_size;
            }

            // Copy the ACE mode of the best hybrid to the array for CAP search!
            for (i = 0; i < VRX_NUM_CAP_SEARCH; i++)
            {
               gsa_AceModeforHybridSelected[i] = gs_PrefiAceMode_Selected;
            }
            AFED_SearchGoodHybridCap();

            //Start search from idx 1 onwards & idx 0 is best hybrid selection from the list
            if(guc_HybridAlgoCtrl == HYB_ALGO_SNR_BASED)
            {
               gs_HybTrStepSize = 2;
            }

            gs_HybTrIndex = 1;
            // Set CAP change is started, because it is not allowed to execute this code part again!
            guc_EnableHybCapacitorSearch |= (CHANGE_CAP_SEARCH_START);
         }
         else
         {
            gs_HybTrFinishUp = 1;

            // Used for Echo-Algo only
            // Note: The if-case is needed, because otherwise the SNR- and Gain-Algo
            //       are not working!
            if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
            {
               gs_HybTrMaxIndex = gs_HybTrMaxIndexEcho;
            }
         }
      } //if(gs_HybTrIndex >= gs_MaxNumHybrSettings)

      //Go to train the next hybrid/PGA training
      gs_HybrTrHandlerState = HYBRID_TRAINING_SET_HYBRID;

      break;

   case HYBRID_TRAINING_RESTORE:

      // Flag to indicate end of Hybrid Training, used for Fast PSD computation during Hybrid Training
      gft_StartHybTrn=0;

      // restore Virtual Noise status
      //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start)
      gft_DSVirtualNoiseStatus = gft_VirtualNoiseStatus_save;
      //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (End)
      // restore original AlgLog2NumFramesToAccum
      gt_FdqConfig.s_AlgLog2NumFramesToAccum = gs_FdqAlgLog2NumFramesToAccum_Saved;
      gt_SnrConfig.s_AlgLog2NumFramesToAccum = gs_SnrAlgLog2NumFramesToAccum_Saved;

      // XDSLRTFW-477 PERF_DS_ALL_ALL_TX_BAND_SWITCH (Start_End)
      // Move back the saved Rx Band Info
      gs_NumOfRxBands = gs_SaveRxNumBands;

      if(guc_EnableHybCapacitorSearch & (ENABLE_CAP_SEARCH_CPC|ENABLE_CAP_SEARCH_CPB))
      {
         int32 l_HybMet_save;

         for (i = 0; i < VRX_NUM_CAP_SEARCH; i++)
         {
            l_HybMet_save = gl_HybTrMetric[i];
            gl_HybTrMetric[i] = gl_HybTrMetric[i + VRX_MAX_NUM_ADAP_HYB_SETTINGS];
            gl_HybTrMetric[i + VRX_MAX_NUM_ADAP_HYB_SETTINGS] = l_HybMet_save;

            l_HybMet_save = gsa_AceModeforHybridSelected[i];
            gsa_AceModeforHybridSelected[i] = gsa_AceModeforHybridSelected[i + VRX_MAX_NUM_ADAP_HYB_SETTINGS];
            gsa_AceModeforHybridSelected[i + VRX_MAX_NUM_ADAP_HYB_SETTINGS] = l_HybMet_save;
         }

         // Used for Echo-Algo only
         // This code can run always. When no additional MIPS should be added
         // for the old SNR-Algo then the if-case should be commented in!
         // Note: Gain- and Echo-algo are running parallel, therefore the if-case is checking
         //       bit "HYB_ALGO_GAIN_BASED" and not "HYB_ALGO_ECHO_BASED".
//         if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
         {
            int16 s_HybRxPthGain_save;

            for (i = 0; i < VRX_NUM_CAP_SEARCH; i++)
            {
               l_HybMet_save = gsa_HybTrMetricEcho[i];
               gsa_HybTrMetricEcho[i] = gsa_HybTrMetricEcho[i + VRX_MAX_NUM_ADAP_HYB_SETTINGS];
               gsa_HybTrMetricEcho[i + VRX_MAX_NUM_ADAP_HYB_SETTINGS] = l_HybMet_save;
               s_HybRxPthGain_save = gsa_HybTrRxPathGain[i];
               gsa_HybTrRxPathGain[i] = gsa_HybTrRxPathGain[i+VRX_MAX_NUM_ADAP_HYB_SETTINGS];
               gsa_HybTrRxPathGain[i+VRX_MAX_NUM_ADAP_HYB_SETTINGS] = s_HybRxPthGain_save;
            }
         }
         gs_HybTrStepSize = 1;
      }


      // Tracee debug info
      {
         int16 s_HybTrMaxHybIndex, s_HybTrMaxCapIndex;
         int16 s_DataToBeStreamed;

         s_HybTrMaxHybIndex = gs_HybTrMaxIndex_save;
         s_HybTrMaxCapIndex = gs_HybTrMaxIndex;
         s_DataToBeStreamed = (VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH);
         s_DataToBeStreamed += ((VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH) >> 1);
         if ((VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH) & 1)
         {
            // Note: Arrays get be aligned on a 32bit address from compiler.
            //     int16 gsa_HybTrRxPathGain[VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH];
            //     one additional 16-bit word
            //     int16 gsa_HybTrMetricEcho[VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH];
            //     one additional 16-bit word
            //     int16 gsa_AceModeforHybridSelected[VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH];
            s_DataToBeStreamed += 2;
         }

         if(guc_HybridAlgoCtrl & HYB_ALGO_GAIN_BASED)
         {
            if(guc_HybridAlgoCtrl & HYB_ALGO_ECHO_BASED)
            {
               gla_HybMetricTrace[0] = HYB_ALGO_ECHO_BASED;
               s_HybTrMaxHybIndex = gs_HybTrMaxIndexEcho_save;
               s_HybTrMaxCapIndex = gs_HybTrMaxIndexEcho;
            }
            else
            {
               gla_HybMetricTrace[0] = HYB_ALGO_GAIN_BASED;
               for (i = 0; i < (VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH); i++)
               {
                  gsa_HybTrMetricEcho[i] = (int16)gl_HybTrMetric[i];
               }
            }
         }
         else // if (guc_HybridAlgoCtrl == HYB_ALGO_SNR_BASED)
         {
            gla_HybMetricTrace[0] = HYB_ALGO_SNR_BASED;
            memcpy(&gsa_HybTrRxPathGain[0], &gl_HybTrMetric[0],((VRX_MAX_NUM_ADAP_HYB_SETTINGS + VRX_NUM_CAP_SEARCH))*sizeof(int32));
         }

         s_DataToBeStreamed += 2;
         gla_HybMetricTrace[0] |= ((s_HybTrMaxHybIndex & 0xFF) << 8);
         gla_HybMetricTrace[0] |= (VRX_MAX_NUM_ADAP_HYB_SETTINGS << 16);
         gla_HybMetricTrace[0] |= (VRX_NUM_CAP_SEARCH << 24);
         gla_HybMetricTrace[1] = s_HybTrMaxCapIndex;
         gla_HybMetricTrace[1] |= (gs_HybTrMaxIndex_CapSearch << 16);
         DSH_SendStream(DSH_HYBRID_METRIC,(s_DataToBeStreamed*sizeof(int32)),&gla_HybMetricTrace[0]);
      }

      gl_HybTrSymcnt_end = gl_RxSymbolCount;
      if (gl_HybTrSymcnt_end == 0)
      {
         gl_HybTrSymcnt_end = 1;
      }

      gs_HybrTrHandlerState = HYBRID_TRAINING_DONE;
      break;

   } //switch
}

