/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *-------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   acc48_ops.c
 *
 *   Functions for using 48-bit accumulators
 *
 *-------------------------------------------------------------------------
 */

#include "common.h"
#include "dsp_op.h"

/*****************************************************************************
;   Subroutine Name: norm_acc48
;
;   Normalizes a 48-bit accumulator A. The 48-bit accumulator
;   is created by using a 32-bit HI portion overlapped with a 32-bit LO
;   portion.
;
;   The 48 bits number is represented as   HI * 2^16 + LO
;   Example: HI = 0x00345678, LO = 0x00123456, then
;
;                 0x00345678
;                       00123456 +
;                 --------------
;             A = 0x0034568A3456
;
;   If A is normalized, then bits 16-31 of LO is all zero: HI = 0x0034568A,
;   LO = 0x00003456
;
;   Typically, A is left unnormalized during the accumulation process and
;   normalization happens at the very end.
;
;   Prototype:
;       void norm_acc48(int32 *pl_AccH, int32 *pl_AccL)
;
;   Input Arguments:
;      pl_AccH          - pointer to the HI part of the 48 bits accum.
;      pl_AccL          - pointer to the LO part of the 48 bits accum.
;
;   Output Arguments:
;
;   Return:
;      Returns the normalized accum. in *pl_AccH and *pl_AccL.
;
;   Global Variables:
;      none
;
*******************************************************************************/

void norm_acc48(int32 *pl_AccH, int32 *pl_AccL)
{
   int32 l_temp;

   // Normalizes the high and low part of uacc48
   l_temp = *pl_AccL;
   *pl_AccL  = (l_temp & 0xFFFF);
   *pl_AccH += (l_temp >> 16);

}

/*****************************************************************************
;   Subroutine Name: round_acc48
;
;   Rounds a 48-bit accumulator A at "bit_position" and then shifts the result
;   right by "bit_position".  The accumulator will always be rounded away from 0;
;   e.g. if A=3 and bit_position=1, round_acc48 will return 2, and if
;   A=-3 and bit_position=1, round_acc48 will return -2.
;
;   The final result must fit within a 32-bit integer.  If the final result
;   cannot fit into a 32-bit integer a saturated result is returned; i.e.
;   if the final result is positive and greater than MAX_32 (the maximum 32-bit integer)
;   the function will return MAX_32, and if the final result is negative and
;   less than MIN_32 (the minimum 32-bit integer), the function will return MIN_32.
;
;   Prototype:
;       int32 round_acc48(int32 l_AccH, int32 l_AccL, int16 s_bit_position)
;
;   Input Arguments:
;      l_AccH          - The HI part of the 48 bits accum.
;      l_AccL          - The LO part of the 48 bits accum.
;       s_bit_position  - The lowers bit position preserved after shifting
;
;   Output Arguments:
;
;   Return:
;      Returns the rounded accum.
;
;   Global Variables:
;      none
;
*******************************************************************************/
int32 round_acc48(int32 l_AccH, int32 l_AccL, int16 s_bit_position)
{
   int32 l_H = l_AccH;
   int32 l_L = l_AccL;
   int32 tmp_l_H;

   /* parameter checking */
   if (s_bit_position >= 48)
   {
      return 0;
   }
   else if (s_bit_position < 0)
   {
      s_bit_position = 0;
   }

   /* find absolute value of int48 */
   norm_acc48(&l_H, &l_L);

   /* positive case */
   if(l_H >= 0)
   {
      if ((s_bit_position > 0) && (s_bit_position <= 16))
      {
         l_L += ((int32)0x1 << (s_bit_position-1));
      }
      else if (s_bit_position > 16)
      {
         l_H += ((int32)0x1 << (s_bit_position-16-1));
      }
      norm_acc48(&l_H, &l_L);

      if(l_H < 0)     // we have overflowed
      {
         l_H = 0x7FFFFFFF;
         l_L = 0x0000FFFF;
      }

      /* shift and return value */
      if(s_bit_position < 16)
      {
         /* check for overflow in l_H */
         tmp_l_H = (l_H >> (32-(16-s_bit_position)-1));   // subtract 1 from shift value to account for the sign bit         tmp_l_H = tmp_l_H & 0xFFFF;

         if (tmp_l_H == 0)
         {
            /* shift right by s_bit_position */
            l_L >>= s_bit_position;
            l_H <<= (16-s_bit_position);
            return (l_H + l_L);
         }
         else
         {
            return(MAX_32);
         }
      }
      else
      {
         /* if s_bit_position >= 16, we do not need to check for overflows since
          * the shifted result will be at most 32 bits.
          */

         /* shift right by s_bit_position */
         l_H >>= (s_bit_position-16);   // this shift is the same as  l_H <<= (16-s_bit_position)
         // except that the compiler cannot handle negative shifts
         return(l_H);
      }
   }
   else     /* negative case */
   {
      if (s_bit_position == 1)
      {
         l_L += ((int32)0x1 << (s_bit_position-1));
      }
      if ((s_bit_position > 1) && (s_bit_position <= 16))
      {
         l_L += ((int32)0x1 << (s_bit_position-1))-1;
      }
      else if (s_bit_position > 16)
      {
         l_H += ((int32)0x1 << (s_bit_position-16-1))-1;
      }

      norm_acc48(&l_H, &l_L);

      /* shift and return value */
      if(s_bit_position < 16)
      {
         /* Check for overflow in l_H.
          * To check for overflow, examine the MSB of the result to be returned
          * and all bit positions more significant than the MSB.
          *
          * If the 48-bit accumulator has been rounded up to 0, then
          * all of these bits should be 0.
          * If the rounded 48-bit accumulator is negative and has no overflow, then
          * all of these bits should be 1, which in 2's complement is -1.
          * If the round 48-bit accumulator is negative and overflows, then
          * these bits should form a number less than -1.
          * If the bits form a number greater than 0, then something has gone wrong
          * in the code.  We do not handle this condition.
          */
         tmp_l_H = (l_H >> (32-(16-s_bit_position)-1));   // subtract 1 from shift value to account for the sign bit

         if (tmp_l_H == -1)
         {
            /* shift right by s_bit_position */
            l_L >>= s_bit_position;
            l_H <<= (16-s_bit_position);
            return (l_H + l_L);
         }
         else if (tmp_l_H == 0)
         {
            return(0);
         }
         else
         {
            return(MIN_32);
         }
      }


      else
      {
         /* if s_bit_position >= 16, we do not need to check for overflows since
          * the shifted result will be at most 32 bits.
           */

         /* shift right by s_bit_position */
         l_H >>= (s_bit_position-16);   // this shift is the same as  l_H <<= (16-s_bit_position)
         // except that the compiler cannot handle negative shifts
         return(l_H);
      }
   }
}

/*****************************************************************************
;   Subroutine Name: leftshift48
;
;   Description:
;
;   Prototype:
;      leftshift48(int32 *high, int32 *low,int shift)
;
;   Input Arguments:
;
;   Output Arguments:
;      none
;
;   Return Value:
;      none
;
;   Global Variables Used:
;****************************************************************************/

void leftshift48(int32 *high, int32 *low, int shift)
{
   int iter;
   int32 last_bit;

   if(shift==0)
   {
      return;
   }

   for(iter=1; iter<= shift; iter++)
   {
      (*high) <<= 1;
      last_bit = (*low)& 0x8000; // Test the 16th bit
      (*low) <<= 1;
      (*low) &= 0xffff;

      if(last_bit!=0)
      {
         *high += 1;
      }

      // printf("%ld %ld %ld\n",*high, *low, last_bit);
   }
}

