/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1995-2003 Aware Incorporated
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------
*
*       All Rights Reserved
*
*       40, Middlesex Turnpike , Bedford, MA 01730-1413
*       Phone (617) 276 - 4000 ; Fax (617) 276 - 4001
*
*       codeswap.c
*
*
*-------------------------------------------------------------------
*/

/*
*--------------------------------------------------------------------
*
* 08/12/16 Vinay: Modified the condition to check the destination
*              address in VRx518 platform. Earlier, the DMA to
*           SDRAM was made only for BAR15. But now we have
*           variables in BAR16 address space also. Hence,
*           modified code to check the destination address
*           space till the end of BAR16.
*--------------------------------------------------------------------
*/



#ifndef CODESWAP_C
#define CODESWAP_C

#define IN_CODESWAP_C

/*******************************************************************
*
*       Include Files
*
*******************************************************************/

#include "common.h"
#include "gdata.h"
#include "codeswap.h"
#include "vdsl_state.h"
#include "vdsl_xception.h"
#include "states.h"
#ifdef DEBUG_TRAIL
#include "LeaveStatesTrail.h"
#endif //DEBUG_TRAIL
#include "PrintTrail.h"
#include "nmp.h"
#include "CustomerTasks.h"
#include "profile.h"
#include "LL_IOF.h"
#include "dsp_op.h"
#include "vcpe_swap_index.h"
#include "aux_regs.h"
#include "hercules_memrymap.h"
#include "hmp_data.h"
#include "xdma.h"

/*******************************************************************
*
*       Define Variables
*
*******************************************************************/


#define CODESWAP_MSG_WORD_0     (HMP_RW_WORD_DATA << 14)
#define CODESWAP_MSG_WORD_1     (1)     //  payload size
#define CODESWAP_MSG_WORD_2     0
#define CODESWAP_MSG_WORD_3     0
#define CODESWAP_MSG_WORD_4     1       // num of page

DATA_LD_ST_MKERNEL_BEGIN
int32 gl_SRAMImageOffset = DEFAULT_SRAM_IMAGE_OFFSET;
DATA_MAP_END;

//#define DEBUG_XDMA

#ifdef DEBUG_XDMA
int16 gl_codeswap_XMEM_XCTRL = 0;
uint32 gul_CodeSwapMode_reg;
int32 gl_ArcConfig_reg;
int32 gl_AccessPeriod_reg;
int32 gl_ArbControl_reg;
int32 gl_XmemControl_reg;
int32 gl_XmemStat_reg;
int32 gl_XdmaStartAdx_reg;
int32 gl_XdmaEndAdx_reg;
int32 gl_PagesSwapped;
int32 gl_XdmaDoneStartAdx_reg;
int32 gl_XdmaDoneEndAdx_reg;

int32 gl_Ilink1;
int32 gl_Ilink2;
#endif

/****************************************************************************
;   Prototype: void SendCodeSwapRequest(void)
;
;   Description:
;       Sends a codeswap request to host.
;
;   Arguments:
;
;   Return Value:
;       none
;
;
;****************************************************************************/

void SendCodeSwapRequest(void)
{
   int32 l_Src, l_Size, l_Dest;

   l_Size = (__gta_SwapTable[gs_CodeSwapSection].l_SwapPageSizeInLong<<2) - 1;

   // read ahead destination address; normally it is the location of swap page within the ARC
   l_Dest = __gta_SwapTable[gs_CodeSwapSection].l_DestinationAddress;

   if (l_Size < 0)
   {
      //if destination address in range of SRAM BAR15, then it's a request to
      //write data to SRAM; note that l_temp3 has the SRAM address of the "generic" SRAM page
      if ((gs_AllowSaveDataToSram) && ((gs_CodeSwapSection == VDSL_WRITE_DATA_TO_SRAM_DM_SWAPPAGE)))
      {
         // we now read the actual SRAM address and length of the transfer
         l_Size = gta_BAR15LookUpTable[guc_BAR15LookUpTableIdx][OFF_CHIP_LENGTH_INDEX];
         l_Dest = gta_BAR15LookUpTable[guc_BAR15LookUpTableIdx][OFF_CHIP_DEST_INDEX];
         l_Src = gta_BAR15LookUpTable[guc_BAR15LookUpTableIdx][OFF_CHIP_SOURCE_INDEX];

         // XDSLRTFW-2853 (Start_End)
         if (l_Size > 0)
         {
            // Check whether we need to write into SDRAM or read from SDRAM and configure
            // the XDMA engines accordingly.
            if (((l_Dest < SRAM_START_ADDR)) && ((l_Src >= SRAM_START_ADDR) && (l_Src <= SRAM_END_ADDR)))
            {
               // Read from SDRAM
               l_Src -= SRAM_START_ADDR;     // offset the SRAM start address in XDMA address space
               SetupReadXdmaBlock(l_Src, l_Dest, l_Size);
            }
            else if (((l_Src < SRAM_START_ADDR)) && ((l_Dest >= SRAM_START_ADDR) && (l_Dest <= SRAM_END_ADDR)))
            {
               // Write to SDRAM
               l_Dest -= SRAM_START_ADDR;     // offset the SRAM start address in XDMA address space
               SetupWriteXdmaBlock(l_Src, l_Dest, l_Size);
            }
            else
            {
               gt_SwapControl.uc_HwStatus = SWAP_HW_DONE;
               ReleaseXdmaLock();
               return;
            }
            gt_SwapControl.uc_HwStatus = SWAP_HW_ACTIVE;
            return;
         }
      }

      // empty page - trick the codeswap engine that this request is done
      gt_SwapControl.uc_HwStatus = SWAP_HW_DONE;
      ReleaseXdmaLock();
   }
//XDSLRTFW-2059 (Start)
   else if (gs_CodeSwapSection == VDSL_COMP_TABLE_V2_DM_SWAPPAGE)
   {
      /*
      * __gta_SwapTable (structure) contains the swap page information table. This table is generated
      * during build process from  VDSL_CPE_62.mem(VR9), VRX318_VDSL_CPE_62.mem (VRX 318), vcpe_inputs.cmd
      * and vcpe_swap_index.h files.
      *
      * Note: To read Swap page data from the SDRAM image __gta_SwapTable[x].l_SwapPageByteOffsetInSram
      *       variable can be used, here x refers to code swap index. Please add 0xB0000 to the variable
      *       value to get the correct location of the page in SDRAM incase data is being readout via winhost.
      */
      l_Src= __gta_SwapTable[gs_CodeSwapSection].l_SwapPageByteOffsetInSram + gl_SRAMImageOffset + gul_CompTableSourceAddrsOffset;
      l_Dest = gul_CompTableDestinationAddrs;
      l_Size = gul_CompTableSize;

      SetupReadXdmaBlock(l_Src, l_Dest, l_Size);
      gt_SwapControl.uc_HwStatus = SWAP_HW_ACTIVE;
   }
   //XDSLRTFW-2059 (End)
   else
   {
      // source start address is the location of swap page within SRAM
      l_Src = __gta_SwapTable[gs_CodeSwapSection].l_SwapPageByteOffsetInSram + gl_SRAMImageOffset;
      SetupReadXdmaBlock(l_Src, l_Dest, l_Size);
      gt_SwapControl.uc_HwStatus = SWAP_HW_ACTIVE;
   }
   // restore interrupt after triggering XDMA for codeswap
   // restore1_save(ul_int_status);
}




/*******************************************************************************
*
*   Prototype: uint8 RequestSwap(uint8 uc_RequestPage)
*
*   This function is called to submit a swap request into the codeswap queue.
*   If successful, it returns a handle to that request in the queue. The caller should
*   use the returned handle for further uses like checking status of that request or
*   referencing other info related to this swap request. When done with
*   the swap page, the code should free the handle.
*
*   Input Arguments:
*       uc_RequestPage:     page number
*
*   Output Arguments:
*
*   Returns:
*       INVALID_CODESWAP_HANDLE  if failed to submit the request.
*       a valid handle if successfully sumitted to queue.
*
*   Global Variables:
*       gt_SwapControl
*
*******************************************************************************/
uint8 RequestSwap1(uint8 uc_RequestPage, FlagT ft_SwapTimingOn)
{

   uint8 uc_QueueIdx;
   int8 i;
   uint8 uc_TempHandle = INVALID_CODESWAP_HANDLE;        // if fails to add to queue
   SwapRequest_t *pt_RequestQueue;

   if (uc_RequestPage < MAX_SWAP_PAGES)
   {
      uc_QueueIdx = gt_SwapControl.uc_ActiveTransferIndex;
      for(i = 0; i < MAX_SWAP_REQUESTS; i++)
      {
         pt_RequestQueue = &gt_SwapRequestQueue[uc_QueueIdx];
         if (pt_RequestQueue->uc_Status == SWAP_FREE_HANDLE)
         {
            pt_RequestQueue->uc_SwapPageNum = uc_RequestPage;
            pt_RequestQueue->uc_Status = SWAP_PENDING;
            pt_RequestQueue->us_FwSwapDuration = 0;
            pt_RequestQueue->us_HwSwapDuration = 0;
            pt_RequestQueue->ft_SwapTimingOn = ft_SwapTimingOn;
            pt_RequestQueue->uc_LookUpTableIdx = 0xFF;

            switch(uc_RequestPage)
            {
            default:
               pt_RequestQueue->s_MaxSwapDuration = gs_MaxSwapDuration;
               break;
            }
            uc_TempHandle = uc_QueueIdx;
            break;
         }
         uc_QueueIdx = (uc_QueueIdx + 1) & MAX_SWAP_REQUESTS_MASK;
      }
   }

   return(uc_TempHandle);
}

// this function is to minimize change to the existing code when
// requesting codeswap.
uint8 RequestSwap(uint8 uc_RequestPage)
{
   if (gul_dbgSnifferControl & SNIFFER_DISABLE_SWAP_TIMING)
   {
      return(RequestSwap1(uc_RequestPage, SWAP_TIMING_OFF));
   }
   else
   {
      return(RequestSwap1(uc_RequestPage, SWAP_TIMING_ON));
   }
}

uint8 RequestSwapOffChip(uint8 uc_RequestPage, FlagT ft_SwapTimingOn, uint8 uc_LookUpTableIdx)
{
   uint8 uc_Temp;

   if ((uc_Temp = RequestSwap1(uc_RequestPage, ft_SwapTimingOn)) != INVALID_CODESWAP_HANDLE)
   {
      gt_SwapRequestQueue[uc_Temp].uc_LookUpTableIdx = uc_LookUpTableIdx;
   }

   if (uc_Temp == INVALID_CODESWAP_HANDLE)
   {
      DSH_SendEvent(DSH_EVT_CODESWAP_ISSUE,sizeof(uint8),&uc_RequestPage);
   }
   return(uc_Temp);
}

/*******************************************************************************
*
*   Prototype: void FreeSwapHandle(uint8 *puc_SwapHandle)
*   This function is called to free the codeswap handle.
*   Once the handle is free, the codeswap server will use it to fullfill other
*   requests.
*
*   Input Arguments:
*   puc_SwapHandle: pointer to the current handle.
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*       gt_SwapControl
*
*******************************************************************************/
void FreeSwapHandle(uint8 *puc_SwapHandle)
{
   if (*puc_SwapHandle < MAX_SWAP_REQUESTS)
   {
      gt_SwapRequestQueue[*puc_SwapHandle].uc_Status = SWAP_FREE_HANDLE;
      *puc_SwapHandle = INVALID_CODESWAP_HANDLE;
   }
}

/*******************************************************************************
*
*   Prototype: uint8 GetRequestStatus(uint8 uc_SwapHandle)
*   This function is called to obtain the status of the swap request referred
*   by the handle.
*
*   Input Arguments:
*   uc_SwapHandle:  a valid handle
*
*   Output Arguments:
*
*   Returns:
*       status of the specified request.
*       if handle is invalid, SWAP_HANDLE_ERROR will be returned.
*
*   Global Variables:
*
*******************************************************************************/
uint8 GetRequestStatus(uint8 uc_SwapHandle)
{
   if (gt_SwapControl.uc_HwStatus == SWAP_HW_DISABLED)
   {

      return(SWAP_HW_DISABLED);
   }
   else if (uc_SwapHandle < MAX_SWAP_REQUESTS)
   {
      return(gt_SwapRequestQueue[uc_SwapHandle].uc_Status);
   }
   else
   {
      //SetCodeSwapError();
      return(SWAP_HANDLE_ERROR);
   }
}

/*******************************************************************************
*
*   Function: UpdateHwSwapStatus
*   This function is called to read the XDMA/ME control to find out the status
*   of the current XDMA/ME transfer and updates the gt_SwapControl.uc_HwStatus
*   accordingly.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*       XDMA or ME status
*
*   Global Variables:
*
*******************************************************************************/
uint8 UpdateHwSwapStatus(void)
{
//--------------------------------------------------------
// if TARGET_ARC, ASIM, or HW
//--------------------------------------------------------
   //--------------------------------------------------------
   // read XDMA or ME interface for actual status
   //--------------------------------------------------------
   // if Codeswap is via ME
   //--------------------------------------------------------
   {
      extern int32 gl_XmemXstat;
      extern int32 gl_XmemDspStat;

      gl_XmemDspStat = GetDspReg(DSP_INT_STAT);

      if ((gt_SwapControl.uc_HwStatus != SWAP_HW_ERROR) && // if there is already a codeswap error, skip status check
            (gl_XmemDspStat & DSP_INT_STAT_XDMA_DONE_MASK))
      {

         gl_XmemXstat = GetDspReg(DSP_XDMA_STAT);

         if (gl_XmemXstat & DSP_XDMA_STAT_ERR_MASK)
         {
            gt_SwapControl.uc_HwStatus = SWAP_HW_ERROR;
         }
         else
         {
            gt_SwapControl.uc_HwStatus = SWAP_HW_DONE;
         }

         // Clear the interrupt for XDMA
         SetDspReg(DSP_INT_STAT, DSP_INT_STAT_XDMA_DONE_MASK);
         ReleaseXdmaLock();

      }
   }
   //--------------------------------------------------------
   // if Codeswap HW is complete, then log information
   //--------------------------------------------------------

   if (gt_SwapControl.uc_HwStatus == SWAP_HW_DONE)
   {
#ifdef DEBUG_TRAIL
      // log code swap completion to states trail
      DebugTrail1(3,0x0001,0x0000,
                  (int16)0xAABB,
                  (int16)gs_CodeSwapSection,
                  (int16)gl_RxSymbolCount);
#endif // DEBUG_TRAIL
   }

   //--------------------------------------------------------
   // return the HwStatus
   //--------------------------------------------------------
   return(gt_SwapControl.uc_HwStatus);

}


/*******************************************************************************
*
*   Prototype: void RunCodeSwapEngine(void)
*   This function will do the followings:
*   1. Update fw/hw timers for each request that is pending/active
*   2. Check HW (ME or XDMA) to see if previous transfer is completed and
*      update the status of the active request accordingly.
*   3. If HW (ME or XDMA) has finished previous transfer and if there is
*      new pending request, set up the next transfer
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*       None
*
*   Global Variables:
*       gt_CodeSwapControl.
*
*******************************************************************************/
void RunCodeSwapEngine(void)
{
   uint8 uc_QueueIdx;
   SwapRequest_t *pt_TheRequest;
   FlagT ft_SwapTimingOn;
   int32 i;

   if (gt_SwapControl.uc_HwStatus != SWAP_HW_DISABLED)
   {
      // update counters and status.
      uc_QueueIdx = gt_SwapControl.uc_ActiveTransferIndex;
      for(i = 0; i < MAX_SWAP_REQUESTS; i++)
      {
         pt_TheRequest = &gt_SwapRequestQueue[uc_QueueIdx];
         if (pt_TheRequest->ft_SwapTimingOn == SWAP_TIMING_ON)
         {
            switch(pt_TheRequest->uc_Status)
            {
            case SWAP_SUBMITTED:
            case SWAP_PENDING:
            case SWAP_HW_QUEUED:
               pt_TheRequest->us_FwSwapDuration++;
               break;

            case SWAP_HW_ACTIVE:
               pt_TheRequest->us_FwSwapDuration++;
               pt_TheRequest->us_HwSwapDuration++;
               pt_TheRequest->s_MaxSwapDuration--;
               break;

            case SWAP_HW_ERROR:
            case SWAP_HW_DONE:
               pt_TheRequest->s_MaxSwapDuration--;
               break;
            }
         }
         uc_QueueIdx = (uc_QueueIdx + 1) & MAX_SWAP_REQUESTS_MASK;
      }

      uc_QueueIdx = gt_SwapControl.uc_ActiveTransferIndex;
      for(i = 0; i < MAX_SWAP_REQUESTS; i++)
      {
         pt_TheRequest = &gt_SwapRequestQueue[uc_QueueIdx];
         ft_SwapTimingOn = pt_TheRequest->ft_SwapTimingOn;
         switch(pt_TheRequest->uc_Status)
         {
         case SWAP_PENDING:
            if (gt_SwapControl.uc_HwStatus == SWAP_HW_IDLE)
            {
               gt_SwapControl.uc_ActiveTransferIndex = uc_QueueIdx;

               // set up for the next transfer
               guc_CodeSwapMsgReq = 1;
               gs_CodeSwapSection = pt_TheRequest->uc_SwapPageNum;

               guc_BAR15LookUpTableIdx = pt_TheRequest->uc_LookUpTableIdx;
               pt_TheRequest->uc_Status = SWAP_SUBMITTED;
               gt_SwapControl.uc_HwStatus = SWAP_HW_QUEUED;

#ifdef DEBUG_TRAIL
               // log code swap request to states trail
               DebugTrail1(3,0x0001,0x0000,
                           (int16)0xAAAA,
                           (int16)gs_CodeSwapSection,
                           (int16)gl_RxSymbolCount);

               // Debug code to analyze codeswap requests (commented out by default in order to avoid flooding Tracee with events which are only required for specific debugging)
               // DSH_SendEvent(DSH_EVT_CODESWAP_ADDED,3*sizeof(int32),&gta_BAR15LookUpTable[guc_BAR15LookUpTableIdx][OFF_CHIP_SOURCE_INDEX]);

#endif // DEBUG_TRAIL
            }
            break;

         case SWAP_SUBMITTED:
         case SWAP_HW_QUEUED:
         case SWAP_HW_ACTIVE:
            if ((ft_SwapTimingOn == SWAP_TIMING_ON) && (pt_TheRequest->s_MaxSwapDuration == 0))
            {
               pt_TheRequest->uc_Status = SWAP_TIMEOUT;
               //gt_SwapControl.uc_HwStatus = SWAP_HW_DISABLED;
            }
            else
            {
               pt_TheRequest->uc_Status = UpdateHwSwapStatus();
            }
            break;

         case SWAP_HW_ERROR:
            DSH_SendEvent(DSH_EVT_CODESWAP_ISSUE,sizeof(uint8) , (void *)pt_TheRequest->uc_SwapPageNum);
            if ((ft_SwapTimingOn == SWAP_TIMING_OFF) || (pt_TheRequest->s_MaxSwapDuration == 0))
            {
               pt_TheRequest->uc_Status = SWAP_ERROR;
               //gt_SwapControl.uc_HwStatus = SWAP_HW_DISABLED;

               gt_SwapControl.uc_HwStatus = SWAP_HW_IDLE;       // release lock on hw

               // updating uc_ActiveTransferIndex to allow prev entry in queue for debugging
               gt_SwapControl.uc_ActiveTransferIndex = (uc_QueueIdx + 1) & MAX_SWAP_REQUESTS_MASK;

            }
            break;

         case SWAP_HW_DONE:
            if ((ft_SwapTimingOn == SWAP_TIMING_OFF) || (pt_TheRequest->s_MaxSwapDuration == 0))
            {
               pt_TheRequest->uc_Status = SWAP_DONE;
               gt_SwapControl.uc_HwStatus = SWAP_HW_IDLE;       // release lock on hw


#ifdef DEBUG_TRAIL
               // log code swap completion to states trail
               DebugTrail1(3,0x0001,0x0000,
                           (int16)0xBBBB,
                           (int16)gs_CodeSwapSection,
                           (int16)gl_RxSymbolCount);
#endif // DEBUG_TRAIL

               // Debug code to analyze codeswap requests (commented out by default in order to avoid flooding Tracee with events which are only required for specific debugging)
               //DSH_SendEvent(DSH_EVT_CODESWAP_EXECUTED,sizeof(int16),&gs_CodeSwapSection);

               // advance ActiveQueueIndex so queue would behave as a log buffer
               gt_SwapControl.uc_ActiveTransferIndex = (uc_QueueIdx + 1) & MAX_SWAP_REQUESTS_MASK;
            }
            break;

         case SWAP_TIMEOUT:
            // do something here so that new request will not be mixed up with timedout
            // request.
            // for THRU_ME, should call updateHw until swap is done, then set to HW_IDLE
            // for XDMA, stop the transfer. Update status to HW_IDLE

            // the transfer might be still going on. Need to halt it.
            int32 l_temp = 0;
            SetDspReg(ARC_XMEM_XCTRL_REG,l_temp);
            // Clear the interrupt for XDMA
            SetDspReg(DSP_INT_STAT, DSP_INT_STAT_XDMA_DONE_MASK);

            ReleaseXdmaLock();

            gt_SwapControl.uc_HwStatus = SWAP_HW_IDLE;       // release lock on hw

            // updating uc_ActiveTransferIndex to allow prev entry in queue for debugging
            gt_SwapControl.uc_ActiveTransferIndex = (uc_QueueIdx + 1) & MAX_SWAP_REQUESTS_MASK;
            DSH_SendEvent(DSH_EVT_CODESWAP_ISSUE,sizeof(uint8) , (void *)pt_TheRequest->uc_SwapPageNum);
            break;

         }
         uc_QueueIdx = (uc_QueueIdx + 1) & MAX_SWAP_REQUESTS_MASK;
      }
   }
}

/*******************************************************************************
*
*   Prototype: int16 PollForCodeSwapDone(uint8 uc_SwapPageNum, uint8 uc_PageHandle)
*
*       The PollForCodeSwapDone function checks the status  for the swap request
*   specified by the parameter uc_PageHandle.  If the swap request times out, then
*   this function forces the tx and rx to the fail states.  If the handle is invalid,
*   this function returns the SWAP_HANDLE_ERROR code.  For other conditions, this
*   function returns the status.
*
*   This function can be used to poll while waiting for a swap to complete.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*       None
*
*   Global Variables:
*
*******************************************************************************/
uint8 PollForCodeSwapDone(uint8 uc_SwapPageNum, uint8 uc_PageHandle)
{
   uint8 uc_SwapStatus;

   // if the page number of the current handle to be checked is equal to the page number of the calling function, then we check the status
   // why is this the case? Why we don't check the status for any handle?
   if ((uc_PageHandle < INVALID_CODESWAP_HANDLE) &&                      // if a valid page request
       (gt_SwapRequestQueue[uc_PageHandle].uc_SwapPageNum == uc_SwapPageNum))
   {
      uc_SwapStatus = GetRequestStatus(uc_PageHandle);
   }
   else if(uc_PageHandle == CODESWAP_NO_ACTION)
   {
      return(SWAP_DONE);
   }
   else
   {
      //removed debug stream since it does not only report a codeswap issue, it also reports if a codeswap is waiting due to multiple codeswaps being in the pipe
      //this is triggered too often during failstate and training, hence removing it
      //DSH_SendEvent(DSH_EVT_CODESWAP_ISSUE,sizeof(uint8),&uc_PageHandle);
      return(SWAP_HANDLE_ERROR);
   }

   return(uc_SwapStatus);
}




#endif
