/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   cosine.c
 *
 *   Fast Cosine implementation.
 *------------------------------------------------------------------------
 */

#include "typedef.h"
#include "cosine.h"
#include "gdata.h"
#include "pll.h"
#include "mul.h"
#include "dsp_op.h"
#include "ieee_flt.h"
#include <stdio.h>
#include <stdlib.h>


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: FastCosine
 *
 *  Description: Computes 4 quadrant cos(x), where x is in radians and
 *               1 radian is normalized to 8192.
 *               x is assumed to be in the range [-pi, +pi]
 *
 *  Returns: 32767*cos(x)
 *
 *  Notes:
 *
 *     Aproximate cos(x) by looking up the points from a sparse cosine table
 *     and interpolate linearly between the points. The sparse table is
 *     generated by computing cos(x) for every (radian/512).
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 FastCosine(int16 s_phase)
{

   int16 s_index, s_remainder, s_diff, s_flag = 0;
   int32 l_result;

   s_phase = abs(s_phase); // cos(-x) = cos(x)

   if (s_phase > PLL_HALF_PI_RADIANS)   // use symmetry around pi/2
   {
      s_phase = (PLL_PI_RADIANS-s_phase);
      s_flag  = 1;
   }

   s_index     = s_phase >> 8; // divide by 256
   s_remainder = s_phase - (s_index << 8);


   // Interpolate linearly between looked up points.
   s_diff    = gsa_ShortCosTable[s_index+1] - gsa_ShortCosTable[s_index];
   MULS16(l_result, s_diff, s_remainder);

   l_result  = (l_result + (1 << 7)) >> 8; // round
   l_result += gsa_ShortCosTable[s_index];

   if (s_flag)
   {
      l_result = -l_result;
   }
   return (int16)(l_result);
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name: FastSine
 *
 *  Description: Computes 4 quadrant sin(x), where x is in radians and
 *               1 radian is normalized to 8192.
 *               x is assumed to be in the range [-pi, +pi]
 *
 *  Returns: 32767*sin(x)
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 FastSine(int16 s_phase)
{
   int32 l_phase;

   l_phase = (int32)s_phase-PLL_HALF_PI_RADIANS;  // sin(x) = cos(x-pi/2)
   if (l_phase <= -PLL_PI_RADIANS)  // check for wrap around
   {
      l_phase += 2*PLL_PI_RADIANS;
   }

   return FastCosine((int16)l_phase);
}

/*****************************************************************************
;   Prototype:
;      int16 Modulo2PI(int32 l_phase)
;
;   Description:
;      This subroutine performs modulo 2*PI on the input phase.
;      Output is translated into the range of -pi to pi.
;
;   Arguments:
;      l_phase -- input phase in radians (1 radian = 8192).
;
;   Return:
;      int16 -PLL_PI_RADIANS to PLL_PI_RADIANS (PLL_PI_RADIANS = 25736)
;
;   Global Variables: none
;
;*************************************************************************************/
Float32 Modulo2PI(Float32 f_phase)
{

   int16 i;

   //if (f_phase < 0)
   if (cmpgtf32(0,f_phase))
   {
      //f_phase = -f_phase;
      f_phase = mpyf32(f_phase,F32_MINUS1);
      i = 1;
   }
   else
   {
      i = 0;
   }

   // map to [-pi pi]
   while (cmpgtf32(f_phase,F32_PI_RADIANS_TIMES8192))
   {
      f_phase = subf32(f_phase,F32_2PI_RADIANS_TIMES8192);
   }

   if (i == 1)
   {
      //f_phase = -f_phase;
      f_phase = mpyf32(f_phase,F32_MINUS1);
   }

   return f_phase;
}

/*****************************************************************************
;   Prototype:
;      void RotateTone(int16 s_NumSamples, int16 s_ToneIdx, int16 *psa_Tone)
;
;   Description:
;      This subroutine rotates a complex tone by the proper amount when
;       the frame is shifted by s_NumSamples due to frame alignment.
;
;       the amount of phase adjust
;           alpha = (s_NumSamples*s_ToneIdx*2*pi)/gs_RxFftLength
;       where alpha is expressed in radians where 1 radian = 8192
;
;   Input Arguments:
;      s_NumSamples    -- the number of samples shifted in frame alignment.
;       s_ToneIdx      -- tone index of the complex tone
;       psa_Tone       -- ptr to real and imag part of the complex tone.
;
;   Output Arguments: 0 - No overflow
;                     1 - Overflow
;
;*************************************************************************************/
int16 RotateTone(int16 s_NumSamples, int16 s_ToneIdx, int16 *psa_Tone)
{
   // @note AH07102015 New Floating point implementation of angle calculation
   // It turned out that the calculation of the rotation angle in fixed point is not accurate enough.
   // The bigger the angle got, the bigger the rotation error was. With this, the SNR based frame alignment handler did not work properly.
   // Therefore changed the angle calculation to floating point (Float32). Detailed information see XDSLRTFW-2487

   Float32 f_Acc0, f_Acc1,f_temp;
   Float32 f_Angle, f_cos, f_sin, f_Real, f_Imag;
   int32   l_AccReal, l_AccImag, l_temp;
   uint16  us_RxFftLen;
   int16   s_Angle, s_c, s_s;
   int16   s_OverflowFlag;

   s_OverflowFlag = FALSE;

   if (s_NumSamples == 0)
   {
      return(s_OverflowFlag);
   }

   us_RxFftLen = (uint16)(1<<gs_RxLog2FftLength);
   l_temp = s_ToneIdx * (-s_NumSamples);

   f_temp = int32toFloat32(l_temp);

   //f_Angle = SNRH_PI_TIMES_2 * (float)l_temp / (float)us_RxFftLen;
   f_temp = mpyf32(f_temp,F32_M_2PI);
   f_Angle = divf32(f_temp,int2f32((int16)us_RxFftLen));

   // FastCosine() expects input angle to be normalized to 8192
   f_Angle = mpyf32(f_Angle , F32_8192);

   //reduce Angle to range between -pi / pi
   f_Angle = Modulo2PI(f_Angle);

   s_Angle = f32toint16(f_Angle,0);

   s_c = FastCosine(s_Angle);
   s_s = FastSine(s_Angle);

   // output of FastCosine(x) is cos(x)*32767 reduce to cos(x) again
   f_cos = int2f32(s_c);
   f_sin = int2f32(s_s);
   f_cos = divf32(f_cos, F32_32767);
   f_sin = divf32(f_sin, F32_32767);

   //f_Real = (float)(*paTone);
   f_Real = int2f32(*psa_Tone);

   //f_Imag = (float)(*(paTone+1));
   f_Imag = int2f32(*(psa_Tone+1));

   // Real part of rotated tone
   //f_Acc0 = f_Real * f_cos;
   f_Acc0 = mpyf32(f_Real , f_cos);

   //Acc1 = Imag * f_sin;
   f_Acc1 = mpyf32(f_Imag , f_sin);

   //l_AccReal = (int32)(Acc0 - Acc1);
   l_AccReal = f32toint32(subf32(f_Acc0 , f_Acc1),0);

   // f_Imag part of rotated tone
   //f_Acc0 = Real * s;
   f_Acc0 = mpyf32(f_Real , f_sin);

   //Acc1 = Imag * f_cos;
   f_Acc1 = mpyf32(f_Imag , f_cos);

   //AccImag = (int32)(Acc0 + f_Acc1);
   l_AccImag = f32toint32(addf32(f_Acc0,f_Acc1),0);

   // Check for overflow
   if ((Abs32(l_AccReal) > 0x7FFF) || (Abs32(l_AccImag) > 0x7FFF))
   {
      l_AccReal >>= 1;
      l_AccImag >>= 1;
      s_OverflowFlag = TRUE;
   }

   *psa_Tone     = (int16)(l_AccReal);
   *(psa_Tone+1) = (int16)(l_AccImag);

   return(s_OverflowFlag);
}

