/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2002 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   File Name: noiseacc.c
;
;   This file contains the noiseacc function used in SNR calculation
;
;***************************************************************************/

// ***********************************************************************************************************
// mtkernel.c
//
// History
//
// 19/09/2013 Varun/Palaksha/Ram : Added new CMV INFO 238 0 ==> gives the Max mips count(Foregraound + Tc task) in showtime if test 0 0 is 0x1000
//                    Note that in VDSL Rx enters showtime first(look timing diagram of VDSL2 for more details). since we want showtime mips when both tx
//                    and rx in showtime we enable the profile tasking when tx enters showtime. foreground mips peaking occurs at tx in RPSynchro6 state.
//                    Hence condition is checked in foreground in showtime. Following additional improvements done
//                  - Added FG Task Averege MIPS profiling
//                  - Added a optimisation code to reduce the RxTc Mips
//                     Grep for "XDSLRTFW-1243 Feature_ALL_VDSL2_ALL_CMV_MipsProfiling"
//
// ************************************************************************************************************

#include "common.h"
#include "acc48_ops.h"
#include "noiseacc.h"
#include "gdata.h"
#include "dsp_op.h"

#include "xy_iof.h"

//*************************************************
//
#ifdef INCLUDE_NON_CACHE_FUNCTIONS
//
//*************************************************


/*
*-------------------------------------------------------------------------------
*
*   Name : NoiseAcc
*
*   Description:
*      This routine accumulates the absolute value of a 16 bit complex vector
*      into a 64 bit accumulator.
*       For each tone i,
*      1) Compute the power
*             pow[i] = (psa_input_buf[i*2])^2 + (psa_input_buf[i*2+1])^2
*
*      2) Add pow[i] to the accumulated noise power array, psa_output_buf[].
*      The code uses a 64-bit accumulator implemented in 2 32-bit counters.
*
*   Memory Allocation:
*      The output and input arrays need to be in X memory and should not overlap.
*      The size of input array is "2*s_num_tones" words.
*      The size of ouput array is "4*s_num_tones" words.
*
*   Implementation:
*      The routine uses the ARC dual mac to do a single cycle computation of
*      the absolute value of each complex number. Also, the accumulator access is
*      pipelined such that the ARC does not stall after the 3 cycle mul instruction
*
*   Prototype: void NoiseAcc(int32 *psa_output_buf, int16 *psa_input_buf, int16 s_num_tones)
*
*  Input Arguments:
*      int16 *psa_input_buf   - Pointer to input vector in X memory
*      int16 s_num_tones      - Number of complex elements in input buffer
*
*  Output Arguments:
*      int16 *psa_output_buf   - Pointer to output accumulator vector in X memory
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void NoiseAcc(int32 *pla_output_buf, int16 *psa_input_buf, int16 s_num_tones)
{
   int32 l_XYDec_OuputVector_X32, l_XYDec_InputVector_X32;

   l_XYDec_OuputVector_X32 = XYAddrDecode(pla_output_buf, ADDR_MODE_32);
   l_XYDec_InputVector_X32 = XYAddrDecode((int32 *)(void *)psa_input_buf, ADDR_MODE_32);

   if(CheckXYAddrRange((int32 *)(void *)psa_input_buf, X_MEM) == PASS)
   {
      NoiseAccLoopXX(l_XYDec_OuputVector_X32, l_XYDec_InputVector_X32, s_num_tones);
   }
   else
   {
      NoiseAccLoopXY(l_XYDec_OuputVector_X32, l_XYDec_InputVector_X32, s_num_tones);
   }

}

/*
*-------------------------------------------------------------------------------
*
*   Name : NoiseAccXY
*
*   Description:
*      This routine accumulates the absolute value of a 16 bit complex vector
*      into a 64 bit accumulator.
*       For each tone i,
*      1) Compute the power
*             pow[i] = (psa_input_buf[i*2])^2 + (psa_input_buf[i*2+1])^2
*
*      2) Add pow[i] to the accumulated noise power array, psa_output_buf[].
*      The code uses a 64-bit accumulator implemented in 2 32-bit counters.
*
*   Memory Allocation:
*      The output and input arrays need to be in X and Y memory respectively.
*      The size of input array is "2*s_num_tones" words.
*      The size of ouput array is "4*s_num_tones" words.
*
*   Implementation:
*      The routine uses the ARC dual mac to do a single cycle computation of
*      the absolute value of each complex number. Also, the accumulator access is
*      pipelined such that the ARC does not stall after the 3 cycle mul instruction
*
*   Prototype: void NoiseAccXY(int32 *psa_output_buf, int16 *psa_input_buf, int16 s_num_tones)
*
*  Input Arguments:
*      int16 *psa_input_buf   - Pointer to input vector in Y memory
*      int16 s_num_tones      - Number of complex elements in input buffer
*
*  Output Arguments:
*      int16 *psa_output_buf   - Pointer to output accumulator vector in X memory
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/




//*************************************************
//
#endif // #ifdef INCLUDE_NON_CACHE_FUNCTIONS
//
//*************************************************








//*************************************************
//
#ifdef INCLUDE_CACHE_FUNCTIONS
//
//*************************************************

/********************************************************************************************
;   Subroutine Name: RoundNoiseAccum
;
;   Description:
;      This routine rounds and shifts the 64-bit accumulator array generated by
;      the NoiseAcc function into an array of 32 bit numbers.  RoundNoiseAccum does not
;      check for saturation or overflow.  The user is responsible for ensuring that the
;      shift value s_bit_position is large enough so that all the final array entries will
;      fit into a 32-bit integer.
;
;   Prototype:
;      void RoundNoiseAccum(int32 *pla_in, int32 *pla_round, int16 s_first_chan, int16 s_last_chan, int16 s_bit_position);
;
;   Input Arguments:
;      pla_in         -- pointer to the array of 64-bit accumulators
;      s_first_chan   -- first channel
;      s_last_chan    -- last channel
;      s_bit_position -- the lowest bit position preserved after shifting
;
;   Output Arguments:
;      pla_round      -- pointer to the array of rounded and shifted numbers
;
;   Return Value:
;      none
;
;   Global Variables:
;      none
;
;****************************************************************************/
int32 round_acc64(int32 l_H, uint32 ul_L, int16 s_bit_position)
{
   uint32 ul_RND;

   /* initializing and parameter checking */
   if (s_bit_position >= 64)
   {
      return 0;
   }
   else if (s_bit_position > 32)
   {
      ul_RND = ((int32)0x1 << (s_bit_position-32-1));
   }
   else if (s_bit_position > 0)
   {
      ul_RND = ((int32)0x1 << (s_bit_position-1));
   }
   else if (s_bit_position <= 0)
   {
      s_bit_position = 0;
      ul_RND = 0;
   }

   /* positive case */
   if(l_H >= 0)
   {
      if (s_bit_position > 32)
      {
         if (ul_RND > (uint32)(MAX_32 - l_H))
         {
            // Overflow while rounding
            return(MAX_32);
         }

         l_H += ul_RND;
         l_H >>= (s_bit_position-32);
         return (l_H);
      }
      else
      {
         if (ul_RND > (0xffffffff - ul_L))
         {
            // Add with carry
            ul_L += ul_RND;
            l_H += 1;
         }
         else
         {
            ul_L += ul_RND;
         }

         if ((l_H && (norm_l(l_H) < (32-s_bit_position))) ||
               (!l_H && ((ul_L >> s_bit_position) > MAX_32)))
         {
            // Overflow while averaging
            return(MAX_32);
         }

         ul_L >>= s_bit_position;
         l_H <<= (32-s_bit_position);
         return (int32)(l_H + ul_L);
      }
   }
   /* negative case */
   else
   {
      /* Not Implemented Yet */
      return (0);
   }
}

void RoundNoiseAccum(int32 *pla_in, int32 *pla_round, int16 s_first_chan, int16 s_last_chan, int16 s_bit_position)
{
   int16 i, s_num_chan;

   s_num_chan = s_last_chan - s_first_chan+1;
   for (i=0; i<s_num_chan; i++)
   {
      *pla_round++ = round_acc64(*(pla_in+1), (uint32)*pla_in, s_bit_position);
      pla_in += 2;
   }
}

//XDSLRTFW-1243 Feature_ALL_VDSL2_ALL_CMV_MipsProfiling (Start)
//Added new function to work based on the output format of ReadAccumulatedNoise2()
void RoundNoiseAccum2(int32 *pla_in, int32 *pla_round, int16 s_first_chan, int16 s_last_chan, int16 s_bit_position)
{
   int16 i, s_num_chan;

   s_num_chan = s_last_chan - s_first_chan+1;
   for (i=0; i<s_num_chan; i++)
   {
      *pla_round++ = round_acc64(*(pla_in+gt_DDSnrFdqConfig.s_DDSnrFdqNumChannelsPerGroup), (uint32)*pla_in, s_bit_position);
      pla_in++;
   }
}
//XDSLRTFW-1243 Feature_ALL_VDSL2_ALL_CMV_MipsProfiling (End)

//*************************************************
//
#endif // ifdef INCLUDE_CACHE_FUNCTIONS
//
//*************************************************
