/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: SelectPilot.c
*
*   This file contains functions to selects pilot tone index.
*
*-------------------------------------------------------------------------------
*/

// ************************************************************************
// BgSelectMultiPilot.c
//
// History
// 27/l1/2012 Ram: Merged IOP fix corresponding to JIRA XDSLRTFW-458
//                 Grep for "XDSLRTFW-458: IOP_ALL_VDSL2_ALL_TACorrection"
//
// ************************************************************************

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "cmv.h"
#include "mul.h"
#include "ConvertToDB.h"
#include "SelectPilot.h"

// XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BgSelectMultiPilot(void)
*
*   This function selects multiple pilot tones based on SNR during R-O-SIGNATURE with the knowledge of the band plan.
*   It tries to find 3 pilot tones which are distributed one per DS band. If the SNR or band-plan allows less than 3 pilot tones,
*   the unused pilot tones get the pilot tone index 0 and will not be communicated to the CO
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      gt_PilotConfig.ta_PilotTones[0].s_PilotToneIdx: (O) pilot tone index
*
*-------------------------------------------------------------------------------
*/

void BgSelectMultiPilot(void)
{
   int16 i, s_BandSize;
   int16 s_LeftChannel, s_RightChannel;
   int16 s_PilotSelectOffset;
   int16 s_HighestAllowedPilotTone;          // Choose pilot tone selection band

   // Backup pilot tone 0 index and metric, in case the new search algorithm does not find any pilot tone
   int16 s_PT0_Idx_bkp = gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_0].s_PilotToneIdx;
   int32 l_PT0_Metric_bkp = gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_0].l_Metric;

   // Multipilot tone feature, search one pilot tone per DS band
   for (i=PT_ARRAY_IDX_0; i<=PT_ARRAY_IDX_2; i++)
   {
      s_HighestAllowedPilotTone = gsa_RxBandRightChannel[i];
      // set the start tone index away from the band edge by minimum of predefined value and 1/4 of band size
      s_BandSize = s_HighestAllowedPilotTone - gsa_RxBandLeftChannel[i] + 1;

      ChoosePilotSelectBand(gpsa_MeasuredSnrBuf, &gsa_RxBandLeftChannel[i], &s_HighestAllowedPilotTone);

      //If no band is selected, give another chance to try
      if(s_HighestAllowedPilotTone < gsa_RxBandLeftChannel[i])
      {
         s_HighestAllowedPilotTone = gsa_RxBandRightChannel[i];
      }

      if ((gt_PilotConfig.s_PilotToneControl & ADD_OFFSET_FROM_BANDEDGE) &&
            (gt_Kl0ElectricalLength.s_kl0_estimate > gs_Kl0_thresh_pilot_select))
      {
         // Left band edge
         s_PilotSelectOffset = MIN(gs_PilotSelectOffset, (s_BandSize >> 2));
         s_LeftChannel = gsa_RxBandLeftChannel[i] + s_PilotSelectOffset;
         // Right band edge
         s_PilotSelectOffset = MIN(gs_PilotSelectRightEdgeOffset, (s_BandSize >> 2));
         s_RightChannel = gsa_RxBandRightChannel[i] - s_PilotSelectOffset;
      }
      else
      {
         s_LeftChannel = gsa_RxBandLeftChannel[i];
         s_RightChannel = gsa_RxBandRightChannel[i];
      }

      // We need to make sure the pilot tone is not selected among the US ISI tones
      if (gft_US0BandUsed && (s_LeftChannel <= gsa_TxBandRightChannel[0]+32))
      {
         s_LeftChannel = gsa_TxBandRightChannel[0]+33;

         // Silvano: a burst of DS CRC at showtime transition have been observed on mid-range loops when the pilot is selected
         // on the DS1 tones affected by US0 echo: this burst of DS CRC seems to be due a slight rotation of the initial showtime FDQ, which is
         // then corrected by FDQ update. Selecting the pilot tone away from the echo dominated tones solves the problem.
         // We apply this fix only if kl0 > 35dB (approx. 4kft).
         if ((s_LeftChannel <= gsa_TxBandRightChannel[0]+45) && (gt_Kl0ElectricalLength.s_kl0_estimate > 350))
         {
            s_LeftChannel = gsa_TxBandRightChannel[0]+45;
         }
      }
      //Avoid using the tones near the right band edge which can be affected by the US echo signal
      if(gsa_RxBandRightChannel[i] >= gs_HighestAllowedRxTone)
      {
         s_RightChannel = gs_HighestAllowedRxTone - MIN(gs_PilotSelectRightEdgeOffset, (s_BandSize >> 2));
      }

      // Note AH (11/09/2017): During analysis of NE_LOS link drop after pilot tone switching, it was
      // found that most of these link drop cases had a pilot tone index > 4095
      // As of now, limit the maximum pilot tone index to this value. As far as I know we did never
      // do investigation about PLL parameters like ki, kp for pilot tone indexes larger
      // than this value.
      // XDSLRTF-4056: Restrict the maximum pilot tone index to 3856.
      // This tone is a grid tone for vector engines using a grid of 16 + x * 32 tones, and it avoids
      // using tones above as PT, where the SNR is sometimes bad due to SRC effects and sub-optimal
      // vectoring. Was 4095, now 3856.
      // Also the offset from the right edge is changed to 32, to avoid using a pilot tone,
      // where the vector engine is not extrapolating the coefficients correctly.

      if(s_RightChannel >= gs_MaxPilotToneIndex)
      {
         s_RightChannel = gs_MaxPilotToneIndex;
      }

      // Perform the pilot tone search within the provided index boarders
      SelectPilotTone(gpsa_MeasuredSnrBuf, s_LeftChannel, s_RightChannel, &gt_PilotConfig.ta_PilotTones[i].s_PilotToneIdx, &gt_PilotConfig.ta_PilotTones[i].l_Metric);
   }

   //XDSLRTFW-458: IOP_ALL_VDSL2_ALL_TACorrection (Start)
   if(gt_CustomerIopBits.us_ECI & ECI_DPBO_LONGREACH_PILOT)
   {
      if ((gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_0].s_PilotToneIdx == 0)&&
          (gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_1].s_PilotToneIdx == 0)&&
          (gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_2].s_PilotToneIdx == 0))
      {
         // If we failed to find a new pilot tone, re-use the one from before O-Signature
         // Fix for XDSLRTFW-3639 [VRX518] No connect on long BT loop (2000m)
         gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_0].s_PilotToneIdx = s_PT0_Idx_bkp;
         gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_0].l_Metric = l_PT0_Metric_bkp;
      }
   }
   //XDSLRTFW-458: IOP_ALL_VDSL2_ALL_TACorrection (End)

   // XDSLRTFW-3280 (Start) Overwrite Pilot tone index at PT_ARRAY_IDX_0 with forced pilot tone
   if(gsa_Optn2_AlgControl[1] & OPTN_UseForcedPilot)
   {
      gt_PilotConfig.ta_PilotTones[PT_ARRAY_IDX_0].s_PilotToneIdx = TESTArray[TEST_RxTestPilotTone];
      gt_PilotConfig.te_UsedPTArrayIdx = PT_ARRAY_IDX_0;
      gt_PilotConfig.te_BestPTArrayIdx = PT_ARRAY_IDX_0;
   }
   // XDSLRTFW-3280 (End)

   gs_RxBkgdProcessFlag = TRAINING_DONE;
}
// XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement
