/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   Bitload_support.c
*
*   This file contains miscellaneous functions used in bitload.
*
*-------------------------------------------------------------------------
*/
#include <string.h>
#include "common.h"
#include "gdata.h"
#include "Bitload_support.h"
#include "dsp_op.h"
#include "decimalgain.h"
#include "mul.h"
#include "nomatp.h"
#include "vdsl_state.h"

/*^^^
*-----------------------------------------------------------------------------
*   Prototype:   void GetBatStats(uint8 *puca_Bat,
*                                void *ps_Num1BitTones,
*                                int16 *ps_TCM_Ovhd,
*                                int32 *pl_Sum_bi)
*
*   Description:
*      Computes statistics of a given Rx bit allocation table (BAT), including
*       number of loaded tones and Sum {bi}.
*
*   Input Parameters:
*      puca_Bat -- pointer to an Rx bit allocation table
*
*   Output Parameters:
*      ps_Num1BitTones -- returns number of 1-bit tones (bi == 1)
*      ps_TCM_Ovhd -- TCM overhead, if ft_TcmFlag set, else 0.
*      pl_Sum_bi -- sum of all bi
*
*   Returns:
*
*   Global variables:
*      gft_RxTcmFlag -- (I) TCM on/off flag
*      gus_ncloaded  -- (O) total number of loaded tones
*      gusa_ncloaded[] -- (O) number of loaded tones per band
*
*  Notes: Initially the second parameter was of type int16 but to suppress/remove a warning
 *         it has been changed to type void. And suitable changes are made in the function
 *         and wherevr it is being employed.
*-----------------------------------------------------------------------------
^^^*/
void GetBatStats(uint8 *puca_Bat, int16 *ps_Num1BitTones, int16 *ps_TCM_Ovhd, int32 *pl_Sum_bi)
{
   int16 i;
   int16 s_CurRxBand;
   uint8 *puc_BatEntry;
   int16 *ps_NumLoadedTones;
   *ps_Num1BitTones = 0;
   *ps_TCM_Ovhd = 0;
   *pl_Sum_bi = 0;

   gus_ncloaded = 0;


   //XDSLRTFW-3493(Start)
   //Number of tones allocated for each bit infornmation is Required for ReTx bit loading and ReTx framing
   //clear an array of number of tones for loaded bits array
   memset(gusa_n_tones_per_Bi, 0, sizeof(uint16)*MAX_BI_INDEX);
   //XDSLRTFW-3493(End)

   for (s_CurRxBand = 0; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
   {
      //gusa_ncloaded[s_CurRxBand] = 0;
      ps_NumLoadedTones = (int16 *)(void *)gusa_ncloaded+s_CurRxBand;
      *ps_NumLoadedTones = 0;

      puc_BatEntry = puca_Bat+gsa_BitloadLeftChannel[s_CurRxBand];

      for (i = gsa_BitloadLeftChannel[s_CurRxBand]; i <= gsa_BitloadRightChannel[s_CurRxBand]; i++)
      {
         if (*puc_BatEntry > 0)
         {
            *pl_Sum_bi += (int32)*puc_BatEntry;
            (*ps_NumLoadedTones)++;
            if (*puc_BatEntry == 1)
            {
               (*ps_Num1BitTones)++;
            }
            //XDSLRTFW-3493(Start)
            //Count non zero loaded tones on their respective Bi index
            if (*puc_BatEntry)
               gusa_n_tones_per_Bi[*puc_BatEntry]++;
            //XDSLRTFW-3493(End)
         }
         puc_BatEntry++;
      }
      gus_ncloaded += *ps_NumLoadedTones;
   }

   if (gft_RxTcmFlag)
   {
      *ps_TCM_Ovhd = (((gus_ncloaded - (*ps_Num1BitTones >> 1) + 1) >> 1) + 4);
   }
   else
   {
      *ps_TCM_Ovhd = 0;
   }
}


/*^^^
*-----------------------------------------------------------------------------
*   Prototype:   FlagT FindToneWithExtremeMargin   (int16 s_SearchType,
*                                                uint8 *puca_RxBat,
*                                                int16 *psa_RxFineGains,
*                                        int16 *psa_SNRBuffer,
*                                                RxToneFlags p_ActiveTones,
*                                        int16 s_MinBitsPerTone,
*                                                int16 s_MaxBitsPerTone,
*                                                int16 *ps_ch,
*                                                int16 *ps_Band,
*                                                int16 *ps_Margin)
*
*   Description:
*      Find the tone with the smallest or largest SNR margin.
*       There are two modes:
*
*      1) LARGEST_MARGIN_AFTER_ADDING_1_BIT - Search for the tone which, after adding a bit
*                          (or adding two in the case of a bi=0 tone for g.992.1/.2)
*                           has the largest resulting SNR margin.
*
*      2) SMALLEST_MARGIN - Search for the tone which has the smallest SNR margin.
*                            This tone is a good candidate for bi reduction.
*
*      3)LARGEST_ACTUAL_MARGIN - Search for the tone with largest SNR margin
*
*      A one-bit-per-tone array mask indicates which tones will be considered
*       in the comparison.  The tones considered can also be restricted by
*       specifying min and max bi values.
*
*      SNR margin is calculated using the input SNR and fine gains buffers.
*       Coding gain is also added to this margin.
*
*   Input Parameters:
*      s_SearchType     -- specifies whether the search should be for the tone
*                           with the smallest or largest margin. Recognized
*                           values are LARGEST_MARGIN_AFTER_ADDING_1_BIT or SMALLEST_MARGIN
*                           (default).
*      puca_RxBat       -- pointer to an Rx bit allocation table.
*      psa_RxFineGains  -- pointer to an Rx fine gain table
*      psa_SNRBuffer    -- measured SNR buffer, not adjusted for coding gain
*                           or anything else.
*      p_ActiveTones    -- One-bit-per-tone array indicating which tones in
*                           the BAT should be tested for margin.
*      s_MinBitsPerTone -- Only tones with current bi >= s_MinBitsPerTone will
*                           be tested for margin.
*      s_MaxBitsPerTone -- Only tones with current bi <= s_MaxBitsPerTone will
*                           be tested for margin.
*      *ps_Band -- if >= 0, indicate the band to be search, if <0, search for all the band
*
*   Output Parameters:
*      ps_ch             -- Tone index of the tone with the smallest/largest margin.
*      *ps_Band        -- Band index of the tone with the smallest/largest margin.
*      ps_Margin         -- Smallest/largest margin for this tone.
*
*   Return value:
*            SUCCEED:   success (no error)
*            FAIL:      only possible if the set of tones to consider
*                           (according to p_ActiveTones, s_MinBitsPerTone and
*                     s_MaxBitsPerTone) is empty.
*-----------------------------------------------------------------------------
^^^*/

FlagT FindToneWithExtremeMargin(int16 s_SearchType,
                                uint8 *puca_RxBat,
                                int16 *psa_RxFineGains,
                                int16 *psa_SNRBuffer,
                                RxToneFlags p_ActiveTones,
                                int16 s_MinBitsPerTone,
                                int16 s_MaxBitsPerTone,
                                int16 *ps_ch,
                                int16 *ps_Band,
                                int16 *ps_Margin)
{
   int16 s_ch, s_bi, s_bi_delta, s_Margin, s_ExtremeMargin;
   int16 s_ToneIndex = -1;
   int16 s_BandIndex = -1;
   int16 s_CurRxBand;
   FlagT ft_FoundBetterChoice;

   //Added a s_SearchType to get actual highest margin, this is required for
   //showtime noise margin equalization
   //XDSLRTFW-1784(START-END)
   //XDSLRTFW-3871:DS-SRA down shift are not executed anymore.DS SNRm remains under US SNRm threshold (start)
   if (s_SearchType == LARGEST_MARGIN_AFTER_ADDING_1_BIT)
   {
      s_ExtremeMargin = (int16) 0x8000;   // Largest negative number.
      // For this mode, bi_delta is the smallest allowed increase to bi.
      s_bi_delta = 1;
   }
   else if(s_SearchType == LARGEST_ACTUAL_MARGIN)
   {
      s_ExtremeMargin = (int16) 0x8000;   // Largest negative number.
      // To get the actual margin no need to add 1 bit
      s_bi_delta = 0;
   }
   else
   {  // SMALLEST_MARGIN
      s_ExtremeMargin = (int16) 0x7fff;   // Largest positive number.
      s_bi_delta = 0;
   }
   //XDSLRTFW-3871:DS-SRA down shift are not executed anymore.DS SNRm remains under US SNRm threshold (End)
   for (s_CurRxBand = 0; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
   {
      //If *ps_Band >= 0, only search the band designated by *ps_Band
      //if it < 0, search for all the band
      if((*ps_Band >= 0) && (s_CurRxBand != *ps_Band))
      {
         continue;
      }

      // !! Debug code to test maximum bits per tone per band, i.e. max constellation per band!!
      if (gsa_RxMaxConstSize[s_CurRxBand] > 0)
      {
         // Note: The function is called with 3 different inputs:
         //       - fixed value "1"
         //       - global variable "gs_RxMaxConstSize"
         //       - global variable minus one "(gs_RxMaxConstSize-1)"
         //       - global variable minus delta "(gs_RxMaxConstSize-(l_DeltaSumbi+gs_RxMinConstSize))"
         if (s_MaxBitsPerTone != 1)
         {
            if (s_MaxBitsPerTone == gs_RxMaxConstSize)
            {
               s_MaxBitsPerTone = gsa_RxMaxConstSize[s_CurRxBand];
            }
            else
            {
               int16 s_DeltaBits;

               s_DeltaBits = (gs_RxMaxConstSize - s_MaxBitsPerTone);
               s_MaxBitsPerTone = gsa_RxMaxConstSize[s_CurRxBand]-s_DeltaBits;
            }
         }
      }

      for (s_ch = gsa_BitloadLeftChannel[s_CurRxBand]; s_ch <= gsa_BitloadRightChannel[s_CurRxBand]; s_ch++)
      {

         s_bi = puca_RxBat[s_ch];

         if (IS_TONEFLAGSET(p_ActiveTones, s_ch) &&
               (s_bi >= s_MinBitsPerTone) &&
               (s_bi <= s_MaxBitsPerTone))
         {
            s_Margin = AddCodingGainToSnr(s_ch, psa_SNRBuffer);

            // If there's any existing fine gain in reserve it could
            // be used here to increase the margin of a newly loaded
            // tone, but that gets complicated.
            s_Margin += psa_RxFineGains[s_ch];

            // When subtracting bits and looking for the tone with
            // the smallest margin, we use bi_delta = 0.
            // When adding bits we want to know the margin after adding
            // the bit, so we set bi_delta = 1.
            if(s_bi == 0)
            {
               s_Margin -= gsa_ConstellationSNR[gs_RxMinConstSize];
            }
            else
            {
               s_Margin -= gsa_ConstellationSNR[s_bi+s_bi_delta];
            }

            //XDSLRTFW-3871:DS-SRA down shift are not executed anymore.DS SNRm remains under US SNRm threshold (start)
            //During Down shift SRA operation often SRA failed due to not fulfilling 1 bit Trellis criteria in this function.
            //This fail was observed due to not finding largest margin tone with FindToneWithExtremeMargin() function.
            //FindToneWithExtremeMargin() function with LARGEST_MARGIN_AFTER_ADDING_1_BIT argument searches for the tone which, after adding a bit
            //has the largest resulting SNR margin and can have higher margin than gs_MinTargetSnrMargin(!). This later condition is not required
            //in FindToneWithExtremeMargin() function because the purpose of this function is only to find tone with max (with or without adding a bit)
            //or min margin. This function was not designed to make the decision based on min bitswap/SRA target margin. This decision is done by its uper
            //layer calling function.

            ft_FoundBetterChoice = FALSE;
            if (s_SearchType == SMALLEST_MARGIN)
            {
               ft_FoundBetterChoice = (s_Margin < s_ExtremeMargin);
            }
            else // if ((s_SearchType == LARGEST_MARGIN_AFTER_ADDING_1_BIT)||(s_SearchType == LARGEST_ACTUAL_MARGIN))
            {
               ft_FoundBetterChoice = (s_Margin > s_ExtremeMargin);
            }
            //XDSLRTFW-3871:DS-SRA down shift are not executed anymore.DS SNRm remains under US SNRm threshold (End)

            if (ft_FoundBetterChoice)
            {
               s_ExtremeMargin = s_Margin;
               s_ToneIndex = s_ch;
               s_BandIndex = s_CurRxBand;
            }
         }
      }
   }

   *ps_ch = s_ToneIndex;
   *ps_Band = s_BandIndex;
   *ps_Margin = s_ExtremeMargin;

   if (s_ToneIndex == -1)
   {
      return (FAIL);
   }
   else
   {
      return (SUCCEED);
   }
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: int16 AddCodingGainToSnr(int16 s_ch, int16 *psa_SnrBuf)
*
*   This function returns combined
*
*   Input Arguments:
*      s_ch: physical tone index
*      psa_SnrBuf: pointer to input SNR buffer in Q8.8 format
*
*   Output Arguments:
*
*   Returns:
*      s_SnrOut: coding gain adjusted SNR in Q8.8 format for a given tone
*
*   Global Variables:
*      gsa_TotalCodingGain: total coding gain for INLV and FAST paths
*      gs_MaxToneForFast: maximum tone index for FAST path
*      gft_EnableMfdq: MFDQ enable/disable flag
*
*-------------------------------------------------------------------------------
*/

int16 AddCodingGainToSnr(int16 s_ch, int16 *psa_SnrBuf)
{
   int16 s_CodingGain, s_SnrOut;
   int32 l_Acc;

   // if the tones aren't in the MEDLEY set,
   // zero out SNR so that we do not bitload on these tones
   if (!IS_TONEFLAGSET(guca_RxSupportedToneSet, s_ch))
   {
      s_SnrOut = 0;
   }
   else
   {
      if (s_ch <= gs_MaxToneForFast)
      {
         s_CodingGain = gsa_TotalCodingGain[FAST];
      }
      else
      {
         s_CodingGain = gsa_TotalCodingGain[INLV];
      }


      // Include extra SNR margin for lower bandedge tones having correlated noise.
      // gs_high_tone_for_extra_snrmargin is the upper tone of these bandedge tones.
      if (s_ch <= gs_high_tone_for_extra_snrmargin)
      {
         s_CodingGain = s_CodingGain - gs_extra_snrmargin_for_low_tones;
      }

      l_Acc = psa_SnrBuf[s_ch] + s_CodingGain;

      if (gft_EnableMfdq)
      {
         l_Acc -= gt_MfdqConfig.s_MfdqMargin;
      }

      s_SnrOut = sature16(l_Acc);
   }


   return (s_SnrOut);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void UpdateFGSumSqAndNOMATP(int16 s_ch, int16 s_finegaindB_init,
*      int16 s_finegaindB_final, int32 *pl_SumFGainLinSq, FlagT ft_ToneUnloaded)
*
*   This function updates fine gain linear sum square and NOMATP by calling UpdateNOMATP().
*
*   Input Arguments:
*      s_ch -- index of the tone being processed
*      s_finegaindB_init -- initial fine gain before update (in dB Q8.8)
*      s_finegaindB_final -- final fine gain after update (in dB Q8.8)
*      pl_SumFGainLinSq -- pointer to the sum of linear gain square (before update, Q6.18)
*      ft_ToneUnloaded -- flag: 1: this tone is being unloaded, 2: this tone is loaded first time
*                                0: this tone is neither unloaded, not loaded first time
*
*   Output Arguments:
*      pl_SumFGainLinSq -- pointer to the sum of linear gain square (after update, Q6.18)
*
*   Returns:
*
*   Global Variables:
*      Check UpdateNOMATP() for the global variables used.
*-------------------------------------------------------------------------------
*/

void UpdateFGSumSqAndNOMATP(int16 s_ch, int16 s_finegaindB_init, int16 s_finegaindB_final, int32 *pl_SumFGainLinSq, FlagT ft_ToneUnloaded)
{
   int16 s_FGainLin;
   int32 l_SumFGainLinSq_add, l_SumFGainLinSq_sub;
   int32 l_GainSqChange;

   // preserve the 12 most significant bits as these are sent in Message
   // so the result is in Q3.9 format
   s_FGainLin = (DecimalGain(s_finegaindB_final)+(int16)(1<<3)) >> 4;

   //l_SumFGainLinSq_add is in Q6.18
   MULS16(l_SumFGainLinSq_add, s_FGainLin, s_FGainLin);

   // finegain-init
   s_FGainLin = (DecimalGain(s_finegaindB_init)+(int16)(1<<3)) >> 4;
   MULS16(l_SumFGainLinSq_sub , s_FGainLin , s_FGainLin);

   //Update the NOMATP
   //Do not do this during the showtime since UpdateNOMATP() use the interleaver data memory
   //to store the interpolated PSD values which is not available during the showtime
   if (gft_UpdateNomAtp_Flag == TRUE)
   {
      l_GainSqChange = l_SumFGainLinSq_add - l_SumFGainLinSq_sub;
      UpdateNOMATP(s_ch, l_GainSqChange);
   }

   //Treat the special case

   //the tone is being unloaded so it should not contribute to the sum of the gain square
   if(ft_ToneUnloaded == 1)
   {
      l_SumFGainLinSq_add = 0;
   }

   //the tone is being loaded first time so this gain has not included in the sum of gain square before
   else if(ft_ToneUnloaded == 2)
   {
      l_SumFGainLinSq_sub = 0;
   }

   *pl_SumFGainLinSq += (l_SumFGainLinSq_add - l_SumFGainLinSq_sub);
}
