/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: CalcAtpCeil.c
*
*   This file contains a function to convert log to linear.
*
*-------------------------------------------------------------------------------
*/

#include "common.h"
#include "gdata.h"
#include "GetTonePsd.h"
#include "decimalgain.h"
#include "ConvertToDB.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *   Description: Calculate and return the total transmit power (ATP) specified by s_MaxNomPsdIn and
 *             the break points in pt_PSDDescMaxIn.  The calculation is subject
 *             to the tx band allocation specified by sa_BandLeftChannel[], sa_BandRightChannel[],
 *             and to a PSD ceiling s_ceil passed as input.
 *
 *
 *  Prototype: int16 CalcAtp(int16 s_ceil, int16 s_MaxNomPsdIn, PSDDescriptorTable_t *pt_PSDDescMaxIn,
 *                      int16 s_NumOfBands, int16 *sa_BandLeftChannel, int16 *sa_BandRightChannel)
 *
 *  Input Arguments:
 *      s_ceil      -   PSD ceiling to apply during calculation of ATP {format -10*(dBm/Hz)}
 *      s_NumOfTones_RxOrTx   -  Number of RX or TX tones; Typical Values:4096 or 8192
 *
 *  Output Arguments:
 *
 *  Return: ATP in 8.8 dBm format
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 CalcAtp(int16 s_ceil, int16 s_MaxNomPsdIn, PSDDescriptorTable_t *pt_PSDDescMaxIn,
              int16 s_NumOfBands, int16 *sa_BandLeftChannel, int16 *sa_BandRightChannel, int16 s_NumOfTones_RxOrTx)
{

   int16 i, j, s_mant, s_exp, s_Psd, s_atp;
   int32 l_acc_lin_power, l_ceil_local, l_temp;
   int16 s_tone_count=0, s_shift = 0;

   // pt_PSDDescMaxIn has attenuation, relative to s_MaxNomPsdIn, in 10*dB format.
   // s_ceil has format -10*(dBm/Hz).
   // The psd ceiling used locally will be on the (gain) shaping relative to
   // s_MaxNomPsdIn, in 10XdB 8.8 format.
   l_ceil_local = -((int32)s_ceil - s_MaxNomPsdIn)<<8;

   l_acc_lin_power = 0;
   for (i=0; i<s_NumOfBands; i++)
   {
      for (j = sa_BandLeftChannel[i]; j <= sa_BandRightChannel[i]; j++)
      {
         s_tone_count++;

         // s_Psd is gain, relative to s_MaxNomPsdIn, in 10*dB format.
         s_Psd = GetTonePsd(j,pt_PSDDescMaxIn, s_NumOfTones_RxOrTx);//   XDSLRTFW-3838

         // 10XdB 8.8 format.
         l_temp = (int32) s_Psd<<8;
         // Apply ceiling
         if (l_temp>l_ceil_local)
         {
            l_temp = (int32) l_ceil_local;
         }

         // Convert to 8.8 dB format for input to db_to_linear.
         // Also, db_to_linear calculates 10^(x/20) rather than 10^(x/10),
         // so include an additional <<1.
         s_Psd = (int16) ((l_temp<<1)/10);

         db_to_linear(s_Psd, &s_mant, &s_exp);

         //s_mant is in 1.15 format.  Convert to 1.31 format in l_temp
         l_temp = ((int32) s_mant) << 16;
         //s_Psd should always be <0, so a normalizing right shift of TX_LOG2_MAX_FFT_LENGTH
         //avoids overflow in the acc.
         //XDSLRTFW-1460
         //In this particular case of PSD difference lower than
         //-57dBm/Hz, dB_to_Linear() returns a exp of -20.
         //As a result we end up right shifting a int32 33 times.
         //Normally the right shift does a modulo 32 and we end up
         //having a very big value for ((s_mant << 16) >> 1)
         s_shift = (gs_TxLog2IfftLength-s_exp);
         if(s_shift > 31)
         {
            s_shift = 31;
         }
         l_temp >>= s_shift;
         //XDSLRTFW-1460

         l_acc_lin_power += l_temp;
      }
   }

   // Convert total power to dB, which is in 8.8 dB format except for the normalizations applied above.
   s_atp = ConvertToDB(l_acc_lin_power);

   // Remove the normalization that was applied to the linear power.
   // The composite normalization was [l_acc_lin_power << (15+16-TX_LOG2_MAX_FFT_LENGTH)].
   // 771 is 10*log10(2)=3.0103 dB, in 8.8 dB format.
   s_atp -= (int16)771*(31-(int16)gs_TxLog2IfftLength);

   // pt_PSDDescMaxIn was relative to s_MaxNomPsdIn (format -10*(dBm/Hz))
   // so account for that and for the tone spacing, to convert to dBm, 8.8 format.
   // s_atp += -(gt_TxPsdControl.s_MaxNomPsdIn<<8)/10;
   s_atp += -(s_MaxNomPsdIn<<8)/10;

   if (gs_frame_rate_is_8khz == 1)
   {
      s_atp += VAL_binw8_in_dBHZ_8p8;
   }
   else
   {
      s_atp += VAL_binw4_in_dBHZ_8p8;
   }

   return(s_atp);

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CalcAtpCeil
 *
 *  Description:  Calculates the PSD ceiling that corresponds to the total power
 *              (ATP) constraint s_MaxAtp applied to the TX PSD
 *              described by s_MaxNomPsdIn, the break points pt_PSDDescMaxIn,
 *              and the band allocation specified by
 *              sa_BandLeftChannel[] and sa_BandRightChannel[].  The result is
 *              returned in -10*(dBm/Hz) format.
 *
 *  Prototype: int16 CalcAtpCeil(int16 s_MaxAtp, int16 s_MaxNomPsdIn, PSDDescriptorTable_t *pt_PSDDescMaxIn,
 *                         int16 s_NumOfBands, int16 *sa_BandLeftChannel, int16 *sa_BandRightChannel)
 *
 *  Input Arguments:
 *      s_NumOfTones_RxOrTx   -  Number of RX or TX tones; Typical Values:4096 or 8192
 *
 *  Output Arguments:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 CalcAtpCeil(int16 s_MaxAtp, int16 s_MaxNomPsdIn, void *pt_PSDDescMaxIn1,
                  int16 s_NumOfBands, int16 *sa_BandLeftChannel, int16 *sa_BandRightChannel, int16 s_NumOfTones_RxOrTx)
{
   int16 s_ceil, s_atp, s_count=0;
   int32 l_excess, l_MaxAtp_8pt8Format;
   PSDDescriptorTable_t *pt_PSDDescMaxIn;
   pt_PSDDescMaxIn = (PSDDescriptorTable_t *)pt_PSDDescMaxIn1;
   l_MaxAtp_8pt8Format = (((int32)s_MaxAtp) << 8)/10;

   // First try a ceiling of gt_TxPsdControl.s_MaxNomPsdIn, which is in -10*(dBm/Hz) format.
   s_ceil = s_MaxNomPsdIn;
   // s_atp, s_excess, and gt_TxPsdControl.s_MaxAtp in 8.8 dBm format
   s_atp = CalcAtp(s_ceil,s_MaxNomPsdIn,pt_PSDDescMaxIn,s_NumOfBands,sa_BandLeftChannel,sa_BandRightChannel,s_NumOfTones_RxOrTx);//XDSLRTFW-3838
   l_excess = s_atp - l_MaxAtp_8pt8Format;

   // Keep reducing the ceiling by the excess ATP, until
   // an ATP no greater than MaxATP is obtained.
   while(l_excess>1)
   {
      s_count++;
      // convert s_excess from 8.8 dBm format so that it can be
      // "subtracted" from s_ceil which has format -10*(dBm/Hz).
      l_excess = (10*l_excess)>>8;
      // s_ceil is "reduced" an extra 0.1 dB to speed convergence.
      s_ceil = s_ceil + l_excess + 1;
      // Get the ATP with this new ceiling.
      s_atp = CalcAtp(s_ceil,s_MaxNomPsdIn,pt_PSDDescMaxIn,s_NumOfBands,sa_BandLeftChannel,sa_BandRightChannel,s_NumOfTones_RxOrTx);//XDSLRTFW-3838
      l_excess = s_atp - l_MaxAtp_8pt8Format;
   }

   return(s_ceil);

}
