/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: CalcMedleySnrMargin.c
*
*   This file contains functions to calculate Medley SNR margin.
*
*-------------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// CalcMedleySnrMargin.c
//
// History
//
// 18/02/2013 Hanyu/Vinjam: Added code to update VDSL2 DS Attainable Net Data Rate (ATTNDRds) in Showtime.
//            grep for: XDSLRTFW-539 FEATURE_ALL_VDSL2_ATTNDR_Update
//
// ************************************************************************************************************

#include "common.h"
#include "gdata.h"
#include "QuickAverage.h"
#include "dsp_op.h"
#include "vdsl_state.h"
#include "ConvertToDB.h"

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void CalcMedleySnrMargin(FlagT ft_ApplyFG, int16 *ps_AvgMargin,
*      int16 *ps_MinMargin, int16 *ps_MinMarginTone)
*
*   This function computes the average and minimum SNR margin for the given SNRs
*   per tone and constellation size per tone.
*
*   Input Arguments:
*      ft_ApplyFG: 1 to apply FG in margin calculation (Medley)
*         assume FG in HW buffer is in dB format, 0 otherwise (Showtime)
*
*   Output Arguments:
*      ps_AvgMargin: pointer to average margin
*      ps_MinMargin: pointer to minimum margin
*      ps_MinMarginTone: pointer to tone index with minimum margin
*
*   Returns:
*
*   Global Variables:
*      gpsa_MeasuredSnrBuf -- (I) pointer to the input SNR buffer
*      ghpuca_RxBat_Inactive -- (I) pointer to the inactive RX BAT table
*      ghpsa_RxFineGains_Inactive -- (I) pointer to the inactive RX BAT table
*      gs_NumOfRxBands -- (I) the number of RX frequency bands
*      gsa_RxBandLeftChannel[] -- (I) an array of the RX left band edge frequency tone
*      gsa_RxBandRightChannel[] -- (I) an array of the RX right band edge frequency tone
*
*-------------------------------------------------------------------------------
*/

//#define SNRMARGIN_DEBUG

#ifdef SNRMARGIN_DEBUG
extern int16 gs_AvFineGain_Pre;         //Average fine gain prior to the margin reduction in dB (Q8.8)
extern int16 gs_AvFineGain_after;      //Average fine gain after the margin reduction in dB (Q8.8)
extern int16 gs_AvSnrMargin_Pre;      //Average SNR prior to the margin reduction in dB (Q8.8)
extern int16 gs_AvSnrMargin_after;      //Average SNR after the margin reduction in dB (Q8.8)
extern int16 gs_MaxSnrMargin;
extern int16 gs_MinSnrMargin;
extern int16 gs_LimitByMinGain;
extern int16 gs_LimitByMinMargin;
#endif //#ifdef SNRMARGIN_DEBUG

void CalcMedleySnrMargin(FlagT ft_ApplyFG, int16 *ps_AvgMargin, int16 *ps_MinMargin, int16 *ps_MinMarginTone)
{
   int16 j, s_ch, s_qc, s_RxChannelsAllocated, s_RxChannelsAllocatedPerRxBand;
   int16 s_CodingGain, s_margin, s_MinMargin, s_MinMarginTone=0;
   int32 l_TotalMargin, l_TotalMarginPerRxBand;
   int32 l_SumFineGaindB;
   int16 s_MaxSnrMargin=0, s_MinSnrMargin=0, s_ExcessMargin=0, s_MarginToCut;
   int16 s_MarginReductionFlag=0, s_MarginIterationCnt=0, s_temp;

   if(gs_ApplyMarginReduction == 1)
   {

      s_MaxSnrMargin = (gt_SnrMgnConfig.s_MAXSNRMds<<8)/10;
      s_MinSnrMargin = (gt_SnrMgnConfig.s_MINSNRMds<<8)/10;
      s_MarginReductionFlag = 0;
      s_ExcessMargin = 0;
      s_MarginIterationCnt = 0;

#ifdef SNRMARGIN_DEBUG
      //Log for debug
      gs_MaxSnrMargin = s_MaxSnrMargin;
      gs_MinSnrMargin = s_MinSnrMargin;
      gs_LimitByMinGain = 0;
      gs_LimitByMinMargin = 0;
#endif //SNRMARGIN_DEBUG
   }


_start:
   l_SumFineGaindB = 0;

   l_TotalMargin = 0;
   s_RxChannelsAllocated = 0;
   s_MinMargin = (int16)0x7FFF;

   // XDSLRTFW-539 FEATURE_ALL_VDSL2_ATTNDR_Update (START)
   gl_SumMedleySNR = 0;
   // XDSLRTFW-539 FEATURE_ALL_VDSL2_ATTNDR_Update (END)

   for(j=0; j<gs_NumOfRxBands; j++)
   {
      l_TotalMarginPerRxBand = 0;
      s_RxChannelsAllocatedPerRxBand = 0;

      for(s_ch=gsa_RxBandLeftChannel[j]; s_ch<=gsa_RxBandRightChannel[j]; s_ch++)
      {
         s_qc= ghpuca_RxBat_Inactive[s_ch];

         if(s_qc == 0)
         {
            continue;
         }

         s_CodingGain = gsa_TotalCodingGain[INLV];

         if (s_ch <= gs_MaxToneForFast)
         {
            s_CodingGain = gsa_TotalCodingGain[FAST];
         }

         // Include extra SNR margin for lower bandedge tones having correlated noise.
         // gs_high_tone_for_extra_snrmargin is the upper tone of these bandedge tones.
         if (s_ch <= gs_high_tone_for_extra_snrmargin)
         {
            s_CodingGain = s_CodingGain - gs_extra_snrmargin_for_low_tones;
         }

         // add coding gain
         s_margin = gpsa_MeasuredSnrBuf[s_ch] + s_CodingGain - gsa_ConstellationSNR[s_qc];

         // apply fine gain
         if (ft_ApplyFG)
         {
            s_margin += ghpsa_RxFineGains_Inactive[s_ch];
         }

         //=======================================================================
         //Reduce the fine gain if the maximum margin requirement is violated
         //=======================================================================
         if((gs_ApplyMarginReduction == 1) && (s_ExcessMargin > 0))
         {
            //Compute the maximum margin that can be cut without violating the minimum margin constraint
            s_MarginToCut = s_margin - s_MinSnrMargin;

            //Check if there is extra margin to cut
            if((ghpsa_RxFineGains_Inactive[s_ch] > gs_min_fine_gain_MarginRed) && (s_MarginToCut > 0))
            {

               //Maximum amount of fine gain that can be cut
               s_temp = ghpsa_RxFineGains_Inactive[s_ch] - gs_min_fine_gain_MarginRed;
               if(s_MarginToCut > s_temp)
               {
                  s_MarginToCut = s_temp;

#ifdef SNRMARGIN_DEBUG
                  //Log for debug
                  gs_LimitByMinGain++;
#endif
               }

               if(s_MarginToCut > s_ExcessMargin)
               {
                  s_MarginToCut = s_ExcessMargin;
               }
#ifdef SNRMARGIN_DEBUG
               else
               {
                  gs_LimitByMinMargin++;
               }
#endif

               //Reduce fine gain
               ghpsa_RxFineGains_Inactive[s_ch] -= s_MarginToCut;

               //Reduce margin
               s_margin -= s_MarginToCut;

               //Set the flag to indicate that the margin reduction is applied
               s_MarginReductionFlag = 1;
            }
         }


         if (s_margin < s_MinMargin)
         {
            s_MinMargin = s_margin;
            s_MinMarginTone = s_ch;
         }

         // compute the total margin per band
         l_TotalMarginPerRxBand += s_margin;
         s_RxChannelsAllocatedPerRxBand++;

         // compute the total margin
         l_TotalMargin += s_margin;
         s_RxChannelsAllocated++;

         // XDSLRTFW-539 FEATURE_ALL_VDSL2_ATTNDR_Update (START)
         gl_SumMedleySNR += (gpsa_MeasuredSnrBuf[s_ch]>>8);  // dB in Q32.0 format to avoid overflow
         // XDSLRTFW-539 FEATURE_ALL_VDSL2_ATTNDR_Update (END)

         // compute the total fine gain
         l_SumFineGaindB += ghpsa_RxFineGains_Inactive[s_ch];

#ifdef SNRMARGIN_DEBUG
         // if SaveMarginPerToneflag is set, then overwrite SNR with margin information
         if (gft_SaveMarginPerTone)
         {
            gpsa_MeasuredSnrBuf[s_ch] = s_margin;
         }
#endif
      } // end for(s_ch=gsa_RxBandLeftChannel[j]; s_ch<=gsa_RxBandRightChannel[j]; s_ch++) loop

      if (s_RxChannelsAllocatedPerRxBand == 0)
      {
         gt_AttenSnrMPerRxBand.t_AttenSnrM[j].s_SNRM = OUT_OF_RANGE_SNRM;
      }
      else
      {
         // compute average margin per band
         gt_AttenSnrMPerRxBand.t_AttenSnrM[j].s_SNRM = QuickAverage(l_TotalMarginPerRxBand, s_RxChannelsAllocatedPerRxBand);
         // convert to 0.1 dB format
         gt_AttenSnrMPerRxBand.t_AttenSnrM[j].s_SNRM = (int16) ((int32) (gt_AttenSnrMPerRxBand.t_AttenSnrM[j].s_SNRM * 10) >> 8);
      }

   } // end for(j=0; j<gs_NumOfRxBands; j++) loop
   // Populate the number of RxBands.
   gt_AttenSnrMPerRxBand.us_NumberOfBands = (uint16)j;

   // compute the average margin
   if (s_RxChannelsAllocated == 0)
   {
      *ps_AvgMargin = OUT_OF_RANGE_SNRM;
   }
   else
   {
      *ps_AvgMargin = QuickAverage(l_TotalMargin, s_RxChannelsAllocated);
   }

   // compute the average fine gain
   gs_RxAvFineGain = QuickAverage(l_SumFineGaindB, s_RxChannelsAllocated);

   //=======================================================================================
   //Set the variables which are used for the maximum margin reduction
   //(Limit the number iteration to 3)
   //=======================================================================================
   if(gs_ApplyMarginReduction == 1)
   {
      //Save the margin before reduction
      if(s_MarginIterationCnt == 0)
      {
         gs_ExcessMarRedDB = *ps_AvgMargin;
         gs_ExcessMarFGReduction = gs_RxAvFineGain;

#ifdef SNRMARGIN_DEBUG
         //Log information for debug
         gs_AvFineGain_Pre = gs_RxAvFineGain;
         gs_AvSnrMargin_Pre = *ps_AvgMargin;
#endif //#ifdef SNRMARGIN_DEBUG

      }

      //Count the number of iterations the margin reduction is applied
      s_MarginIterationCnt++;
      if((s_MarginIterationCnt <= gs_MaxMarginIterations)         //still not reach iteration limit
            && (*ps_AvgMargin > (s_MaxSnrMargin + (int16)(256*0.5)))   //margin is still too high (0.1 dB leeway)
            && ((s_MarginIterationCnt == 1) || (s_MarginReductionFlag == 1))) //if further margin reduction is achievable
      {
         s_ExcessMargin = *ps_AvgMargin - s_MaxSnrMargin;
         s_MarginReductionFlag = 0;

         goto _start;
      }
      //else compute the final results

      //Compute the amount of margin reduction in Q8.8
      gs_ExcessMarRedDB -= *ps_AvgMargin;

      //Compute the amount of finegain reduction in Q8.8
      gs_ExcessMarFGReduction -= gs_RxAvFineGain;

#ifdef SNRMARGIN_DEBUG
      //log information for debug
      gs_AvFineGain_after = gs_RxAvFineGain;
      gs_AvSnrMargin_after = *ps_AvgMargin;
#endif //#ifdef SNRMARGIN_DEBUG

      //==================================================================
      //Apply the margin reduction to those tones with bc = 0
      //=================================================================
      s_ExcessMargin = gs_ExcessMarRedDB;
      if(s_ExcessMargin > 0)
      {
         for(j=0; j<gs_NumOfRxBands; j++)
         {
            for(s_ch=gsa_RxBandLeftChannel[j]; s_ch<=gsa_RxBandRightChannel[j]; s_ch++)
            {
               if(ghpuca_RxBat_Inactive[s_ch] > 0)
               {
                  continue;
               }

               //Compute margin
               s_margin = gpsa_MeasuredSnrBuf[s_ch] - gsa_ConstellationSNR[2];

               // apply fine gain
               if (ft_ApplyFG)
               {
                  s_margin += ghpsa_RxFineGains_Inactive[s_ch];
               }

               //Compute the maximum margin that can be cut without violating the minimum margin constraint
               s_MarginToCut = s_margin - s_MinSnrMargin;

               //Check if there is extra margin to cut
               if((ghpsa_RxFineGains_Inactive[s_ch] > gs_min_fine_gain_MarginRed) && (s_MarginToCut > 0))
               {

                  //Maximum amount of fine gain that can be cut
                  s_temp = ghpsa_RxFineGains_Inactive[s_ch] - gs_min_fine_gain_MarginRed;
                  if(s_MarginToCut > s_temp)
                  {
                     s_MarginToCut = s_temp;
                  }

                  if(s_MarginToCut > s_ExcessMargin)
                  {
                     s_MarginToCut = s_ExcessMargin;
                  }

                  //Reduce fine gain
                  ghpsa_RxFineGains_Inactive[s_ch] -= s_MarginToCut;
               } //if((ghpsa_RxFineGains_Inactive[s_ch] > gs_min_fine_gain_MarginRed) &&
            } //for(s_ch=gsa_RxBandLeftChannel[j]; s_ch<=gsa_RxBandRightChannel[j]; s_ch++)
         } //for(j=0; j<gs_NumOfRxBands; j++)
      } //if(s_ExcessMargin > 0)
   } //if(gs_ApplyMarginReduction == 1)

   // change q8.8 format to 0.1 dB increments
   *ps_AvgMargin = (int16) (((int32)(*ps_AvgMargin) * 10) >> 8);

   // set the minimum margin
   *ps_MinMargin = s_MinMargin;
   *ps_MinMarginTone = s_MinMarginTone;

#ifdef SNRMARGIN_DEBUG
   // if SaveMarginPerTone flag is set, then disable the SNR calculation
   // to provide time for debug read of Margin per Tone
   if (gft_SaveMarginPerTone)
   {
      gft_EnableSnrUpdate = 0;
   }
#endif

   //Reduce fine gains of unloaded tones
   {
      if(gs_UnloadedToneFGreduction_dB != NEG_INFINITY_DB)
      {
         int16 s_FineGainOut, s_FineGain;

         for(j=0; j<gs_NumOfRxBands; j++)
         {
            for(s_ch = gsa_RxBandLeftChannel[j]; s_ch <= gsa_RxBandRightChannel[j]; s_ch++)
            {
               // XDSLRTFW-3280 - Start/End - PLL improvement / pilot tone selection improvement
               // Check for pilot tone
               if (!((s_ch == gt_PilotConfig.ta_PilotTones[0].s_PilotToneIdx) || (s_ch ==gt_PilotConfig.ta_PilotTones[2].s_PilotToneIdx) || (s_ch == gt_PilotConfig.ta_PilotTones[1].s_PilotToneIdx)))
               {
                  if(ghpuca_RxBat_Inactive[s_ch] == 0)
                  {
                     s_FineGain = ghpsa_RxFineGains_Inactive[s_ch];

                     // When s_FineGain is already NEG_INFINITY_DB nothing must be done!
                     if((gs_UnloadedToneFGreduction_dB != NEG_INFINITY_DB) && (s_FineGain != NEG_INFINITY_DB))
                     {
                        s_FineGainOut = s_FineGain - gs_UnloadedToneFGreduction_dB;
                        // When s_FineGainOut is below DEFAULT_MIN_FINEGAIN_DB then disable tone!
                        if(s_FineGainOut < gs_min_fine_gain_MarginRed)
                        {
                           s_FineGainOut = NEG_INFINITY_DB;
                        }
                        ghpsa_RxFineGains_Inactive[s_ch] = s_FineGainOut;
                     }
                  }
               }
            }
         } //for(j=0; j<gs_NumOfRxBands; j++)
      } //if(gs_UnloadedToneFGreduction_dB != 0)
   }
}

