/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   changebat.c
*
*   Routines for modifying a bit allocation table to change the sum of
*   bi or the sum of Lp.
*
*-------------------------------------------------------------------------
*/
// ***********************************************************************************************************
// ChangeBat.c
//
// History
// 09/05/2018 Abu Rahman: XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code
// Issue: After several 10th of thousands of DS bit swapping RMS FG per band tends to reach its maximum limit (2.5dB).
// Due to this DS bit swap stops. If this situation happens CPE does not trigger any DS bit swap even with the presence
// of RFI noise on DS band which could lead to link drop.
//
// Root Cause: Multiple reasons
// i.  After several thousand DS DS, RMS FG per band was not accurate any more. RMS FG  per band was calculated only
//     one time from inactive FG table and then always modified in FW where errors were accumulated and never corrected.
// ii. At every Bit swap initiation calculation RMS FG was modified. If BS decision algorithm decided to not trigger
//     any DS bitswap request then RMS FG was not reinitialized with correct value.
// iii.Due to above points RMS FG thresholds for FG allocation did not work at all and algorithm triggered DS bit swap
//     even actual RMS FG per bands were increasing.
//
// Solution:
// i.  Calculated RMS FG per band from inactive gain table every time before starting
//     BS algorithm in RxBitSwapDecision_VDSL2() function
// ii. Modified BS algorithm in a way that actual RMS FG (DS) always tries to be below
//     the initial RMS FG. This improvement makes sure that per tone DS FGs are evenly
//     distributed and around initial RMS FG.
// iii.If RMS FG go above the threshold then new algorithm tries to reduce FG across the tones in that band to mitigate the
//     RMS FG criteria.
//   Grep for XDSLRTFW-3807
//
// 19/10/2018 Abu Rahman
//            XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case
//
//            DS Bitswap/SRA  algorithm in BgRxBitSwapDecision_VDSL2() assumed that
//            Max FG = +2.5 dB
//            Min FG = -2.5 dB and
//            Monitoring (0 bit loaded) tone FG = 0 dB
//
//            This algorithm does not consider any FG reduction (eg. gs_ExcessMarFGReduction) which is done during initial bit loading.
//            As a consequence, in cases of access margin (where SNR margin is higher than that of Max margin) FG reduction is done in
//            training to reduce the overall received DS ActATP,  DS power is increased over time in conjunction with DS bitswap/SRA.
//
//            As a solution, initial FG reduction (gs_ExcessMarFGReduction) in training for Bitloading is consider in Max FG, Min FG as
//            well as monitoring tone FG calculation during Bitswap/SRA computation.
//            Grep for XDSLRTFW-4005
//
// 20/12/2018 Abu Rahman
//            XDSLRTFW-4051 Fast downstream up shift SRA
//            This code supports Fast downstream up shift SRA
//            Maximum bits loaded/tone = 5
//            This feature is enabled by DTAG bit (CMV DSL 21 0 1) only against vectoring and BRCM based DSLAM.
//
//            Some other controls are done by CNFG 93 cmv
//            CNFG 93 0:
//             bit 0:
//                - value 0x8000: disable feature "Fast downstream up shift SRA"
//                - value 0x0000: enable  feature "Fast downstream up shift SRA" for BRCM CO and any  vector engine
//                - value 0x0001: enable  feature "Fast downstream up shift SRA" for BRCM CO and BRCM vector engine

//            CNFG 93 1:
//             - value 0xFFFF: do not disable feature "Fast downstream up shift SRA" in showtime
//             - value      X: disable feature "Fast downstream up shift SRA" after X sec of Showtime
//                             (supported range: 0-32767 sec, debug range 0-65534 sec)
//            CNFG 93 2:
//             - value 0xFFFF: Never disable DS-SRAs in Showtime
//             - value      Y: Disable DS SRAs if CPE receive 3 consecutive SRA rejections with rejection code SRA "not enabled"
//                             (0x3) after Y sec of Showtime (supported range: 0-32767 sec, debug range 0-65534 sec)
//            Example:
//            DSL 21[0] : 1 -DTAG bit enabled
//            CNFG 93[0]: 0 - meaning feature is enabled for all kind of VCE
//            CNFG 93[1]: 100 - Fast DS up shift SRA is enabled for 100 sec in Showtime
//            CNFG 93[2]: 200 - After 200 sec of Showtime if three consecutive SRA rejection come with SRA not enabled
//                              exception code (0x3) then DS SRA request will be disabled
//            Default CMV configuration: if DTAG bit is enabled
//            CNFG 93[0] = 0
//            CNFG 93[1] = 255
//            CNFG 93[2] = 255
//            Grep for XDSLRTFW-4005

// ***********************************************************************************************************
#include <string.h>
#include "common.h"
#include "gdata.h"
#include "Bitload_support.h"
#include "Bitload.h"
#include "Changebat.h"
#include "ChooseFineGains.h"
#include "nomatp.h"
#include "vdsl_state.h"
/*^^^
*----------------------------------------------------------------------------
*   Prototype:   FlagT ChangeLpKeepMaxMargin(uint8 *puca_RxBat,
*                                           int16 *psa_RxFineGains,
*                                           int32 l_DesiredDeltaSumLp,
*                                           RxToneFlags p_ActiveTones,
*                                           RxToneFlags p_ModifiedTones,
*                                           int16 *ps_ActualDeltaSumLp,
*                                           uint16 us_BiGiModificationControlFlags)
*
*   Description:
*      Changes the bit allocation table (BAT) to produce the specified change
*       (delta) in Sum{Lp}.
*       Recall that the Sum{bi} = Sum{Lp} + TCM_Overhead.  This routine
*       figures out the necessary change in Sum{bi} to get the desired change
*       in Lp, and calls ChangeBatKeepMaxMargin() to make the actual changes
*       to the BAT.
*
*      The delta Lp may be positive or negative.  The specified min tone
*       margin argument (by gs_MinTargetSnrMargin) specifies the minimum margin
*       that must be maintained on all tones.  For positive deltaLp the margin check
*       is performed before adding each bit to the BAT.  For negative deltaLp the margin
*       is checked after all bits have been subtracted.
*
*      The function sets the value of the Actual delta Lp argument.
*      This will only differ from the desired delta Lp when it is not
*       possible to achieve the desired delta Lp.  One ToneFlag array specifies
*       which tones in the BAT are allowed to change.  A second ToneFlag array
*       is written to by the function to indicate which tones in the BAT were
*       actually changed.  The fine gain array input parameter is used
*      to calculate SNR margins, but it is not modified by this routine.
*
*   Input Parameters:
*      puca_RxBat -- pointer to an Rx bit allocation table
*      psa_RxFineGains -- pointer to an Rx fine gain table
*      l_DesiredDeltaSumLp -- desired change in Lp, either positive or negative
*      p_ActiveTones -- One-bit-per-tone array indicating which tones in the BAT may be changed by this routine.  Only tones
*            whose bit is set will have their bi changed, and margin checks will only be done on these tones.
*      us_BiGiModificationControlFlags : only bit0, bit1, bit2 and bit3 bits are used
*        - (bit 0) ft_StopAtMarginThreshold -- flag if set to TRUE, stop changing bits if margin threshold is met
*        - (bit 1) ft_forceEven1Bit -- flag if set to FALSE, do not perform forcing even bit operation
*        - (bit 2) ft_AddfineGainplusdeltaLp -- flag if set to TRUE, added the fine gain in the case delta_lp > 0
*        - (bit 3) ft_Fast_UpShift_SRA_Mode -- flag if set to TRUE, enable fast downstream up shift SRA
*
*   Output Parameters:
*      p_ModifiedTones -- One-bit-per-tone array used to indicate which tones were actually changed by this routine.  In some
*            cases a bit may be set for a tone whose final bi value has not changed, because intermediate steps may have
*            changed the bit and then changed it back again.
*      pl_ActualDeltaSumLp -- the actual change in Lp.  This will only differ from the desired delta Lp when it is not possible to achieve the desired
*      delta Lp.  The two reasons this could happen are: 1) min margin could not be maintained, 2) no room to add bits, or no bits left
*      to subtract.
*
*   Return value:
*       SUCCEED/FAIL
*
*   Inernal error code:
*       0:   success (no error)
*       1:   BAT change violates SNR margin.
*           If deltaLp > 0 : Couldn't increase Lp by desired amount without
*                            violating margin
*           If deltaLp < 0 : Decreased Lp by desired amount, but still didn't
*                            meet margin (used for validating L2 requests).
*
*       2:   impossible to increase/decrease Lp as desired
*
*           If deltaLp > 0 : Not possible to increase Lp as desired because
*                            all tones at maximum bitloading.
*           If deltaLp < 0 : Not possible to decrease Lp as desired because
*                            not enough bits left in BAT!
*
*   Global variables:
*
*      gpsa_MeasuredSnrBuf - (I) measured SNR buffer, not adjusted for coding gain.
*      gs_MinTargetSnrMargin - (I) the minimum margin (in 8.8 dB format)
*                               that must be met by the BAT table after any modifications
*----------------------------------------------------------------------------
^^^*/
FlagT ChangeLpKeepMaxMargin(uint8 *puca_RxBat,
                            int16 *psa_RxFineGains,
                            int32 l_DesiredDeltaSumLp,
                            RxToneFlags p_ActiveTones,
                            RxToneFlags p_ModifiedTones,
                            int32 *pl_ActualDeltaSumLp,
                            uint16 us_BiGiModificationControlFlags)
{
   int32 l_DeltaSumLp;
   int16 s_Num1BitTones, s_RxExtraBits;
   int16 s_Dummy2ReturnValue;
   int16 s_ch, s_CurRxBand;
   int32 l_InitialSumBi, l_FinalSumBi;
   RxToneFlags p_ActiveTones_ToConsider;
   int16 s_Initial_TCM_Ovhd, s_TCM_Ovhd_before, s_TCM_Ovhd_after;
   FlagT ft_Result;
   int16 *psa_FineGain;
   int16 *psa_SNRBuffer, s_MinToneMargin;

   FlagT ft_StopAtMarginThreshold = FALSE;
   FlagT ft_forceEven1Bit = FALSE;
   FlagT ft_AddfineGainplusdeltaLp = FALSE;
   FlagT ft_Fast_UpShift_SRA_Mode = FALSE;

   // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (Start)
   // #define FT_STOP_AT_MARGIN_THRESHOLD                (0x0001)    // bit 0
   // #define FT_FORCE_EVEN_1_BIT                        (0x0002)    // bit 1
   // #define FT_ADD_FINE_GAIN_PLUS_DELTA_LP             (0x0004)    // bit 2
   // #define FT_FAST_UPSHIFT_SRA_MODE                   (0x0008)    // bit 3

   // Decode control flags from the us_BiGiModificationControlFlags bit fields
   if (us_BiGiModificationControlFlags & FT_STOP_AT_MARGIN_THRESHOLD)      // bit 0
   {
      ft_StopAtMarginThreshold = TRUE;
   }
   if (us_BiGiModificationControlFlags & FT_FORCE_EVEN_1_BIT)              // bit 1
   {
      ft_forceEven1Bit = TRUE;
   }
   if (us_BiGiModificationControlFlags & FT_ADD_FINE_GAIN_PLUS_DELTA_LP)   // bit 2
   {
      ft_AddfineGainplusdeltaLp = TRUE;
   }
   if (us_BiGiModificationControlFlags & FT_FAST_UPSHIFT_SRA_MODE)         // bit 3
   {
      ft_Fast_UpShift_SRA_Mode = TRUE;
   }
   // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (End)

   // Clear the past status
   gus_BitloadErrorCode = BITLOAD_NO_ERROR;

   psa_SNRBuffer = gpsa_MeasuredSnrBuf;
   s_MinToneMargin = gs_MinTargetSnrMargin;
   l_DeltaSumLp = l_DesiredDeltaSumLp;

   GetBatStats(puca_RxBat, &s_Dummy2ReturnValue, &s_Initial_TCM_Ovhd, &l_InitialSumBi);

   s_TCM_Ovhd_before = s_Initial_TCM_Ovhd;
   s_TCM_Ovhd_after = s_TCM_Ovhd_before;
   l_FinalSumBi = l_InitialSumBi;

   // Copy the active tone set to another array, where we clear the tones for
   // which fine gain is set to be negative infinity; as we don't want to
   // change Medley set

   //for (s_ch = 0; s_ch < (gs_RxNumTones>>3); s_ch++)
   //   p_ActiveTones_ToConsider[s_ch] = p_ActiveTones[s_ch];
   memcpy(p_ActiveTones_ToConsider, p_ActiveTones, (gs_RxNumTones>>3));

   // Clear the tones from active set for which fine gains may be put to
   // negative infinity to avoid these being detected as tones while doing
   // any bat change.  This explicit step avoids any problems with negative
   // overflow during margin calculation (neg SNR and neg fine gain).
   for (s_CurRxBand = 0; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
   {
      psa_FineGain = psa_RxFineGains + gsa_BitloadLeftChannel[s_CurRxBand];

      for (s_ch = gsa_BitloadLeftChannel[s_CurRxBand]; s_ch <= gsa_BitloadRightChannel[s_CurRxBand]; s_ch++)
      {
         if (*psa_FineGain++ == NEG_INFINITY_DB)
         {
            CLEARTONEFLAG(p_ActiveTones_ToConsider, s_ch);
         }
      }
   }

   //Use these pair of global variables to keep track the last processed tones
   //so all the tones have equal chance to be processed
   gs_CurRxBand_save = 0;
   gs_ch_save = gsa_BitloadLeftChannel[0];

   while (l_DeltaSumLp != 0)
   {
      // Break out of the loop in case of error
      if (gus_BitloadErrorCode != 0)
      {
         break;
      }
      // This code only for TRUE -FALSE - TRUE configuration
      // Used in BS-S-8 and BS-S-9
      if ((l_DeltaSumLp > 0) && (ft_AddfineGainplusdeltaLp))
      {
         //*********************************************************************************************************//
         //BS-S-8. If BS-S-7 can not allocate all the bits removed from BS-S-5 then allocate final bits by modifying//
         //        FG on possible tones. This process always checks for RMS FG  value criteria(Start)               //
         //*********************************************************************************************************//
         //start with 0.5 dB
         //use relative large threshold in bit-swap, compared to that used in bitloading to
         //speed up the bit-swap operation.
         gs_ChooseFgThresh = (int16)(0.5 * 256);

         s_RxExtraBits = ChooseFineGains(puca_RxBat, psa_RxFineGains,l_DeltaSumLp, 0, 0);
         //*********************************************************************************************************//
         //BS-S-8 (End)                                                                                             //
         //*********************************************************************************************************//

         //*********************************************************************************************************//
         //BS-S-9. If reallocating removed tones from BS-S-5 fails due to RMS FG criteria then run again BS-S-8 for //
         //        rest of bits by disabling RMS FG check criteria (i.e. RMS FG <= 0 dB) (Start)                    //
         //*********************************************************************************************************//

         // XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code  (start)
         // If FG allocation fails then try one more time for rest of the bits (ie. (l_DeltaSumLp - s_RxExtraBits)) without checking
         // the RMS FG criteria.
         if (s_RxExtraBits < l_DeltaSumLp)
         {

            l_DeltaSumLp = l_DeltaSumLp - s_RxExtraBits; // number of bits needs to be allocated without checking avg. FG criteria

            // Initial FG threshold (i.e. gs_ChooseFgThresh) is taken from BS-S-8 where final FG threshold of that state was stored in
            // gs_ChooseFgThresh variable at ChooseFineGains() function.
            s_RxExtraBits = ChooseFineGains(puca_RxBat, psa_RxFineGains,l_DeltaSumLp, 0, 1);

            gft_RmsFgOverShoot = 1;
         }
         //*********************************************************************************************************//
         //BS-S-9 (End)                                                                                             //
         //*********************************************************************************************************//

         // XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code (end)
         if (s_RxExtraBits < l_DeltaSumLp)
         {
            gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_LP1;
         }
         // in case more bits are allocated (possible if MinBitsPerTone > 1)
         else if (s_RxExtraBits > l_DeltaSumLp)
         {
            // Bit fields of us_BiGiModificationControlFlags are already initialized before calling ChangeLpKeepMaxMargin()
            // ChangeBatKeepMaxMargin() uses only bit 0 and bit 3 flags of us_BiGiModificationControlFlags variable
            // (bit 0)ft_StopAtMarginThreshold
            // (bit 3)ft_Fast_UpShift_SRA_Mode = FALSE
            ft_Result = ChangeBatKeepMaxMargin(puca_RxBat, psa_RxFineGains, (l_DeltaSumLp - s_RxExtraBits), p_ActiveTones_ToConsider, p_ModifiedTones, us_BiGiModificationControlFlags);

            if ((ft_Result == FAIL) && (gus_BitloadErrorCode == 0))
            {
               gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_LP2;
            }
         }
      }
      else
      {
         // This portion of the code add or reduces Bits form BAT
         // l_DeltaSumLp = -ve remove bits -BS-S-6
         // l_DeltaSumLp = +ve bits bits (no FG changes) - BS-S-7
         // This code section also used by SRA
         // Bit fields of us_BiGiModificationControlFlags are already initialized before calling ChangeLpKeepMaxMargin()
         // ChangeBatKeepMaxMargin() uses only bit 0 and bit 3 flags of us_BiGiModificationControlFlags variable
         // For SRA (bit 0) ft_StopAtMarginThreshold = FALSE for down-shift SRA and TRUE for up-shift SRA
         // (bit 3) ft_Fast_UpShift_SRA_Mode = FALSE/TRUE used only for SRA
         ft_Result = ChangeBatKeepMaxMargin(puca_RxBat, psa_RxFineGains, l_DeltaSumLp, p_ActiveTones_ToConsider, p_ModifiedTones, us_BiGiModificationControlFlags);

         if ((ft_Result == FAIL) && (gus_BitloadErrorCode == 0))
         {
            gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_LP3;
         }
      }

      GetBatStats(puca_RxBat, &s_Num1BitTones, &s_TCM_Ovhd_after, &l_FinalSumBi);

      // Any change in TCM overhead results in further unwanted change to Lp.
      // This becomes a new delta Lp that we need to implement.
      l_DeltaSumLp = (s_TCM_Ovhd_after - s_TCM_Ovhd_before);

      s_TCM_Ovhd_before = s_TCM_Ovhd_after;

   } //while (l_DeltaSumLp != 0)

   // Update the return value ( important for showtime bitswap processing)
   *pl_ActualDeltaSumLp = (l_FinalSumBi - s_TCM_Ovhd_after - (l_InitialSumBi - s_Initial_TCM_Ovhd));

   // Return in case of error
   if(gus_BitloadErrorCode != 0)
   {
      return(FAIL);
   }


   if((gft_RxTcmFlag) && (ft_forceEven1Bit == TRUE) && (s_Num1BitTones&1))
   {
      // Force an even number of 1-bit tones without changing Lp.

      ft_Result = ForceEvenNum1BitTonesInBAT(puca_RxBat, psa_RxFineGains, p_ActiveTones_ToConsider, p_ModifiedTones);

      if (ft_Result == FAIL)
      {
         // Fail here if couldn't force even # of 1-bit tones while maintaining margin.
         // (Forcing an even # of 1-bit tones while maintaining Lp requires bits to be added to the BAT).
         if(gus_BitloadErrorCode == 0)
         {
            gus_BitloadErrorCode = BITLOAD_ERROR_EVEN_1BIT_TONES;
         }
         return (FAIL);
      }
   }

   // Fail here if: Decreased Lp by desired amount, but still didn't meet margin (used for validating L2 requests).
   // (FindToneWithExtremeMargin will return SUCCEED so long as there is one active tone).
   // (If deltaLp was positive and we reach this point, this margin check will always pass, since it was passed earlier).

   //Abu: FindToneWithExtremeMargin() function below is disabled because the result (exception code) was not used! It will decrease the MIPS consumption
   //Set this variable to -1 to indicate search for all the band
   //s_Dummy2ReturnValue = -1;
   //FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, gs_RxMinConstSize, gs_RxMaxConstSize, &s_Dummy1ReturnValue, &s_Dummy2ReturnValue, &s_ActualMinToneMargin);

   //   if (s_ActualMinToneMargin < s_MinToneMargin)
   //   {
   //      gus_BitloadErrorCode = CHGBAT_RETCODE_VIOLATES_MARGIN;
   //      return(FAIL);
   //   }

   // final check if Lp is adjusted correctly
   if ((l_DesiredDeltaSumLp != *pl_ActualDeltaSumLp) && (ft_StopAtMarginThreshold == FALSE))
   {
      return (FAIL);
   }
   else
   {
      return(SUCCEED);
   }
}

/*^^^
*----------------------------------------------------------------------------
*   Prototype: FlagT ChangeBatKeepMaxMargin(uint8 *puca_RxBat,
*                                           int16 *psa_RxFineGains,
*                                           int32 l_DesiredDeltaSumbi,
*                                           RxToneFlags p_ActiveTones,
*                                           RxToneFlags p_ModifiedTones,
*                                           uint16 us_BiGiModificationControlFlags)
*
*   Description:
*      Changes the bit allocation table (BAT) to change the Sum{bi} by the
*       specified delta.
*
*      The delta Sum{bi} may be positive or negative.  The specified min tone
*       margin argument specifies the minimum margin that must be maintained
*       on all tones.  The BAT will be modified so as to maximize the min SNR
*       margin of all tones.
*
*       Note that this function never forces the BAT to have an even number of
*       1-bit tones.  This is handled by the calling function.
*
*       The function sets the value of the Actual Sum{bi} output argument.
*      This will only differ from the desired delta Sum{bi} when it is not
*       possible to achieve the desired delta Sum{bi}.
*
*      One ToneFlag array specifies which tones in the BAT are allowed to
*       change.  A second ToneFlag array is written to by the function to
*       indicate which tones in the BAT were actually changed.  The fine gain
*       array input parameter is used to calculate SNR margins, but it is not
*       modified by this routine.
*
*   Input Parameters:
*      puca_RxBat          -- pointer to an Rx bit allocation table
*      psa_RxFineGains     -- pointer to an Rx fine gain table
*      l_DesiredDeltaSumBi -- desired change in Sum{bi}, either positive or negative
*      p_ActiveTones       -- One-bit-per-tone array indicating which tones
*                              in the BAT may be changed by this routine.
*                              Only tones whose bit is set will have their bi
*                              changed, and margin checks will only be done on
*                              these tones.
*
*      us_BiGiModificationControlFlags : only bit0 and bit3 bits are used
*
*      (bit 0 of us_BiGiModificationControlFlags) ft_StopAtMarginThreshold -- Flag,
*                              if set to TRUE, indicating that changes to the BAT
*                              should stop and the function should return as
*                              soon as the margin requirement is met.  This is
*                              intended for use in two different ways:
*
*          1) When the current BAT violates margin. Set this flag and use
*              the largest negative value of l_DesiredDeltaSumBi and the return
*              value pl_ActualDeltaSumBi will be the reduction in DeltaSumBi
*              needed to meet margin.
*
*          2) When the current BAT has excess margin. Set this flag and use
*              the largest positive value of l_DesiredDeltaSumBi and the return
*              value pl_ActualDeltaSumBi will be the increase in DeltaSumBi that
*              is possible while still meeting margin.
*
*     (bit 3 of us_BiGiModificationControlFlags) ft_Fast_UpShift_SRA_Mode -- FLAG
*              TRUE : Enable up to 5 bits/tone bit loading per iteration in case of Downstream up shift SRA
*              FALSE: load max 1 bit/tone per iteration
*
*   Output Parameters:
*      p_ModifiedTones     -- One-bit-per-tone array used to indicate which tones
*                              were actually changed by this routine. In some
*                           cases a bit may be set for a tone whose final bi
*                              value has not changed, because intermediate steps may
*                              have changed the bit and then changed it back again.
*
*   Returns:
*      SUCCEED/FAIL
*
*   Internal error code:
*      0:   success (no error)
*      1:   BAT change violates SNR margin
*      2:   impossible to increase/decrease Lp as desired
*
*   Global variables:
*      gs_RxMaxConstSize - (I) the maximum number of bits per RX tone.
*      gpsa_MeasuredSnrBuf - (I) measured SNR buffer, not adjusted for coding gain.
*      gs_MinTargetSnrMargin - (I) the minimum margin (in 8.8 dB format)
*                               that must be met by the BAT table after any modifications
*-----------------------------------------------------------------------------
^^^*/
FlagT ChangeBatKeepMaxMargin(uint8 *puca_RxBat,
                             int16 *psa_RxFineGains,
                             int32 l_DesiredDeltaSumbi,
                             RxToneFlags p_ActiveTones_ToConsider,
                             RxToneFlags p_ModifiedTones,
                             uint16 us_BiGiModificationControlFlags)
{

   FlagT ft_Result;
   int16 s_ch, s_ChIdx;            // channel index
   int16 s_Margin;                 //8.8dB format
   int16 sa_PTwithDOP[MAX_NUM_DOP_PTS] = {0}; // holds the pilot tone indexes for data-on-pilot (DOP) enabled pilot tones
   int16 s_CurRxBand;
   int32 l_DeltaSumbi;
   int16 s_FineGaininit, s_excessmargin;
   int16 s_threshold, s_MinMargin_toComp, s_StepSize;
   int16 s_bi;
   int16 s_Dummy;
   int16 *psa_SNRBuffer, s_MinToneMargin;

   FlagT ft_StopAtMarginThreshold = FALSE;
   FlagT ft_Fast_UpShift_SRA_Mode = FALSE;

   // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (Start)
   // ChangeBatKeepMaxMargin() uses only bit0 and bit3 flags of us_BiGiModificationControlFlags variable
   // Decode control flags from the us_BiGiModificationControlFlags bit fields
   if (us_BiGiModificationControlFlags & FT_STOP_AT_MARGIN_THRESHOLD)  // bit 0
   {
      ft_StopAtMarginThreshold = TRUE;
   }
   if (us_BiGiModificationControlFlags & FT_FAST_UPSHIFT_SRA_MODE)     // bit 3
   {
      ft_Fast_UpShift_SRA_Mode = TRUE;
   }
   // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (End)

   psa_SNRBuffer = gpsa_MeasuredSnrBuf;
   s_MinToneMargin = gs_MinTargetSnrMargin;
   l_DeltaSumbi = l_DesiredDeltaSumbi;

   //*********************************************************************************************************//
   //BS-S-7. Add bits only on the tones which have sufficient margin to accommodate extra bits.               //
   //        This process does not change any FG                                                              //
   //*********************************************************************************************************//
   // XDSLRTFW-3918 Start SRA/BS problem due to Data on Pilot
   for (int32 i=PT_ARRAY_IDX_0, uc_k=0;i<MAX_NUM_PILOT_TONES;i++)
   {
      if(gt_PilotConfig.ta_PilotTones[i].ft_DeRotatePilot == TRUE)
      {
         // store the indexes the pilot tone indexes for data-on-pilot enabled pilot tones
         // we map the maximum of 2 enabled DOP tones from the 3 DS bands to the 2 elements of the array sa_PTwithDOP
         sa_PTwithDOP[uc_k++] = gt_PilotConfig.ta_PilotTones[i].s_PilotToneIdx;
      }
   }
   // XDSLRTFW-3918 End

   while (l_DeltaSumbi > 0)
   {

      //Set this variable to -1 to indicate search for all the band
      s_CurRxBand = -1;

      // Find tone with biggest margin.
      // LARGEST_MARGIN_AFTER_ADDING_1_BIT - Search for the tone which, after adding a bit has the largest resulting SNR margin.
      ft_Result = FindToneWithExtremeMargin(LARGEST_MARGIN_AFTER_ADDING_1_BIT, puca_RxBat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, 0, (int16) (gs_RxMaxConstSize - 1), &s_ch, &s_CurRxBand, &s_Margin);

      if (ft_Result == FAIL)
      {
         gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_LP4;
         return(FAIL);
      }

      // If adding bits to this tone will violate margin, stop.
      if ((ft_StopAtMarginThreshold) && (s_Margin < s_MinToneMargin))
      {
         gus_BitloadErrorCode = CHGBAT_RETCODE_VIOLATES_MARGIN;
         return (FAIL);
      }

      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
      // debug code start
      // this code logged Tone, FG, BAT, Margin of the tones before bits are loaded in TRUE-FALSE-FALSE configuration
      s_bi = puca_RxBat[s_ch];

      s_Margin = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
      // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
      // avoid the possible overflow problem (for negative infinity) and also fine for 0db case.
      s_Margin += psa_RxFineGains[s_ch];
      // When subtracting bits and looking for the tone with the smallest margin, we use bi_delta = 0.
      s_Margin -= gsa_ConstellationSNR[s_bi];

      if(gs_RxState == R_O_SHOWTIME_RX)
      {
         if (!(IS_TONEFLAGSET(p_ModifiedTones, s_ch)))
         {
            // log  tone idx, initial BAT, FG and Margin information of bit swap-able tone
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_ch;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = puca_RxBat[s_ch];
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_Margin;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = psa_RxFineGains[s_ch];
         }
      }
      // Debug code end
#endif
      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)

      // XDSLRTFW-4051 Fast downstream up shift SRA(start)
      // Finding SRA Margin on the given tone
      if (ft_Fast_UpShift_SRA_Mode)
      {
         s_bi = puca_RxBat[s_ch];

         s_Margin = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
         // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
         // avoid the possible overflow problem (for negative infinity) and also fine for 0db case.
         s_Margin += psa_RxFineGains[s_ch];
         // When subtracting bits and looking for the tone with the smallest margin, we use bi_delta = 0.
         s_Margin -= gsa_ConstellationSNR[s_bi];

         // margin left for bitloading
         s_Margin -= gs_MinTargetSnrMargin;
      }


      // XDSLRTFW-3918 Start SRA/BS problem due to Data on Pilot
      if ( (s_ch != sa_PTwithDOP[0]) && (s_ch != sa_PTwithDOP[1]) )
      {
         // Handling for normal tone
         // if tone was initially unloaded put the fine gain to be zero
         if (puca_RxBat[s_ch] == 0)
         {
            // Update linear sum square of fine gain
            // In case tone is getting loaded for the first time, initial fine gain does not matter.
            // We set the fine gain of this tone to gs_fgain_adjust dB
            // so we will not violate the average gain square constraint

            // XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case (Start)
            UpdateFGSumSqAndNOMATP(s_ch, psa_RxFineGains[s_ch], (gs_fgain_adjust - gs_ExcessMarFGReduction), &gla_SumFGainLinSq[s_CurRxBand], 2);
            psa_RxFineGains[s_ch] = (gs_fgain_adjust - gs_ExcessMarFGReduction); // FG of monitoring tone
            // XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case (End)
            gusa_ncloaded[s_CurRxBand]++;

                     // Fast Up shift SRA mode where maximum 5 bits/tone can be loaded
            if (ft_Fast_UpShift_SRA_Mode)
            {
                int16 s_Nobit;       // Number of bits to be loaded on tone s_ch

                // SNR computation on monitoring tones (i.e. tone with zero bit loading) are not very accurate
                // so that higher bit loading (more than 1 bit)  on those tones 6 dB extra margin is considered.
                //
                if (s_Margin >= (21 << 8))
                   s_Nobit = 5;                       // 5 bits
                else if (s_Margin >= (18 << 8))
                   s_Nobit = 4;                       // 4 bits
                else if (s_Margin >= (15 << 8))
                   s_Nobit = 3;                       // 3 bits
                else if (s_Margin >= (12 << 8))
                   s_Nobit=2;                         // 2 bits
                else
                   s_Nobit=gs_RxMinConstSize;         // 1 bits

                if (s_Nobit > l_DeltaSumbi) //l_DeltaSumbi must not be negative
                   s_Nobit = l_DeltaSumbi;

                puca_RxBat[s_ch] = puca_RxBat[s_ch] + s_Nobit;

                // Only gs_RxMaxConstSize bits are allowed per tone!
                if (puca_RxBat[s_ch] > gs_RxMaxConstSize)
                {
                   s_Nobit = s_Nobit -(puca_RxBat[s_ch]-gs_RxMaxConstSize); // actual number of bit which will be loaded
                   puca_RxBat[s_ch] = gs_RxMaxConstSize;
                }
                l_DeltaSumbi = l_DeltaSumbi-s_Nobit;
            }
            else // normal SRA mode where 1 bit/tone is loaded
            {
               puca_RxBat[s_ch] += gs_RxMinConstSize;
               l_DeltaSumbi -= gs_RxMinConstSize;
            }
         }
         else //if (puca_RxBat[s_ch] != 0)
         {
            // Fast Up shift SRA mode where maximum 5 bits/tone can be loaded
            if (ft_Fast_UpShift_SRA_Mode)
            {
               int16 s_Nobit;       // Number of bits to be loaded on tone s_ch

               if (s_Margin >= (15 << 8))
                  s_Nobit = 5;                       // 5 bits
               else if (s_Margin >= (12 << 8))
                  s_Nobit = 4;                       // 4 bits
               else if (s_Margin >= (9 << 8))
                  s_Nobit = 3;                       // 3 bits
               else if (s_Margin >= (6 << 8))
                  s_Nobit = 2;                       // 2 bits
               else
                  s_Nobit = 1;                       // 1 bits

               if (s_Nobit > l_DeltaSumbi) //l_DeltaSumbi must not be negative
                  s_Nobit = l_DeltaSumbi;

               puca_RxBat[s_ch] = puca_RxBat[s_ch] + s_Nobit;

               // Only gs_RxMaxConstSize bits are allowed per tone!
               if (puca_RxBat[s_ch] > gs_RxMaxConstSize)
               {
                  s_Nobit = s_Nobit -(puca_RxBat[s_ch]-gs_RxMaxConstSize); // actual number of bit which will be loaded
                  puca_RxBat[s_ch] = gs_RxMaxConstSize;
               }
               l_DeltaSumbi = l_DeltaSumbi-s_Nobit;
            }
            else // normal SRA mode where 1 bit/tone is loaded
            {
               //Note we only add bit, but not change gain
               puca_RxBat[s_ch]++;
               l_DeltaSumbi--;
            }
         }
      }
      else // handling for DOP pilot tone
      {
         // ensure that we have always <=2 bits on data-on-pilot enabled pilot tones
         if ( (puca_RxBat[s_ch] == 0) && (gs_RxMinConstSize <= 2))
         {
            // Note we only add bit, but not change gain
            puca_RxBat[s_ch] += gs_RxMinConstSize;
            l_DeltaSumbi -= gs_RxMinConstSize;
            // Fix for XDSLRTFW-3918: Added missing increment of gusa_ncloaded
            gusa_ncloaded[s_CurRxBand]++;
         }
         else if( (puca_RxBat[s_ch] < 2) && (gs_RxMinConstSize <= 2))
         {
            // Only add bit if we do not exceed 2-bit constellation
            // Note we only add bit, but not change gain
            puca_RxBat[s_ch]++;
            l_DeltaSumbi--;
         }
         else
         {
            // DOP pilot tone can't carry more bits. Clear it from TEMPORARY array ActiveTone_ToConsider in order to avoid
            // that this tone is attempted to be loaded in the next iterations of the while loop
            CLEARTONEFLAG(p_ActiveTones_ToConsider, s_ch);
            continue;
         }
      }
      // XDSLRTFW-3918 End

      // XDSLRTFW-4051 Fast downstream up shift SRA(End)
      SETTONEFLAG(p_ModifiedTones, s_ch);
   }

   //Done with change the Lp
   if(l_DeltaSumbi == 0)
   {
      ft_Fast_UpShift_SRA_Mode =0;
      return(SUCCEED);
   }

   //*********************************************************************************************************//
   //BS-S-6. Remove bit from tones which has lower margin than minimum margin defined in BS-S-4. Remove extra //
   //FG from that tone with maintaining minimum margin criteria and update RMS FG value.                      //
   //*********************************************************************************************************//
   ///////////////////////////////////////////////////////////////////////////
   // Remove bits by looking margin which are below min margin + threshold ///
   ///////////////////////////////////////////////////////////////////////////

   // start with threshold =~ 0.1 dB
   s_StepSize = 25;
   s_threshold = s_StepSize;

   //Set this variable to -1 to indicate search for all the band
   s_Dummy = -1;

   // compute the min margin
   ft_Result = FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, gs_RxMinConstSize, gs_RxMaxConstSize, &s_ch, &s_Dummy, &s_Margin);

   if(ft_Result == FAIL)
   {
      gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_BAT1;
      return (FAIL);
   }

   // if min  margin is below -3dB (=-768), start with -3dB for removing bits
   //This is possible during the showtime when some narrow band noise is injected
   //to a certain frequency
   if (s_Margin < -768)
   {
      s_MinMargin_toComp =-768;
   }
   else
   {
      s_MinMargin_toComp = s_Margin;
   }

   s_CurRxBand = gs_CurRxBand_save;
   s_ChIdx = gs_ch_save;

   while (l_DeltaSumbi < 0) // this loop counts the number of bits to be removed
   {
      // Bits are removed from lowest SNR-Margin tone to higher SNR-Margin tone up to 50 bits or up to all remaining
      // tones with at least Minimum target margin, which come first basis. Minimum target margin is defined in BS-S-4.
      // Margin per tone is calculated with the following formula
      // Margin = (Measured SNR + Coding gain + FG) - constellation SNR
      while (s_CurRxBand < gs_NumOfRxBands) // Rx band loops
      {
         while (s_ChIdx <= gsa_BitloadRightChannel[s_CurRxBand]) // Tones per band loop
         {
            s_ch = s_ChIdx++;

            if (IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch) && (l_DeltaSumbi < 0))
            {
               s_bi = puca_RxBat[s_ch];

               if (s_bi < gs_RxMinConstSize)
               {
                  continue;
               }

               s_Margin = AddCodingGainToSnr(s_ch, psa_SNRBuffer);

               // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
               // avoid the possible overflow problem (for negative infinity) and also fine for 0db case.
               s_Margin += psa_RxFineGains[s_ch];

               // When subtracting bits and looking for the tone with the smallest margin, we use bi_delta = 0.
               s_Margin -= gsa_ConstellationSNR[s_bi];

               // Check if the margin is better than margin to compare + threshold
               if (s_Margin > s_MinMargin_toComp + s_threshold)
               {
                  continue;
               }
               // Abu: Bit unloading problem
               // According to current implementation bit unloading can be done up to (gs_MinTargetSnrMargin + s_StepSize).
               // This means for 3dB target margin cases bit unloading will be done on tones where the per tone margins are
               // less than our equal to 2.1 dB.
               // Later on removed bits allocation algorithm tries load bits on a tone in a way that, that tone has
               // at least gs_MinTargetSnrMargin!
               // This conflicting logic can trigger series of (if not continuous) bit swapping!

               if (puca_RxBat[s_ch] == gs_RxMinConstSize)
               {
                  //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
                  if(gs_RxState == R_O_SHOWTIME_RX)
                  {
                     if (!(IS_TONEFLAGSET(p_ModifiedTones, s_ch)))
                     {
                        // log  tone idx, initial BAT, FG and Margin information of bitswapable tone
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_ch;
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = puca_RxBat[s_ch];
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_Margin;
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = psa_RxFineGains[s_ch];
                     }
                  }
#endif
                  //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)
                  // update fine gain linear sum square
                  UpdateFGSumSqAndNOMATP(s_ch, psa_RxFineGains[s_ch], (0 - gs_ExcessMarFGReduction), &gla_SumFGainLinSq[s_CurRxBand], 1); // XDSLRTFW-4005:

                  s_bi = puca_RxBat[s_ch];
                  puca_RxBat[s_ch] = 0;
                  //psa_RxFineGains[s_ch] = 0;
                  psa_RxFineGains[s_ch] = 0 - gs_ExcessMarFGReduction;  // XDSLRTFW-4005: FG of Monitoring tone
                  gusa_ncloaded[s_CurRxBand]--;
                  SETTONEFLAG(p_ModifiedTones, s_ch);

                  // if # of bits to trim is less than MinBitsPerTone,
                  // then we need to add bits to other tones
                  // Currently we add all bits to a single previously loaded tones
                  // in an attempt to not change the # of loaded tones, rather than
                  // distribute it ideally over all available tones optimally
                  // FIXME: We can do a better job here
                  if (l_DeltaSumbi > -gs_RxMinConstSize)
                  {
                     // Set this variable to -1 to indicate search for all the band
                     s_Dummy = -1;
                     // Find tone with biggest margin to add (l_DeltaSumbi+gs_RxMinConstSize) bits
                     // Only consider tones that satisfy the following constraints:
                     // 1) RxBat[tone] >= gs_RxMinConstSize
                     // 2) RxBat[tone] <= gs_RxMaxConstSize-(l_DeltaSumbi+gs_RxMinConstSize)
                     // 3) tone index is not a DOP enabled pilot tone

                     // XDSLRTFW-3918 Start SRA/BS problem due to Data on Pilot
                     for (int32 i=PT_ARRAY_IDX_0;i<MAX_NUM_PILOT_TONES;i++)
                     {
                        if(gt_PilotConfig.ta_PilotTones[i].ft_DeRotatePilot == TRUE)
                        {
                           // Remove DOP pilot tones in order to avoid that bits are attempted to be loaded here
                           CLEARTONEFLAG(p_ActiveTones_ToConsider, gt_PilotConfig.ta_PilotTones[i].s_PilotToneIdx);
                        }
                     }
                     // XDSLRTFW-3918 End

                     ft_Result = FindToneWithExtremeMargin(LARGEST_MARGIN_AFTER_ADDING_1_BIT, puca_RxBat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, gs_RxMinConstSize, (int16) (gs_RxMaxConstSize- (l_DeltaSumbi + gs_RxMinConstSize)), &s_ch, &s_Dummy, &s_Margin);

                     if (ft_Result == FAIL)
                     {
                        gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_LP5;
                        return(FAIL);
                     }

                     // If adding bits to this tone will violate margin, stop.
                     if ((ft_StopAtMarginThreshold) && (s_Margin < s_MinToneMargin))
                     {
                        gus_BitloadErrorCode = CHGBAT_RETCODE_VIOLATES_MARGIN;
                        return (FAIL);
                     }

                     puca_RxBat[s_ch] += (l_DeltaSumbi+gs_RxMinConstSize);
                     // XDSLRTFW-3112 Invalid DS OLR requests with MinConstellationSize = 2 (Start_End)
                     // Marking modified tone s_ch in p_ModifiedTones was forgotten, hence this tone was missing in OLR message and lead to an inavlid OLR request, which got discarded by CO
                     SETTONEFLAG(p_ModifiedTones, s_ch);
                     l_DeltaSumbi = 0;
                  }
                  else
                  {
                     l_DeltaSumbi += gs_RxMinConstSize;
                  }


               } //if (puca_RxBat[s_ch] == gs_RxMinConstSize)
               else
               {
                  //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
                  if(gs_RxState == R_O_SHOWTIME_RX)
                  {
                     if (!(IS_TONEFLAGSET(p_ModifiedTones, s_ch)))
                     {
                        // log  tone idx, initial BAT, FG and Margin information of bitswapable tone
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_ch;
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = puca_RxBat[s_ch];
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_Margin;
                        gsa_BSDebugBuf[gs_DSDebugBufCnt++] = psa_RxFineGains[s_ch];
                     }
                  }
#endif
                  //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)
                  puca_RxBat[s_ch]--;
                  l_DeltaSumbi++;

                  // remove excess gain (BIS), if not SRA
                  if ((psa_RxFineGains[s_ch] > gs_min_fine_gain) && (gft_FineGainOn==TRUE) )
                  {
                     // calculate excess gain
                     s_excessmargin = s_Margin - s_MinToneMargin + (gsa_ConstellationSNR[s_bi] - gsa_ConstellationSNR[s_bi -1]);
                     s_FineGaininit = psa_RxFineGains[s_ch];

                     if(s_excessmargin > 0)  // to ensure that we are taking the gains out, may e in case of bad tones, margin may be negative here
                     {
                        psa_RxFineGains[s_ch] -= s_excessmargin;
                        if (psa_RxFineGains[s_ch] < gs_min_fine_gain)
                        {
                           psa_RxFineGains[s_ch] = gs_min_fine_gain;
                        }

                        // XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case (Start)
                        // gs_min_fine_gain_MarginRed = -2560 // = 10 dB in 8.8dB format
                        if (psa_RxFineGains[s_ch] < gs_min_fine_gain_MarginRed)
                        {
                           psa_RxFineGains[s_ch] = gs_min_fine_gain_MarginRed;
                        }
                        // XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case (End)

                        // update fine gain linearsum sq
                        UpdateFGSumSqAndNOMATP(s_ch, s_FineGaininit, psa_RxFineGains[s_ch], &gla_SumFGainLinSq[s_CurRxBand], 0);
                     }
                  }

                  SETTONEFLAG(p_ModifiedTones, s_ch);
               } //if (puca_RxBat[s_ch] == gs_RxMinConstSize) else

               if(l_DeltaSumbi >= 0)
               {
                  goto _WhileLoopEnd;
               }

            } //if (IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch) && (l_DeltaSumbi < 0))

         } //while(s_ChIdx <= gsa_BitloadRightChannel[s_CurRxBand])

         s_CurRxBand++;

         if(s_CurRxBand < gs_NumOfRxBands)
         {
            s_ChIdx = gsa_BitloadLeftChannel[s_CurRxBand];
         }

      } //while (s_CurRxBand < gs_NumOfRxBands)

      s_CurRxBand = 0;
      s_ChIdx = gsa_BitloadLeftChannel[s_CurRxBand];

      // Check whether we now meet margin and break if that flag is set.

      //Set this variable to -1 to indicate search for all the band
      s_Dummy = -1;
      ft_Result = FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, gs_RxMinConstSize, gs_RxMaxConstSize, &s_ch, &s_Dummy, &s_Margin);

      if(ft_Result == FAIL)
      {
         gus_BitloadErrorCode = CHGBAT_RETCODE_CANT_CHANGE_BAT2;
         return (FAIL);
      }

      if ((s_Margin >= s_MinToneMargin) && (ft_StopAtMarginThreshold == TRUE))
      {
         break;
      }

      //If the margin is still lower than the threshold,
      //implies that we have not been able to touch the low margin tone,
      //increase the threshold, instead of resetting the margin to compare
      if (s_Margin <=  s_MinMargin_toComp + s_threshold)
      {
         s_threshold += s_StepSize;
      }
      else //otherwise reset the threshold
      {
         s_MinMargin_toComp = s_Margin;
         s_threshold = s_StepSize;
      }

   } // while (l_DeltaSumbi < 0)

_WhileLoopEnd:

   if(s_ChIdx > gsa_BitloadRightChannel[s_CurRxBand])
   {
      s_CurRxBand++;

      if(s_CurRxBand >= gs_NumOfRxBands)
      {
         s_CurRxBand = 0;
      }

      s_ChIdx = gsa_BitloadLeftChannel[s_CurRxBand];
   }

   gs_CurRxBand_save = s_CurRxBand;
   gs_ch_save = s_ChIdx;

   //When ft_StopAtMarginThreshold is TRUE and l_DesiredDeltaSumbi < 0
   //we allow that the number of bits removed is less than the desired
   if((ft_StopAtMarginThreshold == TRUE) && (l_DeltaSumbi <= 0) && (l_DesiredDeltaSumbi < 0))
   {
      return (SUCCEED);
   }

   //Otherwise, the number of removed bits must match to that desired
   if (l_DeltaSumbi == 0)
   {
      return (SUCCEED);
   }
   else
   {
      return (FAIL);
   }
}

/*^^^
*-----------------------------------------------------------------------------
*   Prototype:   FlagT ForceEvenNum1BitTonesInBAT(uint8 *puca_Bat,
*                                                int16 *psa_RxFineGains,
*                                                ToneFlags p_ActiveTones,
*                                                ToneFlags p_ModifiedTones)
*
*   Description:
*      Makes minimal adjustments to an Rx BAT to force the number of 1-bit tones to be even, without changing
*      the value of Lp = Sum{bi} - TCM_Overhead.  It is assumed that TCM is enabled.
*
*   Input Parameters:
*      puca_Bat -- pointer to an Rx bit allocation table
*      psa_RxFineGains -- pointer to an Rx fine gain table
*      psa_SNRBuffer -- measured SNR buffer, not adjusted for coding gain or anything else.
*      s_MinToneMargin -- the minimum margin (in 8.8 dB format) that must be met by the BAT table after any modifications
*      p_ActiveTones -- One-bit-per-tone array indicating which tones in the BAT may be changed by this routine.  Only tones
*            whose bit is set will have their bi changed, and margin checks will only be done on these tones.

*   Output Parameters:
*      p_ModifiedTones -- One-bit-per-tone array used to indicate which tones were actually changed by this routine.  In some
*            cases a bit may be set for a tone whose final bi value has not changed, because intermediate steps may have
*            changed the bit and then changed it back again.
*
*   Returns:
*      FlagT value indicating SUCCESS or FAIL.  SUCCESS is the normal return value.
*         FAIL is returned if the necessary changes to the BAT are impossible or would violate the specified min margin.
*
*   Global variables:
*
  *-----------------------------------------------------------------------------
^^^*/

FlagT ForceEvenNum1BitTonesInBAT(uint8 *puca_Bat, int16 *psa_RxFineGains, RxToneFlags p_ActiveTones, RxToneFlags p_ModifiedTones)
{
   int16  s_Num1BitTones, s_TCM_Ovhd;
   int32 l_Dummy, s_New_bi;
   int16 s_ch, s_band, s_Margin, s_ch_leastmargin, s_band_leastmargin;
   FlagT ft_Result = FAIL;
   int16 s_CaseType;   // Takes on value of 1,2,3 or 4, depending on values of : NumLoadedTones (mod 2)
   // and Num1BitTones (mod 4).
   RxToneFlags p_ActiveTones_ToConsider;
   int16 *psa_SNRBuffer, s_MinToneMargin;

   psa_SNRBuffer = gpsa_MeasuredSnrBuf;
   s_MinToneMargin = gs_MinTargetSnrMargin;


   GetBatStats(puca_Bat, &s_Num1BitTones, &s_TCM_Ovhd, &l_Dummy);

   // Copy the active tone set to another array
   memcpy(p_ActiveTones_ToConsider, p_ActiveTones, sizeof(RxToneFlags));

   // XDSLRTFW-3918 Start SRA/BS problem due to Data on Pilot
   for (int32 i=PT_ARRAY_IDX_0;i<MAX_NUM_PILOT_TONES;i++)
   {
      // Exclude all pilot tones from p_ActiveTones_ToConsider array in order to avoid a loading of trellis bits to them
      // Note: it would be sufficient to clear DOP enabled pilot tones (they are the only ones included in p_ActiveTones_ToConsider),
      // but by removing all pilot tones makes the code is easier, faster and more flexible
      CLEARTONEFLAG(p_ActiveTones_ToConsider, gt_PilotConfig.ta_PilotTones[i].s_PilotToneIdx);
   }
   // XDSLRTFW-3918 End

   if(s_Num1BitTones & 1)
   {
      s_CaseType = (gus_ncloaded & 0x1) + (s_Num1BitTones & 0x3); // There are four different case types of interest.

      // First drop a one-bit tone to zero. Choose the tone with the lowest margin.

      //Set this variable to -1 to indicate search for all the band
      s_band_leastmargin = -1;
      FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_Bat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones, 1, 1, &s_ch_leastmargin, &s_band_leastmargin, &s_Margin);

      CLEARTONEFLAG(p_ActiveTones_ToConsider, s_ch_leastmargin);

      // Under certain conditions this will have decreased Lp.  Check for these cases.
      if ((s_CaseType == 1) || (s_CaseType == 4))
      {
         // Need to increment Lp.
         if (gus_ncloaded > s_Num1BitTones)
         {
            // Since there exists at least one tone with bi > 1, increment Lp by incrementing that bi.
            // This will work unless for all bi > 1, bi = 15, but this is highly unlikely.

            // Find tone that has biggest margin after adding a bit to it.

            //Set this variable to -1 to indicate search for all the band
            s_band = -1;
            ft_Result = FindToneWithExtremeMargin(LARGEST_MARGIN_AFTER_ADDING_1_BIT, puca_Bat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, 2, (int16) (gs_RxMaxConstSize - 1), &s_ch, &s_band, &s_Margin);

            if (ft_Result == FAIL)
            {
               gus_BitloadErrorCode = BITLOAD_ERROR_EVEN_1BIT_TONES1;
               return (FAIL); // Tones that don't have bi=1 all have bi=15. Highly unlikely!!
            }

            // JB ??? - can we guarantee min margin here?
            /*if (s_Margin < s_MinToneMargin)
            {
               return(FAIL);
            }
            */

            // Increment the BAT and mark the tone as modified.
            puca_Bat[s_ch] += 1;
            SETTONEFLAG(p_ModifiedTones, s_ch);
         }
         else
         {


            // All tones have bi=1.
            // In this case increment Lp by performing the following changes on four 1-bit tones:
            // 1->0, 1->0, 1->2, 1->3

            uint8 uca_ChangeToBi[4] = {0,0,2,3};
            int16 s_NumTonesChanged=0;
            int16 s_save_tonenum[4];
            int16 s_save_BandIdx[4];

            while (s_NumTonesChanged < 4)
            {
               // Get next 1-bit tone with largest margin.

               //Set this variable to -1 to indicate search for all the band
               s_band = -1;
               ft_Result = FindToneWithExtremeMargin(LARGEST_MARGIN_AFTER_ADDING_1_BIT, puca_Bat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, 1, 1, &s_ch, &s_band, &s_Margin);

               if (ft_Result == FAIL)
               {
                  gus_BitloadErrorCode = BITLOAD_ERROR_EVEN_1BIT_TONES2;
                  return (FAIL);
               }

               /*
               // Check that adding bits to this tone won't violate margin.  If so, fail.
               if (s_Margin < s_MinToneMargin)
               {
                  return(FAIL);
               }
               */

               s_save_tonenum[s_NumTonesChanged] = s_ch;
               s_save_BandIdx[s_NumTonesChanged] = s_band;

               s_NumTonesChanged++;

               CLEARTONEFLAG(p_ActiveTones_ToConsider, s_ch);
            }

            if (ft_Result != FAIL)
            {
               // do actual bat changes and set modified tones flag

               for (s_NumTonesChanged=0; s_NumTonesChanged < 4; s_NumTonesChanged ++)
               {
                  s_ch = s_save_tonenum[s_NumTonesChanged];
                  s_band = s_save_BandIdx[s_NumTonesChanged];
                  s_New_bi = uca_ChangeToBi[s_NumTonesChanged];
                  puca_Bat[s_ch] = s_New_bi;
                  SETTONEFLAG(p_ModifiedTones, s_ch);

                  // If bi set to 0, reset fine gain to 0dB.
                  if (s_New_bi == 0)
                  {
                     // XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case (Start)
                     //update linear sum finegain
                     UpdateFGSumSqAndNOMATP(s_ch, psa_RxFineGains[s_ch], (0-gs_ExcessMarFGReduction), &gla_SumFGainLinSq[s_band], 1);
                     psa_RxFineGains[s_ch] = (0 - gs_ExcessMarFGReduction); // FG of monitoring tone
                     // XDSLRTFW-4005: DS VDSL Fine gains are increased by bitswaps even in max noise margin case (End)
                     gusa_ncloaded[s_band]--;
                  }
               }
            }
         }

      } //if ((s_CaseType == 1) || (s_CaseType == 4))


      // Zero the BAT for 1 bit tone saved earlier, reset the fine gain, and mark the tone as modified.
      // should update linearsum finegain for current maintenance of sumfinegain
      // not being done here for codeswap reasons.
      UpdateFGSumSqAndNOMATP(s_ch_leastmargin, psa_RxFineGains[s_ch_leastmargin], (0-gs_ExcessMarFGReduction), &gla_SumFGainLinSq[s_band_leastmargin], 1);

      puca_Bat[s_ch_leastmargin] = 0;
      psa_RxFineGains[s_ch_leastmargin] = (0-gs_ExcessMarFGReduction); // 0dB
      gusa_ncloaded[s_band_leastmargin]--;

      SETTONEFLAG(p_ModifiedTones, s_ch_leastmargin);

   } //if(s_Num1BitTones & 1 == 1)

   return(SUCCEED);
}
