/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: ConvertLogToLinear.c
*
*   This file contains a function to convert log to linear.
*
*-------------------------------------------------------------------------------
*/

#include "common.h"
#include "dsp_op.h"

/*^^^
*------------------------------------------------------------------------
*
*  Name : ConvertLogToLinear
*
*   Prototype:
*       void ConvertLogToLinear(void* pusa_LinearPSDOut, uint16 pusa_LogPSDIn)
*
*   Abstract:
*      This function converts a Q7.9 Log value into corresponding a Q3.13 Linear Value.
*
*   Input Parameters:
*
*   Global Variables:
*
*  Notes: Initially the first parameter was of type uint16 but to suppress/remove a warning
 *        it has been changed to type void. And suitable changes are made in the function
 *        and wherever it is being employed.
*-----------------------------------------------------------------------------
^^^*/
/* Coefficients of the Antilog series where input is between 0 and -1 */
#define POW10X_ORDER   (4)
#define POW10X_C0_MANT   (8186)   //0.9993 in Q3.13 format
#define POW10X_C1_MANT   (18665)   //2.2785 in Q3.13 format
#define POW10X_C2_MANT   (20207)   //2.4667 in Q3.13 format
#define POW10X_C3_MANT   (12130)   //1.4807 in Q3.13 format
#define POW10X_C4_MANT   (3225)   //0.3937 in Q3.13 format
#define MAX_32  0x7fffffff
#define MIN_32  0x80000000

void ConvertLogToLinear(uint16* pusa_LinearPSDOut /* Q3.13 */, uint16 pusa_LogPSDIn /* Q7.9 */)
{
   int16 j, s_ScaleLinearOutBy10;
   int32 l_LogPSD, l_x, l_y, l_acc0, l_acc1;
   int16 sa_pow10xCoef[POW10X_ORDER+1];// For inverse log computation.
   // Initialize pow(10,x) coefficient array
   sa_pow10xCoef[0] = POW10X_C0_MANT;
   sa_pow10xCoef[1] = POW10X_C1_MANT;
   sa_pow10xCoef[2] = POW10X_C2_MANT;
   sa_pow10xCoef[3] = POW10X_C3_MANT;
   sa_pow10xCoef[4] = POW10X_C4_MANT;

//   for(i=0; i < s_num_tones; i++)

   // Convert to 6.10 dB format and invert.
   l_LogPSD = (-(int32)pusa_LogPSDIn) << 1;

   // Only perform conversion from dB to linear scale if non-zero value
   if(l_LogPSD == 0)
   {
      // Unity tssi is 1024 in Q1.10. Since 32767 is aprox 1 in Q1.15 then the
      // inverse of 1024 is 32 in Q11.5.
      *pusa_LinearPSDOut = 1024;
   }
   else
   {
      // Range of l_LogPSD is [-63,0] dB. If l_LogPSD < -20 dB (-1024*20 in Q6.10), add 1
      // to the exponent, compute pow(10,l_LogPSD+20) and divide the result by 10 afterwards.
      s_ScaleLinearOutBy10 = 0;
      while (l_LogPSD <= (-1024*20))
      {
         s_ScaleLinearOutBy10++;
         l_LogPSD += (1024*20);
      }
      // Convert to Q1.15 format (mult by 64/40 or 8/5)
      l_x = (int16)(l_LogPSD / 5);
      l_x <<= 3;
      /***********************************************************************/
      /* Compute pow(10,s_x) = C4*s_x^4 + C3*s_x^3 + C2*s_x^2 + C1*s_x + C0  */
      /***********************************************************************/
      // Compute C0
      l_acc0 = ((int32)sa_pow10xCoef[0])<<15; // Left shift C0 by 15 to convert C0 from Q3.13 to Q4.28
      // Initialize s_y to s_x
      l_y = l_x;
      // Compute (C4*x^4 + C3*x^3 + C2*x^2 + C1*x + C0) in Q4.28
      // Each iteration of outer loop generates C(j)*x^j
      for(j=1; j<=POW10X_ORDER; j++)
      {
         if(j>1)
         {
            l_acc1 = ((int32)l_y * l_x);
            l_y = round(l_acc1, 15);
         }

         // Compute C(j)*x^j in Q4.28
         l_acc0 += ((int32)sa_pow10xCoef[j] * l_y);
      }
      // Extract upper word which is Q4.12
      *pusa_LinearPSDOut = (int16)round(l_acc0, 16);

      // Scale by 10^ScaleBy10 if original value of l_LogTssi < -20 dB
      while (s_ScaleLinearOutBy10 > 0)
      {
         *pusa_LinearPSDOut /= 10;
         s_ScaleLinearOutBy10--;
      }
      // Convert Q4.12 to Q6.10
      *pusa_LinearPSDOut >>= 2;
   } // end of else - non-zero LogPSD

   // Finally, convert Q6.10 to Q3.13
   *pusa_LinearPSDOut <<= 3;
   return;
}
