/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   ConvertToDB.c
*
*   This file contains a function to convert decimal number to dB.
*
*-------------------------------------------------------------------------
*/

#include "common.h"
#include "gdata.h"
#include "mul.h"

/****************************************************************************
 ;
 ;   Function Name: ConvertToDB
 ;
 ;   Description:
 ;      Computes the dB value (10log10(l_xin)) of the input argument
 ;
 ;  Algorithm:
 ;      Let l_xin = x*2^n, where 0.25 < x <= 0.5, then
 ;      10*log10(l_xin) = (10/log2(10)) * log2(l_xin)
 ;                      = (10/log2(10)) * (log2(x*2^n))
 ;                      = (10/log2(10)) * (n + log2(x))
 ;                      = (10/log2(10))*n + (10/log2(10))*log2(x)
 ;
 ;      Aproximate (10/log2(10))*log2(x) by
 ;                      p(x) = 30.140153*x^3  -50.423888*x^2   +36.672731*x  -12.508209
 ;                           = ((30.140153*x -50.423888)*x + 36.672731)*x -12.508209
 ;
 ;      Accurate to +/- 0.008 dB
 ;
 ;   Input Arguments:
 ;      int32   l_xin   (Q32.0 format)
 ;
 ;  Return:
 ;      10*log10(l_xin) (in Q8.8 format)
 ;
 *****************************************************************************/
int16 ConvertToDB(int32 l_xin)
{
   int16 n;
   int32 l_Acc, l_Acc1;

   if (l_xin <= 0)
   {
      return (int16)0x8000;   // it shouldn't be <= 0
   }

   n = 16; // Offset initial n by 16.
   // Map input to range [+0.25, +0.5], where the approximation is optimized for.
   // Express l_xin as x*2^n, where 0.25<x<=0.5, represented in Q1.15 format
   while (l_xin >= 32768)
   {
      l_xin >>= 1;
      n++;
   }
   while (l_xin < 16384)
   {
      l_xin <<= 1;
      n--;
   }

   // Constants are multiplied by a factor to get the best resolution but at the
   // same time preventing overflow in the intermediate results.
   // Note that l_xin already has a factor of 2^16 in it.
   //l_Acc   = (7716*l_xin) - 845972661;        // 30.140153*2^8  = 7716;  50.423888*2^24 = 845972661
   //l_Acc   = (l_Acc >> 16)*l_xin + 615266328; // 36.672731*2^24 = 615266328
   //l_Acc   = (l_Acc >> 16)*l_xin -   209852925; // 12.508209*2^24 = 209852925

   MULS16(l_Acc, 7716, (int16)l_xin);
   l_Acc  -= 845972661; // 30.140153*2^8  = 7716;  50.423888*2^24 = 845972661
   MULS16(l_Acc, (int16)(l_Acc >> 16), (int16)l_xin);
   l_Acc  += 615266328; // 36.672731*2^24 = 615266328
   MULS16(l_Acc, (int16)(l_Acc >> 16), (int16)l_xin);
   l_Acc  -= 209852925; // 12.508209*2^24 = 209852925

   // Final answer is in 8.8 format so we need to downshift by a total of 16.
   l_Acc >>= 2;                        // shift by 2 to prevent overflow in next line.

   //l_Acc  += 12626113*n;                      // (10/log2(10))*2^22 = 12626113
   MULS32x16(l_Acc1, 12626113, n);            // (10/log2(10))*2^22 = 12626113

   l_Acc  += l_Acc1;
   l_Acc  = (l_Acc + (1 << 13)) >> 14;        // shift additional 14 for a total of 16 + round

   return (int16)l_Acc;
}
