/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: DDSnrFdqHandler.c
*
*   This file contains functions to calculate SNR & FDQ using data frames.
*
*-------------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// DDSnrFdqHandler.c
//
// History
//
// 11/07/2012 Vinjam: Code added to send the showtime snr (per Group) with out virtual noise using CMV INFO 85,
//                    during normal initialization mode.
//                    Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
//
// 25/07/2012 Vinjam: Extended "CMV INFO 85" for DS SNRps(Per carier group) with & without virtual noise
//                    Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
//
//14/02/2013 Vinjam : Resolved memory corruption by relocating the guca_SNRBuf_NE[ ] to "ONE_PORT_SHOW_ONLY_DATA" section.
//                    Grep for XDSLRTFW-726 Bug_All_VDSL2_All_MemoryCorruption
//
//18/12/2013 Kannan:  While applying 5 secs shine noise FDQ is being diverged and followed by link drop,
//                    So disable the FDQ if PLL phase error exceeds "gs_MaxPhaseErrorThreshold/2".
//                    While applying 5 secs shine noise PLL also getting diverged, in order to
//                    avoid PLL being diverged Max phase error is limited to "gs_MaxPhaseErrorThreshold = 0x10".
//                    This was done as a part of XDSLRTFW-1271.
//                    For Eg. Max Phase error = (PLL_PI_RADIANS/180); //1 degree;   = (25736)/180 = 142
//                    Grep for "XDSLRTFW-1104"
//
// 4/08/2014 Anantha Ramu: Code modified to resolve link drop in Non-ReTx mode, when 5 sec noise is applied.
//                   Code flow is made similar for ReTx & Non-ReTx modes.
//                          Grep for XDSLRTFW_1779.
//
// // 24/11/2015 Abu Rahman : PLL FDQ Adaptation Algorithm Modification.
//                            This new algorithm can handle Micro interruption, REIN,SHINE,PEIN and reasonable reference frequency drift and
//                            it is common for all VDSL2 modes.
//                            Grep for XDSLRTFW-2481
// ************************************************************************************************************


#include <string.h>
#include "common.h"
#include "gdata.h"
#include "cmv.h"
#include "fifo.h"
#include "IRI_Iof.h"
#include "show_iof.h"
#include "noiseacc.h"
#include "snr_b.h"
#include "psd_b.h"
#include "dsp_op.h"
#include "DDSnrFdqHandler.h"
#include "ComplexVectorMult.h"
#include "Bitload_support.h"
#include "CalcShowtimeSnrMargin.h"
#include "ConvertToDB.h"
#include "profile.h"
#include "LL_IOf.h"
#include "eoc.h"
#include "PGAHandler.h"

//XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start_End)
extern uint8 GetSnrForCarrierGroup(const uint16 us_CarrierGroup);

#ifdef INCLUDE_NON_CACHE_FUNCTIONS
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void LogSnrFdqUpdate(void)
*
*   This function logs SNR and FDQ coefficients for a selected tone.
*   This function is used for debug, and may be excluded from a target build,
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

#ifdef LOG_SNR_FDQ_UPDATE
void LogSnrFdqUpdate(void)
{


   /* HW based FDQ */
   gs_ToneSelect = TESTArray[TEST_LogToneIndex];

   if ((gs_ToneSelect >= gs_LeftChannel) && (gs_ToneSelect <= gs_RightChannel))
   {
      if (gs_LogIndex < SNR_FDQ_UPDATE_LOG_LENGTH)
      {
         gsa_SnrLog[gs_LogIndex] = gpsa_MeasuredSnrBuf[gs_ToneSelect];

         ReadFdqMant(&gsa_FdqMantissaLog[gs_LogIndex<<1], gs_ToneSelect, 1);
         ReadFdqExp(&gca_FdqExponentLog[gs_LogIndex], gs_ToneSelect, 1);
         gs_LogIndex++;
      }
   }


}
#endif   // LOG_SNR_FDQ_UPDATE

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DDSnrFdqHandler(void)
*
*   This function performs decision-directed SNR/FDQ calculation.
*
*   1) DD_SNRFDQ_INIT:
*      initialize variables to begin SNR/FDQ calculation called once
*      per each round of SNR/FDQ update.
*
*   2) DD_SNRFDQ_SETUP:
*      set up variables, buffers, and HW (e.g., RTV buffer) for SNR/FDQ
*      calculation of particular tone group, which is defined as
*      gs_AlgNumTonesToProcess.
*
*   3) DD_SNRFDQ_HW_READY:
*      wait until HW is ready (RTV buffer ready)
*
*   4) DD_SNRFDQ_CALC:
*      from count 0 to gs_AlgNumFramesToAccum-1, accumulate noise for
*      SNR calculation, and upon completion, submit background task
*      to compute SNR.
*      compute {E*conj(FFT)} for FDQ coefficients, and submit background
*      task to compute FDQ coefficients. repeat this process as much as
*      possible during noise accumulation. update FDQ coefficients in HW.
*
*   5) DD_SNRFDQ_RXPOWER:
*      measure and accumulate RX power for a given tone group.
*
*   6) DD_SNRFDQ_RXPOWER_WAIT:
*      wait until PSD calculation is done.

*   7) DD_SNRFDQ_NEXT_TONEGROUP:
*      update tone index for the next tone group. if all tones are updated,
*      go to DD_SNRFDQ_DONE so that other showtime task can be run
*      for instance, Bit-Swap algorithm.
*
*   8) DD_SNRFDQ_CALC_MARGIN:
*      update SNR margin and complete calculation of RX power.
*
*   9) DD_SNRFDQ_CALC_MARGIN_WAIT:
*      wait until SNR margin calculation is completed.
*
*   10) DD_SNR_DONE:
*      do nothing until state is reset to DD_SNRFDQ_INIT from outside
*      of this routine.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      gs_DDSnrFdqState: reset in RxDataFrameProcess()
*      gs_AlgNumFramesToAccum: # of training frames to accumulate noise
*      gs_AlgLog2NumFramesToAccum: log2 of gs_AlgNumFramesToAccum
*      gs_AlgNumTonesToProcess: # of tones to be updated at each
*
*-------------------------------------------------------------------------------
*/

void DDSnrFdqHandler(void)
{

   // XDSLRTFW-2481 (Start)
   if ((gus_NoiseDetected == NO_MI_OR_EXTRA_SHOWTIME_NOISE_DETECTED) && gft_EnableFdqUpdate) //XDSLRTFW-2562(Start_End)
   {
      gc_FdqUpdateState = 1;
      gus_FdqUpdateEnableCount++; //Reset the FDQ Enable count
   }
   else
   {
      gc_FdqUpdateState = 0;
      gus_FdqUpdateEnableCount = 0;  //FDQ Enable count after transition from FDQ Disable State or showtime entry state
   }

   //if there is a change in PLL, then change the FDQ State
   if (gc_Prev_FdqUpdateState != gc_FdqUpdateState)
   {
      //Wait for minimum 64 symbols to enable FDQ,
      //this is required while changing state from Disable to enable state
      if ((gc_FdqUpdateState == 1) && (gus_FdqUpdateEnableCount > 64))
      {
         EnableFDQAdapt();
         gc_Prev_FdqUpdateState = gc_FdqUpdateState;
      }
      else if (gc_FdqUpdateState == 0)
      {
         DisableFDQAdapt();
         gc_Prev_FdqUpdateState = gc_FdqUpdateState;
      }
   }
   // XDSLRTFW-2481 (End)

   switch (gs_DDSnrFdqState)
   {
   case DD_SNRFDQ_INIT:
      if ((gus_ModemOperationMode_Status & MODEM_OPERATION_MODE_VDSL2) &&
            (gt_RxOLRPMVars.uc_rxOLRPMState != L0_STEADY_STATE))
      {
         return;
      }
      // Enable FDQ adaptation on completion of OLRs
      if ((gt_RxOLRPMVars.uc_rxOLRPMState == L0_STEADY_STATE) && (!gft_SaveMarginPerTone))
      {
         if ((gsa_Optn2_AlgControl[2] & OPTN_SFDQUpdatesDisable) == 0)
         {
            gft_EnableFdqUpdate = TRUE;
         }

         if ((gsa_Optn2_AlgControl[2] & OPTN_SNRUpdatesDisable) == 0)
         {
            gft_EnableSnrUpdate = TRUE;
         }
      }
      // initialize variables for SNR/FDQ calculation
      gs_AlgLog2NumFramesToAccum = gt_DDSnrFdqConfig.s_AlgLog2NumFramesToAccum;
      gs_AlgNumFramesToAccum = (1 << gs_AlgLog2NumFramesToAccum);
      gft_ApplyConstGain = 1;

      // set up pointer for showtime SNR buffer
      // should not clear SNR buffer here since SNR buffer should contain
      // old values (for ME to read out) until they're replaced with new values
      // moreover, it contains MEDLEY SNR if SNR update is disabled
      gpsa_MeasuredSnrBuf = gsa_SnrBuf;

      gft_EnableGetRxTones = FALSE;

      // set starting channel index
      gs_CurrentRxBand = 0;
      gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];

      // reset power measurement variables
      gl_Pa = 0;

      //XDSLRTFW-3823 (start)
      // Validate NoOf Tone groups
      // atleast one start idx,(in this case end indx will be 8192) from startidx to end idx constant SNR offset will be added.
      // gta_SNROffset.s_NoOfToneGroups = 0 will disable SNR offset (by writing "0" value)
      if(gta_SNROffset.s_NoOfToneGroups >= 1)
      {
         if(gta_SNROffset.s_NoOfToneGroups > SNR_OFFSET_TONEGROUPS)
         {
            gta_SNROffset.s_NoOfToneGroups = SNR_OFFSET_TONEGROUPS;
         }
         // Copy to FW variable to be used for SNR manipulation
         int16 s_idx;
         for(s_idx = 0;s_idx < gta_SNROffset.s_NoOfToneGroups;s_idx++)
         {
            gt_SNROffsetBitLoad[s_idx+1].s_ToneGroupStartIdx = gta_SNROffset.ta_SNROffset[s_idx].s_ToneGroupStartIdx;
            gt_SNROffsetBitLoad[s_idx+1].s_ToneGroupSNROffset  = gta_SNROffset.ta_SNROffset[s_idx].s_ToneGroupSNROffset;
         }
       }
       gt_SNROffsetBitLoad[0].s_ToneGroupStartIdx = 0;
       gt_SNROffsetBitLoad[0].s_ToneGroupSNROffset  = 0;
       gt_SNROffsetBitLoad[gta_SNROffset.s_NoOfToneGroups+1].s_ToneGroupStartIdx = RX_MAX_NUM_TONES;
       gt_SNROffsetBitLoad[gta_SNROffset.s_NoOfToneGroups+1].s_ToneGroupSNROffset  = 0;
       gs_ToneGroupIdx = 0; //XDSLRTFW-3823 (end)


      gs_DDSnrFdqState = DD_SNRFDQ_SETUP;
      break;

   case DD_SNRFDQ_SETUP:

      // set ending channel index
      gs_RightChannel = gs_LeftChannel + gt_DDSnrFdqConfig.s_DDSnrFdqNumChannelsPerGroup - 1;
      if (gs_RightChannel > gsa_RxBandRightChannel[gs_CurrentRxBand])
      {
         gs_RightChannel = gsa_RxBandRightChannel[gs_CurrentRxBand];

      }

      //Note both of these two variables are used in showtime
      gs_AlgNumTonesToProcess = gs_RightChannel - gs_LeftChannel + 1;
      gs_NumChannelsPerGroup = gs_AlgNumTonesToProcess;


      // set tone offset to be stored in RTV0 and RTV1 buffers
      gs_RxToneOffset = gs_LeftChannel;
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetRxToneOffset);

      // set up RTV buffer
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetUpRTVForDDSnrFdqHandler);

      // transition to next state and wait until HW is ready
      gs_DDSnrFdqState = DD_SNRFDQ_HW_READY;
      gs_AlgHandlerCount = 0;
      break;

   case DD_SNRFDQ_HW_READY:
      if (gs_AlgHandlerCount < gs_RTVxCfgLatency)
      {
         gs_AlgHandlerCount++;
      }
      else
      {
         // transition to noise accumulation substate
         gs_DDSnrFdqState = DD_SNRFDQ_CALC;
         // set SNR update substate
         if (gft_EnableSnrUpdate)
         {
            gs_DDSnrState = DD_SNR_NOISE_ACC;
         }
         else
         {
            gs_DDSnrState = DD_SNR_DONE;
         }


         gs_AlgHandlerCount = 0;
      }
      break;

   case DD_SNRFDQ_CALC:
      // run SNR update subhandler
      DDSnrSubHandler();

      /* HW Based FDQ */
      // if SNR subhandler is done, proceed to next state
      if (gs_DDSnrState == DD_SNR_DONE)
      {

         guc_PsdCalcState = TRAINING_DONE;
         gs_DDSnrFdqState = DD_SNRFDQ_RXPOWER_WAIT;
      }

      break;


   case DD_SNRFDQ_RXPOWER_WAIT:
      if (guc_PsdCalcState == TRAINING_DONE)
      {
         // indicate SNR/FDQ update is done
         gs_DDSnrFdqState = DD_SNRFDQ_NEXT_TONEGROUP;
#ifdef LOG_SNR_FDQ_UPDATE
         if (TESTArray[TEST_Control] & TEST_LogSnrFdqControl)
         {
            LogSnrFdqUpdate();
         }
#endif // LOG_SNR_FDQ_UPDATE
      }
      break;

   case DD_SNRFDQ_NEXT_TONEGROUP:
      // if all the tones in this symbol has been processed
      if (gs_RightChannel == gsa_RxBandRightChannel[gs_NumOfRxBands-1])
      {
         // increment # of SNR updates during showtime
         if (gft_EnableSnrUpdate)
         {
            gs_NumSnrUpdate++;
         }

         //Note: if HW FDQ update is used, this count is not very meaningful,
         //but it indicates that the FDQ update is on.
         if (gft_EnableFdqUpdate)
         {
            gs_NumFdqUpdate++;
         }

         // go to state to compute SNR margin and complete RX power measurement
         gs_DDSnrFdqState = DD_SNRFDQ_CALC_MARGIN;
      }
      // else update for next tone group
      else
      {
         if (gs_RightChannel < gsa_RxBandRightChannel[gs_CurrentRxBand])
         {
            gs_LeftChannel = gs_RightChannel+1;
         }
         else
         {
            gs_CurrentRxBand++;
            gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];
         }
         gs_DDSnrFdqState = DD_SNRFDQ_SETUP;
      }
      break;

   case DD_SNRFDQ_CALC_MARGIN:

      // If SNR updates are enabled, then also update the average/minimum SNR margin
      if (gft_EnableSnrUpdate)
      {
         guc_SnrCalcState = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgCalcShowtimeSnrMargin);
      }
      else
      {
         guc_SnrCalcState = TRAINING_DONE;
      }


      gs_DDSnrFdqState = DD_SNRFDQ_CALC_MARGIN_WAIT;
      break;

   case DD_SNRFDQ_CALC_MARGIN_WAIT:
      if (guc_SnrCalcState == TRAINING_DONE)
      {

         // XDSLRTFW-3280 - PLL improvement / pilot tone selection improvement
         // Perform a metric comparison (which might result in pilot tone switching)
         // if the showtime pilot tone switching and the multi-pilot feature is enabled
         if((gt_PilotConfig.s_PilotToneControl & SWITCH_PILOT_IN_SHOWTIME) && (gft_EnableShowPllToneSwitch))
         {
            ComparePilotMetric();
         }


         // XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
         // At least 1 SNR update is complete since last SRA
         gft_SNRUpdated_PostSRA = TRUE;
         // XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]
         // indicate SNR/FDQ update is done
         gs_DDSnrFdqState = DD_SNRFDQ_DONE;
      }
      break;

   case DD_SNRFDQ_DONE:
      break;
   }

}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void DDSnrSubHandler(void)
*
*   This function performs SNR updates for a given tone group.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      gs_DDSnrState: DD SNR update substate
*
*-------------------------------------------------------------------------------
*/


void DDSnrSubHandler(void)
{

   /* HW Based SNR */
   switch (gs_DDSnrState)
   {

   case DD_SNR_NOISE_ACC:
      if (gs_AlgHandlerCount == 0 )
      {
         guc_SnrCalcState = TRAINING_IN_PROGRESS;

         // Clear RxAccumBuf for DD
         memset(gpla_RxAccumBuf, 0, sizeof(int32)*gt_DDSnrFdqConfig.s_DDSnrFdqNumChannelsPerGroup*2); // Need 2 32-bit words for holding 1 tone data
         // Reset the Noise Accumulator Buffer and trigger the SNR in hardware
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, ResetNoisePowerBuffer);
      }
      else if (gs_AlgHandlerCount == gs_AlgNumFramesToAccum+1)
      {
         // queue background task to calculate SNR
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgSnrCalc);
         gs_DDSnrState = DD_SNR_CALC;
      }
      //In the next symbol, this function will be run due to the RX synch symbol reception
      //we should disable the noise accumulation for one symbol
      if(guc_SnrCalcState == TRAINING_IN_PROGRESS)
      {
         if(gs_RxPMDFrameCount == (RX_DATA_SYMBOLS_PER_SUPERFRAME-3))
         {
            AddFunctionToFifo(gp_RxLoadingFunctionFifo, DisableNPRWrite);
         }
      }

      break;

   case DD_SNR_CALC:
      // wait for background SNR calculation is done
      if (guc_SnrCalcState == TRAINING_DONE)
      {
         gs_DDSnrState = DD_SNR_DONE;
      }
      break;

   case DD_SNR_DONE:
      break;
   }

}

#endif // INCLUDE_NON_CACHE_FUNCTIONS

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BgDDFdqAdapt(void)
*
*   This function performs FDQ adaptation using the sign-sign LMS (SSLMS)
*   algorithm during showtime according to the following formula:
*
*   C(i,k+1) = C(i,k) - slew * sign{E(i,k) * conj(X(i,k))}
*
*   where   C(i,k) = complex-valued FDQ tap of tone i at time k
*         slew   = step size
*         E(i,k) = complex-valued QAM decision error
*         X(i,k) = complex-valued FFT output
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

#ifdef INCLUDE_CACHE_FUNCTIONS

#ifndef HW_SNR_FDQ
void BgDDFdqAdapt(void)
{
   int16 i, j, j2;
   int16 *psa_ErrCorr;
   int16 s_err_corr_real, s_err_corr_imag;
   int32 l_mant_real, l_mant_imag;
   int16 s_exp_real, s_exp_imag, s_exp_min, s_exp, s_final_exp;
   int16 s_FDQ_coef_real, s_FDQ_coef_imag;
   uint8 uc_FDQ_exp;

   psa_ErrCorr = gpsa_FftErrCorrBuf;

   for (i=gs_LeftChannel; i<=gs_RightChannel; i++)
   {

      j = i - gs_LeftChannel;
      j2 = j<<1;

      // read FDQ exponent from inactive table
      uc_FDQ_exp = ghpuca_RxExtGains_Inactive[i];
      s_FDQ_coef_real = gsa_pre_FDQ_coef[j2];
      s_FDQ_coef_imag = gsa_pre_FDQ_coef[j2+1];

      s_err_corr_real = *psa_ErrCorr++;
      s_err_corr_imag = *psa_ErrCorr++;

      // do not adapt FDQ for tones with g=0
      if (ghpsa_RxFineGains_Inactive[i] == NEG_INFINITY_DB)
      {
         continue;
      }

      // extract sign bit
      s_err_corr_real = (s_err_corr_real<0)? -gs_DDFdqAdaptSlew: gs_DDFdqAdaptSlew;
      s_err_corr_imag = (s_err_corr_imag<0)? -gs_DDFdqAdaptSlew: gs_DDFdqAdaptSlew;

      // add sign bit of gradient to LSB
      l_mant_real = (int32)s_FDQ_coef_real - s_err_corr_real;
      l_mant_imag = (int32)s_FDQ_coef_imag - s_err_corr_imag;

      // due to the HW limit, the imaginary part cannot be -32768
      // double the step size if the imaginary number hits -32768
      // -16384 should be avoided as well since it can be -32768 during normalization
      if ((l_mant_imag == -32768) || (l_mant_imag == -16384))
      {
         l_mant_real -= s_err_corr_real;
         l_mant_imag -= s_err_corr_imag;
      }

      // normalize FDQ coeffs
      s_exp_real = s_exp_imag = 31;

      if (l_mant_real != 0)
      {
         s_exp_real = norm_l(l_mant_real);
      }

      if (l_mant_imag != 0)
      {
         s_exp_imag = norm_l(l_mant_imag);
      }

      // take the minimum of the two
      s_exp_min = (s_exp_real < s_exp_imag)? s_exp_real: s_exp_imag;

      // positive means right shift and negative means left shift
      s_exp = (32-FDQ_MANTISSA_WORDLENGTH) - s_exp_min;

      // update the final exponent
      s_final_exp = (int16)uc_FDQ_exp + s_exp;
      uc_FDQ_exp = (uint8)s_final_exp;

      // since the final exponent has to be positive, increase the number of right shift to make the final exponent to be 0
      if (s_final_exp < 0)
      {
         s_exp -= s_final_exp;
         uc_FDQ_exp = 0;
      }

      // reduce the number of right shift to makethe final exponent not exceeding the maximum
      if (s_final_exp > FDQ_EXPONENT_MAX)
      {
         s_exp -= (s_final_exp - FDQ_EXPONENT_MAX);
         uc_FDQ_exp = FDQ_EXPONENT_MAX;
      }

      // apply the shift to the mantissa
      if (s_exp > 0)
      {
         s_FDQ_coef_real = sature16(round(l_mant_real, s_exp));
         s_FDQ_coef_imag = sature16(round(l_mant_imag, s_exp));
      }
      else
      {
         s_exp = -s_exp;
         s_FDQ_coef_real = sature16(l_mant_real<<s_exp);
         s_FDQ_coef_imag = sature16(l_mant_imag<<s_exp);
      }

      // due to the HW limit, the imaginary part cannot be 0x8000
      // this is just a precautionary check
      if (s_FDQ_coef_imag <= -32768)
      {
         s_FDQ_coef_imag = -32767;
      }

      gsa_pre_FDQ_coef[j2] = s_FDQ_coef_real;
      gsa_pre_FDQ_coef[j2+1] = s_FDQ_coef_imag;

      // write FDQ exponent to inactive table
      guca_pre_FDQ_exp[j] = uc_FDQ_exp;
      ghpuca_RxExtGains_Inactive[i] = uc_FDQ_exp;
   }

   guc_FdqTrainingState = TRAINING_DONE;
}

#endif //#ifndef HW_SNR_FDQ

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BgDDRxPowerCalc(void)
*
*      This function accumulates the rx power for the current FFT output power
*    signal created by inputting the FFT Ouptut to the NoiseAcc function.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

#ifndef HW_VECPOW
void BgDDRxPowerCalc(void)
{
   int32 *pla_NoisePower;

   int16 *psa_PsdTemp;

   psa_PsdTemp = gpsa_FftErrCorrBuf; // this buffer temporarily holds the PSD(in dB) for the gs_AlgNumTonesToProcess tones

   /* convert 48 bits accumulated value to 32 bits; */
   /* shift is 2 extra to match NoiseAcc() */
   pla_NoisePower = gpla_RxAccumBuf; // Overwrite input buffer with 32-bit output.
   RoundNoiseAccum(gpla_RxAccumBuf, pla_NoisePower, gs_LeftChannel,
                   gs_RightChannel, 1);

   pla_NoisePower = gpla_RxAccumBuf;
   PsdCalc(pla_NoisePower, psa_PsdTemp, 0, (int16)(gs_AlgNumTonesToProcess-1));

   guc_PsdCalcState = TRAINING_DONE;
}

#endif //#ifdef HW_VECPOW

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BgCalcShowtimeSnrMargin(void)
*
*   This function calculates SNR margin in background.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void BgCalcShowtimeSnrMargin(void)
{
   // calculate AVG/MIN margin for updated SNR
   CalcShowtimeSnrMargin(1, &gt_LineStatusDS.s_SnrMargin, &gs_RxMinMargin, &gs_RxMinMarginTone);

   //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start)
   //Code added to send the showtime snr (per Group) with out virtual noise
   //using CMV INFO 85 during normal initialization mode.
   //Extended "CMV INFO 85" for DS SNRps(Per carier group) with & without virtual noise
   //XDSLRTFW-726 Bug_All_VDSL2_All_MemoryCorruption (Start_End)
   if(guc_PortMode == 0) //valid for single port mode
   {
      int j;
      // SNR, 512 byte
      for(j=0; j<512; j++)
      {
         // Downsample the SNR buffer and convert to messaging format
         guca_SNRBuf_NE[j] = GetSnrForCarrierGroup(j);
      }
   }
   //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (End)

   // calculate the lom failure here, comparing the updated average updated snrm
   // with the negotiated one. Comparisons are done in the 0.1 dB format so
   // the received SNRMargin is converted to the 0.1 dB format by
   // x_in_0.1dB = (x_in_q8.8 * 10) >> 8

   if (gt_LineStatusDS.s_SnrMargin < gt_SnrMgnConfig.s_MINSNRMds)
   {
      gt_TxIbData.uc_lom_fail = PRESENT;
   }
   else
   {
      gt_TxIbData.uc_lom_fail = TERMINATED;
   }

   // override the lom failure for test purposes
//   if (gs_lom_override_cnt > 0)
//   {
//      gt_TxIbData.uc_lom_fail = PRESENT;
//      if (gs_lom_override_cnt > 0)
//         gs_lom_override_cnt--;
//   }
#ifdef DEBUG_TRAIL
   // Log LOM events
   if (gt_debugTrailControl.s_ShowtimeEventControl & DEBUG_TRAIL_SHOW_EVENTS_NE_LOM_ENABLE)
   {
      int16 s_thresh, s_transition;

      s_thresh = gt_SnrMgnConfig.s_TARSNRMds - gs_SnrMarginLomThresh;

//      if (gs_lom_override_cnt > 0)
//      {
//         gt_TxIbData.uc_lom_fail = PRESENT;
//         s_thresh = (gt_LineStatusDS.s_SnrMargin + 10);
//         if (gs_lom_override_cnt > 0)
//            gs_lom_override_cnt--;
//      }

      if(s_thresh < 0)
      {
         s_thresh = 0;
      }

      s_transition = 0;
      if((gt_LineStatusDS.s_SnrMargin < s_thresh) &&
            ((gt_debugTrailStatus.s_debugTrailPrevEvent & DEBUG_TRAIL_SHOW_PREEVENT_NE_LOM_ON) == 0))
      {
         s_transition = (int16)DEBUG_TRAIL_SHOW_EVENTS_NE_LOM_DEF_ON;
      }
      else if((gt_LineStatusDS.s_SnrMargin >= s_thresh) &&
              (gt_debugTrailStatus.s_debugTrailPrevEvent & DEBUG_TRAIL_SHOW_PREEVENT_NE_LOM_ON))
      {
         s_transition = (int16)DEBUG_TRAIL_SHOW_EVENTS_NE_LOM_DEF_OFF;
      }

      if(s_transition != 0)
      {
         int16 TempVar;

         // Toggle event between ON and OFF!
         TempVar = (gt_debugTrailStatus.s_debugTrailPrevEvent & DEBUG_TRAIL_SHOW_PREEVENT_NE_LOM_ON);
         TempVar ^= DEBUG_TRAIL_SHOW_PREEVENT_NE_LOM_ON;
         gt_debugTrailStatus.s_debugTrailPrevEvent &= (~DEBUG_TRAIL_SHOW_PREEVENT_NE_LOM_ON);
         gt_debugTrailStatus.s_debugTrailPrevEvent |= TempVar;

         // Log LOM change into ShowTime Debug Trail
         DebugTrail1(5,DEBUG_TRAIL_SHOWTIME_EVENTS_ENABLE,0,
                     (int16)s_transition,
                     (int16)gl_RxSymbolCount,
                     (int16)(gl_RxSymbolCount>>16),
                     (int16)gt_LineStatusDS.s_SnrMargin,
                     (int16) gs_RxMinMargin);
      }
   }
#endif // DEBUG_TRAIL

   guc_SnrCalcState = TRAINING_DONE;
}

// XDSLRTFW-3280 Start - PLL improvement / pilot tone selection improvement
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ComparePilotMetric(void)
*
*   This function Compares the pilot tone metric of all pilot tones (called after each SNR update)
*   In a first attempt we mark the pilot tone with the best metric in gt_PilotConfig.te_BestPTArrayIdx
*   If the usage of this best pilot tone is avoided because it creates a too big phase error (e.g. due to instable SNR), we
*   will mark the pilot tone with the second best metric for usage in gt_PilotConfig.te_BestPTArrayIdx.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables: gt_PilotConfig.te_BestPTArrayIdx
*
*-------------------------------------------------------------------------------
*/

void ComparePilotMetric(void)
{
   PT_ArrIdx_te t_UsedArrayIdx = gt_PilotConfig.te_UsedPTArrayIdx;
   //                    Slot for:   3rd best      , 2nd best     , best
   PT_ArrIdx_te t_SortArrayIdx[3] = {PT_ARRAY_IDX_0,PT_ARRAY_IDX_1,PT_ARRAY_IDX_2};
   PT_ArrIdx_te t_tmp;
   PT_ArrIdx_te t_newIdx;
   int32 l_best_metric_3per_thresh;

   // calculation of 3percent threshold
   l_best_metric_3per_thresh = gt_PilotConfig.ta_PilotTones[t_UsedArrayIdx].l_Metric + (gt_PilotConfig.ta_PilotTones[t_UsedArrayIdx].l_Metric>>5);

   // sort BestArrayIdx from worst (element 0) to best index (element 2)
   if (gt_PilotConfig.ta_PilotTones[0].l_Metric > gt_PilotConfig.ta_PilotTones[2].l_Metric)
   {  //if a > c, swap a & c
      t_tmp = t_SortArrayIdx[0];
      t_SortArrayIdx[0] = t_SortArrayIdx[2];
      t_SortArrayIdx[2] = t_tmp;
   }
   if (gt_PilotConfig.ta_PilotTones[0].l_Metric > gt_PilotConfig.ta_PilotTones[1].l_Metric)
   {  // if a > b, swap a & b
      t_tmp = t_SortArrayIdx[0];
      t_SortArrayIdx[0] = t_SortArrayIdx[1];
      t_SortArrayIdx[1] = t_tmp;
   }
   if (gt_PilotConfig.ta_PilotTones[1].l_Metric > gt_PilotConfig.ta_PilotTones[2].l_Metric)
   {  // if b > c, swap b & c
      t_tmp = t_SortArrayIdx[1];
      t_SortArrayIdx[2] = t_SortArrayIdx[1];
      t_SortArrayIdx[1] = t_tmp;
   }

   // Now the t_SortArrayIdx[] should look like
   // t_SortArrayIdx[0] worst PT Idx
   // t_SortArrayIdx[1] 2nd best PT Idx
   // t_SortArrayIdx[2] best PT Idx
   t_newIdx = t_SortArrayIdx[2];
   if(gft_Try2ndBestPTIdx == TRUE)
   {
      // Try 2nd best Pilot Tone array Index because best failed in SwithToBetterPilotTone()
      t_newIdx = t_SortArrayIdx[1];
      gft_Try2ndBestPTIdx = FALSE; // XDSLRTFW-3702 (Start/End) Need to reset this variable to avoid being stuck with second best pilot tone
   }


   // if we found a metric which is better than 3% we mark this for usage as the new pilot tone
   if ((t_newIdx != t_UsedArrayIdx) &&
       (gt_PilotConfig.ta_PilotTones[t_newIdx].l_Metric > l_best_metric_3per_thresh) &&
       (gs_PLLAvgHandlerState == PLL_AVG_IDLE))        //make sure that we are not midst the calculation of a new pilot tone
   {
      gt_PilotConfig.te_BestPTArrayIdx = t_newIdx;
   }
}
// XDSLRTFW-3280 End
#endif  //INCLUDE_CACHE_FUNCTIONS

