/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), Lantiq GmbH. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*
*
*   DetecDetectReverbSegueG9935ST.c
*
*   This file contains function to detect Sync Symbol in Showtime
*   at VDSL Vectoring mode
*
*-------------------------------------------------------------------------
*/

#include "gdata.h"
#include "vdsl_state.h"
#include "typedef.h"


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : DetectReverbSegueG9935ST.c
 *
 *  Description:    Detects if the signal R-REVERB or R-SEGUE is present
 *  in the current symbol by examining a window containing N tones.
 *
 *  Prototype:  int16 DetectReverbSegue_G9935_ST(int16 s_NumTones)
 *
 *  Input Arguments:
 *
 *      s_NumTones : the number of tones in the buffer ( not used in current code!)
 *
 *  Return:
 *      =  0, REVERB detected
 *      =  1, SEGUE detected
 *
 *  Global Variables Used:
 *         gsa_SyncSymConstelForOlr  : Buffer for capturing 30 showtime OLR related constellation points ( used as input)
 *         gul_SyncSybmolRef  : Store Reference SEGUE constellation points for Showtime Sync Symbol.
 *                                   Two bits for each constellation, only sign bit.
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 DetectReverbSegueG9935ST(int16 s_NumTones)
{
   // complete function need to be modified !
   int16 i;
   int16 s_NumReverb, s_NumSegue;
   int16 s_real, s_imag;
   int16 s_real_ref, s_imag_ref;
   // These indices are for HW_DECIM = 2nd tone
   // since with subsampling = 2nd Tone we will have after 10n+1 th tone
   // Actual tone number   (10*n+1)        :   1  3  5  7  9  11 13 15 17 19 21 23 25 27 29 31 33 35 37 39 41 43 45 47 49 51 53 55 57 59 61 ..
   // Tone(cmpx) possition in the array    :   0  1  2  3  4  5  6  7  8  9  10 11 12 13 14 15 16 17 18 19 20 21 22 23 24 25 26 27 28 29 30 ..
   // FLAG/PROBE tone (G993.5)             :   F  P  P  F  P  F  P  P  F  P  F  P  P  F  P  F  P  P  F    P  F  P  P  F  P  F  P  P  F  P  F  ..
   // the following indices will contain the flag tones
   int16 s_index[13] = {0,3,5,8,10,13,15,18,20,23,25,28,30};

   //Tone by Tone based detection algorithm
   s_NumReverb = 0;
   s_NumSegue = 0;

   if (gft_FirstRxSyncSybmolInShowtime == TRUE)
   {
      gul_SyncSybmolRef = 0;

      for (i = 0; i < 6 ; i++)      // only 6 flag tones are used to determine OLR
      {

         s_real_ref =  (gsa_SyncSymConstelForOlr[2*s_index[i]] >> 15) & 0x1;
         s_imag_ref =  (gsa_SyncSymConstelForOlr[2*s_index[i] + 1] >> 15) & 0x1;

         gul_SyncSybmolRef |= (uint32)(s_real_ref << (2*i));
         gul_SyncSybmolRef |= (uint32)(s_imag_ref << (2*i +1));
      }

      gft_FirstRxSyncSybmolInShowtime = FALSE;

      return(SEGUE); //At the beginning of showtime, the first sync symbol transmitted shall be modulated by a sync frame of all ones (SEGUE) Section 10.5.3/G993.2.

   }

   for(i=0; i<6; i++)
   {

      s_real_ref = (int16)((gul_SyncSybmolRef >> (2*i)) & 0x1);
      s_imag_ref = (int16)((gul_SyncSybmolRef >> (2*i + 1)) & 0x1);

      s_real =  (gsa_SyncSymConstelForOlr[2*s_index[i]] >> 15) & 0x1;
      s_imag =  (gsa_SyncSymConstelForOlr[2*s_index[i] + 1] >> 15) & 0x1;


      //At the beginning of showtime, the first sync symbol transmitted shall be modulated by a sync frame of all ones (SEGUE) Section 10.5.3/G993.2.
      if (( s_real == s_real_ref) && (s_imag == s_imag_ref))
      {
         s_NumSegue++;
      }
      else if (( s_real != s_real_ref) && (s_imag != s_imag_ref))
      {
         s_NumReverb++;
      }

   }
   if (s_NumReverb > s_NumSegue)
   {
      return(REVERB);
   }
   else if (s_NumReverb < s_NumSegue)
   {
      return(SEGUE);
   }
   else
   {
      return(NEITHER);
   }

}

//XDSLRTFW-2875 (Start)
int16 DetectReverbSegueVectorFriendlyST (int16 *psa_RxToneBuf, int16 s_NumTones)
{
   int16 i;
   int16 s_NumReverb, s_NumSegue;
   int16 s_real, s_imag;

   // Actual tone number   (10*n+1)        :   1  3  5  7  9  11 13 15 17 19 21 23 25 27 29 31 33 35 37 39 41 43 45 47 49 51 53 55 57 59 61 ..
   // Tone(cmpx) possition in the array    :   0  2  4  6  8  10 12 14 16 18 20 22 24 26 28 30 32 34 36 38 40 42 44 46 48 50 52 54 56 58 60 ..
   // FLAG/PROBE tone (G993.5)             :   F  P  P  F  P  F  P  P  F  P  F  P  P  F  P  F  P  P  F  P  F  P  P  F  P  F  P  P  F  P  F  ..
   // the following indices will contain the flag tones
   //int16 s_index[13] = {1,7,11,17,21,27,31,37,41,47,51,57,61};
   int16 s_index[13] = {0,6,10,16,20,26,30,36,40,46,50,56,60};

   //Tone by Tone based detection algorithm
   s_NumReverb = 0;
   s_NumSegue = 0;

   // only 6 flag tones are used to determine OLR. Max 6 flag tones are available out of the max
   // 30 tones captured for OLR detection. See data above.
   for(i=0; i<6; i++)
   {
      if (s_index[i] < s_NumTones)
      {
         s_real = psa_RxToneBuf[2*s_index[i]];      //Constell point is rotated to the right quadrant
         s_imag = psa_RxToneBuf[2*s_index[i] + 1];  //Constell point is rotated to the right quadrant

         if((s_real > 0) && (s_imag > 0))
         {
            s_NumReverb++;
         }
         else if((s_real < 0) && (s_imag < 0))
         {
            s_NumSegue++;
         }
      }
   }

   if (s_NumReverb > s_NumSegue)
   {
      return(REVERB);
   }
   else if (s_NumReverb < s_NumSegue)
   {
      return(SEGUE);
   }
   else
   {
      return(NEITHER);
   }
}
//XDSLRTFW-2875 (End)
