/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   FdqHandler.c
 *
 *   This file contains the routine that calculates FDQ.
 *
 *------------------------------------------------------------------------
 */
#include <string.h>
#include "common.h"
#include "gdata.h"
#include "FdqHandler.h"
#include "fifo.h"
#include "fdq.h"
#include "accum32.h"
#include "IRI_Iof.h"
#include "ghs.h"
#include "cmv.h"

extern void ResetFdQBuffer(void);

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : FdqHandler
 *
 *  Description:  This function is used to calculate the FDQ.
 *
 *  Prototype:  void FdqHandler(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *
 *  Global Variables Used:
 *      gs_AlgHandlerState       - (I/O) state of FDQ calculation state machine
 *      gpla_RxAccumBuf[]      - (I/O) accumulated frames
 *
 *
 *  Before FdqHandler() is called for the first time, the following initialization
 *  needs to be performed:
 *
 *      1) gs_AlgHandlerState = FDQ_TRAIN_INIT.
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
void FdqHandler(void)
{
   switch(gs_AlgHandlerState)
   {

   case FDQ_TRAIN_INIT:
      // Initialize variables for FDQ calculation
      gft_EnableGetRxTones = TRUE;

      // For VRx518 the RTV is twice, i.e. for 1024 tones.
      // Note: Normally RTV size should be considered!
      //       The code should also handle on the fly configured values for debuging!
      gs_NumChannelsPerGroup = gt_FdqConfig.s_NumChannelsPerGroup;

      //Set the number of symbols used for FDQ training
      gs_AlgLog2NumFramesToAccum = gt_FdqConfig.s_AlgLog2NumFramesToAccum;
      gs_AlgNumFramesToAccum = (1 << gs_AlgLog2NumFramesToAccum);

      if ( (gs_RxMedleyStateFlag == FALSE) ||
           (guc_ExchangePhaseRxNumBytesPerSymbol == 1) ||
           (gs_FDQTrainTypeCntrlFlag & FDQ_TRAIN_TYPE_MEDLEY_ONEPASS) )
      {
         if(gs_FDQTrainTypeCntrlFlag & FDQ_TRAIN_TYPE_MEDLEY_ONEPASS)
         {
            // Set to MEDLEY_FDQ_TRAIN_ONEPASS to indicate the onepass FDQ training.
            // Note: Normally used during Medley!
            gs_FDQTrainType = MEDLEY_FDQ_TRAIN_ONEPASS;
         }
         else
         {
            // Set to COARSE_FDQ_TRAIN to indicate the coarse FDQ training
            // Note: Normally used during ChannelDiscovery and TransceiverTraining!
            gs_FDQTrainType = COARSE_FDQ_TRAIN;
            if (gs_FDQTrainTypeCntrlFlag & FDQ_TRAIN_TYPE_FINE_TUNE_ONLY)
            {
               gs_FDQTrainType  = FINE_TUNE_FDQ_TRAIN;
            }
         }

         // Set RTV flag to read FFT output
         gs_RtvSelect = FFT_OUTPUT;
      }
      else //if(gs_MedleyStateFlag == TRUE)
      {
         //At the Analysis phase, set to MEDLEY_FDQ_TRAIN to indicate the FDQ training using Medley signal
         gs_FDQTrainType = MEDLEY_FDQ_TRAIN;
         // Set flag to read SFDQ output
         gs_RtvSelect = SFDQ_OUTPUT;
      }

      //Set the first frequency band to process
      gs_CurrentRxBand = 0;
      gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];

      //go to next substate
      gs_AlgHandlerState = FDQ_SETUP;
      break;

   case FDQ_SETUP:
      // Setup variables for FDQ calculation on the current tone group

      // Update the right channel
      // Note: ReadFDQ(void) is using "us_length = gs_RightChannel - gs_LeftChannel + 1;"
      //       Therefore gs_RightChannel-gs_LeftChannel <=(gs_NumChannelsPerGroup-1)
      gs_RightChannel = (gs_LeftChannel + (gs_NumChannelsPerGroup - 1));
      if(gs_RightChannel > gsa_RxBandRightChannel[gs_CurrentRxBand])
      {
         gs_RightChannel = gsa_RxBandRightChannel[gs_CurrentRxBand];
      }
      gs_NumOfTonesInBand = ((gs_RightChannel- gs_LeftChannel) + 1);

      // Clear RxAccumBuf for FDQ accumulation
      memset(gpla_RxAccumBuf, 0, sizeof(int32)*gs_NumOfTonesInBand*2);
      gs_AlgHandlerCount = 0;

      // Set offset for loading gpsa_RxToneBuf from RTV buffer 0 HW
      gs_RxToneOffset = gs_LeftChannel;
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetRxToneOffset);

// This was debug code and not needed!
//      if (gs_FDQTrainType == COARSE_FDQ_TRAIN)
//      {
//         AddFunctionToFifo(gp_RxLoadingFunctionFifo, ResetFdQBuffer);
//      }
      // Transition to next substate
      gs_AlgHandlerState = FDQ_WAIT_FOR_HW_SET;
      break;

   case FDQ_WAIT_FOR_HW_SET:

      // Wait for RTV buffer 0 HW settings to take effect
      if (gs_AlgHandlerCount < gs_RTVxCfgLatency)
      {
         gs_AlgHandlerCount++;
      }
      else
      {
         // Transition to accumulation substate

         // Even for COARSE_FDQ_TRAIN, use RotateDataTones, so that data-tones can be accumulated
         // with sufficient accuracy to approximate their magnitudes. If training is one-pass
         // however, this is not necessary.
         if (gs_FDQTrainType == MEDLEY_FDQ_TRAIN_ONEPASS)
         {
            guc_FdqTrainingState = TRAINING_DONE;
            gs_AlgHandlerState = FDQ_ACCUM;
         }
         else if (gs_FDQTrainType == COARSE_FDQ_TRAIN)
         {
            gs_AlgHandlerState = FDQ_PRE_ACCUM;
         }
         else
         {
            // Update global FDQ arrays since needed for FDQ refinement of message carrying tones
            AddFunctionToFifo(gp_RxLoadingFunctionFifo, ReadFDQ);
            gs_AlgHandlerState = FDQ_PRE_ACCUM;
         }

         // Clear counter
         gs_AlgHandlerCount = 0;
      }
      break;


   case FDQ_PRE_ACCUM:
      // XDSLRTFW-1111
      // Remove Tone9 marker for vectoring before Medley, i.e. for ChannelDiscovery and TransceiverTraining!
      // Note: XDSLRTFW-1111 and XDSLRTFW-2666 cannot be done merged together at state "FDQ_ACCUM".
      //
      // Note: For multiple joining line scenarios, i.e. all lines are training at the same time, the SNR on tone 10n+9 was always bad.
      //       But all other tones showed a kind of noise free SNR. Reason is that all lines are synchronized in vectoring
      //       and are sending a constant pattern (i.e. IDLE-flags on odd tones and 00-constelation on even tones) + the PRBS in reset mode.
      //       Therefore the xtalk leads to a positive superposition.
      //       On tone 10n+9 this is not the case for CO's starting with different idx in the pilot-sequences. This idx is modulated in the
      //       symbols 256 to 26 symbols of a superframe and seen as real xtalk.
      //       Therefore these symbols must be excluded during accumulation for SNR and FDQ.
      //
      //       Usable symbols for accumulation are:
      //       => 26 < sym_idx < 256
      //       => 27 <= sym_idx <= 255
      //
      //       To be on the save side a guard of 2 additional symbol was selected, i.e. on upper and lower bound side.
      //       => 28 < sym_idx < 254
      //       => 29 <= sym_idx <= 253
      //
      // The if-case "(gs_RxMedleyStateFlag == FALSE)" is intended security code.
      // The flag "EN_CHDIS_SKIP_TONE9_MARKER_SYM_FDQSNR" is set and reset only correct for vectoring.
      // But the code is common for better testing with non-vectoring.
      // Even when the flag "EN_CHDIS_SKIP_TONE9_MARKER_SYM_FDQSNR" is wrongly set in gs_MedleyFrameSynchEnableFlag for non-vectoring
      // it will not lead to any fault behavior or timing penalty during medley.
      // Note: In case of low code memory, the if-case can be removed, when the functionality is not usable for non-vectoring.
      if (!((gs_MedleyFrameSynchEnableFlag & EN_CHDIS_SKIP_TONE9_MARKER_SYM_FDQ) &&
           (gs_RxMedleyStateFlag == FALSE) &&
           ((gs_RxFrmCnt < 29) || (gs_RxFrmCnt >= (RX_DATA_SYMBOLS_PER_SUPERFRAME-2)))) )
      {
         // Prohibit copying Iridia buffer to gpsa_RxToneBuf.
         gft_EnableGetRxTones = FALSE;
         guc_FdqTrainingState = TRAINING_IN_PROGRESS;

         //Rotate message carring tones
         AddFunctionToBkgdFifo((PtrToBkgdFunc)RotateDataTones);
         gs_AlgHandlerState = FDQ_ACCUM;
      }
      break;


   case FDQ_ACCUM:
      // Accum representative frame for FDQ calculation
      if (guc_FdqTrainingState == TRAINING_DONE)
      {
         uint8 uc_DoAccu;

         uc_DoAccu = TRUE;
         // XDSLRTFW-2666 (Start-End)
         // Remove Tone9 marker for vectoring when "MEDLEY_FDQ_TRAIN_ONEPASS" fdqtype
         // is used before Medley!
         // Note: For "MEDLEY_FDQ_TRAIN_ONEPASS" a known not changing signal is needed (SOC-IDLE),
         //       because a reference based training (known constellation) is done.
         //       Due to the accumulation over 128/256 tones, it is not allowed to have
         //       a wrong/changing constellation during this phase. This would be the case for tone 10n+9
         //       in vectoring, when these symbols would not be excluded. On tone 10n+9 the pilot-sequence-idx marker
         //       is modulated, i.e. on symbols 256 to 26.
         //
         //       Usable symbols for accumulation are:
         //       => 26 < sym_idx < 256
         //       => 27 <= sym_idx <= 255

         //       To be on the save side a guard of 1 additional symbol was selected, i.e. on upper- and lower-bound side.
         if ((gft_DSVectoringEnabled == TRUE) &&
             (gs_RxMedleyStateFlag == FALSE) &&
             (gs_FDQTrainType == MEDLEY_FDQ_TRAIN_ONEPASS) &&
             ((gs_RxFrmCnt < 28) || (gs_RxFrmCnt > 254)))
         {
            uc_DoAccu = FALSE;
         }

         if (uc_DoAccu)
         {
            gs_AlgHandlerCount++;
            Accum16to32(gpla_RxAccumBuf, 0, gpsa_RxToneBuf, 0, (int16)(gs_NumOfTonesInBand<<1));
         }
         gft_EnableGetRxTones = TRUE; // Restore copying Iridia buffer to gpsa_RxToneBuf.

         if (gs_AlgHandlerCount == gs_AlgNumFramesToAccum)
         {
            // Compute average of the accumulated frames in background
            guc_FdqTrainingState = TRAINING_IN_PROGRESS;
            AddFunctionToBkgdFifo((PtrToBkgdFunc)BgAvgFDQAccumulation);
            gs_AlgHandlerState = FDQ_TRAIN;
         }
         else if ( (gs_FDQTrainType==FINE_TUNE_FDQ_TRAIN) ||
                   (gs_FDQTrainType==MEDLEY_FDQ_TRAIN) ||
                   (gs_FDQTrainType==COARSE_FDQ_TRAIN) )
         {
            gs_AlgHandlerState = FDQ_PRE_ACCUM;
         }
      }
      break;

   case FDQ_TRAIN:
      // Call FDQ training as a background task
      if (guc_FdqTrainingState == TRAINING_DONE)
      {
         guc_FdqTrainingState = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)FDQTrain);
         gs_AlgHandlerState = FDQ_LOAD;
      }
      break;

   case FDQ_LOAD:
      // Load FDQ coefficients to HW. Load FDQ exponents to inactive EGT table.
      if (guc_FdqTrainingState == TRAINING_DONE)
      {
         // Load FDQ coeffs
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, LoadFDQ);
         gs_AlgHandlerState = FDQ_NEXT_TONEGROUP;
      }
      break;

   case FDQ_NEXT_TONEGROUP:
      // Set up FDQ training vars for the next tone group
      gs_AlgHandlerState = FDQ_SETUP;

      // Determine whether all FDQ tones have been processed
      if(gs_RightChannel == gsa_RxBandRightChannel[gs_NumOfRxBands-1])
      {
         // Set flag to read SFDQ output
         gs_RtvSelect = SFDQ_OUTPUT;
         gs_RtvSelectLastFrame = RTV_NOTYPE;

         if ((gs_FDQTrainType == COARSE_FDQ_TRAIN) && (!(gs_FDQTrainTypeCntrlFlag & FDQ_TRAIN_TYPE_COARSE_ONLY)))
         {
            // Reset gs_LeftChannel to perform second pass FDQ:
            gs_CurrentRxBand = 0;
            gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];

            // Set FDQ training for fine tuning the message carrying tones
            gs_FDQTrainType  = FINE_TUNE_FDQ_TRAIN;
         }
         else
         {
            // If fine tuning FDQ or running Medley FDQ or in the diag mode, then training complete, i.e.
            // if ((gs_FDQTrainType == FINE_TUNE_FDQ_TRAIN) ||
            //     (gs_FDQTrainType == MEDLEY_FDQ_TRAIN) ||
            //     (gs_FDQTrainType == MEDLEY_FDQ_TRAIN_ONEPASS))
            gs_NumChannelsPerGroup = gs_NumChannelsPerGroupSave;
            gs_AlgHandlerState = FDQ_TRAIN_DONE;
         }
      }
      else
      {
         // Continue to process the remaining tones
         if(gs_RightChannel < gsa_RxBandRightChannel[gs_CurrentRxBand])
         {
            gs_LeftChannel = (gs_RightChannel + 1);
         }
         else
         {
            gs_CurrentRxBand++;
            gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];
         }
      }

      break;
   }
}

