/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2007 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/**********************************************************************
 *
 * Aware DMT Technology. Proprietary and Confidential.
 *
 * ADDRESS:          40 Middlesex Turnpike, Bedford, MA 01730-1432 USA
 * TELEPHONE:        (781) 276-4000
 * FAX:              (781) 276-4001
 * WEB:              http://www.aware.com
 *
 * FILE:             FrameAlign.c
 * DESCRIPTION:      Computes VDSL frame alignment offset.
 *
 **********************************************************************/

#include "common.h"
#include "gdata.h"
#include "arctan.h"
#include "stdlib.h"
#include "FrameAlign.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : RxFrameSync
 *
 *  Description:    Determines the offset necessary to achieve frame
 *  (symbol) synchronization with the signal received from the VTU-O or VTU-R.
 *
 *  Prototype:
 *      void RxFrameSync(int16 sa_X[], int16 s_StartTone, int16 *ps_offset)
 *
 *  Input Arguments:
 *      sa_X[]             --   descrambled frequency domain samples of
 *                              representative received frame
 *      s_StartTone        --   an integer multiple of 10: e.g. 10, 20, 30, etc.
 *
 *  Output Arguments:
 *      ps_offset         --   pointer to offset to from current frame
 *                              boundary to new frame boundary.
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void RxFrameSync(int16 sa_X[], int16 s_StartTone, int16 *ps_offset)
{
   /* Use only tones 8, 9, 10, 18, 19, 20, 28, 29, 30, etc. which have (+, +)
      constellation prior to scrambling.

     s_StartTone must be an integer multiple of 10: e.g. 10, 20, 30, etc.

     gs_NumFrameAlignClusters is the number of 3 tone clusters to use.
     (A 3 tone cluster consists of tones k*10+8, k*10+9 and k*10+10).
     (A 3 tone cluster consists of 2 tone pairs: (k*10+8, k*10+9) and (k*10+9, k*10+10)).

   */
   int16 i, j, k;
   int16 s_phase1, s_phase2, s_count, s_bin;
   int32 l_SumOfPhaseDiff;
   int32 l_temp, l_temp1, l_temp2;
   int16 s_bin_count1[1<<LOG2_NUM_FSYNC_BINS], s_bin_count2[1<<LOG2_NUM_FSYNC_BINS];
   int16 s_log2_bin_width, s_start_bin, s_stop_bin, num_in_average;

   int16 s_NumFSyncBins, s_Log2NumFSyncBins;

   // Number of freq bins usind for histogramming depends on the number of clusters
   if (gs_NumFrameAlignClusters > 2)
   {
      s_Log2NumFSyncBins = LOG2_NUM_FSYNC_BINS;
   }
   else
   {
      s_Log2NumFSyncBins = LOG2_NUM_FSYNC_BINS-2;
   }

   s_NumFSyncBins = (1 << s_Log2NumFSyncBins);

   // Bin width is 65536 (actually 2*pi = 2*25736, but rounded up) divided by number of bins,
   // This is the amount of unnormalized group delay in a bin.
   s_log2_bin_width =  16 - s_Log2NumFSyncBins;

   j = 0;
   s_count = 0;
   for (k=0; k < (s_NumFSyncBins); k++)
   {
      s_bin_count1[k]=0;
      s_bin_count2[k]=0;
   }

   //For the first pass through the tone clusters, the group delay is calculated for each
   //pair of adjacent tones, and the bin containing the group delay value is incremented.

   for (k = 0; k < gs_NumFrameAlignClusters; k++)
   {

      for (j = 0; j < 2; j++)
      {

         i = 2*(k*10+s_StartTone-(2-j));
         // FastAtan() output: 8192  is equiv to 1 radian. 25736 is equiv to pi radians.
         s_phase1 = FastAtan(sa_X[i+1], sa_X[i]);
         s_phase2 = FastAtan(sa_X[i+3], sa_X[i+2]);

         l_temp = (int32)s_phase2-s_phase1;

//      for (j = 0; j < 4; j=j+2) {
//
//         i = 2*(k*10+s_StartTone-(4-j));
//            // FastAtan() output: 8192  is equiv to 1 radian. 25736 is equiv to pi radians.
//            s_phase1 = FastAtan(sa_X[i+1], sa_X[i]);
//            s_phase2 = FastAtan(sa_X[i+5], sa_X[i+4]);
//
//            l_temp = (int32)((s_phase2-s_phase1)>>1);

         // wrap around for range 1 [-pi, +pi]:
         l_temp1 = l_temp;
         if (l_temp >=  (25736))
         {
            l_temp1 -= (2*25736);   /* wrap around +pi by subtracting 2*pi */
         }
         if (l_temp <= -(25736))
         {
            l_temp1 += (2*25736);   /* wrap around -pi by adding 2*pi */
         }

         // count number of points within range [-0.5*pi, +0.5*pi]:
         if ((l_temp1 >=  -12868) && (l_temp1 <=  12868))
         {
            s_count++;
         }

         // wrap around for range 2 [0, 2*pi]:
         l_temp2 = l_temp;
         if (l_temp >=  (2*25736))
         {
            l_temp2 -= (2*25736);   /* wrap around +2pi by subtracting 2*pi */
         }
         if (l_temp <=  0)
         {
            l_temp2 += (2*25736);   /* wrap around 0 by adding 2*pi */
         }

         // Count how many group delays are in each of the 1<<LOG2_NUM_FSYNC_BINS intervals
         // of length 1<<log2_bin_width spanning the total group delay range.
         // Increment the appropriate bin counter.
         s_bin_count1[ (int16) ((l_temp1 + 25736) >> s_log2_bin_width) ]++;
         s_bin_count2[ (int16) ( l_temp2          >> s_log2_bin_width) ]++;

#ifdef LOG_FA_DEBUG_INFO
#if 1
         // for each iteration, capture the 2 phase diffs for each cluster.
         // Only the first 3 clusters are captured.
         if (k < 3)
         {
            gsa_FA_debug_buf[((6*gs_FrameAlignIteration)+2*k)+j] = (int16)l_temp1;
         }
#else
         // capture the 2 tones k*10+8 and k*10+9 for first 2 clusters.
         // both real and imag parts are logged for 7 iterations.
         if ((k < 2) && (gs_FrameAlignIteration < 7))
         {
            gsa_FA_debug_buf[((TONE_CLUSTER_INDEX8*gs_FrameAlignIteration)+4*k)+2*j+0] = sa_X[i];
            gsa_FA_debug_buf[((TONE_CLUSTER_INDEX8*gs_FrameAlignIteration)+4*k)+2*j+1] = sa_X[i+1];
         }
#endif
#endif // LOG_FA_DEBUG_INFO

      }
   }

   // choose the group delay corresponding to the right range based on s_count. Use s_bin_count1 subsequently,
   // overwriting with s_bin_count2, if necessary.
   if (s_count < gs_NumFrameAlignClusters)
   {
      for (k=0; k < s_NumFSyncBins; k++)
      {
         s_bin_count1[k] = s_bin_count2[k];
      }
   }

   // Determine which consecutive bins contain the most group delays.
   // Group delays outside this range will be considered outliers.
   s_start_bin = 0;
   l_temp1 = -1;
   for (k=0; k<=s_NumFSyncBins-NUM_CONSECUTIVE_FSYNC_BINS; k++)
   {
      l_temp = 0;
      for (i=k; i<= k+NUM_CONSECUTIVE_FSYNC_BINS-1; i++)
      {
         l_temp += s_bin_count1[i];
      }
      if (l_temp>l_temp1)
      {
         l_temp1 = l_temp;
         s_start_bin = k;
      }
   }
   s_stop_bin = s_start_bin + NUM_CONSECUTIVE_FSYNC_BINS-1;
   if (s_stop_bin > s_NumFSyncBins-1)
   {
      s_stop_bin = s_NumFSyncBins-1;
   }

   // The group delay estimates for the first two iterations can be scattered normally, so
   // disable outlier deletion for first two iterations of frame alignment,
   if (gft_FrameAlignRemoveOutliers == 0)
   {
      s_start_bin = 0;
      s_stop_bin  = s_NumFSyncBins - 1;
   }

   // For the second pass through the tone clusters, recalculate the group delay for each
   // pair of adjacent tones, but include it in the average only if it lies in the acceptable
   // range of bins.
   num_in_average = 0;
   l_SumOfPhaseDiff = 0;
   for (k = 0; k < gs_NumFrameAlignClusters; k++)
   {

      for (j = 0; j < 2; j++)
      {

         i = 2*(k*10+s_StartTone-(2-j));
         // FastAtan() output: 8192  is equiv to 1 radian. 25736 is equiv to pi radians.
         s_phase1 = FastAtan(sa_X[i+1], sa_X[i]);
         s_phase2 = FastAtan(sa_X[i+3], sa_X[i+2]);

         l_temp = (int32)s_phase2-s_phase1;

//      for (j = 0; j < 4; j=j+2) {
//
//         i = 2*(k*10+s_StartTone-(4-j));
//
//            // FastAtan() output: 8192  is equiv to 1 radian. 25736 is equiv to pi radians.
//            s_phase1 = FastAtan(sa_X[i+1], sa_X[i]);
//            s_phase2 = FastAtan(sa_X[i+5], sa_X[i+4]);
//
//            l_temp = (int32)((s_phase2-s_phase1)>>1);

         // choose the group delay corresponding to the right range based on s_count.
         if (s_count >= gs_NumFrameAlignClusters)
         {
            // wrap around for range 1 [-pi, +pi]:
            if (l_temp >=  (25736))
            {
               l_temp -= (2*25736);   /* wrap around +pi by subtracting 2*pi */
            }
            if (l_temp <= -(25736))
            {
               l_temp += (2*25736);   /* wrap around -pi by adding 2*pi */
            }
            // Translate into bin number
            s_bin = (int16) ((l_temp+25736) >> s_log2_bin_width);

         }
         else
         {
            // wrap around for range 2 [0, 2*pi]:
            if (l_temp >=  (2*25736))
            {
               l_temp -= (2*25736);   /* wrap around +2pi by subtracting 2*pi */
            }
            if (l_temp <=  0)
            {
               l_temp += (2*25736);   /* wrap around 0 by adding 2*pi */
            }
            // Translate into bin number
            s_bin = (int16) (l_temp >> s_log2_bin_width);
         }

         // Outliers will NOT be included in the averaging.
         if ((s_bin >= s_start_bin) & (s_bin <=s_stop_bin))
         {
            num_in_average++;
            l_SumOfPhaseDiff += l_temp;
         }

      }
   }

   if (num_in_average != 0)
   {
      // take average. Divide by 2*gs_NumFrameAlignClusters.
      l_SumOfPhaseDiff /= num_in_average;
   }
   else
   {
      l_SumOfPhaseDiff = 25736; // =pi
   }

   // normalize. mult by N/(2*pi*8192)
   l_SumOfPhaseDiff = ((l_SumOfPhaseDiff*gs_RxFftLength) >> 13);     // mult by N/8192
   l_SumOfPhaseDiff = ((l_SumOfPhaseDiff*326 >> 10) + 1) >> 1;       // divide by 2*pi and round
                                                                     // FW: 326/1024 = 0.318359375; real: 1/pi = 0.31830988618379067153776752674503

   *ps_offset = (int16)l_SumOfPhaseDiff;
}

void BgRxFrameSync(void)
{

   RxFrameSync(gpsa_RxRepBuf, 10, &gs_AlignmentOffset);
   guc_FrameAlignmentState = TRAINING_DONE;

}


