/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: MfdqAdjust.c
*
*   This file contains functions to adjust MFDQ coefficient for showtime.
*
*-------------------------------------------------------------------------------
*/

#include "common.h"
#include "gdata.h"
#include "dsp_op.h"
#include "ieee_flt.h"
#include "decimalgain.h"
#include "MfdqHandler.h"
#include "MfdqAdjust.h"
#include "MfdqSupport.h"
#include "decimalgain.h"

#ifdef DEBUG_MFDQ
#include <stdio.h>
#endif // DEBUG_MFDQ

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BgAdjustMfdq(void)
*
*   This function adjusts MFDQ tone table for showtime.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void BgAdjustMfdq(void)
{
   int16 i, j, s_PhyToneIdx, s_LogToneIdx;
   int16 s_MfdqPhyToneIdx, s_FbckPhyToneIdx, s_MfdqLogToneIdx, s_FbckLogToneIdx;

   for (i=0; i<gt_MfdqConfig.s_NumMfdqTones; i++)
   {

      j = i << 1;

//#define USE_MFDQ_HW_WORK_AROUND

#ifdef USE_MFDQ_HW_WORK_AROUND
      // Don't use feedback in showtime for HW MFDQ workaround
      gt_MfdqConfig.psa_fb_coef[j] = 0;
      gt_MfdqConfig.psa_fb_coef[j+1] = 0;
#endif

#ifndef USE_MFDQ_HW_WORK_AROUND
      {
         int16 sa_MfdqCoef[3];
         // adjust MFDQ coefficients for showtime
         if (AdjustMfdqCoef(i, sa_MfdqCoef) == SUCCEED)
         {
            // if error occurs, load original unadjusted MFDQ
            gt_MfdqConfig.psa_fb_coef[j] = sa_MfdqCoef[0];
            gt_MfdqConfig.psa_fb_coef[j+1] = sa_MfdqCoef[1];
            gt_MfdqConfig.puca_fb_exp[i] = (uint8)sa_MfdqCoef[2];
         }
      }
#endif

      // convert physical tone index to logical tone index for MFDQ tone table,
      // which is prepared here and will be written to HW right before showtime
      s_MfdqPhyToneIdx = gt_MfdqConfig.psa_MfdqTones[i];
      s_FbckPhyToneIdx = gt_MfdqConfig.psa_FbckTones[i];
      s_PhyToneIdx = s_LogToneIdx = 0;
      s_MfdqLogToneIdx = s_FbckLogToneIdx = 0;
      // assume s_MfdqPhyToneIdx > s_FbckPhyToneIdx & s_MfdqLogToneIdx > s_FbckLogToneIdx
      while (s_PhyToneIdx != s_MfdqPhyToneIdx)
      {
         s_PhyToneIdx = ghpsa_RxToneOrder_Inactive[s_LogToneIdx];
         if (s_PhyToneIdx == s_FbckPhyToneIdx)
         {
            s_FbckLogToneIdx = s_LogToneIdx;
         }
         if (s_PhyToneIdx == s_MfdqPhyToneIdx)
         {
            s_MfdqLogToneIdx = s_LogToneIdx;
         }
         s_LogToneIdx++;
      }
      gt_MfdqConfig.pla_ToneTable[i] = ((uint32)s_FbckLogToneIdx << 16) | (uint32)s_MfdqLogToneIdx;
   }

   guc_BgRxShowInitState = TRAINING_DONE;

#ifdef DEBUG_MFDQ
   {
      int16 i;
      FILE *fptr;

      fptr = fopen("mfdq_table_cpe.txt","w");
      for (i=0; i<gt_MfdqConfig.s_NumMfdqTones; i++)
      {
         fprintf(fptr, "%d %d ", gt_MfdqConfig.psa_MfdqTones[i], gt_MfdqConfig.psa_FbckTones[i]);
         fprintf(fptr, "0x%08lx\n", gt_MfdqConfig.pla_ToneTable[i]);
      }
      fclose(fptr);

      fptr = fopen("mfdq_trt_cpe.txt","w");
      for (i=0; i<gs_RxNumTonesUsed+2; i++)
      {
         fprintf(fptr, "%d\n", ghpsa_RxToneOrder_Inactive[i]);
      }
      fclose(fptr);
   }
#endif // DEBUG_MFDQ
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: int16 AdjustMfdqCoef(int16 s_tidx, int16 *psa_MfdqCoef);
*
*   This function adjusts MFDQ coefficients for showtime.
*
*   Input Arguments:
*      s_tidx: MFDQ tone index (0 to gt_MfdqConfig.s_NumMfdqTones-1)
*
*   Output Arguments:
*      psa_MfdqCoef: pointer to MFDQ coefficients
*         psa_MfdqCoef[0]: Feedback (Bk) real component (Q1.15)
*         psa_MfdqCoef[1]: Feedback (Bk) imaginary component (Q1.15)
*         psa_MfdqCoef[2]: Feedback (Bk) exponent component (-9 to +6)
*
*   Returns:
*      s_return: SUCCEED or FAIL
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

int16 AdjustMfdqCoef(int16 s_tidx, int16 *psa_MfdqCoef)
{
   int16 i, j, s_mt_tone, s_fb_tone, sa_coef[3], s_exp, s_return = SUCCEED;
   FloatG f_mt_cgain, f_fb_cgain, f_mt_fgain, f_fb_fgain;
   FloatG f_cgain_adj, f_fgain_adj, f_gain_adj;
   FloatG f_temp, fa_coef[2], f_scale;
#ifdef DEBUG_MFDQ
   FILE *fptr;
   double d_mt_cgain, d_mt_fgain, d_fb_cgain, d_fb_fgain;

   if (s_tidx == 0)
   {
      fptr = fopen("mfdq_coef_adj_cpe.txt", "w");
   }
   else
   {
      fptr = fopen("mfdq_coef_adj_cpe.txt", "a");
   }
#endif

   i = s_tidx;
   j = i << 1;

#ifdef DEBUG_MFDQ
   fprintf(fptr, "%d %d %d ", gt_MfdqConfig.psa_fb_coef[j], gt_MfdqConfig.psa_fb_coef[j+1], gt_MfdqConfig.puca_fb_exp[i]);
#endif

   // get constellation decode gain and fine gain for MFDQ tone
   s_mt_tone = gt_MfdqConfig.psa_MfdqTones[i];
   f_mt_cgain = GetRxConstGain(s_mt_tone);
   f_mt_fgain = GetRxFineGain(s_mt_tone);

   // get constellation decode gain and fine gain for feedback tone
   s_fb_tone = gt_MfdqConfig.psa_FbckTones[i];
   f_fb_cgain = GetRxConstGain(s_fb_tone);
   f_fb_fgain = GetRxFineGain(s_fb_tone);

#ifdef DEBUG_MFDQ
   d_mt_cgain = Float32toDouble(f_mt_cgain)/8192.0;
   d_mt_fgain = Float32toDouble(f_mt_fgain)/8192.0;
   d_fb_cgain = Float32toDouble(f_fb_cgain)/8192.0;
   d_fb_fgain = Float32toDouble(f_fb_fgain)/8192.0;
   fprintf(fptr, "%f %f ", d_mt_cgain, d_mt_fgain);
   fprintf(fptr, "%f %f ", d_fb_cgain, d_fb_fgain);
#endif

   // compute gain adjustment for FB term
   // scaling factor = G[i]/G[k] where Gi = CDEC[i]*FGAIN[i], Gk = CDEC[k]*FGAIN[k],
   // FGAIN[i] = 1/TX fine gain for tone i
   f_cgain_adj = divf32(f_mt_cgain, f_fb_cgain);
   f_fgain_adj = divf32(f_mt_fgain, f_fb_fgain);
   f_gain_adj = mpyf32(f_cgain_adj, f_fgain_adj);

   // apply gain adjustment to MFDQ FB coefficient
   // new FB coef = old FB coef * (G[i]/G[k])
   // where i is an index for MFDQ tone, k is an index for feedback tone
   s_exp = (int16)MFDQ_MANTISSA_FRAC_BITS - ((int16)gt_MfdqConfig.puca_fb_exp[i] - MFDQ_MAX_FB_EXP_RSHFT);
   f_scale = int32toFloat32((int32)(1 << s_exp));
   f_temp = int32toFloat32((int32)gt_MfdqConfig.psa_fb_coef[j]);
   f_temp = mpyf32(f_temp, f_gain_adj);
   fa_coef[0] = divf32(f_temp, f_scale);
   f_temp = int32toFloat32((int32)gt_MfdqConfig.psa_fb_coef[j+1]);
   f_temp = mpyf32(f_temp, f_gain_adj);
   fa_coef[1] = divf32(f_temp, f_scale);

   // convert to HW-ready int16 coefficient
   ConvertMfdqCoef(sa_coef, fa_coef, MFDQ_MAX_FB_EXP_RSHFT);
   if (sa_coef[2] > MFDQ_MAX_EXP_SHFT)
   {
      sa_coef[2] = MFDQ_MAX_EXP_SHFT;
      gt_MfdqConfig.s_ErrorCode |= MFDQ_COEFF_ADJ_OVFLOW;
      s_return = FAIL;
   }
   *psa_MfdqCoef++ = sa_coef[0];
   *psa_MfdqCoef++ = sa_coef[1];
   *psa_MfdqCoef++ = (uint8)sa_coef[2];

#ifdef DEBUG_MFDQ
   fprintf(fptr, "%d %d %d\n", sa_coef[0], sa_coef[1], sa_coef[2]);
   fclose(fptr);
#endif

   return (s_return);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FloatG GetRxConstGain(int16 s_tone)
*
*   This function returns constellation decode gain in floating point
*   for a given physical tone index.
*
*   Input Arguments:
*      s_tone: physical tone index
*
*   Output Arguments:
*
*   Returns:
*      f_cgain: constellation gain in floating point
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/


FloatG GetRxConstGain(int16 s_tone)
{
   FloatG f_cgain = 0;

   return (f_cgain);
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: FloatG GetRxFineGain(int16 s_tone)
*
*   This function returns RX fine gain in floating point for a given
*   physical tone index.
*
*   Input Arguments:
*      s_tone: physical tone index
*
*   Output Arguments:
*
*   Returns:
*      f_fgain: fine gain in floating point
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

FloatG GetRxFineGain(int16 s_tone)
{
   int16 s_fgain;
   FloatG f_fgain;

   // convert fine gain to decimal form
   // note that at this point fine gain is still in dB format
   // it will be converted into decimal form in PackBGDescriptor()
   s_fgain = -ghpsa_RxFineGains_Inactive[s_tone];
   s_fgain = DecimalGain(s_fgain);

   // convert fine gain to floating point
   f_fgain = int32toFloat32((int32)s_fgain);

   return (f_fgain);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ConvertMfdqCoef(int16 *psa_coef, FloatG *pfa_coef, int16 s_rsh)
*
*   This function converts floating point MFDQ coefficients to HW-ready int16
*   coefficients.
*
*   Input Arguments:
*      pfa_coef: pointer to 2x1 floating point input vector [real; imag]
*      s_rsh: maximum amount of right shift
*
*   Output Arguments:
*      psa_coef: pointer to 3x1 16-bit MFDQ coefficient [real; imag; exp]
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void ConvertMfdqCoef(int16 *psa_coef, FloatG *pfa_coef, int16 s_rsh)
{
   int16 s_exp;
   int32 l_scale, l_real, l_imag, l_tmp_real, l_tmp_imag;
   FloatG f_scale, f_temp;

   l_scale = (int32)(1 << (int16)(MFDQ_MANTISSA_FRAC_BITS + s_rsh));
   f_scale = int32toFloat32(l_scale);
   f_temp = mpyf32(pfa_coef[0], f_scale);
   l_real = f32toint32(f_temp, 0);
   f_temp = mpyf32(pfa_coef[1], f_scale);
   l_imag = f32toint32(f_temp, 0);
   s_exp = -1;
   do
   {
      s_exp++;
      l_tmp_real = round(l_real, s_exp);
      l_tmp_imag = round(l_imag, s_exp);
   }
   while ((l_tmp_real > (int32)0x00007FFFL) || (l_tmp_real < (int32)0xFFFF8000L) ||
          (l_tmp_imag > (int32)0x00007FFFL) || (l_tmp_imag < (int32)0xFFFF8000L));
   psa_coef[0] = (int16)l_tmp_real;
   psa_coef[1] = (int16)l_tmp_imag;
   psa_coef[2] = s_exp;
}
