/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   PsdHandler.c
 *
 *   This file contains the routine that calculates Psd.
 *
 *------------------------------------------------------------------------
 */

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "psdhandler.h"
#include "psd_b.h"
#include "fifo.h"
#include "noiseacc.h"
#include "IRI_Iof.h"
#include "dsp_op.h"
#include "ConvertToDB.h"

extern TxToneFlags guca_TxSupportedToneSet;


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : PsdHandler
 *
 *  Description:
 *
 *      This function is used to calculate the Psd of the received signal.
 *   It uses the NoiseAcc function, which typically is used to calculate the
 *  avg Squared error signal.  In this case, the received signal, rather than
 *   an error signal, is fed into NoiseAcc.  The result is an avg signal power,
 *   rather than an average error squared.
 *
 *   The PSD is calculated over all (gt_PsdConfig.s_LastChannel-1) tones.
 *
 *  Prototype:  void PsdHandler(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gs_AlgHandlerState       - (I/O) state of PSD calculation state machine
 *      gpla_RxAccumBuf[]      - (I/O) accumulated frames
 *
 *  Notes:
 *   Before PsdHandler() is called for the first time, the following initialization
 *   needs to be performed:
 *
 *      1) gs_AlgHandlerState = PSD_SETUP.
 *      2) gs_AlgNumFramesToAccum, gs_AlgLog2NumFramesToAccum must be set up
 *
 *------------------------------------------------------------------------
 *^^^
 */

void PsdHandler(void)
{
   switch(gs_AlgHandlerState)
   {
      //==============================================================================
      // initialize variables for PSD calculation
      //==============================================================================
   case PSD_INIT:
      gs_AlgLog2NumFramesToAccum = gt_PsdConfig.s_AlgLog2NumFramesToAccum;
      gs_NumChannelsPerGroup = gt_PsdConfig.s_NumChannelsPerGroup;

      if (gs_Pga_XtakNoiseScale)
      {
         signed int i,j;

         // Clear the RX supported tone set to re-initialize it below
         memset(guca_TxSupportedToneSet, 0, sizeof(TxToneFlags));

         for (i=0; i<gs_NumOfTxBandsOSignature; i++)
         {
            for (j = gsa_TxBandLeftChannelOSignature[i]; j <= gsa_TxBandRightChannelOSignature[i]; j++)
            {
               // Indicates tone that is transmitted by near end
               SETTONEFLAG(guca_TxSupportedToneSet, j);
            }
         }
      }

      // For VRx518 the RTV is twice, i.e. for 1024 tones.
      // Note: Normally RTV size should be considered!
      //       The code should also handle on the fly configured values for debuging!
      gpla_RxAccumBuf = gpla_SnrAccuBuf;

      gs_AlgNumFramesToAccum = (1 << gs_AlgLog2NumFramesToAccum);

      gs_MaxRxTonePwr = (int16)(-100*256);
      gl_LinMaxRxTonePwr = 0x80000000; //Initialise to the minimum value
      gl_Pa  = 0;

      gft_ApplyConstGain = 0;

      //Disable GetRxTone() if the HW noise accumulation is used to save time
      gft_EnableGetRxTones = FALSE;

      // Set start tone for PSD calculation. Note: (default is 0-gs_RxNumTones).
      gs_LeftChannel = gt_PsdConfig.s_FirstChannel;
      gs_AlgHandlerState = PSD_SETUP;
      break;

      //==============================================================================
      // set up variables for PSD calculation in each group
      //==============================================================================
   case PSD_SETUP:
      gs_RightChannel = gs_LeftChannel + gs_NumChannelsPerGroup - 1;

      gs_AlgNumTonesToProcess = gs_RightChannel - gs_LeftChannel + 1;

      if (gt_PsdConfig.s_AlgCalcType == PSD_CALC_TYPE_NOMSRDBUFF)
      {
         gpsa_AlgMsrdBuffer = NULL;
      }
      else
      {
         gpsa_AlgMsrdBuffer = &gpsa_MeasuredSnrBuf[gs_LeftChannel];
         // Initialize msrd o/p buffer to 0
         if (gt_PsdConfig.s_AlgCalcType == PSD_CALC_TYPE_INITMSRDBUFF_AND_INC)
         {
            memset(gpsa_AlgMsrdBuffer, 0, (sizeof(int16)*gs_AlgNumTonesToProcess));
         }
      }

      gs_AlgHandlerCount = 0;

      // Set RTV flag to accumulate the power of FFT output
      gs_RtvSelect = FFT_OUTPUT | HW_ACCUM_POW;

      // Set RX tone offset to the start tone from which the HW should do the processing, i.e. RTV buffer 0.
      gs_RxToneOffset = gs_LeftChannel;
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, SetRxToneOffset);

      // Transition to accumulation substate
      gs_AlgHandlerState = PSD_WAIT_FOR_HW_SET;
      break;

      //==============================================================================
      // Wait for RTV buffer 0 HW settings to take effect
      //==============================================================================
   case PSD_WAIT_FOR_HW_SET:

      if (gs_AlgHandlerCount < gs_RTVxCfgLatency)
      {
         gs_AlgHandlerCount++;
      }
      else
      {
         gs_AlgHandlerState = PSD_ACCUM;
         // Clear counter
         gs_AlgHandlerCount = 0;

         // Reset the Noise Accumulator Buffer and trigger the SNR in hardware
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, ResetNoisePowerBuffer);
      }

      break;

      //==============================================================================
      // Accum representative frame for PSD calculation
      //==============================================================================
   case PSD_ACCUM:

      gs_AlgHandlerCount++;
      if (gs_AlgHandlerCount == gs_AlgNumFramesToAccum)
      {
         // Queue TC task to disable any writing to NPR buffer
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, DisableNPRWrite);
      }
      else if(gs_AlgHandlerCount == gs_AlgNumFramesToAccum+1)
      {
         // Calculate the PSD in background
         guc_PsdCalcState = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgPsdCalc);
         gs_AlgHandlerState = PSD_NEXT_TONEGROUP;
      }

      break;

      //==============================================================================
      // Update for next tonegroup (or goto finish)
      //==============================================================================
   case PSD_NEXT_TONEGROUP:
      if (guc_PsdCalcState == TRAINING_DONE)
      {
         //If all the tones in this symbol has been processed
         if(gs_RightChannel == (gt_PsdConfig.s_LastChannel-1))
         {
            gpla_RxAccumBuf = &gla_SharedBuffer_X[RX_ACCUM_BUF0_OFFSET_LW];
            gs_NumChannelsPerGroup = gs_NumChannelsPerGroupSave;

            // restore the initial setting
            gs_RtvSelect = FFT_OUTPUT;
            gs_RtvSelectLastFrame = RTV_NOTYPE;

            //Enable GetRxTone() before exit
            gft_EnableGetRxTones = TRUE;

            gs_MaxRxTonePwr = sature16((int32)ConvertToDB(gl_LinMaxRxTonePwr));
            gs_AlgHandlerState = PSD_CALC_DONE;
         }
         //Continue to process the remaining tones
         else
         {
            // Switch RTV0 to recevie tone
            // We are doing this simply for reruning "gs_RtvSelect = FFT_OUTPUT | HW_ACCUM_POW;"
            gs_RtvSelect = FFT_OUTPUT;

            gs_LeftChannel = gs_LeftChannel + gs_NumChannelsPerGroup;
            gs_AlgHandlerState = PSD_SETUP;
         }
      }
      break;
   }
}

