/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2011 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   filename: SelectPilot.c
*
*   This file contains functions to selects pilot tone index.
*
*-------------------------------------------------------------------------------
*/

// ************************************************************************
// SelectPilot.c
//
// History
// 27/l1/2012 Ram: Merged IOP fix corresponding to JIRA XDSLRTFW-474
//           The No-connectivity or long training time issue is because of Wrong
//           pilot tone selection in DS band DS3 at loop lengths >2000m, where
//           we don't expect any SNR.But from the logs of all failing cases, we
//           do see good SNR in DS2 and DS2 bands and also passes all the
//           conditions for pilot tone selection, And we choose the pilot from
//           the DS3 band and link fails.
//
//           As a workaround  we tried not to choose the pilot tone from DS3 by
//           changing the metric based on Kl0( loop estimation), and tested at
//           different loop lengths for the configured profile,without any problems.
//           Root cause for further debugging is the input signal at some frames to
//           the FDQ calculation in DS2 and DS2 doesn't seem to be ok.
//           This fix is controlled by CMV INFO 232 0 bit3(mask 0x0008)
//             bit3 value - 0 (default)
//                        - 1  (for ECIBT testing)
//           Grep for: "XDSLRTFW-474: IOP_DS_VDSL2_ALL_WrongPilot_Selection"
// ************************************************************************

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "dsp_op.h"
#include "cmv.h"
#include "mul.h"
#include "ConvertToDB.h"
#include "vdsl_state.h"
#include "IRI_Iof.h"
#include "fifo.h"
#include "PGAHandler.h"


//Set the SNR threshold to be 15 dB
int16 gs_SnrThresh = (15<<8);
int16 gs_BiasThresh = (20<<8); //only apply frequency Bias if SNR > 20 dB

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void SelectPilotTone(int16 *psa_SnrBuf, int16 s_LeftChannel,
*      int16 s_RightChannel, int16 *ps_PilotToneIdx, int32 *pl_Metric)
*
*   This function selects the pilot tone based on SNR (or PSD based SNR).
*
*   Input Arguments:
*      psa_SnrBuf: pointer to SNR buffer
*      s_LeftChannel: left bound of the region of interest
*      s_RightChannel: right bound of the region of interest
*
*   Output Arguments:
*      ps_PilotToneIdx: pointer to pilot tone index
*      pl_Metric: pointer to the corresponding metric
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void SelectPilotTone(int16 *psa_SnrBuf, int16 s_LeftChannel, int16 s_RightChannel, int16 *ps_PilotToneIdx, int32 *pl_Metric)
{
   int16 i, s_ch, s_ch_delta, s_pilotbias, s_flag, s_snr_threshold, s_best_tone;
   int32 l_temp, l_metric, l_best_metric;
   FlagT ft_BiasFlag;
   int16 s_SnrThresh; //XDSLRTFW-2387 (Start_End)
   ft_BiasFlag = TRUE; //XDSLRTFW-522 (start_end)

   // pilot tone chosen is a multiple of 10 so that we have a fixed pilot reference
   s_LeftChannel = (s_LeftChannel+9)/10;
   MULS16(l_temp, s_LeftChannel, (int16)10);
   s_LeftChannel = (int16)l_temp;

   s_best_tone = -1;
   l_best_metric = 0;

   if (gul_dbgMiscControl & FORCE_MULTIPLE_OF_64_PILOTTONE)
   {
      // Use multiple of 64 pilot tone
      s_ch_delta = 64;

      // Find the first multiple of 64 pilot tone
      for (s_ch=s_LeftChannel; s_ch < s_RightChannel; s_ch++)
      {
         if (!(s_ch & 0x003f))
         {
            break;
         }
      }
   }
   else
   {
      // Use multiple of 10 pilot tone
      s_ch_delta = 10;
      s_ch=s_LeftChannel;
   }

   //XDSLRTFW-2387 (Start)
   if(gft_ReducePilotThr == TRUE)
   {
      s_SnrThresh = (gs_SnrThresh - (3<<8));
   }
   else
   {
      s_SnrThresh = gs_SnrThresh;
   }
   //XDSLRTFW-2387 (End)

   for (; s_ch < s_RightChannel; s_ch+=s_ch_delta)
   {
      // skip tones whose SNR is too low
      if (psa_SnrBuf[s_ch] < s_SnrThresh) //XDSLRTFW-2387 (Start_End)
      {
         continue;
      }

      //XDSLRTFW-522: IOP_US_VDSL2_ALL_US_Data_Rate_Variation_In_BT_ECI_Tests (start)
      // For PilotToneIdx_Initial (ChDisc phase) don't consider tones above
      // 3000 as we don't have good estimate of SNR, for Initial estimate
      // SNR is derived as the (PSD-QLN)
      if(gt_CustomerIopBits.us_ECI & ECI_PILOT1_TONEIDX_BIAS)
      {
         if((s_ch > 3000) && (gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].s_PilotToneIdx == 0))
         {
            ft_BiasFlag = FALSE;
         }
      }


      // apply bias on pilot tone selection only if SNR is better than 20 dB
      // otherwise, select pilot tone with best SNR regardless of frequency of the tone
      if ((psa_SnrBuf[s_ch] > gs_BiasThresh)&&(ft_BiasFlag == TRUE))
      {
         s_pilotbias = (ConvertToDB(s_ch)) << 1;   // 20*log10(k)
      }
      else
      {
         s_pilotbias = 0;
      }
      //XDSLRTFW-522: IOP_US_VDSL2_ALL_US_Data_Rate_Variation_In_BT_ECI_Tests (end)

      // XDSLRTFW-474: IOP_DS_VDSL2_ALL_WrongPilot_Selection (Start)
      if(gt_CustomerIopBits.us_ECI & ECI_TXECHO_PILOT_FIXES )
      {
         // loop lengths above 1800m and tone above 2780
         // make bias as '0' , this will ensure tones above 2780 will not
         // be selected for pilot.
         if((gt_Kl0ElectricalLength.s_kl0_estimate > 300)&& (s_ch > 2780))
         {
            s_pilotbias = 0;
         }

         // loop lengths above 2200m and tone above 2780
         // make bias as '0' , this will ensure tones above 1182 will not
         // be selected for pilot.
         if((gt_Kl0ElectricalLength.s_kl0_estimate > 450)&& (s_ch > 1182))
         {
            s_pilotbias = 0;
         }
      }
      // XDSLRTFW-474: IOP_DS_VDSL2_ALL_WrongPilot_Selection(end)

      l_metric = s_pilotbias + psa_SnrBuf[s_ch];

      // choose the tone that gives the best [20log(k) + snr(k)] metric
      if (l_metric >= l_best_metric)
      {
         // make sure this tone is not near an outlier
         s_snr_threshold = psa_SnrBuf[s_ch] - (int16)(4 << 8);

         s_flag = 1;
         for (i=s_ch-1; i<=s_ch+1; i++)
         {
            if (psa_SnrBuf[i] < s_snr_threshold)
            {
               s_flag = 0;
               break;
            }
         }

         if (s_flag)
         {
            s_best_tone = s_ch;
            l_best_metric = l_metric;
            gs_pilotbias = s_pilotbias;
         }
      }
   }
   // XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
   if(s_best_tone != -1)
   {
      *ps_PilotToneIdx = s_best_tone;
      *pl_Metric = l_best_metric;
   }
   else
   {
      *ps_PilotToneIdx = 0;
      *pl_Metric = 0;
   }
   // XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void ChoosePilotSelectBand(int16 *psa_SnrBuf, int16 *ps_FirstTone, int16 *ps_LastTone)
*
*   This function selects the tone range for the pilot tone selection.
*   For simplicity, the first tone is set to be the first input tone, which is usually theleft band edge.
*   the last tone is chosen to be the last tone in the given band, whose SNR is greater than 15 dB.
*
*   Input Arguments:
*      psa_SnrBuf: pointer to SNR buffer
*      *ps_FirstTone: left bound of the region of interest
*      *ps_LastTone: right bound of the region of interest
*
*   Output Arguments:
*      *ps_LastTone: the updated right bound of the pilot tone selection region.
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

void ChoosePilotSelectBand(int16 *psa_SnrBuf, int16 *ps_FirstTone, int16 *ps_LastTone)
{
   int16 i, s_LastTone;

   s_LastTone = -1;

   for(i = *ps_FirstTone; i <= *ps_LastTone; i++)
   {
      //Find the last tone whose SNR is greater than the SNR threshold
      if(psa_SnrBuf[i] > gs_SnrThresh)
      {
         s_LastTone = i;
      }
   }

   *ps_LastTone = s_LastTone;

} //DecidePilotSelectBand()




// XDSLRTFW-3280 Start VRx518 - PLL improvement / pilot tone selection improvement
/*
*-------------------------------------------------------------------------------
*
*   Prototype: void MultiPilotUpdate()
*
*   This function checks if a better pilot tone than the currently used pilot tone was found by the ComparePilotMetric() function.
*   In case there is a better pilot tone than the used pilot, it will calculate a new reference for the new best pilot. After determining
*   a new reference tone (we are still using the old pilot) the phase error of the new pilot tone is observed for gs_PLLSymbolsToAvg symbols.
*   If this average phase error for the new pilot tone is below the threshold gs_PTSwitchMaxPhaseError, we will give the PLL the new pilot
*   tone as input.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*     - gt_PilotConfig.te_UsedPTArrayIdx
*     - gft_Try2ndBestPTIdx
*
*-------------------------------------------------------------------------------
*/
void MultiPilotUpdate()
{
   PT_ArrIdx_te BestArrayIdx = gt_PilotConfig.te_BestPTArrayIdx;
   PT_ArrIdx_te UsedArrayIdx = gt_PilotConfig.te_UsedPTArrayIdx;

   switch (gs_PLLAvgHandlerState)
   {
      case PLL_AVG_IDLE:
         if (gt_PilotConfig.te_UsedPTArrayIdx != gt_PilotConfig.te_BestPTArrayIdx)
         {
            gs_PLLAvgHandlerCnt=0;
            gla_AvgRefPilotToneReal[BestArrayIdx] = gt_PilotConfig.ta_PilotTones[BestArrayIdx].s_PilotTone_Re;
            gla_AvgRefPilotToneImag[BestArrayIdx] = gt_PilotConfig.ta_PilotTones[BestArrayIdx].s_PilotTone_Im ;
            gs_PLLAvgHandlerState = PLL_AVG_ACCUM;
         }
      break;

      case PLL_AVG_ACCUM:
         //@todo AH: consider improving the precision by avoiding shifting down in ComputePLLRefAverage and then up in ResetPllRefTone
         // compute average for reference, as sometimes reference could be noisy
         ComputePLLRefAvg(BestArrayIdx, &gsa_AvgRefPilotToneReal[BestArrayIdx], &gsa_AvgRefPilotToneImag[BestArrayIdx],512,9);
      break;

      case PLL_AVG_DONE:
         // Grab a new PLL reference.
         ResetPllRefTone(BestArrayIdx, NULL, &gsa_AvgRefPilotToneReal[0], &gsa_AvgRefPilotToneImag[0], (int16)1);

         //Initialize for phase error accumulation
         gs_PLLAvgHandlerCnt=0;
         gt_PilotConfig.ta_PilotTones[0].ul_AvgPhaseError = Abs16(gt_PilotConfig.ta_PilotTones[0].gs_PhaseError);
         gt_PilotConfig.ta_PilotTones[1].ul_AvgPhaseError = Abs16(gt_PilotConfig.ta_PilotTones[1].gs_PhaseError);
         gt_PilotConfig.ta_PilotTones[2].ul_AvgPhaseError = Abs16(gt_PilotConfig.ta_PilotTones[2].gs_PhaseError);
         gs_PLLAvgHandlerState = PLL_AVG_PHASE_ERROR;
      break;

      case PLL_AVG_PHASE_ERROR:
         uint32 ul_temp;
         if (gs_PLLAvgHandlerCnt < gs_PLLSymbolsToAvg)   //
         {
            gt_PilotConfig.ta_PilotTones[0].ul_AvgPhaseError += Abs16(gt_PilotConfig.ta_PilotTones[0].gs_PhaseError);
            gt_PilotConfig.ta_PilotTones[1].ul_AvgPhaseError += Abs16(gt_PilotConfig.ta_PilotTones[1].gs_PhaseError);
            gt_PilotConfig.ta_PilotTones[2].ul_AvgPhaseError += Abs16(gt_PilotConfig.ta_PilotTones[2].gs_PhaseError);
            gs_PLLAvgHandlerCnt++;
         }
         else
         {
            // XDSLRTFW-3695 (Start) Make it possible to switch to a pilot tone which is in fluctuating SNR tone range
            for(int i=PT_ARRAY_IDX_0;i<=PT_ARRAY_IDX_2;i++)
            {
               gt_PilotConfig.ta_PilotTones[i].ul_AvgPhaseError=(gt_PilotConfig.ta_PilotTones[i].ul_AvgPhaseError>>gs_PLLSymbolsToAvgLog2);
               // Normalize the average phase error. Reference is the phase error of the used pilot tone
               ul_temp = gt_PilotConfig.ta_PilotTones[i].ul_AvgPhaseError * gt_PilotConfig.ta_PilotTones[UsedArrayIdx].s_PilotToneIdx;
               //@todo AH: consider using dsp function for division
               gt_PilotConfig.ta_PilotTones[i].ul_AvgPhaseErrorNorm = ul_temp / gt_PilotConfig.ta_PilotTones[i].s_PilotToneIdx;
            }

            // Comparison of the phase error of old and new pilot tone
            // Switch to the best pilot tone index if the normalized phase error of is is smaller or equal to the currently used one
            if( ( gt_PilotConfig.ta_PilotTones[BestArrayIdx].ul_AvgPhaseErrorNorm <= gt_PilotConfig.ta_PilotTones[UsedArrayIdx].ul_AvgPhaseErrorNorm) ||
                ( gt_PilotConfig.s_PilotToneControl & IGNORE_PHASE_ERROR_WHEN_SWITCH ) )
            {
               gt_PilotConfig.te_UsedPTArrayIdx = gt_PilotConfig.te_BestPTArrayIdx;
               gft_Try2ndBestPTIdx=FALSE;
               // XDSLRTFW-3609 Start/End Re-calculate the pilot tone threshold for declaring a LOS defect, otherwise we might get a wrong LOS indication when switching from a pilot tone with high power to one with lower power
               guc_LosDefect_state = LOS_INIT;
               DSH_SendStream(DSH_PILOT_TONE_ARRAY,sizeof(gt_PilotConfig),&gt_PilotConfig.s_NumPilotTones);
            }   // XDSLRTFW-3695 (End) Make it possible to switch to a pilot tone which is in fluctuating SNR tone range
            else
            {
               // The phase error does not allow to switch to the pilot with the currently best pilot metric, hence discard this proposed pilot tone
               // This scenario can happen if we have 'fluctuating' SNR on a proposed pilot tone with very high index
               // In this case we want to give the second best pilot tone (should be normally in DS2 band) also a try
               gt_PilotConfig.te_BestPTArrayIdx = gt_PilotConfig.te_UsedPTArrayIdx;
               // Toggle between best and 2nd best pilot tone in case we fail to select the intended pilot tone
               gft_Try2ndBestPTIdx ^= TRUE;
               // event which notifies that a pilot tone switching was discarded
               DSH_SendStream(DSH_SKIP_PT_SWITCHING,sizeof(gt_PilotConfig),&gt_PilotConfig.s_NumPilotTones);
            }
            gs_PLLAvgHandlerState = PLL_AVG_IDLE;
         }

      break;
   }
   // XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement
}

