/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   SnrFrameAlignHandler.c
 *
 *   This file contains the routine that conducts SNR-based frame alignment.
 *
 *------------------------------------------------------------------------
 */

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "SnrFrameAlignHandler.h"
#include "fifo.h"
#include "DSLEngin.h"
#include "cosine.h"
#include "SnrHandler.h"
#include "mul.h"
#include "IRI_sync.h"

#include "IRI_Iof.h"
#include "V_STR_IOf.h"

#include "ieee_flt.h"
#include "FdqHandler.h"
#include "vdsl_const.h"

#include "cmv_Data.h"
#include "cmv.h"
#include "ll_iof.h"

#include "A_STR_IOf.h"
#include "arctan.h"
extern void NmsAGC1Config(void);

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : SnrFrameAlignHandler
 *
 *  Description:  This function performs the SNR-based RX frame realignment.
 *   Given the initial frame synch point found during the training phase,
 *  this process realigns frame boundary within (-offset, +offset) in relative to the initial
 *  frame boundary. For each new frame boundary, the FDQ is retrained and then SNR is recalculated.
 *  The frame synch point with the highest SNR sum is saved in in each try.
 *  Finally, the frame synch point corresponding to the highest SNR sum is used for the bitloading
 *  calculation. In this implementation, we limits the maximum frame realignment offset to be less
 *  or equal to the RX cyclic prefix length so we can achieve the frame realignment by changing
 *  RX CP length.
 *
 *  Prototype:  void SnrFrameAlignHandler(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gs_AlgHandlerState       - (I/O) state of this handler
 *      gs_MaxMedleyAlignOffset   - (I/O) the maximum frame alignment offset
 *      gs_RxCPLength - (I) RX cyclic prefix length
 *      gs_AlignmentOffset - (I/O) frame realignment offset
 *      gs_NumOfMedleyAlignPoints -- no. of realignment points to try
 *      gl_BestSnrSum -- (O) the highest SNR sum
 *      gs_BestSynchPointIdx -- (O) realignment offset corresponding to the highest SNR sum
 *      gs_CurrSynchPointIdx -- (O) index of the current realignment offset
 *      gs_RxBkgdProcessFlag -- (I/O) flag to indicate the state of the RX background process
 *      gs_RxSubStateCnt -- (I/O) substate count
 *      gs_SyncSearchMode -- 0=idle; 1=pos dir; -1=neg dir; 2=done
 *      gs_RxCPLength_save -- save original CP length
 *
 *
 *  Example:
 *   s_NumOfMedleyAlignPoints = 17
 *   s_MedleyAlignOffsetStep  = 30
 *   Note:
 *      If we've reached the maximum offset in a direction or
 *      if the current channel capacity is more than 12.5% BELOW best capacity so far,
 *      don't advance offsets in a direction any further.
 *
 *                             neg| mid |pos
 *                             -1  |  0  |  1                                       gs_SyncSearchMode
 *  <-----------------------------|-----|------------------------------------->
 *     0   1   2   3   4  5  6  7    8      9  10  11   12   13   14   15   16      gs_CurrSynchPointIdx
 *   240 210 180 150 120 90 60 30         -30 -60 -90 -120 -150 -180 -210 -240      gs_AlignmentOffset * num calls AdjustAlignmen()
 *                              |                                            |
 *                              <-----------------------------------<---------
 *                                         9*30 = 270
 *   |                          |
 *   ----------------------------
 *              |
 *   Algo finishs always in this range,
 *   i.e. gs_SyncSearchMode = 2
 *   Therefore the direct to go back to
 *   best sync point is (pos),i.e. -------------------->
 *   negativ samples
 *
 *------------------------------------------------------------------------
 *^^^
 */
void SnrFrameAlignHandler(void)
{
   int32 l_temp;
   int16 sa_tempPllRefTone[2];

   switch(gt_SnrFrameAlignConfig.s_AlgHandlerState)
   {

   case SNR_BASED_FRAME_ALIGN_INIT:

      // save # of SNR training symbols
      gs_SnrAlgLog2NumFramesToAccum_Saved = gt_SnrConfig.s_AlgLog2NumFramesToAccum;
      gt_SnrConfig.s_AlgLog2NumFramesToAccum = FRAME_REALIGNMENT_LOG2_NUM_SNR_TRAINING_SYMBOLS;

      gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_POSTTDQ_FA;
      break;

      //Start Post-TDQ frame synch search here
   case SNR_BASED_FRAME_ALIGN_POSTTDQ_FA:

      gft_RestoreOrigFdq = TRUE;

      // clear metric
      gl_BestSnrSum = -1;
      memset(gla_MedleySnrSum, 0, sizeof(int32)*NUM_FRAME_REALIGNMENT_POINTS);

      MULS16(l_temp, gt_SnrFrameAlignConfig.s_MedleyAlignOffsetStep, (gt_SnrFrameAlignConfig.s_NumOfMedleyAlignPoints) >> 1);
      gs_MaxMedleyAlignOffset = (int16)l_temp;

      gs_CurrSynchPointIdx    = (gt_SnrFrameAlignConfig.s_NumOfMedleyAlignPoints) >> 1; // corresponding to the middle point
      gs_BestSynchPointIdx    = gs_CurrSynchPointIdx;
      gs_AlignmentOffset      = 0;
      // gs_SyncSearchMode: 0=idle; 1=pos dir; -1=neg dir; 2=done
      gs_SyncSearchMode       = 0;                                                      // start off with 0 offset

      gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_SET;
      break;

      //Apply the frame realignment by moving frame start boundary forward by gs_AlignmentOffset
   case SNR_BASED_FRAME_ALIGN_SET:
      gs_RxSubStateCnt = 0;

      gft_EnablePLL = 0;           // Turn off PLL
      gs_RxFrameAlignRemain = 0;
      AddFunctionToFifo(gp_RxLoadingFunctionFifo, AdjustAlignment);

      gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_RESET;
      break;

      //Stop frame applying frame realignment
   case SNR_BASED_FRAME_ALIGN_RESET:

      //For the 6.2 HW, it is not allowed to make a strymon frame less than FFT size
      //so it may take more than one frame to do frame adjustment if gs_AlignmentOffset is greater than
      //CE Length. Note the variable gs_RxFrameAlignRemain stores the remaining adjustment
      //after each call to AdjustAlignment()
      if(gs_RxFrameAlignRemain != 0)
      {
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, AdjustAlignment);
      }
      else
      {
         //Restore the original RX CP length
         if(gs_RxSubStateCnt == 0)
         {
            AddFunctionToFifo(gp_RxLoadingFunctionFifo, ResetRxAlign);
         }
         // wait for the frames boundary realignment to settle
         else if (gs_RxSubStateCnt == 4)
         {
            // XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
            // Adjust PLL reference of 1st pilot tone
            sa_tempPllRefTone[0] = gt_PilotConfig.ta_PilotTones[0].s_PllRefTone_Re;
            sa_tempPllRefTone[1] = gt_PilotConfig.ta_PilotTones[0].s_PllRefTone_Im;
            RotateTone(gs_AlignmentOffset, gt_PilotConfig.ta_PilotTones[0].s_PilotToneIdx, &sa_tempPllRefTone[0]);
            gt_PilotConfig.ta_PilotTones[0].s_PllRefTone_Re = sa_tempPllRefTone[0];
            gt_PilotConfig.ta_PilotTones[0].s_PllRefTone_Im = sa_tempPllRefTone[1];

            // Adjust PLL reference of 2nd pilot tone
            sa_tempPllRefTone[0] = gt_PilotConfig.ta_PilotTones[1].s_PllRefTone_Re;
            sa_tempPllRefTone[1] = gt_PilotConfig.ta_PilotTones[1].s_PllRefTone_Im;
            RotateTone(gs_AlignmentOffset, gt_PilotConfig.ta_PilotTones[1].s_PilotToneIdx, &sa_tempPllRefTone[0]);
            gt_PilotConfig.ta_PilotTones[1].s_PllRefTone_Re = sa_tempPllRefTone[0];
            gt_PilotConfig.ta_PilotTones[1].s_PllRefTone_Im = sa_tempPllRefTone[1];

            // Adjust PLL reference of 3rd pilot tone
            sa_tempPllRefTone[0] = gt_PilotConfig.ta_PilotTones[2].s_PllRefTone_Re;
            sa_tempPllRefTone[1] = gt_PilotConfig.ta_PilotTones[2].s_PllRefTone_Im;
            RotateTone(gs_AlignmentOffset, gt_PilotConfig.ta_PilotTones[2].s_PilotToneIdx, &sa_tempPllRefTone[0]);
            gt_PilotConfig.ta_PilotTones[2].s_PllRefTone_Re = sa_tempPllRefTone[0];
            gt_PilotConfig.ta_PilotTones[2].s_PllRefTone_Im = sa_tempPllRefTone[1];
            // XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement
            gft_EnablePLL = 1; // Turn on PLL
         }
         else if (gs_RxSubStateCnt == (5+gs_PllWaitMedley))  // Wait for PLL to be stable!    // XDSLRTFW-3837 (Start_End)
//         else if (gs_RxSubStateCnt == (5))                   // This would be the exact value!
         {
            gpsa_MeasuredSnrBuf = gsa_SnrBuf;
            gs_AlgHandlerState = SNR_INIT;
            gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_CALC_SNR;
         }

         gs_RxSubStateCnt++;
      }

      break;

   case SNR_BASED_FRAME_ALIGN_CALC_SNR:

      if (gs_AlgHandlerState != SNR_CALC_DONE)
      {
         SnrHandler();                    // calls BgMedleyRotateFDQ
      }
      else
      {
         if (gs_SyncSearchMode == 2)
         {
            // restore # of SNR training symbols
            gt_SnrConfig.s_AlgLog2NumFramesToAccum = gs_SnrAlgLog2NumFramesToAccum_Saved;
            gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_DONE;
         }
         else
         {
            // run the background task to compute the sum of SNR and search for the best SNR sum
            gs_RxBkgdProcessFlag = TRAINING_IN_PROGRESS;
            AddFunctionToBkgdFifo((PtrToBkgdFunc)SearchHighestSnrSum);

            gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_UPDATE;
         }
      }
      break;

   case SNR_BASED_FRAME_ALIGN_UPDATE:

      if (gs_RxBkgdProcessFlag == TRAINING_DONE)
      {
         if (gs_SyncSearchMode == 0)    // Midpoint
         {
            gs_SyncSearchMode  = 1;     // Change over to positive direction
            gs_AlignmentOffset = -gt_SnrFrameAlignConfig.s_MedleyAlignOffsetStep;

            // Update synch point index
            gs_CurrSynchPointIdx++;
         }
         else if (gs_SyncSearchMode == 1) // Positive direction
         {
            // If we've reached the maximum offset in this direction or if
            // the current channel capacity is more than 12.5% BELOW best
            // capacity so far, don't advance offsets in this direction any
            // further.
            if ((gs_CurrSynchPointIdx == gt_SnrFrameAlignConfig.s_NumOfMedleyAlignPoints-1) ||
                  ((gl_BestSnrSum-gla_MedleySnrSum[gs_CurrSynchPointIdx]) > (gl_BestSnrSum >> 3)))
            {
               // now go in negative direction, starting at midpoint-1
               gs_SyncSearchMode  = -1; // Change over to negative direction
               gs_AlignmentOffset = (1+gs_CurrSynchPointIdx-((gt_SnrFrameAlignConfig.s_NumOfMedleyAlignPoints) >> 1))*gt_SnrFrameAlignConfig.s_MedleyAlignOffsetStep;
               gs_CurrSynchPointIdx = ((gt_SnrFrameAlignConfig.s_NumOfMedleyAlignPoints) >> 1)-1;
            }
            else
            {
               // Update synch point index
               gs_CurrSynchPointIdx++;
            }
         }
         else if (gs_SyncSearchMode == -1) // Negative direction
         {
            // If we've reached the maximum offset in this direction or if
            // the current channel capacity is more than 12.5% BELOW best
            // capacity so far, don't advance offsets in this direction any
            // further.
            if ((gs_CurrSynchPointIdx == 0) ||
                  ((gl_BestSnrSum-gla_MedleySnrSum[gs_CurrSynchPointIdx]) > (gl_BestSnrSum >> 3)) )
            {
               gs_SyncSearchMode  = 2;           // indicates search done.
               gft_RestoreOrigFdq = FALSE;

               // Compute the offset (relative to the current frame alignment)
               // which gives the highest SNR sum
               gs_AlignmentOffset = -(gs_BestSynchPointIdx-gs_CurrSynchPointIdx)*gt_SnrFrameAlignConfig.s_MedleyAlignOffsetStep;
               // Overwrite gs_CurrSynchPointIdx with gs_BestSynchPointIdx
               // so BgMedleyRotateFDQ() would rotate the FDQs associated with
               // gs_BestSynchPointIdx correctly.
               gs_CurrSynchPointIdx = gs_BestSynchPointIdx;

               // When the final medley FDQ/SNR measurement gets be skipt. Then the last
               // SNR for best sync point must be done with default high averaging.
               if (gs_MedleyFrameSynchEnableFlag & EN_MEDLEY_SKIP_FINAL_FDQSNR)
               {
                  // restore # of SNR training symbols
                  gt_SnrConfig.s_AlgLog2NumFramesToAccum = gs_SnrAlgLog2NumFramesToAccum_Saved;
                  //XDSLRTFW-3823 (start)
                  if(!(gs_MedleyFrameSynchEnableFlag & EN_MEDLEY_FINAL_SNR))
                  {
                      //XDSLRTFW-3823 (start)
                      // Validate NoOf Tone groups
                      // atleast one start idx,(in this case end indx will be 8192) from startidx to end idx constant SNR offset will be added.
                      // gta_SNROffset.s_NoOfToneGroups = 0 will disable SNR offset (by writing "0" value)
                      if(gta_SNROffset.s_NoOfToneGroups >= 1)
                      {
                         if(gta_SNROffset.s_NoOfToneGroups > SNR_OFFSET_TONEGROUPS)
                         {
                            gta_SNROffset.s_NoOfToneGroups = SNR_OFFSET_TONEGROUPS;
                         }
                         // Copy to FW variable to be used for SNR manipulation
                         int16 s_idx;
                         for(s_idx = 0;s_idx < gta_SNROffset.s_NoOfToneGroups;s_idx++)
                         {
                            gt_SNROffsetBitLoad[s_idx+1].s_ToneGroupStartIdx = gta_SNROffset.ta_SNROffset[s_idx].s_ToneGroupStartIdx;
                            gt_SNROffsetBitLoad[s_idx+1].s_ToneGroupSNROffset  = gta_SNROffset.ta_SNROffset[s_idx].s_ToneGroupSNROffset;
                         }
                       }
                       gt_SNROffsetBitLoad[0].s_ToneGroupStartIdx = 0;
                       gt_SNROffsetBitLoad[0].s_ToneGroupSNROffset  = 0;
                       gt_SNROffsetBitLoad[gta_SNROffset.s_NoOfToneGroups+1].s_ToneGroupStartIdx = RX_MAX_NUM_TONES;
                       gt_SNROffsetBitLoad[gta_SNROffset.s_NoOfToneGroups+1].s_ToneGroupSNROffset  = 0;
                       gs_ToneGroupIdx = 0; //XDSLRTFW-3823 (end)
                   }

               }
            }
            else
            {
               // Update synch point index
               gs_CurrSynchPointIdx--;
               gs_AlignmentOffset = gt_SnrFrameAlignConfig.s_MedleyAlignOffsetStep;
            }
         }

         gt_SnrFrameAlignConfig.s_AlgHandlerState = SNR_BASED_FRAME_ALIGN_SET;
      }
      break;

   case SNR_BASED_FRAME_ALIGN_DONE:
      break;

   } // switch (gt_SnrFrameAlignConfig.s_AlgHandlerState)

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : SearchHighestSnrSum
 *
 *  Description:  This function computes the sum of the SNRs over all the valid tones
 *   and searchs for the maximum SNR sum.
 *
 *  Prototype:  void SearchHighestSnrSum(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gl_BestSnrSum -- (O) the highest SNR sum
 *      gs_BestSynchPointIdx -- (O) realignment offset corresponding to the highest SNR sum
 *      gs_CurrSynchPointIdx -- (O) index of the current realignment offset
 *      gs_RxBkgdProcessFlag -- (I/O) flag to indicate the state of the RX background process
 *      gla_MedleySnrSum[] -- (O) an array to store the SYN sum for each synch point
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
//extern int16 gs_debug_count;
void SearchHighestSnrSum(void)
{

   int16 i, j;
   int32 l_SnrSum;

   //Compute the sum of the SNRs over all tones
   l_SnrSum = 0;
   for(j = 0; j<gs_NumOfRxBands ; j++)
   {
      for(i = gsa_RxBandLeftChannel[j]; i <= gsa_RxBandRightChannel[j]; i++)
      {
         // Cap snr at that for about 15 bits, and don't include
         // snrs below that supporting 1 bit.
         if (gpsa_MeasuredSnrBuf[i]> (57<<8))
         {
            l_SnrSum += (57<<8);
         }
         else if (gpsa_MeasuredSnrBuf[i] > (8<<8))
         {
            l_SnrSum += gpsa_MeasuredSnrBuf[i];
         }
      }
   }

   //Compare the SNR sum with the maxium SNR sum computed so far
   if(l_SnrSum > gl_BestSnrSum)
   {
      gl_BestSnrSum = l_SnrSum;
      gs_BestSynchPointIdx  = gs_CurrSynchPointIdx;
   }

   gla_MedleySnrSum[gs_CurrSynchPointIdx] = l_SnrSum;
   //gsa_RxHlog[gs_debug_count++]=(int16) (l_SnrSum>>8);

   gs_RxBkgdProcessFlag = TRAINING_DONE;
}


//CO HW build should use FdqHandler() loaded to the swappage ALGHD_3_V2
