/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   TimingAdvanceHandler.c
 *
 *   This file contains the routine that calculates and implements the
 *   initial timing advance.
 *
 *------------------------------------------------------------------------
 */

#include "common.h"
#include "gdata.h"
#include "TimingAdvanceHandler.h"
#include "IRI_sync.h"
#include "fifo.h"
#include "sys_const.h"
#include "SharedFuncs.h"
#include "CRI_Iof.h"
#include "SharedFuncs.h"
#include "str_memmap.h"

/*^^^
 *---------------------------------------------------------------------------------
 *
 *  Name :
 *
 *  Description:  This function is used to compute and save the timing advance
 *                debug information.
 *
 *  Prototype:  void SavePostTimeAdvanceInfo(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gs_TxTimerVal              -- (O) TX timer value (in DSP clock cycles)
 *      gs_TxTimerVal              -- (O) RX timer value (in DSP clock cycles)
 *      gs_TimingAdvanceFixedDelay -- (I) Sum of const TX and RX filter delay (in samples at 4096 IDFT sample rate)
 *      gs_TimingAdvanceTxIirDelay -- (I) TX IIR filter delay                 (in samples at 4096 IDFT sample rate)
 *      gs_TimingAdvanceRxIirDelay -- (I) RX IIR filter delay                 (in samples at 4096 IDFT sample rate)
 *
 *      gs_TimingAdvanceTrailIndex -- (I/O) the TA trail index
 *      gs_TxTimerValPostTA[]      -- (O) the TX timer value post applying TA
 *      gsa_TaTargetTrail[]        -- (O) the targeted TA
 *      gsa_TaActualTrail[]        -- (O) the actual TA
 *
 *----------------------------------------------------------------------------------
 *^^^
 */
void SavePreTimeAdvanceInfo(void)
{
   int16 s_tx_lead_rx_amount, i;

   i = TX_LOG2_MAX_FFT_LENGTH-gs_TxLog2IfftLength;

   s_tx_lead_rx_amount = ((gl_txRxFrameOffset & V_FRAME_SKEW_MASK) << i);

   // Save the debug information
   // (Log the target TA value expressed in the x tone IFFT sampling rate, i.e. according to the define TX_LOG2_MAX_FFT_LENGTH)
   gsa_TaAdjustTrailPre[gs_TimingAdvanceTrailIndex] = (gs_TimingAdvance << i);
   gsa_TaTargetTrail[gs_TimingAdvanceTrailIndex] = (gs_TargetTimingAdvance << i);
   gsa_TaActualTrailPre[gs_TimingAdvanceTrailIndex] = s_tx_lead_rx_amount;
}

void SavePostTimeAdvanceInfo(void)
{
   int16 s_tx_lead_rx_amount, i;
   int16 s_TimingAdvanceAtU2;
//   FlagT ft_SkipCpAdjustSave;

   i = TX_LOG2_MAX_FFT_LENGTH-gs_TxLog2IfftLength;

//   ft_SkipCpAdjustSave = gft_SkipCpAdjust;
//   gft_SkipCpAdjust = 1;
//   s_tx_lead_rx_amount = CalcTxRxFrameOffset();
//   gft_SkipCpAdjust = ft_SkipCpAdjustSave;
   s_tx_lead_rx_amount = ((gl_txRxFrameOffset & V_FRAME_SKEW_MASK) << i);
   // XDSLRTFW-3737 (Start_End)
   s_TimingAdvanceAtU2 = s_tx_lead_rx_amount - ((gs_TimingAdvanceFixedDelay + gs_TimingAdvanceTxIirDelay + gs_TimingAdvanceRxIirDelay) & 0xFFFE);

   // Save the debug information
   // (Log the target TA value expressed in the x tone IFFT sampling rate, i.e. according to the define TX_LOG2_MAX_FFT_LENGTH)
   gsa_TaAdjustTrail[gs_TimingAdvanceTrailIndex] = (gs_TimingAdvance << i);
//   gsa_TaTargetTrail[gs_TimingAdvanceTrailIndex] = (gs_TargetTimingAdvance << i);
   gsa_TaActualTrail[gs_TimingAdvanceTrailIndex++] = s_TimingAdvanceAtU2;
}


/*^^^
 *---------------------------------------------------------------------------------
 *
 *  Name : TimingAdvanceHandler
 *
 *  Description:  This function is used to compute the timing advance applied at CPE
 *
 *  Prototype:  void TimingAdvanceHandler(void)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gs_TimingAdvanceHandlerState -- (I/O) substate number of this handler
 *      gs_TxTimerVal                -- (O) TX timer value (in DSP clock cycles)
 *      gs_TxTimerVal                -- (O) RX timer value (in DSP clock cycles)
 *      gs_TimingAdvance             -- (I/O) either input TA specified by CO or the output TA computed by the CPE
 *                                      (expressed in samples at the US IDFT sampling rate)
 *      gs_TimingAdvanceFixedDelay   -- (I) Sum of const TX and RX filter delay (in samples at 4096 IDFT sample rate)
 *      gs_TimingAdvanceTxIirDelay   -- (I) TX IIR filter delay                 (in samples at 4096 IDFT sample rate)
 *      gs_TimingAdvanceRxIirDelay   -- (I) RX IIR filter delay                 (in samples at 4096 IDFT sample rate)
 *      gs_TargetTimingAdvance       -- (I) the targeted TA at U2 interface
 *                                         (expressed in samples at the US IDFT sampling rate)
 *
 *      gs_TimingAdvanceTrailIndex   -- (I/O) the TA trail index
 *      gsa_TaAdjustTrail[]          -- (O) the Tx/Rx frame offset adjustment
 *
 *----------------------------------------------------------------------------------
 *^^^
 */
void TimingAdvanceHandler(void)
{
   static int16 s_wait_cnt;                               // Note: static is needed here, because the variable is used over symbols.
                                                          //       Normally a global variable should be used!

   switch(gs_TimingAdvanceHandlerState)
   {
   case TIMING_ADVANCE_INIT:

      //Save the initial value
      gs_TimingAdvance_sav = gs_TimingAdvance;

      // queue loading function to measure Tx/Rx Counters, which
      // are returned in gl_TxTimerVal and gl_RxTimerVal
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, GetTxRxCounters);

      // Transition to next substate to compute the TA
      gs_TimingAdvanceHandlerState = TIMING_ADVANCE_CALCULATION;

      break;

   case TIMING_ADVANCE_CALCULATION:

      SavePreTimeAdvanceInfo();

      if(gs_TimingAdvance == 0)
      {
         int16 i;

         // NOTE: entering this substate,
         //       - gs_TimingAdvance and gs_TargetTimingAdvance are in samples at the US IDFT sampling rate
         //       - s_TxRxOffset and TotalFilterDelay = (gs_TimingAdvanceFixedDelay + gs_TimingAdvanceTxIirDelay + gs_TimingAdvanceRxIirDelay)
         //         are in samples at 4096 IDFT sample rate
         //
         // First convert all variables in samples expressed at the 4096 IDFT sample rate.
         // Note: In VDSL2, gs_TargetTimingAdvance is obtained either from
         //               - O-Signature message or
         //               - Default value (if the O-Signature sets the initial TA value to 0x7FFF).
         //
         i = TX_LOG2_MAX_FFT_LENGTH-gs_TxLog2IfftLength;

         // Calculate the Tx/Rx offset in samples expressed at the 4096 IDFT sample rate,
         // where the frame start is assumed to be the first sample of the cyclic prefix.
         // Note: gs_TimingAdvance = 0,               the value is already adjusted for the Rx path delay, Tx path delay, and channel delay!
         //                        = negative value - TX frame start is ahead the RX frame start
         //                        = positive value - RX frame start is ahead the TX frame start
         gs_TimingAdvance = CalcTxRxFrameOffset();

         // Shift the previous normalized values back to the real used IFFT, i.e.
         // convert gs_TimingAdvance back to the US sampling rate.
         if(i>0)
         {
            gs_TimingAdvance += (1<<(i-1));   //perform rounding
            gs_TimingAdvance >>= i;
         }
      }
      else if(gs_DbgTimingAdvanceOffset == 0x7FFF)
      {
         // Disable TA for test purpose
         gs_TimingAdvance = 0;
      }

      // Queue loading function to make Timing Advance adjustment
      // If gs_TimingAdvance > 0, this implies that TX frame should be advanced to get correct TA value,
      //                          thus we need to shorten this frame (the next frame is the one where TA takes effect).
      // If gs_TimingAdvance < 0, this implies that TX frame should be delayed to get correct TA value,
      //                          thus we need to lengthen this frame.
      gs_TxFrameAlignRemain = 0;
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, AdjustTxTimingAdvance);

      gs_TimingAdvanceHandlerState = TIMING_ADVANCE_RESET;

      break;

   case TIMING_ADVANCE_RESET:

      //For the 6.2 HW, it is not allowed to make a strymon frame less than IFFT size
      //so it may take more than one frame to do TA adjustment if gs_TimingAdvance is greater than
      //CE Length. Note the variable gs_TxFrameAlignRemain stores the remaining adjustment
      //after each call to AdjustTxTimingAdvance()
      if(gs_TxFrameAlignRemain != 0)
      {
         // queue loading function to make Timing Advance adjustment
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, AdjustTxTimingAdvance);
      }
      else
      {
         // queue loading function to make Timing Advance adjustment
         AddFunctionToFifo(gp_TxLoadingFunctionFifo, ResetTxTimingAdvance);

         s_wait_cnt = 0;
         gs_TimingAdvanceHandlerState = TIMING_ADVANCE_WAIT;
      }

      break;

   case TIMING_ADVANCE_WAIT:
      //In the 6.2 build, since the strymon and DSP task run asynchronously
      //we want to wait for a few of frames until the frame alignment is done
      //In the 5.0 build, this probably doesn't matter, but should not hurt
      s_wait_cnt++;
      if(s_wait_cnt > 4)
      {
         //go to the next substate
         gs_TimingAdvanceHandlerState = TIMING_ADVANCE_GET_COUNTERS_FOR_TEST;
      }

      break;

   case TIMING_ADVANCE_GET_COUNTERS_FOR_TEST:

      // queue loading function to measure Tx/Rx Counters, which
      // are returned in gs_TxTimerVal and gs_RxTimerVal
      AddFunctionToFifo(gp_TxLoadingFunctionFifo, GetTxRxCounters);

      // Transition to next substate
      gs_TimingAdvanceHandlerState = TIMING_ADVANCE_CHECK;
      break;

   case TIMING_ADVANCE_CHECK:
      // save debug information
      SavePostTimeAdvanceInfo();

      // Transition to next substate
      gs_TimingAdvanceHandlerState = TIMING_ADVANCE_DONE;
      if(gs_ReduceInitFrameOffsetCnt > 0)
      {
         //Do another time to reduce the error of frame offset estimate
         gs_ReduceInitFrameOffsetCnt--;
         gs_TimingAdvance = gs_TimingAdvance_sav;
         gs_TimingAdvanceHandlerState = TIMING_ADVANCE_INIT;
      }

      break;
   }
}

