/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;   File Name: ToneReorder.c
;
;   This file contains routine of perform tone ordering.
;
;***************************************************************************/

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "ToneReorder.h"
#ifdef HW_TRT
#include "Bitload_support.h"
#endif

/*****************************************************************************
;   Prototype:
;      FlagT ToneOrdering(int16 *psa_BAT, uint8 *puca_ExtGains, int16 s_NumTones,
;                         int16 s_MaxBitPerTone, int16 *psa_ToneIndices)
;
;   This subroutine performs the tone ordering.
;
;
;   Input Arguments:
;      s_direction       -- TX/RX
;      psa_BAT          -- pointer to the Bit Allocation Table
;
;      puca_ExtGains    -- pointer to the Extended Gains Table (for use
;                     of direction bit
;
;      s_NumTone       -- total number of frequency tones or bands (DC and
;                           Nyquist together are considered to be one tone)
;
;      s_MaxBitsPerTone -- maximum number of bits per tone
;
;   Output Arguments:
;      psa_ToneIndices  -- pointer to an array of tone indices with the
;                           number of bits per tone in ascending order (for
;                           those tones with the same number of bits, the
;                           tone indices are in ascending order)
;
;   Global Variables Used:
;
;****************************************************************************/

FlagT ToneOrdering(int16 s_direction, uint8 *puca_BAT, uint8 *puca_ExtGains, int16 s_NumTones, int16 s_MaxBitsPerTone, int16 *psa_ToneIndices)
{
   int16 i, j, k, s_ToneOrderType;
   int16 sa_count[RX_MAX_BITS_PER_TONE+1];
   int16 sa_idx[RX_MAX_BITS_PER_TONE+1];
   int16 s_NumOfBands, sa_BandLeftChannel[MAX_NUM_BANDS], sa_BandRightChannel[MAX_NUM_BANDS];
   int16 s_NumTcmContrTone, s_ToneSaved, s_NumTonesUsed, s_StartIndex, s_EndIndex, s_TotalTones, s_MinToneForReorder;
   FlagT ft_TcmFlag, ft_ToneOrderStatus = SUCCEED;

#ifdef HW_TRT   // Dummy variables required as parameters for GetBatStats
   int16 s_Num1bitTones, s_Tcm_Ovhd;
   int32 l_ActualDeltaSumLp;
#endif
   int16 s_TxNumTones;

   //==========================================================================
   // set path-dependent variables
   //==========================================================================
   if (s_direction == TX)
   {
      s_NumOfBands = gs_NumOfTxBands;
      ft_TcmFlag = gft_TxTcmFlag;
      gs_TxNumTonesUsed = 0;
      for (i=0; i<s_NumOfBands; i++)
      {
         sa_BandLeftChannel[i] = gsa_TxBandLeftChannel[i];
         sa_BandRightChannel[i] = gsa_TxBandRightChannel[i];
         gs_TxNumTonesUsed += gsa_TxBandRightChannel[i] - gsa_TxBandLeftChannel[i] + 1;
      }
      s_NumTonesUsed = gs_TxNumTonesUsed;
      s_ToneOrderType = gs_TxToneOrderType;
   }
   else
   {
      s_NumOfBands = gs_NumOfRxBands;
      ft_TcmFlag = gft_RxTcmFlag;
      gs_RxNumTonesUsed = 0;
      for (i=0; i<s_NumOfBands; i++)
      {
         sa_BandLeftChannel[i] = gsa_RxBandLeftChannel[i];
         sa_BandRightChannel[i] = gsa_RxBandRightChannel[i];
         gs_RxNumTonesUsed += gsa_RxBandRightChannel[i] - gsa_RxBandLeftChannel[i] + 1;
      }
      s_NumTonesUsed = gs_RxNumTonesUsed;
      s_ToneOrderType = gs_RxToneOrderType;
   }

   //==========================================================================
   // count # of tones in each bit-assignment goup
   //==========================================================================
   // initialize tone counts
   memset(sa_count, 0, sizeof(int16)*(RX_MAX_BITS_PER_TONE+1));

   for (j=0; j<s_NumOfBands; j++)
   {
      for (i=sa_BandLeftChannel[j]; i<=sa_BandRightChannel[j]; i++)
      {
         // process tones for a given direction
         if ((puca_ExtGains[i] & MASK_BIT4) == (s_direction << 4))
         {
            k = (int16)puca_BAT[i];
            sa_count[k]++;
         }
         else
         {
            ft_ToneOrderStatus = FAIL;
         }
      }
   }

   //==========================================================================
   // determine starting index of TRT
   //==========================================================================
   // get the first tone location for each bit-assignment group
   // for RX path: reserve first two TRT indices for out-band 0-bit tones (tone 0, tone 1)
   // (1) to ensure TRT starts with 0-bit tone in case all in-band tones are loaded
   // (2) to compute SNR for all in-band tones including unloaded tones
   // (SNR is not calculated correctly for the in-band 0-bit tone if it is paired with the first nonzero-bit tone in TCM mode)
   // for TX path: no need to reserve out-band 0-bit tone even though all tones are loaded
   // since zero-bit unused tones precede TX TRT
   if (s_direction == TX)
   {
      sa_idx[0] = gs_TxNumTones - s_NumTonesUsed;
      if (gft_TxTcmFlag)
      {
         gus_Tx_Tcm_Num1bits = sa_count[1];
      }
      else
      {
         gus_Tx_Tcm_Num1bits = 0;
      }
   }
   else
   {
      sa_idx[0] = RX_NUM_OUT_BAND_ZERO_BIT_TONE;
      psa_ToneIndices[sa_idx[0]-2] = RX_OUT_BAND_ZERO_BIT_TONE_1; // tone 0
      psa_ToneIndices[sa_idx[0]-1] = RX_OUT_BAND_ZERO_BIT_TONE_2; // tone 1
      if (gft_RxTcmFlag)
      {
         gus_Rx_Tcm_Num1bits = sa_count[1];
      }
      else
      {
         gus_Rx_Tcm_Num1bits = 0;
      }
   }

   //==========================================================================
   // move all 1-bit tones at the end of TRT if TCM is enabled
   //==========================================================================
   if (ft_TcmFlag)
   {
      // put 1-bit tone at the end
      sa_idx[1] = sa_idx[0] + s_NumTonesUsed - sa_count[1];
      sa_idx[2] = sa_idx[0] + sa_count[0];
      // error if # of 1-bit tones is odd
      if (sa_count[1] & 0x1)
      {
         ft_ToneOrderStatus = FAIL;
      }
      s_MinToneForReorder = 2; // tone less than 2 will be reordered
   }
   else
   {
      sa_idx[1] = sa_idx[0] + sa_count[0];
      s_MinToneForReorder = 1; // tone less than 1 will be reordered
   }

   //==========================================================================
   // determine RX_MIN_TONE
   // include 0-bit tones for SNR calculation for all in-band tones
   //==========================================================================
   if (s_direction == RX)
   {
      gs_RxMinTone = sa_idx[0];
      gs_RxMaxTone = gs_RxNumTonesUsed + 1; // assume 2 extra outband 0-bit tones
      gus_Rx_1Bit_Index = 0;
      if (ft_TcmFlag)
      {
         gs_RxMinTone = sa_idx[2];
         s_NumTcmContrTone = gs_RxNumTonesUsed - sa_count[0] - (sa_count[1] >> 1);
         if (s_NumTcmContrTone & 0x1)
         {
            gs_RxMinTone--;
         }
         // include all 0-bit tones so that SNR calculation works on these tones
         if (gs_RxMinTone & 0x1)
         {
            gs_RxMinTone = 1;
         }
         else
         {
            gs_RxMinTone = 0;
         }
         // save the first 1-bit tone index
         if (sa_count[1])
         {
            gus_Rx_1Bit_Index = sa_idx[1];
         }
      }
      else
      {
         // make MAX_TONE - MIN_TONE + 1 even
         // JBK ??? - need to verify TCM case later
         s_TotalTones = gs_RxMaxTone - gs_RxMinTone + 1;
         if (s_TotalTones & 0x1)
         {
            gs_RxMaxTone++;
         }
      }
   }

   //==========================================================================
   // fill in TRT
   //==========================================================================
   switch (s_ToneOrderType)
   {

   case NORMAL_TONE_ORDER:

      for (k=s_MinToneForReorder+1; k<=s_MaxBitsPerTone; k++)
      {
         j = k-1;
         sa_idx[k] = sa_idx[j] + sa_count[j];
      }

      // sort indices according to tone ordering rule
      for (j=0; j<s_NumOfBands; j++)
      {
         for (i=sa_BandLeftChannel[j]; i<=sa_BandRightChannel[j]; i++)
         {
            k = (int16)puca_BAT[i];
            psa_ToneIndices[sa_idx[k]] = i;
            sa_idx[k]++;
         }
      }

      break;

   case NO_TONE_ORDER:

      // sort indices according to tone ordering rule
      for (j=0; j<s_NumOfBands; j++)
      {
         for (i=sa_BandLeftChannel[j]; i<=sa_BandRightChannel[j]; i++)
         {
            k = (int16)puca_BAT[i];
            if (k > s_MinToneForReorder)
            {
               k = s_MinToneForReorder;   // set TRT based on physical tone order for tones loaded more than 2-bit
            }
            psa_ToneIndices[sa_idx[k]] = i;
            sa_idx[k]++;
         }
      }

      break;
   }

   //==========================================================================
   // post processing TRT for RX path
   // reshuffle 0-bit tones if TCM is enabled
   //==========================================================================
   if (s_direction == RX)
   {
      if (ft_TcmFlag)
      {
         // get the starting index of 0-bit tone
         sa_idx[0] -= sa_count[0];
         // put two out-band 0-bit tones to the start and the end of 0-bit tone group
         // since these tones won't be part of SNR calculation
         // (1) the first 0-bit tone can be excluded from MIN-MAX tone range depending on
         // the number of in-band 0-bit tones
         // (2) the last 0-bit tone may generate bad SNR in Trellis mode if it is paired
         // with the first non-zero bit tone in Trellis decoding
         // note that we have total of (sa_count[0]+2) 0-bit tones in RX TRT
         j = sa_idx[0] - 1;
         s_ToneSaved = psa_ToneIndices[j];
         for (i=0; i<sa_count[0]; i++)
         {
            psa_ToneIndices[j] = psa_ToneIndices[j+1];
            j++;
         }
         psa_ToneIndices[j] = s_ToneSaved;
      }
      if (gs_RxNumTones > gs_TxNumTones)
         s_TxNumTones = gs_RxNumTones;
      else
         s_TxNumTones = gs_TxNumTones;
      // fill TRT with unused tones
      s_StartIndex = gs_RxNumTonesUsed + RX_NUM_OUT_BAND_ZERO_BIT_TONE;
      s_EndIndex = s_TxNumTones - gs_TxNumTonesUsed - 1;
      if (FillUnusedToneInTrt(puca_ExtGains, psa_ToneIndices, s_StartIndex, s_EndIndex) == FAIL)
      {
         ft_ToneOrderStatus = FAIL;
      }
   }

#ifdef HW_TRT
   if (s_direction == RX)
   {
      GetBatStats(puca_BAT, &s_Num1bitTones, &s_Tcm_Ovhd, &l_ActualDeltaSumLp);
   }
#endif
   return (ft_ToneOrderStatus);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT FillUnusedToneInTrt(uint8 *puca_ExtGains, int16 *psa_ToneIndices,
*      int16 s_StartIndex, int16 s_EndIndex)
*
*   This function fills unused tones in TRT. Assume that all unused tones are
*   configured as RX tones (i.e., in direction bit in EGT).
*
*   Input Arguments:
*      puca_ExtGains: pointer to extended gain table
*      s_StartIndex: start index within TRT to fill unused tones
*      s_EndIndex: end index within TRT to fill unused tones
*
*   Output Arguments:
*      psa_ToneIndices: pointer to tone ordering table
*
*   Returns:
*      SUCCESS/FAIL
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

FlagT FillUnusedToneInTrt(uint8 *puca_ExtGains, int16 *psa_ToneIndices, int16 s_StartIndex, int16 s_EndIndex)
{
   int16 i, s_NumOfBands, s_BandLeftChannel, s_BandRightChannel;
   FlagT ft_Status = SUCCEED;

   s_NumOfBands = 0;
   s_BandLeftChannel = gsa_RxBandLeftChannel[0];
   s_BandRightChannel = gsa_RxBandRightChannel[0];

   for (i=0; i<gs_TxNumTones; i++)
   {
      // for all RX tones (including unused tones)
      if ((puca_ExtGains[i] & MASK_BIT4) == 0)
      {
         if ((i < s_BandLeftChannel) && (i != RX_OUT_BAND_ZERO_BIT_TONE_1) && (i != RX_OUT_BAND_ZERO_BIT_TONE_2))
         {
            psa_ToneIndices[s_StartIndex] = i;
            s_StartIndex++;
         }
         if (i == s_BandRightChannel)
         {
            s_NumOfBands++;
            if (s_NumOfBands == gs_NumOfRxBands)
            {
               s_BandLeftChannel = gs_TxNumTones;
               s_BandRightChannel = gs_TxNumTones;
            }
            else
            {
               s_BandLeftChannel = gsa_RxBandLeftChannel[s_NumOfBands];
               s_BandRightChannel = gsa_RxBandRightChannel[s_NumOfBands];
            }
         }
      }
   }

   if (s_StartIndex != s_EndIndex+1)
   {
      ft_Status = FAIL;
   }

   return (ft_Status);
}
