/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*****************************************************************************
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   decimal_gain.c
;
;   dB-to-linear conversion
;
;****************************************************************************/

#include "common.h"
#include "dsp_op.h"
#include "mul.h"

/*^^^
*-------------------------------------------------------------------
*
*  void db_to_linear(int16 s_dB_input, int16 *ps_lin_mant, int16 *ps_lin_exp);
*
*   Description:
*      Computes the decimal equivalent of the fine gain in 1.15 format,
*      Assumes the the input gain_db is in 8.8 format and is within the
*      range [-128.0 dB, +127.0 dB].
*      Returns 10^(gain_db/20) as mantissa (Q1.15 format) and exponent (# left shifts)
*
*      Algorithm:
*         1)  Map input x to range [-6.0 dB, 0.0 dB])
*
*         2)   Polynomial Expansion Details:
*            The original coeffficients are : 0.0002    0.0064    0.1148    1.0000
*            To fit the 16 bit coef size they are stored as
*            c1*8*8*8  c2*8*8  c3*8  c4
*            0.0929    0.4078    0.9187    1.0000
*            0.0002    0.0064    0.1148    1.0000
*
*   Input Arguments:
*      s_dB_input -- input value in dB (Q8.8 format)
*
*   Ouput Arguments:
*      *ps_lin_mant -- ouput linear value mantissa (Q1.15 format)
*      *ps_lin_exp  -- ouput linear value exponent (# of left shifts)
*
*   Return:
*
*-------------------------------------------------------------------
*^^^
*/

//  note this is not 6*256 but round(20*log10(2)*256)
#define   LOG2x20_DB 1541   //   in Q8.8 format

// coefs for power seies approx of 10^(x/20)
int16 gsa_logexpansion_coefs[4]= { 3045    ,   13364  ,     30104    ,   32767};

void db_to_linear(int16 s_dB_input, int16 *ps_lin_mant, int16 *ps_lin_exp)
{
   int16 norm_dB_input, exp, n; // by keeping it as an int avoid sex instructions.
   int32 l_acc;
   int16 s_acc, *ps_coef;

   // Prescale input within range
   // Map input to range [-6,0], where the approximation is valid
   norm_dB_input = s_dB_input;   // by keeping it as an int avoid sex instructions.
   exp = 0;
   while (norm_dB_input < -LOG2x20_DB)
   {
      norm_dB_input += LOG2x20_DB;
      exp--;
   }
   while (norm_dB_input > 0)
   {
      norm_dB_input -= LOG2x20_DB;
      exp++;
   }

   // 3rd order power series expansion from -6 to 0
   // Calculate  ((((c3*x)+c2)*x)+c1)*x+c0      (ideal)
   // or as      ((((c3*x)/8 +c2)*x)/8 +c1)*x/8 +c0   (coef precision enhancement)
   // Each iteration is
   //  (s_acc[1.15] * norm_dB_input[8.8] )/256  + coef(i)
   // While it may appear that the product is 9.23 and will overflow when you take the
   // the bottom 16 bits -> it won't because the worst case s_db is -6 and the coefs are small
   ps_coef= gsa_logexpansion_coefs;
   s_acc = *ps_coef++;
   n=3;
   do
   {
      MULS16(l_acc, s_acc,norm_dB_input) ;
      l_acc= ((l_acc>>8)>>3);  // /256 to convert a into a 1.15 quantity.
      // /8 is to compensate for the coef scaling.
      s_acc = (int16) (l_acc + *ps_coef++ );
   }
   while(--n);

   // return
   *ps_lin_mant = s_acc;
   *ps_lin_exp = (int16)exp;
}

/*^^^
*-------------------------------------------------------------------
*
*  int16 DecimalGain(int16 s_gain_db);
*
*   Description:
*      Computes the decimal equivalent of the fine gain in 3.13 format,
*      Assumes the the input gain_db is in 8.8 format and is within the
*      range [-14.5 dB, +2.5 dB].
*      Returns 10^(gain_db/20) in 3.13 format (i.e. 8192 is eqv to 1.0)
*
*      Algorithm:
*      OLD IMPLEMENTATION:
*          1) Map input x to range [256, 768] (=[+1.0 dB, 3.0 dB])
*
*          2) Approximates 10^(x/20) in the desired range by
*             p(x) = 0.00104309497*x^2 + 3.57736185790*x + 8208.07332753051
*
*          3) Adjust approximation if original input x was not in
*             range [256, 768]
*      NEW IMPLEMENTATION:
*         Calls more generic db_to_linear()
*
*   Input Arguments:
*      s_gain_db -- file gain in dB (Q8.8 format)
*
*   Return:
*      decimal gain (in Q3.13 format)
*
*-------------------------------------------------------------------
*^^^
*/
int16 DecimalGain(int16 s_gain_db)
{
   int16 s_DecimalValue;

   if (s_gain_db == 0)
   {
      s_DecimalValue = UNITY_FINEGAIN_LIN;
   }
   if (s_gain_db == NEG_INFINITY_DB)
   {
      s_DecimalValue = 0;
   }
   // Check that input is within allowed range of [-14.5dB, +2.5dB].
   else if (s_gain_db >= MAX_FINEGAIN_DB)
   {
      s_DecimalValue = MAX_FINEGAIN_LIN;
   }
   else if (s_gain_db <= MIN_FINEGAIN_DB)
   {
      s_DecimalValue = MIN_FINEGAIN_LIN;
   }
   else
   {
      int16 s_mant, s_exp;
      db_to_linear(s_gain_db, &s_mant, &s_exp);
      // s_exp should always be < 2 since fgain range is [-14.0, 2.5] dB
      s_exp = -s_exp;   // To convert to # of right shifts
      s_exp += 2;      // To convert from Q1.15 to Q3.13 format
      // round result and return
      s_DecimalValue = (int16) ((s_mant+((1<<s_exp)>>1)) >> s_exp);
   }

   return (s_DecimalValue);
}
