/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*****************************************************************************
;
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   fdq.c
;
;   Routine to perform frequency-domain equalization (FDQ) initialization.
;
;****************************************************************************/

#include "common.h"
#include "dsp_op.h"
#include "gdata.h"
#include "SharedFuncs.h"
#include "IRI_Iof.h"
#include "fdq.h"
#include "cosine.h"
#include "ghs.h"
#include "FdqHandler.h"

// Local functions.
void FDQcoef(int16 *psa_in_tones);
void FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp);


/*****************************************************************************
;   Subroutine Name: FDQTrain(void)
;
;   Description:
;      This routine uses the accumulated frequency domain frames and
;      calls a subroutine to calculate the FDQ coefficients.
;
;   Prototype:
;      void FDQTrain(void);
;
;   Input Arguments:
;      none
;
;   Output Arguments:
;      none
;
;   Return Value:
;      none
;
;   Global Variables:
;      gpsa_RxRepBuf - received representative frame from FFT output
;                             (this array of data is lost in this program)
;      gsa_pre_FDQ_coef[] (O) -- FDQ coefficient matissas stored in the order of:
;            {Wr[0], Wi[0], Wr[1], Wi[1], ..., Wr[2*RX_NUM_TONES-2], Wi[2*RX_NUM_TONES-1]}
;      gsa_pre_FDQ_exp (O) -- FDQ coefficient exponent stored in the order of
;            {Ws[0], Ws[1], ..., Ws[RX_NUM_TONES-1]
;      gsa_RxRefTones[]   -- Rx reference tones
;
;****************************************************************************/
void FDQTrain(void)
{
   /* ==================================================================================== */
   /* Compute FDQ coefficients */
   /* ==================================================================================== */
   FDQcoef(gpsa_RxRepBuf);

   guc_FdqTrainingState = TRAINING_DONE;
   return;
}


/*****************************************************************************
;   Subroutine Name: FDQcoef(void)
;
;   Description:
;      This routine calculates the subset of FDQ coefficients.
;
;      For non-message-modulated tones k*10, k*10+2, k*10+4, k*10+6, k*10+8, k*10+9:
;      reference tone=[0x2000, 0x2000] (note that tones have been descrambled).
;
;      For message-modulated tones k*10+1, k*10+3, k*10+5, k*10+7:
;      First assume their FDQ coefficients are the same as their preceding neighbors,
;      e.g., assume the FDQ coefficient of tone 41 is the same as that of tone 40.
;      Copying and FDQ loading are done in the state machine. Then for the purpose of
;      frame accumulation, each message-modulated tone in every frame is rotated to the
;      first quadrant. Therefore, When calling this routine,
;      the averaged frame has the reference tone [0x2000, 0x2000].
;      FDQ coefficients found this way are multiplied by the coarse FDQ coefficients.
;
;   Prototype:
;      void FDQcoef(int16 *psa_in_tones);
;
;   Input Arguments:
;      *psa_in_tones
;
;   Output Arguments:
;      none
;
;   Return Value:
;      none
;
;   Global Variables:
;      gsa_RxBandLeftChannel[], gsa_RxBandRightChannel[], gs_NumOfRxBands, gs_LeftChannel
;
;      gsa_pre_FDQ_coef[RX_NUM_TONES*2] (O) -- FDQ coefficient matissas stored in the order of:
;            {Wr[0], Wi[0], Wr[1], Wi[1], ..., Wr[2*RX_NUM_TONES-2], Wi[2*RX_NUM_TONES-1]}
;      gsa_pre_FDQ_exp[RX_NUM_TONES] (O) -- FDQ coefficient exponent stored in the order of
;            {Ws[0], Ws[1], ..., Ws[RX_NUM_TONES-1]
;
;****************************************************************************/
void ApplyFDQAdjustment(int16 *psa_FDQ_coef, int16 *psa_FDQCoefAdjustment, uint8 *puc_FDQ_exp, uint8 uc_FDQExpAdjustment);
void InterpFdq(int16 *psa_FDQ_coef, uint8 *puca_FDQ_exp);
void ApplyFdqExpReduction(int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp, int16 s_exp_reduction);

void FDQcoef(int16 *psa_in_tones)
{
   int16 i, m, m2, k;
   int16 s_cluster_index, s_1st_cluster_index, sa_FDQCoefAdjustment[2];
   int16 sa_ref_tones[2];
   uint8 uc_FDQExpAdjustment;


   //Compute the first cluster index by floor(gs_LeftChannel/10)
   s_1st_cluster_index = s_cluster_index = ClusterIndex(gs_LeftChannel);

   for(i = gs_LeftChannel; i <= gs_RightChannel; i++)
   {
      k = i - s_cluster_index*10;
      m = i-gs_LeftChannel;
      m2 = m << 1;

      sa_ref_tones[0]=gs_RxRefToneLevel;
      sa_ref_tones[1]=gs_RxRefToneLevel;

      // Need to figure the reference constellation for each tone for this FDQ training (referenced based)!
      // Note: It is only allowed to run HW for redoing the quadrant scrambler (PRBS).
      if (gs_FDQTrainType == MEDLEY_FDQ_TRAIN_ONEPASS)
      {
         {
            int16 k_ref;

            // Set it to 0 by default, which corresponds to (0x2000, 0x2000)
            k_ref = 0;

            // If we send 1 byte per Medley frame, then tones 10N+1, 10N+3, 10N+5 and 10N+7 contains one Flag 0x7E.
            // Note: G.993.2, Table 12-37 or 12-45.
            //       SOC IDEL Byte       0x7E
            //       SOC MSG  Bits      | b7b6 | b5b4 | b3b2 | b1b0 |
            //       Byte bits Binary   |  01  |  11  |  11  |  10  |
            //                 DEC      |   1  |   3  |   3  |   2  |
            //       Tone cluster (k)   |10N+7 |10N+5 |10N+3 |10N+1 |
            //
            //       As it can be seen above the 0x7E is generating for odd-tones 3 constellations.
            //       The fourth constellation is coming from the even tones.
            //
            // If we send 2 bytes per Medley frame, then tones 10N+1, 10N+2, 10N+3 and 10N+4 contains first  Flag 0x7E,
            //                                           tones 10N+6, 10N+7, 10N+8 and 10N+9 contains second Flag 0x7E.
            // Note:
            //         Tone cluster (k)            |        SOC MSG  Bits           | SOC IDEL Bytes 0x7E7E
            //     ----------------------------------------------------------------------------------------
            //      5, 10, 15, ..., 5 n, ...       |       00                       |         00
            //      1, 11, 21, ..., 10 n + 1, ...  |   SOC message bits  0 and  1   |         10
            //      2, 12, 22, ..., 10 n + 2, ...  |   SOC message bits  2 and  3   |         11
            //      3, 13, 23, ..., 10 n + 3, ...  |   SOC message bits  4 and  5   |         11
            //      4, 14, 24, ..., 10 n + 4, ...  |   SOC message bits  6 and  7   |         01
            //      6, 16, 26, ..., 10 n + 6, ...  |   SOC message bits  8 and  9   |         10
            //      7, 17, 27, ..., 10 n + 7, ...  |   SOC message bits 10 and 11   |         11
            //      8, 18, 28, ..., 10 n + 8, ...  |   SOC message bits 12 and 13   |         11
            //      9, 19, 29, ..., 10 n + 9, ...  |   SOC message bits 14 and 15   |         01

            if(gs_RxNumBytesPerSymbol == 2)
            {
               // Get the new cluster ref-tone index.
               // Tone cluster (10n + k):
               //    k     |    k_ref
               // ---------------------
               //    0     |      0      (same coding as for one byte)
               //    1     |      1      (same coding as for one byte)
               //    2     |      2      (same coding as for one byte)
               //    3     |      3      (same coding as for one byte)
               //    4     |      4      (same coding as for one byte)
               //    5     |      0
               //    6     |      1
               //    7     |      2
               //    8     |      3
               //    9     |      4
               //
               k_ref = k;
               if(k >= 5)
               {
                  k_ref = (k_ref - 5);
               }
            }
            else if((gs_RxNumBytesPerSymbol == 1) &&
                    ((k & 1) == ODD_CLUSTER_INDEX) &&
                    (k != TONE_CLUSTER_INDEX9))
            {
               // Get the new cluster ref-tone index.
               // Tone cluster (10n + k):
               //    k+1   |    k_ref
               // ---------------------
               //    1+1   |      1
               //    3+1   |      2
               //    5+1   |      3
               //    7+1   |      4
               //
               k_ref = ((k+1) >> 1);
            }

            // For k_ref == 0 or 5 the ref-tone will not be changed, i.e. constellation 00 (Quadrant 0).
            // Note: 5 only for 2 bytes per Symbol!
//            sa_ref_tones[0] = gs_RxRefToneLevel;
//            sa_ref_tones[1] = gs_RxRefToneLevel;

            if(k_ref == 1)
            {
               // Constellation 10 (Quadrant 1)
               sa_ref_tones[0] = -gs_RxRefToneLevel;
               // sa_ref_tones[1] = gs_RxRefToneLevel;
            }
            else if(k_ref & 0x2)
            {
               // Constellation 11 (Quadrant 2)
               // For k_ref = 2 or 3.
               sa_ref_tones[0] = -gs_RxRefToneLevel;
               sa_ref_tones[1] = -gs_RxRefToneLevel;
            }
            else if(k_ref == 4)
            {
               // Constellation 01 (Quadrant 3)
               // sa_ref_tones[0] = gs_RxRefToneLevel;
               sa_ref_tones[1] = -gs_RxRefToneLevel;
            }
         }
      } // end of if (gs_FDQTrainType==MEDLEY_FDQ_TRAIN_ONEPASS)
//    else
         // Fix the reference tone to the first quadrant
         // - non-data tones are rotated to the first quadrant by the hardware PRBS
         // - data tones are rotated to first quadrant by the RotateDataTones function
         // in both cases, the tone is rotated prior to accumulation


      // If the given tone has zero energy, set the FDQ for set channel to be zero)
      // Note: Checking of Real + Imag part of the tone!
      if((psa_in_tones[m2] == 0) && (psa_in_tones[m2+1] == 0))
      {
         guca_pre_FDQ_exp[m]=0;
         gsa_pre_FDQ_coef[m2] = 0;
         gsa_pre_FDQ_coef[m2+1] = 0;
      }
      else
      {
         //
         if (gs_FDQTrainType == MEDLEY_FDQ_TRAIN_ONEPASS)
         {
            // all the tones are treated in the same way)
            FDQcoefPerTone(&(psa_in_tones[m2]), sa_ref_tones, &(gsa_pre_FDQ_coef[m2]), &guca_pre_FDQ_exp[m]);
         }
         // COARSE_FDQ_TRAIN is performed on the non-data tones of the TRAIN signal.
         // rotation is performed by PRBS HW
         else if (gs_FDQTrainType == COARSE_FDQ_TRAIN)
         {
            // XDSLRTFW-2576 (Start-End)
            // Non-Vectoring: if (k==0 || k==2 || k==4 || k==6 || k==8 || k==9)
            // Vectoring:     if (k==0 || k==2 || k==4 || k==6 || k==8)
            // Note1: To avoid additional if-conditions like
            //           if(((k&1) == 0) || ((gft_DSVectoringEnabled == FALSE) && (k==9)))
            //        the tone9 should always be treated as an odd tone. No differentiation
            //        between Non-Vectoring and Vectoring!
            // Note2: In diag mode, all the tones can be treated in the same way!
            if((k & 1) == EVEN_CLUSTER_INDEX)
            {
               FDQcoefPerTone(&(psa_in_tones[m2]), sa_ref_tones, &(gsa_pre_FDQ_coef[m2]), &guca_pre_FDQ_exp[m]);
            }
         }
         // FINE_FDQ_TRAIN is a decsion directed update of the data tone FDQs during the TRAIN signal.
         // The initial FDQ's were estimates based on the adjacent non-data tones (COARSE_FDQ_TRAIN)
         else if (gs_FDQTrainType == FINE_TUNE_FDQ_TRAIN)
         {
            // XDSLRTFW-2576 (Start-End)
            // Vectoring:     if (k==1 || k==3 || k==5 || k==7 || k==9)
            // Non-Vectoring: if (k==1 || k==3 || k==5 || k==7)
            // Note: To avoid additional if-conditions like
            //          if((((k&1) == 1) && ((gft_DSVectoringEnabled == TRUE) || (k != 9))) ||
            //       the tone9 should always be treated as an odd tone. No differentiation
            //       between Non-Vectoring and Vectoring!
            if(((k & 1) == ODD_CLUSTER_INDEX) ||
               (gs_FDQTrainTypeCntrlFlag & FDQ_TRAIN_TYPE_FINE_TUNE_ALL_TONE))
            {
               FDQcoefPerTone(&(psa_in_tones[m2]), sa_ref_tones, sa_FDQCoefAdjustment, &uc_FDQExpAdjustment);

               // ApplyFDQAdjustment computes the final FDQ coefficients by multiplying the initial FDQ and
               // the   corrective FDQ.
               // Given A +jB and C+jD, calculate E+jF = (A+jB)*(C+jD)
               // E = (A-B)*D + (C-D)*A
               // F = (A-B)*D + (C+D)*B
               ApplyFDQAdjustment(&gsa_pre_FDQ_coef[m2], sa_FDQCoefAdjustment, &guca_pre_FDQ_exp[m], uc_FDQExpAdjustment);
            }
         }
         // MEDLEY_FDQ_TRAIN is a decsion directed update of all FDQs during the MEDLEY signal.
         else if(gs_FDQTrainType == MEDLEY_FDQ_TRAIN)
         {
            //FDQcoefPerTone(&(psa_in_tones[m2]), &sa_ref_tones[k*2], &(gsa_pre_FDQ_coef[m2]), &guca_pre_FDQ_exp[m]);
            FDQcoefPerTone(&(psa_in_tones[m2]), sa_ref_tones, sa_FDQCoefAdjustment, &uc_FDQExpAdjustment);

            // ApplyFDQAdjustment computes the final FDQ coefficients by multiplying the initial FDQ and
            // the   corrective FDQ.
            // Given A +jB and C+jD, calculate E+jF = (A+jB)*(C+jD)
            // E = (A-B)*D + (C-D)*A
            // F = (A-B)*D + (C+D)*B
            ApplyFDQAdjustment(&gsa_pre_FDQ_coef[m2], sa_FDQCoefAdjustment, &guca_pre_FDQ_exp[m], uc_FDQExpAdjustment);
         }
      } //else

      if (k == TONE_CLUSTER_END_INDEX)
      {
         s_cluster_index++;
      }
   } //for(i = gs_LeftChannel; i <= gs_RightChannel; i++)


   // Compute the FDQs for data tones by interploting two adjacent tones
   // Note: This step is not needed in the diag mode!
   if (gs_FDQTrainType == COARSE_FDQ_TRAIN)
   {
      int16 sa_exp_magsq[3]= {0,0,0};
      int16 s_exp_reduction;
      uint32 ul_magsq;

      s_cluster_index = s_1st_cluster_index;
      for(i = gs_LeftChannel; i <= gs_RightChannel; i++)
      {
         k = (i - (s_cluster_index*10));
         m = i - gs_LeftChannel;
         m2 = m << 1;

         {
            // Calculate the magnitude-squared of the tone reception, and determine the exponent (+1)
            // of the mag-squared value.  Retain the exponents for three consecutive tones in sa_exp_magsq[].
            sa_exp_magsq[0] = sa_exp_magsq[1];   //previous tone
            sa_exp_magsq[1] = sa_exp_magsq[2];   //current tone

            //next tone
            ul_magsq = ((uint32) psa_in_tones[m2+2]*psa_in_tones[m2+2] + (uint32) psa_in_tones[m2+3]*psa_in_tones[m2+3])>>1;
            if (ul_magsq!=0)
            {
               sa_exp_magsq[2] = -norm_l((int32) ul_magsq);
            }
            else
            {
               sa_exp_magsq[2] = -32;
            }

            // If this is a left-edge tone, need to get the exponent of the mag-squared for the
            // current tone, which goes in sa_exp_magsq[1].
            if(i == gs_LeftChannel)
            {
               ul_magsq = ( (uint32) psa_in_tones[m2]*psa_in_tones[m2] + (uint32) psa_in_tones[m2+1]*psa_in_tones[m2+1] )>>1;
               if (ul_magsq!=0)
               {
                  sa_exp_magsq[1] = -norm_l((int32) ul_magsq);
               }
               else
               {
                  sa_exp_magsq[1] = -32;
               }
            }
         }

         // XDSLRTFW-2576 (Start-End)
         // Vectoring:     if (k==1 || k==3 || k==5 || k==7 || k==9)
         // Non-Vectoring: if (k==1 || k==3 || k==5 || k==7)
         // Note: To avoid additional if-conditions like
         //         if(((k&1)==1) && ((gft_DSVectoringEnabled == TRUE) || (k != 9)))
         //       the tone9 should always be treated as an odd tone. No differentiation
         //       between Non-Vectoring and Vectoring!
         if((k & 1) == ODD_CLUSTER_INDEX)
         {
            //If this tone is left band edge, copy its right neighboring tone to it
            if(i == gs_LeftChannel)
            {
               gsa_pre_FDQ_coef[m2] = gsa_pre_FDQ_coef[m2+2];
               gsa_pre_FDQ_coef[m2+1] = gsa_pre_FDQ_coef[m2+3];
               guca_pre_FDQ_exp[m] = guca_pre_FDQ_exp[m+1];

               {
                  // If the magnitudes of the current and next tones differ, adjust the fdq exponent
                  // according to the difference.
                  if(sa_exp_magsq[1]!=sa_exp_magsq[2])
                  {
                     s_exp_reduction = ((2+sa_exp_magsq[1]-sa_exp_magsq[2])>>1);
                     ApplyFdqExpReduction(&gsa_pre_FDQ_coef[m2],&guca_pre_FDQ_exp[m],s_exp_reduction);
                  }
               }
            }
            //If this tone is right band edge, copy its left neighboring tone to it
            else if(i == gs_RightChannel)
            {
               gsa_pre_FDQ_coef[m2] = gsa_pre_FDQ_coef[m2-2];
               gsa_pre_FDQ_coef[m2+1] = gsa_pre_FDQ_coef[m2-1];
               guca_pre_FDQ_exp[m] = guca_pre_FDQ_exp[m-1];

               {
                  // If the magnitudes of the current and previous tones differ, adjust the fdq exponent
                  // according to the difference.
                  if (sa_exp_magsq[1]!=sa_exp_magsq[0])
                  {
                     s_exp_reduction = ((2+sa_exp_magsq[1]-sa_exp_magsq[0])>>1);
                     ApplyFdqExpReduction(&gsa_pre_FDQ_coef[m2],&guca_pre_FDQ_exp[m],s_exp_reduction);
                  }
               }
            }
            //For all other tones, interpolating two neighboring tones
            else
            {
               //always interpolate FDQ for the data tones.
               InterpFdq(&(gsa_pre_FDQ_coef[m2-2]), &(guca_pre_FDQ_exp[m-1]));

               // If the three consecutive tones don't have similar magnitudes, adjust the exponent or
               // mantissa of current tone.
               // this avoids OVF during subsequent fine-training due to
               // getting an overly large fdq during interpolation.
               if ( (sa_exp_magsq[0]!=sa_exp_magsq[1]) || (sa_exp_magsq[1]!=sa_exp_magsq[2]) || (sa_exp_magsq[0]!=sa_exp_magsq[2]))
               {
                  if (sa_exp_magsq[2]>sa_exp_magsq[0])
                  {
                     s_exp_reduction = ((2+sa_exp_magsq[1]-sa_exp_magsq[2])>>1);
                     ApplyFdqExpReduction(&gsa_pre_FDQ_coef[m2],&guca_pre_FDQ_exp[m],s_exp_reduction);
                  }
                  else
                  {
                     s_exp_reduction = ((2+sa_exp_magsq[1]-sa_exp_magsq[0])>>1);
                     ApplyFdqExpReduction(&gsa_pre_FDQ_coef[m2],&guca_pre_FDQ_exp[m],s_exp_reduction);
                  }
               }
            }//else

         } // if((k & 1) == 1)

         if (k == TONE_CLUSTER_END_INDEX)
         {
            s_cluster_index++;
         }
      } //for(i = gs_LeftChannel; i <= gs_RightChannel; i++)


      // Interpolation of two neighboring tones need to take into account the RFI tones info.
      for (m = 0; m < gs_NumOfRFIBands; m++)
      {

         i = gsa_RFIBandLeftChannel[m];
         // check to see if gsa_RFIBandLeftChannel[m] is in [gs_LeftChannel+2, gs_RightChannel]
         if ((i >= gs_LeftChannel+2) && (i <= gs_RightChannel))
         {

            s_cluster_index = ClusterIndex(i);
            k = i - s_cluster_index*10;

            // If gsa_RFIBandLeftChannel[m] is one of the right neigbouring tones, do
            // not interpolate. Use its left neighboring tone instead.

            // normalize i to within [0, gs_NumChannelsPerGroup] range
            i -= gs_LeftChannel;
            //if (k==2 || k==4 || k==6 || k==8)
            if (((k & 1) == 0) && (k != 0))
            {

               gsa_pre_FDQ_coef[(i-1)*2]   = gsa_pre_FDQ_coef[(i-2)*2];
               gsa_pre_FDQ_coef[(i-1)*2+1] = gsa_pre_FDQ_coef[(i-2)*2+1];
               guca_pre_FDQ_exp[i-1]       = guca_pre_FDQ_exp[i-2];
            }
         }

         i = gsa_RFIBandRightChannel[m];
         // check to see if gsa_RFIBandRightChannel[m] is in [gs_LeftChannel, gs_RightChannel-2]
         if ((i >= gs_LeftChannel) && (i <= gs_RightChannel-2))
         {

            s_cluster_index = ClusterIndex(i);
            k = i - s_cluster_index*10;

            // If gsa_RFIBandRightChannel[m] is one of the left neigbouring tones, do
            // not interpolate. Use its right neighboring tone instead.

            // normalize i to within [0, gs_NumChannelsPerGroup] range
            i -= gs_LeftChannel;
            //if (k==0 || k==2 || k==4 || k==6)
            if (((k & 1) == 0) && (k != 8))
            {

               gsa_pre_FDQ_coef[(i+1)*2]   = gsa_pre_FDQ_coef[(i+2)*2];
               gsa_pre_FDQ_coef[(i+1)*2+1] = gsa_pre_FDQ_coef[(i+2)*2+1];
               guca_pre_FDQ_exp[i+1]       = guca_pre_FDQ_exp[i+2];
            }
         }

      } //for (m = 0; m < gs_NumOfRFIBands; m++)
   } //if (gs_FDQTrainType==COARSE_FDQ_TRAIN)
}

/*****************************************************************************
;   Prototype: void ApplyFDQAdjustment(int16 *psa_FDQ_coef, int16 *psa_FDQCoefAdjustment,
;                              uint8 *puc_FDQ_exp, uint8 uc_FDQExpAdjustment)
;
;   Description:
;      This routine computes the final FDQ coefficients by multiplying the coarse FDQ and
;      corrective FDQ.
;
;   Input Arguments:
;      psa_FDQ_coef - pointer to the coarse FDQ coefficient buffer
;      psa_FDQCoefAdjustment - pointer to the corrective FDQ coefficient buffer
;      psa_FDQ_exp -- pointer to the exponent of the coarse FDQ (positive means left shift)
;      uc_FDQExpAdjustment -- the exponent of the corrective FDQ
;
;   Output Arguments:
;      psa_FDQ_coef - pointer to the final FDQ coefficient buffer
;      psa_FDQ_exp -- pointer to the exponent of the final FDQ (positive means left shift)
;
;   Return Value:
;      none
;
;   Global Variables:
;      none
;
;****************************************************************************/
void ApplyFDQAdjustment(int16 *psa_FDQ_coef, int16 *psa_FDQCoefAdjustment, uint8 *puc_FDQ_exp, uint8 uc_FDQExpAdjustment)
{
   int32 l_Part1, l_Part2, l_Part3, l_Real, l_Imag;
   int16 s_exp, s_exp_Real, s_exp_Imag, s_final_exp;

   // Given A +jB and C+jD, calculate E+jF = (A+jB)*(C+jD)
   // E = (A-B)*D + (C-D)*A
   // F = (A-B)*D + (C+D)*B

   // l_Part1 = (A-B)*D;
   l_Part1 = (psa_FDQ_coef[0] - psa_FDQ_coef[1]) * psa_FDQCoefAdjustment[1];

   // l_Part2 = (C-D)*A;
   l_Part2 = (psa_FDQCoefAdjustment[0] - psa_FDQCoefAdjustment[1]) * psa_FDQ_coef[0];

   // l_Part3 = (C+D)*B;
   l_Part3 = (psa_FDQCoefAdjustment[0] + psa_FDQCoefAdjustment[1]) * psa_FDQ_coef[1];

   l_Real = l_Part1 + l_Part2;
   l_Imag = l_Part1 + l_Part3;

   // Get normalization shift count for real and imag parts
   s_exp_Real = s_exp_Imag = 31;

   if(l_Real != 0)
   {
      s_exp_Real = norm_l(l_Real);
   }

   if(l_Imag != 0)
   {
      s_exp_Imag = norm_l(l_Imag);
   }

   //Find the minimum of the two
   if(s_exp_Real < s_exp_Imag)
   {
      s_exp = s_exp_Real;
   }
   else
   {
      s_exp = s_exp_Imag;
   }

   //Compute the number of shift (positive means right shift, and negative means left shift
   s_exp = (32-FDQ_MANTISSA_WORDLENGTH) - s_exp;

   //Update the final exponent
   s_final_exp = (int16)uc_FDQExpAdjustment + (int16)*puc_FDQ_exp + s_exp - FDQ_MANTISSA_FRAC_BITS;
   *puc_FDQ_exp = (uint8)s_final_exp;

   //Since the final exponent has to be positive, increase the number of right shift to make the final exponent to be 0
   if(s_final_exp < 0)
   {
      s_exp -= s_final_exp;
      *puc_FDQ_exp = 0;
   }
   //Reduce the number of right shift to makethe final exponent not exceeding the maximum
   if(s_final_exp > FDQ_EXPONENT_MAX)
   {
      s_exp -= (s_final_exp - FDQ_EXPONENT_MAX);
      *puc_FDQ_exp = FDQ_EXPONENT_MAX;
   }

   //Apply the shift to the mantissa
   if(s_exp > 0)
   {
      psa_FDQ_coef[0] = sature16(round(l_Real, s_exp));
      psa_FDQ_coef[1] = sature16(round(l_Imag, s_exp));
   }
   else
   {
      s_exp = -s_exp;
      psa_FDQ_coef[0] = sature16(l_Real<<s_exp);
      psa_FDQ_coef[1] = sature16(l_Imag<<s_exp);
   }

   //Due to the HW limit, the imaginary part cannot be 0x8000
   if(psa_FDQ_coef[1] <= -32768)
   {
      psa_FDQ_coef[1] = -32767;
   }
}

/*****************************************************************************
;   Prototype: void InterpFDQ(int16 *psa_FDQ_coef, uint8 *puca_FDQ_exp)
;
;   Description:
;      This routine computes the interpolated FDQ of a given tone by linearly interpolating
;      its two adjancent tones.
;
;   Input Arguments:
;      psa_FDQ_coef - pointer to the FDQ coefficient array of three adjacent tones
;      puca_FDQ_exp -- pointer to the FDQ exponent array of three adjacent tones
;                  (positive exp means left shift)
;
;   Output Arguments:
;      Same as the input arguments by the parameters of middle tone set to the interpolated
;      values of two adjacent tones.
;
;   Return Value:
;      none
;
;   Global Variables:
;      none
;
;****************************************************************************/
void InterpFdq(int16 *psa_FDQ_coef, uint8 *puca_FDQ_exp)
{
   int16 i, sa_exp[2], s_exp;
   int32 l_Acc[2];

   //Perform interpolation for real and imaginary part
   for(i=0; i<2; i++)
   {
      l_Acc[i] = (psa_FDQ_coef[i]<<puca_FDQ_exp[0]) + (psa_FDQ_coef[4+i]<<puca_FDQ_exp[2]);
      l_Acc[i] += 1;   //perform rounding
      l_Acc[i] >>= 1;

      if(l_Acc[i] != 0)
      {
         sa_exp[i] = norm_l(l_Acc[i]);
      }
      else
      {
         sa_exp[i] = 31;
      }
   }

   //Find the smallest exponent
   if(sa_exp[0] < sa_exp[1])
   {
      s_exp = sa_exp[0];
   }
   else
   {
      s_exp = sa_exp[1];
   }

   //Compute the number of right shifts that should be applied to mantissa to ensure
   //the word length equals to FDQ_MANTISSA_WORDLENGTH
   s_exp = (32-FDQ_MANTISSA_WORDLENGTH) - s_exp;

   if(s_exp > 0)
   {
      psa_FDQ_coef[2] = sature16(round(l_Acc[0], s_exp));
      psa_FDQ_coef[3] = sature16(round(l_Acc[1], s_exp));
   }
   else
   {
      //If s_exp <= 0, set it to 0 since the negative exponent is not allowed
      s_exp = 0;
      psa_FDQ_coef[2] = sature16(l_Acc[0]);
      psa_FDQ_coef[3] = sature16(l_Acc[1]);
   }

   //Due to the HW limit, the imaginary part cannot be 0x8000
   if(psa_FDQ_coef[3] <= -32768)
   {
      psa_FDQ_coef[3] = -32767;
   }

   puca_FDQ_exp[1] = (uint8)s_exp;
}


/*****************************************************************************
;   Subroutine Name: FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp)
;
;   Description:
;      This routine calculates the FDQ coefficient for a single channel using
;      the following method:
;
;      FDQ_coef[i] = reference_tone[i]/input_tone[i] for each channel i, where
;      all of FDQ_coef[i], reference_tone[i] and input_tone[i] are complex numbers.
;      FDQ_coef[i] is represented Wr[i], Wi[i] and Ws[i] in the form of (Wr[i] + jWi[i]) * 2^Ws[i]
;      where Wr[i] and Wi[i] are mantissas and Ws[i] is their exponent.
;      Wr[i], Wi[i] and Ws[i] are all 16-bit values.
;
;      The fixed point format of the FDQ coefficients Wr[i] and Wi[i] is
;            Q(FDQ_MANTISSA_WORDLENGTH - FDQ_MANTISSA_FRAC_BITS,FDQ_MANTISSA_FRAC_BITS).
;
;   Prototype:
;       void FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp);
;
;   Input Arguments:
;      psa_in_tone - pointer to representative input tone
;      psa_ref_tone -- pointer to reference tone
;
;   Output Arguments:
;      psa_FDQ_coef -- pointer to the computed mantissa of complex FDQ coefficient
;      puc_FDQ_exp -- pointer to the common exponent of this pair of FDQ coefficients
;
;   Return Value:
;      none
;
;   Global Variables:
;      none
;
;****************************************************************************/
void FDQcoefPerTone(int16 *psa_in_tone, int16 *psa_ref_tone, int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp)
{
   int16 s_Xr, s_Xi, s_Yr, s_Yi;
   int32 l_Acc;
   int16 s_exp_den, s_denum, s_rcp_denum, s_exp_rcp;
   int16 s_exp_num_r, s_num_r;
   int16 s_exp_num_i, s_num_i;
   int32 l_mant_r, l_mant_i;
   int16 s_exp_r, s_exp_i, s_exp_delta, s_RightShift, s_final_exp;

   /* Get input tone */
   s_Xr = psa_in_tone[0];
   s_Xi = psa_in_tone[1];

   /* Get reference tone */
   s_Yr = psa_ref_tone[0];
   s_Yi = psa_ref_tone[1];

   // ================================================================ */
   //
   //   FDQ coef = ref/input = (Yr+jYi)/(Xr+jXi)
   //
   //   Multiply numerator and denominator by (Xr-jXi) to make
   //   denominator real.
   //
   //   ref/input = (Yr+jYi)/(Xr+jXi) * (Xr-jXi)/(Xr-jXi)
   //
   //           = [(Xr*Yr + Xi*Yi) + j(Xr*Yi - Xi*Yr)] / (Xr^2 + Xi^2)
   //
   //
   // ================================================================ */

   // ================================================================
   //   Step 1. Calculate (real) denominator
   // ================================================================ */
   l_Acc = (int32)s_Xr * s_Xr + (int32)s_Xi * s_Xi;

   /* Get the normalization left shift count for the denominator */
   //s_exp_den = -norm_l(l_Acc);
   s_exp_den = norm_l(l_Acc);

   /* Normalize the denominator and represent it in 16 bit */
   /* and Round it Up */
   s_denum = sature16((round((l_Acc << s_exp_den), 16)));
   s_exp_den = -s_exp_den;

   // ==================================================================================
   //   Step 2: Compute real part of numerator = (Xr*Yr + Xi*Yi)
   // ==================================================================================
   l_Acc = (int32)s_Xr * s_Yr + (int32)s_Xi * s_Yi;

   /* Get the normalization shift count */
   //s_exp_num_r = -norm_l(l_Acc);
   s_exp_num_r = norm_l(l_Acc);

   /* Normalize and represent it in 16 bit, with rounding up */
   s_num_r = sature16((round((l_Acc << s_exp_num_r), 16)));
   s_exp_num_r = - s_exp_num_r;

   // =====================================================================================
   // Step 3: Compute imaginary part of numerator = j(Xr*Yi - Xi*Yr)
   // =====================================================================================

   l_Acc =  (int32)s_Xr * s_Yi - (int32)s_Xi * s_Yr;   // Cannot overflow.

   /* Get the normalization shift count */
   //s_exp_num_i = -norm_l(l_Acc);
   s_exp_num_i = norm_l(l_Acc);

   /* Normalize and represent it in 16 bit, with rounding up */
   s_num_i = sature16((round((l_Acc << s_exp_num_i), 16)));
   s_exp_num_i = -s_exp_num_i;

   // =====================================================================================
   // Step 4: Compute reciprocal of denominator = 1 / (Xr^2 + Xi^2)
   // =====================================================================================

   // Representation of '1' is mantissa=0x4000, exponent is -14.  Note the
   // divide routine requires normalized Q1.15 format.

   Divide_16bit(0x4000, -14, s_denum, s_exp_den, &s_rcp_denum, &s_exp_rcp  );

   // =====================================================================================
   // Step 5: Compute real part of FDQ coefficient = (Xr*Yr + Xi*Yi) * [1 / (Xr^2 + Xi^2)]
   // =====================================================================================

   /* Compute mantissa */
   l_mant_r = (int32)s_num_r * s_rcp_denum;

   /* Get normalization shift count */
   //s_exp_r = -norm_l(l_mant_r);
   s_exp_r = norm_l(l_mant_r);

   /* Normalize it */
   //l_mant_r = l_mant_r << -s_exp_r;
   l_mant_r = l_mant_r << s_exp_r;
   s_exp_r = -s_exp_r;

   /* Compute exponent of real part of FDQ coefficient */
   s_exp_r += s_exp_num_r + s_exp_rcp;

   // =====================================================================================
   // Step 6: Compute imaginary part of FDQ coefficient = j(Xr*Yi - Xi*Yr) * [1 / (Xr^2 + Xi^2)]
   // =====================================================================================

   /* Compute mantissa */
   l_mant_i = (int32)s_num_i * s_rcp_denum;

   /* Get normalization shift count */
   //s_exp_i = -norm_l(l_mant_i);
   s_exp_i = norm_l(l_mant_i);

   /* Normalize mantissa */
   //l_mant_i = l_mant_i << -s_exp_i;
   l_mant_i = l_mant_i << s_exp_i;
   s_exp_i = -s_exp_i;

   /* Compute exponent of imaginary part of FDQ coefficient */
   s_exp_i += s_exp_num_i + s_exp_rcp;

   // =====================================================================================
   // Step 7:
   // Renormalize real or imaginary part of FDQ coefficient so both
   // share the same exponent.
   // =====================================================================================

   // Common exponent value is stored in s_exp_r.
   // If either mantissa is zero, don't do any renormalizing.

   if (l_mant_r == 0)
   {
      s_exp_r = s_exp_i;   // Do this to avoid normalizing either component.
   }

   else if (l_mant_i == 0)
   {
      s_exp_i = s_exp_r;   // Do this to avoid normalizing either component.
   }

   s_exp_delta = s_exp_r - s_exp_i;

   if (s_exp_delta >= 0)
   {
      // Adjust imaginary component (or do nothing if s_exp_delta == 0).
      l_mant_i = round(l_mant_i, s_exp_delta);
   }
   else
   {
      // Adjust real component, with rounding up
      s_exp_delta = -s_exp_delta;
      l_mant_r = round(l_mant_r, s_exp_delta);
      s_exp_r = s_exp_i;
   }

   // Check whether input tone energy was saturated.  If so, reduce FDQ coefficient
   // magnitude by additional factor of 2.  This is done to allow faster adaptation
   // of the coefficient in the downward (smaller magnitude) direction.  Currently the
   // reference signal level is 1/4 of the input tone's saturation level. Therefore the
   // minimum FDQ coefficient that would be generated here has a magnitude
   // on the order of 0.25.  If, for example, the true input signal level is >4x the saturation
   // level, it would take three iterations of adaptation for the FDQ coefficient to reach its
   // proper value.  By decreasing the coefficient magnitude by a factor of 2 for saturated
   // input tones, we reduce the minimum coefficient magnitude to 0.125 and allow adaptation
   // after only 2 iterations.

   if ( ((s_Xr == MAX_TONE_AMPLITUDE) || (s_Xr == MIN_TONE_AMPLITUDE)) &&
         ((s_Xi == MAX_TONE_AMPLITUDE) || (s_Xi == MIN_TONE_AMPLITUDE)) )
   {
      s_exp_r -= 1;
   }

   /* =====================================================================   */
   // Step 8:
   // Format the FDQ coefficients into required fixed point format
   // based on the word length, fractional
   /* format of the mantissa, and exponent.                           */
   /* The resulting format will be                                    */
   /* Q(   FDQ_MANTISSA_WORDLENGTH - FDQ_MANTISSA_FRAC_BITS,               */
   /*      FDQ_MANTISSA_FRAC_BITS).                                 */
   /*                                                         */
   /* =====================================================================   */

   /* limit the wordsize if necessary and adjust exponent      */
   s_RightShift = 32 - FDQ_MANTISSA_WORDLENGTH;
   psa_FDQ_coef[0] = sature16((round(l_mant_r, s_RightShift)));
   psa_FDQ_coef[1] = sature16((round(l_mant_i, s_RightShift)));
   s_final_exp = s_exp_r + s_RightShift + FDQ_MANTISSA_FRAC_BITS;
   *puc_FDQ_exp = (uint8)s_final_exp;

   if (s_final_exp < 0)
   {
      // Exponent must be zero or positive.
      s_RightShift -= s_final_exp;

      psa_FDQ_coef[0] = sature16(round(l_mant_r, s_RightShift));
      psa_FDQ_coef[1] = sature16(round(l_mant_i, s_RightShift));

      *puc_FDQ_exp = 0;
   }

   else if (s_final_exp > FDQ_EXPONENT_MAX)
   {
      *puc_FDQ_exp = FDQ_EXPONENT_MAX;
   }

   //Due to the HW limit, the imaginary part cannot be 0x8000
   if(psa_FDQ_coef[1] <= -32768)
   {
      psa_FDQ_coef[1] = -32767;
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *
 *  Prototype: void RotateDataTones(int16 *psa_in_tone)
 *
 *  Description:
 *   Rotate each message-modulated tone to the first quadrant
 *   for the purpose of frame accumulation for FDQ computation.
 *
 *  Input Arguments: int16 *psa_in_tone
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */
void RotateDataTones(void)
{
   int16 k, m, i, s_tmp, s_cluster_index;
   int16 *psa_in_tone;

   psa_in_tone = gpsa_RxToneBuf;

   //Compute the first cluster index by floor(gs_LeftChannel/10)
   s_cluster_index = ClusterIndex(gs_LeftChannel);

   for(i = gs_LeftChannel; i <= gs_RightChannel; i++)
   {
      k = i - s_cluster_index*10;
      m = (i-gs_LeftChannel)*2;

      // XDSLRTFW-2576 (Start-End)
      // Vectoring:     if (k==1 || k==3 || k==5 || k==7 || k==9)
      // Non-Vectoring: if (k==1 || k==3 || k==5 || k==7)
      // Note: To avoid additional if-conditions like
      //          if( (((k&1)==1) && ((gft_DSVectoringEnabled == TRUE) || (k != 9)) && (gs_FDQTrainType != MEDLEY_FDQ_TRAIN)) ||
      //       the tone9 should always be treated as an odd tone. No differentiation
      //       between Non-Vectoring and Vectoring!
      if((((k & 1) == ODD_CLUSTER_INDEX) && (gs_FDQTrainType != MEDLEY_FDQ_TRAIN)) ||
         (gs_FDQTrainType == MEDLEY_FDQ_TRAIN))
      {
         if((psa_in_tone[m]<0) && (psa_in_tone[m+1]>=0))
         {
            // Rotate from the 2nd quadrant to the 1st quadrant
            s_tmp = psa_in_tone[m];
            psa_in_tone[m] = psa_in_tone[m+1];
            psa_in_tone[m+1] = -s_tmp;
         }
         else if ((psa_in_tone[m]<=0) && (psa_in_tone[m+1]<0))
         {
            //Rotate from the 3rd quadrant to the 1st quadrant.
            psa_in_tone[m] = -psa_in_tone[m];
            psa_in_tone[m+1] = -psa_in_tone[m+1];
         }
         else if ((psa_in_tone[m]>0) && (psa_in_tone[m+1]<0))
         {
            // Rotate from the 4th quadrant to the 1st quadrant
            s_tmp = psa_in_tone[m];
            psa_in_tone[m] = -psa_in_tone[m+1];
            psa_in_tone[m+1] = s_tmp;
         }
      }

      if (k == TONE_CLUSTER_END_INDEX)
      {
         s_cluster_index++;
      }
   }

   guc_FdqTrainingState = TRAINING_DONE;
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Description: Background function to scale FDQ accumulation
 *
 *  Prototype: void BgAvgFDQAccumulation(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */
void BgAvgFDQAccumulation(void)
{
   // shift to average of the accumulated frames
   RightShiftAndRound32to16(gpsa_RxRepBuf, 0, gpla_RxAccumBuf, 0, (int16)(gs_NumOfTonesInBand<<1), gs_AlgLog2NumFramesToAccum);


   guc_FdqTrainingState = TRAINING_DONE;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Description: Background function to rotate FDQ coeffs based on
 *               Medley Alignment Offset.
 *
 *  Prototype: void BgRotateFDQ(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */
void BgMedleyRotateFDQ(void)
{
   int16 i;
   signed int j, j2;

   for (i = gs_LeftChannel; i <= gs_RightChannel; i++)
   {
      j = i - gs_LeftChannel;
      j2 = j<<1;

      // First save a copy of orig FDQ coeffs:
      gpuca_MedleyTempFDQExps[j]     = guca_pre_FDQ_exp[j];
      gpsa_MedleyTempFDQCoeffs[j2]   = gsa_pre_FDQ_coef[j2];
      gpsa_MedleyTempFDQCoeffs[j2+1] = gsa_pre_FDQ_coef[j2+1];

      // Rotate FDQ mantissas and adjust exponent
      guca_pre_FDQ_exp[j] += RotateTone((int16)(gs_CurrSynchPointIdx*gt_SnrFrameAlignConfig.s_MedleyAlignOffsetStep-gs_MaxMedleyAlignOffset),
                                        i, &(gsa_pre_FDQ_coef[j2]));


      //Due to the HW limit, the imaginary part cannot be 0x8000
      if(gsa_pre_FDQ_coef[j2+1] <= -32768)
      {
         gsa_pre_FDQ_coef[j2+1] = -32767;
      }
   }

   guc_RotateFDQState = TRAINING_DONE;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Description: Adjust FDQ exponent *puc_FDQ_exp downward by s_exp_reduction.
 *             Cap exponent at 15.  If resulting exponent would be <0, saturate at 0
 *             and adjust the Real/Imag FDQ coefficient in psa_FDQ_coef[0],psa_FDQ_coef[1].
 *
 *  Prototype: void ApplyFdqExpReduction(int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp, int16 s_exp_reduction)
 *
 *  Input Arguments:
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ApplyFdqExpReduction(int16 *psa_FDQ_coef, uint8 *puc_FDQ_exp, int16 s_exp_reduction)
{
   if (((int16)(*puc_FDQ_exp))>=s_exp_reduction)
   {
      *puc_FDQ_exp = (uint8) (((int16)(*puc_FDQ_exp)) - s_exp_reduction);
      if (*puc_FDQ_exp>15)
      {
         *puc_FDQ_exp=15;
      }
   }
   else
   {
      psa_FDQ_coef[0] >>= (s_exp_reduction - (int16) (*puc_FDQ_exp));
      psa_FDQ_coef[1] >>= (s_exp_reduction - (int16) (*puc_FDQ_exp));
      *puc_FDQ_exp = 0;
   }
}


