/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   NomAtp.c
*
*   This file contains the functions used to compute the NOMATP.
*
*-------------------------------------------------------------------------
*/

#include "common.h"
#include "gdata.h"
#include "dsp_op.h"
#include "decimalgain.h"
#include "mul.h"
#include "nomatp.h"
#include "ConvertToDB.h"
#include "bitload.h"

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT CalcInitNOMATP(void)
*
*   This function computes the initial (or Medley) NOMATP based on MREFPSD[i]
*   received from PRM message. It also converts MAXNOMATP to a linear value
*   for later use.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*      SUCCEED or FAIL
*
*   Global Variables:
*      ghpsa_RxFineGains_Inactive -- (I) pointer the fine gain table (all fine gain should
*                                be set to 0 dB at this point.
*
*      gs_MedleyNOMATP_dBm -- (O) the computed Medley NOMATP
*
*   Also see CalcLinearMAXNOMATP(), CalcNOMATP() for global variables used
*-------------------------------------------------------------------------------
*/
FlagT CalcInitNOMATP(void)
{

   //Convert the MREFPSD from dB to linear format
   gft_BitloadOK = CalcLinearPsdTable();

   if(gft_BitloadOK == FAIL)
   {
      return(FAIL);
   }

   //Compute the maximum linear NOMATP, which is defined as 10^(x/10),
   //where x=MAXNOMATP-10log(subcarrier_space) ans store the result in gul_MAXNOMATP_linear
   CalcLinearMAXNOMATP();

   //Compute the nominal aggregate transmit power (NOMATP) based on
   //Medley Reference PSD (MREFPSD) (with all the fine gains of 0 dB)
   CalcNOMATP(ghpsa_RxFineGains_Inactive);

   //Save the Medley State NOMATP
   gs_MedleyNOMATP_dBm = gs_NOMATP_dBm;

   return(SUCCEED);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT CalcLinearPsdTable(void)
*
*   For the giving PSD breaking points, this function interpolates these PSD breakpoints
*   to produce the PSD level for each tone within the RX bands.
*   Then it converts these PSD levels from the dB value to the linear representation,
*   which are stored in a table.
*   If the breakpoints do not cover the RX bands properly, the function returns FAIL.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*      SUCCEED or FAIL
*
*   Global Variables:
*      gt_PwrConfigParam -- (I) the structure containing s_Up_MaxNomAggrPwr, s_Up_MaxNomPSD
*                                s_Dn_MaxNomAggrPwr and s_Dn_MaxNomPSD
*      gt_UsREFPSDDescriptorTable -- (I) the structure containing US Medley Reference PSD
*      gt_DsREFPSDDescriptorTable -- (I) the structure containing DS Medley Reference PSD
*      gs_NumOfRxBands -- (I) the number of RX bands
*      gsa_RxBandLeftChannel -- (I) the array of the RX band left edges
*      gsa_RxBandRightChannel -- (I) the array of the RX band right edges
*
*      gpsa_LinearPsdBuffer -- (O) the pointer to the buffer to store matissa of interpolated PSD in dB
*      gpsa_LinearPsdExpBuffer -- (O) the pointer to the buffer to store exponet of interpolated PSD in dB
*      gs_PsdOffset_dB -- (o) the PSD offset which is set the highest PSD value in MREFPSD[i]
*-------------------------------------------------------------------------------
*/
FlagT CalcLinearPsdTable(void)
{
   int16 s_MaxNomPSD;
   PSDDescriptorTable_t *pt_PSDDescriptor;
   int16 sa_NormalizedPsdLevel[MAX_NUM_PSD_POINTS];

   int16 i;
   int16 s_band, s_ch, s_ch_min, s_ch_max;
   int16 s_LeftBreakPointIdx;
   int16 s_NumPsdBreakPoint, s_LeftBreakPoint, s_RightBreakPoint;
   int16 s_LeftPsdLevel, s_RightPsdLevel, s_PsdLevel;
   int16 *ps_InterpPsdTable, *ps_InterpPsdExpTable;

   //Initialize the variables based on CO or CPE
   gs_MAXNOMATP_dBm = gt_PwrConfigParam.s_Dn_MaxNomAggrPwr;
   s_MaxNomPSD = gt_PwrConfigParam.s_Dn_MaxNomPSD;
   pt_PSDDescriptor = (PSDDescriptorTable_t*)(void *)&gt_DsREFPSDDescriptorTable;

   s_NumPsdBreakPoint = pt_PSDDescriptor->us_NumberOfTones;

   //According to the standard, valid range of MAXNOMATP is between -25.6 and +25.6
   if((gs_MAXNOMATP_dBm < -256) || (gs_MAXNOMATP_dBm > 256))
   {
      gus_BitloadErrorCode = BITLOAD_ERROR_INVALID_MAXNOMATP;
      return(FAIL);
   }


   //Compute the true PSD value and find the highest PSD point
   gs_PsdOffset_dB = (int16)NEG_INFINITY_DB;
   for(i=0; i<s_NumPsdBreakPoint; i++)
   {
      s_PsdLevel = -pt_PSDDescriptor->ut_PSDRecord[i].s_PSDLevelOfTone-s_MaxNomPSD;

      sa_NormalizedPsdLevel[i] = s_PsdLevel;

      if(s_PsdLevel > gs_PsdOffset_dB)
      {
         gs_PsdOffset_dB = s_PsdLevel;
      }
   }

   //Subtract all PSD level by gs_PsdOffset_dB to reduce its dynamic range
   for(i=0; i<s_NumPsdBreakPoint; i++)
   {
      sa_NormalizedPsdLevel[i] -= gs_PsdOffset_dB;
   }

   //Compute the NOMATP over the medley set
   s_LeftBreakPointIdx = 0;

   for(s_band = 0; s_band < gs_NumOfRxBands; s_band++)
   {
      //Find the left PSD break point for the left band edge
      for(i = s_LeftBreakPointIdx; i < s_NumPsdBreakPoint; i++)
      {
         if(pt_PSDDescriptor->ut_PSDRecord[i].us_IndexOfTone <= gsa_RxBandLeftChannel[s_band])
         {
            s_LeftBreakPointIdx = i;
         }
         else
         {
            break;
         }
      }

      //Get the left and right PSD break point
      s_LeftBreakPoint = pt_PSDDescriptor->ut_PSDRecord[s_LeftBreakPointIdx].us_IndexOfTone;
      s_RightBreakPoint = pt_PSDDescriptor->ut_PSDRecord[s_LeftBreakPointIdx+1].us_IndexOfTone;

      //Get the tone range for this pair of break point
      s_ch_min = MAX(s_LeftBreakPoint, gsa_RxBandLeftChannel[s_band]);
      s_ch_max = MIN(s_RightBreakPoint, gsa_RxBandRightChannel[s_band]);

_Start:
      //Get left and right PSD level (represented in integer multiple of 0.1 dB)
      s_LeftPsdLevel = sa_NormalizedPsdLevel[s_LeftBreakPointIdx];
      s_RightPsdLevel = sa_NormalizedPsdLevel[s_LeftBreakPointIdx+1];

      //Set the table pointers to the first tone of this tone range
      ps_InterpPsdTable = gpsa_LinearPsdBuffer + s_ch_min;
      ps_InterpPsdExpTable = gpsa_LinearPsdExpBuffer + s_ch_min;

      for(s_ch = s_ch_min; s_ch <= s_ch_max; s_ch++)
      {
         //Interpolate the left and right PSD break point to get the PSD level for this tone
         InterpolatePsd(s_LeftBreakPoint, s_RightBreakPoint, s_LeftPsdLevel, s_RightPsdLevel, s_ch, &s_PsdLevel);

         //Convert the PSD in dB to PSD in linear (ie. 10^(MREFPSD[i]/10)) in Q1.15
         CalcLinearPsd(s_PsdLevel, ps_InterpPsdTable, ps_InterpPsdExpTable);

         ps_InterpPsdTable++;
         ps_InterpPsdExpTable++;

      } //for(s_ch =

      //Update the break points
      if(s_ch <= gsa_RxBandRightChannel[s_band])
      {
         s_LeftBreakPointIdx++;
         s_LeftBreakPoint = s_RightBreakPoint;
         s_RightBreakPoint = pt_PSDDescriptor->ut_PSDRecord[s_LeftBreakPointIdx+1].us_IndexOfTone;

         s_ch_min = s_ch;
         s_ch_max = MIN(s_RightBreakPoint, gsa_RxBandRightChannel[s_band]);
         goto _Start;
      }

   } //for(s_band = 0; s_band < gs_NumOfRxBands; s_band++)

   //Compute the difference between MAXNOMATP and NOMATP
   return(SUCCEED);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void CalcLinearMAXNOMATP(void)
*
*   This function computes the adjusted MAXNOMATP, which is defined as:
*
*   adjusted MAXNOMATP = 10^((MAXNOMATP - 10log(subcarrier_space))/10 - PsdOffset)
*   where the PSD offset is used to normalize the PSD level to get the maximum
*   dynamic range.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*      None
*
*   Global Variables:
*      gs_frame_rate_is_8khz -- (I) flag to indicate the frame rate
*      gs_MAXNOMATP_dBm      -- (I) the MAXNOMATP in dBm (in multiple of 0.1 dB)
*      gs_PsdOffset_dB       -- (I) the PSD level offset (in multiple of 0.1 dB)
*      gul_MAXNOMATP_linear_lw  -- (O) the adjusted linear MAXNOMATP's lower word
*      gul_MAXNOMATP_linear_hw  -- (O) the adjusted linear MAXNOMATP's high word
*      gs_LogSubCarrierSpace    -- (O) set to 10log(subcarrier_space) * 10
*-------------------------------------------------------------------------------
*/
void CalcLinearMAXNOMATP(void)
{

   int32 l_Acc;
   int16 s_mant, s_x, s_exp;

   //Decide 10log(subcarrier_space) Q8.8
   //This global variable is used by some of other functions
   if(gs_frame_rate_is_8khz == 0)
   {
      gs_LogSubCarrierSpace = LOG_SUBCARRIER_SPACE_4KHZ;
   }
   else
   {
      gs_LogSubCarrierSpace = LOG_SUBCARRIER_SPACE_8KHZ;
   }

   s_x = gs_MAXNOMATP_dBm - gs_PsdOffset_dB;

   //Convert gs_LogSubCarrierSpace from Q8.8 to the multiple of 0.1 dB format
   MULS16(l_Acc, gs_LogSubCarrierSpace, (int16)10);

   //Subtract 10log(subcarrier_space) from the input
   l_Acc = (s_x<<8) - l_Acc;

   //Convert the PSD in multiple of 0.1 dB to linear
   s_x = (int16)((l_Acc + (1<<7)) >> 8);

   //s_mant is in Q1.15
   CalcLinearPsd(s_x, &s_mant, &s_exp);

   //Convert the mantissa from Q1.15 to Q1.30
   l_Acc = (int32)s_mant << 15;

   //Store the result in low and upper word of 48-bit register
   if(s_exp == 0)
   {
      gul_MAXNOMATP_linear_lw = l_Acc;
      gul_MAXNOMATP_linear_hw = 0;
   }
   else if(s_exp < 0)
   {
      s_exp = - s_exp;
      gul_MAXNOMATP_linear_lw = (l_Acc + (1<<(s_exp-1)))>>s_exp;
      gul_MAXNOMATP_linear_hw = 0;
   }
   else //s_exp>0
   {
      gul_MAXNOMATP_linear_lw = (l_Acc&0x0000FFFF) << s_exp;
      gul_MAXNOMATP_linear_hw = (l_Acc>>16)<<s_exp;
   }

   //Add the upper 16 bits of the 32-bit lower word to the lower 16 bits of the 32-bit high word
   //and clean up the upper 16 bits of the 32-bit lower word.
   //(this will simplify the comparison in CheckNOMATPLimit()
   gul_MAXNOMATP_linear_hw += gul_MAXNOMATP_linear_lw>>16;
   gul_MAXNOMATP_linear_lw &= 0x0000FFFF;
}


/*
*-------------------------------------------------------------------------------
*
*   Prototype: void CalcNOMATP(int16 *psa_RxFineGains)
*
*   This function computes the nominal aggregate transmit power (NOMATP)
*   using the equation given in Section 10.3.4.2 of VDSL2 standard.
*
*   To simplify the computation, we compute the adjusted NOMATP and adjust MAXNOMATP
*   in linear domain as defined below:
*
*   adjusted NOMATP = sum of {(10^(MREFPSD[i]/10) * gain[i]^2} over all the tones below to Medley set
*
*   adjusted MAXNOMATP = 10^((MAXNOMATP - 10log(subcarrier_space))/10)
*
*   Input Arguments:
*      psa_RxFineGains -- pointer to the RX fine gain table in dB (Q8.8)
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*      gs_NumOfRxBands -- (I) the number of RX bands
*      gsa_RxBandLeftChannel -- (I) the array of the RX band left edges
*      gsa_RxBandRightChannel -- (I) the array of the RX band right edges
*      gpsa_LinearPsdBuffer -- (I) the pointer to the buffer to store matissa of interpolated PSD in dB
*      gpsa_LinearPsdExpBuffer -- (I) the pointer to the buffer to store exponet of interpolated PSD in dB
*
*      gul_NOMATP_linear -- (O) the adjusted linear NOMATP (Q17.15)
*      gul_NOMATP_linear_lw - (O) the low word of linear NOMATP
*      gul_NOMATP_linear_hw - (O) the high word of linear NOMATP
*      gs_NOMATP_dBm -- (O) the NOMATP as defined by the standard (in multiple of 0.1 dB)
*-------------------------------------------------------------------------------
*/

void CalcNOMATP(int16 *psa_RxFineGains)
{
   int16 s_band, s_ch;
   int16 s_FineGain_Linear;
   int16 *ps_FineGain;
   int32 l_Power;

   //Compute the NOMATP over the medley set
   gul_NOMATP_linear_hw = 0;
   gul_NOMATP_linear_lw = 0;

   for(s_band = 0; s_band < gs_NumOfRxBands; s_band++)
   {
      //Set the table pointers to the first tone of this tone range
      ps_FineGain = psa_RxFineGains + gsa_RxBandLeftChannel[s_band];

      for(s_ch = gsa_RxBandLeftChannel[s_band]; s_ch <= gsa_RxBandRightChannel[s_band]; s_ch++)
      {
         //Convert the fine gain to the linear value in Q3.9 format (match to the message)
         s_FineGain_Linear = (DecimalGain(*ps_FineGain)+(int16)(1<<3)) >> 4;

         // If the far end does not transmit the bi=0 tones in spite of gi != 0,
         // do not consider them in NOMATP calculation by assuming gi = 0
         if ((gul_dbgMiscControl & IGNORE_BI0_TONES_FOR_NOMATP_COMPUTATION) && (ghpuca_RxBat_Inactive[s_ch] == 0))
         {
            s_FineGain_Linear = 0;
         }

         //Compute gain^2 (Q6.18)
         MULS16(l_Power, s_FineGain_Linear, s_FineGain_Linear);

         //Update the NOMATP and store the result in gul_NOMATP_linear_hw and gul_NOMATP_linear_lw
         UpdateNOMATP(s_ch, l_Power);

         //Update the table pointer
         ps_FineGain++;

      } //for(s_ch =

   } //for(s_band = 0; s_band < gs_NumOfRxBands; s_band++)


   //From gul_NOMATP_linear_hw and gul_NOMATP_linear_lw,
   //compute the final linear NOMATP (stored in gul_NOMATP_linear)
   //and the final NOMATP in dBm (multiple of 0.1 dB) stored in gl_NOMATP_dBm
   CalcFinalNOMATP();
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT InterpolatePsd(int16 s_LeftBreakPoint, int16 s_RightBreakPoint,
*                           int16 s_LeftPsdLevel, int16 s_RightPsdLevel,
*                           int16 s_ToneIdx, int16 *ps_PsdLevel)
*
*   This function computes the PSD level of a given tone by linearly interpolating
*   two PSD break points.
*
*   P0 = P1 + (P2-P1) * (t0-t1)/(t2-t1)
*
*   where P1 and P2 are the PSD level of left and right break point respectively,
*   t1 abd t2 are the tone index of left and right break point respectively,
*   and t0 is the index of the tone at which the interpolated PSD level, P0,
*   is computed.
*
*   Input Arguments:
*      s_LeftBreakPoint -- tone index of left PSD break point
*      s_RightBreakPoint -- tone index of right PSD break point
*      s_LeftPsdLevel -- PSD level of left PSD break point (in multiple of 0.1 dB)
*      s_RightPsdLevel -- PSD level of right PSD break point (in multiple of 0.1 dB)
*      s_ToneIdx -- current tone index (to be interpolated)
*
*   Output Arguments:
*      ps_PsdLevel -- pointer to the interpolated PSD level (in multiple of 0.1 dB)
*
*   Returns:
*      None
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void InterpolatePsd(int16 s_LeftBreakPoint, int16 s_RightBreakPoint,
                    int16 s_LeftPsdLevel, int16 s_RightPsdLevel,
                    int16 s_ToneIdx, int16 *ps_PsdLevel)
{

   int16 s_P_delta, s_t0_delta, s_t1_delta;
   int32 l_Acc;

   //Convert the PSD level from multiple of 0.1 dB format to Q8.8 format)
   if(s_ToneIdx == s_LeftBreakPoint)
   {
      *ps_PsdLevel = s_LeftPsdLevel;
      return;
   }
   else if(s_ToneIdx == s_RightBreakPoint)
   {
      *ps_PsdLevel = s_RightPsdLevel;
      return;
   }

   //Perform the interpolation, such that
   //Psd0 = Psd1 + (Psd2-Psd1) * (t0-t1)/(t2-t1), as described in the function header
   s_P_delta = s_RightPsdLevel - s_LeftPsdLevel;
   s_t0_delta = s_ToneIdx - s_LeftBreakPoint;
   s_t1_delta = s_RightBreakPoint - s_LeftBreakPoint;

   MULS16(l_Acc, s_P_delta, s_t0_delta);

   //Compute abs(l_Acc)/s_t1_delta
   if(l_Acc < 0)
   {
      l_Acc = - l_Acc;
   }

   l_Acc = NormAndDivide_32by16bit(l_Acc, s_t1_delta);

   //Restore the sign
   if(s_P_delta < 0)
   {
      l_Acc = -l_Acc;
   }

   l_Acc += s_LeftPsdLevel;

   *ps_PsdLevel = (int16)l_Acc;
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT CalcLinearPsd(int16 s_Psd_dB, int16 *pl_PsdMant_linear, int16 *ps_PsdExp_linear)
*
*   This function computes the transmit power of the current tone in linear domain.
*
*   TP_linear = 10^(MREFPSD[i]/10)
*
*   Input Arguments:
*      s_Psd_dB -- input PSD level in dB (in unit of 0.1 dB)
*
*   Output Arguments:
*      ps_PsdMant_linear -- pointer to the mantissa of the linear PSD for the given tone (in Q1.15)
*      ps_PsdExp_linear -- pointer to the exponent of the linear PSD (>0 left shift, <0 right shift)
*
*   Returns:
*      None
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/

// note this is not 6*256 but round(20*log10(2)*256)
#define   LOG2x20_DB   (1541)      //20*log(2)      in Q8.8 format

#define MIN_PSD_DB   (-128*256)   //=-128 dB in Q8.8
#define MAX_PSD_DB   (127*256)   //=127 dB in Q8.8
void CalcLinearPsd(int16 s_Psd_dB, int16 *pl_PsdMant_linear, int16 *ps_PsdExp_linear)
{
   int16 s_mant, s_exp, s_exp1;
   int16 s_PsdLevel;
   int32 l_Acc, l_PsdLevel;

   //Change the PSD (in multiple of 0.1 dB) to 8.8 format
   l_Acc = s_Psd_dB << 8;

   if(s_Psd_dB < 0)
   {
      l_Acc = -l_Acc;
   }

   l_PsdLevel = NormAndDivide_32by16bit(l_Acc, (int16)10);

   if(s_Psd_dB < 0)
   {
      l_PsdLevel = -l_PsdLevel;
   }

   //Since db_to_linear() computes 10^(x/20), instead of 10^(x/10),
   //so we need to scale input x up by the factor of 2 to get the desired result
   l_PsdLevel <<= 1;

   //Limit the PSD range to be greater than -127 since input to db_to_linear()
   //is in Q8.8 format and use s_exp to store the number of right (<0)
   //or left (>0) shifts which should be applied to the result
   s_exp = 0;
   while(l_PsdLevel < MIN_PSD_DB)
   {
      l_PsdLevel += LOG2x20_DB;
      s_exp--;
   }

   while(l_PsdLevel > MAX_PSD_DB)
   {
      l_PsdLevel -= LOG2x20_DB;
      s_exp++;
   }

   s_PsdLevel = (int16)l_PsdLevel;

   //Convert the dB to linear (i.e., compute 10^(Psd/20))
   db_to_linear(s_PsdLevel, &s_mant, &s_exp1);

   *pl_PsdMant_linear = s_mant;
   *ps_PsdExp_linear = (s_exp+s_exp1);
}

#undef LOG2x20_DB
#undef MIN_PSD_DB

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void UpdateNOMATP(int16 s_ToneIdx, int32 l_GainSq)
*
*   This function updates the adjusted linear NOMATP for a given tone based.
*
*   Giving:
*   adjusted NOMATP = sum of {(10^(MREFPSD[i]/10) * gain[i]^2} over all the tones belong to Medley set
*
*   Updating is done by:
*   adjusted NOMATP = adjusted NOMATP + PSD_linear * gain^2
*
*   Input Arguments:
*      s_ToneIdx -- tone index
*      l_GainSq -- linear gain square (Q6.18)
*
*   Output Arguments:
*
*   Returns:
*      None
*
*   Global Variables:
*      gul_NOMATP_linear_lw - (I/O) the low word of linear NOMATP
*      gul_NOMATP_linear_hw - (I/O) the high word of linear NOMATP
*-------------------------------------------------------------------------------
*/

void UpdateNOMATP(int16 s_ToneIdx, int32 l_GainSq)
{

   int16 s_PsdMant, s_PsdExp;
   int16 s_FineGain_Linear;
   int32 l_Power;

   //Get the linear PSD for the given tone, represented in a mantissa and exponent
   s_PsdMant = gpsa_LinearPsdBuffer[s_ToneIdx];   //in Q1.15
   s_PsdExp  = gpsa_LinearPsdExpBuffer[s_ToneIdx]; //>0 left shifts, <0 right shift

   //Comvert l_GainSq from Q6.18 to Q2.14
   //(Note l_GainSq can also represent the different between two gain square
   //so it can be both possitive and negative)
   s_FineGain_Linear = (int16)(l_GainSq>>(18-14));

   //Compute 10^(MREFPSD[i]/10) * gain^2
   //Note PSD is in Q1.15 and gain^2 is in Q2.14 so l_Power is in Q3.29
   MULU16(l_Power, s_PsdMant, s_FineGain_Linear);

   //Note since the linear PSD is always positive so its sign bit is redudant
   //we can represent the above result in Q2.30 to get more precision
   l_Power <<= 1;

   //Note Exp <= 0
   if(s_PsdExp < 0)
   {
      //Convert the negative exponent to number of right shift
      //(Note: since the PSD is alway negative so the shift is alway right shift)
      s_PsdExp = - s_PsdExp;

      l_Power = (l_Power + (1<<(s_PsdExp-1))) >> s_PsdExp;
   }

   //Compute the sum of 10^(MREFPSD[i]/10) * gain^2 using 48-bit accumulator
   //the final result should be (gul_NOMATP_linear_hw<<16) + gul_NOMATP_linear_lw
   //still in Q(48-30).30
   SignedAccum48(l_Power, &gul_NOMATP_linear_hw, &gul_NOMATP_linear_lw);
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void CalcFinalNOMATP(void)
*
*   This function updates the NOMATP as define in the standard by the following equation
*
*   NOMATP = 10*log(subcarrier_space) + 10*log(l_NOMATP_linear)
*
*   where l_NOMATP_linear = sum of {(10^(MREFPSD[i]/10) * gain[i]^2}) over all the tones belong to Medley set
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*      None
*
*   Global Variables:
*      gul_NOMATP_linear_lw  -- (I) low word of the linear NOMATP
*      gul_NOMATP_linear_hw  -- (I) high word of the linear NOMATP
*      gs_PsdOffset_dB       -- (I) PSD offset (in multiple of 0.1 dB)
*      gs_LogSubCarrierSpace -- (I) 10log10(subcarrier_space) * 10
*      gs_NOMATP_dBm        -- (O) the NOMATP (in multiple of 0.1 dB)
*
*-------------------------------------------------------------------------------
*/
void CalcFinalNOMATP(void)
{

   int16 s_NOMATP_dBm;
   int32 l_Acc, l_Acc1;
   int32 ul_Acc_hw;
   uint16 us_Acc_lw;
   int16 s_exp;

   //Note: the linear NOMATP is now represented in gul_NOMATP_linear_lw and gul_NOMATP_linear_hw
   //the combined value should be (gul_NOMATP_linear_hw<<16) + gul_NOMATP_linear_lw
   //and the result is in QX.30 format

   //Add the upper 16 bits of the lower word to the lower 16 bits of high word
   //and clean up the upper 16 bits of the lower word.
   gul_NOMATP_linear_hw += gul_NOMATP_linear_lw>>16;
   gul_NOMATP_linear_lw &= 0x0000FFFF;

   //First, normalize the upper word
   s_exp = norm_l(gul_NOMATP_linear_hw);
   ul_Acc_hw = gul_NOMATP_linear_hw <<s_exp;

   //Shift the upper "s_exp" bits of the lower 16-bit of gul_NOMATP_linear_lw into gul_NOMATP_linear_hw
   us_Acc_lw = ((int16)gul_NOMATP_linear_lw) >> (16-s_exp);
   ul_Acc_hw += us_Acc_lw;

   //Compute the total number of shifts to be applied to the final result
   //note 30 counts for QX.30 format, 16 counts for dropping of the lower 16 bits
   s_exp = -s_exp - 30 + 16;

   //Normalize the upper word
   if(ul_Acc_hw == 0)
   {
      gs_NOMATP_dBm = NEG_INFINITY_DB;
   }
   else
   {
      //Compute 10*log(l_NOMATP_linear)
      //Input is assumed to be Q32.0, and output is Q8.8
      s_NOMATP_dBm = ConvertToDB(ul_Acc_hw);

      //Adjust the results by s_exp * 10log(2) = s_exp * (10log(2) * 256) = s_exp * 771
      //Since l_NOMATP_linear is in Q2.30 and ConvertToDB() assumes the input is Q32.0
      //we need to adjust the result by -10log(2^30)*256 = 23119 (Q8.8)
      MULS16(l_Acc, s_exp, (int16)771);

      l_Acc += s_NOMATP_dBm;

      l_Acc += gs_LogSubCarrierSpace;

      //Convert the result from Q8.8 to the multiple of 0.1 dB
      MULS32x16(l_Acc1, l_Acc, 10);

      gs_NOMATP_dBm = (int16)((l_Acc1 + (1<<7)) >> 8);

      gs_NOMATP_dBm += gs_PsdOffset_dB;
   }

}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT CheckNOMATPLimit(void)
*
*   This function compute the linear NOMATP with the linear MAXNOMATP
*   and return 0 if no violation, 1 if there is violation.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*      1: if NOMATP > MAXNOMATP, else 0
*
*   Global Variables:
*      gul_NOMATP_linear_lw -- (I) low word of the linear NOMATP
*      gul_NOMATP_linear_hw -- (I) high word of the linear NOMATP
*      gul_MAXNOMATP_linear_lw -- (I) low word of linear MAXNOMATP
*      gul_MAXNOMATP_linear_hw -- (I) high word of linear MAXNOMATP
*-------------------------------------------------------------------------------
*/
FlagT CheckNOMATPLimit(void)
{

   //Add the upper 16 bits of the 32-bit lower word to the lower 16 bits of the 32-bit high word
   //and clean up the upper 16 bits of the lower word.
   gul_NOMATP_linear_hw += gul_NOMATP_linear_lw>>16;
   gul_NOMATP_linear_lw &= 0x0000FFFF;

   if(gul_NOMATP_linear_hw < gul_MAXNOMATP_linear_hw)
   {
      return(0);
   }
   else if(gul_NOMATP_linear_hw > gul_MAXNOMATP_linear_hw)
   {
      return(1);
   }
   else //if two upper word equal, compare two lower words
   {
      if(gul_NOMATP_linear_lw < gul_MAXNOMATP_linear_lw)
      {
         return(0);
      }
      else
      {
         return(1);
      }
   }
}

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void SignedAccum48(int32 l_Input, uint32 *pul_Acc48H, uint32 *pul_Acc48L)
*
*   This function adds the 32-bit signed integer to the 48-bit accumulator.
*   The 48-bit accumulator is represented by two 32-bit integer, i.e., LW32 and HW32,
*   and the final 48-bit value should be computed as:
*
*   48-bit Results = (HW32<<16) + LW32.
*
*   Input Arguments:
*      l_Input -- input 32-bit value to be added to the accumulator
*      pul_Acc48H -- pointer to the high 32-bit word of the accumulator (i.e HW32)
*      pul_Acc48L -- pointer to the low 32-bit word of the accumulator (i.e LW32)
*
*   Output Arguments:
*      pul_Acc48H -- pointer to the high 32-bit word of the accumulator after update
*      pul_Acc48L -- pointer to the low 32-bit word of the accumulator after update
*
*   Returns:
*
*   Global Variables:
*
*-------------------------------------------------------------------------------
*/
void SignedAccum48(int32 l_Input, uint32 *pul_Acc48H, uint32 *pul_Acc48L)
{
   uint32 ul_Acc, ul_Input;
   uint16 us_flag = 0;

   if(l_Input >= 0)
   {
      *pul_Acc48L += (l_Input & 0x0000FFFF);
      *pul_Acc48H += (l_Input >> 16);
   }
   else //if(Input < 0)
   {
      ul_Input = -l_Input;

      if(ul_Input < *pul_Acc48L)
      {
         *pul_Acc48L -= ul_Input;
      }
      else
      {

         //Add the upper 16 bits of the lower 32-bit accumulator to
         //the upper 32-bit accumulator
         *pul_Acc48H += (*pul_Acc48L>>16);
         *pul_Acc48L &= (uint32)0x0000FFFF;

         //If the accumulator value is greater than 32-bit representation
         //Use the largest positive 32-bit number
         if(*pul_Acc48H & 0xFFFF0000)
         {
            //Borrow 1 from high-16 bit word of *pul_Acc48H
            *pul_Acc48H -= 0x00010000;

            ul_Acc = 0xFFFFFFFF - ul_Input;
            ul_Acc += 1;

            *pul_Acc48L += (ul_Acc & 0x0000FFFF);
            *pul_Acc48H += (ul_Acc >> 16);
         }
         else
         {
            //Form the lower 32-bit of 48-bit accumulator
            ul_Acc = *pul_Acc48L + (*pul_Acc48H<<16);

            ul_Acc -= ul_Input;

            *pul_Acc48L = ul_Acc & 0x0000FFFF;
            *pul_Acc48H = ul_Acc >> 16;
         }
      }
   }
}
