/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   pll.c
;
;   This file contains routines to implement the phase lock loop (PLL).
;
;***************************************************************************/
//********************************************************************************************
// 19/12/2012 Abu: FAST Loss detection in training (vectoring mode)
//            Grep XDSLRTFW-645 / XDSLRTFW-674
// 21/11/2013 Anantha Ramu: Observed that ReTx test noise(REIN/SHINE/PEIN) was disturbing PLL.
//                          Disturbance of PLL leads to uncorrected DTUs in ReTx mode. Did
//                     modifications to detect high phase error change and saturating
//                          phase error input to PLL at those events.
//                          Added CMV control bit TEST_PLL_fix to enable(1)/disable(0) the fix.
//                     Grep for: XDSLRTFW-1271 DSReTx_uncorrected_DTUs_PLL_changes
//
// 18/12/2013 Kannan:  Disabled the Micro interruption handler for DS ReTx ,
//                    since Micro interruption state machine disable the PLL and Enable the PLL again.
//                    Completely disabling of PLL is not working in 5secs shine noise test cases,
//                    instead PLL adaptation is slow down by limiting the Max phase error threshold.
//                    Grep for "XDSLRTFW-1104"
//
// 3/01/2014 Anantha Ramu: Did additional changes to PLL control uncorrected DTUs.
//                         Grep for: XDSLRTFW-1271 DSReTx_uncorrected_DTUs_PLL_changes
//
// 7/01/2014 Anantha Ramu: As random link drops were obsereved in showtime, the pll ref resetting and
//                         pll scaling modifications removed.
//                         Grep for: XDSLRTFW-1483 DSReTx_uncorrected_DTUs_PLL_changes
//
// 24/11/2015 Abu Rahman : PLL FDQ Adaptation Algorithm Modification.
//                            This new algorithm can handle Micro interruption, REIN,SHINE,PEIN and reasonable reference frequency drift and
//                            it is common for all VDSL2 modes.
//                            Grep for XDSLRTFW-2481
// 07/12/2016 XDSLRTFW-3015 SHINE protection lower than reported : it is a PLL issue!
//            i. Pilot tone phase error saturation value while external noise detected was 1 degree.
//               When long SHINE noise comes, sometime it deviated the pilot tone phase in a way that
//               SHINE noise pulse appeared longer in Rx Path than the original SHINE pulse. As a result,
//               in some corner cases ( eg. SHINE protection 58 DMT symbol, SHINE noise 56 DMT symbol),
//               SHINE protection did not work well. To solved this issue, new pilot tone phase saturation values are introduced.
//
//           ii a. On very short loops, with low power SHINE pulses ( eg. 10 dB more than the background  noise),
//                 external noise detection threshold was not optimum.
//           ii b. On very long loops, where the DS data rates are < 5Mbps and Max Erasure Metric in Showtime is around 10, fixed threshold
//                 for noise detection logic may lead false external noise detection in case of margin verification test.
//                 To solve this issue, new noise detection threshold logic is introduced.
//
//              Grep for: XDSLRTFW-3015
//********************************************************************************************

#include "common.h"
#include "dsp_op.h"
#include "pll.h"
#include "gdata.h"
#include "arctan.h"
#include "cosine.h"
#include "dsp_op.h"
#include "PrintTrail.h"
#include "IRI_Iof.h"
#include "TimingRecov.h"
#include "mul.h"
#include "ghs.h"
#include "profile.h"
#include "vdsl_state.h"
#include "DshInterface.h"
#include "cmv_Data.h"
#include "cmv.h"
#include "cri_iof.h"
#include <stdlib.h>


static FlagT gft_RxFirstShowtime = 1;
int16 gs_PhaseErrThresh = (PLL_PI_RADIANS/30); //6 degree

// XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement

/*****************************************************************************
;   Prototype:
;      void PLL(int16 *psa_InPilotTone)
;
;   Description:
;      This subroutine implements the Phase Lock Loop Operation using three steps:
;
;      1) Computes the phase error between the input pilot tone and reference tone.
;      2) Filter the phase error by the loop filter (having the transfer function of
;      (Kp + Ki/(1-1/z))
;      3) Apply the noise shaping filter to the output of the loop filter to reduce
;      the quantization noise (due to the ADC of the VCXO) in low frequency
;
;   Arguments:
;      psa_InPilotTone -- pointer to the buffer storing the input pilot tone
;
;   Global Variables: none
;
;*************************************************************************************/

void PLL(PT_ArrIdx_te PilotCalc)
{
   int32 l_Acc0, l_Acc1, l_temp;
   int16 sa_PilotToneTemp1_Re,sa_PilotToneTemp1_Im;
   int16 sa_PilotToneTemp2_Re,sa_PilotToneTemp2_Im;;
   int16 s_temp = 0;
   sa_PilotToneTemp1_Re = gt_PilotConfig.ta_PilotTones[PilotCalc].s_PilotTone_Re;
   sa_PilotToneTemp1_Im = gt_PilotConfig.ta_PilotTones[PilotCalc].s_PilotTone_Im;

   {
      extern FlagT gs_Overflow;

      //Scale up the input pilot tone to improve the numerical precision
      //note: gt_PilotConfig.ta_PilotTones[UsedPilot].s_PilotToneScale_over_4 is computed in ResetPllRefTone()
      l_Acc0 = sa_PilotToneTemp1_Re * gt_PilotConfig.ta_PilotTones[PilotCalc].s_PilotToneScale_over_4;
      l_Acc1 = sa_PilotToneTemp1_Im * gt_PilotConfig.ta_PilotTones[PilotCalc].s_PilotToneScale_over_4;

      //XDSLRTFW-1271 DSReTx_uncorrected_DTUs_PLL_changes   (End)
      //Check if there is any overflow
      sa_PilotToneTemp2_Re = sature16(l_Acc0);

      if(gs_Overflow == FALSE)
      {
         sa_PilotToneTemp2_Im = sature16(l_Acc1);

         if(gs_Overflow == FALSE)
         {
            //If no overflow, use the scaled pilot tone
            sa_PilotToneTemp1_Re = sa_PilotToneTemp2_Re;
            sa_PilotToneTemp1_Im = sa_PilotToneTemp2_Im;
         }
      }
   }

   /* ========================================================================== */
   /* Multiply the input tone with the complex conjugate of the refence tone */
   /* and compute the phase error (which is the angle of the multiplied vector) */
   /* ========================================================================== */
   //XDSLRTFW-1271 DSReTx_uncorrected_DTUs_PLL_changes   (Start)

   /* Real part of (InPilotTone * complex conjugate of PllRefTone) */
   MULS16(l_Acc0, gt_PilotConfig.ta_PilotTones[PilotCalc].s_PllRefTone_Re, sa_PilotToneTemp1_Re);
   MULS16(l_temp, gt_PilotConfig.ta_PilotTones[PilotCalc].s_PllRefTone_Im, sa_PilotToneTemp1_Im);
   l_Acc0 += l_temp;

   /* Imag part of (InPilotTone * complex conjugate of PllRefTone) */
   MULS16(l_Acc1, gt_PilotConfig.ta_PilotTones[PilotCalc].s_PllRefTone_Re, sa_PilotToneTemp1_Im);
   MULS16(l_temp, gt_PilotConfig.ta_PilotTones[PilotCalc].s_PllRefTone_Im, sa_PilotToneTemp1_Re);
   l_Acc1 -= l_temp;


   //XDSLRTFW-1271 DSReTx_uncorrected_DTUs_PLL_changes   (End)
   gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError = FastAtan((int16)(l_Acc1 >> 16), (int16)(l_Acc0 >> 16));


   //XDSLRTFW-2508 Start
   if (gs_RxState == R_O_SHOWTIME_RX)
   {
      //XDSLRTFW-2509 Start
      if (gft_RxFirstShowtime)
      {
         gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError = MapPllRefTone(PilotCalc, gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError);
         if (PilotCalc == PT_ARRAY_IDX_2) // We have three calls of the PLL() function for each pilot tone, switch of at the last call PT2
         {
            gft_RxFirstShowtime = FALSE;
            gft_DeRotateMedleyPilot = 0;
            //gft_EnablePilotStream = TRUE;
         }
      }
      //XDSLRTFW-2509 End

      //PllPhaseOffsetZoneMap(gs_PhaseError);//XDSLRTFW-2508
   }


   // Disable rotating the pilot tone reference
   // During decision-directed PLL, assume abs(phase error) <= 45 degrees.   .
   // Otherwise, map phase error to within +/- 45 degrees region.
   if (gft_DeRotateMedleyPilot || gt_PilotConfig.ta_PilotTones[PilotCalc].ft_DeRotatePilot)
   {

      if (gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError > PLL_HALF_PI_RADIANS+PLL_QUARTER_PI_RADIANS)
      {
         gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError -= PLL_PI_RADIANS;
      }

      else if (gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError > PLL_QUARTER_PI_RADIANS)
      {
         gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError -= PLL_HALF_PI_RADIANS;
      }

      else if (gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError < -(PLL_HALF_PI_RADIANS+PLL_QUARTER_PI_RADIANS))
      {
         gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError += PLL_PI_RADIANS;
      }

      else if (gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError < -PLL_QUARTER_PI_RADIANS)
      {
         gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError += PLL_HALF_PI_RADIANS;
      }
   }

   if (gt_PilotConfig.te_UsedPTArrayIdx == PilotCalc)
   {
      gs_PhaseError = gt_PilotConfig.ta_PilotTones[PilotCalc].gs_PhaseError;

      // Limit Phase Error
      if (gs_PhaseError >= gs_MaxPhaseErrorThreshold)
      {
         gs_PhaseError = gs_MaxPhaseErrorThreshold;
      }
      else if (gs_PhaseError <= -gs_MaxPhaseErrorThreshold)
      {
         gs_PhaseError = -gs_MaxPhaseErrorThreshold;
      }

      /* ============================================================================= */
      /* Filter the phase error by the loop filter (Kp + Ki/(1-z^(-1))) */
      /* x(n) = x(n-1) + (Kp + Ki)* PhaseError(n) - Kp*PhaseError(n-1) */
      /* ============================================================================= */

      /* Compute integral portion: */
      MULS16(l_temp, gt_PilotConfig.ta_PilotTones[PilotCalc].s_Ki, gs_PhaseError);
      l_Acc0 = l_add(gl_pll_freq_offset, l_temp);
      gl_pll_freq_offset = l_Acc0;

      /* Add proportional part: */
      MULS16(l_temp, gt_PilotConfig.ta_PilotTones[PilotCalc].s_Kp, gs_PhaseError);
      l_Acc0 = l_add(l_Acc0, l_temp);
      gl_pll_loopfilter_out = l_Acc0;

      // Keep track of Max Absolute Phase Error
      l_temp = abs(gs_PhaseError);
      if (gs_MaxPhaseError < (int16)l_temp)
      {
         gs_MaxPhaseError = (int16)l_temp;
      }
   }//endif (gt_PilotConfig.te_UsedPTArrayIdx == PilotCalc)
}
// XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement



// XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement

/*****************************************************************************
;  Prototype:
;     void MapPllRefTone(PT_ArrIdx_te ArrayIdx, int16 s_InTone_Imag);
;
;  Description:
;     Maps the  PLL reference point once when reaching showtime (shift the PLL-reference point by 90, 180 or -90 degree) based on the value of gs_PhaseError
;         This will allow 180deg Phase Error
;  Input Arguments:
;     PT_ArrIdx_te UsedPilot -- Pilot tone index of currently used pilot tone
;     s_PhaseError -- Phase error calculated based on initial value value of PLL reference tone
;
;  Output Arguments: s_PhaseError
;
;  Global Variables: none
;
;*************************************************************************************/

int16 MapPllRefTone(PT_ArrIdx_te ArrayIdx, int16 s_PhaseError)
{
   int16 s_temp;

   if (s_PhaseError > (PLL_HALF_PI_RADIANS+PLL_QUARTER_PI_RADIANS)) // > 135 Degrees
   {
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re = -gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im = -gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im;
      s_PhaseError -= PLL_PI_RADIANS;
   }
   else if (s_PhaseError < -(PLL_HALF_PI_RADIANS+PLL_QUARTER_PI_RADIANS)) // < -135 Degrees
   {
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re = -gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im = -gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im;
      s_PhaseError += PLL_PI_RADIANS;
   }
   else if (s_PhaseError > PLL_QUARTER_PI_RADIANS)   // > 45 Degrees
   {
      s_temp = gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re = -gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im = s_temp;
      s_PhaseError -= PLL_HALF_PI_RADIANS;
   }
   else if (s_PhaseError < -PLL_QUARTER_PI_RADIANS)  // < -45 Degrees
   {
      s_temp = gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Re = gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PllRefTone_Im = -s_temp;
      s_PhaseError += PLL_HALF_PI_RADIANS;
   }

   return (s_PhaseError);
}
// XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement


// gs_SectorIdx_PhaseError =  -3 if        Phase       [ -90      :  -180  ] degree
// gs_SectorIdx_PhaseError =  -2 if        Phase       [ -45      :   -90  ] degree
// gs_SectorIdx_PhaseError =  -1 if        Phase       [ -22.5    :   -45  ] degree
// gs_SectorIdx_PhaseError =   0 if        Phase       [ -22.5    :    22.5] degree
// gs_SectorIdx_PhaseError =   1 if        Phase       [  22.5    :    45  ] degree
// gs_SectorIdx_PhaseError =   2 if        Phase       [  45      :    90  ] degree
// gs_SectorIdx_PhaseError =   3 if        Phase       [  90      :   180  ] degree
// gs_SectorIdx_PhaseError =  99 if        pilot tone is not valid (amplitude too low)

void PllPhaseOffsetZoneMap(int16 s_PhaseError)
{
   gs_Prev_SectorIdx_PhaseError = gs_SectorIdx_PhaseError;
   gs_SectorIdx_PhaseError =  PHASE_ERROR_p00_to_p22deg;       // between 0 and +/- 22.5 deg

   if (s_PhaseError > PLL_HALF_PI_RADIANS)                     // between    90 and  180 deg
   {
      gs_SectorIdx_PhaseError = PHASE_ERROR_p90_to_p180deg;
   }
   else if (s_PhaseError > PLL_QUARTER_PI_RADIANS)             // between    45 and   90 deg
   {
      gs_SectorIdx_PhaseError = PHASE_ERROR_p45_to_p90deg;
   }
   else if (s_PhaseError > PLL_PI_BY_8_RADIANS_22p5DEG)        // between  22.5 and   45 deg
   {
      gs_SectorIdx_PhaseError = PHASE_ERROR_p22_to_p45deg;
   }
   else if (s_PhaseError < -PLL_HALF_PI_RADIANS)               // between   -90 and -180 deg
   {
      gs_SectorIdx_PhaseError = PHASE_ERROR_m90_to_m180deg;
   }
   else if (s_PhaseError < -PLL_QUARTER_PI_RADIANS)            // between   -45 and  -90 deg
   {
      gs_SectorIdx_PhaseError = PHASE_ERROR_m45_to_m90deg;
   }
   else if (s_PhaseError < -PLL_PI_BY_8_RADIANS_22p5DEG)       // between -22.5 and  -45 deg
   {
      gs_SectorIdx_PhaseError = PHASE_ERROR_m90_to_m180deg;
   }

   if (gs_SectorIdx_PhaseError == gs_Prev_SectorIdx_PhaseError)
   {
      if (gs_NumOfIdenticalAdjacentPTPhases < gs_MinNumberOfAdjPhaseErrorsToDeclareNewPhase)
      {
         gs_NumOfIdenticalAdjacentPTPhases++;
      }
      else
      {
         if (gs_SectorIdx_PhaseError != gs_Last_Reported_Phase_Error)
         {
            //
            gs_Last_Reported_Phase_Error = gs_SectorIdx_PhaseError;
            gta_PllPhaseRotInfo[gus_PhaseErrorReporting_Idx].l_SymbolCount  = gl_RxSymbolCount;
            gta_PllPhaseRotInfo[gus_PhaseErrorReporting_Idx].s_SectorIdx    = gs_SectorIdx_PhaseError;
            gta_PllPhaseRotInfo[gus_PhaseErrorReporting_Idx].s_PhaseError   = s_PhaseError;
            gus_PhaseErrorReporting_Idx++;

            if (gus_PhaseErrorReporting_Idx == 20)
            {
               gus_PhaseErrorReporting_Idx = 0;
            }
         }
      }
   }
   else
   {
      gs_NumOfIdenticalAdjacentPTPhases = 0;
   }
}

// XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
/*****************************************************************************
;  Prototype:
;     void ResetPLL(PT_ArrIdx_te ArrayIdx, int16 s_Kp, int16 s_Ki, s_PhaseErrThreshold);
;
;   Description:
;      This subroutine resets the loop filter coefficients of PLL.
;
;   Input Arguments:
;      PT_ArrIdx_te UsedPilot -- Array Index of the currently used pilot tone
;      s_Kp -- new value (before scaling) which Kp variable to be set to
;      s_Ki -- new value (before scaling) which Ki variable to be set to
;      s_PhaseErrThreshold -- new value for max phase error.
;
;   Output Arguments: none
;
;   Global Variables:
;      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PilotToneIdx   --   To determine scaling of coeficient which
;                     are in initialized to work for PILOT_TONE
;
;*************************************************************************************/

void ResetPLL(PT_ArrIdx_te ArrayIdx, int16 s_Kp, int16 s_Ki, int16 s_PhaseErrThreshold)
{
   int32 l_Kp, l_Ki;

   // This scales the loop filter coefficients of PLL.
   // which are initialized to work for PILOT_TONE
   l_Kp = ((int32)PILOT_TONE*s_Kp)/gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PilotToneIdx;
   l_Ki = ((int32)PILOT_TONE*s_Ki)/gt_PilotConfig.ta_PilotTones[ArrayIdx].s_PilotToneIdx;

   // Lower and upper bound Kp & Ki while assuring that ratio is constant
   // Note: Kp >= Ki always
   if (l_Ki < 1)
   {
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Ki = 1;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Kp = (int16) ((int32)(gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Ki * s_Kp) / s_Ki);
   }
   else if (l_Kp > 32767)
   {
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Kp = 32767;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Ki = (int16) ((int32)(gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Kp * s_Ki) / s_Kp);
   }
   else
   {
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Kp = (int16)l_Kp;
      gt_PilotConfig.ta_PilotTones[ArrayIdx].s_Ki = (int16)l_Ki;
   }

   gs_MaxPhaseErrorThreshold = s_PhaseErrThreshold;
}
// XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement


// XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
/*****************************************************************************
;   Prototype:
;      void ResetPllRefTone(PT_ArrIdx_te PilotIdx, PilotConfig_t *pt_PilotConfig, int16 *ps_InTone_Real, int16 *ps_InTone_Imag, int16 s_NumPllRefTones);
;
;   Description:
;      This subroutine resets the reference tone for PLL, i.e. multiple or single pilot tone reference.
;
;   Input Arguments:
;      PilotIdx         -- Pilot tone Array index
;      pt_PilotConfig   -- Pointer to pilot tone struct
;      ps_InTone_Real   -- Pointer to real part of the input reference tone
;      ps_InTone_Imag   -- Pointer to imaginary part of the input reference tone
;      s_NumPllRefTones -- Number of pilot tones to be reset
;
;   Output Arguments:
;         gt_PilotConfig.ta_PilotTones[j].s_PllRefTone_Re
;         gt_PilotConfig.ta_PilotTones[j].s_PllRefTone_Im
;         gt_PilotConfig.ta_PilotTones[j].ul_PilotToneRefPwr
;         gt_PilotConfig.ta_PilotTones[j].s_PilotToneScale, with j = 0 to (MAX_NUM_PILOT_TONES-1)
;
;   Global Variables:
;         gt_PilotConfig.ta_PilotTones[j].s_PilotTone_Re
;         gt_PilotConfig.ta_PilotTones[j].s_PilotTone_Im, with j = 0 to (MAX_NUM_PILOT_TONES-1)
;
;   Note: Only one pointer input set is allowed to be used, i.e. struct pt_PilotConfig or array pointers ps_InTone_Real/ps_InTone_Imag.
;         Examples:
;           Array:   All PTs  : ResetPllRefTone(PT_ARRAY_IDX_0, NULL, &gsa_AvgRefPilotToneReal[0], &gsa_AvgRefPilotToneImag[0], (int16)3);
;                    Single PT: ResetPllRefTone(PT_ARRAY_IDX_X, NULL, &gsa_AvgRefPilotToneReal[0], &gsa_AvgRefPilotToneImag[0], (int16)1);  with X = 0, 1 and 2
;           Struct:  ResetPllRefTone(PT_ARRAY_IDX_0, &gt_PilotConfig, NULL, NULL, (int16)3);
;
;*************************************************************************************/
void ResetPllRefTone(PT_ArrIdx_te ArrayIdx, PilotConfig_t *pt_PilotConfig, int16 *ps_InTone_Real, int16 *ps_InTone_Imag, int16 s_NumPllRefTones)
{
   int32 l_magnitude, l_temp;
   int16 q, i, j;
   // XDSLRTFW-3835 (Start)
   int16 sa_RefPilotToneReal[MAX_NUM_PILOT_TONES];
   int16 sa_RefPilotToneImag[MAX_NUM_PILOT_TONES];


   // Mapping of PilotTone struct into an array to get a common code.
   if (pt_PilotConfig != NULL)
   {
      for (j = 0; j < MAX_NUM_PILOT_TONES; j++)
      {
         sa_RefPilotToneReal[j] = pt_PilotConfig->ta_PilotTones[j].s_PilotTone_Re;
         sa_RefPilotToneImag[j] = pt_PilotConfig->ta_PilotTones[j].s_PilotTone_Im;
      }
      ps_InTone_Real = &sa_RefPilotToneReal[0];
      ps_InTone_Imag = &sa_RefPilotToneImag[0];
   }

   // Set the correct Loop counter.
   // NOTE: Every pilot tone can be independently reset.
   //       e.g. s_NumPllRefTones = 1 and ArrayIdx = 2  -> 1 + 2 = 3
   s_NumPllRefTones = s_NumPllRefTones + ArrayIdx;
   // Secure check!
   // Todo: Replace later with "MTK_ASSERT(s_NumOfValues != 0, break);" which is planned for R6.
   if (s_NumPllRefTones > MAX_NUM_PILOT_TONES)
   {
      s_NumPllRefTones = MAX_NUM_PILOT_TONES;
   }

   for (j = ArrayIdx; j < s_NumPllRefTones; j++)
   {
   // XDSLRTFW-3835 (End)

      l_temp       = abs((int32)ps_InTone_Real[j]);
      l_magnitude  = abs((int32)ps_InTone_Imag[j]);

      /* Approx. magnitude by max(|I|, |Q|) + (11/32)*min(|I|, |Q|) */
      if (l_magnitude > l_temp)
      {
         l_magnitude = l_magnitude + (11*l_temp >> 5);
      }
      else
      {
         l_magnitude = l_temp + (11*l_magnitude >> 5);
      }

      /* Long division: */
      q = 0;
      l_temp = 1;
      for (i=0; i<15; i++)
      {
         q <<= 1;
         l_temp <<= 1;
         if (l_temp >= l_magnitude)
         {
            l_temp -= l_magnitude;
            q += 1;
         }
      }

      // if magnitude is very high, q result can be 0.
      // in this case, set q to 1.
      if (q == 0)
      {
         q=1;
      }

      // XDSLRTFW-645 / XDSLRTFW-674 (start)
      // This portion of the code makes a fast reboot of the CPE when it sees the pilot tone got corrupted.
      // Compute the pilot tone threshold for the FAST Loss!
      gt_PilotConfig.ta_PilotTones[j].s_PllRefTone_Re = (ps_InTone_Real[j]);  //NOTE: This code is obsolete when using s_InTone to calculate power in the next lines
      gt_PilotConfig.ta_PilotTones[j].s_PllRefTone_Im = (ps_InTone_Imag[j]);  //NOTE: This code is obsolete when using s_InTone to calculate power in the next lines

      // Compute the reference Pilot tone power
      gt_PilotConfig.ta_PilotTones[j].ul_PilotToneRefPwr =  (uint32)( ((int32)ps_InTone_Real[j]*(int32)ps_InTone_Real[j]) + ((int32)ps_InTone_Imag[j]*(int32)ps_InTone_Imag[j]) );

      // Initialize gs_FrameWithLowPilotPwrCnt to zero at the point when we (re)calculate the pilot tone reference power. This initialization make sure that
      // legacy counter value does not have any influence of rebooting the system.
      gs_FrameWithLowPilotPwrCnt = 0;

      // compute the threshold pilot tone power. Currently it is 1/16th of  the reference pilot tone (Squared)power
      gt_PilotConfig.ta_PilotTones[j].ul_PilotThresholdPwr = (gt_PilotConfig.ta_PilotTones[j].ul_PilotToneRefPwr >> 4);
      // XDSLRTFW-645 / XDSLRTFW-674 (end)

      /* Set reference tone to the input tone */
      gt_PilotConfig.ta_PilotTones[j].s_PllRefTone_Re = (int16)(ps_InTone_Real[j]*q);
      gt_PilotConfig.ta_PilotTones[j].s_PllRefTone_Im = (int16)(ps_InTone_Imag[j]*q);

      //Set the scale factor for scaling the pilot tone
      //(making the scale smaller to avoid overflowing the pilot tone)
      gt_PilotConfig.ta_PilotTones[j].s_PilotToneScale = q;
      gt_PilotConfig.ta_PilotTones[j].s_PilotToneScale_over_4 = gt_PilotConfig.ta_PilotTones[j].s_PilotToneScale >> 2;
      if(gt_PilotConfig.ta_PilotTones[j].s_PilotToneScale_over_4 == 0)
      {
         gt_PilotConfig.ta_PilotTones[j].s_PilotToneScale_over_4 = 1;
      }
   }
}
// XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement

// XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
void RunPLL(void)
{
   PT_ArrIdx_te UsedPilot = gt_PilotConfig.te_UsedPTArrayIdx;
   uint32 ul_CyclesBeforePll;
   int32 l_PllCycles;

   /* XDSLRTFW-2302 */
   if (gs_RxPMDFrameCount != RX_DATA_SYMBOLS_PER_SUPERFRAME)
   {
      // We're in data symbol
      CopyPilotTone(FALSE);
   }
   else
   {
      // We're in sync symbol
      CopyPilotTone(TRUE);   // XDSLRTFW-3034 (Start_End)
   }


#ifdef NTR_INTRPT_POLL_ENABLE
   if (gs_RxState==32)
   {
      poll_ntr();
   }
#endif //NTR_INTRPT_POLL_ENABLE


   // disable the PLL at each synch symbol since the synch symbol may not send correct pilot tone
   if ((gft_EnablePLL) && (gs_RxPMDFrameCount != RX_DATA_SYMBOLS_PER_SUPERFRAME))
   {
      // Compute Pilot tone amplitude square
      gt_PilotConfig.ta_PilotTones[0].ul_PilotTonePwr =  (uint32)( ((int32)gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Re*(int32)gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Re) + ((int32)gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Im *(int32)gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Im ) );
      gt_PilotConfig.ta_PilotTones[1].ul_PilotTonePwr =  (uint32)( ((int32)gt_PilotConfig.ta_PilotTones[1].s_PilotTone_Re*(int32)gt_PilotConfig.ta_PilotTones[1].s_PilotTone_Re) + ((int32)gt_PilotConfig.ta_PilotTones[1].s_PilotTone_Im *(int32)gt_PilotConfig.ta_PilotTones[1].s_PilotTone_Im ) );
      gt_PilotConfig.ta_PilotTones[2].ul_PilotTonePwr =  (uint32)( ((int32)gt_PilotConfig.ta_PilotTones[2].s_PilotTone_Re*(int32)gt_PilotConfig.ta_PilotTones[2].s_PilotTone_Re) + ((int32)gt_PilotConfig.ta_PilotTones[2].s_PilotTone_Im *(int32)gt_PilotConfig.ta_PilotTones[2].s_PilotTone_Im ) );

      // this section works only in Showtime
      if (gus_ShowtimeControl & MASK_TX_RX_SHOWTIME)
      {
         // XDSLRTFW-2481 (Start)
         if (gft_Calculate_EdMetric_PilotPwr_Threshold)  //@todo AH: move this calculation to foreground instead of time critical
         {
            // This section calculate the Average Pilot Tone Amplitude Square from frist three seconds of showtime data.
            // First three seconds of showtime (12000 symbol) gft_Calculate_EdMetric_PilotPwr_Threshold flag is set to one and then it reset to zero in
            // Calculate_DiscardFrame_Threshold(void) function.
            for(int j=0;j<MAX_NUM_PILOT_TONES;j++)
            {
               // NOTE 1 : The logic of finding Average power need to be changed!
               // NOTE 2 : This logic assume the FG(Pilot tone) is constant!!!
               if ( gt_PilotConfig.ta_PilotTones[j].ul_PilotTonePwr > gt_PilotConfig.ta_PilotTones[j].ul_RxMaxPilotPWR)
               {
                  gt_PilotConfig.ta_PilotTones[j].ul_RxMaxPilotPWR =  gt_PilotConfig.ta_PilotTones[j].ul_PilotTonePwr;
               }
               if ( gt_PilotConfig.ta_PilotTones[j].ul_PilotTonePwr < gt_PilotConfig.ta_PilotTones[j].ul_RxMinPilotPWR )
               {
                  gt_PilotConfig.ta_PilotTones[j].ul_RxMinPilotPWR =  gt_PilotConfig.ta_PilotTones[j].ul_PilotTonePwr;
               }

               if (gus_metric_eval_cnt == gus_metric_eval_period-10 )
               {
                  gt_PilotConfig.ta_PilotTones[j].ul_RxAvgPilotPWR = (gt_PilotConfig.ta_PilotTones[j].ul_RxMaxPilotPWR+gt_PilotConfig.ta_PilotTones[j].ul_RxMinPilotPWR) >>1;
                  gs_PhaseErrorThreshold = (5*143);    // 5 degree
               }
            }
         }
         else   // Calculate the average Pilot tone power
         {
            gus_NoiseDetected = 0;
            // XDSLRTFW-3015 (Start)
            // Mirco Interruption detection logic
            if (      (gt_PilotConfig.ta_PilotTones[UsedPilot].ul_PilotTonePwr < (gt_PilotConfig.ta_PilotTones[UsedPilot].ul_RxAvgPilotPWR>>2))
                  || ((gt_PilotConfig.ta_PilotTones[UsedPilot].ul_PilotTonePwr < (gt_PilotConfig.ta_PilotTones[UsedPilot].ul_RxAvgPilotPWR>>PILOT_PWR_REDUCTION_MI))&&((uint8)gul_metric_value <= guc_EdMetricThresholdDuringExtNoise)))
            {
               // Mirco interruption detected

               // During Micro Interruption, pilot tone does not have any real signal.It is better not to update pll based on this corrupted data.
               // A very small pilot tone phase error is used in cases false detection of micro interruption.
               gs_MaxPhaseErrorThreshold = gs_PhaseErrorThresholdDuringMI;   //    0.021 degree, XDSLRTFW-3015
               gus_NoiseDetected = MICRO_INTERRUPTION_DETECTED;
               gus_MicroInterruptionDetectionCnt++;
            }
            else if ((uint8)gul_metric_value <= guc_EdMetricThresholdDuringExtNoise)   //External Noise detection logic
            {
               // External noise detected

               // External noise can come from REIN,SHINE or even from the frequency shift of the CPE or CO crystal.
               // In case of REIN or SHINE nose, PLL could be freeze but in case of frequency drift of crystal,
               // PLL has to follow the phase. Since it is not possible to distinguish different noise sources,
               // a COMPROMISED phase error saturation value is used in case of external noise detected.
               // Experiment showed phase error saturation value 0.14 degree is good enough
               // to sustain respective SHINE noise for 62 SHINE INP protection and also good enough to follow 0.2 ppm/sec
               // crystal frequency ramp.
               gs_MaxPhaseErrorThreshold = gs_PhaseErrorThresholdDuringExtNoise;   //0.14 degree, XDSLRTFW-3015
               gus_NoiseDetected = SUDDEN_SHOWTIME_NOISE_DETECTED;
               gus_SymbolWithLowEdMetricCnt++;
            }
            else
            {
                // No noise or MI detected
               gs_MaxPhaseErrorThreshold = gs_PhaseErrorThreshold;  // 5 degree
            }
            // XDSLRTFW-3015 (End)
         }
      }
      // XDSLRTFW-2481 (End)

      //Call the PLL for all three pilot tones
      ReadRxTimer((uint32 *)&ul_CyclesBeforePll);
      PLL(PT_ARRAY_IDX_0);
      PLL(PT_ARRAY_IDX_1);
      PLL(PT_ARRAY_IDX_2);
      ReadRxTimer((uint32 *)&(uint32)l_PllCycles);
      l_PllCycles = l_PllCycles - ul_CyclesBeforePll;
      if ((l_PllCycles > 0 )&& (l_PllCycles > gul_MaxPllCycles))
      {
         gul_MaxPllCycles = l_PllCycles;
         DSH_SendEvent(DSH_EVT_PLL_Cycles,sizeof(uint32),(void *)&gul_MaxPllCycles);
      }


      UpdateTimingRecoveryHW(gl_pll_loopfilter_out);
   }
   else
   {
      UpdateTimingRecoveryHW(gl_pll_freq_offset);
   }


   // Tracee debug info
   // XDSLRTFW-3922
   if (gft_EnablePLL != gft_PrevPLLState)
   {
      // Excecute the stream only after Ghs.
      if ((gsa_IndirectStat0[0] >= STAT_FullInitState) &&
          (gsa_IndirectStat0[0] <= STAT_LoopDiagMode))
      {
         int32 l_MsgRaw[4];

         l_MsgRaw[0] = gt_PilotConfig.ta_PilotTones[0].ul_PilotTonePwr;
         l_MsgRaw[1] = gt_PilotConfig.ta_PilotTones[1].ul_PilotTonePwr;
         l_MsgRaw[2] = gt_PilotConfig.ta_PilotTones[2].ul_PilotTonePwr;
         l_MsgRaw[3] = gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].ul_PilotThresholdPwr;

         DSH_SendEvent(DSH_EVT_PLL_STATE, sizeof(l_MsgRaw), (void *)&l_MsgRaw);
      }
//      else
//      {
//         gta_DebugPll.gft_EnablePLL = gft_EnablePLL;
//         gta_DebugPll.gl_pll_freq_offset = gl_pll_freq_offset;
//         gta_DebugPll.gs_PhaseError = gs_PhaseError;
//         gta_DebugPll.gl_pll_loopfilter_out = gl_pll_loopfilter_out;
//
//         DSH_SendEvent(DSH_EVT_PLL_STATE,sizeof(gta_DebugPll),&gta_DebugPll);
//      }

      gft_PrevPLLState = gft_EnablePLL;
   }

#ifdef LOG_PILOT_CONSTELLATION
   if(gft_StartPilotLog)
   {
      //log pilot tone every gs_PilotLogStep symbol(s)
      if((gs_PilotLogSymCnt == 0) && (gs_PilotLogCnt <= (gs_PilotBufSize-2)))
      {
         gpsa_PilotBuf[gs_PilotLogCnt++] = gt_PilotConfig.ta_PilotTones[UsedPilot].s_PilotTone_Re;
         gpsa_PilotBuf[gs_PilotLogCnt++] = gt_PilotConfig.ta_PilotTones[UsedPilot].s_PilotTone_Im ;
      }

      gs_PilotLogSymCnt++;

      if(gs_PilotLogSymCnt == gs_PilotLogSymStep)
      {
         gs_PilotLogSymCnt = 0;
      }

      //Disable the logging when the buffer is full
      if(gs_PilotLogCnt == gs_PilotBufSize)
      {
         gft_StartPilotLog = 0;
      }
   }
#endif //#ifdef LOG_PILOT_CONSTELLATION
}
// XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement

