/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2005 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   snr_b.c
 *
 *   Background functions for computing SNRs.
 *
 *------------------------------------------------------------------------
 */

// ***********************************************************************************************************
// snr_b.c
//
// History
//
// 16/05/2012 Vinjam: Code Pointers connected to Downstream Virtual Noise feature
//            Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
//
// 10/07/2012 Vinjam: Added code to store the Show time SNR with out virtual noise in a seperate buffer.
//                     SNR is converted (floor) from 8.8 format to 7.1 format to save physical memory.
//                     Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
//
// 08/08/2012 Vinjam: Added a seprate buffer to hold SNR with out virtual noise for DELT mode
//                    It is valid for single port mode only (unbonded - 8a,12a,17a & 30a profiles)
//                    Grep for XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise
//
// 15/02/2013 Vinjam : Memory Optimization. Re-used gsa_RxPathAtten_Tmp[] to store SNR with out virtual noise during LDM in channel Ananlysis phase.
//                  gpca_DELT_RxWithOutVnSnrBuf = (int8 *)(&gsa_RxPathAtten_Tmp[0]);
//                     Grep for XDSLRTFW-726 Bug_All_VDSL2_All_MemoryOptimization
//
// 19/09/2013 Varun/Palaksha/Ram : Added new CMV INFO 238 0 ==> gives the Max mips count(Foregraound + Tc task) in showtime if test 0 0 is 0x1000
//                    Note that in VDSL Rx enters showtime first(look timing diagram of VDSL2 for more details). since we want showtime mips when both tx
//                    and rx in showtime we enable the profile tasking when tx enters showtime. foreground mips peaking occurs at tx in RPSynchro6 state.
//                    Hence condition is checked in foreground in showtime. Following additional improvements done
//                  - Added FG Task Averege MIPS profiling
//                  - Added a optimisation code to reduce the RxTc Mips
//                     Grep for "XDSLRTFW-1243 Feature_ALL_VDSL2_ALL_CMV_MipsProfiling"
// 19/02/2015 Fuss : XDSLRTFW-2112 - Fix for NMS (NoiseMarginSeparation). SNR add value was differenet between Medley
//                                   and Showtime! Some clean-up done too.
//                   Grep for XDSLRTFW-2112
// ************************************************************************************************************

#include "common.h"
#include "gdata.h"
#include "noiseacc.h"
#include "Snr_b.h"
#include "ConvertToDB.h"
#include "dsp_op.h"
#include "vdsl_state.h"
#include "ConvertToDB.h"
#include "SharedFuncs.h"
#include "GetTonePsd.h"
#include "ghs.h"
#include "IRI_Iof.h"
#include "cmv.h"

//XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start_End)
//Added a buffer to store SNR with out virtual noise in DELT mode
//SNR in 8.8 format is converted to 7.1 format to same physical memory
//XDSLRTFW-726 Bug_All_VDSL2_All_MemoryOptimization (Start_End)
extern int8 *gpca_DELT_RxWithOutVnSnrBuf;
extern uint8 guca_NMS[RX_MAX_NUM_TONES];

int8 gc_NMS_Offset_dB = 0;

/*^^^
 *------------------------------------------------------------------------
 *
 *  Description: Background function to calculate SNR
 *
 *  Prototype: void BgSnrCalc(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *------------------------------------------------------------------------
 *^^^
 */
void BgSnrCalc(void)
{
   uint16 s_ch;
   int16 s_pilotbias;
   int32 l_metric;
   int32 *pla_NoisePower;

#ifdef HW_SNR_FDQ
   //Read out the accumulated signal power
   //(Note: we run ReadAccumulatedNoise() in background only during the training, but not in showtime,
   //       since the noise accumulation buffer is also used for store the synch frame during showtime)
   //       so we have to make sure this buffer is not being used simultanously by two tasks.)
   if(gs_RxState != R_O_SHOWTIME_RX)
   {
      ReadAccumulatedNoise();
   }
#endif //#ifdef HW_SNR_FDQ

   /* convert 48 bits accumulated value to 32 bits; */
   /* shift is 2 extra to match NoiseAcc() */
   pla_NoisePower = gpla_RxAccumBuf; // Overwrite input buffer with 32-bit output.
//XDSLRTFW-1243 Feature_ALL_VDSL2_ALL_CMV_MipsProfiling (Start)
   if(gs_RxState != R_O_SHOWTIME_RX)
   {
      RoundNoiseAccum(gpla_RxAccumBuf, pla_NoisePower, gs_LeftChannel, gs_RightChannel, gs_AlgLog2NumFramesToAccum);
   }
   else
   {
      RoundNoiseAccum2(gpla_RxAccumBuf, pla_NoisePower, gs_LeftChannel, gs_RightChannel, gs_AlgLog2NumFramesToAccum);
   }

//XDSLRTFW-1243 Feature_ALL_VDSL2_ALL_CMV_MipsProfiling (End)
   pla_NoisePower = gpla_RxAccumBuf;

   SnrCalc(pla_NoisePower, gpsa_MeasuredSnrBuf+gs_LeftChannel, gs_LeftChannel, gs_RightChannel);
   // XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
   // Update pilot tone metric values of each pilot tone with the new SNR measurement
   for (uint16 i = 0; i < MAX_NUM_PILOT_TONES; i++)
   {
      s_ch = gt_PilotConfig.ta_PilotTones[i].s_PilotToneIdx;
      s_pilotbias = (ConvertToDB(s_ch)) << 1;   // 20*log10(tone idx) for weighting metric with tone idx
      l_metric = s_pilotbias + gpsa_MeasuredSnrBuf[s_ch];
      gt_PilotConfig.ta_PilotTones[i].l_Metric = l_metric;
   }

   // XDSLRTFW-3702 (Start) Give DS3-band pilot tone higher priority for 35B if SNR is above a certain threshold
   if( ((gt_PilotConfig.s_PilotToneControl & DISABLE_PREFER_DS3_PILOT)==FALSE) &&
       (gpsa_MeasuredSnrBuf[gt_PilotConfig.ta_PilotTones[2].s_PilotToneIdx] > (30<<8)) )      // boost the DS3 band pilot tone if flag is enabled and we have more than 30dB on this pilot tone
   {
      gt_PilotConfig.ta_PilotTones[2].l_Metric += (6<<8); //boost the pilot tone metric by 6dB. This value was obtained empirically
   }
   // XDSLRTFW-3702 (End)

   // XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement
   guc_SnrCalcState = TRAINING_DONE;
}


/********************************************************************************************
;   Subroutine Name: SnrCalc
;
;   Description:
;      This routine computes the SNR for each channel in [s_first_chan, s_last_chan] in the
;      following equation:
;
;      SNR = 10*log10(referece_tone_power/average_noise_power)
;
;      reference_power = (0x2000)^2 + (0x2000)^2) = 2^26 + 2^26 =  2^27
;
;      average_noise_power = la_NoisePower[i]
;
;      SNR[i] = 10*log10(2^27) - 10*log10(la_NoisePower[i])
;
;      The implementation is done in the fixed-point algorithmetic and final SNR
;      is expressed in Q8.8 format
;
;   Prototype:
;      void SnrCalc(int32 *pla_NoisePower, int16 *psa_snr, int16 s_first_chan, int16 s_last_chan);
;
;   Input Arguments:
;      s_first_chan -- first channel
;      s_last_chan -- last channel
;      pla_NoisePower -- pointer to accumulators for storing average noise power
;
;   Output Arguments:
;      psa_snr -- pointer to an array of SNR values (in dB)
;
;   Return Value:
;      none
;
;   Global Variables:
;      none
;
;****************************************************************************/

void SnrCalc(int32 *pla_NoisePower, int16 *psa_snr, int16 s_first_chan, int16 s_last_chan)
{
   int16 i, s_ch, s_VirtualSNR=0;
   int16 s_NumBrkpts;
   int32 l_Acc, l_Acc1, l_NoisePower;
   int16 *psa_FineGain, s_RxMaxNomPsd;
   int16 s_FineGain, s_VirtNoisePsd;
   int16 s_RefSignalPSD;
   int32 l_Gain;
   PSDDescriptorTable_t *pt_PSDDesc;


   psa_FineGain = ghpsa_RxFineGains_Inactive;

   /* ============================================================================= */
   /* Compute SNR for each channel i */
   /* ============================================================================= */
   for (s_ch=s_first_chan; s_ch<=s_last_chan; s_ch++)
   {
      i = s_ch - s_first_chan;

      l_NoisePower  = pla_NoisePower[i];

      // If noise power is 0, increment to 1 to avoid getting an "infinite" SNR.
      if (l_NoisePower == 0)
      {
         l_NoisePower = 1;
      }

      /* ================================================================================ */
      /* compute 10*log10(pla_NoisePower[i]), in Q24.8 format */
      /* ================================================================================ */
      l_Acc = (int32)ConvertToDB(l_NoisePower);


      /* ================================================================================ */
      /* Load signal_dB = 10*log10(2^27) = SNR_SIG_MANT */
      /* and represent it in Q24.8 format */
      /* ================================================================================ */
      l_Acc1 = (int32)SNR_SIG_MANT;

      // apply constellation gain to signal power calculation since it is added to
      // error signal used during showtime noise accumulation
      if (gft_ApplyConstGain)
      {
         // add 20*log10(Constellation Gains[s_ch]/Constellation Gains[2])
         l_Acc1 += gusa_ConstGainPow_dB[ghpuca_RxBat_Inactive[s_ch]];
      }

      /* ================================================================================ */
      /* Compute SNR = signal_dB - noise_dB in Q10.6 format (1 bit for sign bit) */
      /* ================================================================================ */
      l_Acc = l_Acc1 - l_Acc;

      /* Saturate the result and represent SNR in Q8.8 format */
      // fine gain is set to 0 (negative infinity in dB) at bitloading to block tones with
      // extremely low SNR from being bitloaded. do not update SNR for these tones during showtime.
      // does not interfere with Reverb/Medley SNR since fine gain is initialized to unity at link start
      if (psa_FineGain[s_ch] != 0)
      {
         psa_snr[i] = sature16(l_Acc);
         //XDSLRTFW-3823 Add SNR Offset to the Showtime SNR (start)
         if(s_ch < gt_SNROffsetBitLoad[gs_ToneGroupIdx+1].s_ToneGroupStartIdx)
         {
            psa_snr[i] += gt_SNROffsetBitLoad[gs_ToneGroupIdx].s_ToneGroupSNROffset; //in 8.8 format (ex: -2dB => 0xFE00)

         }
         else
         {
            gs_ToneGroupIdx = gs_ToneGroupIdx+1;
            //RX_MAX_NUM_TONES value @gt_SNROffsetBitLoad[gta_SNROffset.s_NoOfToneGroups+1].s_ToneGroupStartIdx
            // will ensure gs_ToneGroupIdx will not go above "gta_SNROffset.s_NoOfToneGroups".&& "gta_SNROffset.s_NoOfToneGroups" capped to SNR_OFFSET_TONEGROUPS
            psa_snr[i] +=  gt_SNROffsetBitLoad[gs_ToneGroupIdx].s_ToneGroupSNROffset;
         }


         if ((gft_dbg_ShowSnr == 0) &&
               (gus_ShowtimeControl & MASK_TX_RX_SHOWTIME) &&
               (ghpsa_RxFineGains_Inactive[s_ch] != 0x2000))
         {
            // Skip in case of 0dB fine gain

            /* ================================================================================ */
            /* Compute SNR = SNR_dB - Fine Gains_dB */
            /* ================================================================================ */

            l_Gain = NormAndDivide_32by16bit(((int32)1<<30), psa_FineGain[s_ch]);

            //l_Gain is the linear gain in Q3.13
            l_Gain = round(l_Gain, 4);

            //Convert the linear fine gain to dB (Q8.8)
            s_FineGain = ConvertToDB(l_Gain);

            //Since the input to ConvertToDB Q3.13 format and
            //ConvertToDB computes 10log10(gain), but we need to compute 20*log10(gain)
            //so we need to adjust s_FineGain to get the desired dB (in Q8.8)
            s_FineGain = (s_FineGain<<1) - 0x4E44; // 0x4E44 = 20*log10(2^13) * 256

            psa_snr[i] = psa_snr[i] - s_FineGain;

         } // end of if showtime

      }   // if (psa_FineGain[s_ch] != 0)
      else
      {
         // SNR for tones with zero RX FG should be set to a special value
         // these are (1) unloaded tones with "less-than threshold" bad Medley SNR, (2) PARR tones, (3) RFI tones
         // pilot tones should have a real SNR
         // unused RX tones have 0 for SNR (SNR is not computed at all)
         psa_snr[i] = (int16)OUT_OF_RANGE_SNR_PER_TONE;
      }

      //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start)

      /* ================================================================================ */
      /* Compute virtual noise SNR = Reference PSD - virtual noise_PSD in Q10.6 format (1 bit for sign bit) */
      /* ================================================================================ */
      // Get the virtual noise PSD
      pt_PSDDesc = (PSDDescriptorTable_t*)(void *)&gt_DS_RefVirtNoiseLevel_VDSL2;
      s_NumBrkpts = pt_PSDDesc->us_NumberOfTones;

      //Added Number of virtual noise break points check, as this code is also invoked even before
      //extracting the virtual noise break points
      if((gft_DSVirtualNoiseStatus) && (s_NumBrkpts > 0))
         //Below statement is as per Vinax CPE Firmware, need to cross check the reson for it
         //if((gft_DSVirtualNoiseStatus) && (gt_DsREFPSDDescriptorTable.us_NumberOfTones > 0))
      {

         if(gsa_IndirectStat0[0] == STAT_ShowTimeTCSyncState)
         {
            //Store SNR with out virtual noise breakpoints for normal mode
            //SNR in 8.8 format is converted to 7.1 format to same physical memory.
            //Truncated SNR contains 0.5 dB (floor) granularity
            gca_SnrBuf_WithOutVN[s_ch] = (char)((psa_snr[i] >> 7)&0xFF);
         }
         else
         {
            //Store SNR with out virtual noise breakpoints for Loop Diagnostics Mode (LDM)
            //Valid for single port mode only (12a, 17a & 30a Profiles only - in unbonded mode)
            //Valid for 8a profile in "unbonded" mode
            if((gsa_IndirectStat0[0] == STAT_LoopDiagMode) && (guc_PortMode == 0))
            {
               //Valid for Channel Analysis & Exchange Phase of LDM
               if(gs_RxState >= VDSL2_R_O_P_MEDLEY_RX)
               {
                  //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (Start_End)
                  //Store SNR with out virtual noise breakpoints for normal mode
                  //SNR in 8.8 format is converted to 7.1 format to same physical memory.
                  //Truncated SNR contains 0.5 dB (floor) granularity
                  //XDSLRTFW-726 Bug_All_VDSL2_All_MemoryOptimization (Start_End)
                  gpca_DELT_RxWithOutVnSnrBuf[s_ch] = (char)((psa_snr[i] >> 7) & 0xFF);
               }
            }
         }

         // Get the virtual noise PSD in 0.5 dB steps
         s_VirtNoisePsd = GetTonePsd(s_ch, pt_PSDDesc, gs_RxNumTones);//XDSLRTFW-3838

         // Get the reference signal PSD
//         if (gs_RxState == R_O_SHOWTIME_RX)
         {
            s_RxMaxNomPsd = gt_PwrConfigParam.s_Dn_MaxNomPSD;
            pt_PSDDesc = (PSDDescriptorTable_t*)(void *)&gt_DsREFPSDDescriptorTable;
            // Get the reference signal PSD in 0.1 dB steps
            s_RefSignalPSD = GetTonePsd(s_ch, pt_PSDDesc, gs_RxNumTones) - s_RxMaxNomPsd;//XDSLRTFW-3838
         }
// Note: The code brings only 200m during Analysis when VN is enabled!
//       But a clean concept would be needed so that the array contains the correct PSD for all cases when SnrHandler() is called!
//         else
//         {
//            s_RefSignalPSD = gsa_PsdPerTone[s_ch];
//         }

         // values from 201 to 255, correspond to a virtual noise level of minus infinity
         // dBm/Hz. Here we check against -281 because s_VirtNoisePsd is in dBm/Hz in 0.5dB format
         // and we have already added the offset of 80 (i.e. -40dBm/Hz).
         if (s_VirtNoisePsd <= -281)
         {
            s_VirtualSNR = (int16)0x7FFF; // biggest 16 bit number
         }
         else
         {
            if(s_VirtNoisePsd > -80)
            {
               // Convert to 0.5 dB steps while computing virtual noise SNR
               // limit the VN PSD to -40dBm if it is set higher than -40dBm by ME
               // this cannot happen at CPE side since decoded VN PSD is always lower than -40dBm
               s_VirtualSNR = (s_RefSignalPSD/5) + 80;
            }
            else
            {
               // Convert to 0.5 dB steps while computing virtual noise SNR
               s_VirtualSNR = (s_RefSignalPSD/5) - s_VirtNoisePsd;

            }
            // Convert to Q8.8 format
            s_VirtualSNR = s_VirtualSNR << 7;
         }

         // Take the minimum of the two SNRs
         if ((psa_snr[i] != (int16)OUT_OF_RANGE_SNR_PER_TONE) && (psa_snr[i] > s_VirtualSNR))
         {
            psa_snr[i] = s_VirtualSNR;
         }

      }

      //Limit the minimum SNR value to a predefined number
      if ((psa_snr[i] != (int16)OUT_OF_RANGE_SNR_PER_TONE) && (psa_snr[i] < MIN_SNR_DB))
      {
         psa_snr[i] = MIN_SNR_DB;
      }

      //XDSLRTFW-439: Feature_All_DS_All_All_SupportDsTxRefVirtualNoise (End)

      // XDSLRTFW-954 - Noise Margin Separation to Improve DS performance (START)
      // XDSLRTFW-2112 (Start_End)
      // XDSLRTFW-4216 :CTL, VDSL: Downstream low perf in mid range loops from 2000ft to 3000ft (NMS) (Start)
      // Code below are generic for NMS algorithm
      if (guc_NMS_AlgHandler == CALCULATE_SNR_DEFAULT_AGC_GAIN)
      {
         int16 s_MedleySnrWithAGCBoost, s_MedleySnr, s_NmsGain_dB;

         s_MedleySnrWithAGCBoost = gsa_PsdAnalysis_out[s_ch];             // 8.8 format
         // Note:Negative values are considered two lines below!
         s_MedleySnr = MAX(psa_snr[i], 0);                    // 8.8 format

         s_NmsGain_dB = s_MedleySnrWithAGCBoost - s_MedleySnr;

         // Lower bound of NMS gain
         // Note: gt_NoiseMarginChange.s_min_Used_NMS_Gain is >= 0.
         if (s_NmsGain_dB < gt_NoiseMarginChange.s_min_Used_NMS_Gain)
         {
            s_NmsGain_dB = 0;
            // This code could lead to an offset of xdB, even if the SNR with 3dB AGC boost will lead to an worth SNR than the original MedleySNR.
            // E.g. if all s_NmsGain_dB are negative then all values are set to xdB!
            // s_NmsGain_dB = gt_NoiseMarginChange.s_min_Used_NMS_Gain;
         }

         // Upper bound of NMS gain
         if (s_NmsGain_dB > gt_NoiseMarginChange.s_max_Used_NMS_Gain)
         {
            s_NmsGain_dB = gt_NoiseMarginChange.s_max_Used_NMS_Gain;
         }

         // Convert NMS gain to 4.4 dB format and store it to guca_NMS
         guca_NMS[s_ch] = (uint8)(s_NmsGain_dB >> 4);          // 4.4 format

         // Debug feature (start)
         if (gt_NoiseMarginChange.s_NM_Ctrl & OPTN_NoiseMarginChange_NM_Ctrl_NMS_NoSnr)
         {
            s_NmsGain_dB = 0;
         }
         // Debug feature (end)

         psa_snr[i] += (s_NmsGain_dB & (0xFFF0));                           // Add NMS gain to the measured SNR (8.8 dB format)
      }
      // XDSLRTFW-4216 :CTL, VDSL: Downstream low perf in mid range loops from 2000ft to 3000ft (NMS) (End)
      // Debug code to test SRA in 35bLite.
      // Note: Only non vectoring.
      if (gs_RxState == R_O_SHOWTIME_RX)
      {
         if (guc_NMS_AlgHandler == NOISE_MARGIN_SEP_DONE)
         {
            // Modify showtime SNR with the estimated NMS value
            psa_snr[i] += (guca_NMS[s_ch] << 4);
         }
        // commenting below lines this now handled through OPTN_29 CMV option
        //else if (guc_NMS_AlgHandler == NOISE_MARGIN_SEP_SRA_TEST)
        // {
        //    psa_snr[i] += (int16)(guca_NMS[s_ch] << 8);
        // }

         psa_snr[i] += (int16)(gc_NMS_Offset_dB << 8);

      }
      // XDSLRTFW-954 - Noise Margin Separation to Improve DS performance (END)
   }
}
