/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2008 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;
;   File Name: BitSwapDecision_VDSL2.c
;
;   This file contains functions for making the VDSL bit-swapping decision.
;
;   Change 25/05/2012 - VENKATESH S N
;   XDSLRTFW-464 : DS_MARGIN_LOWER_TO_BRCM_DISTRIBUTE_FINEGAIN
//
//  10/09/2013 Sooraj: Added Support of DS-SRA with ReTx
//                 Grep for XDSLRTFW-1063: SRAds_Support_with_DS_ReTx
//  05/12/2013 Kannan: When US ReTx is enabled consider 24Bits of DS RRC for
//                     DS bit swap Lp check.
//                     Grep for XDSLRTFW-1427
// 06/02/2014 Kannan:
//    (1) Showtime margin equalization is done in such a way that reduce fine gain
//    from Highest margin tone and increase the fine gain in smallest margin
//    by meeting per band Linear squared sum of Fine gain <= 1. Gain swap
//    is done across the bands, i.e both highest & lowest margin tones are in
//    different band. In case if not possible to swap the gain across the bands,
//    gain swap is done within the band.
//
//    (2) Gain swap is triggered if highest margin on a tone is > 1 dB and lowest margin is  < 1dB
//    as compared to average margin.
//
//
//    (3) While performing DS SRA, enabled a flag to stop adding bits beyond per tone margin threshold.
//    This is enabled by default. It can be disabled if required, this can done
//    by clearing bit Mask 2 at Test CMV 46 offset 0.
//
//
//    (4) Showtime margin equalization is enabled by default, It can be disabled if required, this can done
//    by clearing bit Mask 1 at Test CMV 46 offset 0.
//    Grep for XDSLRTFW-1784
// 15/12/2017 Kannan:
//    Used 2 different error codes for Bitloading and ReTx framing failures to identify
//        the cause for not triggering DS SRA.
//    Grep for XDSLRTFW-3562
//
// 09/05/2018 Abu Rahman: XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code
// Issue: After several 10th of thousands of DS bit swapping RMS FG per band tends to reach its maximum limit (2.5dB).
// Due to this DS bit swap stops. If this situation happens CPE does not trigger any DS bit swap even with the presence
// of RFI noise on DS band which could lead to link drop.
//
// Root Cause: Multiple reasons
// i.  After several thousand DS DS, RMS FG per band was not accurate any more. RMS FG  per band was calculated only
//     one time from inactive FG table and then always modified in FW where errors were accumulated and never corrected.
// ii. At every Bit swap initiation calculation RMS FG was modified. If BS decision algorithm decided to not trigger
//     any DS bitswap request then RMS FG was not reinitialized with correct value.
// iii.Due to above points RMS FG thresholds for FG allocation did not work at all and algorithm triggered DS bit swap
//     even actual RMS FG per bands were increasing.
//
// Solution:
// i.  Calculated RMS FG per band from inactive gain table every time before starting
//     BS algorithm in RxBitSwapDecision_VDSL2() function
// ii. Modified BS algorithm in a way that actual RMS FG (DS) always tries to be below
//     the initial RMS FG. This improvement makes sure that per tone DS FGs are evenly
//     distributed and around initial RMS FG.
// iii.If RMS FG go above the threshold then new algorithm tries to reduce FG across the tones in that band to mitigate the
//     RMS FG criteria.
//   Grep for XDSLRTFW-3807
//
// 13/07/2018 Abu Rahman: XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs
// DS FG violation after DS BS is resolved in XDSLRTFW-3807 by initializing Sum of fine gain linear square in gla_SumFGainLinSq[]
// variable every time before starting BS decision algorithm. In XDSLRTFW-3323,  this above mentioned  variable is initialized
// commonly every time before starting BS/SRA algorithm.
// Grep for XDSLRTFW-3323
//
// 20/12/2018 Abu Rahman
//            XDSLRTFW-4051 Fast downstream up shift SRA
//            This code supports Fast downstream up shift SRA
//            Maximum bits loaded/tone = 5
//            This feature is enabled by DTAG bit (CMV DSL 21 0 1) only against vectoring and BRCM based DSLAM.
//
//            Some other controls are done by CNFG 93 cmv
//            CNFG 93 0:
//             bit 0:
//                - value 0x8000: disable feature "Fast downstream up shift SRA"
//                - value 0x0000: enable  feature "Fast downstream up shift SRA" for BRCM CO and any  vector engine
//                - value 0x0001: enable  feature "Fast downstream up shift SRA" for BRCM CO and BRCM vector engine

//            CNFG 93 1:
//             - value 0xFFFF: do not disable feature "Fast downstream up shift SRA" in showtime
//             - value      X: disable feature "Fast downstream up shift SRA" after X sec of Showtime
//                             (supported range: 0-32767 sec, debug range 0-65534 sec)
//            CNFG 93 2:
//             - value 0xFFFF: Never disable DS-SRAs in Showtime
//             - value      Y: Disable DS SRAs if CPE receive 3 consecutive SRA rejections with rejection code SRA "not enabled"
//                             (0x3) after Y sec of Showtime (supported range: 0-32767 sec, debug range 0-65534 sec)
//            Example:
//            DSL 21[0] : 1 -DTAG bit enabled
//            CNFG 93[0]: 0 - meaning feature is enabled for all kind of VCE
//            CNFG 93[1]: 100 - Fast DS up shift SRA is enabled for 100 sec in Showtime
//            CNFG 93[2]: 200 - After 200 sec of Showtime if three consecutive SRA rejection come with SRA not enabled
//                              exception code (0x3) then DS SRA request will be disabled
//            Default CMV configuration: if DTAG bit is enabled
//            CNFG 93[0] = 0
//            CNFG 93[1] = 255
//            CNFG 93[2] = 255
//            Grep for XDSLRTFW-4005
//
// 05/04/2019 Stefan Krause
//            XDSLRTFW-4037 SRA can always lead to an unsupported framing
//            Framing generation for IDTU does cannot reach exactly the desired Lp (result shall be always equal to the desired
//            setting or a little bit lower
//            Concept for IDTU:
//             1. Calculate desired Lp (here: ul_Reconfig_Lp)
//             2. Calculate an IDTU-framing which result in an actual Lp which is smaller than / equal to the desired Lp
//             3. Modify SRA-request to ensure that it fits to the calculated "actual Lp"
//
//            Current limitation:
//            To avoid conflicts with Trellis bits, we consider for the bit-reduction only tones which have a bitloading
//            of more than 2 bits
//            Grep for XDSLRTFW-4037

;
;***************************************************************************/
#include <string.h>
#include <stdlib.h>
#include "common.h"
#include "gdata.h"
#include "dsp_op.h"
#include "cmv.h"
#include "Bitload_support.h"
#include "changebat.h"
#include "decimalgain.h"
#include "BitswapDecision_VDSL2.h"
#include "ConvertToDB.h"
#include "vdsl_xception.h"
#include "ToneReorder.h"
#include "mul.h"
#include "Bitload.h"
#include "GenFramingParams_VDSL2.h"
#include "DebugBuffer.h"
#include "vdsl_const.h"
#include "gdata.h"

int16 ConvertGi_dBtoLinear(void);

// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
int16 CountNumTonesChanged(void);


//XDSLRTFW-1784 (Showtime Margin Equalization - START)
#ifdef NOISE_MARGIN_EQU_DEBUG
uint16 gus_dbg_MinMgn_Neg = 0;
uint16 gus_dbg_minfg_exceed = 0;
#endif //NOISE_MARGIN_EQU_DEBUG
//XDSLRTFW-1784 (Showtime Margin Equalization - END)

FlagT CheckForSRAConstraints(int16 s_delta_Lp, int16 s_path);

//XDSLRTFW-1784 (Showtime Margin Equalization - START)
uint16 ShowtimeNoiseMarginEqualization(uint8 *puca_RxBat, int16 *psa_RxFineGains);
FlagT DistributeFineGainsOneTone(uint8 *puca_RxBat,
                                 int16 *psa_RxFineGains,
                                 int16 *psa_SNRBuffer,
                                 int16 s_ch,
                                 int16 s_band,
                                 int16 s_currMargin,
                                 int16 s_reqdMargin);
//XDSLRTFW-1784 (Showtime Margin Equalization - END)


/*^^^
*-----------------------------------------------------------------------------
*   Prototype : int16 CountNumTonesChanged(void)
*
*
*   Description : Count the number of tones changed during bitswap/ SRA
*
*   Input Parameters:
*
*   Output Parameters:
*
*
*   Global Variables:
*------------------------------------------------------------------------------------
*/

int16 CountNumTonesChanged(void)
{
   int16 s_NumOfBitSwapTones = 0;
   int16 i, s_band;

   for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
   {
      for (i=gsa_RxBandLeftChannel[s_band]; i <= gsa_RxBandRightChannel[s_band]; i++)
      {
         //If this tone is a bit-swap tone, store it directly (for forming message)
         if(IS_TONEFLAGSET(guca_RxBitswapToneSet, i))
         {
            s_NumOfBitSwapTones++;
         }

      } //for (i=gsa_RxBandLeftChannel[s_band]
   } //for (s_band=0; s_band <


   return(s_NumOfBitSwapTones);
}
// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]
#ifdef BS_INSIDE_DEBUG
void Log_BS_Info(uint8 *puca_RxBat, int16 *psa_RxFineGains,int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones_ToConsider)
{
   int16 s_ch, s_band;
   int16 s_BitAlloc, s_FineGain;
   int16 s_Margin, s_SNR;

   //gsa_StVectoringBuffer[gs_DbgBufferCnt++] = 31693; // 0x7BCD = 31693
   //gsa_StVectoringBuffer[gs_DbgBufferCnt++] = 2;
   //gsa_StVectoringBuffer[gs_DbgBufferCnt++] = gs_MaxFgReductionAfterOverShoot;

   //Loop through all the bands
   for (s_band = 0; s_band < gs_NumOfRxBands; s_band++)
   {

       //store  s_MinMargin info

         //Loop through all the channels in this band
         for (s_ch = gsa_RxBandLeftChannel[s_band]; s_ch <= gsa_RxBandRightChannel[s_band]; s_ch++)
         {
            //Get the bit allocation and gain of this tone
            s_BitAlloc = puca_RxBat[s_ch];
            s_FineGain = psa_RxFineGains[s_ch];

            //Only change the tones which have been affected in bit-swap, which can be added more gain
            //and which have been loaded
            if (IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch))
            {

               //Compute the SNR margin adjusted with fine gain and coding gain
               s_SNR = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
               s_SNR += psa_RxFineGains[s_ch];

               s_Margin = s_SNR - gsa_ConstellationSNR[s_BitAlloc];

               //if(gs_RxState == R_O_SHOWTIME_RX)
               {
                  // log  tone idx, final BAT, FG and Margin information of bit swapped tone
                  gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_ch;
                  gsa_BSDebugBuf[gs_DSDebugBufCnt++] = puca_RxBat[s_ch];
                  gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_Margin;
                  gsa_BSDebugBuf[gs_DSDebugBufCnt++] = psa_RxFineGains[s_ch];
               }

            } //if((IS_TONEFLAGSET(p_ActiveTones_ToConsider, i) == 1) &&
         } //for (i = gsa_RxBandLeftChannel[s_band];

   } //for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
}
#endif

// XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs (Start)
// XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs (End)
/*****************************************************************************
;   Subroutine Name: BgRxBitSwapDecision_VDSL2( )
;
;   This subroutine computes the SNR margin and decide if the bit-swap
;   should be initiated or not. If the bit-swap should be initiated,
;   it computes the new bits and gains for the bit-swap tones.
;
;   BS algorithm main sequence:
;   Notes :
;   i.   All steps below are independent.
;   ii.  Each step below is done for down-stream bands and after completing the current step move forward to the next step.
;   iii. RMS FGs computations and checks are done per band
;   iv.  RMS FG criteria: RMS FG per band shall never exceed the initial RMS FG per band.
;   v.   Initial RMS FG per band is calculated based on initial bit loading.
;
;   BS-S --> Bit Swap Step
;   BS-S-1.   Copy active BAT-FG to Inactive BAT-FG (done in RxBitswapHandler_VDSL2())
;   BS-S-2.   Read FG from inactive FG table and convert it from 3.13 linear to 8.8 dB format and store it in Inactive FG Table.
;             (done in RxBitswapHandler_VDSL2())
;   BS-S-3.   Compute RMS FG per band inside BS-S-2 in form of Sum of fine gain linear square in gla_SumFGainLinSq[] variable.
;   BS-S-4.   Define adaptive SNR based threshold which is use for initiating core bit swap algorithm
;   BS-S-5.   Decision algorithm which decides core BS algorithm or  Margin equalization(if enabled) or none will be run
;
;   Core BS algorithm:
;   BS-S-6.   Remove bit from tones which has lower margin than minimum margin defined in BS-S-4. Remove extra FG from that tone with maintaining
;             minimum margin criteria and update RMS FG value.
;   BS-S-7.   Add bits only on the tones which have sufficient margin to accommodate extra bits. This process does not change any FG.
;   BS-S-8.   If BS-S-7 can not allocate all the bits removed from BS-S-5 then allocate final bits by modifying FG on possible tones.
;             This process always checks for RMS FG  value criteria
;   BS-S-9.   If re-allocating removed tones from BS-S-5 fails due to RMS FG criteria then run again BS-S-8 for rest of bits by disabling
;             RMS FG check criteria (i.e. notes iv)
;   BS-S-10.  If the bit allocation is complete ( i.e. removing and adding bits) with BS-S-9 then remove FG from each "to be swapped" tones from
;             respective DS band with 0.1 dB steps. This process reduces the RMS FG per band to maintain the RMS FG per band criteria(i.e. note iv).
;   BS-S-11.  Add extra gain to the modified tones so that maximum margin is achieved. This process always check for RMS FG criteria to
;             add more gain.
;   BS-S-12.  Finally if the minimum margin after intended bit swapping is more than the minimum margin before bit swapping then convert
;             the inactive FG table from 8.8dB format to 3.13 linear format and send BS request FG and BAT to Active buffer and trigger
;             BS OLR message to perform DS BS.
;   BS-S-13.  Margin Equalization Algorithm
;
;   Prototype:
;   void BgRxBitSwapDecision_VDSL2(void)
;
;   Input Arguments:
;      psa_SnrBuf -- pointer to the SNR buffer computed during SHOWTIME
;
;   Output Arguments:
;      none
;
;   Return:
;      none
;
;   Global Variables:
;      gl_rx_sframe_count      -- (I) RX superframe count
;      gt_RxBitSwapInfo      -- (I) structure containing RX bit swap info
;
;      gsa_RxBat[]            -- (O) RX Bit Allocation Table
;      gsa_RxFineGains[]      -- (O) RX fine gain table
;      gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/

int16 gs_MinMarginBeforeSwap;

void BgRxBitSwapDecision_VDSL2(void)
{
   int16 *psa_RxFineGains, *psa_SnrBuf;
   uint8 *puca_RxBat;
   int16 s_MinMarginAfterSwap, s_ch, s_band, i, s_Avg_Margin;
   int16 s_Tcm_Ovhd;
   int32 l_delta_Lp, l_ActualDeltaSumLp;
   FlagT ft_Result, ft_MAXNOMATP_Flag;
   int16 s_ch_hi, s_band_hi, s_Margin_hi = 0;
   uint8 uc_bitswap_decision;
   int16 s_temp = 0;
   int32 l_SnrMgn;
   uint16 us_BiGiModificationControlFlags;
#ifdef BS_INSIDE_DEBUG
   int16 s_i;
#endif

   //Initialize variables
   gft_SRA_FAIL = FALSE;
   gft_RmsFgOverShoot = 0; // this variable is modified by BS-S-9 (in ChangeLpKeepMaxMargin() function) and is used by BS-S-10

   gft_RxBitSwapFlag = FALSE;
   gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_NO_ERROR;

   //XDSLRTFW-1784 (Showtime Margin Equalization - START)
   gs_NumOfBitSwapTones = 0;
   memset(guca_RxBitswapToneSet, 0, (gs_RxNumTones>>3));
   //XDSLRTFW-1784 (Showtime Margin Equalization - END)

   //Disable the MAXNOMATP check during the bit-swap
   ft_MAXNOMATP_Flag = gft_MAXNOMATP_Flag;
   gft_MAXNOMATP_Flag = FALSE;

   psa_RxFineGains = ghpsa_RxFineGains_Inactive;
   puca_RxBat = ghpuca_RxBat_Inactive;
   psa_SnrBuf = gpsa_MeasuredSnrBuf;

   //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
   // Debug Per Tone Margin Start
   for (i = 0; i < 4096; i++)
   {
      gsa_PerToneMargin[i] = 0;
   }

   //Loop through all the bands
   for (s_band = 0; s_band < gs_NumOfRxBands; s_band++)
   {
         //Loop through all the channels in this band
         for (s_ch = gsa_RxBandLeftChannel[s_band]; s_ch <= gsa_RxBandRightChannel[s_band]; s_ch++)
         {
            int16 s_BitAlloc;
            int16 s_FineGain;
            int16 s_SNR;

            //Get the bit allocation and gain of this tone
            s_BitAlloc = puca_RxBat[s_ch];
            s_FineGain = psa_RxFineGains[s_ch];

            if ((IS_TONEFLAGSET(guca_RxSupportedToneSet, s_ch)) && (s_BitAlloc !=0))
            {
               //Compute the SNR margin adjusted with fine gain and coding gain
               s_SNR = AddCodingGainToSnr(s_ch, psa_SnrBuf);
               s_SNR += psa_RxFineGains[s_ch];



               gsa_PerToneMargin[s_ch] = s_SNR - gsa_ConstellationSNR[s_BitAlloc]; // per tone margin in 8.8 dB format

            } //end if (IS_TONEFLAGSET(guca_RxSupportedToneSet, s_ch))
         } //end for (i = gsa_RxBandLeftChannel[s_band];

   } //end for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
   // Debug Per Tone Margin end
#endif


   //*******************************************************************************//
   // BS-S-3.   Compute RMS FG per band from 8.8dB formated inactive FG table(Start)//
   //*******************************************************************************//

   // In case of Bit swap decision failure, gla_SumFGainLinSq[band] are not restored back in many cases!
   // solution : Compute the sum of the linear gain square for each band at the beginning of RxBitswapHandler_VDSL2()
   // This computation is done in RxBitSwapGainConversion_VDSL2() function under XDSLRTFW-3323
   //*********************************************************************************************************//

   //*********************************************************************************************************//
   // BS-S-3 (End)                                                                                            //
   //*********************************************************************************************************//

   //*********************************************************************************************************//
   //BS-S-4.   Define adaptive SNR based threshold which is use for initiating core bit swap algorithm(Start) //
   //*********************************************************************************************************//
   //Compute the minimum allowed SNR margin
   // Bit swap algorithm tries to maintain DS margin at least the value of gs_MinTargetSnrMargin on bit swapped tones
   gs_MinTargetSnrMargin = MAX((gt_BitloadParam.s_TarSnrMgn - gs_BITSWAP_TRIGGER_DELTA),
                               (gs_BITWAP_MIN_SNR_MARGIN));

   gs_BsDecisionThreshold = gs_MinTargetSnrMargin;


   // Convert average Rx margin from x10 dB format to 8.8 dB format
   // Note: The downstream SNR margin ranges from -64 dB to +63 dB with 0.1 dB steps.
   l_SnrMgn = (((int32)gt_LineStatusDS.s_SnrMargin) << 8);
   s_Avg_Margin = (int16)(l_SnrMgn / 10); // s_Avg_Margin is in 8.8 dB format


   // If average Rx margin falls below 2dB margin (gs_BITWAP_MIN_SNR_MARGIN) then by bit-swapping maintain
   // the original min BS target margin (i.e. gs_MinTargetSnrMargin = 2 dB or more) does not make sense.
   // In that case better solution will be if the minimum BS target margin follows the average Rx margin.
   if(s_Avg_Margin <= gs_BITWAP_MIN_SNR_MARGIN) // Average margin <= 2dB
   {
      gs_MinTargetSnrMargin = s_Avg_Margin;

      // SNR calculation is not optimum due to not having high averaging time.
      // When average Rx SNR falls below 2 dB, Bit swap decision logic will be triggered
      // if the Min margin of any tone (ie gs_MinMarginBeforeSwap ) falls 0.5 dB below the average margin.
      gs_BsDecisionThreshold = gs_MinTargetSnrMargin-gs_BsDecisionExtraThreshold; // s_Avg_Margin - 0.5 dB
   }
   //*********************************************************************************************************//
   //BS-S-4 (End)                                                                                             //
   //*********************************************************************************************************//


   //***************************************************************************************************************************//
   //BS-S-5.   Decision algorithm which decides core BS algorithm or Margin equalization(if enabled) or none will be run(Start)//
   //***************************************************************************************************************************//

   //Check if a bitswap needs to be carried out by determining the minimum margin over all tones
   //Set this variable to -1 to indicate search for all the band
   s_band = -1;
   FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SnrBuf,
                             guca_RxSupportedToneSet, gs_RxMinConstSize, gs_RxMaxConstSize,
                             &s_ch, &s_band, &gs_MinMarginBeforeSwap);


   //XDSLRTFW-1784 (Showtime Margin Equalization - START)
   uc_bitswap_decision = 0;

   //Explore bitswap in case minimum margin is below the desired margin
   //where gs_REDUCE_LP_DELTA_BITSWAP controls how many bits can be bits can be swapped each time
   if (gs_MinMarginBeforeSwap < gs_BsDecisionThreshold)
   {
      //Do the bit swap whose margin less than threshold
      uc_bitswap_decision = 1;

      if (TESTArray[TEST_Control4] & TEST_Control4_Showtime_NoiseMgn_Equ_Bit1_Mask)
      {
         //Given below logic is required to avoid continues endless bit swap.
         guc_GainSwapCntr++;

         if (guc_GainSwapCntr > 8)
         {
            guc_GainSwapCntr = 0;
            if (gs_MinMarginBeforeSwap > 0)
            {
               // If Minimum margin before bit swap is greater than 0dB then force Margin Equalization algorithm (BS-S-13)
               // Abu: looks like global variable needed for margin equalization are not initialize in proper way! need to be checked
               uc_bitswap_decision = 3;
            }
         }
      }
   }
   else // preparation for Margin Equalization algorithm
   {
      if (TESTArray[TEST_Control4] & TEST_Control4_Showtime_NoiseMgn_Equ_Bit1_Mask)
      {
         //Do the bit swap on bit loadable high margin tones to reduce the margin.
         //Calling of this logic is required to reduce high margin, Call this logic
         //if normal bit swap logic is not called.
         //This is required if some tone got good SNR in showtime to do the bit swap.
         s_band_hi = -1;
         // Find bit loadable largest margin tone.
         ft_Result = FindToneWithExtremeMargin(LARGEST_MARGIN_AFTER_ADDING_1_BIT, puca_RxBat,
                                               psa_RxFineGains, psa_SnrBuf,
                                               guca_RxSupportedToneSet, 0,
                                               (int16)(gs_RxMaxConstSize-1),
                                               &s_ch_hi, &s_band_hi, &s_Margin_hi);

         //Abu: ft_Result = Fail code is missing!!

         //look for 3dB threshold
         if  (s_Margin_hi - s_Avg_Margin > 0x0300)
         {
            uc_bitswap_decision = 2; // Perform normal bit swap algorithm
            gs_MinTargetSnrMargin = s_Avg_Margin - 0x0300;
         }
         else
         {

            guc_slow_gainswap_cnt++;

            if (gs_MinMarginBeforeSwap > gs_Margin_Hi_Prev)
               gs_Margin_Hi_Prev = gs_MinMarginBeforeSwap;

            if (gs_MinMarginBeforeSwap < gs_Margin_Lo_Prev)
               gs_Margin_Lo_Prev = gs_MinMarginBeforeSwap;

            if (guc_slow_gainswap_cnt > 16)
            {

               guc_slow_gainswap_cnt = 0;
               s_temp = abs(gs_Margin_Hi_Prev - gs_Margin_Lo_Prev);

               if (s_temp < 0x0300)
                  uc_bitswap_decision = 3;

               //if (gs_dbg_idx < 4089)
               //{
               //   gsa_StVectoringBuffer[gs_dbg_idx++] = (int16)0xAAAA;  // Buffer for capturing 2048 Showtime constellation points
               //   gsa_StVectoringBuffer[gs_dbg_idx++] = gs_Margin_Hi_Prev;
               //   gsa_StVectoringBuffer[gs_dbg_idx++] = gs_Margin_Lo_Prev;
               //}
               gs_Margin_Hi_Prev = (int16) 0x8000;      // Largest negative number.
               gs_Margin_Lo_Prev = (int16) 0x7fff;      // Largest positive number.

            }
         } //if-else
      } //(TESTArray[TEST_Control4] & TEST_Control4_Showtime_NoiseMgn_Equ_Bit1_Mask)
   }//if-else
   //XDSLRTFW-1784 (Showtime Margin Equalization - END)
   // XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)
   //*********************************************************************************************************//
   //BS-S-5 (End)                                                                                             //
   //*********************************************************************************************************//


   //Explore bitswap in case minimum margin is below the desired margin
   //where gs_REDUCE_LP_DELTA_BITSWAP controls how many bits can be bits can be swapped each time
   //if((gs_MinMarginBeforeSwap < gs_MinTargetSnrMargin) && (gs_REDUCE_LP_DELTA_BITSWAP > 0))
   if ((uc_bitswap_decision > 0) && (uc_bitswap_decision < 3)) // BS-S-5. Decision algorithm
   {
      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
      gula_BSFailStat[E_CODE_BITSWAP_INFO_ALGO_STARTED]++; // number of time BS core algorithm is called

      //Core BS algorithm

#ifdef BS_INSIDE_DEBUG
      //Debug code start
      for (s_i = 0; s_i < 1200; s_i++)
      {
         gsa_BSDebugBuf[s_i] = 0;
      }


      //gsa_BSDebugBuf[0] = used buffer size
      //gsa_BSDebugBuf[1] = (int16)gft_RxBitSwapFlag;
      //gsa_BSDebugBuf[2] = (int16)gft_RmsFgOverShoot;
      //gsa_BSDebugBuf[3] = s_AvgRxMargin;
      //gsa_BSDebugBuf[4] = (uint16) gula_BSFailStat[0];
      //gsa_BSDebugBuf[5] = gs_MinTargetSnrMargin;

      gsa_BSDebugBuf[3] = s_Avg_Margin;
      gsa_BSDebugBuf[5] = gs_MinTargetSnrMargin;
      gsa_BSDebugBuf[6] = gs_Debug5; // no of time code comes to Margin Equalization code
      gs_DSDebugBufCnt = 7; // gsa_BSDebugBuf[0] contains the size of the used buffer , gsa_BSDebugBuf[1] & gsa_BSDebugBuf[2]
      gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 31402; // 0x7AAA = 0x31402
      gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 1;
      gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 1;
      // debug code end
#endif
      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)

      //*********************************************************************************************************//
      //BS-S-6. Remove bit from tones which has lower margin than minimum margin defined in BS-S-4. Remove extra //
      //FG from that tone with maintaining minimum margin criteria and update RMS FG value.                      //
      //*********************************************************************************************************//
      //Limit the maximum number of bits to be reduced by gs_REDUCE_LP_DELTA_BITSWAP
      //Note: The OLR message limits the maximum number of tones, where bit or gain
      //can be changed, to 128.
      //Thus, we should choose gs_REDUCE_LP_DELTA_BITSWAP such that
      //resulting number of tones changed will not exceed this number.
      //At the worse case, each of N tones is cut by 1 bit and each of N other tones is added by 1 bit,
      //Thus the total number of affected tones are 2N. When the TCM is on, it is possible
      //to change 4 more tones (see ForceEvenNum1BitTonesInBAT()) to meet the TCM overhead constraint
      //Thus the total number of tones changed is 2N+4.
      //We need to have: 2N+4 <= 128. or N <= 62
      l_delta_Lp = -gs_REDUCE_LP_DELTA_BITSWAP; // -50

      //Reduce Lp in BAT for those tones which have lower margin than gs_MinTargetSnrMargin
      //the maximum number of bits can be reduced is limited by s_delta_Lp, and

      // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (Start)
      // Initialize us_BiGiModificationControlFlags bit fields for ChangeLpKeepMaxMargin() function
      // #define FT_STOP_AT_MARGIN_THRESHOLD                (0x0001)    // bit 0
      // #define FT_FORCE_EVEN_1_BIT                        (0x0002)    // bit 1
      // #define FT_ADD_FINE_GAIN_PLUS_DELTA_LP             (0x0004)    // bit 2
      // #define FT_FAST_UPSHIFT_SRA_MODE                   (0x0008)    // bit 3

      us_BiGiModificationControlFlags = (FT_STOP_AT_MARGIN_THRESHOLD );

      ft_Result = ChangeLpKeepMaxMargin(puca_RxBat, psa_RxFineGains, l_delta_Lp, guca_RxSupportedToneSet, guca_RxBitswapToneSet, &l_ActualDeltaSumLp, us_BiGiModificationControlFlags);
      // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (End)

      if((l_ActualDeltaSumLp == 0) || (ft_Result == FAIL))
      {
         gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_CUT_BITS;
         gula_BSFailStat[E_CODE_BITSWAP_DECISION_CUT_BITS]++; //XDSLRTFW-3807
         goto _return;
      }

      //*********************************************************************************************************//
      //BS-S-6 (End)                                                                                             //
      //*********************************************************************************************************//


      //*********************************************************************************************************//
      //BS-S-7. Add bits only on the tones which have sufficient margin to accommodate extra bits.               //
      //        This process does not change any FG(Start)                                                       //
      //*********************************************************************************************************//
      //l_ActualDeltaSumLp is the actual number of bits being removed from BAT
      //we need to add this amount bits to some other tones to keep the Lp unchanged
      l_delta_Lp = -l_ActualDeltaSumLp; // +Ve

      // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (Start)
      // Initialize us_BiGiModificationControlFlags bit fields for ChangeLpKeepMaxMargin() function
      // TRUE FALSE FALSE FALSE configuration
      // #define FT_STOP_AT_MARGIN_THRESHOLD                (0x0001)    // bit 0
      // #define FT_FORCE_EVEN_1_BIT                    (0x0002)    // bit 1
      // #define FT_ADD_FINE_GAIN_PLUS_DELTA_LP             (0x0004)    // bit 2
      // #define FT_FAST_UPSHIFT_SRA_MODE                   (0x0008)    // bit 3

      us_BiGiModificationControlFlags = (FT_STOP_AT_MARGIN_THRESHOLD | FT_FORCE_EVEN_1_BIT);

      ft_Result = ChangeLpKeepMaxMargin(puca_RxBat, psa_RxFineGains, l_delta_Lp, guca_RxSupportedToneSet, guca_RxBitswapToneSet, &l_ActualDeltaSumLp, us_BiGiModificationControlFlags);

      // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (End)

      //*********************************************************************************************************//
      //BS-S-7 (End)                                                                                             //
      //*********************************************************************************************************//

      //*********************************************************************************************************//
      //BS-S-8. If BS-S-7 can not allocate all the bits removed from BS-S-5 then allocate final bits by modifying//
      //        FG on possible tones. This process always checks for RMS FG  value criteria                      //
      //BS-S-9. If reallocating removed tones from BS-S-5 fails due to RMS FG criteria then run again BS-S-8 for //
      //        rest of bits by disabling RMS FG check criteria (i.e. RMS FG <= 0 dB) (Start)                    //
      //*********************************************************************************************************//

      //If the number of bits added back is less than s_delta_Lp,
      //check to see if we can add more bits by adding some fine gains
      if (ft_Result == FAIL)
      {
         if(gft_FineGainOn)
         {
            l_delta_Lp -= l_ActualDeltaSumLp;

            for (i = RX_MAX_BITS_PER_TONE; i >= 0; i--)
            {
               gsa_SNRRequired[i] = gsa_ConstellationSNR[i] + gs_MinTargetSnrMargin;
            }
            // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (Start)
            // Initialize us_BiGiModificationControlFlags bit fields for ChangeLpKeepMaxMargin() function
            // TRUE FALSE TRUE FALSE configuration
            // #define FT_STOP_AT_MARGIN_THRESHOLD                (0x0001)    // bit 0
            // #define FT_FORCE_EVEN_1_BIT                        (0x0002)    // bit 1
            // #define FT_ADD_FINE_GAIN_PLUS_DELTA_LP             (0x0004)    // bit 2
            // #define FT_FAST_UPSHIFT_SRA_MODE                   (0x0008)    // bit 3

            us_BiGiModificationControlFlags = (FT_STOP_AT_MARGIN_THRESHOLD | FT_FORCE_EVEN_1_BIT | FT_ADD_FINE_GAIN_PLUS_DELTA_LP);

            ft_Result = ChangeLpKeepMaxMargin(puca_RxBat, psa_RxFineGains, l_delta_Lp, guca_RxSupportedToneSet, guca_RxBitswapToneSet, &l_ActualDeltaSumLp, us_BiGiModificationControlFlags);
            // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (End)

            if(ft_Result == FAIL)
            {
               //Set error code
               gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_ADD_BITS2;
               gula_BSFailStat[E_CODE_BITSWAP_DECISION_ADD_BITS2]++; //XDSLRTFW-3807
               goto _return;
            }
         }
         else
         {
            //Set error code
            gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_ADD_BITS1;
            gula_BSFailStat[E_CODE_BITSWAP_DECISION_ADD_BITS1]++; //XDSLRTFW-3807
            goto _return;
         }
      }


      //*********************************************************************************************************//
      //BS-S-8 & BS-S-9(End)                                                                                     //
      //*********************************************************************************************************//

      if(ft_Result != FAIL)
      {

         //Addition of s_Lp_delta was ok,
         //Increase the fine gains of affected tones with the constraint of
         //not violating the fine gain restriction
         if(gft_FineGainOn)
         {
            //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)

            //*********************************************************************************************************//
            // BS-S-10.  BS-S-10.  If the bit allocation is complete ( i.e. removing and adding bits) with BS-S-9 then //
            // remove FG from each "to be swapped" tones from respective DS band with 0.1 dB steps. This process       //
            // reduces the RMS FG per band to maintain the RMS FG per band criteria(i.e. note iv)                      //
            //*********************************************************************************************************//
            if (gft_RmsFgOverShoot == 1)
            {
               gs_MaxFgReductionAfterRmsFgOverShoot = ReduceFineGains_InCaseOfFgOverShoot(puca_RxBat, psa_RxFineGains, psa_SnrBuf, guca_RxBitswapToneSet);
            }
            //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)

            //*********************************************************************************************************//
            //BS-S-10 (End)                                                                                            //
            //*********************************************************************************************************//
            //*********************************************************************************************************//
            //BS-S-11. Add extra gain to the modified tones so that maximum margin is achieved. This process always    //
            //         check for RMS FG criteria to add more gain(Start)                                               //
            //*********************************************************************************************************//
            DistributeFineGains(puca_RxBat, psa_RxFineGains, psa_SnrBuf, guca_RxBitswapToneSet);
            //*********************************************************************************************************//
            //BS-S-11 (End)                                                                                            //
            //*********************************************************************************************************//

         }

         //*********************************************************************************************************//
         //BS-S-12. Finally if the minimum margin after intended bit swapping is more than the minimum margin before//
         //         bit swapping then convert the inactive FG table from 8.8dB format to 3.13 linear format and send//
         //         BS request FG and BAT to Active buffer and trigger BS OLR message to perform DS BS (Start)      //
         //*********************************************************************************************************//
         //Determine margin now
         //Set this variable to -1 to indicate search for all the band
         s_band = -1;

         FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SnrBuf, guca_RxSupportedToneSet, gs_RxMinConstSize, gs_RxMaxConstSize, &s_ch, &s_band, &s_MinMarginAfterSwap);

         if (s_MinMarginAfterSwap < (gs_MinMarginBeforeSwap))  //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code
         {
            //Set error code
            gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_MARGIN;
            gula_BSFailStat[E_CODE_BITSWAP_DECISION_MARGIN]++; //XDSLRTFW-3807
            goto _return;
         }
         else
         {
            int32 l_FinalLp;
            //Verify the final BAT, where s_x = s_TCM_Ovhd, l_ActualDeltaSumLp = l_Sum_bi
            GetBatStats(puca_RxBat, &i, &s_Tcm_Ovhd, &l_ActualDeltaSumLp);

            l_ActualDeltaSumLp -= s_Tcm_Ovhd;
            //XDSLRTFW-1427 (START)
            l_FinalLp = (int32)(gt_rx_config_v2.ul_Lp[LP0]+gt_rx_config_v2.ul_Lp[LP1]);
            if (gt_ReTXParams.uc_UsReTxStatus == US_RETX_IN_USE)
            {
               l_FinalLp += US_RRC_BITS;
            }
            if(l_ActualDeltaSumLp != (int32)(l_FinalLp))  //XDSLRTFW-1427 (END)
            {
               gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_LP;
               gula_BSFailStat[E_CODE_BITSWAP_DECISION_LP]++; //XDSLRTFW-3807
               goto _return;
            }
            //Check if the number of 1 bit-tones is even or not
            else if((gft_RxTcmFlag) && (i & 0x1))
            {
               gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_NUM1BIT_TONES;
               gula_BSFailStat[E_CODE_BITSWAP_NUM1BIT_TONES]++; //XDSLRTFW-3807
               goto _return;
            }

            //For those tones which bi or gi have been changed, convert their
            //gains to linear gain in Q3.13 format for being used in OLR message
            //Also, count the number of tones being changed
            gs_NumOfBitSwapTones = ConvertGi_dBtoLinear();


            if(gs_NumOfBitSwapTones > MAX_NUM_BITSWAP_TONES)
            {
               gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_TONES;
               gula_BSFailStat[E_CODE_BITSWAP_DECISION_TONES]++; //XDSLRTFW-3807
               goto _return;
            }
         } //if(s_MinMarginAfterSwap < s_MinMarginBeforeSwap) else
         //*********************************************************************************************************//
         //BS-S-12 (End)                                                                                            //
         //*********************************************************************************************************//
      } //if(ft_Result != FAIL)
   } //if(s_MinMarginBeforeSwap < gs_MinTargetSnrMargin)
   //XDSLRTFW-1784 (Showtime Margin Equalization - START)
   //*********************************************************************************************************//
   //BS-S-13.  Margin Equalization Algorithm                                                                  //
   //*********************************************************************************************************//
   if ((TESTArray[TEST_Control4] & TEST_Control4_Showtime_NoiseMgn_Equ_Bit1_Mask) &&
         (uc_bitswap_decision == 3))
   {

      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
      // Debug code Start
      for (s_i = 0; s_i < 1200; s_i++)
      {
         gsa_BSDebugBuf[s_i] = 0;
      }

      gs_Debug5++; // no of time code comes to Margin Equalization code
      gsa_BSDebugBuf[3] = s_Avg_Margin;
      gsa_BSDebugBuf[5] = gs_MinTargetSnrMargin;
      gsa_BSDebugBuf[6] = gs_Debug5; // no of time code comes to Margin Equalization code
      gs_DSDebugBufCnt = 7; // gsa_BSDebugBuf[0] contains the size of the used buffer , gsa_BSDebugBuf[1] & gsa_BSDebugBuf[2]
      gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 31402; // 0x7AAA = 0x31402
      gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 11;
      gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 11;
      // debug code end
#endif
      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code (End)
      if (ShowtimeNoiseMarginEqualization(puca_RxBat, psa_RxFineGains))
      {
         //For those tones which bi or gi have been changed, convert their
         //gains to linear gain in Q3.13 format for being used in OLR message
         //Also, count the number of tones being changed
         gs_NumOfBitSwapTones = ConvertGi_dBtoLinear();

         if(gs_NumOfBitSwapTones > MAX_NUM_BITSWAP_TONES)
         {
            gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_DECISION_TONES_GAIN_SWAP;
            gula_BSFailStat[E_CODE_BITSWAP_DECISION_TONES_GAIN_SWAP]++; //XDSLRTFW-3807
            goto _return;
         }

         //Don't do gain swap if no of tones is < threshold
         //This is restricted to avoid frequent gain swap as a part
         //bit swap protocol
         if(gs_NumOfBitSwapTones < (MAX_NUM_BITSWAP_TONES >> 2))
         {
            gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_NUM_TONES_GAIN_SWAP;
            gula_BSFailStat[E_CODE_BITSWAP_NUM_TONES_GAIN_SWAP]++; //XDSLRTFW-3807
            goto _return;
         }
      } //if (ShowtimeNoiseMarginEqualization(puca_RxBat, psa_RxFineGains))
   } //if (TESTArray[TEST_Control4] & TEST_Control4_Showtime_NoiseMgn_Equ_Bit1_Mask)
   //*********************************************************************************************************//
   //BS-S-13 (End)                                                                                            //
   //*********************************************************************************************************//

   if ((gs_NumOfBitSwapTones) && (uc_bitswap_decision == 1))
   {
      guc_dbg_bitswap_decision_1++;
   }

   if ((gs_NumOfBitSwapTones) && (uc_bitswap_decision == 2))
   {
      guc_dbg_bitswap_decision_2++;
   }

   if ((gs_NumOfBitSwapTones) && (uc_bitswap_decision == 3))
   {
      guc_dbg_bitswap_decision_3++;
   }


   //XDSLRTFW-1784 (Showtime Margin Equalization - END)
   // Check if any tones changed
   if (gs_NumOfBitSwapTones > 0)
   {

#ifndef STANDALONE_BITSWAP_TEST

      // Perform tone-reordering
      gft_ToneOrderOK = ReorderTRT_Rx();

      if(gft_ToneOrderOK == FAIL)
      {
         gl_RxBitSwapDecisionErrorCode = E_CODE_BITSWAP_TONEREORDER;
         gula_BSFailStat[E_CODE_BITSWAP_TONEREORDER]++; //XDSLRTFW-3807
         goto _return;
      }
#endif
      if(gl_RxBitSwapDecisionErrorCode == E_CODE_BITSWAP_DECISION_NO_ERROR)
      {
         gft_RxBitSwapFlag = TRUE;
         gula_BSFailStat[E_CODE_BITSWAP_DECISION_NO_ERROR]++; //XDSLRTFW-3807
      }

      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
      //Debug code start
      if (gft_RxBitSwapFlag == TRUE )
      {
         gsa_BSDebugBuf[1] = (int16)gft_RxBitSwapFlag;
         gsa_BSDebugBuf[2] = (int16)gft_RmsFgOverShoot;

         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 31402; // 0x7AAA = 0x31402
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 2;
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 2;
         // this function logged Tone, FG, BAT, Margin of the tones after bits are removed from the tone
         Log_BS_Info(puca_RxBat, psa_RxFineGains, psa_SnrBuf, guca_RxBitswapToneSet);

         gsa_BSDebugBuf[1] = (int16)gft_RxBitSwapFlag;
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 31402; // 0x7AAA = 0x31402
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 3;
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 3;

         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = gs_MinMarginBeforeSwap;
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_MinMarginAfterSwap;
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = gs_MaxFgReductionAfterRmsFgOverShoot;


         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 31402; // 0x7AAA = 0x31402
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = 4;
         gsa_BSDebugBuf[gs_DSDebugBufCnt++] = (int16)gula_BSFailStat[0];
         gsa_BSDebugBuf[4] = (uint16)gula_BSFailStat[0]; // data verification
         gsa_BSDebugBuf[0] = gs_DSDebugBufCnt;

         // to align PerToneMargin with BS info. Please ignore this two tones in PerToneMargin analysis
         gsa_PerToneMargin[0] = (int16)gula_BSFailStat[0];
         gsa_PerToneMargin[4095] = (int16)gula_BSFailStat[0];
      }
      // Debug code end
#endif
      //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)
   }

_return:

   //XDSLRTFW-1784 (Showtime Margin Equalization - START)
   if (gl_RxBitSwapDecisionErrorCode != E_CODE_BITSWAP_DECISION_NO_ERROR)
   {
      DSH_SendEvent(DSH_EVT_BITSWAP_ALGO_FAIL, sizeof(int32), (void *)&gl_RxBitSwapDecisionErrorCode);
   }
   //XDSLRTFW-1784 (Showtime Margin Equalization - END)

   // Restore gsa_SNRRequired[] if it has been changed previously
   // Add target margin to constellation SNR
   for (i = RX_MAX_BITS_PER_TONE; i >= 0; i--)
   {
      gsa_SNRRequired[i] = gsa_ConstellationSNR[i] + gt_BitloadParam.s_TarSnrMgn;
   }

   //Restore the original flag
   gft_MAXNOMATP_Flag = ft_MAXNOMATP_Flag;

   //Indicate the background process is done
   gs_RxBkgdProcessFlag = TRAINING_DONE;

} //void BgRxBitSwapDecision_VDSL2(void)

/*^^^
*-----------------------------------------------------------------------------
*   Prototype : int16 ConvertGi_dBtoLinear(void)
*
*
*   Description : Convert the gains of those tones being changed, to
*                 linear fine gain in Q3.13
*                 For VR9 HW, we also save the changed BAT entries to a temporary buffer
*                 for later use.
*
*
*   Input Parameters:
*
*
*   Output Parameters:
*
*
*   Global Variables:
*      ghpsa_RxFineGains_Inactive: (I) pointer to inactive fine gain table
*      ghpuca_RxBat_Inactive:      (I) pointer to inactive bit allocation table
*      gpsa_BitSwapGain:           (O) pointer to the changed fine gain table
*       gpuca_BitSwapBat:           (O) pointer to the changed bit allocation table
*------------------------------------------------------------------------------------
*/

int16 ConvertGi_dBtoLinear(void)
{
   int16 *psa_RxFineGains;
   uint8 *puca_RxBat;
   int16 s_NumOfBitSwapTones = 0;
   int16 i, s_band, s_RxFineGain;

   psa_RxFineGains = ghpsa_RxFineGains_Inactive;
   puca_RxBat = ghpuca_RxBat_Inactive;

   for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
   {
      for (i=gsa_RxBandLeftChannel[s_band]; i <= gsa_RxBandRightChannel[s_band]; i++)
      {
         //If this tone is a bit-swap tone, compute the gain in linear format which is used for OLR message
         //(then the RX gain stored in HW, we need to convert it to 1/gain_linear and this is done
         // in FillBinsInfo() of ovhd_tx.c)
         if(IS_TONEFLAGSET(guca_RxBitswapToneSet, i))
         {
            // Convert gain from dB to linear in Q3.13 format
            // Since only 12-bit fine gain in Q3.9 is sent in the OLR message,
            // so we round and clear the 4 LSBs
            s_RxFineGain = DecimalGain(psa_RxFineGains[i]);
            s_RxFineGain = (s_RxFineGain + (1 << 3)) & 0xFFF0;

            //In VR9 HW, we cannot copy bat and gain table separately
            //so we have to save the changed bits value during the active to inactive table copy
            gpuca_BitSwapBat[s_NumOfBitSwapTones] = puca_RxBat[i];

            //Temporarily store the modified linear fine gain in this buffer
            gpsa_BitSwapGain[s_NumOfBitSwapTones++] = s_RxFineGain;
            // XDSLRTFW-4132 (Start_End)
            // Avoid writing more values than the array behind the pointer can store.
            // This can happen during bitswap algorithm in case of trellis and a
            // great number of 1-bit tones, because the headroom of 12bits (tones) is not enough.
            // Used BS formula : 2N+4 <= 128. or N <= 62
            //                   l_MaxLpChangeBS = (-gs_REDUCE_LP_DELTA_BITSWAP); // -50
            if (s_NumOfBitSwapTones >= MAX_NUM_BITSWAP_TONES)
            {
               // This return value triggers a safety check at the calling function
               return(MAX_NUM_BITSWAP_TONES+1);
            }

         }

      } //for (i=gsa_RxBandLeftChannel[s_band]
   } //for (s_band=0; s_band <

   return(s_NumOfBitSwapTones);
}

/*^^^
*-----------------------------------------------------------------------------
*   Prototype : DistributeFineGains(uint8 *puca_RxBat, int16 *psa_RxFineGains, int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones_ToConsider)
*
*
*   Description :
*
*   Add fine gains to the bit-swap affected tones to increase margin as much as possible, without violating the RMS
*   fine gain constraint.
*
*   Input Parameters:
*      puca_RxBat      : BAT
*      psa_RxFineGains : Fine Gain Array (dB)
*      psa_SNRBuffer   : SNR
*      p_ActiveTones_ToConsider: Tones to which fine gains can be added
*
*   Output Parameters:
*      pl_SumFGainLinSq   : Linear Sum Square Fine Gains
*      psa_RxFineGains   : Modified fine gain array
*
*   Global Variables:
*      gs_RxMinConstSize: (I) the minimum constellation size allowed
*      gs_RxMaxConstSize: (I) the maximum constellation size allowed
*      gs_NumOfRxBands: (I) the number of RX bands
*      gsa_RxBandLeftChannel: (I) the left channels of the bands
*      gsa_RxBandRightChannel: (I) the right channels of the bands
*      gs_max_fine_gain: (I) the maximum fine gain
*      gsa_ConstellationSNR: (I) the required SNR for bitloading of different constellation size
*      gla_AvgFGainLinSqInit[]: (I) the average of gain squared of each band computed during the initialization
*      gla_SumFGainLinSq[]: (O) the sum of gain squared of each band computed in showtime
*-----------------------------------------------------------------------------*/

void DistributeFineGains(uint8 *puca_RxBat, int16 *psa_RxFineGains, int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones_ToConsider)
{

   int16 s_ch, s_band, i_BandIdx;
   int16 s_finegainreqd, s_BitAlloc, s_FineGain;
   int16 s_threshold, s_Margin, s_MinMargin, s_SNR;
   int32 l_SumFGainLinSq, l_MaxSumFGainLinSq, l_SumFGainLinSq_ToConsider;
   FlagT ft_AnyToneLefttotakeGain, ft_flag_rms_increase;


   ft_AnyToneLefttotakeGain = 1;

   //Loop through all the bands
   for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
   {
      MULS32xU16(l_MaxSumFGainLinSq, gla_AvgFGainLinSqInit[s_band], gusa_ncloaded[s_band]);
      l_SumFGainLinSq = gla_SumFGainLinSq[s_band];

      // !! Debug code to test maximum bits per tone per band, i.e. max constellation per band!!
      if (gsa_RxMaxConstSize[s_band] > 0)
      {
         gs_RxMaxConstSize = gsa_RxMaxConstSize[s_band];
      }

      //Set this variable to the current band so the function only search for the current band
      i_BandIdx = s_band;
      FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SNRBuffer,
                                p_ActiveTones_ToConsider, gs_RxMinConstSize, gs_RxMaxConstSize, &s_ch, &i_BandIdx, &s_MinMargin);

      if (s_MinMargin < 0)
      {
         s_MinMargin = 0;
      }

      //Set it to 0.5 dB
      s_threshold = 128;

      //If there is still chance to add more fine gain
      while((l_SumFGainLinSq < l_MaxSumFGainLinSq) && (ft_AnyToneLefttotakeGain))
      {
         ft_AnyToneLefttotakeGain = 0;
         ft_flag_rms_increase = 0;

         //Loop through all the channels in this band
         for (s_ch = gsa_RxBandLeftChannel[s_band]; s_ch <= gsa_RxBandRightChannel[s_band]; s_ch++)
         {
            //Get the bit allocaton and gain of this tone
            s_BitAlloc = puca_RxBat[s_ch];
            s_FineGain = psa_RxFineGains[s_ch];

            //Only change the tones which have been affected in bit-swap, which can be added more gain
            //and which have been loaded
            if ((IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch)) && (s_FineGain < gs_max_fine_gain) && (s_BitAlloc != 0))
            {
               ft_AnyToneLefttotakeGain = 1;

               //Compute the SNR margin adjusted with fine gain and coding gain
               s_SNR = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
               s_SNR += psa_RxFineGains[s_ch];

               s_Margin = s_SNR - gsa_ConstellationSNR[s_BitAlloc];

               //skip these tones if the margin is greater than the threshold
               if (s_Margin  >= (s_MinMargin + s_threshold))
               {
                  continue;
               }

               //Compute the required fine gain to make the margin pass the threshold
               s_finegainreqd =  s_MinMargin + s_threshold - s_Margin ;

               if (s_finegainreqd > s_threshold)
               {
                  s_finegainreqd = s_threshold;
               }

               s_finegainreqd += s_FineGain;

               //Limit the fine gain by its upper limit
               if(s_finegainreqd > gs_max_fine_gain)
               {
                  s_finegainreqd = gs_max_fine_gain;
               }

               //Update fine gain linear sum square
               l_SumFGainLinSq_ToConsider = l_SumFGainLinSq;
               UpdateFGSumSqAndNOMATP(s_ch, s_FineGain, s_finegainreqd, &l_SumFGainLinSq_ToConsider, 0);

               //Check if adding this fine gain violates the average gain square constraint
               if(l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)
               {
                  //No violation, update the fine gain and the sum of gain squares
                  psa_RxFineGains[s_ch] = s_finegainreqd;
                  l_SumFGainLinSq = l_SumFGainLinSq_ToConsider;
               }
               else
               {
                  //Indicate a sucessful try
                  ft_flag_rms_increase = 1;
               }

            } //if((IS_TONEFLAGSET(p_ActiveTones_ToConsider, i) == 1) &&

         } //for (i = gsa_RxBandLeftChannel[s_band];

         if ((ft_flag_rms_increase == 1) || (ft_AnyToneLefttotakeGain == 0))
         {
            break;
         }

         s_threshold += 128;
      } //while((l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)

      gla_SumFGainLinSq[s_band] = l_SumFGainLinSq; //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code
   } //for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
}

//XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
/*
*-----------------------------------------------------------------------------
*   Prototype : int16 ReduceFineGains_InCaseOfFgOverShoot(uint8 *puca_RxBat, int16 *psa_RxFineGains,
*                                                         int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones_ToConsider)
*
*   Description :
*
*   Reduce FGs of bit swap-able tones in a way that final RMS FG/ band obeys the RMS FG criteria ( i.e. RMS FG <= 0 dB)
*
*   Use case :
*
*   Using ChooseFineGains function with s_NoAvgGainCheck = 1 argument re-distributes FG without maintaining RMS FG criteria.
*   This can results in RMS FG/ band  violation (overshoot).  Whenever ChooseFineGains() with s_NoAvgGainCheck = 1 argument
*   is called, to maintain the RMS FG criteria ReduceFineGains_InCaseOfFgOverShoot() is called.
*
*   Input Parameters:
*      puca_RxBat      : BAT
*      psa_RxFineGains : Fine Gain Array (dB)
*      psa_SNRBuffer   : SNR
*      p_ActiveTones_ToConsider: Tones to which fine gains can be added
*
*   Return Parameters:
*       Maximum FG reduction in 8.8 dB format
*
*   Global Variables:
*      gs_NumOfRxBands: (I) the number of RX bands
*      gusa_ncloaded: (I) number of loaded tones in a band
*      gsa_RxBandLeftChannel: (I) the left channels of the bands
*      gsa_RxBandRightChannel: (I) the right channels of the bands
*      gla_AvgFGainLinSqInit[]: (I) the average of gain squared of each band computed during the initialization
*      gla_SumFGainLinSq[]: (O) the sum of gain squared of each band computed in Showtime.
*-----------------------------------------------------------------------------*/
int16 ReduceFineGains_InCaseOfFgOverShoot(uint8 *puca_RxBat, int16 *psa_RxFineGains,int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones_ToConsider)
{
   int16 s_ch, s_band;
   int16 s_finegainreqd, s_BitAlloc, s_FineGain;
   int16 s_threshold;
   int32 l_SumFGainLinSq, l_MaxSumFGainLinSq, l_SumFGainLinSq_ToConsider;
   FlagT ft_AnyToneLefttotakeGain, ft_flag_rms_below_threshold;
   int16 s_Iteration_counter, s_MaxIteration;

   s_MaxIteration = 0;

   //Loop through all the bands
   for (s_band = 0; s_band < gs_NumOfRxBands; s_band++)
   {
      MULS32xU16(l_MaxSumFGainLinSq, gla_AvgFGainLinSqInit[s_band], gusa_ncloaded[s_band]);
      l_SumFGainLinSq = gla_SumFGainLinSq[s_band];

      //Set it to ~0.1 dB (8.8db format)
      s_threshold = 26;
      s_Iteration_counter = 0;

      //If there is still chance to add more fine gain
      while (l_SumFGainLinSq > l_MaxSumFGainLinSq)
      {
         ft_AnyToneLefttotakeGain = 0;
         ft_flag_rms_below_threshold = 0;

         //find maximum number of iteration
         s_Iteration_counter++;

         //if (s_Iteration_counter > 5)
         //   break;

         if (s_Iteration_counter >s_MaxIteration)
            s_MaxIteration = s_Iteration_counter;

         //Loop through all the channels in this band
         for (s_ch = gsa_RxBandLeftChannel[s_band]; s_ch <= gsa_RxBandRightChannel[s_band]; s_ch++)
         {
            //Get the bit allocation and gain of this tone
            s_BitAlloc = puca_RxBat[s_ch];
            s_FineGain = psa_RxFineGains[s_ch];

            //Only change (i.e. reduce FG) the tones which have been affected in bit-swap
            if ((IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch)) && (s_FineGain > gs_min_fine_gain) && (s_BitAlloc != 0))
            {
               ft_AnyToneLefttotakeGain = 1;

               //Reduce FG by s_threshold
               s_finegainreqd = s_FineGain - s_threshold;

               //Limit the fine gain by its lower limit
               if (s_finegainreqd < gs_min_fine_gain)
                  s_finegainreqd = gs_min_fine_gain;


               //Update fine gain linear sum square
               l_SumFGainLinSq_ToConsider = l_SumFGainLinSq;
               UpdateFGSumSqAndNOMATP(s_ch, s_FineGain, s_finegainreqd, &l_SumFGainLinSq_ToConsider, 0);

               psa_RxFineGains[s_ch] = s_finegainreqd; // update FG with reduced value
               l_SumFGainLinSq = l_SumFGainLinSq_ToConsider; // update sum of gain squares with reduced value

               //Check if reducing this fine gain full fills the RMS FG criteria
               if (l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)
               {
                  //no violation!
                  ft_flag_rms_below_threshold = 1;
               }

            } //if((IS_TONEFLAGSET(p_ActiveTones_ToConsider, i) == 1) &&
         } //for (i = gsa_RxBandLeftChannel[s_band];

         if ((ft_flag_rms_below_threshold == 1) ||(ft_AnyToneLefttotakeGain == 0))
            break;

      } //while((l_SumFGainLinSq_ToConsider > l_MaxSumFGainLinSq)

      gla_SumFGainLinSq[s_band] = l_SumFGainLinSq;

   } //end for (s_band=0; s_band < gs_NumOfRxBands; s_band++)

   //gs_MaxFgReductionAfterRmsFgOverShoot = s_MaxIteration*s_threshold; // 8.8 dB format
   return(s_MaxIteration*s_threshold);
}
//XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)

//XDSLRTFW-464 : DS_MARGIN_LOWER_TO_BRCM_DISTRIBUTE_FINEGAIN (Start)
/****************************************************************************************
;Function : DistributeFineGainsForShowtime()
;Input :  puca_RxBat  ------------------------ Rx Bit Table
;        psa_RxFineGains -------------------- Fine Gain Table
;        psa_SNRBuffer ---------------------- SNR Table
;        p_ActiveTones_ToConsider ----------- Medley Used Set
;Output : Distributed Fine Gain Table around 0dB RMS GI
;Description: This function is a wrapper above the final fine gains to compensate 2.5dB
;             which got allocated due to MIN_FINE_GAIN (2.5) and distribute the fine gains
******************************************************************************************/

void DistributeFineGainsForShowtime(uint8 *puca_RxBat, int16 *psa_RxFineGains, int16 *psa_SNRBuffer, RxToneFlags p_ActiveTones_ToConsider)
{

   int16 s_ch, s_band, i_BandIdx, s_CurRxBand;
   int16 s_finegainreqd, s_BitAlloc, s_FineGain;
   int16 s_threshold, s_Margin, s_MinMargin, s_SNR;
   int32 l_SumFGainLinSq, l_MaxSumFGainLinSq, l_SumFGainLinSq_ToConsider, s_DeltaFGain;
   FlagT ft_AnyToneLefttotakeGain, ft_flag_rms_increase;
   int32 l_SumFGainLinSqTrain;
   int16 s_FGainLin;
   int16 s_FineGain_dB;
   int32 l_Gain;


   ft_AnyToneLefttotakeGain = 1;

   //Add the s_DeltaFGain of 2.5dB which we used for Init FineGain
   s_DeltaFGain = 640; // +2.5dB in 8.8 format
   for (s_CurRxBand = 0; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
   {
      gla_AvgFGainLinSqInitTrain[s_CurRxBand] = 0;
      gla_AvgFGainLinInitTrain[s_CurRxBand] = 0;
      gla_AvgFGaindBInitTrain[s_CurRxBand] = 0;
      //XDSLRTFW-1677 : DS Fine Gain Violation
      for (s_ch = gsa_RxBandLeftChannel[s_CurRxBand]; s_ch <= gsa_RxBandRightChannel[s_CurRxBand]; s_ch++)
      {
         //Only change the tones which have been affected in bit-swap, which can be added more gain
         //and which have been loaded
         if ((IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch)) && (puca_RxBat[s_ch] != 0))
         {
            s_FineGain_dB = psa_RxFineGains[s_ch];

            // preserve the 12 most significant bits as these are sent in R_params
            // s_FGgainLin is Q3.9 format
            s_FGainLin = (DecimalGain(s_FineGain_dB) + (int16)(1<<3)) >> 4;
            MULS16(l_SumFGainLinSqTrain, s_FGainLin, s_FGainLin);

            gla_AvgFGainLinSqInitTrain[s_CurRxBand] += l_SumFGainLinSqTrain;
         }
      }
      gla_AvgFGainLinSqInitTrain[s_CurRxBand] = gla_AvgFGainLinSqInitTrain[s_CurRxBand]/gusa_ncloaded[s_CurRxBand];

      gla_AvgFGainLinInitTrain[s_CurRxBand] = sqrt32(gla_AvgFGainLinSqInitTrain[s_CurRxBand]);
      // Convert to Q3.13 Format from 3.9
      l_Gain = NormAndDivide_32by16bit(((int32)1<<30), ((gla_AvgFGainLinInitTrain[s_CurRxBand]*8192)/512));

      //l_Gain = gla_AvgFGainLinInitTrain[s_CurRxBand];
      //l_Gain is the linear gain in Q3.13
      l_Gain = round(l_Gain, 4);

      //Convert the linear fine gain to dB (Q8.8)
      s_FineGain_dB = ConvertToDB(l_Gain);

      //Since the input to ConvertToDB Q3.13 format and
      //ConvertToDB computes 10log10(gain), but we need to compute 20*log10(gain)
      //so we need to adjust s_FineGain to get the desired dB (in Q8.8)
      s_FineGain_dB = (s_FineGain_dB<<1) - 0x4E44; // 0x4E44 = 20*log10(2^13) * 256

      gla_AvgFGaindBInitTrain[s_CurRxBand] = -(s_FineGain_dB);
      gla_AvgFGaindBInitTrain[s_CurRxBand] &= 0xFFFF;
      s_FineGain_dB = (gla_AvgFGaindBInitTrain[s_CurRxBand] + s_DeltaFGain);
      if(s_FineGain_dB <= 0)
      {
         if(s_FineGain_dB == 0)
         {
            gla_AvgFGainLinSqRefTrain[s_CurRxBand] = 262144;//512^2 - 0dB Linear Fine Gain
         }
         else
         {
            s_FGainLin = (DecimalGain(s_FineGain_dB) + (int16)(1<<3)) >> 4;
            gla_AvgFGainLinSqRefTrain[s_CurRxBand] = s_FGainLin * s_FGainLin;
         }

      }
      else
      {
         //gla_AvgFGainLinSqRefTrain[s_CurRxBand] = gla_AvgFGainLinSqInitTrain[s_CurRxBand];
         gla_AvgFGainLinSqRefTrain[s_CurRxBand] = 262144; //512^2 - 0dB Linear Fine Gain
      }

   }

   //Loop through all the bands
   for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
   {
      MULS32xU16(l_MaxSumFGainLinSq, gla_AvgFGainLinSqRefTrain[s_band], gusa_ncloaded[s_band]);
      MULS32xU16(l_SumFGainLinSq, gla_AvgFGainLinSqInitTrain[s_band], gusa_ncloaded[s_band]);
      //l_SumFGainLinSq = gla_AvgFGainLinSqInitTrain[s_band];

      // !! Debug code to test maximum bits per tone per band, i.e. max constellation per band!!
      if (gsa_RxMaxConstSize[s_band] > 0)
      {
         gs_RxMaxConstSize = gsa_RxMaxConstSize[s_band];
      }

      //Set this variable to the current band so the function only search for the current band
      i_BandIdx = s_band;
      FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, psa_SNRBuffer, p_ActiveTones_ToConsider, gs_RxMinConstSize, gs_RxMaxConstSize, &s_ch, &i_BandIdx, &s_MinMargin);

      if (s_MinMargin < 0)
      {
         s_MinMargin = 0;
      }

      //Set it to 0.5 dB
      s_threshold = 128;

      //If there is still chance to add more fine gain
      while((l_SumFGainLinSq < l_MaxSumFGainLinSq) && (ft_AnyToneLefttotakeGain))
      {
         ft_AnyToneLefttotakeGain = 0;
         ft_flag_rms_increase = 0;

         //Loop through all the channels in this band
         for (s_ch = gsa_RxBandLeftChannel[s_band]; s_ch <= gsa_RxBandRightChannel[s_band]; s_ch++)
         {
            //Get the bit allocaton and gain of this tone
            s_BitAlloc = puca_RxBat[s_ch];
            s_FineGain = psa_RxFineGains[s_ch];

            //Only change the tones which have been affected in bit-swap, which can be added more gain
            //and which have been loaded
            if ((IS_TONEFLAGSET(p_ActiveTones_ToConsider, s_ch)) && (s_FineGain < gs_max_fine_gain) && (s_BitAlloc != 0))
            {
               ft_AnyToneLefttotakeGain = 1;

               //Compute the SNR margin adjusted with fine gain and coding gain
               s_SNR = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
               s_SNR += psa_RxFineGains[s_ch];

               s_Margin = s_SNR - gsa_ConstellationSNR[s_BitAlloc];

               //skip these tones if the margin is greater than the threshold
               if (s_Margin  >= (s_MinMargin + s_threshold))
               {
                  continue;
               }

               //Compute the required fine gain to make the margin pass the threshold
               s_finegainreqd =  s_MinMargin + s_threshold - s_Margin ;

               if (s_finegainreqd > s_threshold)
               {
                  s_finegainreqd = s_threshold;
               }

               s_finegainreqd += s_FineGain;

               //Limit the fine gain by its upper limit
               if(s_finegainreqd > gs_max_fine_gain)
               {
                  s_finegainreqd = gs_max_fine_gain;
               }

               //Update fine gain linear sum square
               l_SumFGainLinSq_ToConsider = l_SumFGainLinSq;
               UpdateFGSumSqAndNOMATP(s_ch, s_FineGain, s_finegainreqd, &l_SumFGainLinSq_ToConsider, 0);

               //Check if adding this fine gain violates the average gain square constraint
               if(l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)
               {
                  //No violation, update the fine gain and the sum of gain squares
                  psa_RxFineGains[s_ch] = s_finegainreqd;
                  l_SumFGainLinSq = l_SumFGainLinSq_ToConsider;
               }
               else
               {
                  //Indicate a sucessful try
                  ft_flag_rms_increase = 1;
               }

            } //if((IS_TONEFLAGSET(p_ActiveTones_ToConsider, i) == 1) &&

         } //for (i = gsa_RxBandLeftChannel[s_band];

         if ((ft_flag_rms_increase == 1) || (ft_AnyToneLefttotakeGain == 0))
         {
            break;
         }

         s_threshold += 128;

      } //while((l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)

      gla_SumFGainLinSq[s_band] = l_SumFGainLinSq;

   } //for (s_band=0; s_band < gs_NumOfRxBands; s_band++)

}
//XDSLRTFW-464 : DS_MARGIN_LOWER_TO_BRCM_DISTRIBUTE_FINEGAIN (End)


/*****************************************************************************
;   Subroutine Name: RxBitSwapDecisionForTest_VDSL2( )
;
;   This subroutine computes the bit swap parameters for the bit-swap test.
;
;   Prototype:
;   void RxBitSwapDecisionForTest_VDSL2(void)
;
;   Input Arguments:
;
;   Output Arguments:
;      none
;
;   Return:
;      none
;
;   Global Variables:
;      gl_rx_sframe_count      -- (I) RX superframe count
;      gt_RxBitSwapInfo      -- (I) structure containing RX bit swap info
;
;      gsa_RxBat[]            -- (O) RX Bit Allocation Table
;      gsa_RxFineGains[]      -- (O) RX fine gain table
;      gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/

void RxBitSwapDecisionForTest_VDSL2(void)
{
   int16 s_from_tone, s_to_tone;
   int16 s_ActualNumBitsToSwap;
   int16 *psa_RxFineGains;

   psa_RxFineGains = ghpsa_RxFineGains_Inactive;

   gft_RxBitSwapFlag = FALSE;

   s_ActualNumBitsToSwap = 0;

   {
      /* manual bitswap */
      // From tone
      s_from_tone = TESTArray[TEST_BitSwapFromTone];
      // XDSLRTFW-3280 - Start / End - PLL improvement / pilot tone selection improvement
      if ((s_from_tone>= gs_RxNumTones) || (s_from_tone==gt_PilotConfig.ta_PilotTones[0].s_PilotToneIdx)|| (s_from_tone==gt_PilotConfig.ta_PilotTones[1].s_PilotToneIdx) || (s_from_tone==gt_PilotConfig.ta_PilotTones[2].s_PilotToneIdx) )//XDSLRTFW-2302
      {
         goto _end;
      }

      // To tone
      s_to_tone = TESTArray[TEST_BitSwapToTone];
      // XDSLRTFW-3280 - Start / End - PLL improvement / pilot tone selection improvement
      if ((s_to_tone >= gs_RxNumTones) || (s_to_tone==gt_PilotConfig.ta_PilotTones[0].s_PilotToneIdx)|| (s_to_tone==gt_PilotConfig.ta_PilotTones[1].s_PilotToneIdx) || (s_to_tone==gt_PilotConfig.ta_PilotTones[2].s_PilotToneIdx) )//XDSLRTFW-2302
      {
         goto _end;
      }

      //Simply doing bitswap without checking
      s_ActualNumBitsToSwap = 1;

      // Update the inactive bank as per the request
      *(ghpuca_RxBat_Inactive+s_from_tone) -= s_ActualNumBitsToSwap;
      *(ghpuca_RxBat_Inactive+s_to_tone) += s_ActualNumBitsToSwap;

      //Mark the tones that have changed
      SETTONEFLAG(guca_RxBitswapToneSet, s_from_tone);
      SETTONEFLAG(guca_RxBitswapToneSet, s_to_tone);

      gs_NumOfBitSwapTones = ConvertGi_dBtoLinear();

      gft_ToneOrderOK = ReorderTRT_Rx();

      if(gft_ToneOrderOK == FAIL)
      {
         goto _end;
      }

      // Reset Test Bitswap Flag
      TESTArray[TEST_ReconfigControl] &= ~TEST_ForceBitswap;
   }

   gft_RxBitSwapFlag = TRUE;

_end:

   //Indicate the background process is done
   gs_RxBkgdProcessFlag = TRAINING_DONE;
}

/*****************************************************************************
;   Subroutine Name: RxSRADecisionForTest_VDSL2( )
;
;   This subroutine computes the SRA parameters for the SRA test.
;
;   Prototype:
;   void RxSRADecisionForTest_VDSL2(void)
;
;   Input Arguments:
;
;   Output Arguments:
;      none
;
;   Return:
;      none
;
;   Global Variables:
;      gl_rx_sframe_count      -- (I) RX superframe count
;      gt_RxBitSwapInfo      -- (I) structure containing RX bit swap info
;
;      gsa_RxBat[]            -- (O) RX Bit Allocation Table
;      gsa_RxFineGains[]      -- (O) RX fine gain table
;      gsa_RxCombinedGains[]   -- (O) RX combined gain table
;
*******************************************************************************/

void RxSRADecisionForTest_VDSL2(void)
{
   int16 i;
   int16 s_NumSRATones;
   int16 *psa_RxFineGains;

   psa_RxFineGains = ghpsa_RxFineGains_Inactive;

   gft_RxBitSwapFlag = FALSE;

   // Start cutting from start tone till the programmed num of tones to cut.
   // Cut by the programmed delta value
   i = TESTArray[TEST_SRA_StartTone];

   for (s_NumSRATones = 0; s_NumSRATones < TESTArray[TEST_SRA_NumTones]; s_NumSRATones++, i++)
   {
      *(ghpuca_RxBat_Inactive+i) += TESTArray[TEST_SRA_delta];

      SETTONEFLAG(guca_RxBitswapToneSet, i);

   }

   gt_rx_config_v2.ul_Reconfig_Lp[LP0] = gt_rx_config_v2.ul_Lp[LP0] + (s_NumSRATones * TESTArray[TEST_SRA_delta]);

   gs_NumOfBitSwapTones = ConvertGi_dBtoLinear();


   gft_ToneOrderOK = ReorderTRT_Rx();

   if(gft_ToneOrderOK == FAIL)
   {
      goto _end;
   }

   // Reset Test Bitswap Flag
   TESTArray[TEST_ReconfigControl] &= ~TEST_TestOLR;

   gft_RxBitSwapFlag = TRUE;

_end:

   //Indicate the background process is done
   gs_RxBkgdProcessFlag = TRAINING_DONE;
}



/*****************************************************************************
;   Subroutine Name: RxSRADecision_VDSL2( )
;
;   This subroutine implements the automatic SRA.
;
;   Prototype:
;   void RxSRADecision_VDSL2(void)
;
;   Input Arguments:
;
;   Output Arguments:
;      none
;
;   Return:
;      none
;
;   Global Variables:
*******************************************************************************/
void RxSRADecision_VDSL2(void)
{
   int16 s_path, s_delta_Lp;
   int16 *psa_RxFineGains, *psa_SnrBuf;
   uint8 *puca_RxBat;
   int32 l_ActualDeltaSumLp, l_SRATotalLp_Delta;
   FlagT ft_Result;

   FlagT ft_SRA_LimitAtMargin;
   FlagT ft_forceEven1Bit; // XDSLRTFW-3118 (Start_End)
   FlagT ft_Fast_UpShift_SRA_Mode;
   uint16 us_BiGiModificationControlFlags;

   int32 l_Dummy;

   // Initialization of local variables
   l_ActualDeltaSumLp = 0;
   l_SRATotalLp_Delta = 0;
   us_BiGiModificationControlFlags = 0;
   ft_SRA_LimitAtMargin = FALSE;
   ft_Fast_UpShift_SRA_Mode = FALSE;
   ft_forceEven1Bit = TRUE;

   s_path = gs_data_path;
   psa_RxFineGains = ghpsa_RxFineGains_Inactive;
   puca_RxBat = ghpuca_RxBat_Inactive;
   psa_SnrBuf = gpsa_MeasuredSnrBuf;

   // SRA-process:         SRA period until desired margin/rate has been reached
   // SRA-task:            SRA block using the same SNR measurement (typically up to 10 SRA requests)
   // SRA-request:         SRA request message send to CO (max 128 tones)
   // SRA-bkgtask:         SRA background task calculating the

   //                            | SRA-bkgtask              | SRA-request              | SRA-task               | SRA-process
   //                            | (max 10/20/50 bits)      | (max 128 tones)          | (max 10 requests)      |
   // ===========================+==========================+==========================+========================+==========================
   // gs_delta_Lp                | max Lp delta for each    |                          |                        |
   //                            | SRA-bkgtask              |                          |                        |
   // ---------------------------+--------------------------+--------------------------+------------------------+--------------------------
   // gl_ActualDeltaSumLp        |                          | Desired Lp change in     |                        |
   //                            |                          | current SRA request      |                        |
   // ---------------------------+--------------------------+--------------------------+------------------------+--------------------------
   // gs_MaxTones_SRAIteration   |                          | max number of tones per  |                        |
   //                            |                          | SRA-request              |                        |
   // ---------------------------+--------------------------+--------------------------+------------------------+--------------------------
   // gs_NumOfBitSwapTones       |                          | number of tones changed  |                        |
   //                            |                          | in the current SRA-req.  |                        |
   // ---------------------------+--------------------------+--------------------------+------------------------+--------------------------
   // gl_SRATotalLp_Delta        |                          |                          | Desired Lp change in   |
   //                            |                          |                          | current SRA task       |
   // ---------------------------+--------------------------+--------------------------+------------------------+--------------------------
   // gl_SRATotalLp_DeltaCnt     |                          |                          | Lp bits changed in     |
   //                            |                          |                          | current SRA task       |
   // ---------------------------+--------------------------+--------------------------+------------------------+--------------------------
   // gs_NoOfDsSraAfterSnrUpdate |                          |                          | number of SRA (perform)|
   //                            |                          |                          | requests in SRA task   |
   // ===========================+==========================+==========================+========================+==========================
   // used States / functions    | SRA_DECISION_INIT        |                          |                        |
   //                            | SRA_DECISION_CHANGE_     |                          |                        |
   //                            |     INACTIVE_TABLES      |                          |                        |



   switch(guc_SRAState)
   {
   case SRA_INIT:
      {
         // Clear the past status every SRA request
         gft_RxBitSwapFlag = FALSE;
         gl_ActualDeltaSumLp = 0;
         gft_RxSraInProgress = TRUE;
         gs_NumOfBitSwapTones = 0;
         gus_BitloadErrorCode = BITLOAD_NO_ERROR;


         // XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
         //   gl_SRATotalLpDeltaMax = MAX_LP_CHANGED_PER_SNRUPDATE; 1000
         //   gs_delta_Lp = LP_CHANGE_PER_BKGD_TASK; 10
         //   gs_MaxTones_SRAIteration = MAX_TONES_CHANGED_PER_SRA; 100
         // XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]


         // Get the count of loaded tones: gus_ncloaded based on the current BAT
         GetBatStats(puca_RxBat, (int16 *)(void *)&gus_Rx_Tcm_Num1bits, &gs_Tcm_Oh, &l_Dummy);

         // Compute the SRA delta once per SRA task ( which may involve several SRA requests)
         if (gl_SRATotalLp_DeltaCnt == 0)
         {
            int16 UpshiftSNRM, DownshiftSNRM;

            UpshiftSNRM = gt_SnrMgnConfig.s_TARSNRMds;
            if (guc_SraShiftControl & (SHIFT_TO_SHIFT_THRESH|SHIFT_TO_PROTECT_IMPROVE))
            {
               UpshiftSNRM = (int16)(gt_RxAutoSRA_Ctrl.us_RA_USNRM - guc_SraShiftOffset);
            }

            DownshiftSNRM = gt_SnrMgnConfig.s_TARSNRMds;
            if(guc_SraShiftControl & (SHIFT_TO_SHIFT_THRESH|SHIFT_TO_RATE_IMPROVE))
            {
               DownshiftSNRM = (int16)(gt_RxAutoSRA_Ctrl.us_RA_DSNRM + guc_SraShiftOffset);
            }

            gs_NoOfDsSraAfterSnrUpdate = 0;


#ifndef TEST_VDSL_SRA
            // After every SNR update, look at things afresh. Note that the G997 monitor margin is only
            // called once per second and may be too late to change the SRAInitiate flags according to the margin
            // changes in showtime.
            // Note: It is not allowed to switch off the SRA when the real down- or upshift margin is achieved!
            //       At this point the TR115 requirement must be considered, i.e. +/- 1 dB from the down- and upshift snr margin.
            //       Actually the FW uses guc_SraShiftOffset = SRA_TR115_SHIFT_OFFSET_1DOT3DB!
            if(gt_LineStatusDS.s_SnrMargin > UpshiftSNRM)
            {
               gft_DSRAInitiate = FALSE;
            }
            else // if (gt_LineStatusDS.s_SnrMargin <= UpshiftSNRM)
            {
               gft_USRAInitiate = FALSE;
            }

            if(gt_LineStatusDS.s_SnrMargin < DownshiftSNRM)
            {
               gft_USRAInitiate = FALSE;
            }
            else // if (gt_LineStatusDS.s_SnrMargin >= DownshiftSNRM)
            {
               gft_DSRAInitiate = FALSE;
            }
#endif

            if(gft_USRAInitiate)
            {
#ifdef TEST_VDSL_SRA
               gt_LineStatusDS.s_SnrMargin = (int16)(gt_RxAutoSRA_Ctrl.us_RA_USNRM);
               l_SRATotalLp_Delta = 100;
#else


               // Compute total change in Lp required at the start of a new SRA to decrease the average margin below the threshold
               // This computation assumes that each bit provides a 3db SNR. Measured average margin as well as the SRA margin threshold are in steps of 0.1db
               // ->  DeltaLp = ((CurrentAverageMargin  (Upshift Threshold 1))*NcLoadedTones/3
               // Note: A negative "l_SRATotalLp_Delta" denotes that the SRA upshift is done!
               l_SRATotalLp_Delta = ((gt_LineStatusDS.s_SnrMargin - UpshiftSNRM)* gus_ncloaded) / 30;
#endif
               if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled != RETX_SELECTED)
               {
                  // IFEC mode
                  if ((int32)(gt_rx_config_v2.ul_Lp[s_path] + l_SRATotalLp_Delta) > (int32)gul_AutoSRALmax)
                  {
                     // Variable gul_AutoSRALmax contains the TargetLp!
                     l_SRATotalLp_Delta = (int32)(gul_AutoSRALmax - gt_rx_config_v2.ul_Lp[s_path]);
                  }
               }
               // Minimum margin to be maintained on all tones, check prior to adding bits
               // Note: This MIN decision was added due to error "CHGBAT_RETCODE_CANT_CHANGE_LP4" when UpshiftSNRM was used!
               gs_MinTargetSnrMargin = MIN((((UpshiftSNRM)<<8)/10), gt_BitloadParam.s_TarSnrMgn);

            }
            else if(gft_DSRAInitiate)
            {
#ifdef TEST_VDSL_SRA
               gt_LineStatusDS.s_SnrMargin = (int16)(gt_RxAutoSRA_Ctrl.us_RA_DSNRM);
               l_SRATotalLp_Delta = 100;
#else

               // Compute total change in Lp required at the start of a new SRA to increase the average margin above the threshold
               // This computation assumes that each bit provides a 3db SNR. Measured average margin as well as the SRA margin threshold are in steps of 0.1db
               // -> DeltaLp = (CurrentAverageMargin (Downshift Threshold + 1))*NcLoadedTones/3
               // Note: A negative "l_SRATotalLp_Delta" denotes that the SRA downshift is done!
               l_SRATotalLp_Delta = (-(gt_LineStatusDS.s_SnrMargin - DownshiftSNRM) * gus_ncloaded) / 30;
#endif
               if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled != RETX_SELECTED)
               {
                  // IFEC mode
                  if (l_SRATotalLp_Delta > ((int32)gt_rx_config_v2.ul_Lp[s_path] - (int32)gul_AutoSRALmin))
                  {
                     // Variable gul_AutoSRALmin contains the TargetLp!
                     l_SRATotalLp_Delta = (int32)gt_rx_config_v2.ul_Lp[s_path] - (int32)gul_AutoSRALmin;
                  }
               }
               // Minimum margin to be maintained on all tones, check prior to cutting bits
               // Note: This MAX decision was added to relax the margin and to avoid a problem as seen for Upshift!
               gs_MinTargetSnrMargin = MAX((((DownshiftSNRM)<<8)/10),gt_BitloadParam.s_TarSnrMgn);

            }

            // Account for the flat fine gain reduction done as part of excess margin reduction
            if (gs_ApplyMarginReduction)
            {
               gs_MinTargetSnrMargin += gs_ExcessMarFGReduction_save;
            }

            // Check if any add/ cut possible to meet the margin
            // Margin may have changed in the SNR update during the ongoing SRA task. Don't continue with the SRA
            // if margin changed drastically in the other direction. Note that g997 monitoring is called only once per
            // second and you may come here before the monitor margin clears the SRA initiate flag based on the margin
            // changes. The idea here is that any  add/ cut should follow only as per SRA initiate flag

            if (l_SRATotalLp_Delta > 0)
            {
               // Clear the flag at the beginning of an SRA task
               gft_SNRUpdated_PostSRA = FALSE;

               gl_SRATotalLp_Delta = l_SRATotalLp_Delta;
            }
            else
            {
               gl_SRATotalLp_Delta = 0;
            }

         }

         // Check if any further SRA possible/ desired to meet the margin
         if (gl_SRATotalLp_Delta)
         {
            guc_SRAState = SRA_CHANGE_INACTIVE_TABLES;
         }
         else
         {
            // SRA flagged in the desired direction can't help further as Lp limit reached
            // or the margin has changed drastically and any SRA in the flagged direction is not desirable
            // any more
            gus_BitloadErrorCode = SRA_LP_LIMIT_REACHED;
            guc_SRAState = SRA_PROCESS_DONE;
         }
      }
      break;


   case SRA_CHANGE_INACTIVE_TABLES:
      {

         // General comment: gs_delta_Lp is used by ChangeLpKeepMaxMargin() to load or unload bits. gs_delta_Lp variable gives
         // the size (ie.number of bits) and direction (ie. loading or unloading) of bit loading modification per ChangeLpKeepMaxMargin()
         // function call. With a signal bit swap or SRA maximum 100 bit (except trellis bits) are changed with the default initialization
         // of gs_delta_Lp = LP_CHANGE_PER_BKGD_TASK = 10.
         // This means for a single bit swap or SRA
         // with gs_delta_Lp = 10 , ChangeLpKeepMaxMargin() will be called at least 10 times
         // with gs_delta_Lp = 20 , ChangeLpKeepMaxMargin() will be called at least 5 times
         // with gs_delta_Lp = 50 , ChangeLpKeepMaxMargin() will be called at least 2 times
         // with higher gs_delta_Lp SRA speed can be increased significantly!

         ft_Fast_UpShift_SRA_Mode = FALSE; // configuration for normal operation
         gs_delta_Lp = LP_CHANGE_PER_BKGD_TASK; // load maximum 10 bits per ChangeLpKeepMaxMargin() function calling

         if (gft_USRAInitiate)
         {
            // XDSLRTFW-4148 : OLR failure code "CHGBAT_RETCODE_VIOLATES_MARGIN" (gus_BitloadErrorCode) is often seen (Start)
            // Relax target margin criteria for up shift SRA in a way that it could load 1 extra bit/tone even when
            // the margin of that tone is at target margin level.
            // New margin criteria is target margin - 1.5dB, but minimum 1.5dB
            // i.e. for 6dB target margin the criteria will be at 4.5dB
            gs_MinTargetSnrMargin = MAX(gt_BitloadParam.s_TarSnrMgn - 0x180, 0x180);
            // XDSLRTFW-4148 : OLR failure code "CHGBAT_RETCODE_VIOLATES_MARGIN" (gus_BitloadErrorCode) is often seen (End)

            // Fast Up Shift SRA
            if(gus_FastDsSraControl & FAST_DS_UPSHIFT_SRA_ENABLED )
            {
               // Note: it is seen from the test that when total number of DS bit change is less than the number of DS tones, sometimes with
               // gs_delta_Lp = 50 value the  ChangeLpKeepMaxMargin() algorithm fails due to not finding tones to load the bits.
               // This problem never observed with lower gs_delta_Lp  value. To overcome this problem it is empirically decided to configure
               // gs_delta_Lp = 20 when total number of bits need to be changed is 1.125 times higher than the number of DS tones.
               if ((gl_SRATotalLp_Delta - gl_SRATotalLp_DeltaCnt) > (gus_ncloaded + (gus_ncloaded >> 3)) )
               {
                  ft_Fast_UpShift_SRA_Mode = TRUE;  // Enable Fast SRA mode for Up shift direction
                  gs_delta_Lp = 50; // load maximum 50 bits per ChangeLpKeepMaxMargin() function calling
               }
               else if ((gl_SRATotalLp_Delta - gl_SRATotalLp_DeltaCnt) > (gus_ncloaded >> 1) )
               {
                  ft_Fast_UpShift_SRA_Mode = TRUE;  // Enable Fast SRA mode for Up shift direction
                  // Configuration for average 1.125 bits/tone loading!
                  gs_delta_Lp = 20; // load maximum 20 bits per ChangeLpKeepMaxMargin() function calling
               }
               // Note: if above conditions are not satisfied then FW uses the default ft_Fast_UpShift_SRA_Mode = FALSE
               // and gs_delta_Lp = LP_CHANGE_PER_BKGD_TASK values


               // XDSLRTFW-4169 : Maintaining maximum number of SRA modified tones
               // For each SRA request, Old SRA algorithm internally modified maximum 10 bits per iteration. This internal
               // iteration continues as long as the maximum number of modified tones are less than 100 (MAX_TONES_CHANGED_PER_SRA).
               // This means maximum 109 tones (considering last but one iteration can modify 99 tones) can be modified by SRA
               // algorithm before the trellis coding function.
               // Algorithm designed in XDSLRTFW-4051 changes the maximum number of modified bits/iteration from 10 to 50 and 20.
               // This means at corner cases, maximum 99+50 = 149 or 99+20 = 119 tones can be modified by SRA algorithm before
               // the trellis coding function. In case of modified tone greater than 128, SRA algorithm fails due to maximum SRA
               // tone criteria and in case of modified tones 119 before trellis coding, SRA algorithm can fail with the same
               // criteria if trellis coding algorithm adds more than 9 (128-119 = 9) bits.
               // Solution:
               // A check is introduced in the SRA: if pre calculation shows that the last internal iteration can overshoot
               // MAX_TONES_CHANGED_PER_SRA (100) tones then that iteration can modified maximum 10 bits (ie.tones). This new
               // check will maintain the maximum number of modified tones per SRA before the trellis coding function to 109
               // like the old algorithm.

               // Note: Headroom is (MAX_NUM_BITSWAP_TONES - gs_MaxTones_SRAIteration)
               //       Example:
               //       gs_delta_Lp = 10: 10 * 1bit tones + trellis -> 10 tones + 3 trellis tones + 1 trellis trellis tone + even trellis correction
               //                         -> ~15 swap tones
               if ((gs_MaxTones_SRAIteration-gs_NumOfBitSwapTones) < gs_delta_Lp)
               {
                  gs_delta_Lp = LP_CHANGE_PER_BKGD_TASK;
               }

            }
         }

         // limit the number of bits changed in one call to the function to gs_delta_Lp
         // this ensures that overhead messaging handler can be added to background queue
         // without long delay
         s_delta_Lp = gs_delta_Lp;

         // Limit the cut/ add for the last run
         if ((gl_SRATotalLp_DeltaCnt + gs_delta_Lp) > gl_SRATotalLp_Delta)
         {
            s_delta_Lp = gl_SRATotalLp_Delta - gl_SRATotalLp_DeltaCnt;
         }
         // if a down-shifting SRA then negate the Lp value
         if(gft_DSRAInitiate)
         {
            s_delta_Lp = -s_delta_Lp;
         }

         if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled != RETX_SELECTED)
         {
            // Check if it violates the rate constraints
            gft_SRARateViolated = CheckForSRAConstraints(s_delta_Lp, s_path);
         }

         // Cut s_delta_Lp bits from the tones with lowest margins or
         // Add s_delta_Lp bits to tones with highest margins

         // No need to force even number of 1-bit tones yet, We will take care of it in the end when done with
         // the complete desired Lp change for the SRA

         // XDSLRTFW-3118 (Start)
         // Odd number of 1 bit tones violates trellis and leads to bad DTUs / CRCs. It leads to link drops if we can't correct this violation on time, e.g. last SRA gets rejected by CO. Hence disallow odd number of 1-bit tones
         // NOTE: This code change needs a follow up, in order to avoid not executed DS SRAs when many 1-bit tones are used. In this case, with this code change we fail with BITLOAD_ERROR_EVEN_1BIT_TONES1
         // XDSLRTFW-3120 Follow up DS downshift SRA with many 1-bit tones
         /*
         if ((gs_NumOfBitSwapTones < (gs_MaxTones_SRAIteration - gs_delta_Lp)) && (gl_SRATotalLp_DeltaCnt < (gl_SRATotalLp_Delta - gs_delta_Lp))
               && (gl_SRATotalLp_DeltaCnt < (gl_SRATotalLpDeltaMax - gs_delta_Lp)))
         {
            ft_forceEven1Bit = FALSE;
         }
         else
         {
            ft_forceEven1Bit = TRUE;
         }
         */
         // XDSLRTFW-3118 (End)

         //XDSLRTFW-1784 (Showtime Margin Equalization - START)
         ft_SRA_LimitAtMargin = FALSE;
         if (TESTArray[TEST_Control4] & TEST_Control4_SRA_LimitAtMgn_Bit2_Mask)
         {
            //While adding bits make sure, each tones meet the required per tone margin
            ft_SRA_LimitAtMargin = TRUE;
         }
         //XDSLRTFW-1784 (Showtime Margin Equalization - END)
         //Link drop is seen if we Enable fine gain check for DS SRA,
         //This is observed when sudden removal of the noise is applied.

         // XDSLRTFW-4135 Showtime drop after 20sec in 35B tests with DS SRA enabled (Start)
         // Background: During interoperability testing against ADTRAN based DSLAM with 35B and DS SRA enabled
         // we observed a NE_ESE link drop after about 20 seconds in Showtime.

         // Root cause : SRA algorithm does not check the minimum margin criteria while loading or unloading a tone.
         // For unloading (down shift SRA) it is desired, but it could lead to a very bad or even negative per tone margin
         // while loading a tone without checking the final per tone margin. This negative margin could
         // cause CRC/Bad DTU and even link drop.

         // Solution: Enabling ft_SRA_LimitAtMargin flag make sure that before loading any bit for downstream up shift SRA,
         // tone is checked for sufficient SNR. If tone does not have enough margin to maintain it's (min SRA) margin then
         // SRA algorithm does not load any bit on that tone.
         // This constrain is not needed for DS down shift SRA because there the goal is to remove bits from tone to
         // improve the per tone margin even though the per tone margin could not meet the min margin criteria.

         if (gft_USRAInitiate)
         {
            ft_SRA_LimitAtMargin = TRUE;
         }
         // XDSLRTFW-4135 Showtime drop after 20sec in 35B tests with DS SRA enabled (End)


         // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (Start)
         // Initialize us_BiGiModificationControlFlags bit fields for ChangeLpKeepMaxMargin() function
         if (ft_SRA_LimitAtMargin == TRUE)
         {
            us_BiGiModificationControlFlags = FT_STOP_AT_MARGIN_THRESHOLD;
         }
         if (ft_forceEven1Bit == TRUE)
         {
            us_BiGiModificationControlFlags |= FT_FORCE_EVEN_1_BIT;
         }
         if (ft_Fast_UpShift_SRA_Mode == TRUE)
         {
            us_BiGiModificationControlFlags |= FT_FAST_UPSHIFT_SRA_MODE;
         }

         // note: RxSRADecision_VDSL2() function does not use FT_ADD_FINE_GAIN_PLUS_DELTA_LP flag bit

         ft_Result = ChangeLpKeepMaxMargin(puca_RxBat, psa_RxFineGains, (int32)s_delta_Lp,  guca_RxSupportedToneSet,
                                           (uint8 *)guca_RxBitswapToneSet, &l_ActualDeltaSumLp, us_BiGiModificationControlFlags);
         // XDSLRTFW-4075 Optimize input arguments of Bitswap and SRA functions (End)

         // We expect the call to change Lp as specified. If not, declare failure with the SRA decision
         if((ft_Result != FAIL) && ((l_ActualDeltaSumLp == (int32)s_delta_Lp)))
         {

            // keep track of total number of bits added/cut per SRA task
            // Nota that this count is always positive whether doing upshift or downshift
            if(gft_USRAInitiate)
            {
               gl_SRATotalLp_DeltaCnt += (int32)s_delta_Lp;
            }
            else
            {
               gl_SRATotalLp_DeltaCnt -= (int32)s_delta_Lp;
            }


            // keep track of total number of bits added/cut per SRA request
            // Note that this count is a signed number..negative for downshift and positive for upshift
            gl_ActualDeltaSumLp += l_ActualDeltaSumLp;

            // Convert the gains of all the tones, but those being changed, to 1/finegain in Q3.13 for storing back to HW
            // Return the number of tones that are being changed
            gs_NumOfBitSwapTones = CountNumTonesChanged();

            // move to next state if more than 120 tones have been changed
            // or if all the total number of bits to be cut/added is attained
            if ((gs_NumOfBitSwapTones >= gs_MaxTones_SRAIteration) || (gl_SRATotalLp_DeltaCnt >= gl_SRATotalLp_Delta))
            {

               gula_SRAFailStat[SRA_ALGO_FINDS_NECESSARY_BITS_FOR_CHANGING]++;
               // Write OLR Event into debug buffer
#ifdef DEBUG_TRACES
               {
                  int16 s_MsgRaw[3];

                  s_MsgRaw[0] = (int16)(gl_SRATotalLp_DeltaCnt | (ft_forceEven1Bit << 12));
                  s_MsgRaw[1] = (int16)gl_ActualDeltaSumLp;
                  s_MsgRaw[2] = (int16)gs_NumOfBitSwapTones;

                  DebugBuffWriteEv(DEBUG_EVT_OLR_SRA_DECISION, sizeof(s_MsgRaw), s_MsgRaw);
               }
#endif

               if (gs_NumOfBitSwapTones <= MAX_NUM_BITSWAP_TONES)
               {
                  gs_NoOfDsSraAfterSnrUpdate++;

                  gula_SRAFailStat[SRA_ALGO_SATISFY_MAX_NUM_BITSWAP_TONES]++;  // Statistics variable
//XDSLRTFW-1063: SRAds_Support_with_DS_ReTx (START)
                  if (gt_ReTXParams.uc_OMSG1_DsReTxEnabled == RETX_SELECTED)
                  {
                     guc_SRAState = SRA_RETX_FRM;
                  }
                  else
//XDSLRTFW-1063: SRAds_Support_with_DS_ReTx (END)
                  {
                     if (gft_FineGainOn)
                     {
                        // Increase the fine gains of affected tones with the constraint of
                        // not violating the fine gain restriction
                        DistributeFineGains(puca_RxBat, psa_RxFineGains, psa_SnrBuf, guca_RxBitswapToneSet);
                     }
                     // Convert the fine gains from dB to Linear domain for all the tones changed
                     ConvertGi_dBtoLinear();

                     guc_SRAState = SRA_IN_PROGRESS;
                  }
               }
               else
               {
                  // exceeded the max number of tones
                  gus_BitloadErrorCode = SRA_EXCEED_MAX_TONES;
                  guc_SRAState = SRA_PROCESS_DONE;
                  gula_SRAFailStat[SRA_ALGO_FAILED_MAX_NUM_BITSWAP_TONES]++;
               }
            }
         }
         else
         {
            // Flag error as it couldn't be completed as desired
            if (gus_BitloadErrorCode == 0)
            {
               //XDSLRTFW-3562_DS_SRA_NOT_TRIGGERED_AT_150m_START
               //Different error code is required to identify the cause for
               //not triggering DS SRA.
               gus_BitloadErrorCode = SRA_ERROR_BITLOAD_NOT_FEASIBLE;
               //XDSLRTFW-3562_DS_SRA_NOT_TRIGGERED_AT_150m_END
               gula_SRAFailStat[SRA_FAILED_BITLOAD_NOT_FEASIBLE]++;
            }
            guc_SRAState = SRA_PROCESS_DONE;
         }
      }
      break;

      //XDSLRTFW-1063: SRAds_Support_with_DS_ReTx (START)

   case SRA_RETX_FRM:
      {
         uint32 ul_Reconfig_Lp;

         ul_Reconfig_Lp = (uint32)(gt_rx_config_v2.ul_Lp[s_path] + gl_ActualDeltaSumLp);
         //XDSLRTFW-3493(Start)
         if (TESTArray[TEST_Control4] & TEST_Control4_Enhanced_ReTx_Framing_Bit10_Mask)
         {
            ft_Result = RtxFramingVdsl_V3(ul_Reconfig_Lp, &gt_rx_config_v2);
            // Note: Before calling the function "RtxFramingVdsl_V3()" the variables
            //          - gt_rx_config_v2.ul_Reconfig_Lp[s_path] and
            //          - gt_rx_config_v2.ul_Lp[s_path];
            //       are always equal!
            //       FramingGen can be success if input=ul_Reconfig_Lp is equal or greater than gt_rx_config_v2.ul_Reconfig_Lp[s_path], i.e.
            //       Output is smaller equal Input -> (gt_rx_config_v2.ul_Reconfig_Lp[s_path] <= ul_Reconfig_Lp)!
            //       The mentioned FramingGen behaviour is same and valid for upshift and downshift, i.e. downshift a bigger downshift and
            //       upshift a smaller upshift would take place.
         }
         else
         {
            ft_Result = RtxFramingVdsl_V1(ul_Reconfig_Lp, &gt_rx_config_v2);
         }
         //XDSLRTFW-3493(End)
         if(ft_Result != FAIL)
         {
            // XDSLRTFW-4037 SRA can always lead to an unsupported framing (Start)
            // Intra-DTU framing cannot always find framing parameters which fit exactly to the required Lp
            // Concept for IDTU:
            // 1. Calculate desired Lp (here: ul_Reconfig_Lp)
            // 2. Calculate an IDTU-framing which result in an actual Lp which is smaller than / equal to the desired Lp
            // 3. Modify SRA-request to ensure that it fits to the calculated "actual Lp"
            //
            // Current limitation:
            // To avoid conflicts with Trellis bits, we consider for the bit-reduction only tones which have a bitloading
            // of more than 2 bits
            if (gt_rx_config_v2.ul_Reconfig_Lp[s_path] < ul_Reconfig_Lp)
            {
               int16 s_ch, s_CurRxBand;

               for (s_CurRxBand=0; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
               {
                  for (s_ch=gsa_RxBandLeftChannel[s_CurRxBand]; s_ch <= gsa_RxBandRightChannel[s_CurRxBand]; s_ch++)
                  {
                     //If this tone is a SRA tone
                     if(IS_TONEFLAGSET(guca_RxBitswapToneSet, s_ch))
                     {
                        //Consider only tones which a bitloading of > 2bits
                        while (puca_RxBat[s_ch] > 2)
                        {
                           puca_RxBat[s_ch]--;
                           ul_Reconfig_Lp--;
                           // Reduce "desired Lp" until it fits to the "desired Lp"
                           if (ul_Reconfig_Lp == gt_rx_config_v2.ul_Reconfig_Lp[s_path])
                           {
                              // Set new delta value!
                              // -> gl_ActualDeltaSumLp = (NewLp - OldLp), with
                              //    - Downshift (NewLp < OldLp): more bits will be removed, i.e. smaller negative value.
                              //    - Upshift   (NewLp > OldLp): less bits will be removed, i.e. smaller positive value.
                              //      Note: It must be taken care that the upshift will not end-up in a downshift, i.e. new delta
                              //            must be greater zero!
                              if ((gl_ActualDeltaSumLp > 0) && ((int32)(ul_Reconfig_Lp - gt_rx_config_v2.ul_Lp[s_path]) < 0))
                              {
                                  gl_ActualDeltaSumLp = 0;    // Indicate fail!
                              }
                              else
                              {
                                 gl_ActualDeltaSumLp = (ul_Reconfig_Lp - gt_rx_config_v2.ul_Lp[s_path]);
                              }

                              goto EndOf_SRA_Lp_Reduction_For_Framing;
                           }
                        }
                     }
                  } //for (s_ch=gsa_RxBandLeftChannel[s_CurRxBand]
               } // for (s_CurRxBand=0; s_CurRxBand <
            }
EndOf_SRA_Lp_Reduction_For_Framing:
            // XDSLRTFW-4037 SRA can always lead to an unsupported framing (End)

            if (ul_Reconfig_Lp == gt_rx_config_v2.ul_Reconfig_Lp[s_path])
            {
               // If we reach this point we have found:
               // - a correct framing and
               // - a correct SRA-request message which fits to the calculated framing

               if (gft_FineGainOn)
               {
                  // Increase the fine gains of affected tones with the constraint of
                  // not violating the fine gain restriction
                  DistributeFineGains(puca_RxBat, psa_RxFineGains, psa_SnrBuf, guca_RxBitswapToneSet);
               }
               // Convert the fine gains from dB to Linear domain for all the tones changed
               ConvertGi_dBtoLinear();
               guc_SRAState = SRA_IN_PROGRESS;
            }
            else
            {
               if (gus_BitloadErrorCode == 0)
               {
                  //XDSLRTFW-3562_DS_SRA_NOT_TRIGGERED_AT_150m_START
                  //Different error code is required to identify the cause for
                  //not triggering DS SRA.
                  gus_BitloadErrorCode = SRA_ERROR_RETX_FRAMING_NOT_FEASIBLE;
                  //XDSLRTFW-3562_DS_SRA_NOT_TRIGGERED_AT_150m_END
               }
               guc_SRAState = SRA_PROCESS_DONE;
            }
         }
         else
         {
            guc_SRAState = SRA_PROCESS_DONE;
         }

         if(guc_SRAState == SRA_PROCESS_DONE)
         {
            if(gft_USRAInitiate)
            {
               gl_SRATotalLp_DeltaCnt -= (int32)gs_delta_Lp;
            }

            if(gft_DSRAInitiate)
            {
               gl_SRATotalLp_DeltaCnt += (int32)gs_delta_Lp;
            }
         }

      }
      break;
      //XDSLRTFW-1063: SRAds_Support_with_DS_ReTx (END)
   case SRA_IN_PROGRESS:
      {
         // Perform tone-reordering
         gft_ToneOrderOK = ReorderTRT_Rx();

         // if no change in Lp then do not send a SRA request
         if((gl_ActualDeltaSumLp != 0)  && (gft_ToneOrderOK != FAIL))
         {

            // Compute the new Lp after adding/cutting bits
            gt_rx_config_v2.ul_Reconfig_Lp[s_path] = (uint32)(gt_rx_config_v2.ul_Lp[s_path] + gl_ActualDeltaSumLp);

            gft_RxBitSwapFlag = TRUE;
         }

         guc_SRAState = SRA_PROCESS_DONE;
      }
      break;

   case SRA_PROCESS_DONE:
      {
         gl_ActualDeltaSumLp = 0;     // restore these global variables so that bit-swaps are not affected
         gs_delta_Lp = 10;            // value 10 is according to default SRA configuration.
         gft_RxSraInProgress = FALSE;
         guc_SRAState = SRA_DONE;
         if (gft_RxBitSwapFlag == FALSE)
         {

            // XDSLRTFW-4152: Enhance exit condition of SRA state machine (Start)
            // When code pointer comes to the point where SRA is rejected internally due to
            // any reason (i.e Margin failure, Trellis failure, Tone reordering failure etc.),
            // it is better to terminate the current SRA task so that after updating SNR freshly
            // new SRA can be triggered.

            gl_SRATotalLp_DeltaCnt = 0;
            gl_SRATotalLp_Delta = 0;
            gs_NoOfDsSraAfterSnrUpdate = 0;
            // XDSLRTFW-4152: Enhance exit condition of SRA state machine (End)

            // give a chance for bitswaps decision to execute in next symbol
            gft_SRA_FAIL = TRUE;
            gt_rx_config_v2.ul_Reconfig_Lp[s_path] = gt_rx_config_v2.ul_Lp[s_path];    //XDSLRTFW-1475
         }
         // Write OLR Event into debug buffer
#ifdef DEBUG_TRACES
         {
            int16 s_MsgRaw[3];

            s_MsgRaw[0] = (int16)gt_rx_config_v2.ul_Reconfig_Lp[s_path];
            s_MsgRaw[1] = (int16)gl_SRATotalLp_Delta;
            s_MsgRaw[2] = (int16)gus_BitloadErrorCode;

            DebugBuffWriteEv(DEBUG_EVT_OLR_SRA_DECISION_DONE, sizeof(s_MsgRaw), s_MsgRaw);
         }
#endif

      }
      break;
   }

   // Break up the current background task into 3 background tasks
   gs_RxBkgdProcessFlag = TRAINING_DONE;

}
// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [End]

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void RxBitSwapGainConversion_VDSL2(void)
*
*   This function converts the inverse of the RX linear fine gains to the fine gains
*   in 8.8 dB format.
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      gs_NumOfRxBands - (I) the number of RX bands
*      gsa_RxBandLeftChannel[] - (I) the array of left band edges
*      gsa_RxBandRightChannel[] - (I) the array of right band edges
*      ghpsa_RxFineGains_Inactive[] - (I/O) pointer to the inactive fine gain table
*      gs_RxBkgdProcessFlag - (O) flag to indicate the background process done or not
*      gla_SumFGainLinSq[] - (O) Sum of FG Linear Square per band
*-------------------------------------------------------------------------------
*/
void RxBitSwapGainConversion_VDSL2(void)
{
   int16 i, j;
   int16 s_NumOfBands;
   int16 *psa_BandLeftChannel, *psa_BandRightChannel;
   int16 *psa_FineGainTable, *psa_FineGain;
   int16 s_FineGain;
   int32 l_Gain;

   // XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs (Start)
   uint8 *puca_RxBat;
   uint8 c_Bits;
   int16 s_FGainLin;
   int32 l_SumFGainLinSq_add;
   int32 l_SumFGainLinSq;
   // XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs (End)

   //Get the input variables
   s_NumOfBands = gs_NumOfRxBands;
   psa_BandLeftChannel = gsa_RxBandLeftChannel;
   psa_BandRightChannel = gsa_RxBandRightChannel;
   psa_FineGainTable = ghpsa_RxFineGains_Inactive;

   puca_RxBat = ghpuca_RxBat_Inactive; // XDSLRTFW-3323

   for(j=0; j<s_NumOfBands; j++)
   {
      l_SumFGainLinSq = 0;
      //Set the pointer to the first gain in the current band
      psa_FineGain = psa_FineGainTable + psa_BandLeftChannel[j];

      for(i = psa_BandLeftChannel[j]; i <= psa_BandRightChannel[j]; i++)
      {
         //Get the inverse of Rx linear fine gain (Q3.13)
         s_FineGain = *psa_FineGain;

         //Invert this gain to get the RX linear fine gain (Q3.13)
         if(s_FineGain > 0)
         {
            l_Gain = NormAndDivide_32by16bit(((int32)1<<30), s_FineGain);

            //l_Gain is the linear gain in Q3.13
            l_Gain = round(l_Gain, 4);

            //Convert the linear fine gain to dB (Q8.8)
            s_FineGain = ConvertToDB(l_Gain);

            //Since the input to ConvertToDB Q3.13 format and
            //ConvertToDB computes 10log10(gain), but we need to compute 20*log10(gain)
            //so we need to adjust s_FineGain to get the desired dB (in Q8.8)
            s_FineGain = (s_FineGain<<1) - 0x4E44; // 0x4E44 = 20*log10(2^13) * 256

            // XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs (start)
            c_Bits = puca_RxBat[i];
            if (c_Bits != 0) // consider tones with only loaded bits
            {
               s_FGainLin = (int16)((l_Gain +  (int32)(1 << 3)) >> 4); // convert linear FG to Q3.9 format
               MULS16(l_SumFGainLinSq_add, s_FGainLin, s_FGainLin);
               l_SumFGainLinSq +=l_SumFGainLinSq_add ; // Sum of fine gain linear square of each band
            }
            // XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs (end)
         }
         else
         {
            s_FineGain = NEG_INFINITY_DB;
         }

         //Store this gain back in place
         *psa_FineGain++ = s_FineGain;

         //XDSLRTFW-1784 (Showtime Margin Equalization - START)
         //set bit 1 to log rx fine gain
         //XDSLRTFW-1784 (Showtime Margin Equalization - END)
      } //for(i = psa_BandLeftChannel[j];

      // Update Sum of Linear FG square per band
      gla_SumFGainLinSq[j] = l_SumFGainLinSq; //XDSLRTFW-3323: DS fine gain violation after Bitswaps / SRAs
   } //for(j=0

   //Indicate the background process is done
   gs_RxBkgdProcessFlag = TRAINING_DONE;

}

void ChangeRxGainsForBitSwap(void)
{
   int16 s_NumOfBitSwapTones = 0;
   int16 i, s_band;

   for (s_band=0; s_band < gs_NumOfRxBands; s_band++)
   {
      for (i=gsa_RxBandLeftChannel[s_band]; i <= gsa_RxBandRightChannel[s_band]; i++)
      {
         //If this tone is a bit-swap tone, store it directly (for forming message)
         if(IS_TONEFLAGSET(guca_RxBitswapToneSet, i))
         {

            //In VR9 HW, we cannot copy bat and gain table separately
            //so we have to save the changed bits value during the active to inactive table copy
            //and now copy them back to inactive BAT table
            ghpuca_RxBat_Inactive[i] = gpuca_BitSwapBat[s_NumOfBitSwapTones];
            //gpsa_BitSwapGain[] store the modified linear fine gain
            ghpsa_RxFineGains_Inactive[i] = gpsa_BitSwapGain[s_NumOfBitSwapTones++];
         }

      } //for (i=gsa_RxBandLeftChannel[s_band]
   } //for (s_band=0; s_band <

   //Indicate the background process is done
   gs_RxBkgdProcessFlag = TRAINING_DONE;
}


// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra [Start]
/*
*-------------------------------------------------------------------------------
*
*   Prototype: FlagT CheckForSRAConstraints(int16 s_delta_Lp, int16 s_path)
*
*   This function checks for the violation of rate constraints
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*      gs_NumOfRxBands - (I) the number of RX bands
*      gsa_RxBandLeftChannel[] - (I) the array of left band edges
*      gsa_RxBandRightChannel[] - (I) the array of right band edges
*      ghpsa_RxFineGains_Inactive[] - (I/O) pointer to the inactive fine gain table
*      gs_RxBkgdProcessFlag - (O) flag to indicate the background process done or not
*-------------------------------------------------------------------------------
*/

FlagT CheckForSRAConstraints(int16 s_delta_Lp, int16 s_path)
{
   FlagT ft_retval = FALSE;
   uint32 ul_TargetLp;

   ul_TargetLp = (uint16)(gt_rx_config_v2.ul_Lp[s_path] + (int32)s_delta_Lp);

   // Check if new target Lp is within the allowed Lp range
   if ((ul_TargetLp < gul_AutoSRALmin) || (ul_TargetLp > gul_AutoSRALmax))
   {
      ft_retval = TRUE;
   }

   return(ft_retval);
}

// XDSRTFW-442: Feature_All_All_All_All_Vdsl2Sra[End]

//******************  Noise Margin Equalization code - start ****************//
//XDSLRTFW-1784 (Showtime Margin Equalization - START)
/*^^^
*-----------------------------------------------------------------------------
*FlagT DistributeFineGainsOneTone(uint8 *puca_RxBat,
*                                                               int16 *psa_RxFineGains,
*                                                               int16 *psa_SNRBuffer,
*                                                               int16 s_ch,
*                                                               int16 s_band,
*                                                               int16 s_currMargin,
*                                                               int16 s_reqdMargin)
*
*       Description :
*
*       Add fine gains to the required tone to increase margin as much as possible, without violatine the RMS
*       fine gain constraint.
*
*       Input Parameters:
*               puca_RxBat              : BAT
*               psa_RxFineGains : Fine Gain Array (dB)
*               psa_SNRBuffer   : SNR
*     s_ch : required tone
*     s_band: Band at which fine gain boots required
*     s_currMargin: current margin
*     s_reqdMargin: required margin threshold.
*       Output Parameters:
*       pl_SumFGainLinSq        : Linear Sum Square Fine Gains
*               psa_RxFineGains : Modified fine gain array
*
*       Global Variables:
*               gs_RxMinConstSize: (I) the minimum constellation size allowed
*               gs_RxMaxConstSize: (I) the maximum constellation size allowed
*               gs_NumOfRxBands: (I) the number of RX bands
*               gsa_RxBandLeftChannel: (I) the left channels of the bands
*               gsa_RxBandRightChannel: (I) the right channels of the bands
*               gs_max_fine_gain: (I) the maximum fine gain
*               gsa_ConstellationSNR: (I) the required SNR for bitloading of different constellation size
*               gla_AvgFGainLinSqInit[]: (I) the average of gain squared of each band computed during the initialization
*               gla_SumFGainLinSq[]: (O) the sum of gain squared of each band computed in showtime
*-----------------------------------------------------------------------------*/

FlagT DistributeFineGainsOneTone(uint8 *puca_RxBat,
                                 int16 *psa_RxFineGains,
                                 int16 *psa_SNRBuffer,
                                 int16 s_ch,
                                 int16 s_band,
                                 int16 s_currMargin,
                                 int16 s_reqdMargin)
{

   //int16 s_ch, s_band, i_BandIdx;
   int16 s_finegainreqd, s_BitAlloc, s_FineGain;
   int16 s_threshold, s_Margin, s_MinMargin, s_SNR;
   int32 l_SumFGainLinSq, l_MaxSumFGainLinSq, l_SumFGainLinSq_ToConsider;
   FlagT ft_flag_rms_increase, ft_Result;

   s_MinMargin = s_currMargin;

   ft_flag_rms_increase = 0;

   ft_Result = 0;

   MULS32xU16(l_MaxSumFGainLinSq, gla_AvgFGainLinSqInit[s_band], gusa_ncloaded[s_band]);
   l_SumFGainLinSq = gla_SumFGainLinSq[s_band];


   if (s_MinMargin < 0)
      s_MinMargin = 0;

   //Set it to 0.5 dB
   s_threshold = 128;
   //s_threshold = 64;  //0.25dB

   //If there is still chance to add more fine gain
   while((l_SumFGainLinSq < l_MaxSumFGainLinSq))
   {
      //ft_AnyToneLefttotakeGain = 0;
      ft_flag_rms_increase = 0;

      //Get the bit allocaton and gain of this tone
      s_BitAlloc = puca_RxBat[s_ch];
      s_FineGain = psa_RxFineGains[s_ch];

      //Only change the tones which have been affected in bit-swap, which can be added more gain
      //and which have been loaded
      if ((s_FineGain < gs_max_fine_gain) && (s_BitAlloc !=0 ))
      {
         //ft_AnyToneLefttotakeGain = 1;

         //Compute the SNR margin adjusted with fine gain and coding gain
         s_SNR = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
         s_SNR += psa_RxFineGains[s_ch];

         s_Margin = s_SNR - gsa_ConstellationSNR[s_BitAlloc];

         //skip these tones if the margin is greater than the threshold
         //if (s_Margin  >= (s_MinMargin + s_threshold))
         //     continue;
         if (s_Margin  >= s_reqdMargin)
         {
            //Max margin reached
            ft_flag_rms_increase = 1;
            break;
         }

         //Compute the required fine gain to make the margin pass the threshold
         s_finegainreqd =  s_MinMargin + s_threshold - s_Margin ;

         if (s_finegainreqd > s_threshold)
            s_finegainreqd = s_threshold;

         s_finegainreqd += s_FineGain;

         //Limit the fine gain by its upper limit
         if(s_finegainreqd > gs_max_fine_gain)
            s_finegainreqd = gs_max_fine_gain;

         //Update fine gain linear sum square
         l_SumFGainLinSq_ToConsider = l_SumFGainLinSq;
         UpdateFGSumSqAndNOMATP(s_ch, s_FineGain, s_finegainreqd, &l_SumFGainLinSq_ToConsider, 0);

         //Check if adding this fine gain violates the average gain square constraint
         if(l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)
         {
            //No violation, update the fine gain and the sum of gain squares
            psa_RxFineGains[s_ch] = s_finegainreqd;
            l_SumFGainLinSq = l_SumFGainLinSq_ToConsider;
            ft_Result = 1;
         }
         else
         {
            //Indicate a sucessful try
            //Max RMS reached
            ft_flag_rms_increase = 1;
         }

      }
      else
      {
         //Max fine gain reached
         ft_flag_rms_increase = 1;
         break;
      }

      if ((ft_flag_rms_increase == 1))
         break;

      s_threshold += 128;
      //s_threshold += 64;

   } //while((l_SumFGainLinSq_ToConsider < l_MaxSumFGainLinSq)

   gla_SumFGainLinSq[s_band] = l_SumFGainLinSq;


   return (ft_Result);
} //FlagT DistributeFineGainsOneTone()


uint16 ShowtimeNoiseMarginEqualization(uint8 *puca_RxBat, int16 *psa_RxFineGains)
{
   int16 s_ch;
   int16 s_band, s_CurRxBand, s_CurRxBand_start;
   int16 s_MinMargin, s_AdditionalSNRNeeded_for_boost, s_FineGainToConsider, s_NumSkipBands;
   FlagT ft_Result, ft_PerBandMgnEqu;
   int16 s_ch_hi, s_band_hi, s_Margin_hi, s_AvgMargin, s_AdditionalSNRNeeded, s_AdditionalSNRNeeded_for_Reduction;
   RxToneFlags p_ActiveTones_ToConsider;
   int32 l_SumFGainLinSq_toconsider_hi, l_temp_hi, l_MaxSumFGainLinSq;
   uint16 us_nc_changed, us_FineGainEq_IterationCnt;
   int16 sa_flag_rms_increase[MAX_NUM_RX_BANDS];
   int16 s_MinMargin_save, s_ch_save, s_band_save;

   uint8 *puca_Bat = puca_RxBat;
   //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
   // Debug code start
   int16 *psa_SNRBuffer;
   int16 s_bi;
   int16 s_Margin;
   int16 s_fg;

   psa_SNRBuffer = gpsa_MeasuredSnrBuf; // measured SNR buffer

   // Debug code end
#endif
   //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)

   s_AvgMargin = ((gt_LineStatusDS.s_SnrMargin << 8) / 10); //Avg margin Q8.8 format

   s_NumSkipBands = 0;
   us_nc_changed = 0;
   us_FineGainEq_IterationCnt = 0;
   ft_PerBandMgnEqu = 0;
   s_MinMargin_save = 0;
   s_ch_save = 0;
   s_band_save = 0;

   for (s_CurRxBand = 0; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
   {
#if 1
      gusa_ncloaded[s_CurRxBand] = 0;
      puca_Bat = puca_RxBat + gsa_BitloadLeftChannel[s_CurRxBand];
      for (s_ch = gsa_BitloadLeftChannel[s_CurRxBand]; s_ch <= gsa_BitloadRightChannel[s_CurRxBand]; s_ch++)
      {
         if(*puca_Bat++ > 0)
            gusa_ncloaded[s_CurRxBand]++;
      }
#endif
      sa_flag_rms_increase[s_CurRxBand] = RMS_GAIN_INIT;
   }

   // Copy the active tone set to another array
   memcpy(p_ActiveTones_ToConsider, guca_RxSupportedToneSet, sizeof(RxToneFlags));

   // Start/end XDSLRTFW-3050 [VRX518] E_CODE_G997_1SEC_TASK_FAIL link drops in 35B mode
   // Changed abortion criteria for fine gain equalization from us_FineGainEq_IterationCnt < 128 to us_FineGainEq_IterationCnt < 64
   // Background is a corner case of the showtime-margin/fine-gain equalization in which we are failing to distribute the fine gains
   // for every iteration. Each iteration takes ~35symbols. When we have the IterationCnt abortion criteria at 128 we block for
   // 128*35symbols = 4480symbols (>1sec) all other background tasks. This leads to E_CODE_G997_1SEC_TASK_FAIL link drops because the
   // G997 counter update is unable to queue itself every 1sec. To avoid this we allow a maximum of 64 fine gain equalization iterations.
   // LIMITATION: This leads to fine gain equalization bitswap messages in which 64 instead of 128 tones are changed, hence a slower equalization
   // algorithm. But, better to have a stable link, than a fast showtime margin equalization.
   while ((us_nc_changed < MAX_NUM_BITSWAP_TONES) && (us_FineGainEq_IterationCnt < (MAX_NUM_BITSWAP_TONES >> 1)))
   {
      us_FineGainEq_IterationCnt++;
      s_AdditionalSNRNeeded = 0x7fff; // BIG number
      s_MinMargin_save = 0x7fff; // Initialize BIG number
      s_CurRxBand_start = 0;

      if ((s_NumSkipBands == (gs_NumOfRxBands)) && (ft_PerBandMgnEqu == 0))
      {
         ft_PerBandMgnEqu = 1;
         s_NumSkipBands = 0;
      }


      if  (ft_PerBandMgnEqu)
      {

         if (s_NumSkipBands < gs_NumOfRxBands)
         {
            sa_flag_rms_increase[s_NumSkipBands] = RMS_GAIN_INIT;
            s_CurRxBand_start = s_NumSkipBands;
         }
         else
         {
            guc_MarginEqualization_ExitCode = 8; //All search done
            break;
            //All search done & exit
         }

      }

      for (s_CurRxBand = s_CurRxBand_start; s_CurRxBand < gs_NumOfRxBands; s_CurRxBand++)
      {
         //Skip the band which cannot be added by more gains
         if(sa_flag_rms_increase[s_CurRxBand] == RMS_GAIN_SKIP)
            continue;

         sa_flag_rms_increase[s_CurRxBand] = RMS_GAIN_INIT;

         // !! Debug code to test maximum bits per tone per band, i.e. max constellation per band!!
         if (gsa_RxMaxConstSize[s_CurRxBand] > 0)
         {
            gs_RxMaxConstSize = gsa_RxMaxConstSize[s_CurRxBand];
         }

         s_band = s_CurRxBand;
         FindToneWithExtremeMargin(SMALLEST_MARGIN, puca_RxBat, psa_RxFineGains, gpsa_MeasuredSnrBuf,
                                   p_ActiveTones_ToConsider, gs_RxMinConstSize, gs_RxMaxConstSize,
                                   &s_ch, &s_band, &s_MinMargin);

         if (s_MinMargin < s_MinMargin_save)
         {
            s_MinMargin_save = s_MinMargin;
            s_ch_save = s_ch;
            s_band_save = s_band;
         }

      }


      s_MinMargin = s_MinMargin_save;
      s_ch = s_ch_save;
      s_band = s_band_save;

#ifdef NOISE_MARGIN_EQU_DEBUG
      if (s_MinMargin < 0) gus_dbg_MinMgn_Neg++;
#endif //NOISE_MARGIN_EQU_DEBUG

      s_band_hi = -1;
      if (ft_PerBandMgnEqu)
         s_band_hi = s_band;  //if flag is set do the perband margin equalization, no need to eqalize across the bands

      ft_Result = FindToneWithExtremeMargin(LARGEST_ACTUAL_MARGIN, puca_RxBat,
                                            psa_RxFineGains, gpsa_MeasuredSnrBuf,
                                            p_ActiveTones_ToConsider, gs_RxMinConstSize,
                                            (int16)(gs_RxMaxConstSize),
                                            &s_ch_hi, &s_band_hi, &s_Margin_hi);

      //Check margin diff between highest margin & lowest margin, if no difference or very small difference (say 1dB)
      //don't do anything or
      //don't continue to adjust the fine gain if Highest margin is below the Avg margin
      //both highest & lowest margin tones required margin boost or bit swap
      if (((s_Margin_hi - s_MinMargin) < 0x0100) || ( (s_AvgMargin - s_Margin_hi) > 0 ))
      {
         guc_MarginEqualization_ExitCode = 1; //No change is required for Margin Equalization
         sa_flag_rms_increase[s_band] = RMS_GAIN_SKIP;
         s_NumSkipBands++;
         continue;  //exit the while loop
      }


      //Is required margin can be increased through fine gains of smallest margin tone.
      //If fine gain is already max limit, fine gain can't increased further, such tones can't be considered
      //for fine gain boot and remove those tones from the list.
      if (psa_RxFineGains[s_ch] >= gs_max_fine_gain )
      {
         CLEARTONEFLAG(p_ActiveTones_ToConsider, s_ch);
         guc_MarginEqualization_ExitCode = 2;
         continue;
      }

      //Is required margin can be reduced through fine gains of highest margin tone.
      //If fine gain is already min limit, fine gain can't be reduced further, such tones can't be considered
      //for fine gain reduction and remove those tones from the list.
      if (psa_RxFineGains[s_ch_hi] <= gs_min_fine_gain_MarginRed )
      {
         CLEARTONEFLAG(p_ActiveTones_ToConsider, s_ch_hi);
         guc_MarginEqualization_ExitCode = 3;
         continue;
      }

      //if  s_ch & s_ch_hi are same don't do anything.

      //Estimate the required margin to meet the Avg margin
      //s_AdditionalSNRNeeded = ((gt_LineStatusDS.s_SnrMargin << 8) / 10) - s_MinMargin;
      s_AdditionalSNRNeeded = s_AvgMargin - s_MinMargin;

      //If smallest margin is above the Avg margin, no need to do anything
      /*
      if (s_AdditionalSNRNeeded < 0)
      {
         gs_dbg_SmallMgn_above_avg_margin++;
         continue;
      }
      */

      //If additional SNR needed is less than 1.0dB, don't change anything
      if (s_AdditionalSNRNeeded < 0x0100)
      {
         guc_MarginEqualization_ExitCode = 9; //No change is required for Margin Equalization
         sa_flag_rms_increase[s_band] = RMS_GAIN_SKIP;
         s_NumSkipBands++;
         continue;  //exit the while loop
      }

      //if additional fine gain boost required is reaching max fine gain,
      //limit fine gain to it's max level.
      if (s_AdditionalSNRNeeded > (gs_max_fine_gain - psa_RxFineGains[s_ch]))
      {
         s_AdditionalSNRNeeded = gs_max_fine_gain - psa_RxFineGains[s_ch];
      }

      s_AdditionalSNRNeeded_for_boost = s_AdditionalSNRNeeded;

      //Need to reduce only excess margin above average margin for the highest margin tone
      if (s_AdditionalSNRNeeded > (s_Margin_hi - s_AvgMargin))
         s_AdditionalSNRNeeded = (s_Margin_hi - s_AvgMargin);


      //Choose min of the max & min fine gain limit
      s_AdditionalSNRNeeded = MIN(s_AdditionalSNRNeeded_for_boost, s_AdditionalSNRNeeded);

      s_AdditionalSNRNeeded_for_Reduction = -s_AdditionalSNRNeeded;  //Change the sign to reduce the fine gain

      //Limit min fine gain threshold to -10dB
      if (s_AdditionalSNRNeeded_for_Reduction < (gs_min_fine_gain_MarginRed - psa_RxFineGains[s_ch_hi]))
      {
         s_AdditionalSNRNeeded_for_Reduction = gs_min_fine_gain_MarginRed - psa_RxFineGains[s_ch_hi];
      }

      //Need to verify that the required changes in the fine gain will meet per band lin squared sum of Gi <= 1.
      //Estimate for a band where fine gain reduction is required.
      //s_ncloaded = gusa_ncloaded[s_band_hi];
      l_SumFGainLinSq_toconsider_hi = gla_SumFGainLinSq[s_band_hi];
      s_FineGainToConsider = psa_RxFineGains[s_ch_hi];

      //save the current value
      l_temp_hi = l_SumFGainLinSq_toconsider_hi;

#ifdef NOISE_MARGIN_EQU_DEBUG
      //for debug
      if ((s_FineGainToConsider + s_AdditionalSNRNeeded_for_Reduction) < (-10*256))
         gus_dbg_minfg_exceed++;
#endif //NOISE_MARGIN_EQU_DEBUG

      UpdateFGSumSqAndNOMATP(s_ch_hi, s_FineGainToConsider, (int16)(s_FineGainToConsider + s_AdditionalSNRNeeded_for_Reduction), &l_SumFGainLinSq_toconsider_hi, 0);


      //Due to memory limitation, we cannot update NOMATP during the bit-swap
      //so we cannot check the NOMATP violation.
      //Though sub-optimal, here we choose to limit the sum of gain square not
      //exceed that computed during the initialization
      //In this way, we also limit the NOMATP to some degree
      MULS32xU16(l_MaxSumFGainLinSq, gla_AvgFGainLinSqInit[s_band_hi], gusa_ncloaded[s_band_hi]);

      //check lin squared sum of Gi <= 1 for highest margin tone
      if ((l_SumFGainLinSq_toconsider_hi < l_MaxSumFGainLinSq))
      {
         //Update fine gain for highest margin tone
         gla_SumFGainLinSq[s_band_hi] = l_SumFGainLinSq_toconsider_hi;

         //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
         // debug code start
         // this code logged Tone, FG, BAT, Margin of the tones before bits are loaded in TRUE-FALSE-FALSE configuration
         s_bi = puca_RxBat[s_ch];

         s_Margin = AddCodingGainToSnr(s_ch, psa_SNRBuffer);
         // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
         // avoid the possible overflow problem (for negative infinity) and also fine for 0db case.
         s_Margin += psa_RxFineGains[s_ch];
         s_fg = psa_RxFineGains[s_ch];
         // When subtracting bits and looking for the tone with the smallest margin, we use bi_delta = 0.
         s_Margin -= gsa_ConstellationSNR[s_bi];
#endif
         //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)
         ft_Result = DistributeFineGainsOneTone(puca_RxBat,
                                                psa_RxFineGains,
                                                gpsa_MeasuredSnrBuf,
                                                s_ch,
                                                s_band,
                                                s_MinMargin,
                                                (s_MinMargin + s_AdditionalSNRNeeded));

         if (ft_Result)
         {
            //Update fine gain for lowest margin tone
            // DistributeFineGainsOneTone() function modified the FineGain of tone s_ch!! Need to modify that function in a way that
            // it can take care guca_RxBitswapToneSet and us_nc_changed by its own!

            SETTONEFLAG(guca_RxBitswapToneSet, s_ch);
            us_nc_changed++;

            //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(Start)
#ifdef BS_INSIDE_DEBUG
            // Debug code Start
            // log  tone idx, initial BAT, FG and Margin information of bit swap-able tone
            //logging for lowest margin tone
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_ch;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_bi;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_Margin;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_fg;


            // debug code start
            // this code logged Tone, FG, BAT, Margin of the tones before bits are loaded in TRUE-FALSE-FALSE configuration
            s_bi = puca_RxBat[s_ch_hi];

            s_Margin = AddCodingGainToSnr(s_ch_hi, psa_SNRBuffer);
            // For bi=0 tones, its fine gain either is negative infinity db or 0db, so ignore its fine gain here
            // avoid the possible overflow problem (for negative infinity) and also fine for 0db case.
            s_Margin += psa_RxFineGains[s_ch_hi];
            s_fg = psa_RxFineGains[s_ch_hi];
            // When subtracting bits and looking for the tone with the smallest margin, we use bi_delta = 0.
            s_Margin -= gsa_ConstellationSNR[s_bi];
            //logging for highest margin tone
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_ch_hi;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_bi;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_Margin;
            gsa_BSDebugBuf[gs_DSDebugBufCnt++] = s_fg;

            // Debug code end
#endif
            //XDSLRTFW-3807 Improve bitswap algorithm in VRX518 code(End)

            //Update fine gain for highest margin tone
            psa_RxFineGains[s_ch_hi] += s_AdditionalSNRNeeded_for_Reduction;
            SETTONEFLAG(guca_RxBitswapToneSet, s_ch_hi);
            us_nc_changed++;
         }
         else
         {

            gla_SumFGainLinSq[s_band_hi] = l_temp_hi;
            sa_flag_rms_increase[s_band] = RMS_GAIN_SKIP;
            s_NumSkipBands++;

         }
      }
      else
      {
         guc_MarginEqualization_ExitCode = 7;
         gla_SumFGainLinSq[s_band_hi] = l_temp_hi;
      }

   } //while

   return (us_nc_changed);
} //uint16 ShowtimeNoiseMarginEqualization(uint8 *puca_RxBat, int16 *psa_RxFineGains)
//XDSLRTFW-1784 (Showtime Margin Equalization - END)
