/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
 *------------------------------------------------------------------------
 *
 *   Aware DMT Technology. Proprietary and Confidential.
 *
 *   40 Middlesex Turnpike, Bedford, MA 01730-1413
 *   Phone (781) 276 - 4000
 *   Fax   (781) 276 - 4001
 *
 *   BitSynchHandler_VDSL2.c
 *
 *   This file contains the routine that searchs for the beginning symbol
 *   of each five consecutive symbol group representing 1 bit in the VDSL2
 *   loop diagnostic mode.
 *
 *------------------------------------------------------------------------
 */

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "BitSynchHandler_VDSL2.h"
#include "fifo.h"
#include "ModulateSocMessages.h"

/*
*-------------------------------------------------------------------------------
*
*   Prototype: void BitSynchHandler(void)
*
*   In the diagnostic mode, each bit in SOC message is represented in five consecutive
*   symbols whose tones are modulated by either all (++) or (--) phase for bit of 0 and 1
*   respectively.
*
*   This function finds the starting symbol of each five consecutive symbol group (representing 1 bit).
*
*   Input Arguments:
*
*   Output Arguments:
*
*   Returns:
*
*   Global Variables:
*
*   gs_AlgHandlerCount      -- (I/O) count the number of symbols added to the accumulator
*   gs_BitSearchSymCount   -- (I/O) track which symbol in the five symbol group
*   gl_RxBitStartSym      -- (O) the RX symbol count of the first symbol in the three consecutive symbol group
*   gs_NumOfSymbolsForBitSearchAccum -- (I) the number of symbols to accumulate
*   gs_RxBkgdProcessFlag   -- (I/O) flag to indicate if the RX background process is done or not
*   gpsa_RxRepBuf           -- (I/O) pointer to the accumulator array
*-------------------------------------------------------------------------------
*/

void BitSynchHandler(void)
{
   static uint8 uc_bit, uc_bit_pre;
   static int16 s_NumOfSymWithPhaseReverse;

   switch(gs_AlgHandlerState)
   {

   case BIT_SYNCH_START:

      gs_BitSynchInterationCnt = 0;
      gs_AlgHandlerState = BIT_SYNCH_INIT;

      break;

      //--------------------------------------------------
      //Initialization
      //--------------------------------------------------
   case BIT_SYNCH_INIT:

      //Initialize various counts
      gs_AlgHandlerCount = 0;
      gs_BitSearchSymCount = 0;
      s_NumOfSymWithPhaseReverse = 0;

      //Initialize the start symbol count (for a bit group of five symbols) to be the next symbol
      gl_RxBitStartSym = gl_RxSymbolCount+1;

      //Clear the five 16-bit accumulator buffer
      memset(gpsa_RxRepBuf, 0, (sizeof(int16)*gus_NumSymbolsPerBit));

      //Decode the current symbol (return 1 or 0) and save it to previous byte
      DecodeSOCMessage(gpsa_RxToneBuf, gs_NumOfToneClustersForMsgDecode, gs_RxNumBytesPerSymbol, &uc_bit_pre);

      //go to the next substate
      gs_AlgHandlerState = BIT_SYNCH_ACCUM;

      break;

   case BIT_SYNCH_ACCUM:

      //Decode the current symbol (return 1 or 0)
      DecodeSOCMessage(gpsa_RxToneBuf, gs_NumOfToneClustersForMsgDecode, gs_RxNumBytesPerSymbol, &uc_bit);

      //Check if the current bit is differnt from the previous bit
      //(1 different, 0 same)
      uc_bit_pre = (uc_bit ^ uc_bit_pre);

      //Count the number of symbols which are phase reversed from their previous symbol
      s_NumOfSymWithPhaseReverse += (int16)uc_bit_pre;

      //Add the distance between current and previous bit to the corresponding accumulator
      //where we have five accumulators[i], i = 0, 1, .. 4
      //and the accumulator[i] stores the sum of the distances for symbol i+m*5,
      //for m = 0, 1, ..., gs_NumOfSymbolsForBitSearchAccum
      gpsa_RxRepBuf[gs_BitSearchSymCount] += uc_bit_pre;

      //Save the current to the previous bit
      uc_bit_pre = uc_bit;

      //Update counts
      gs_BitSearchSymCount++;
      if(gs_BitSearchSymCount == gus_NumSymbolsPerBit)
      {
         //reset this count every 5 symbols
         gs_BitSearchSymCount = 0;

         //update count which counts the number of 5-symbol group that have been added to accumulator
         gs_AlgHandlerCount++;

         //If the desired number of symbols have been reached
         if(gs_AlgHandlerCount == gs_NumOfSymbolsForBitSearchAccum)
         {
            //Make sure not all the bits in this accumulation period are the same
            //(for this case, we cannot find the bit boundary)
            if((s_NumOfSymWithPhaseReverse == 0) &&
                  (gs_BitSynchInterationCnt < MAX_BIT_SYNCH_ITERRATIONS))
            {
               //Restart the bit synchronization search
               gs_AlgHandlerState = BIT_SYNCH_INIT;
            }
            else
            {
               //Start the backgroup task to select
               //the best three symbols in a five symbol group which
               //have the lowest accumulated distance
               gs_RxBkgdProcessFlag = TRAINING_IN_PROGRESS;
               AddFunctionToBkgdFifo((PtrToBkgdFunc)BgBitSymbolSearch);

               //go to the next substate
               gs_AlgHandlerState = BIT_SYNCH_SELECT;
            }

            //Update the number of iteration
            gs_BitSynchInterationCnt++;
         }
      }

      break;

   case BIT_SYNCH_SELECT:

      //Wait for the backgroud process to finish
      if(gs_RxBkgdProcessFlag == TRAINING_DONE)
      {
         gs_AlgHandlerState = BIT_SYNCH_DONE;
      }

   } //switch(gs_AlgHandlerState)

} //void BitSynchHandler(void)

void BgBitSymbolSearch(void)
{

   int16 i, s_BestStartSym, s_max;


   //Search for the largest value in the accumulator and its index represent
   //the beginning symbol of a bit group of five symbols

   s_BestStartSym = 0;
   s_max = gpsa_RxRepBuf[0];
   for(i=1; i<gus_NumSymbolsPerBit; i++)
   {
      if(gpsa_RxRepBuf[i] > s_max)
      {
         s_max = gpsa_RxRepBuf[i];
         s_BestStartSym = i;
      }
   }

   //Set the symbol count for the first symbol of a bit group of five symbols, say, of index N.
   //Then all the symbol of index N + m*5, N+1 +m*5, N+2+m*5, for m=0,1,2,...
   //are the first symbol of the corresponding bit symbol groups
   gl_RxBitStartSym += s_BestStartSym;

   gs_RxBkgdProcessFlag = TRAINING_DONE;

} //void BgBitSymbolSearch(void)
