/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*****************************************************************************
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   HlinHandler.c
;
;   Routine to Hlin, the frequency response of the channel in linear domain
;
;****************************************************************************/

#include <string.h>
#include "common.h"
#include "gdata.h"
#include "fifo.h"
#include "fdq.h"
#include "accum32.h"
#include "IRI_Iof.h"
#include "HlinHandler.h"
#include "dsp_op.h"
#include "decimalgain.h"
#include "mul.h"
#include "GetQlnHlogForCarrierGroup.h"
#include "vdsl_xception.h"
#include "FdqHandler.h"

void BgInitHlinBuf(void);
void BgHlinCalc(void);
void CalcHlinParam(int16 s_Hlog, int16 *psa_Fdq, uint16 *pusa_HlinBuf);
void BgHlinScale(void);

/*^^^
 *------------------------------------------------------------------------
 *
 *  Function Name: HlinHandler
 *
 *  Description: This function computes the Hlin parameter for all
 *  the Medley Set freqency tones.
 *
 *  Prototype: void HlinHandler(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *  Global Variables:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void HlinHandler(void)
{
   int16 j;

   switch(gs_AlgHandlerState)
   {

      //----------------------------------------------
      //Initialize variables for Hlin calculation
      //----------------------------------------------
   case HLIN_INIT:

      //Initialize Hlin Buffer
      //Call the backgroud process to initialize Hlin buffer
      //(note, this cannot be done in real-time for 30a band plan)
      gs_RxBkgdProcessFlag = TRAINING_IN_PROGRESS;
      AddFunctionToBkgdFifo((PtrToBkgdFunc)BgInitHlinBuf);

      // For VRx518 the RTV is twice, i.e. for 1024 tones.
      // Note: Due to memory layout the buffers gsa_pre_FDQ_coef and guca_pre_FDQ_exp cannot be increased,i.e.
      //       the correct size leads to an compiler error!
      //       Therefore the number of processed tones is adapted for the HlinHandler().
      //       Further it is not clear, if the bigger number leads to MIPS problems!
      // Note: Devision by 4 due to byte conversion and array size of gsa_pre_FDQ_coef[(NUM_CHANNELS_PER_GROUP_512 <<1)].
//      if ((gs_NumChannelsPerGroup > NUM_CHANNELS_PER_GROUP_512) &&
//          ((sizeof(gsa_pre_FDQ_coef) >> 2) == NUM_CHANNELS_PER_GROUP_512))
//      {
//         gs_NumChannelsPerGroup = NUM_CHANNELS_PER_GROUP_512;
//      }

      //Set the first frequency band to process
      gs_CurrentRxBand = 0;
      gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];

      //go to next substate
      gs_AlgHandlerState = HLIN_SETUP;

      break;

      //---------------------------------------------------------------
      // Setup variables for Hlin calculation on the current tone group
      //---------------------------------------------------------------
   case HLIN_SETUP:

      //Make sure the left channel starting at the beginning of subcarrier group
      j = gs_LeftChannel;
      gs_LeftChannel >>= gs_Log2CarrierGroupSizeRx_Medley;
      gs_LeftChannel <<= gs_Log2CarrierGroupSizeRx_Medley;

      if(gs_LeftChannel < j)
      {
         gs_LeftChannel += gs_CarrierGroupSizeRx_Medley;
      }

      // Update the right channel
      gs_RightChannel = gs_LeftChannel + gs_NumChannelsPerGroup - 1;
      if(gs_RightChannel > gsa_RxBandRightChannel[gs_CurrentRxBand])
      {
         gs_RightChannel = gsa_RxBandRightChannel[gs_CurrentRxBand];
      }

      //Make sure the right channel starting at the beginning of subcarrier group
      gs_RightChannel >>= gs_Log2CarrierGroupSizeRx_Medley;
      gs_RightChannel <<= gs_Log2CarrierGroupSizeRx_Medley;

      gs_NumOfTonesInBand = gs_RightChannel-gs_LeftChannel+1;

      //If there is not enough tones in this band (at least one subcarrier group,
      //go to next band
      if(gs_NumOfTonesInBand <= 0)
      {
         gs_AlgHandlerState = HLIN_NEXT_TONEGROUP;
      }
      else
      {
         //Load the FDQ coefficients to the array gsa_pre_FDQ_coef[]
         AddFunctionToFifo(gp_RxLoadingFunctionFifo, ReadFDQ);

         //go to next substate
         gs_AlgHandlerState = HLIN_CALC;
      }

      break;

      //---------------------------------------------------------------
      // Compute the Hlin
      //---------------------------------------------------------------
   case HLIN_CALC:

      //Wait until the initialization of Hlin buffer is done
      if(gs_RxBkgdProcessFlag == TRAINING_DONE)
      {
         //Call the backgroud process to compute Hlin
         gs_RxBkgdProcessFlag = TRAINING_IN_PROGRESS;
         AddFunctionToBkgdFifo((PtrToBkgdFunc)BgHlinCalc);

         //go to the next substate
         gs_AlgHandlerState = HLIN_NEXT_TONEGROUP;
      }

      break;

      //----------------------------------------------------
      //Set up Hlin variables for the next tone group
      //----------------------------------------------------
   case HLIN_NEXT_TONEGROUP:

      if(gs_RxBkgdProcessFlag == TRAINING_DONE)
      {
         // Determine whether all in-band tones have been processed
         j = gs_RightChannel + gs_CarrierGroupSizeRx_Medley;
         if(j > gsa_RxBandRightChannel[gs_NumOfRxBands-1])
         {
            //Call the backgroud process to scale Hlin
            gs_RxBkgdProcessFlag = TRAINING_IN_PROGRESS;
            AddFunctionToBkgdFifo((PtrToBkgdFunc)BgHlinScale);

            gs_AlgHandlerState = HLIN_SCALE;
         }
         else
         {
            // Continue to process the remaining tones
            if(j <= gsa_RxBandRightChannel[gs_CurrentRxBand])
            {
               gs_LeftChannel = gs_RightChannel+1;
            }
            else
            {
               gs_CurrentRxBand++;
               gs_LeftChannel = gsa_RxBandLeftChannel[gs_CurrentRxBand];
            }

            // go to the next substate
            gs_AlgHandlerState = HLIN_SETUP;
         }
      }
      break;

      //----------------------------------------------------
      //Scale Hlin Coefficients
      //----------------------------------------------------
   case HLIN_SCALE:

      if(gs_RxBkgdProcessFlag == TRAINING_DONE)
      {
//         gs_NumChannelsPerGroup = gs_NumChannelsPerGroupSave;

         // Set flag to indicate Hlin calculation is done
         gs_AlgHandlerState = HLIN_DONE;
      }

      break;
   } //switch(gs_AlgHandlerState)


} //void HlinHandler(void)

void BgInitHlinBuf(void)
{
   int16 j;
   uint16 *pusa_HlinBuf;

   pusa_HlinBuf = gpusa_HlinBuf_NE;
   for(j=0; j<3*512; j++)
   {
      *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
   }

   gs_RxBkgdProcessFlag = TRAINING_DONE;

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Function Name: BgHlinCalc
 *
 *  Description: This function computes the Hlin for all subcarrier groups
 *  in a given band (in the range between gs_LeftChannel and gs_RightChannel)
 *
 *  Prototype: void BgHlinCalc(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *  Global Variables:
 *      gs_LeftChannel -- (I) left channel index
 *      gs_RightChannel -- (I) right channel index
 *      gs_NumOfTonesInBand -- (I) number of tones between left and right channel
 *      gs_CarrierGroupSizeRx_Medley -- (I) the size of subcarrier group (G)
 *      gs_Log2CarrierGroupSizeRx_Medley -- (I) Log2(G)
 *
 *      gpusa_HlinBuf_NE -- pointer to the computed Hlin in the format described in Message O/R-MSG-LD
 *------------------------------------------------------------------------
 *^^^
 */
void BgHlinCalc(void)
{
   int16 s_log2_G, s_log2_G_ori;
   int16 s_CarrierGroupStartIdx, s_CarrierGroupEndIdx;
   int16 s_FirstTone;
   uint16 *pusa_HlinBuf;
   int16 j, k;
   uint16 us_return, us_Hlog=OUT_OF_RANGE_HLOG;

   //Hlog table is currently computed using the discovery phase G
   s_log2_G_ori = gs_Log2CarrierGroupSizeRx_Disc;

   //Using the subcarrier group size computed based on MedleySet
   s_log2_G = gs_Log2CarrierGroupSizeRx_Medley;

   //We cannot perform the conversion from a smaller G to a larger G
   if(s_log2_G > s_log2_G_ori)
   {
      EnterFailStates(E_CODE_HLIN_G_CONVERSION);
   }

   //Compute the beginning subcarrier group index of Hlin for this frequency band
   s_CarrierGroupStartIdx = gs_LeftChannel >> s_log2_G;

   //Compute the ending subcarrier group index of Hlin for this frequency band
   s_CarrierGroupEndIdx = gs_RightChannel >> s_log2_G;

   pusa_HlinBuf = &gpusa_HlinBuf_NE[s_CarrierGroupStartIdx*3];

   //Loop through all the subcarrier groups in this frequency range
   for(k = s_CarrierGroupStartIdx; k <= s_CarrierGroupEndIdx; k++)
   {

      //Get the Hlog value for this subcarrier group
      us_return = GetCarrierGroupIdx(k, OUT_OF_RANGE_HLOG, s_log2_G_ori, s_log2_G);

      if(us_return != (uint16)0xFFFF)
      {
         us_Hlog = gsa_RxHlog[us_return];
      }

      if((us_return != (uint16)0xFFFF) && (us_Hlog != OUT_OF_RANGE_HLOG))
      {
         //compute the FDQ index for tone k*G
         s_FirstTone = k << s_log2_G;

         j = s_FirstTone - gs_LeftChannel;
         j <<= 1;

         //Compute the parameter Hlin parameters: s(k), a(k) and b(k)
         CalcHlinParam(gsa_RxHlog[us_return], &gsa_pre_FDQ_coef[j], pusa_HlinBuf);
         pusa_HlinBuf += 3;
      }
      else
      {
         //This is an invalid subcarrier group, set s, a, b to OUT_OF_RANGE_HLIN
         *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
         *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
         *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
      }

   }

   gs_RxBkgdProcessFlag = TRAINING_DONE;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Function Name: CalcHlinParam
 *
 *  Description:
 *  This function computes the Hlin parameter s, a, b
 *  (as defined in the standard) for a subcarrier group.
 *  The way in which it is computed is the following:
 *
 *  Hlin(k) = s(k) * (a(k) + jb(k))      //not counting the scaling for now
 *         = A(k) exp (j w(k))
 *         = A(k) (cos (w(k)) + j sin(w(k)))
 *
 *  The amplitude A(k) is computed from Hlog(k) by:
 *
 *  Hlog(k) = 20 log10(A(k)), thus
 *  A(k) = 10^(Hlog(k)/20)
 *
 *  The angle w(k) is computed from FDQ coefficients.
 *  If FDQ(k) = X(k) + jY(k), then,
 *
 *
 *  cos(w(k)) = X(k)/sqrt(X(k)*X(k) + Y(k)*Y(k))
 *  sin(w(k)) = -Y(k)/sqrt(X(k)*X(k) + Y(k)*Y(k))
 *
 *  Prototype: void CalcHlinParam(int16 s_Hlog_in, int16 *psa_Fdq, uint16 *pusa_HlinBuf)
 *
 *  Input Arguments: none
 *      s_Hlog_in - (I) input HLOG value in the format given by the standard
 *      psa_Fdq -- (I) pointer to the input FDQ coefficients (real and imaginary)
 *
 *  Output Arguments: none
 *      pusa_HlinBuf -- (O) pointer to the output Hlin buffer
 *
 *  Return: none
 *
 *  Global Variables:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void Hlog2Hlin(int16 s_Hlog_in, int16 *ps_A_mant, int16 *ps_A_exp);
void CosSinOfHlinPhase(int16 *psa_Fdq, int16 *ps_cos_mant, int16 *ps_cos_exp,
                       int16 *ps_sin_mant, int16 *ps_sin_exp);
void CalcHlinDivision(int16 s_num, int16 s_denom_mant, int16 s_denom_exp,
                      int16 *ps_quot_mant, int16 *ps_quot_exp);

void CalcHlinParam(int16 s_Hlog_in, int16 *psa_Fdq, uint16 *pusa_HlinBuf)
{
   int16 s_A_mant, s_A_exp;
   uint16 us_scale_mant;
   int16 s_cos_mant, s_cos_exp, s_sin_mant, s_sin_exp, s_exp, s_exp1;
   int16 s_ab_exp, s_total_exp, s_scale_exp;
   int32 l_scale;

   //If for some reason, FDQ is 0, no need to continue
   if((psa_Fdq[0] == 0) && (psa_Fdq[1] == 0))
   {
      pusa_HlinBuf[0] = (uint16)OUT_OF_RANGE_HLIN;
      pusa_HlinBuf[1] = (uint16)OUT_OF_RANGE_HLIN;
      pusa_HlinBuf[2] = (uint16)OUT_OF_RANGE_HLIN;
      return;
   }

   //Convert the Hlog (decimal format) to linear format
   Hlog2Hlin(s_Hlog_in, &s_A_mant, &s_A_exp);

   //Compute cos(w(k)) and sin(w(k)) from the FDQ coefficients
   CosSinOfHlinPhase(psa_Fdq, &s_cos_mant, &s_cos_exp, &s_sin_mant, &s_sin_exp);

   //At this point, Hlin can be computed as:
   //
   // Hlin = s_A_mant * (2^s_A_exp) * (s_cos_mant * 2^s_cos_exp + j s_sin_mant * 2^s_sin_exp)
   //------------------------------------------------------------------------------------

   //--------------------------------------------------------------------------------
   //In the following, we need to reformat Hlin so it is represented by:
   //
   // Hlin = (us_scale_mant/(2^15)) * (a(k) * j b(k))/(2^15)
   //

   //Normalize cos and sin value (this step might be redundant, but for double insurance)
   if(s_cos_mant != 0)
   {
      s_exp = norm_l((int32)s_cos_mant<<16);
      s_cos_mant <<= s_exp;
      s_cos_exp -= s_exp;
   }

   if(s_sin_mant != 0)
   {
      s_exp = norm_l((int32)s_sin_mant<<16);
      s_sin_mant <<= s_exp;
      s_sin_exp -= s_exp;
   }

   //Find the common exponent for a(k) and b(k)
   if(s_cos_mant == 0)
   {
      s_ab_exp = s_sin_exp;
      s_cos_exp = s_sin_exp;
   }
   else if(s_sin_mant == 0)
   {
      s_ab_exp = s_cos_exp;
      s_sin_exp = s_cos_exp;
   }
   else
   {
      s_ab_exp = MAX(s_cos_exp, s_sin_exp);
   }

   //Shift a(k) or b(k) so they share the same exponent
   if(s_cos_exp < s_ab_exp)
   {
      s_cos_mant >>= (s_ab_exp - s_cos_exp);
   }
   if(s_sin_exp < s_ab_exp)
   {
      s_sin_mant >>= (s_ab_exp - s_sin_exp);
   }

   //Normalize s_A_mant
   s_exp = norm_l((int32)s_A_mant<<16);

   //add 1 to change A_mant from sign to unsigned number
   s_exp++;
   l_scale = (int32)s_A_mant<<s_exp;
   us_scale_mant = (uint16)l_scale;
   s_scale_exp = s_A_exp - s_exp;

   //Compute the combined exponent of scale*a(k) or scale*b(k)
   s_total_exp = s_scale_exp + s_ab_exp;

   //Since The combined exponent must be -30, we need to make proper left or right shifts
   //to the mantissa to achieve this

   //Since all the mantissa is normalized, if the combined exponent is still greater than
   //-30, treat this group as the out of range Hlin
   if(s_total_exp > -30)
   {
      //Hlin is too high to represent
      *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
      *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
      *pusa_HlinBuf++ = (uint16)OUT_OF_RANGE_HLIN;
   }
   /*
      else if(s_total_exp == -29)
      {
         //set s_cos_mant = us_scale_mant * s_cos_mant/0xFFFF
         //~= us_scale_mant * s_cos_mant/0x10000
         //= (us_scale_mant * s_cos_mant) >> 16
         l_scale = us_scale_mant * s_cos_mant;
         l_scale >>= 16;
         s_cos_mant = sature16(l_scale);

         //set s_sin_mant = us_scale_mant * s_sin_mant/0xFFFF
         //~= us_scale_mant * s_sin_mant/0x10000
         //= (us_scale_mant * s_sin_mant) >> 16
         l_scale = us_scale_mant * s_sin_mant;
         l_scale >>= 16;
         s_sin_mant = sature16(l_scale);

         //Set the scale to the largest value
         us_scale_mant = 0xFFFF;
      }
   */
   //Scale down the mantissas to get the total exponent to -30
   else if(s_total_exp < -30)
   {
      //Compute the total number of combined right shifts need to perform
      s_exp = -30 - s_total_exp;

      //Apply half of this shift to scale and another half to a and b
      s_exp1 = s_exp>>1;
      us_scale_mant >>= s_exp1;

      s_exp -= s_exp1;
      s_cos_mant >>= s_exp;
      s_sin_mant >>= s_exp;
   }

   //Limit a(k) and b(k) betwee (-2^15+1) (= 0x8001) and (2^15-1) (=0x7FFF)
   if(s_cos_mant < (int16)0x8001)
   {
      s_cos_mant = (int16)0x8001;
   }

   if(s_sin_mant < (int16)0x8001)
   {
      s_sin_mant = (int16)0x8001;
   }

   //Store the results in the output buffer
   pusa_HlinBuf[0] = us_scale_mant;
   pusa_HlinBuf[1] = s_cos_mant;
   pusa_HlinBuf[2] = s_sin_mant;

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Function Name: BgHlinScale
 *
 *  Description:
 *  The VDSL2 Standard requires having one common scale for all a(k) and b(k)
 *  and max(|a(k)|, |b(k)|) to be 0x7FFF =(2^15) - 1.
 *  This function finds the one common scale and apply it to all a(k) and b(k)
 *  to meet the standard's requirement.
 *
 *  Prototype: void BgHlinScale(void)
 *
 *  Input Arguments: none
 *
 *  Output Arguments: none
 *
 *  Return: none
 *
 *  Global Variables:
 *
 *      gpusa_HlinBuf_NE - (I) pointer to Hlin array consisting of 512 triplets,
 *         where each triplet contains (scale(k), a(k) and b(k)), all in Q1.15 format.
 *         where scale is an unsigned number, a(k) and b(k) are signed number.
 *
 *      gpusa_HlinBuf_NE - (O) pointer to Hlin array consisting of 512 triplets,
 *         where each triplet contains (scale, a(k) and b(k)), all in Q1.15 format.
 *         where scale is an unsigned number, a(k) and b(k) are signed number.
 *         It is noted that "scale" in the output array is same for all k.
 *
 *------------------------------------------------------------------------
 *^^^
 */

void BgHlinScale(void)
{
   int16 i, k;
   uint16 *pusa_HlinBuf;
   int16 sa_HlinCoef[2], s_sign, s_shift;
   uint16 us_scale, us_scale_common, us_scale1;
   int32 l_Acc, l_Acc_max;

   //Find max(|scale(k)*a(k)|, |scale(k)*b(k)|) over all k
   pusa_HlinBuf = gpusa_HlinBuf_NE;
   l_Acc_max = 0;
   for(k=0; k<512; k++)
   {
      us_scale = *pusa_HlinBuf++;
      sa_HlinCoef[0] = (int16)*pusa_HlinBuf++;
      sa_HlinCoef[1] = (int16)*pusa_HlinBuf++;

      //Skip inactive tone
      if(sa_HlinCoef[0] == (int16)OUT_OF_RANGE_HLIN)
      {
         continue;
      }

      for(i=0; i<2; i++)
      {
         l_Acc = us_scale * sa_HlinCoef[i];

         if(l_Acc < 0)
         {
            l_Acc = - l_Acc;
         }

         if(l_Acc > l_Acc_max)
         {
            l_Acc_max = l_Acc;
         }
      }
   } //for(k=0; k<512; k++)

   //No valid tone, return
   if(l_Acc_max == 0)
   {
      us_scale_common = 0;
      goto _scale;
   }

   //Find scale such that scale*0x7FFF = l_Acc_max
   //=> scale = l_Acc_max/0x7FFF ~= l_Acc_max>>15 (use the shift to simplify operation)
   us_scale_common = (uint16)(l_Acc_max >> 15);

   //Limit the minimum scale to be 1
   if(us_scale_common < 1)
   {
      us_scale_common = 1;
   }

   //Convert it to a signed number since NormAndDivide_32by16bit() only take signed number
   us_scale1 = us_scale_common;

   if(us_scale1 & (uint16)0x8000)
   {
      us_scale1 >>= 1;
      s_shift = 1;
   }
   else
   {
      s_shift = 0;
   }

   //Rescale all a(k), b(k) so that they share the same scale factor
   pusa_HlinBuf = gpusa_HlinBuf_NE;
   for(k=0; k<512; k++)
   {
      us_scale = *pusa_HlinBuf++;
      sa_HlinCoef[0] = (int16)*pusa_HlinBuf++;
      sa_HlinCoef[1] = (int16)*pusa_HlinBuf++;

      //Skip inactive tone
      if(sa_HlinCoef[0] == (int16)OUT_OF_RANGE_HLIN)
      {
         continue;
      }

      //Compute scale(k) * a(k)/scale_common or scale(k) * b(k)/scale_common
      for(i=0; i<2; i++)
      {
         if(sa_HlinCoef[i] != 0)
         {
            l_Acc = us_scale * sa_HlinCoef[i];

            s_sign = 0;

            if(sa_HlinCoef[i] < 0)
            {
               l_Acc = - l_Acc;
               s_sign = 1;
            }

            l_Acc >>= s_shift;
            l_Acc = NormAndDivide_32by16bit(l_Acc, us_scale1);

            if(s_sign)
            {
               l_Acc = - l_Acc;
            }

            sa_HlinCoef[i] = sature16(l_Acc);

            //Limit a(k) and b(k) betwee (-2^15+1) (= 0x8001) and (2^15-1) (=0x7FFF)
            //per standard requirement
            if(sa_HlinCoef[i] < (int16)0x8001)
            {
               sa_HlinCoef[i] = (int16)0x8001;
            }
         }
      }

      //save the final Hlin to the output array
      pusa_HlinBuf -= 2;
      *pusa_HlinBuf++ = sa_HlinCoef[0];
      *pusa_HlinBuf++ = sa_HlinCoef[1];

   } //for(k=0; k<512; k++)

_scale:

   //Populate all entries for "scale" with us_scale_common
   pusa_HlinBuf = gpusa_HlinBuf_NE;
   for(k=0; k<512; k++)
   {
      *pusa_HlinBuf = us_scale_common;
      pusa_HlinBuf += 3;
   }

   //Set the CMV structure
   gt_ChannelMeasurement_NE.us_HlinSC = us_scale_common;

   //Indicate the background process is done
   gs_RxBkgdProcessFlag = TRAINING_DONE;

} //BgHlinScale()

//--------------------------------------------------
//Convert the Hlog (decimal format) to linear format
//--------------------------------------------------
void Hlog2Hlin(int16 s_Hlog_in, int16 *ps_A_mant, int16 *ps_A_exp)
{
   int16 s_Hlog_dB;
   int16 s_mant, s_exp;
   int32 l_Acc;

   //Hlog_dB = 6 - s_Hlog_in/10 = (60 - s_Hlog_in)/10 in dB
   s_Hlog_dB = 60 - s_Hlog_in;      //in unit of 0.1 dB

   //Since s_Hlog_dB is between +6 and -96 dB, we can represent it in Q8.8 format
   if(s_Hlog_dB > 60)
   {
      s_Hlog_dB = 60;
   }
   else if(s_Hlog_dB < -960)
   {
      s_Hlog_dB = -960;
   }

   //Change s_Hlog_dB (in multiple of 0.1 dB) to 8.8 format (in dB)
   //equivalent to multiply s_Hlog_dB by (256/10)
   s_mant = (256*16/10);      //x16 is to give 4 more digit precision
   MULS16(l_Acc, s_Hlog_dB, s_mant);

   //perform rounding and convert it back to Q8.8 format
   s_Hlog_dB = (int16)((l_Acc + (1<<3))>>4);

   //Compute 10^(s_Hlog_dB/20) put in s_mant (in Q1.15) and s_exp
   db_to_linear(s_Hlog_dB, &s_mant, &s_exp);

   *ps_A_mant = s_mant;
   *ps_A_exp = s_exp-15;   //now the mantissa is in Q16.0 format
}


//---------------------------------------------------------
//Compute Cos(w(k)) and Sin(w(k)) from the FDQ coefficients
//---------------------------------------------------------
void CosSinOfHlinPhase(int16 *psa_Fdq, int16 *ps_cos_mant, int16 *ps_cos_exp,
                       int16 *ps_sin_mant, int16 *ps_sin_exp)
{
   int16 s_X, s_Y;
   int16 s_denom, s_denom_exp;
   int32 l_sum, l_Acc;

   //Given, FDQ = X + jY, compute X*X + Y*Y
   s_X = psa_Fdq[0];
   s_Y = psa_Fdq[1];
   MULS16(l_sum, s_X, s_X);
   MULS16(l_Acc, s_Y, s_Y);
   l_sum += l_Acc;

   if(l_sum == 0)
   {
      *ps_cos_mant = 0;
      *ps_sin_mant = 0;
      *ps_cos_exp = 0;
      *ps_sin_exp = 0;

      return;
   }

   if(l_sum > (int32)(0x7FFF * 0x7FFF))
   {
      l_sum >>= 2;
      s_X >>= 1;
      s_Y >>= 1;
   }

   //Compute the sqrt of l_sum
   s_denom = sqrt32(l_sum);

   //Normalize the denominator and convert it to 16-bit number
   s_denom_exp = norm_l((int32)s_denom<<16);
   s_denom <<= s_denom_exp;
   s_denom_exp = - s_denom_exp;

   //Compute cos(w(k))
   if(s_X == 0)
   {
      *ps_cos_mant = 0;
      *ps_cos_exp = 0;
   }
   else
   {
      CalcHlinDivision(s_X, s_denom, s_denom_exp, ps_cos_mant, ps_cos_exp);
   }

   //Compute sin(W(k)
   if(s_Y == 0)
   {
      *ps_sin_mant = 0;
      *ps_sin_exp = 0;
   }
   else
   {
      CalcHlinDivision((int16)(-s_Y), s_denom, s_denom_exp, ps_sin_mant, ps_sin_exp);
   }
}


void CalcHlinDivision(int16 s_num, int16 s_denom_mant, int16 s_denom_exp,
                      int16 *ps_quot_mant, int16 *ps_quot_exp)
{
   int16 s_sign;
   int16 s_num_exp;
   int32 l_Acc;

   //Convert numerator to positive number if needed
   s_sign = 0;
   if(s_num < 0)
   {
      s_sign = 1;
      s_num = -s_num;
   }

   //Normalize the numerator
   s_num_exp = norm_l((int32)s_num<<16);
   s_num <<= s_num_exp;
   s_num_exp = -s_num_exp;

   //Compute s_num/s_denom
   Divide_16bit(s_num , s_num_exp, s_denom_mant, s_denom_exp, ps_quot_mant, ps_quot_exp);

   //Recover the sign
   if(s_sign)
   {
      *ps_quot_mant = -*ps_quot_mant;
   }

   //Since the maximum cos or sin is 1, we want to represent it in Q1.15 format
   if(*ps_quot_exp > -15)
   {
      s_num_exp = *ps_quot_exp + 15;
      l_Acc = *ps_quot_mant;
      l_Acc <<= s_num_exp;
      *ps_quot_mant = sature16(l_Acc);
      *ps_quot_exp = -15;
   }

}

