/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/**********************************************************************************************

**********************************************************************************************/

#include <string.h> // for memset and memcpy
#include "common.h"
#include "DshStreamAndEventDefines.h"
#include "DshInterface.h"
#include "dsp_op.h"
#include "fcs.h"
#include "nmp.h"
#include "hmp_data.h"
#include "cmv.h"
#include "gdata.h"
#include "fifo.h"
#include "vdsl_state.h"




static int32 DSH_IncQueueIndex(int32 l_Index)
{
   if (++l_Index >= DSH_QUEUE_DEPTH)
   {
      return(0);
   }
   else
   {
      return(l_Index);
   }
}


static int32 DSH_DecQueueIndex(int32 l_Index)
{
   if (--l_Index < 0 )
   {
      return(DSH_QUEUE_DEPTH - 1);
   }
   else
   {
      return(l_Index);
   }
}


static int32 DSH_WriteQueue(DSH_Statics_t * pt_Statics, uint16 us_StreamID, uint16 us_BlockSize, FlagT ft_DoCrc, void * p_Data, void (*Func)(P_DSH_Statics, P_DSH_QueueEntry))
{
   uint16 us_Crc16 = 0;
   int32 l_ReturnCode=-1;
   uint32 ul_int_status;
   uint8 *pc_c_Data;

   // as the host controler can't handle debug streams with an odd number of bytes
   // the block size has to be padded with a byte. To restore the data correctly
   // the odd flag in the status word is set in case the block size is odd.
   FlagT ft_IsOdd = (FlagT)(us_BlockSize & 0x01);
   us_BlockSize = (uint16)(((us_BlockSize + 1) >> 1)<<1);
   pc_c_Data = (uint8 *)p_Data;
   // calculate CRC
   if(ft_DoCrc)
   {
      for(uint16 j=0; j < us_BlockSize;j++)
      {
         us_Crc16 = calcCrc16(us_Crc16,*pc_c_Data++);
      }
   }
   disable1_save(&ul_int_status);

   int32 l_Idx = pt_Statics->c_QueueWrIdx;

   if( pt_Statics->c_QueueFill < DSH_QUEUE_DEPTH)
   {
      DSH_QueueEntry_t * pt_Entry = &pt_Statics->ta_Queue[l_Idx];
      pt_Entry->us_Id        = (uint16)(us_StreamID | DSH_PROTOCOL_VERSION);
      pt_Entry->us_Size      = us_BlockSize;
      pt_Entry->us_Length    = us_BlockSize;
      pt_Entry->us_Status    = (uint16)(pt_Statics->c_StreamNum++ | DSH_FIRST_FRAGMENT);
      pt_Entry->us_Status    |= DSH_STREAM_SOURCE_CPE;

      if(ft_IsOdd)
      {
         pt_Entry->us_Status |= DSH_STREAM_SIZE_IS_ODD;
      }
      pt_Entry->us_Crc       = us_Crc16;
      pt_Entry->ul_SymCount  = gul_DSH_SymCount;
      pt_Entry->p_Data       = p_Data;
      pt_Entry->Func         = Func;

      pt_Statics->c_QueueFill += 1;
      pt_Statics->c_QueueWrIdx = (int8)DSH_IncQueueIndex(l_Idx);

      l_ReturnCode = pt_Statics->c_StreamNum;
   }
   else
   {
      //@todo AH: Consider to send a message which indicates a Buffer Overflow
      static uint16 us_MailboxCode, us_Word0, us_Word1, us_Word2, us_Word3, us_Word4;
      //static AutoMsgPayLoadEntry_t *AutoMsgBufferOverflow;

      us_MailboxCode = AUTONOMOUS_MSG_NOTIFICATION;
      us_Word0 = AUTO_MSG_INF117_HEADER_0;
      us_Word1 = AUTO_MSG_INF117_HEADER_1;
      us_Word2 = AUTO_MSG_INF117_HEADER_2;
      us_Word3 = AUTO_MSG_INF117_HEADER_3;
      us_Word4 = AUTO_MSG_INF117_HEADER_4;

      //SubmitAutoMsg(pt_PayLoadToSend, us_MailboxCode, us_Word0, us_Word1, us_Word2, us_Word3, us_Word4);

      /*
      T_HMH_AutoMsg Msg;
      Msg.MessageId   = HMH_EVT_DBG_BUFFER_OVERFLOW;
      Msg.Length      = 4;
      Msg.Payload[0]  = StreamID;
      Msg.Payload[1]  = BlockSize;
      HMH_SendAutoMsg(&Msg);
      */
      l_Idx = DSH_DecQueueIndex(l_Idx);
      pt_Statics->ta_Queue[l_Idx].us_Status |= DSH_OVERFLOW_INDICATION;
      pt_Statics->ta_Queue[l_Idx].us_Status |= DSH_STREAM_SOURCE_CPE;
      l_ReturnCode = -1;
   }
   restore1_save(ul_int_status);        // Restore interrupts

   // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
   // 5 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
   if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-5))
   {
      AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);
   }
   else
   {
      // we ran out of space, keep track of this event
      gus_Dbg_DSH_skip_BGF_add++;
   }

   return l_ReturnCode;
}


static FlagT DSH_BgDbgStr(DSH_Statics_t * pt_Statics)
{
   uint16 * pus_Data;
   uint16 us_Status, us_Length, us_Chunk;
   uint32 ul_int_status;
   int32 l_Fill, l_Idx;
   DSH_QueueEntry_t * pt_Entry;



   if(NMP_IsDebugMsgPending() != 0)
   {
      // HMH busy? Quit and retry later...
      return TRUE;
   }

   // The call of NMP_SendDebugMsg will add a function to BG stack. Therefore, check if space on BG stack is really available
   // Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks
   // 5 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
   if (gp_BGTaskFifo->PendingFunctionCount >= (NUM_BG_BUFFERS-5))
   {
      // we ran out of space, keep track of this event
      gus_Dbg_DSH_skip_BGF_add++;
      return TRUE;
   }

   // copy current queue entry to local variables
   l_Idx = pt_Statics->c_QueueRdIdx;

   if(pt_Statics->ft_CleanUpLastStream)
   {
      pt_Statics->ft_CleanUpLastStream = FALSE;

      pt_Entry = &pt_Statics->ta_Queue[l_Idx];
      if(pt_Entry->Func != 0)
      {
         pt_Entry->Func(pt_Statics, pt_Entry);
      }


      l_Idx = DSH_IncQueueIndex(l_Idx);
      pt_Statics->c_QueueRdIdx = (int8)l_Idx;

      // we need to block interrupt to prevent race condition
      disable1_save(&ul_int_status);;
      l_Fill = pt_Statics->c_QueueFill;
      l_Fill--;
      pt_Statics->c_QueueFill = (int8)l_Fill;
      restore1_save(ul_int_status);         // Restore interrupts
   }
   else
   {
      disable1_save(&ul_int_status);;
      l_Fill = pt_Statics->c_QueueFill;
      restore1_save(ul_int_status);
   }

   if(l_Fill == 0)
   {
      // no pending stream? Quit.
      return FALSE;
   }

   pt_Entry = &pt_Statics->ta_Queue[l_Idx];

   us_Status  = pt_Entry->us_Status;
   us_Length  = pt_Entry->us_Length;
   pus_Data   = pt_Entry->p_Data;
   us_Chunk   = us_Length;

   // maximum chunk size is MID_MAX_PAYLOAD_SIZE - sizeof(stream id) - sizeof(stream status) - sizeof(sym. count) - sizeof(crc16)
   // Note: AndreasH 20151112: On Avinax the Chunk size was MID_MAX_PAYLOAD_SIZE -10 (see calculation above)
   // on VR9 we need -16 instead of -10 to prevent a Mailbox overflow for large messages. I don't know why, maybe due a larger message headers on VR9
   if(us_Length > (MAX_MAILBOX_PAYLOAD_LENGTH - 16))
   {
      us_Chunk = MAX_MAILBOX_PAYLOAD_LENGTH - 16;
   }
   else
   {
      us_Status |= DSH_LAST_FRAGMENT;
   }

   NMP_SendDebugMsg(pt_Entry->us_Id, pt_Entry->ul_SymCount, pt_Entry->us_Crc, us_Status, us_Chunk, pus_Data);

   if(us_Status & DSH_LAST_FRAGMENT)
   {
      // wait for last chunkt to be sent befor the entry is removed from the queue
      pt_Statics->ft_CleanUpLastStream = TRUE;
      pt_Entry->us_Status = us_Status;
   }
   else
   {
      // prepare next chunk
      us_Length -= us_Chunk;
      pus_Data  += us_Chunk>>1; // watch up!! 16 bit pointer!
      us_Status &= ~DSH_MASK_FRAGMENT;

      pt_Entry->us_Length = us_Length;
      pt_Entry->us_Status = us_Status;
      pt_Entry->p_Data  = pus_Data;
   }

   return TRUE;
}


static void  DSH_ClearBuffer(DSH_Statics_t * pt_Statics, DSH_QueueEntry_t * pEntry)
{
   uint32 ul_int_status;
   disable1_save(&ul_int_status);;
   if(pEntry->us_Status & DSH_STREAM_SIZE_IS_ODD)
   {
      pEntry->us_Size--;
   }
   pt_Statics->s_BufferFill -= pEntry->us_Size;
   restore1_save(ul_int_status);
}


static FlagT DSH_BgEvtBuf(DSH_Statics_t * pt_Statics)
{

   int32 l_WrIdx = pt_Statics->s_BufferWrIdx;
   int32 l_RdIdx = pt_Statics->s_BufferRdIdx;

   if(l_WrIdx < l_RdIdx)
   {
       uint16 BlockSize       = (uint16)(DSH_BUFFER_SIZE - l_RdIdx);
       DSH_WriteQueue(pt_Statics, DSH_STATE_TRAIL, BlockSize, FALSE, &pt_Statics->uca_Buffer[l_RdIdx], DSH_ClearBuffer);
       l_RdIdx                  = 0;
       pt_Statics->s_BufferRdIdx = 0;
   }

   uint16 BlockSize = (uint16)(l_WrIdx - l_RdIdx);
   if(pt_Statics->ft_BufferFlush || (BlockSize >= DSH_BUFFER_FILL_THRESHOLD))
   {
       pt_Statics->ft_BufferFlush = FALSE;
       if(BlockSize)
       {
           DSH_WriteQueue(pt_Statics, DSH_STATE_TRAIL, BlockSize, FALSE, &pt_Statics->uca_Buffer[l_RdIdx], DSH_ClearBuffer);
           pt_Statics->s_BufferRdIdx = (int16)(l_RdIdx + BlockSize);
       }
       return FALSE;
   }
   return FALSE;
}


void DSH_FgService( void )
{
   // Get local copy of the global symbol count from persistent memory
   // we need a counter which does not get reset with a new link start for tracee, and we want a counter which
   // gets incremented every symbol (including unused frame and in idle state)
   uint32 ul_SymCnt = gul_DSH_SymCount;

   //if Mailbox is available, try to stream version info
   if(TxMailBoxPending() == FALSE)
   {
      if(guc_DSH_VersionInfoStreamed < 1)
      {
         int32 StreamVersionInfoResult;
         StreamVersionInfoResult = DSH_SendStream(DSH_VERSION_INFO,10,&gusa_FW_version_number[0]); //XDSLRTFW-3364 (Start_End)
         if (StreamVersionInfoResult>0)
         {
            guc_DSH_VersionInfoStreamed++;
         }
      }
   }


   // send event to signal counter wrap around
   if(ul_SymCnt == 0xFFFFFFFF)
   {
      //DEBUG: Avoid new record being created by ul_SymCnt equal to 0. We see this being triggered unexcpectedly. Hence use a new Event for debugging
      //DSH_SendEvent(DSH_EVT_SYM_CNT_ZERO,0,0);
      DSH_SendEvent(DSH_EVT_CPE_SYM_ZERO_DEBUG,0,0);
      // queue events now, because all events/streams later than symCnt == 0
      // have to be received by Tracee after this event
      gt_DshStatics.ft_BufferFlush = TRUE;

      // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
      // 5 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
      if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-5))
      {
         AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);  //we need to call DSH_BgService (and can't call directly DSH_BgEvtBuf) because we can't pass arguments to BGfunctions
      }
      else
      {
         // we ran out of space, keep track of this event
         gus_Dbg_DSH_skip_BGF_add++;
      }
   }

   // Update the persistent symbol count (and also the local copy)
   ul_SymCnt++;
   gul_DSH_SymCount = ul_SymCnt;

   // XDSLRTFW-3280 - Start - PLL improvement / pilot tone selection improvement
   if((ul_SymCnt & 0x000003FF) == 0 && gft_EnablePilotStream)
   {
      // PT0
      gta_DebugPilot.s_PilotToneIdx0_Re = gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Re * gt_PilotConfig.ta_PilotTones[0].s_PilotToneScale;
      gta_DebugPilot.s_PilotToneIdx0_Im = gt_PilotConfig.ta_PilotTones[0].s_PilotTone_Im * gt_PilotConfig.ta_PilotTones[0].s_PilotToneScale;
      gta_DebugPilot.s_PllRefToneIdx0_Re = gt_PilotConfig.ta_PilotTones[0].s_PllRefTone_Re;
      gta_DebugPilot.s_PllRefToneIdx0_Im = gt_PilotConfig.ta_PilotTones[0].s_PllRefTone_Im;
      gta_DebugPilot.gs_PhaseErrorIdx0 = gt_PilotConfig.ta_PilotTones[0].gs_PhaseError;
      // PT1
      gta_DebugPilot.s_PilotToneIdx1_Re = gt_PilotConfig.ta_PilotTones[1].s_PilotTone_Re * gt_PilotConfig.ta_PilotTones[1].s_PilotToneScale;
      gta_DebugPilot.s_PilotToneIdx1_Im = gt_PilotConfig.ta_PilotTones[1].s_PilotTone_Im * gt_PilotConfig.ta_PilotTones[1].s_PilotToneScale;
      gta_DebugPilot.s_PllRefToneIdx1_Re = gt_PilotConfig.ta_PilotTones[1].s_PllRefTone_Re;
      gta_DebugPilot.s_PllRefToneIdx1_Im = gt_PilotConfig.ta_PilotTones[1].s_PllRefTone_Im;
      gta_DebugPilot.gs_PhaseErrorIdx1 = gt_PilotConfig.ta_PilotTones[1].gs_PhaseError;
      // PT2
      gta_DebugPilot.s_PilotToneIdx2_Re = gt_PilotConfig.ta_PilotTones[2].s_PilotTone_Re * gt_PilotConfig.ta_PilotTones[2].s_PilotToneScale;
      gta_DebugPilot.s_PilotToneIdx2_Im = gt_PilotConfig.ta_PilotTones[2].s_PilotTone_Im * gt_PilotConfig.ta_PilotTones[2].s_PilotToneScale;
      gta_DebugPilot.s_PllRefToneIdx2_Re = gt_PilotConfig.ta_PilotTones[2].s_PllRefTone_Re;
      gta_DebugPilot.s_PllRefToneIdx2_Im = gt_PilotConfig.ta_PilotTones[2].s_PllRefTone_Im;
      gta_DebugPilot.gs_PhaseErrorIdx2 = gt_PilotConfig.ta_PilotTones[2].gs_PhaseError;

      DSH_SendEvent(DSH_EVT_PILOT_TONES,sizeof(gta_DebugPilot),&gta_DebugPilot);
   }
   // XDSLRTFW-3280 - End - PLL improvement / pilot tone selection improvement

   // XDSLRTFW-2831
   // stream showtime SNR every ~2sec
   // Disabled, since API/MEI problems lead to API crash after using this stream for a longer period
   if((gs_RxState == R_O_SHOWTIME_RX) && ((gul_DSH_SymCount & 0x00001FFF) == 0))
   {
   #ifdef ILV_DBG_BUFFER
         if(gft_WriteToIlvDbgBuffer == FALSE)
   #endif
            DSH_SendStream(DSH_SHOWTIME_SNR,gs_RxNumTones*sizeof(int16),(void*)gpsa_MeasuredSnrBuf);
   }


#ifdef ILV_DBG_BUFFER

   if( (gs_IlvDbgBufferControl & ILV_DBG_BUF_CNTRL_ENABLE_MASK) == TRUE )
   {
      //@todo: I would make the InDirStatSelector global to avoid per symbol access
      int16 s_InDirStatSelector = ((gs_IlvDbgBufferControl&(ILV_DBG_BUF_CNTRL_STAT_SELECT_MASK<<ILV_DBG_BUF_CNTRL_STAT_SELECT_POS))>>ILV_DBG_BUF_CNTRL_STAT_SELECT_POS);
      if(gs_IlvDbgBufferStartTrigger == gsa_IndirectStat0[s_InDirStatSelector])
      {
         gft_WriteToIlvDbgBuffer = TRUE;
      }
      if(gs_IlvDbgBufferStopTrigger == gsa_IndirectStat0[s_InDirStatSelector])
      {
         gft_WriteToIlvDbgBuffer = FALSE;
      }

      //********************** ILV DBG Buffer ***********************************
      // if buffer exists and start trigger was hit
      if(gft_WriteToIlvDbgBuffer == TRUE)
      {
         gs_IlvDbgBufferDecimatorCnt++;

         // @todo: add decimator handling with modulo
         if ((gs_IlvDbgBufferDecimatorCnt % gs_IlvDbgBufferDecimator) == 0)
         {
            gs_IlvDbgBufferDecimatorCnt=0;
            if( (gs_IlvDgbBufferContentSelector & ILVDBG_PLL_ANYLYSIS) &&
                (gs_IlvDbgBufferWriteIdx <= (ILV_DBG_BUFFER_SIZE-gs_IlvDbgBufferBytesPerCycle)))
            {
               *gpsa_IlvDbgBuffer++ = (uint16)(ul_SymCnt&0xFFFF);
               *gpsa_IlvDbgBuffer++ = (uint16)((ul_SymCnt&(0xFFFF<<16))>>16);
               *gpsa_IlvDbgBuffer++ = gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].s_PilotTone_Re;
               *gpsa_IlvDbgBuffer++ = gt_PilotConfig.ta_PilotTones[gt_PilotConfig.te_UsedPTArrayIdx].s_PilotTone_Im;
               *gpsa_IlvDbgBuffer++ = gs_PhaseError;
               *gpsa_IlvDbgBuffer++ = (int16)(gl_pll_loopfilter_out&0xFFFF);
               *gpsa_IlvDbgBuffer++ = (int16)((gl_pll_loopfilter_out&(0xFFFF<<16))>>16);
               *gpsa_IlvDbgBuffer++ = (int16)(gl_pll_freq_offset&0xFFFF);
               *gpsa_IlvDbgBuffer++ = (int16)((gl_pll_freq_offset&(0xFFFF<<16))>>16);
               *gpsa_IlvDbgBuffer++ = (int16)(gul_metric_value&0xFFFF);
               *gpsa_IlvDbgBuffer++ = (int16)((gul_metric_value&(0xFFFF<<16))>>16);
               *gpsa_IlvDbgBuffer++ = (int16)(gt_ReTXStats.ul_UncorrectedDtuCnt&0xFFFF);
               *gpsa_IlvDbgBuffer++ = (int16)(gt_ReTXStats.ul_BadDtuCnt&0xFFFF);
               *gpsa_IlvDbgBuffer++ = (gft_frame_with_lowmetric<<8) + gft_EnablePLL;
               *gpsa_IlvDbgBuffer++ = gus_MicroInterruptionDetectionCnt;
               *gpsa_IlvDbgBuffer++ = gus_SymbolWithLowEdMetricCnt;
               *gpsa_IlvDbgBuffer++ = gus_lowmetric_discard_cnt;
               *gpsa_IlvDbgBuffer++ = (int16)(gul_num_frame_lowmetric&0xFFFF);
               gs_IlvDbgBufferWriteIdx+=gs_IlvDbgBufferBytesPerCycle;
            }

            // Divide Stream in Ping Pong Buffer
            if (gs_IlvDbgBufferWriteIdx >=  ((ILV_DBG_BUFFER_SIZE-gs_IlvDbgBytesPerCycleModSize)>>1))
            {
               gft_IlvDbgBufferPingPong = PONG;
               if(gs_IlvDbgBufferWriteIdx >=  (ILV_DBG_BUFFER_SIZE-gs_IlvDbgBytesPerCycleModSize))
               {
                  if(((gs_IlvDbgBufferControl&(ILV_DBG_BUF_CNTRL_MODE_MASK<<ILV_DBG_BUF_CNTRL_MODE_POS))>>ILV_DBG_BUF_CNTRL_MODE_POS) == TRUE)
                  {  // if circular mode
                     gs_IlvDbgBufferWriteIdx = 0;
                     gft_IlvDbgBufferPingPong = PING;
                  }
                  else
                  {  // if one shot mode
                     gft_WriteToIlvDbgBuffer = FALSE;
                  }
                  gpsa_IlvDbgBuffer = (int16 *)(void *)glp_IlvDbgBufferStart;
               }
            }

            // Stream Interleaver Buffer if we had a tranisition in write index from/to Ping/Pong buffer
            if(gft_IlvDbgBufferPingPong_Prev != gft_IlvDbgBufferPingPong)
            {
               uint8 *puca_tempBufferPointer = (void *)glp_IlvDbgBufferStart;
               if(gft_IlvDbgBufferPingPong == PONG)  // stream PING buffer when PONG is currently written
               {
                  DSH_SendStream(DSH_ILV_DBG_BUFFER_PLL,(ILV_DBG_BUFFER_SIZE-gs_IlvDbgBytesPerCycleModSize)>>1,(void *)puca_tempBufferPointer);
               }
               if(gft_IlvDbgBufferPingPong == PING) // stream PONG buffer when PING is currently written
               {
                  puca_tempBufferPointer+=(ILV_DBG_BUFFER_SIZE>>1);
                  // To have the pointer at the correct start we need to adjust it by the modulo bytes
                  puca_tempBufferPointer-=(gs_IlvDbgBytesPerCycleModSize);
                  DSH_SendStream(DSH_ILV_DBG_BUFFER_PLL,(ILV_DBG_BUFFER_SIZE-gs_IlvDbgBytesPerCycleModSize)>>1,(void *)puca_tempBufferPointer);
               }
               gft_IlvDbgBufferPingPong_Prev = gft_IlvDbgBufferPingPong;
            }
         }//Decimator
      }//gft_WriteToIlvDbgBuffer
   }
#endif //ILV_DBG_BUFFER
}


// DSH background handler, called by event system
void DSH_BgService( void )
{
   DSH_Statics_t *pt_Statics = &gt_DshStatics;

   FlagT ft_Res = FALSE;
   ft_Res |= DSH_BgEvtBuf(pt_Statics);
   ft_Res |= DSH_BgDbgStr(pt_Statics);

   if (ft_Res)
   {
      //In case Background tasks want to be executed again (e.g. because the mailbox was busy) queue BG Task in again
      gus_Dbg_RequeueCnt++;
      gus_RequeueBGFunction |= DSH_REQUEUE_DSH_BG_SERVICE;
   }
}

int32 DSH_SendEvent(uint8 uc_EventId, uint8 uc_BlockSize, void * p_Data)
{

   if(((gt_debugStreamControl.Parameter0 & CMV_INFO115_CONTROL_MASK) == 0)
   ||(gt_debugStreamConfigure.Mask1 & (1 << DSH_STATE_TRAIL)) == 0)
   {
      //debug streams or state trail stream have been deactivated, return
      return 0;
   }


   uint32 ul_int_status;
   DSH_Statics_t * pt_Statics = &gt_DshStatics;

   int32 l_Result      = 0;
   uint8 * puc_FillGap  = 0;
   uint8 * puc_FillData = 0;

   disable1_save(&ul_int_status);

   int32 l_GapSize = DSH_BUFFER_SIZE - pt_Statics->s_BufferWrIdx;
   if(l_GapSize > (int32)(uc_BlockSize + DSH_EVENT_HEADER_SIZE))
   {
      l_GapSize = 0;
   }

   int32 l_WrIdx = pt_Statics->s_BufferWrIdx;
   int32 l_Fill  = pt_Statics->s_BufferFill;
   if((int32)(l_Fill + uc_BlockSize + DSH_EVENT_HEADER_SIZE + l_GapSize) >= DSH_BUFFER_SIZE)
   {
      pt_Statics->ft_Overflow = TRUE;
      l_Result = -1;
      goto WriteToBuffer_End;
   }

   if(l_GapSize)
   {
      puc_FillGap = &pt_Statics->uca_Buffer[l_WrIdx];
      l_WrIdx = 0;
   }

   puc_FillData = &pt_Statics->uca_Buffer[l_WrIdx];

   l_WrIdx += uc_BlockSize + DSH_EVENT_HEADER_SIZE;
   l_Fill  += uc_BlockSize + DSH_EVENT_HEADER_SIZE + l_GapSize;

   if(l_WrIdx >= DSH_BUFFER_SIZE)
   {
      l_WrIdx = 0;
   }

   pt_Statics->s_BufferWrIdx = (int16)l_WrIdx;
   pt_Statics->s_BufferFill  = (int16)l_Fill;

WriteToBuffer_End:
   restore1_save(ul_int_status);
   if(puc_FillGap)
   {
      memset(puc_FillGap, -1, l_GapSize);
   }

   if(puc_FillData)
   {
      if (pt_Statics->ft_Overflow)
      {
         pt_Statics->ft_Overflow = FALSE;
         uc_EventId |= 0x80;
      }

      uint32 ul_SymCount = gul_DSH_SymCount;
      *puc_FillData++ = uc_EventId;
      *puc_FillData++ = (uint8)(ul_SymCount & 0xFF);
      *puc_FillData++ = (uint8)((ul_SymCount >>= 8) & 0xFF);
      *puc_FillData++ = (uint8)((ul_SymCount >>= 8) & 0xFF);
      *puc_FillData++ = (uint8)((ul_SymCount >>= 8) & 0xFF);
      *puc_FillData++ = uc_BlockSize;
      memcpy(puc_FillData, p_Data, uc_BlockSize);
   }


   // XDSLRTFW-2701 Make sure we leave sufficient space in BGTaskFifo and do not flood it with debug stream tasks to avoid E_CODE_BKGD_FIFO_ADD_EXCEPTION
   // 5 was the BGTaskFifio size before debug streams were implemented, so make sure this space is always reserved
   if (gp_BGTaskFifo->PendingFunctionCount < (NUM_BG_BUFFERS-5))
   {
      AddFunctionToBkgdFifo((PtrToBkgdFunc)DSH_BgService);
   }
   else
   {
      // we ran out of space, keep track of this event
      gus_Dbg_DSH_skip_BGF_add++;
   }
   return l_Result;
}


int32 DSH_SendStream(uint16 us_StreamID, uint16 us_BlockSize, void * p_Data)
{
   DSH_Statics_t * pt_Statics = &gt_DshStatics;
   uint16 us_MaskIdx, us_MaskBit;

   const uint32 * pHostConfigurationMask = &gt_debugStreamConfigure.Mask1;

   if((gt_debugStreamControl.Parameter0 & CMV_INFO115_CONTROL_MASK) == 0)
   {
      // return if debug streams are disabled globally
      return 0;
   }

   // compute the offset into Debug Stream Config Mask
   us_MaskIdx = (uint16)(us_StreamID / (sizeof(DSH_MaskEntry_t) << 3));
   us_MaskBit = (uint16)(us_StreamID % (sizeof(DSH_MaskEntry_t) << 3));

   // check if the ID is within the allowed limit and if the particular debug stream is configured via message
   // OR if the ID is equal to DSH_VERSION_INFO, which is always sent, regardless of the configuration
   // OR if the ID is equal the special IPTC Debug Stream forwarding us_Id
   if (((us_StreamID < DSH_CTRL_MAX_NUM) && ((pHostConfigurationMask[us_MaskIdx] & (1 << us_MaskBit)) != 0)) || (us_StreamID == DSH_VERSION_INFO))
   {
      if(us_StreamID == DSH_FSM_STATE)
      {
         //As the FSM state is used to put all other events into a timely context it
         //is important to make sure all that all events before the change have been sent.
         pt_Statics->ft_BufferFlush = TRUE;
         DSH_BgEvtBuf(pt_Statics);
      }
      //Note: Disabled CRC calculation for debug streams due to MIPS overrun on large Debug Stream messages
      //return DSH_WriteQueue(pt_Statics, us_StreamID, us_BlockSize, TRUE, p_Data, 0);
      return DSH_WriteQueue(pt_Statics, us_StreamID, us_BlockSize, FALSE, p_Data, 0);
   }

   return 0;
}


