/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   Name:   HDLC.c
*
*   VDSL HDLC messages handling code.
*
*-------------------------------------------------------------------------
*/

#include <stdio.h>
#include <string.h>

#include "common.h"
#include "dsp_op.h"
#include "fcs.h"
#include "hdlc.h"
#include "gdata.h"
#include "ModulateSocMessages.h"
#ifdef DEBUG_TRAIL
#include "LeaveStatesTrail.h"
#endif //DEBUG_TRAIL
#include "PrintTrail.h"
#include "vdsl_xception.h"
#include "cmv.h"

extern int16 gs_dbgLogSocMsg;
extern int16 gs_dbgLogSocMsgPause;

/*
*-------------------------------------------------------------------------
*
*   TX HDLC routines.
*
*-------------------------------------------------------------------------
*/

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : GetNextHDLCTxOctet
 *
 *  Description:  Generates one TX HDLC message octet on the fly
 *
 *  Prototype:
 *           uint8 GetNextHDLCTxOctet(uint8 *puc_HDLCPayload)
 *
 *
 *  Input Arguments:
 *      puc_HDLCPayload -- pointer to the array containing the message payload bytes
 *
 *  Output Arguments:
 *      None
 *
 *  Global Variables Used:
 *      gs_TxHDLCMsgState -- (I/O) the substate indicator
 *      uc_octet2 -- (I/O) the next byte to send (in case that HDLC transparent stuff is used)
 *      gs_IdleFlagCount -- (I/O) no. of the HDLC idle flags sent between two adjacent message
 *      gs_NumOctetsInTxHDLCMsg -- (I) no. of octests in TX HDLC message
 *      gs_UpdateTxMsgIdxFlag -- (I) Flag indicating if the message index should be increased or not
 *      guc_TxHDLCMsgIndex -- (I/O) TX HDLC message index
 *      gus_TxHDLCMsgFCS -- (I/O) the TX HDLC message FCS check bytes
 *      uc_NumSegments -- (I/O) the number of segment in this message
 *      uc_CurSegment -- (I/O) the current segment number
 *      s_CurOctetIndex -- (I/O) the current octet index
 *      gft_TxHDLCSegmentedAutoRepeatMode -- (I) Segmentation flag in AR mode
 *  Notes:
 *      How to use Segmentation in AR mode :
 *      - If generated SOC data in AR mode is greater than 1024 then set gft_TxHDLCSegmentedAutoRepeatMode = 1 else
 *        make it 0, this flag will make sure that Segmented SOC transmission will not wait for Ack for next segment
 *        transmission in AR mode.
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
#define MIN_NUM_HDLC_FLAGS   (5)      //minimum number of HDLC flags sent in between the message

// This function is typically called in the foreground.
int16 gs_IdleFlagCount = 0;
uint8 GetNextHDLCTxOctet(uint8 *puc_HDLCPayload)
{
   static uint8 uc_octet, uc_octet2 = 0;
   static uint8 uc_NumSegments, uc_CurSegment;
   static int16 s_NumOctetsInCurSegment, s_CurOctetIndex;
   static uint8 *puc_PayLoadBuf;
   static uint8 uc_RepeatReqMsgFlag = FALSE;

   if (uc_octet2)   // Transmit octet2 if prev octet required octet transparency
   {
      uc_octet  = uc_octet2;
      uc_octet2 = 0;
#ifdef DEBUG_TRAIL
      //Log all non-FLAG Tx SOC bytes (Pause variable pauses logging during AutoRepeat messages
      if ((gs_dbgLogSocMsg < (STATES_TRAIL_LENGTH-3)) && (uc_octet != 0x7E) && (gs_dbgLogSocMsgPause==0))
      {
         gsa_StatesTrail[gs_dbgLogSocMsg++] = (int16)uc_octet;
      }
#endif
      return uc_octet;
   }

   switch (gs_TxHDLCMsgState)
   {

   case HDLC_TX_START_NEW_MSG:

      //Make sure at least 4 HDLC flags are sent between the messages
      if(gs_IdleFlagCount >= MIN_NUM_HDLC_FLAGS)
      {
         //Indicate no ACK-SEG is received
         gs_RevAckSegFlag = RECEIVE_ACK_SEG_IDLE;

         //Check if this message is repeat request message
         if(*puc_HDLCPayload == SOC_MSG_REPEAT_REQUEST)
         {
            uc_RepeatReqMsgFlag = TRUE;
         }
         else
         {
            uc_RepeatReqMsgFlag = FALSE;
         }

         // Each segment has at most 1024/2048(Profile 35B) payload octets
         {
            uint16 us_MaxSocMsgSegSize;

            us_MaxSocMsgSegSize = SOC_MAX_MSG_SEGMENT_SIZE;
            if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
            {
               us_MaxSocMsgSegSize = SOC_MAX_MSG_SEGMENT_35B_SIZE;
            }

            uc_NumSegments = 1;
            while(gs_NumOctetsInTxHDLCMsg > (uc_NumSegments * us_MaxSocMsgSegSize))
            {
               uc_NumSegments++;
            }
         }

         if (uc_NumSegments > SOC_MAX_NUM_MSG_SEGMENTS)
         {
            //Enter fail state
            EnterFailStates(E_CODE_TX_HDLC_MSG_SEGMENTS);
         }

         uc_CurSegment = 1;     // range: 1 to 15 (SOC_MAX_NUM_MSG_SEGMENTS)
         puc_PayLoadBuf = puc_HDLCPayload;

         gs_TxHDLCMsgState = HDLC_TX_MSG_INDEX;
#ifdef DEBUG_TRAIL
         // log SOC info into gsa_StatesTrail
         LeaveTxSOCTrail(gpuca_TxSocMsg[0]);
#endif //DEBUG_TRAIL

      }

      // starting flag, return immediately since no octet transparency is needed.
      gs_IdleFlagCount++;
      return((uint8)HDLC_FLAG);

   case HDLC_TX_MSG_INDEX:

      //Update the message index
      if((gs_UpdateTxMsgIdxFlag == TRUE) && (uc_CurSegment == 1))
      {
         if(guc_TxHDLCMsgIndex == 255)
         {
            guc_TxHDLCMsgIndex = 1;
         }
         else
         {
            guc_TxHDLCMsgIndex++;
         }

         gs_UpdateTxMsgIdxFlag = FALSE;
      }

      if(uc_RepeatReqMsgFlag == FALSE)
      {
         uc_octet = guc_TxHDLCMsgIndex;
      }
      else
         //message index shall be 0 for repeat request message
      {
         uc_octet = 0;
      }

      /* Compute Frame Check Sequence (FCS) */
      gus_TxHDLCMsgFCS = 0xFFFF;
      gus_TxHDLCMsgFCS = calcCrc16(gus_TxHDLCMsgFCS, uc_octet);

      gs_TxHDLCMsgState = HDLC_TX_SEGM_INDEX;

      //reset IDLE flag count to 0
      gs_IdleFlagCount = 0;

      break;

   case HDLC_TX_SEGM_INDEX:

      {
         uint16 us_MaxSocMsgSegSize;

         us_MaxSocMsgSegSize = SOC_MAX_MSG_SEGMENT_SIZE;
         if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
         {
            us_MaxSocMsgSegSize = SOC_MAX_MSG_SEGMENT_35B_SIZE;
         }

         if (uc_CurSegment < uc_NumSegments)
         {
            s_NumOctetsInCurSegment = us_MaxSocMsgSegSize;
         }
         else
         {
            s_NumOctetsInCurSegment = (gs_NumOctetsInTxHDLCMsg - (us_MaxSocMsgSegSize*(uc_NumSegments-1)));
         }
      }

      if (uc_RepeatReqMsgFlag == FALSE)
      {
         uc_octet = (uc_NumSegments << 4) + uc_CurSegment;
      }
      else
      {
         uc_octet = (uc_NumSegments << 4);   //segment index shall be 0 for repeat request message
      }

      /* Compute Frame Check Sequence (FCS) */
      gus_TxHDLCMsgFCS = calcCrc16(gus_TxHDLCMsgFCS, uc_octet);

      guc_TxSegmentIdx = uc_CurSegment;

      s_CurOctetIndex   = 0;
      gs_TxHDLCMsgState = HDLC_TX_PAYLOAD;
      break;

   case HDLC_TX_PAYLOAD:

      uc_octet = *puc_PayLoadBuf++;
      s_CurOctetIndex++;
      if (s_CurOctetIndex == s_NumOctetsInCurSegment)
      {
         s_CurOctetIndex = 0;
         uc_CurSegment++;

         gs_TxHDLCMsgState = HDLC_TX_FCS_OCTET1;

      }
      /* Compute Frame Check Sequence (FCS) */
      gus_TxHDLCMsgFCS = calcCrc16(gus_TxHDLCMsgFCS, uc_octet);

      break;

   case HDLC_TX_FCS_OCTET1:

      if (!(gul_dbgSocMsgControl & TX_SKIP_HDLC_FCS_INVERSION))
      {
         gus_TxHDLCMsgFCS = ~gus_TxHDLCMsgFCS;   /*  take 1's compliment */
      }

      if (gul_dbgSocMsgControl & TX_SWITCH_HDLC_FCS_BYTEORDER)
      {
         uc_octet = (uint8)(gus_TxHDLCMsgFCS >> 8);   /* Send MSB first */
      }
      else
      {
         uc_octet = (uint8)(gus_TxHDLCMsgFCS);   /* Send LSB first */
      }

      if (gul_dbgSocMsgControl & TX_BIT_REVERSE_HDLC_FCS)  /*  Bit reverse FCS */
      {
         uc_octet = (uint8)BitRev((uint32)uc_octet, 8);
      }

      gs_TxHDLCMsgState = HDLC_TX_FCS_OCTET2;

      break;

   case HDLC_TX_FCS_OCTET2:

      if (gul_dbgSocMsgControl & TX_SWITCH_HDLC_FCS_BYTEORDER)
      {
         uc_octet = (uint8)(gus_TxHDLCMsgFCS);   /* Send LSB second */
      }
      else
      {
         uc_octet = (uint8)(gus_TxHDLCMsgFCS >> 8);   /* Send MSB second */
      }

      if (gul_dbgSocMsgControl & TX_BIT_REVERSE_HDLC_FCS)  /*  Bit reverse FCS */
      {
         uc_octet = (uint8)BitRev((uint32)uc_octet, 8);
      }

      gs_TxHDLCMsgState = HDLC_TX_CLOSING_FLAG;
      break;

   case HDLC_TX_CLOSING_FLAG:

      if (uc_CurSegment <= uc_NumSegments)
      {
         //go to the state to wait for ACK-SEG message
         gs_TxHDLCMsgState = HDLC_TX_WAIT_ACK_SEG;
         gs_RevAckSegFlag = RECEIVE_ACK_SEG_EXP;
      }
      else
      {
         if(gs_IdleFlagCount >= MIN_NUM_HDLC_FLAGS)
         {
            gs_TxHDLCMsgState = HDLC_TX_CLOSING_FLAG_DONE;
         }
      }

      gs_IdleFlagCount++;
      return(HDLC_FLAG);   // return immediately since no octet transparency is needed.

   case HDLC_TX_CLOSING_FLAG_DONE:

      gs_IdleFlagCount++;
      return(HDLC_FLAG);   // return immediately since no octet transparency is needed.

   case HDLC_TX_WAIT_ACK_SEG:

      if ((gs_IdleFlagCount >= MIN_NUM_HDLC_FLAGS) && ((gft_TxHDLCSegmentedAutoRepeatMode == 1) || (gs_RevAckSegFlag == RECEIVE_ACK_SEG_DONE)))
      {
         //reset the flag to idle
         gs_RevAckSegFlag = RECEIVE_ACK_SEG_IDLE;

         //Increase msg index for each new segment
         gs_UpdateTxMsgIdxFlag = TRUE;

         // go to next segment
         gs_TxHDLCMsgState = HDLC_TX_MSG_INDEX;
      }

      gs_IdleFlagCount++;
      return(HDLC_FLAG);   // return immediately since no octet transparency is needed.
   }

   // Gen escape octet if octet transparency is needed:
   if ((uc_octet == 0x7D) || (uc_octet == 0x7E))
   {
      uc_octet2 = uc_octet^0x20;
      uc_octet  = 0x7D;
   }
   else
   {
      uc_octet2 = 0;
   }
#ifdef DEBUG_TRAIL
   //Log all non-FLAG Tx SOC bytes (Pause variable pauses logging during AutoRepeat messages
   if ((gs_dbgLogSocMsg < (STATES_TRAIL_LENGTH-3)) && (uc_octet != 0x7E) && (gs_dbgLogSocMsgPause==0))
   {
      gsa_StatesTrail[gs_dbgLogSocMsg++] = (int16)uc_octet;
   }
#endif
   return uc_octet;
}


/*
*-------------------------------------------------------------------------
*
*   RX HDLC routines.
*
*-------------------------------------------------------------------------
*/

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : InitHDLCFifo
 *
 *  Description:  Initialize the RX HDLC FIFO structure
 *
 *  Prototype:
 *      void InitHDLCFifo(uint8 *puca_OctetBuffer, HDLC_Fifo_attrib_t *pt_FIFO)
 *
 *
 *  Input Arguments:
 *      puca_OctetBuffer -- pointer to the output octet array
 *
 *  Output Arguments:
 *      pt_FIFO -- pointer to the RX HDLC FIFO structure
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void InitHDLCFifo(uint8 *puca_OctetBuffer, HDLC_Fifo_attrib_t *pt_FIFO)
{
   pt_FIFO->puca_OctetBuffer = puca_OctetBuffer;
   pt_FIFO->s_CurFIFODepth = 0;
   pt_FIFO->s_MaxFIFODepth = MAX_MSG_LEN;
   pt_FIFO->s_PreFIFODepth = 0;
   pt_FIFO->uc_PreSegmentIdx = 0;
   pt_FIFO->s_StartFlag = 0;

   for(pt_FIFO->uc_NumOfSegments = 0; pt_FIFO->uc_NumOfSegments < SOC_MAX_NUM_MSG_SEGMENTS; pt_FIFO->uc_NumOfSegments++)
   {
      pt_FIFO->sa_SegmentLength[pt_FIFO->uc_NumOfSegments] = 0;
   }
   pt_FIFO->uc_NumOfSegments = 0;

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ReceiveHDLCOctet
 *
 *  Description:  This function receives one HDLC octet, and performs the message
 *  delineation, removes the HDLC transparent stuff octet.
 *
 *  Prototype:
 *      void ReceiveHDLCOctet(uint8 uc_octet, uint8 *puca_OctetBuffer, HDLC_Fifo_attrib_t *pt_FIFO)
 *
 *
 *  Input Arguments:
 *      uc_octet -- the current received octet
 *      puca_OctetBuffer -- pointer to the output octet array
 *
 *  Output Arguments:
 *      pt_FIFO -- pointer to the RX HDLC FIFO structure
 *
 *  Global Variables Used:
 *      uc_prev_octet -- (I/O) the previous received octet
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void ReceiveHDLCOctet(uint8 uc_octet, uint8 *puca_OctetBuffer, HDLC_Fifo_attrib_t *pt_FIFO)
{
   static uint8 uc_prev_octet = 0;

   if(uc_octet != HDLC_FLAG)
   {
      // Non-flag octet found.
#ifdef DEBUG_TRAIL
      //Log all non-FLAG Tx SOC bytes (Pause variable pauses logging during AutoRepeat messages
      if (gs_dbgLogSocMsg < STATES_TRAIL_LENGTH-3)
      {
         gsa_StatesTrail[gs_dbgLogSocMsg++] = (int16)uc_octet | 0x1000;
      }
#endif
      //Process only if the openning flag is found
      if(pt_FIFO->s_StartFlag == 1)
      {
         //If the buffer is full, discard this message by resetting FIFO
         if (pt_FIFO->s_CurFIFODepth >= pt_FIFO->s_MaxFIFODepth)
         {
            InitHDLCFifo(puca_OctetBuffer, pt_FIFO);
         }

         if (uc_prev_octet == 0x7D)
         {
            // octet transparency
            if ((uc_octet == 0x5D) || (uc_octet == 0x5E)) // Valid combos
            {
               pt_FIFO->puca_OctetBuffer[pt_FIFO->s_CurFIFODepth++] = (uint8)(uc_octet^0x20);
            }
            else  // Invalid combo.
            {
               //discard this message by resetting FIFO
               InitHDLCFifo(puca_OctetBuffer, pt_FIFO);
            }
         }
         else if(uc_octet != 0x7D)
         {
            pt_FIFO->puca_OctetBuffer[pt_FIFO->s_CurFIFODepth++] = uc_octet;
         }

      } //if(pt_FIFO->s_StartFlag == 1)

   }
   //HDLC flag is found
   else
   {
      //Starting flag is found
      if(pt_FIFO->s_PreFIFODepth == pt_FIFO->s_CurFIFODepth)
      {
         pt_FIFO->s_StartFlag = 1;
      }

      //where 4 counts 1 Address + 1 Control and 2 FCS bytes
      else if(pt_FIFO->s_StartFlag == 1)
      {
         //Compute the current segment length
         pt_FIFO->sa_SegmentLength[pt_FIFO->uc_NumOfSegments] =
            pt_FIFO->s_CurFIFODepth - pt_FIFO->s_PreFIFODepth;

         {
            uint16 us_MaxHdlcMsgLen;

            us_MaxHdlcMsgLen = HDLC_MAX_MSG_LEN;
            if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
            {
               us_MaxHdlcMsgLen = HDLC_MAX_MSG_LEN_35B;
            }
            // The message should be discarded, if it is
            //   - less than 5 bytes (too short) or
            //   - greater than the max segment length.
            //
            if(((pt_FIFO->uc_NumOfSegments == 0) &&
                (pt_FIFO->sa_SegmentLength[pt_FIFO->uc_NumOfSegments] < HDLC_MIN_MSG_LENGTH)) ||
               (pt_FIFO->sa_SegmentLength[pt_FIFO->uc_NumOfSegments] > us_MaxHdlcMsgLen))
            {
               InitHDLCFifo(puca_OctetBuffer, pt_FIFO);

               // Keep a count of rogue octets (incomplete hdlc message)
               gs_dbgRogueHdlcOctets++;
            }
            else
            {
               //Store the current segment length
               pt_FIFO->uc_NumOfSegments++;
               pt_FIFO->s_PreFIFODepth = pt_FIFO->s_CurFIFODepth;
               pt_FIFO->s_StartFlag = 0;

               // Reset count of rogue octets (incomplete hdlc message) on reception of a valid message
               gs_dbgRogueHdlcOctets = 0;
            }
         }
      } //else if(pt_FIFO->s_StartFlag == 1)

   } //if(uc_octet != HDLC_FLAG) else

   uc_prev_octet = uc_octet;
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CheckRxHDLCMsgFCS
 *
 *  Description:  This function computes FCS check bytes over the received message
 *   and compares them with the received ones to decide if this frame contains erorr or not.
 *
 *  Prototype:
 *      int16 CheckRxHDLCMsgFCS(HDLC_Fifo_attrib_t *pt_FIFO)
 *
 *
 *  Input Arguments:
 *      pt_FIFO -- pointer to the RX HDLC FIFO structure
 *
 *  Output Arguments:
 *
 *   Return:
 *      TRUE -- FCS match, FALSE otherwise
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
// This function is typically called in the background.
int16 CheckRxHDLCMsgFCS(HDLC_Fifo_attrib_t *pt_FIFO)
{
   uint8  uc_octet;
   uint16 us_RxComputedFCS;
   int16  i, j, k1, k2, s_len, s_Start;

   //j = current segment index
   j = pt_FIFO->uc_NumOfSegments-1;

   //s_len = segment length
   s_len = pt_FIFO->sa_SegmentLength[j];

   //s_Start = the index of the first byte of this segment
   s_Start = pt_FIFO->s_CurFIFODepth - s_len;

   us_RxComputedFCS = 0xFFFF;

   for(i=s_Start; i< s_Start+s_len-2; i++)
   {
      uc_octet = pt_FIFO->puca_OctetBuffer[i];

      /* Compute FCS only after the escape octets have been removed */
      us_RxComputedFCS = calcCrc16(us_RxComputedFCS, uc_octet);
   }

   // Process the 2 FCS bytes separately:
   if (gul_dbgSocMsgControl & RX_SWITCH_HDLC_FCS_BYTEORDER)
   {
      // switched order of 2 FCS bytes
      k1 = 1;
      k2 = 2;
   }
   else
   {
      k1 = 2;
      k2 = 1;
   }

   uc_octet = pt_FIFO->puca_OctetBuffer[s_Start+s_len-k1];
   if (gul_dbgSocMsgControl & RX_INVERT_HDLC_FCS)
   {
      uc_octet = ~uc_octet;
   }
   if (gul_dbgSocMsgControl & RX_BIT_REVERSE_HDLC_FCS)
   {
      uc_octet = (uint8)BitRev((uint32)uc_octet, 8);
   }
   us_RxComputedFCS = calcCrc16(us_RxComputedFCS, uc_octet);

   uc_octet = pt_FIFO->puca_OctetBuffer[s_Start+s_len-k2];
   if (gul_dbgSocMsgControl & RX_INVERT_HDLC_FCS)
   {
      uc_octet = ~uc_octet;
   }
   if (gul_dbgSocMsgControl & RX_BIT_REVERSE_HDLC_FCS)
   {
      uc_octet = (uint8)BitRev((uint32)uc_octet, 8);
   }
   us_RxComputedFCS = calcCrc16(us_RxComputedFCS, uc_octet);

   if(us_RxComputedFCS != MAGIC_FCS_CHECK)
   {
      return(FALSE);
   }

   return(TRUE);
}

//The background function to call CheckRxHDLCMsgFCS()
void BkgdCheckRxHDLCMsgFCS(void)
{
   gs_RxMsgCheckOkFlag = CheckRxHDLCMsgFCS(&RxHDLCFifo);

   gs_RxBkgdProcessFlag = MSG_DECODE_DONE;
}


