/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/**********************************************************************
 *
 * Aware DMT Technology. Proprietary and Confidential.
 *
 * ADDRESS:          40 Middlesex Turnpike, Bedford, MA 01730-1432 USA
 * TELEPHONE:        (781) 276-4000
 * FAX:              (781) 276-4001
 * WEB:              http://www.aware.com
 *
 * FILE:             ModulateSocMessages.c
 * DESCRIPTION:      Encodes and decodes SOC Messages.
 *
 **********************************************************************/
#include "sys_const.h"
#include "gdata.h"
#include "iri_iof.h"
#include "ghs.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : DecodeSOCMessage
 *
 *  Description:  This function decodes the message octet(s) from the received
 *   tones after descrambling.
 *
 *  Prototype:
 *      void DecodeSOCMessage(int16 *psa_RxToneBuf, int16 s_NumClusters, int16 s_NumBytesInMsg, uint8 *puca_DecodedBytes)
 *
 *  Input Arguments:
 *      psa_RxToneBuf -- pointer to the buffer containing the descrambled RX tones
 *      s_NumClusters -- number of tone clusters to be used for decoding
 *      s_NumBytesInMsg -- number of bytes (1 or 2) modulated to each tone cluster
 *
 *  Output Arguments:
 *      puca_DecodedBytes -- pointer to the output octet buffer
 *
 *  Return:
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void DecodeSOCMessage(int16 *psa_RxToneBuf, int16 s_NumClusters, int16 s_NumBytesInMsg, uint8 *puca_DecodedBytes)
{
   int16 i, j, i2;
   int32 l_X, l_Y, l_DecodedBits;
   FlagT ft_DbgFlipRx01Sense;

   ft_DbgFlipRx01Sense = (gul_dbgSocMsgControl & RX_FLIP_01_SENSE)? TRUE: FALSE;

   //Decode each tone to the corresponding bits (do from tone 9 to 0)
   l_DecodedBits = 0;
   for(i=NUM_CHANNELS_PER_CLUSTER-1; i>=0; i--)
   {
      //Average over s_NumClusters clusters to reduce noise effect
      l_X = 0;
      l_Y = 0;
      for(j=0; j<s_NumClusters; j++)
      {
         i2 = (gsa_SOCMsgRxToneClusterStarts[j] + i)*2;

         l_X += psa_RxToneBuf[i2];
         l_Y += psa_RxToneBuf[i2+1];
      }

      //Decide the bit map
      l_DecodedBits <<= 2;
      if(l_X < 0)
      {
         l_DecodedBits |= (1<<(1-ft_DbgFlipRx01Sense));
      }

      if(l_Y < 0)
      {
         l_DecodedBits |= (1<<ft_DbgFlipRx01Sense);
      }
   }

   //Pack one byte by extracting bits (b15, b14, b11,b10, b7,b6, b3,b2), where b0 is LSB
   if(s_NumBytesInMsg == 1)
   {
      puca_DecodedBytes[0] = (l_DecodedBits>>2) & 3;
      puca_DecodedBytes[0] |=(l_DecodedBits>>(6-2)) & (3<<2);
      puca_DecodedBytes[0] |=(l_DecodedBits>>(10-4)) & (3<<4);
      puca_DecodedBytes[0] |=(l_DecodedBits>>(14-6)) & (3<<6);
   }
   else
   {
      //Pack the first byte by extracting bits (b9 b8 b7 b6 b5 b4 b3 b2)
      puca_DecodedBytes[0] = (uint8)((l_DecodedBits>>2)&0x00FF);

      //Pack the second byte by extracting bits (b19 b18 b17 b16 b15 b14 b13 b12)
      puca_DecodedBytes[1] = (uint8)((l_DecodedBits>>12)&0x00FF);
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ModulateHdlcOctets
 *
 *  Description:  This function generates the lookup data for generating
 *  message based TX tones.
 *
 *  Prototype:
 *      void ModulateHdlcOctets(uint8 uca_MsgBytes[], int16 s_NumBytesInMsg, int16 s_Gain)
 *
 *  Input Arguments:
 *      uca_MsgBytes -- pointer to the input message byte(s)
 *      s_NumBytesInMsg -- number of bytes modulated to each symbol
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      guca_TxOctet -- (I/O) array to store tx bytes for debug
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void ModulateHdlcOctets(uint8 uca_MsgBytes[], int16 s_NumBytesInMsg)
{

   uint32 ul_data_extended[12] = { 0,0,0,0,0,0,0,0,0,0,0,0 };
   int16 j;
   uint32 u_x;
   int16  i;
   uint8  uc_x2, uca_DbgTx01SenseLookUp[4];

   for (i=0; i<4; i++)
   {
      uca_DbgTx01SenseLookUp[i] = (uint8)i;
   }

   if (gul_dbgSocMsgControl & TX_FLIP_01_SENSE)
   {
      uca_DbgTx01SenseLookUp[1] = 2;
      uca_DbgTx01SenseLookUp[2] = 1;
   }

   //Handle the SOC message transmission differently in loop diagnostic mode
   if((gul_OperationModeStatus_VDSL2 & V2_LOOP_DIAG) && (gft_DisableTxDiagModulation == FALSE))
   {
      //Compute the number of bits to send
      gs_NumOfTxSocBits = 8;
      gus_TxSocMsgBitBuf = uca_MsgBytes[0];
      gs_TxSocMsgBitPosition = 0;
      gs_NumOfTxSocBitRepeat = 0;

      return;
   }

   switch (s_NumBytesInMsg)
   {

   case 1: //spread out bits so 00 are on even tones
      // octet [abcdefgh] ->[00 00 ab 00 cd 00 ef 00 gh 00]  where gh is lsb on right

      u_x = (uint32)uca_MsgBytes[0];
      for (i= 2 ; i<= 14 ; i+=4)    //for i=(2,6,10,14)
      {
         uc_x2 = (uint8)(u_x&3);
         uc_x2 = uca_DbgTx01SenseLookUp[uc_x2];
         ul_data_extended[0] |=  uc_x2 << (i);
         u_x=u_x>>2;
      }

      // update debug guca_TxOctet array
      guca_TxOctet[0]=uca_MsgBytes[0];
      guca_TxOctet[1]=0x00;

      break;

   default:
      // covering cases 2,4,6,8,10,12,14,16,18,20,22,24
      for(j = 0; j<(s_NumBytesInMsg>>1); j++)
      {
         u_x = (uint32)uca_MsgBytes[(j<<1)];
         for (i= 2 ; i <= 8 ; i+=2)     // for i=(2,4,6,8)
         {
            uc_x2 = (uint8)(u_x&3);
            uc_x2 = uca_DbgTx01SenseLookUp[uc_x2];
            ul_data_extended[j] |=  uc_x2 << (i);
            u_x=u_x>>2;
         }
         u_x = (uint32)uca_MsgBytes[(j<<1)+1];
         for (i= 12 ; i <= 18 ; i+=2)     // for i=(12,14,16,18)
         {
            uc_x2 = (uint8)(u_x&3);
            uc_x2 = uca_DbgTx01SenseLookUp[uc_x2];
            ul_data_extended[j] |=  uc_x2 << (i);
            u_x=u_x>>2;
         }
      }

      // update debug guca_TxOctet array
      guca_TxOctet[0]=uca_MsgBytes[0];
      guca_TxOctet[1]=uca_MsgBytes[1];

      break;
   }

   // now format the max 12 long words into max 8 PRBS_DATA regs before loading
   if(s_NumBytesInMsg > 2)
   {
      gl_SocTxDataForPrbs          = ul_data_extended[0];           // 0000 0000 0000 xxxx xxxx 00xx xxxx xx00
      gl_SocTxDataForPrbs         |= ul_data_extended[1] << 20;     // 00xx xxxx xx00

      gl_SocTxDataForPrbs_ext[0]   = ul_data_extended[1] >> 12;     // 0000 0000 0000 0000 0000 0000 xxxx xxxx
      gl_SocTxDataForPrbs_ext[0]  |= ul_data_extended[2] <<  8;     // 0000 xxxx xxxx 00xx xxxx xx00
      gl_SocTxDataForPrbs_ext[0]  |= ul_data_extended[3] << 28;     // xx00

      gl_SocTxDataForPrbs_ext[1]   = ul_data_extended[3] >>  4;     // 0000 0000 0000 0000 xxxx xxxx 00xx xxxx
      gl_SocTxDataForPrbs_ext[1]  |= ul_data_extended[4] << 16;     // xxxx 00xx xxxx xx00

      gl_SocTxDataForPrbs_ext[2]   = ul_data_extended[4] >> 16;     // 0000 0000 0000 0000 0000 0000 0000 xxxx
      gl_SocTxDataForPrbs_ext[2]  |= ul_data_extended[5] <<  4;     // 0000 0000 xxxx xxxx 00xx xxxx xx00
      gl_SocTxDataForPrbs_ext[2]  |= ul_data_extended[6] << 24;     // xxxx xx00

      gl_SocTxDataForPrbs_ext[3]   = ul_data_extended[6] >>  8;     // 0000 0000 0000 0000 0000 xxxx xxxx 00xx
      gl_SocTxDataForPrbs_ext[3]  |= ul_data_extended[7] << 12;     // xxxx xxxx 00xx xxxx xx00

      gl_SocTxDataForPrbs_ext[4]   = ul_data_extended[8];           // 0000 0000 0000 xxxx xxxx 00xx xxxx xx00
      gl_SocTxDataForPrbs_ext[4]  |= ul_data_extended[9] << 20;     // 00xx xxxx xx00

      gl_SocTxDataForPrbs_ext[5]   = ul_data_extended[9] >> 12;     // 0000 0000 0000 0000 0000 0000 xxxx xxxx
      gl_SocTxDataForPrbs_ext[5]  |= ul_data_extended[10] <<  8;     // 0000 xxxx xxxx 00xx xxxx xx00
      gl_SocTxDataForPrbs_ext[5]  |= ul_data_extended[11] << 28;     // xx00

      gl_SocTxDataForPrbs_ext[6]   = ul_data_extended[11] >>  4;     // 0000 0000 0000 0000 xxxx xxxx 00xx xxxx
   }
   else
   {
      gl_SocTxDataForPrbs = ul_data_extended[0];
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : GenerateSynSymLookUpData
 *
 *  Description:  This function generates the lookup data for generating
 *  the R-P-SYNCHRO or O-P-SYNCHRO symbols.
 *
 *  Prototype:
 *      void GenerateSynSymLookUpData(int16 s_sign)
 *
 *  Input Arguments:
 *      s_gain -- >=0: modulate all tones with (+,+), <0: modulate all tones with (-,-)
 *
 *  Output Arguments:
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gl_SocTxDataForPrbs -- (O) Data to be sent to the PRBS generator (IT_PRBS_DATA registor)
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void GenerateSynSymLookUpData(int16 s_gain)
{

   if (s_gain >= 0)
   {
      gl_SocTxDataForPrbs = 0;
   }
   else
   {
      gl_SocTxDataForPrbs = 0xFFFFFFFF;
   }

   //Go to the tone trail for debug purpose
   guca_TxOctet[0] = (uint8)gl_SocTxDataForPrbs;
   guca_TxOctet[1] = (uint8)gl_SocTxDataForPrbs;

}



