/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   File Name: ProcessCpBeta.c
;
;   This file contains the function which processes the CP length and Beta length
;   provided by the remote modem for the local modem to use.
;
*****************************************************************************/

#include "vdsl_compiler.h"
#include "sys_const.h"
#include "gdata.h"
#include "vdsl_xception.h"
#include "decimalgain.h"
#include "socmessage.h"
#include "cmv.h"

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : ProcessCpBeta
 *
 *  Description:  This function saves the CP length and Beta length
 *  expressed in samples of the sampling rate corresponding to the IDFT size
 *  of the remote modem in the global variabls.
 *  It also computes the difference between the CS length of the local and remote
 *  modem expressed in the local modem's sampling rate
 *
 *  Prototype: void ProcessCpBeta(uint16 us_CpLength, uint16 BetaLength)
 *
 *  Input Arguments:
 *       us_CpLength -- CP length provided by the remote modem
 *      us_BetaLength -- Beta length provided by the remote modem
 *
 *  Output Arguments:
 *      NONE
 *
 *  Return:
 *      NONE
 *
 *  Global Variables Used:
 *      gs_FeModemLog2IfftSize - (I) log2 of IFFT size provided by the remote modem
 *      gs_RxLog2FftLength - (I) log2 of the FFT size provided the local modem
 *      gs_RemoteTxCPLength - (O) TX CP length sent by the remote modem
 *      gs_RemoteTxBetaLength - (O) TX Beta length sent by the remote modem
 *      gs_RxTxCsDiff - (O) The difference between the CS length of the local and remote
 *                          modem expressed in the local modem's sampling rate
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void ProcessCpBeta(uint16 us_CpLength, uint16 us_BetaLength)
{
   int16 i, s_temp, s_FeTxToNeRxSampleConvShift;

   // Check far-end IDFT size
   {
      int16 s_AllowedLog2IfftSize;

      s_AllowedLog2IfftSize = 13;
      if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
      {
         s_AllowedLog2IfftSize = 14;
      }

      if((gs_FeModemLog2IfftSize < 7) || (gs_FeModemLog2IfftSize > s_AllowedLog2IfftSize))
      {
         EnterFailStates(E_CODE_INVALID_FE_IDFT_SIZE);
         return;
      }
   }

   //Save the remote CP and beta length
   // gs_RemoteTxCPLength in CO-Tx sampling rate, e.g. 512/8192 in 35MHz mode
   // gs_RemoteTxCPLength in CO-Tx sampling rate, e.g. 256/4096 in 17MHz mode

   gs_RemoteTxCPLength = us_CpLength;
   gs_RemoteTxBetaLength = us_BetaLength;

   //Compute the sample rate conversion factor between
   //far-end and near-end modem based on the FFT size difference
   s_FeTxToNeRxSampleConvShift = gs_RxLog2FftLength - gs_FeModemLog2IfftSize;

   //Compute the CS in local modem sampling rate
   s_temp = us_CpLength - us_BetaLength;

   if(s_FeTxToNeRxSampleConvShift >= 0)
   {
      s_temp <<= s_FeTxToNeRxSampleConvShift;
      s_temp = gs_RxCELength - s_temp;
   }
   else
   {
      i = -s_FeTxToNeRxSampleConvShift;

      //First compute CS in higher sampling rate to get better precision
      s_temp = (gs_RxCELength << i) - s_temp;

      //Then convert the CS to the local sampling rate with rounding
      s_temp += 1<<(i-1);
      s_temp >>= i;
   }

   if(s_temp < 0)
   {
      EnterFailStates(E_CODE_INVALID_FE_CS_SIZE);
   }

   //Compute the difference between the local RX CS length and remote TX CS length
   gs_RxTxCsDiff = gs_RxCSLength - s_temp;
}
