/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright C 2016 Intel Corporation
    Copyright (C), 1994-2003 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/****************************************************************************
;
;   Aware DMT Technology. Proprietary and Confidential.
;
;   40 Middlesex Turnpike, Bedford, MA 01730-1413
;   Phone (781) 276 - 4000
;   Fax   (781) 276 - 4001
;
;   File Name: SocMsgs_Common.c
;
;   Common functions used by both CO and RT for processing
;   Special Operation Channel Messages.
;
;
*****************************************************************************/
//*****************************************************************************
// SocMsgs_Common.c
//
// History
//
// 23/03/2015 Fuss: VDSL2 - CPE bug in SOC message handling for O-REPEAT_REQUEST
//            Grep for XDSLRTFW-2284
//*****************************************************************************

#include "vdsl_compiler.h"
#include "sys_const.h"
#include "gdata.h"
#include "vdsl_xception.h"
#include "decimalgain.h"
#include "socmessage.h"
#include "TxPSDCOntrol.h"
#include "SharedFuncs.h"


#ifdef INCLUDE_NON_CACHE_FUNCTIONS
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CheckMsgCode
 *
 *  Description:  This function checks the message code only
 *  in order to early detect of unexpected message.
 *
 *  Prototype:  int16 CheckMsgCode(HDLC_Fifo_attrib_t *pt_FIFO)
 *
 *  Input Arguments:
 *      pt_FIFO -- pointer to the RX message fifo structure
 *
 *  Output Arguments:
 *      NONE
 *
 *  Return:
 *      TRUE or FALSE
 *
 *  Global Variables Used:
 *      gs_RxMsgErrorCode - (O) RX Message Error Code
 *      guc_NumberOfPossibleRxMsgs - number of possible expected messages
 *      guca_PossibleRxMsgCodes[] -- possible message codes
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 CheckMsgCode(HDLC_Fifo_attrib_t *pt_FIFO)
{
   int16 i, s_Start;

   pt_FIFO = &RxHDLCFifo;

   s_Start = pt_FIFO->s_CurFIFODepth - pt_FIFO->sa_SegmentLength[pt_FIFO->uc_NumOfSegments-1];

   //Check Control byte of HDLC frame, containing segment index
   s_Start++;
   guc_RxSegmentIdx = pt_FIFO->puca_OctetBuffer[s_Start]&0x0F;
   guc_RxNumOfSegments = pt_FIFO->puca_OctetBuffer[s_Start++]>>4;

   //Process the first segment
   if(guc_RxSegmentIdx <= 1)
   {
      //get message code
      guc_RxMsgCode = pt_FIFO->puca_OctetBuffer[s_Start];

      if(guc_RxMsgCode == SOC_MSG_ACK_SEG)
      {
         gs_RxMsgErrorCode = E_CODE_MSG_NO_ERROR;
         return(TRUE);
      }

      //If the message code is not expected, return error
      gs_RxMsgErrorCode = E_CODE_MSG_CODE_ERROR;
      for(i=0; i<guc_NumberOfPossibleRxMsgs; i++)
      {
         if(guc_RxMsgCode == guca_PossibleRxMsgCodes[i])
         {
            //If this is expected message code, clean error code
            gs_RxMsgErrorCode = E_CODE_MSG_NO_ERROR;

            break;
         }
      }

      if(gs_RxMsgErrorCode != E_CODE_MSG_NO_ERROR)
      {
         return(FALSE);
      }
   }

   return(TRUE);
}
#endif // INCLUDE_NON_CACHE_FUNCTIONS








#ifdef INCLUDE_CACHE_FUNCTIONS

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : BkgdInterpretMsg
 *
 *  Description:  The function to start the message interpretation function.
 *
 *  Prototype:  void BkgdInterpretMsg(void)
 *
 *  Input Arguments:
 *      NONE
 *
 *  Output Arguments:
 *      NONE
 *
 *  Return:
 *
 *  Global Variables Used:
 *      gs_RxMsgCheckOkFlag       - (O) flat to indicate if the message interpretation is ok or not
 *      gs_RxBkgdProcessFlag   - (O) flat to indicate if the background process is done or not
 *      RxHDLCFifo            - (I) RX message structure
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 InterpretMsg(HDLC_Fifo_attrib_t *pt_FIFO, uint8 uc_MsgCode);
void BkgdInterpretMsg(void)
{
   gs_RxMsgCheckOkFlag = InterpretMsg(&RxHDLCFifo, guc_RxMsgCode);

   gs_RxBkgdProcessFlag = MSG_DECODE_DONE;
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : InterpretMsg
 *
 *  Description:  The function to perform the message interpretation
 *
 *  Prototype:  int16 InterpretMsg(HDLC_Fifo_attrib_t *pt_FIFO, uint8 uc_MsgCode)
 *
 *  Input Arguments:
 *      pt_FIFO -- pointer to the RX message fifo structure
 *       uc_MsgCode -- message code to be interpreted
 *
 *  Output Arguments:
 *      NONE
 *
 *  Return:
 *      TRUE or FALSE
 *
 *  Global Variables Used:
 *      gs_RxMsgErrorCode   - (O) Message error code
 *      gpuca_RxHDLCMsgBuf   - (O) pointer to Buffer to store a complete RX message
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

int16 InterpRepeatReqMsg(void);
int16 InterpretMsg(HDLC_Fifo_attrib_t *pt_FIFO, uint8 uc_MsgCode)
{
   int16 s_segment_idx, s_NumOfSegments;
   int16 i, j, s_Start;

   //Process the multi-segment message
   s_Start = 0;
   gs_RxWholeMsgLength = 0;

   //Interpret REPEAT-REQUEST message (common to CO and CPE)
   if(uc_MsgCode == SOC_MSG_REPEAT_REQUEST)
   {
      return(TRUE);
   }

   for(j=0; j<pt_FIFO->uc_NumOfSegments; j++)
   {
      i = s_Start;

      //====================================================================
      //Check Address byte of HDLC frame containing the message index
      //====================================================================
      guc_RxHDLCMsgIndex = pt_FIFO->puca_OctetBuffer[i++];

      /* May allow this error to increase the chance to be interoperable with some uncompliant modem??
      if(guc_ExpectedMsgIdx != 0)
      {
         //We are expecting a particular message index
         if(guc_RxHDLCMsgIndex != guc_ExpectedMsgIdx)
         {
            //Received message is not equal to expected message, return FALSE
            gs_RxMsgErrorCode = E_CODE_MSG_IDX_ERROR;

            return(FALSE);
         }
      }
      */

      //Check Control byte of HDLC frame, containing segment index
      s_NumOfSegments = (uint16)pt_FIFO->puca_OctetBuffer[i]>>4;
      s_segment_idx = pt_FIFO->puca_OctetBuffer[i++]&0x0F;

      //Segment index is wrong
      if((s_segment_idx == 0) || (s_segment_idx > s_NumOfSegments) ||
            (s_segment_idx != (pt_FIFO->uc_PreSegmentIdx+1)))
      {
         gs_RxMsgErrorCode = E_CODE_MSG_SEG_IDX_ERROR;
         return(FALSE);
      }

      //Remove Address, Control and FCS bytes
      for( ; i<s_Start+pt_FIFO->sa_SegmentLength[j]-2; i++)
      {
         gpuca_RxHDLCMsgBuf[gs_RxWholeMsgLength++] = pt_FIFO->puca_OctetBuffer[i];
      }
      pt_FIFO->uc_PreSegmentIdx++;

      //Point to the start address of next segment
      s_Start += pt_FIFO->sa_SegmentLength[j];

   } //for(j=0; j<pt_FIFO->uc_NumOfSegments; j++)

   //=============================================================
   //If all segment(s) are received, decode and interpret message
   //=============================================================

   i = InterpMsg_RT_VDSL2(uc_MsgCode);

   return(i);
}


void Skip12Or16BitPairDescriptors(int16 *ps_PtrMsgIndex, int16 s_NumberOfPairs)
{
   uint8 uc_UnpackXBitsOctetNum;

   uc_UnpackXBitsOctetNum = 3;
   // ANXQ_SUPPORT
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      uc_UnpackXBitsOctetNum++;      // 4
   }
   *ps_PtrMsgIndex = (*ps_PtrMsgIndex + (s_NumberOfPairs*uc_UnpackXBitsOctetNum));

}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : InitTwelveBitsPair
 *
 *  Description:  Sets the two 12-bits number in structure. These 2 numbers
 *                are spread over 3 bytes (24 bits).
 *
 *  Prototype:
 *           void InitTwelveBitsPair(int16 First, int16 Second, TwelveBitsPair_t *Pair);
 *
 *
 *  Input Arguments:
 *      int16 First              First  12 bits number. (bits 0-11)
 *      int16 Second             Second 12 bits number. (bits 12-23)
 *
 *
 *  Output Arguments:
 *      TwelveBitsPair_t *Pair   Pointer to 3-byte structure that contains the
 *                               two 12-bits numbers.
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void InitTwelveBitsPair(int16 First, int16 Second, TwelveBitsPair_t *Pair)
{

   Pair->uc_Byte0 = (uint8)((Second & 0x0FF0) >> 4);
   Pair->uc_Byte1 = (uint8)((Second & 0x000F) << 4 | (First & 0x0F00) >> 8);
   Pair->uc_Byte2 = (uint8)(First & 0x00FF);
}


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : Unpack12BitsPair
 *
 *  Description:  Unpack an array of 3 bytes (24 bits), where each 3 bytes
 *   produce two 12-bits numbers: LSBWord = bits 0-11, MSBWord = bits 12-23.
 *
 *  Prototype:
 *           void Unpack12BitsPair(int16 *ps_PtrMsgIndex, int16 s_NumberOfPairs, uint8 *puca_ByteBuf,
 *                                 int16 *psa_LSBword, int16 *psa_MSBword)
 *
 *  Input Arguments:
 *      ps_PtrMsgIndex  -- Pointer to message index
 *      s_NumberOfPairs -- no. of 12-bit pairs to be unpacked
 *      puca_ByteBuf    -- pointer to the input byte array
 *
 *  Output Arguments:
 *       ps_PtrMsgIndex -- message index
 *       psa_LSBword -- pointer to the 12-bit array storing the Least Significant words
 *       psa_MSBword -- pointer to the 12-bit array storing the Most Significant words
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void DecodeBands(int16 *ps_PtrMsgIndex, int16 s_NumberOfPairs, uint8 *puca_OctetBuffer,
                 int16 *psa_LSBword, int16 *psa_MSBword)
{
   // ANXQ_SUPPORT
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      Unpack16BitsPair(ps_PtrMsgIndex,
                       s_NumberOfPairs,
                       puca_OctetBuffer,
                       psa_LSBword,
                       psa_MSBword);
   }
   else
   {
      Unpack12BitsPair(ps_PtrMsgIndex,
                       s_NumberOfPairs,
                       puca_OctetBuffer,
                       psa_LSBword,
                       psa_MSBword);
   }
}

void Unpack12BitsPair(int16 *ps_PtrMsgIndex, int16 s_NumberOfPairs, uint8 *puca_ByteBuf, int16 *psa_LSBword, int16 *psa_MSBword)
{
   unsigned int i;
   uint16 us_word;

   for(i=0; i<s_NumberOfPairs; i++)
   {
      //Get next 3 bytes
      us_word = puca_ByteBuf[(*ps_PtrMsgIndex)++];        // byte 1
      us_word <<= 8;
      us_word |= puca_ByteBuf[(*ps_PtrMsgIndex)++];       // byte 2
      psa_MSBword[i] = (us_word>>4)&0x0FFF;
      us_word <<= 8;
      us_word |= puca_ByteBuf[(*ps_PtrMsgIndex)++];       // byte 3
      psa_LSBword[i] = us_word&0x0FFF;
   }
}

// ANXQ_SUPPORT
void Unpack16BitsPair(int16 *ps_PtrMsgIndex, int16 s_NumberOfPairs, uint8 *puca_ByteBuf, int16 *psa_LSBword, int16 *psa_MSBword)
{
   unsigned int i;
   uint16 us_word;

   for(i=0; i<s_NumberOfPairs; i++)
   {
      //Get next 4 bytes
      us_word  = (puca_ByteBuf[(*ps_PtrMsgIndex)++] << 8);  // byte 1
      us_word |= puca_ByteBuf[(*ps_PtrMsgIndex)++];         // byte 2
      psa_MSBword[i] = us_word;
      us_word  = (puca_ByteBuf[(*ps_PtrMsgIndex)++] << 8);  // byte 3
      us_word |= puca_ByteBuf[(*ps_PtrMsgIndex)++];         // byte 4
      psa_LSBword[i] = us_word;
   }
}

// XDSLRTFW-1696 (Start)
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : Unpack3Bits8Data
 *
 *  Description:  Unpack an array of 3 bytes (24 bits), where each 3 bytes
 *  contains eight 3 bits of data.
 *
 *  Prototype:
 *         void Unpack3Bits8Data(uint8 *puca_ByteBuf, uint8 *puca_data)
 *
 *  Input Arguments:
 *    puca_ByteBuf -- pointer to the input byte array
 *
 *  Output Arguments:
 *     puca_data -- pointer to the 8-bit array storing the 3 bit data at each array element
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void Unpack3Bits8Data(uint8 *puca_ByteBuf, uint8 *puca_data)
{
   int16 i;
   uint32 ul_word;

   // form 24 bit word out of three octets
   ul_word = puca_ByteBuf[0];
   ul_word <<= 8;
   ul_word |= puca_ByteBuf[1];
   ul_word <<= 8;
   ul_word |= puca_ByteBuf[2];

   // store each 3 bit data to the output array
   for (i =0 ; i < 8 ; i++)
   {
      puca_data[i] = (uint8)(ul_word & 0x7);
      ul_word >>= 3;
   }
}
// XDSLRTFW-1696 (end)


/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : UnpackPSD
 *
 *  Description:  Unpack the PSD
 *
 *  Prototype:
 *           void UnpackPSD(int16 *ps_PtrMsgIndex, int16 s_NumOfPoints, uint8 *puca_OctetBuffer,
 *                          int16 *ps_MaxNomPSD,
 *                          int16 s_MaxNomPSD,
 *                          void *pt_PSDDesc)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Notes: Initially the last parameter was of type PSDDescriptorTable_t but to suppress/remove a warning
 *         it has been changed to type void. And suitable changes are made in the function and wherever
 *         it is being employed.
 *
 *------------------------------------------------------------------------
 *^^^
 */
void UnpackPSD(int16 *ps_PtrMsgIndex,
               int16 s_NumOfPoints,
               uint8 *puca_OctetBuffer,
               int16 *ps_MaxNomPSD,
               int16 s_MaxNomPSD,
               PSDDescriptorTable_t *pt_PSDDesc)
{
   DecPSD_Descriptor_t  t_TempPSDDesc;
   int16 j, s_PSDLevel;

   // ANXQ_SUPPORT
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      // Unpack message into temp PSD buffer:
      Unpack16BitsPair(ps_PtrMsgIndex, s_NumOfPoints, puca_OctetBuffer,
                       &(t_TempPSDDesc.sa_ToneIndex[0]),
                       &(t_TempPSDDesc.sa_PSDLevel[0]));
   }
   else
   {
      // Unpack message into temp PSD buffer:
      Unpack12BitsPair(ps_PtrMsgIndex, s_NumOfPoints, puca_OctetBuffer,
                       &(t_TempPSDDesc.sa_ToneIndex[0]),
                       &(t_TempPSDDesc.sa_PSDLevel[0]));
   }

   // Compute Max Nom PSD
   if (ps_MaxNomPSD)
   {
      // Find maximum PSD level within active bands:
      s_MaxNomPSD = 0;   /* min pssible psd level value recieved */
      for (j = 0; j < s_NumOfPoints; j++ )
      {
         if (s_MaxNomPSD < t_TempPSDDesc.sa_PSDLevel[j])
         {
            s_MaxNomPSD = t_TempPSDDesc.sa_PSDLevel[j];
         }
      }

      if (gul_dbgSocMsgControl & FORCE_PSD_DESCRIPTOR_IN_0PT5_dB_FORMAT)
      {
         /* psd in message is in 0.5 dB format */
         s_MaxNomPSD = s_MaxNomPSD*5;
      }

      *ps_MaxNomPSD = (1400 - s_MaxNomPSD);
   }
   else
   {
      s_MaxNomPSD = (1400 - s_MaxNomPSD);
   }

   for (j = 0; j < s_NumOfPoints; j++ )
   {

      pt_PSDDesc->ut_PSDRecord[j].us_IndexOfTone = t_TempPSDDesc.sa_ToneIndex[j];

      // Convert PSD level from VDSL message format to PSD descriptor format:
      if (gul_dbgSocMsgControl & FORCE_PSD_DESCRIPTOR_IN_0PT5_dB_FORMAT)
      {
         /* psd in message is in 0.5 dB format */
         s_PSDLevel  = t_TempPSDDesc.sa_PSDLevel[j]*5;
      }
      else
      {
         /* psd in message is in 0.1 dB format */
         s_PSDLevel  = t_TempPSDDesc.sa_PSDLevel[j];
      }
      s_PSDLevel = s_PSDLevel-s_MaxNomPSD;
      pt_PSDDesc->ut_PSDRecord[j].s_PSDLevelOfTone = -s_PSDLevel;
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : CalculatePSD
 *
 *  Description:  Calculates the PSD depending on whether mode is VDSL1 or VDSL2
 *
 *  Prototype: int16 CalculatePSD(int16 s_psdLevelOfTone, int16 s_power)
 *
 *  Input Arguments:
 *
 *  Output Arguments:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
int16 CalculatePSD(int16 s_psdLevelOfTone, int16 s_power)
{
   int16 s_PSDLevel;

   s_PSDLevel = -s_psdLevelOfTone-s_power+1400;   // in 0.1 dB frmat with an offest to -140dBm/Hz
   if (gul_dbgSocMsgControl & FORCE_PSD_DESCRIPTOR_IN_0PT5_dB_FORMAT)
   {
      s_PSDLevel = s_PSDLevel/5;   // For VDSL1, represent in 0.5dB format
   }

   // If user has configured PSD values below -140 dbM/Hz, limit them to -140 dbM/Hz
   if (s_PSDLevel < 0)
   {
      s_PSDLevel = 0;
   }

   return s_PSDLevel;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EncodeNumOfSamples
 *
 *  Description:  Encodes number of samples.
 *                For AnxQ it shall be expressed in multiples of 2 samples at the upstream sampling rate
 *                corresponding to the IDFT size communicated during the ITU-T G.994.1 handshake phase.
 *                Therefore it used for TimingAdvance, Cyclic prefix Length (CP)
 *
 *  Prototype:
 *          void EncodeNumOfSamples(int16 *ps_PtrMsgIndex, int16 s_InputVal, uint8 *puca_TxSocMsg)
 *
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EncodeNumOfSamples(int16 *ps_PtrMsgIndex, int16 s_InputVal, uint8 *puca_TxSocMsg)
{
   // ANXQ_SUPPORT
// Note: In file InterpOSignatureMsg_VDSL2.c, fct. UnpackOSignatureMsg_VDSL2() a check
//       for the special value is done. Therefore a special value will never be transmitted
//       by the CPE!
//   if ((gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK) &&
//       (s_InputVal != 0x7FFF))
   if ((gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK) &&
       (gs_TxLog2IfftLength > US_LOG2_FFT_LENGTH_8192))                // US_LOG2_FFT_LENGTH_8192 = 13
   {
      //  ... shall be expressed either in multiples of 2 samples
      //      if profile 35b is used with extended IDFT Size, or otherwise,
      //      in samples at the upstream sampling rate corresponding to the
      //      IDFT size communicated during
      //           - the ITU-T G.994.1 handshake phase or
      //           - in Field #6 of R-PRM.
      //  Note: Profile 35b being used with extended IDFT Size is equivalent
      //        with Field #6 of R-PRM having value 14.
      s_InputVal = (s_InputVal >> 1);
   }
   puca_TxSocMsg[(*ps_PtrMsgIndex)++] = (s_InputVal >> 8);
   puca_TxSocMsg[(*ps_PtrMsgIndex)++] = (uint8)s_InputVal;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EncodeTones
 *
 *  Description:  Encodes tones information
 *                The first octet of the tone descriptor shall contain the number of tones selected
 *                by the VTU-R to be transmitted. If this number is zero, there shall be no further octets
 *                in the descriptor. If the number of tones is not equal to zero, each group of
 *                three consecutive octets in the descriptor shall describe the location
 *                (i.e., the subcarrier index) of two tones. If the number of tones is odd,
 *                the last 12 bits of the field shall be set to ZERO (and ignored by the receiver).
 *
 *  Prototype:
 *          void EncodeTones(int16 *ps_PtrMsgIndex, int16 s_NumOfTones,
 *                           int16 *psa_ToneArray, uint8 *puca_TxSocMsg, int16 s_SwapToneTransmission)
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EncodeTones(int16 *ps_PtrMsgIndex, int16 s_NumOfTones,
                 int16 *psa_ToneArray, uint8 *puca_TxSocMsg, int16 s_SwapToneTransmission)
{
   int16 s_NumOfTonesTemp;
   signed int j, i;
   int16 s_first, s_second, s_temp;
   TwelveBitsPair_t temp24;

   // Get message index
   i = *ps_PtrMsgIndex;

   // If the number of stones is odd, the last 12 bits of the field
   //  shall be set to ZERO (and ignored by the receiver).
   s_NumOfTonesTemp = s_NumOfTones;
   if(s_NumOfTonesTemp & 1)
   {
      s_NumOfTonesTemp--;
   }

   // ANXQ_SUPPORT
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      for (j = 0; j<s_NumOfTonesTemp; )
      {
         // VDSL2, AnxQ (35b): Encode two 16 bit values into 4 bytes message
         s_first = psa_ToneArray[j++];
         s_second = psa_ToneArray[j++];


         if (s_SwapToneTransmission)
         {
            // Ikanos way
            s_temp= s_first;
            s_first = s_second;
            s_second = s_temp;
         }

         puca_TxSocMsg[i++] = (uint8)(s_second >> 8);     // MSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[i++] = (uint8)(s_second);          // LSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[i++] = (uint8)(s_first >> 8);
         puca_TxSocMsg[i++] = (uint8)(s_first);
      }
   }
   else
   {
      for (j = 0; j<s_NumOfTonesTemp; )
      {
         // VDSL2: Encode two 12 bit values into 3 bytes message
         s_first = psa_ToneArray[j++];
         s_second = psa_ToneArray[j++];

         if (s_SwapToneTransmission)
         {
            // Ikanos way
            s_temp= s_first;
            s_first = s_second;
            s_second = s_temp;
         }

         InitTwelveBitsPair(s_first, s_second, &temp24);
         puca_TxSocMsg[i++] = temp24.uc_Byte0;
         puca_TxSocMsg[i++] = temp24.uc_Byte1;
         puca_TxSocMsg[i++] = temp24.uc_Byte2;
      }
   }

   // Note: This was done to safe cycles.
   //       But for sure more code is needed!
   if(s_NumOfTones & 1)
   {
      // Encode two 12 bit values into 3 bytes message
      s_first = psa_ToneArray[j];
      s_second = 0;

      if (s_SwapToneTransmission)
      {
         // Ikanos way
         s_temp= s_first;
         s_first = s_second;
         s_second = s_temp;
      }

      // ANXQ_SUPPORT
      if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
      {
         puca_TxSocMsg[i++] = (uint8)(s_second >> 8);     // MSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[i++] = (uint8)(s_second);          // LSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[i++] = (uint8)(s_first >> 8);
         puca_TxSocMsg[i++] = (uint8)(s_first);
      }
      else
      {
         InitTwelveBitsPair(s_first, s_second, &temp24);
         puca_TxSocMsg[i++] = temp24.uc_Byte0;
         puca_TxSocMsg[i++] = temp24.uc_Byte1;
         puca_TxSocMsg[i++] = temp24.uc_Byte2;
      }
   }

   // Return message index
   *ps_PtrMsgIndex = i;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EncodeBands
 *
 *  Description:  Encodes bands information
 *
 *  Prototype:
 *           void EncodeBands(int16 *ps_PtrMsgIndex, int16 s_NumOfBands,
 *                            int16 sa_LeftChannel[], int16 sa_RightChannel[], uint8 *puca_TxSocMsg)
 *
 *
 *  Input Arguments:
 *      int16 *ps_PtrMsgIndex   - ptr to O-Sig msg index
 *      int16 s_NumOfBands      - number of bands
 *      int16 sa_LeftChannel[]  - array of left channels in band to be encoded
 *      int16 sa_RightChannel[] - array of right channels in band to be encoded
 *      uint8 *puca_TxSocMsg    - ptr to SOC message array
 *
 *  Output Arguments:
 *
 *
 *  Global Variables Used:
 *      gpuca_TxSocMsg -- (I) pointer to the TX message buffer
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EncodeBands(int16 *ps_PtrMsgIndex, int16 s_NumOfBands,
                 int16 sa_LeftChannel[], int16 sa_RightChannel[],
                 uint8 *puca_TxSocMsg)
{
   int16 i, j;

   // Get message index
   i = *ps_PtrMsgIndex;

   // ANXQ_SUPPORT
   //OPEN_VRX518 InitSexteenBitsPair function being in PD_518
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      // A field can contain parameter values expressed in more than one byte.
      // In this case, the field shall be split into bytes with the byte containing the MSBs
      // of the parameter value sent first.
      for (j = 0; j < s_NumOfBands; j++)
      {
         puca_TxSocMsg[i++] = (uint8)(sa_RightChannel[j] >> 8);     // MSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[i++] = (uint8)(sa_RightChannel[j]);          // LSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[i++] = (uint8)(sa_LeftChannel[j] >> 8);
         puca_TxSocMsg[i++] = (uint8)(sa_LeftChannel[j]);
      }
   }
   else
   {
      TwelveBitsPair_t temp24;

      for (j = 0; j < s_NumOfBands; j++)
      {
         InitTwelveBitsPair(sa_LeftChannel[j], sa_RightChannel[j], &temp24);

         puca_TxSocMsg[i++] = temp24.uc_Byte0;
         puca_TxSocMsg[i++] = temp24.uc_Byte1;
         puca_TxSocMsg[i++] = temp24.uc_Byte2;
      }
   }

   // Return message index
   *ps_PtrMsgIndex = i;
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : EncodePSDs
 *
 *  Description:  Encodes PSD Descriptor information
 *
 *  Prototype:
 *           void EncodePSDs(int16 *ps_PtrMsgIndex, int16 s_NumOfPoints,
 *                   PSDDescriptorTable_t * pt_PSDDescriptorTable,
 *                   int16 s_MaxNomPSD, uint8 *puca_TxSocMsg)
 *
 *
 *  Input Arguments:
 *      int16 *ps_PtrMsgIndex   - ptr to O-Sig msg index
 *      int16 s_NumOfPoints     - number of PSD breakpoints
 *      t_PSDDescriptorTable    - PSD Descriptor
 *      int16 s_MaxNomPSD       - maximum nominal PSD level
 *
 *  Output Arguments:
 *      uint8* puca_TxSocMsg    - Soc msg o/p buffer
 *      Updated O-Sig msg index
 *
 *  Global Variables Used:
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void EncodePSDs(int16 *ps_PtrMsgIndex, int16 s_NumOfPoints,
                void *pt_PSDDescriptorTable1,
                int16 s_MaxNomPSD, uint8 *puca_TxSocMsg)
{
   int16 j;
   int16 s_PSDLevel;
   PSDDescriptorTable_t *pt_PSDDescriptorTable;

   pt_PSDDescriptorTable = (PSDDescriptorTable_t *)pt_PSDDescriptorTable1;

   // ANXQ_SUPPORT
   if (gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK)
   {
      for (j = 0; j < s_NumOfPoints; j++)
      {
         // Convert PSD level from PSD descriptor format to VDSL message format:
         s_PSDLevel = CalculatePSD(pt_PSDDescriptorTable->ut_PSDRecord[j].s_PSDLevelOfTone,
                                   s_MaxNomPSD);

         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = (uint8)(s_PSDLevel >> 8);     // MSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = (uint8)(s_PSDLevel);          // LSB, masking is done implicitly by (uint8)
         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = (uint8)(pt_PSDDescriptorTable->ut_PSDRecord[j].us_IndexOfTone >> 8);
         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = (uint8)(pt_PSDDescriptorTable->ut_PSDRecord[j].us_IndexOfTone);
      }
   }
   else
   {
      TwelveBitsPair_t temp24;

      for (j = 0; j < s_NumOfPoints; j++)
      {
         // Convert PSD level from PSD descriptor format to VDSL message format:
         s_PSDLevel = CalculatePSD(pt_PSDDescriptorTable->ut_PSDRecord[j].s_PSDLevelOfTone,
                                   s_MaxNomPSD);

         InitTwelveBitsPair(pt_PSDDescriptorTable->ut_PSDRecord[j].us_IndexOfTone,
                            s_PSDLevel, &temp24);

         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = temp24.uc_Byte0;
         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = temp24.uc_Byte1;
         puca_TxSocMsg[(*ps_PtrMsgIndex)++] = temp24.uc_Byte2;
      }
   }
}

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : UnpackedMsgLengthCheck
 *
 *  Description:  Checks if decoded message length corresponds to length
 *               of raw octets recieved
 *
 *  Prototype:
 *
 *
 *  Input Arguments:
 *      s_DecodedMsgLength -- number of decoded bytes
 *      s_ActualWholeMsgLength -- number of received message bytes (between HDLC flags)
 *
 *  Output Arguments:
 *
 *  Global Variables Used:
 *      gul_dbgSocMsgControl -- flag to control which condition to go to fail state
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
FlagT UnpackedMsgLengthCheck(int16 s_DecodedMsgLength, int16 s_ActualWholeMsgLength)
{
   // Do not ignore undecoded bytes @ end of recieved soc message
   if ((gul_dbgSocMsgControl & CHECK_MSG_LENGTH_MATCH) && (s_DecodedMsgLength != s_ActualWholeMsgLength))
   {
      return(FALSE);
   }
   else if (s_DecodedMsgLength > s_ActualWholeMsgLength)
   {
      return(FALSE);
   }
   else
   {
      return(TRUE);
   }
}


//XDSLRTFW-2541 (Start_End)
/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : LimitBandPlanToHighestTone
 *
 *  Description: Checks the CO DS bands descriptor for 35b, i.e. tones greater equal to 4096.
 *               When this is the case the following gets performed:
 *
 *                - Fake the DS3 high (right) index to 4095
 *                - Fake the number of DS bands to ignore all bands above DS_START_IDX_35B = 4096.
 *                - Note: The original bands descriptor gets stored in
 *                   (gsa_RxBandLeftChannelOSignature, gsa_RxBandRightChannelOSignature)
 *
 *  Prototype:
 *      void LimitBandPlanToHighestTone(int16 *psa_BandLeft, int16 *psa_BandRight, int16 *ps_NumOfBands, int16 s_FakeBandRight)
 *
 *  Input Arguments:
 *      psa_BandLeft    -- pointer to band descriptor for left tone index
 *      psa_BandRight   -- pointer to band descriptor for right tone index
 *      ps_NumOfBands   -- pointer to number of bands
 *      s_FakeBandRight -- Right channel index to be faked
 *
 *  Output Arguments:
 *
 *  Notes:
 *      The extended DS band can be
 *         1.1 DS3 band extended upto tone 8192, i.e. one big band
 *         1.2 X DS bands above DS3
 *
 *------------------------------------------------------------------------
 *^^^
 */
// XDSLRTFW-3741 Start_End
void LimitBandPlanToHighestTone(int16 *psa_BandLeft, int16 *psa_BandRight, int16 *ps_NumOfBands, int16 s_FakeBandRight)
{
   if (s_FakeBandRight < psa_BandRight[((*ps_NumOfBands)-1)])
   {
      signed int j, i;

      j = 0;

      for(i = 0; i < *ps_NumOfBands; i++)
      {
         if (psa_BandRight[i] <= s_FakeBandRight)
         {
            // Take bands as they are.
            psa_BandLeft[j] = psa_BandLeft[i];
            psa_BandRight[j++] = psa_BandRight[i];
         }
         else // if (psa_BandRight[i] > DS_START_IDX_35B)
         {
            // Fake DSx high (right) index
            if(psa_BandLeft[i] < s_FakeBandRight)
            {
               psa_BandLeft[j] = psa_BandLeft[i];
               psa_BandRight[j++] = s_FakeBandRight;
            }
         }
      }

      // clear rest of the entries
      for(i = j; i < *ps_NumOfBands; i++)
      {
         psa_BandLeft[i] = 0;
         psa_BandRight[i] = 0;
      }

      // Set new number of bands
      *ps_NumOfBands = j;
   }
}

void GetMaxNomPSDPerBand(int16 *psa_MaxNomPSDBand, PSDDescriptorTable_t *pt_PSDDesc, int16 s_MaxNomPSD, int16 *psa_BandRight, int16 s_NumOfBands)
{
   signed int i,j;

   // Set MaxNomPsd per band to highest value.
   // Note: A bigger value means a lower psd, becuase positive values are used
   //       instead of negative values.
   psa_MaxNomPSDBand[0] = 0x7fff;
   i = 0;
   j = 0;

   // Limit number of bands to the max value!
   if (s_NumOfBands > MAX_NUM_DS_BANDS)
   {
      s_NumOfBands = MAX_NUM_DS_BANDS;
   }

   // Find maximum PSD level within active bands:
   while ((j < pt_PSDDesc->us_NumberOfTones) && (i < s_NumOfBands))
   {
      if((pt_PSDDesc->ut_PSDRecord[j].us_IndexOfTone >= psa_BandRight[i]))
      {
         if((pt_PSDDesc->ut_PSDRecord[j].us_IndexOfTone == psa_BandRight[i]))
         {
            j++;
         }
         // Case where psa_MaxNomPSDBand is a mask, but psa_MaxNomPSDBand should be used as a template.
         // Take 3.5 dB off of gt_PwrConfigParam.s_Dn_MaxNomPSD to do this conversion.
         psa_MaxNomPSDBand[i++] = (psa_MaxNomPSDBand[i] + s_MaxNomPSD);
         // Set MaxNomPsd per band to highest value.
         // Note: A bigger value means a lower psd, becuase positive values are used
         //       instead of negative values.
         if (i < s_NumOfBands)
         {
            psa_MaxNomPSDBand[i] = 0x7fff;
         }
      }

      // Store minimum value, i.e. MaxNomPsd per band.
      // Note: A bigger value means a lower psd, becuase positive values are used
      //       instead of negative values.
      if (psa_MaxNomPSDBand[i] > pt_PSDDesc->ut_PSDRecord[j].s_PSDLevelOfTone)
      {
         if (i < s_NumOfBands)
         {
            psa_MaxNomPSDBand[i] = pt_PSDDesc->ut_PSDRecord[j].s_PSDLevelOfTone;
         }
      }

      j++;
   }
}

#endif // INCLUDE_CACHE_FUNCTIONS
