/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2004 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   Name:   TxRxMsgHandler.c
*
*   TX and RX message handlers.
*
*-------------------------------------------------------------------------
*/

#include "common.h"
#include "hdlc.h"
#include "gdata.h"
#include "ModulateSocMessages.h"
#include "fifo.h"
#include "vdsl_xception.h"
#ifdef DEBUG_TRAIL
#include "LeaveStatesTrail.h"
#endif //DEBUG_TRAIL
#include "PrintTrail.h"
#include "ghs.h"
#include "profile.h"

/*
*-------------------------------------------------------------------------
*
*   TX HDLC routines.
*
*-------------------------------------------------------------------------
*/

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : TxHDLCMsgHandler
 *
 *  Prototype:  void TxHDLCMsgHandler(void)
 *
 *  This function transmits the TX message
 *
 *  Input Arguments:
 *      s_NumOctetsInPayload -- no. of octets in message payload field
 *
 *  Output Arguments:
 *      NONE
 *
 *   Return:
 *      NONE
 *
 *  Global Variables Used:
 *      gpuca_TxSocMsg -- (I) pointer to the TX message byte array
 *
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */
void TxHDLCMsgHandler(void)
{
   uint8 uca_octet[24];
   int16 i;

   for(i=0; i<gs_TxNumBytesPerSymbol; i++)
   {
      uca_octet[i] = GetNextHDLCTxOctet((uint8 *)(gpuca_TxSocMsg));
      //if(gs_TxState == 69)   only for ERB SOC debugging
      //gpsa_MonitorConstallation[gs_Debug1++]  = uca_octet[i];
   }
#ifdef DEBUG_TRAIL
   LeaveHDLCOctetTrail(DEBUG_TRAIL_TXHDLC_ENABLE, uca_octet[0], uca_octet[1], gs_TxNumBytesPerSymbol);
#endif //DEBUG_TRAIL
   ModulateHdlcOctets(uca_octet, gs_TxNumBytesPerSymbol);
}



/*
*-------------------------------------------------------------------------
*
*   RX HDLC routines.
*
*-------------------------------------------------------------------------
*/

/*^^^
 *------------------------------------------------------------------------
 *
 *  Name : RxHDLCMsgHandler
 *
 *  Prototype:  void RxHDLCMsgHandler(void)
 *
 *  This function performs the message decoding.
 *
 *  Input Arguments:
 *      NONE
 *
 *  Output Arguments:
 *      NONE
 *
 *   Return:
 *      NONE
 *
 *  Global Variables Used:
 *      gpuca_RxHDLCMsgBuf -- (O) pointer to RX HDLC message octet buffer containing the decoded message octets
 *      RxHDLCFifo -- (O) RX HDLC message structure containing information on the decoded message buffer
 *      gpsa_RxToneBuf -- (I/O) RX Tone buffer (containing unscrambled and then scrambled tones)
 *      gs_NumOfToneClustersForMsgDecode -- (I) number of consecutive tones to be descrambled
 *      gs_RxHDLCMsgState -- (I/O) state of the algorithm operation
 *      gs_RxMsgCheckOkFlag -- (O) flag to indicate the FCS check result
 *      guc_RxMsgCode -- (I) the message code to be decoded
 *  Notes:
 *
 *------------------------------------------------------------------------
 *^^^
 */

void RxHDLCMsgHandler(void)
{
   static uint8 uc_StartDecodingNextMsgNow = 0;
   uint8 uca_byte[2];

   //Decode the tones to get the message byte(s)
   DecodeSOCMessage(gpsa_RxToneBuf, gs_NumOfToneClustersForMsgDecode, gs_RxNumBytesPerSymbol, uca_byte);
#ifdef DEBUG_TRAIL
   LeaveHDLCOctetTrail(DEBUG_TRAIL_RXHDLC_ENABLE, uca_byte[0], uca_byte[1], gs_RxNumBytesPerSymbol);
#endif //DEBUG_TRAIL
   switch (gs_RxHDLCMsgState)
   {

   case RX_HDLC_MSG_DECODE_MSG_INIT:

      //Initialize the message buffer
      InitHDLCFifo(gpuca_RxHDLCMsgBuf, &RxHDLCFifo); // Decoded msg will reside in guca_RxHDLCMsgBuf[].
      if (uc_StartDecodingNextMsgNow)
      {
         RxHDLCFifo.s_StartFlag = 1;
         uc_StartDecodingNextMsgNow = 0;
      }

      //Set segment information to 0
      guc_RxSegmentIdx = 0;
      guc_RxNumOfSegments = 0;

      //Indicate no message error
      gs_RxMsgErrorCode = E_CODE_MSG_NO_ERROR;

      //Set the receiving flag to FALSE
      gs_MsgReceiveFlag = FALSE;

      //Set the check message flag to FALSE
      gs_RxMsgCheckOkFlag = FALSE;

      //Set the flag to indicate not transmission of ACK-SEG
      gs_SendAckSegFlag = FALSE;

      //go to next substate
      gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_DEMODULATE_MSG;
      //break;    // Do NOT break initentionally, so that we can start decoding message immediately
      /* no break */

   case RX_HDLC_MSG_DECODE_DEMODULATE_MSG: // demodulate and delineate HDLC msg.

      //Decode the tones to get the message byte(s)

      //For the diag mode, the demodulation of message byte is done in RxStateFunc()
      if(gul_OperationModeStatus_VDSL2 & V2_LOOP_DIAG)
      {
         //wait until a whole byte has been received
         if(gs_RxNumSocBitPerByte < 8)
         {
            break;
         }
         else
         {
            uca_byte[0] = guc_RxSocByte;
         }
      }

      //Process the received message bytes
      ReceiveHDLCOctet(uca_byte[0], gpuca_RxHDLCMsgBuf, &RxHDLCFifo);

      if(gs_RxNumBytesPerSymbol == 2)
      {
         ReceiveHDLCOctet(uca_byte[1], gpuca_RxHDLCMsgBuf, &RxHDLCFifo);
      }

      //Check if the number of segment exceeds the limit
      if((RxHDLCFifo.uc_NumOfSegments == SOC_MAX_NUM_MSG_SEGMENTS) && (guc_RxSegmentIdx != guc_RxNumOfSegments))
      {
         gs_RxMsgErrorCode = E_CODE_MSG_RX_SEG_LIMIT;

         gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_LEAVE_TRAIL;
      }

      if (RxHDLCFifo.uc_NumOfSegments > guc_RxSegmentIdx)
      {

         //Check message header information (msg index, msg segment number and message code)
         gs_RxMsgCheckOkFlag = CheckMsgCode(&RxHDLCFifo);

         if(gs_RxMsgCheckOkFlag == TRUE)
         {
            // perform the FCS check in background whenever each segment is received
            gs_RxBkgdProcessFlag = MSG_DECODE_IN_PROGRESS;
            AddFunctionToBkgdFifo((PtrToBkgdFunc)BkgdCheckRxHDLCMsgFCS);

            //go to next substate
            gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_CHECK_MSG_FCS;
         }
         else
         {
            //if the message is not expected, ignore it, and restart the decoding process
            uc_StartDecodingNextMsgNow = 1;
            gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_MSG_INIT;
         }
      }

      break;

   case RX_HDLC_MSG_DECODE_CHECK_MSG_FCS:

      if(gs_RxBkgdProcessFlag == MSG_DECODE_DONE)
      {

         if (gs_RxMsgCheckOkFlag == TRUE)
         {

            //FCS match,
            if((gus_ModemOperationMode_Status & MODEM_OPERATION_MODE_VDSL2) &&
                  (guc_RxMsgCode == VDSL2_SOC_MSG_O_ACK))
            {
               //Since O-ACK only contains the message code, skip interpretation
               //set the flag to indicate that message is received correctly


               gs_MsgReceiveFlag = TRUE;
               gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_LEAVE_TRAIL;
            }

            else if(guc_RxMsgCode == SOC_MSG_ACK_SEG)
            {
               gs_MsgReceiveFlag = TRUE;

               //Set the flag to let the TX side know ACK_SEG has been received
               if(gs_RevAckSegFlag == RECEIVE_ACK_SEG_EXP)
               {
                  gs_RevAckSegFlag = RECEIVE_ACK_SEG_DONE;
               }

               gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_LEAVE_TRAIL;
            }
            //If all the segment is received, perform interpretion
            else if(guc_RxSegmentIdx == guc_RxNumOfSegments)
            {
               // Bypass message interpretation in sniffer mode
               if (gul_dbgSnifferControl & SNIFFER_RX_BYPASS_SOC_MSG_INTERP)
               {
                  gs_RxMsgCheckOkFlag = TRUE;
                  gs_RxBkgdProcessFlag = MSG_DECODE_DONE;
               }
               //perform the message interpretation in background
               else
               {
                  gs_RxBkgdProcessFlag = MSG_DECODE_IN_PROGRESS;
                  AddFunctionToBkgdFifo((PtrToBkgdFunc)BkgdInterpretMsg);
               }

               // Go to interpret.
               gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_INTERPRET_MSG;
            }
            else
            {
               //Set this flag to true to indicate the TX need to send ACK-SEG for the messages
               //sent in the RQ mode
               if((guc_RxMsgCode != VDSL2_SOC_MSG_O_SIGNATURE) && (guc_RxMsgCode != VDSL2_SOC_MSG_R_MSG))
               {
                  gs_SendAckSegFlag = TRUE;
               }

               gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_DEMODULATE_MSG;
            }
         }
         else
         {
            //Set error code
            gs_RxMsgErrorCode = E_CODE_MSG_FCS_ERROR;

            // Save msg info in case of FCS error
            // Note: uc_LastGoodMsgCode logs the last correctly decoded
            // msgcode before the current FCS error occurs.
            gt_RxSocMsgFCSErrInfo.uc_LastGoodMsgCode = guc_RxMsgCode;
            gt_RxSocMsgFCSErrInfo.uc_MsgSegmentIdx   = guc_RxSegmentIdx;
            gt_RxSocMsgFCSErrInfo.uc_NumMsgSegments  = guc_RxNumOfSegments;

            //FCS is not match, go to idle
            gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_LEAVE_TRAIL;
         }
      }

      break;

   case RX_HDLC_MSG_DECODE_INTERPRET_MSG:

      if(gs_RxBkgdProcessFlag == MSG_DECODE_DONE)
      {
         //If the message is interpolated ok, go to the state to receive next message
         if(gs_RxMsgCheckOkFlag == TRUE)
         {
            //set the flag to indicate that message is received correctly
            gs_MsgReceiveFlag = TRUE;
         }
         else
         {
            if(gus_OSignatureErrorCode == O_SIG_FIELD21_VEC_XTALK_LINK)
            {
               EnterFailStates(E_CODE_RX_VECTORING_XTALK_LINK);
            }
         }

         gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_LEAVE_TRAIL;
      }
      break;

   case RX_HDLC_MSG_DECODE_LEAVE_TRAIL:
#ifdef DEBUG_TRAIL
      // log SOC info into gsa_StatesTrail
      LeaveRxSOCTrail(guc_RxMsgCode, gs_RxMsgErrorCode, gs_MsgReceiveFlag);
#endif //DEBUG_TRAIL

      if(guc_RxMsgCode == SOC_MSG_ACK_SEG)
      {
         gul_RxTimer = 0;
         gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_MSG_INIT;
      }
      else
      {
         gs_RxHDLCMsgState = RX_HDLC_MSG_DECODE_IDLE;
      }

      break;

   case RX_HDLC_MSG_DECODE_IDLE:
      break;

   }
}

void TxSocAckSeg(void)
{

   static int16 s_InitFlag = 1;
   uint8 uca_octet[2];

   if(s_InitFlag == 1)
   {
      //Form ACK-SEG message
      gpuca_TxSocMsg[0] = SOC_MSG_ACK_SEG;  // Msg code
      gs_NumOctetsInTxHDLCMsg = 1;      // Set msg buffer length for HDLC encapsulation
      gs_UpdateTxMsgIdxFlag = TRUE;      //increase the message index for each segment

      //Init. HDLC state machine state
      gs_TxHDLCMsgState = HDLC_TX_START_NEW_MSG;

      //Modulate two HDLC flags into one symbol
      uca_octet[0] = (uint8)HDLC_FLAG;
      uca_octet[1] = (uint8)HDLC_FLAG;
      ModulateHdlcOctets(uca_octet, gs_TxNumBytesPerSymbol);

      s_InitFlag = 0;
   }
   else
   {

      //transmit SOC message
      TxHDLCMsgHandler();

      if(gs_TxHDLCMsgState == HDLC_TX_CLOSING_FLAG_DONE)
      {
         s_InitFlag = 1;

         //Indicate that ACK-SEG has been sent
         gs_SendAckSegFlag = FALSE;

      }
   }
}
