/* **COPYRIGHT******************************************************************
    INTEL CONFIDENTIAL
    Copyright (C) 2017 Intel Corporation
    Copyright (C), 1994-2006 Aware Inc. All Rights Reserved.
******************************************************************COPYRIGHT** */
/* **DISCLAIMER*****************************************************************
    The source code contained or described herein and all documents related
    to the source code ("Material") are owned by Intel Corporation or its
    suppliers or licensors. Title to the Material remains with Intel
    Corporation or its suppliers and licensors. The Material may contain
    trade secrets and proprietary and confidential information of Intel
    Corporation and its suppliers and licensors, and is protected by
    worldwide copyright and trade secret laws and treaty provisions. No part
    of the Material may be used, copied, reproduced, modified, published,
    uploaded, posted, transmitted, distributed, or disclosed in any way
    without Intel's prior express written permission.

    No license under any patent, copyright, trade secret or other
    intellectual property right is granted to or conferred upon you by
    disclosure or delivery of the Materials, either expressly, by
    implication, inducement, estoppel or otherwise. Any license under
    such intellectual property rights must be express and approved by
    Intel in writing.
*****************************************************************DISCLAIMER** */
/*
*-------------------------------------------------------------------------
*
*   Aware DMT Technology. Proprietary and Confidential.
*
*   40 Middlesex Turnpike, Bedford, MA 01730-1413 USA
*   Phone (781) 276 - 4000
*   Fax   (781) 276 - 4001
*
*   RMsg1TxF_VDSL2.c
*
*   This file contains RT's R_MSG1_TX state machine (VDSL2).
*
*-------------------------------------------------------------------------
*/

// ***********************************************************************************************************
// RMsg1TxF_VDSL2.c
//
// History
//
// 19/09/2013 Fuss: Global timeout clean-up for vectoring
//            Grep for XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts
// 08/10/2014 Vinay: Added code to support reporting of 3 more parameters (Recieve signal, Transmit signal, SOC message) to STAT 0
//            Grep for XDSLRTFW-1901
// 13/01/2015 Fuss: Glitch at the CPE side maybe happening when CPE is configuring the US-path
//            Grep for XDSLRTFW-2155
// 27/11/2017 Stefan: Upstream performance dip on short loops (KPN)
//            - kl0 measurments with calibrated setup showed that our looplength estimation is
//              resulting in too high kl0 estimations
//            - add BRCM-like PSD boost on short loops (for UPBO and non-UPBO configurations)
//            Grep for XDSLRTFW-3212 : Upstream performance dip on short loops
//
// 15/03/2018 Sriram Shastry: Do changes for KPN operator also for OTE_GREECE and MT_HUNGARY
// - Used TEST_TX_PSD_CONTROL_BRCM_MODE instead of TEST_TX_PSD_CONTROL_KPN_MODE.
// - Changed BRCM PSD boost for US1 and US2, i.e. from 0.5dB (US1) and 1.5dB (US2) to 1.0dB (US1) and 1.0dB
// - No KL0 reduction, i.e. bit "TEST_TX_PSD_CONTROL_REDUCE_KL0_BY_10_PERC" should not be set
// Search for XDSLRTFW-3740
//
// 13/06/2018 Hanyu Liu: Added changes for CTL operator to use TEST_TX_PSD_CONTROL_CTL_MODE to boost US1&US2 by 1.0dB.
// Search for XDSLRTFW-3833
//
// 10/10/2018 Abu Rahman
// XDSLRTFW-3990:  TR115A: sync losses in AA12a and AA17a due to tx path clipping
// On top of Tx PSD compensation, FW uses different Tx PSD boost to meet the customer requirement (see XDSLRTFW-3212, XDSLRTFW-3742,XDSLRTFW-3833).
// These boosts are frequency or band dependent so that conventionally these are done in frequency domain. These boosts could lead
// overflow in IFFT block which can cause US CRC as well as DS CRC. To minimize the risk of IFFT block overflow, shift this extra
// Tx PSD boost in time domain (i.e. IFFT scaling, Tx vargain, AFE gain etc.) and boost the frequency domain part in negative direction ( i.e. reduction).
//
// Example behavior if the power is reduced in frequency domain by 1 dB and  increased in time domain in same amount
// Prio 1: IFFT-gain may increase by 6 dB if FD power reduction allows one more step
//         (in this case we would expect that tx-vargain is reduced by 5 dB)
// Prio 2: if increase of IFFT gain is NOT possible, tx-vargain should increase by 1 dB
// Prio 3: if increase of tx-vargain is NOT possible, afe-gain should increase by 1 dB
// Search for XDSLRTFW-3990
//
// ************************************************************************************************************

#include "common.h"
#include "gdata.h"
#include "fifo.h"
#include "TimingAdvanceHandler.h"
#include "TxPSDCOntrol.h"
#include "PGAHandler.h"
#include "ModulateSocMessages.h"
#include "vdsl_state.h"
#include "RTrainingTxF.h"
#include "V_STR_IIR_IOf.h"
#include "IRI_Iof.h"
#include "V_STR_IOf.h"
#include "DSLEngin.h"
#include "ghs.h"
#include "SharedFuncs.h"
#include "cmv.h"
#include "DshInterface.h"
#include "LL_IOf.h"

/*
*-------------------------------------------------------------------------------
*
*  Prototype: void RMsg1TxF_VDSL2(void)
*
*  This function implements three state machines.
*
*   i.   R_P_QUITE1_TX          (159) : Contains R-P Quiet 1 related code
*   ii.  VDSL2_R_P_VECTOR1_TX   (190) : Contains R-P Vector 1 related code
*   iii. R_MSG1_TX              (10)  : Contanis R-P Channel Discovery 1 related code
*
*  Input Arguments:
*
*  Output Arguments:
*
*  Returns:
*
*  Global Variables:
*
*-------------------------------------------------------------------------------
*/

// Sub state definations of R_P_QUITE1_TX tx state
#define R_P_QUITE1_TX_INIT                   (0)
#define R_P_QUITE1_TX_WAIT_FOR_OSIGNATURE    (1)
#define R_P_QUITE1_TX_TIMING_ADVANCE         (2)
#define R_P_QUITE1_TX_SET_TXGAINS_INIT       (3)

// Sub state definations of VDSL2_R_P_VECTOR1_TX state
#define R_P_VECTOR1_TX_INIT                  (0)
#define R_P_VECTOR1_TX_SIGNAL                (1)

// Sub state definations of R_MSG1_TX tx state
#define R_MSG1_TX_SET_TXGAINS                (0)
#define R_MSG1_TX_WAIT_FOR_PGA_TRAINED       (1)
#define R_MSG1_TX_READY_TO_SEND_MSG          (2)


void RMsg1TxF_VDSL2(void)
{
   uint8  uc_octet;
   int16  s_temp;
   int16  s_RevSeg;
   uint16 us_TxPSD_Mode;
   int16  s_Use_US0_ONLY_TxFilter;

   int16 s_MaxUsPSDBoost = 0;
   int16 s_BandCnt = 0;


   if( gs_TxState == R_P_QUITE1_TX)       // R-P Quiet 1 state
   {
//XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
      gsa_IndirectStat0[4]=VDSL2_R_P_QUIET1;
      switch (gs_TxSubState)
      {

      case R_P_QUITE1_TX_INIT:

         // Initialize flag to false for O-SIGNATURE
         guc_TxRxCommunicator = TX_COMMUNICATOR_RESET;

         // log start of Channel Discovery phase
         gpt_DebugTiming->l_TxRMsg1_Start = gl_TotalTxSymbolCount;

         AddFunctionToFifo(gp_TxLoadingFunctionFifo, InitTrainTxF);
         gs_TxSubState = R_P_QUITE1_TX_WAIT_FOR_OSIGNATURE;
         gs_TxSubStateCnt = 0;
         break;

      case R_P_QUITE1_TX_WAIT_FOR_OSIGNATURE:

         if (!(guc_TxRxCommunicator & TX_IIRFILTER_RECONF_COMPLETED))
         {
            // If the O-SIGNATURE Rx Message has been demodulated,
            // then training, including frame alignment, has been performed
            // run TimingAdvanceHandler to align Tx/Rx frame boundaries.
            if (guc_TxRxCommunicator & TX_IIRFILTER_RECONF_START)
            {
               if(gs_TxSubStateCnt == 0)
               {
                  // log start of Tx configuration
                  gpt_DebugTiming->l_TxRMsg1_TxConfig = gl_TotalTxSymbolCount;
                  // Bypass the HW settle time, if no tx filter auto-reconfig is enabled!
                  // Note: For Tx filter auto-reconfig the variable gets be cleared again.

                  if(TESTArray[TEST_JIRA2257_CONFIG] & TEST_JIRA2257_CLEAR_US0_ONLY)
                  {
                     gus_UseUS0OnlyCntrl = 0;
                  }

                  if (gsa_Optn4_FilterControl[OPTN_4_IDX1_FILTER_CTRL] & OPTN_Tx_Filter_Auto_Reconfig)
                  {
                     // For profiles 17a/12x/8x and 35B
                     {
                        int16 s_TxIIRFilterSelect;
                        int16 s_FirstSupportedUS0Channel, s_LastSupportedUS0Channel;

                        s_TxIIRFilterSelect = IIR_FILTER_NO_CHANGE;
                        s_FirstSupportedUS0Channel = gsa_TxBandLeftChannel[0];
                        s_LastSupportedUS0Channel = gsa_TxBandRightChannel[0];

                        // --------------+-------+-----------+--------------------------------------- +----+
                        // Profile       | US0   | US1       | Used TxIIR-filter                      | Idx|
                        // --------------+-------+-----------+--------------------------------------- +----+
                        //  8/12/17MHz   | oPOTS | not used  | OPTN_POTS_LP_Filter_Select             |  7 |
                        //  8/12/17MHz   | EU64  | not used  | OPTN_POTS_DOUBLE_LP_Filter_Select      |  9 |
                        //  8/12/17MHz   | EU128 | not used  | OPTN_POTS_QUAD_LP_Filter_Select        | 10 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        //  8/12/17MHz   | oPOTS | in use    | OPTN_POTS_Filter_Select                |  1 |
                        //  8/12/17MHz   | EU64  | in use    | OPTN_POTS_DOUBLE_Filter_Select         |  2 |
                        //  8/12/17MHz   | EU128 | in use    | OPTN_BYPASS_Filter_Select              |  0 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        //  8/12/17MHz   | oISDN | not used  | OPTN_ISDN_LP_Filter_Select             |  8 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        //  8/12/17MHz   | oISDN | in use    | OPTN_ISDN_Filter_Select                |  3 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        // 35MHz         | oPOTS | not used  | OPTN_35B_POTS_LP_Filter_Select         | 11 |
                        // 35MHz         | EU64  | not used  | OPTN_35B_POTS_DOUBLE_LP_Filter_Select  | 13 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        // 35MHz         | oPOTS | in use    | OPTN_35B_VDSL_Filter_Select            |  4 |
                        // 35MHz         | EU64  | in use    | OPTN_35B_VDSL_POTS_DOUBLE_Filter_Select|  5 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        // 35MHz         | oISDN | not used  | OPTN_35B_ISDN_LP_Filter_Select         | 12 |
                        // --------------+-------+-----------+--------------------------------------- +----+
                        // 35MHz         | oISDN | in use    | OPTN_35B_VDSL_Filter_Select            |  6 |
                        // --------------+-------+-----------+--------------------------------------- +----+

                        s_Use_US0_ONLY_TxFilter = 0;
                        if ((!(TESTArray[TEST_JIRA2257_CONFIG] & TEST_JIRA2257_VDSL_FILTER)) &&
                            (gus_UseUS0OnlyCntrl & TX_US0_ONLY_CONF_IIR))
                        {
                           s_Use_US0_ONLY_TxFilter = 1;
                        }

                        // A different TX filter depending on the TX band information must be selected!
                        if ( (gt_ProfileAct.us_ProfileSelected & (CNFG_V2_PROFILE_8xALL_MASK|CNFG_V2_PROFILE_12xALL_MASK|CNFG_V2_PROFILE_17A_MASK)) &&
                             (gft_Debug35bHwConfig17aModeEnable == FALSE)) //XDSLRTFW-3457 (Start_End)
                        {
                           // Choose the correct US0 band Tx filter
                           // Note: No reconfiguration should take place, when the Tx IIR is already loaded!
                           //       To be independent of code changes at other places the "!=" is used in every if-case.
                           if (s_FirstSupportedUS0Channel <= US0_POTS_ISDN_DETECT_TONE)
                           {
                              // POTS profile
                              if (s_LastSupportedUS0Channel <= US0_POTS_MAX_TONE)
                              {
                                 if (s_Use_US0_ONLY_TxFilter)
                                    s_TxIIRFilterSelect = OPTN_POTS_LP_Filter_Select;
                                 else
                                    s_TxIIRFilterSelect = OPTN_POTS_Filter_Select;
                              }
                              else if(s_LastSupportedUS0Channel <= US0_ISDN_POTSDOUBLE_MAX_TONE)
                              {
                                 if (s_Use_US0_ONLY_TxFilter)
                                    s_TxIIRFilterSelect = OPTN_POTS_DOUBLE_LP_Filter_Select;
                                 else
                                    s_TxIIRFilterSelect = OPTN_POTS_DOUBLE_Filter_Select;
                              }
                              else if(s_LastSupportedUS0Channel <= US0_ISDNDOUBLE_POTSQUAD_MAX_TONE)
                              {  //XDSLRTFW-3496 Start
                                 if (s_Use_US0_ONLY_TxFilter)
                                    s_TxIIRFilterSelect = OPTN_POTS_QUAD_LP_Filter_Select;
                                 else
                                    s_TxIIRFilterSelect = OPTN_BYPASS_Filter_Select;
                           //XDSLRTFW-3496 End
                              }
                           }
                           else
                           {
                              // ISDN profile or NO US0
                              if (s_Use_US0_ONLY_TxFilter)
                                 s_TxIIRFilterSelect = OPTN_ISDN_LP_Filter_Select;
                              else
                                 s_TxIIRFilterSelect = OPTN_ISDN_Filter_Select;
                           }
                        } // End 8x, 12x and 17a
                        else if ((gt_ProfileAct.us_ProfileSelected & CNFG_V2_PROFILE_35B_MASK) || (gft_Debug35bHwConfig17aModeEnable == TRUE)) //XDSLRTFW-3457 (Start_End)
                        {
                        // Choose the correct US0 band Tx filter
                        // Note: No reconfiguration should take place, when the Tx IIR is already loaded!
                        //       To be independent of code changes at other places the "!=" is used in every if-case.
                           if (s_FirstSupportedUS0Channel <= US0_POTS_ISDN_DETECT_TONE)
                           {
                           //XDSLRTFW-3457 (Start)
                              // POTS profile
                              if (s_LastSupportedUS0Channel <= US0_POTS_MAX_TONE)
                              {
                                 if (s_Use_US0_ONLY_TxFilter)
                                    s_TxIIRFilterSelect = OPTN_35B_POTS_LP_Filter_Select;
                                 else
                                    s_TxIIRFilterSelect = OPTN_35B_VDSL_Filter_Select;
                              }
                              else if(s_LastSupportedUS0Channel <= US0_ISDN_POTSDOUBLE_MAX_TONE)
                              {
                                 if (s_Use_US0_ONLY_TxFilter)
                                    s_TxIIRFilterSelect = OPTN_35B_POTS_DOUBLE_LP_Filter_Select; //XDSLRTFW-3570
                                 else
                                    s_TxIIRFilterSelect = OPTN_35B_VDSL_POTS_DOUBLE_Filter_Select; //XDSLRTFW-3715
                              }
                           //XDSLRTFW-3457 (End)
                           }
                           else
                           {
                              //XDSLRTFW-3457 (Start)
                              // ISDN profile or NO US0
                                 if (s_Use_US0_ONLY_TxFilter)
                                 {
                                    s_TxIIRFilterSelect = OPTN_35B_ISDN_LP_Filter_Select;
                                 }
                                 else if (s_FirstSupportedUS0Channel > US0_ISDNDOUBLE_POTSQUAD_MAX_TONE)
                                 {
                                    //XDSLRTFW-3792 In No US0 case, with VDSL ISDN filter we see strong SNR degredation in DS3 because of TX mirroring effect
                                    //35b VDSL POTS Filter has 70dB OOB attenution
                                    //35b VDSL ISDN Filter has ~60dB OOB Attenuation
                                    //Work Around is to use the 35b VDSL POTS Filter incase case No US0
                                    s_TxIIRFilterSelect = OPTN_35B_VDSL_Filter_Select;
                                 }
                                 else
                                 {
                                    s_TxIIRFilterSelect = OPTN_35B_VDSL_ISDN_Filter_Select;//XDSLRTFW-3715
                                 }
                              //XDSLRTFW-3457 (End)
                           }
                        } // end 35B

                        if (s_TxIIRFilterSelect >= OPTN_POTS_LP_Filter_Select)
                        {
                           //-------------------------------------------+-----+-------------+
                           // Filter                                    | Idx | POFI Cutoff |
                           //-------------------------------------------+-----+-------------+
                           // OPTN_POTS_LP_Filter_Select                |   7 |      80 kHz |
                           // OPTN_ISDN_LP_Filter_Select                |   8 |     320 kHz |
                           // OPTN_POTS_DOUBLE_LP_Filter_Select         |   9 |     320 kHz |
                           // OPTN_POTS_QUAD_LP_Filter_Select           |  10 |     552 kHz |
                           // OPTN_35B_POTS_LP_Filter_Select            |  11 |      80 kHz |
                           // OPTN_35B_ISDN_LP_Filter_Select            |  12 |     320 kHz |
                           // OPTN_35B_POTS_DOUBLE_LP_Filter_Select     |  13 |     320 kHz |
                           //-------------------------------------------+-----+-------------+

                           // gus_AFE_TxMode
                           // BYPASSED           (0)
                           // VDSL35b_CPE        (1)
                           // VDSL17_CPE         (2)
                           // VDSL8_CPE          (3)
                           // ADSL2_CPE          (4)

                           if (gus_DebugControlVRX518 & TX_US0_POFI_FC_SELECT_EN) //XDSLRTFW-3352
                           {
                              gs_TxPathAfeDfeCustomSettings |= AFE_POFI_MODE_ADSL2_CPE;

                              if((s_TxIIRFilterSelect == OPTN_POTS_LP_Filter_Select) ||(s_TxIIRFilterSelect == OPTN_35B_POTS_LP_Filter_Select))
                              {
                                 gla_fc_pofi_lp_kHz[gus_AFE_TxMode] = 80;
                              }
                              else if(s_TxIIRFilterSelect == OPTN_POTS_QUAD_LP_Filter_Select)
                              {
                                 gla_fc_pofi_lp_kHz[gus_AFE_TxMode] = 552;
                              }
                              else
                              {
                                 gla_fc_pofi_lp_kHz[gus_AFE_TxMode] = 320;
                              }
                           }
                        }

                        if (s_TxIIRFilterSelect != IIR_FILTER_NO_CHANGE)
                        {
                           //Update CMV and variable
                           gs_TxIIRFilterSelect = s_TxIIRFilterSelect;
                           gsa_Optn4_FilterControl[OPTN_4_IDX4_FILTER_CD] = (gsa_Optn4_FilterControl[OPTN_4_IDX4_FILTER_CD] & (~OPTN_TX_MASK_Filter_Select));
                           gsa_Optn4_FilterControl[OPTN_4_IDX4_FILTER_CD] |= s_TxIIRFilterSelect;
                           //Set filter id before "SetTransmitGains"
                           //"LoadTxIIR" before "TimingAdvanceHandler"
                           AddFunctionToFifo(gp_TxLoadingFunctionFifo, LoadTxIIR);
                        }

                     } //17M/12M/8M and 35M
                  } //if (gsa_Optn4_FilterControl[OPTN_4_IDX1_FILTER_CTRL] & OPTN_Tx_Filter_Auto_Reconfig)

               } // if(gs_TxSubStateCnt == 0)

               if (gs_TxSubStateCnt > 4)
               {
                  // Settle time over
                  guc_TxRxCommunicator |= TX_IIRFILTER_RECONF_COMPLETED;
               }
               gs_TxSubStateCnt++;

            } // if (guc_TxRxCommunicator & TX_IIRFILTER_RECONF_START)
         }
         else if ( ((gft_DSVectoringEnabled == TRUE) && (guc_TxRxCommunicator & TX_TIMING_ADVANCE_START)) ||
                   ((gft_DSVectoringEnabled == FALSE) && (guc_TxRxCommunicator & TX_RMSG1_START)) )
         {
            gs_TxSubStateCnt = 0;
            gs_TimingAdvance = 0;
            gs_TimingAdvanceHandlerState = TIMING_ADVANCE_INIT;

            gs_TxSubState = R_P_QUITE1_TX_TIMING_ADVANCE;
         }
         break;

      case R_P_QUITE1_TX_TIMING_ADVANCE:

         if (gs_TimingAdvanceHandlerState != TIMING_ADVANCE_DONE)
         {
            TimingAdvanceHandler();
         }
         else
         {
            // Setup TxPSDControl control structure used for Transmit Signal
            gt_TxPsdControl.s_MaxNomPsdIn = gt_PwrConfigParam.s_Up_MaxNomPSD;

            // Increase the initial US PSD from the default -52 dBm/Hz to help the CO signal
            // detection on long loops.
            // Sliding scale with 1dB of PSD ceiling increase per 2 dB of calculated attenuation:
            // Start linear scale with the default -52 dBm/Hz at 50dB kl0 (~2000m), -51 dBm/Hz at 52dB kl0,
            // up to -44 dBm/Hz at 66dB kl0 (~2600m)
            // Enable the sliding initial US ceiling only for band plans with single US0 over POTS.
            // Link failures on long loops have been observed with double US0 band plans if the initial US ceiling
            // is increased from the default -52 dBm/Hz value.
            if ((gft_SlidingUsInitialCeiling == TRUE) && (gs_frame_rate_is_8khz == 0) && (gsa_TxBandRightChannel[0] <= 32))
            {
               s_temp = (gt_Kl0ElectricalLength.s_kl0_estimate - 500) >> 1;
               if (s_temp > 0)
               {
                  gt_PwrConfigParam.s_Up_InitialPsdCeiling = MAX( INITIAL_UP_PSDCEIL-s_temp , gs_MAX_US0_CEILING-10 );
               }
            }

            // Standard says, the CDPSDus should be 'limited to' min(PSDMASKus,MAXPSDus) and
            // that REFPSDus should limited to (min(PSDMASKus,MAXMASKus)-3.5dB)
            // Hence, we compute CDPSDus/REFPSDus by equating it to the limit
            gt_TxPsdControl.s_PsdCeiling = gt_PwrConfigParam.s_Up_InitialPsdCeiling;
            gt_TxPsdControl.pt_MaxPsdDescIn = (PSDDescriptorTable_t*)(void *)&gt_MaxUsPSDDescriptorTable;
            if (gt_UPBOPSDDescriptorTable.us_NumberOfBands)
            {
               gt_TxPsdControl.pt_RefPsdDescIn = (UPBOPSDDescriptorTable_t*)&gt_UPBOPSDDescriptorTable;
               gt_TxPsdControl.s_kl0In = gt_Kl0ElectricalLength.s_kl0_estimate;
               // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (START)
               // the ELE from the CO side is available first only after O-UPDATE
               // here we are after O-SIGNATURE and will start calculation of UPBO based on the calculated kl0
               gt_TxPsdControl.usa_Kl0EstimOPb[0] = gt_Kl0ElectricalLength.s_kl0_estimate;
               gt_TxPsdControl.usa_Kl0EstimOPb[1] = gt_Kl0ElectricalLength.s_kl0_estimate;
               gt_TxPsdControl.usa_Kl0EstimOPb[2] = gt_Kl0ElectricalLength.s_kl0_estimate;
               gt_TxPsdControl.usa_Kl0EstimOPb[3] = gt_Kl0ElectricalLength.s_kl0_estimate;
               // if we are in us_AeleMode = 1,2,3 we need to use for kl0 the calculated ELEDS = min{CPE}
               if(gt_AELEM_UPBOInfo.us_AeleMode != 0)
               {
                  uint16 ELEDsMinLocal = gus_ELEDsMin;
                  // let's do capping:
                  if(ELEDsMinLocal < 18)
                  {
                     // If the estimated value of kl0 is smaller than 1.8 dB, the VTU shall be allowed to perform power
                     // back-off as if kl0 were equal to 1.8 dB                     
                     ELEDsMinLocal = 18;
                  }
                  if( ELEDsMinLocal > 1280)
                  {
                     // The valid range of values is from 0 dB to 128 dB with a 0.1 dB step
                     ELEDsMinLocal = 1280;
                  }
                  // for all bands we apply this ELEDS
                  gt_TxPsdControl.usa_Kl0EstimOPb[0] = ELEDsMinLocal;
                  gt_TxPsdControl.usa_Kl0EstimOPb[1] = ELEDsMinLocal;
                  gt_TxPsdControl.usa_Kl0EstimOPb[2] = ELEDsMinLocal;
                  gt_TxPsdControl.usa_Kl0EstimOPb[3] = ELEDsMinLocal;
               }
               // XDSLRTFW-487_VR9_VRX318_VDSL2_All_AELEM_Support (END)
            }
            else
            {
               gt_TxPsdControl.pt_RefPsdDescIn = NULL;
            }
            gt_TxPsdControl.pt_ActPsdDescOut = (PSDDescriptorTable_t*)(void *)&gt_UsREFPSDDescriptorTable;
            // Set MAXNOMATP
            gt_TxPsdControl.s_MaxAtp = gt_PwrConfigParam.s_Up_MaxNomAggrPwr;

            // get TxPSD Mode (bits 8-11 of TESTArray[TEST_TX_PSD_CONTROL])
            us_TxPSD_Mode = (TESTArray[TEST_TX_PSD_CONTROL] >> 8) & 0x000F;

            //-----------------------+----------+--------+--------+--------+--------+
            // Mode                  | kl0      | US0    | US1    | US2    | US3    |
            //-----------------------+----------+--------+--------+--------+--------+
            // DEFAULT               | all      | 0.0 dB | 0.0 dB | 0.0 dB | 0.0 dB |
            //-----------------------+----------+--------+--------+--------+--------+
            // BRCM                  | all      | 0.0 dB | 1.0 dB | 1.0 dB | 1.0 dB |
            //-----------------------+----------+--------+--------+--------+--------+
            // BRCM_MODE_VECTORING   | all      | 0.0 dB | 1.0 dB | 1.0 dB | 1.0 dB |
            //-----------------------+----------+--------+--------+--------+--------+
            // KPN                   | < 3.0 dB | 0.0 dB | 0.5 dB | 1.5 dB | 1.5 dB |
            //                       | < 4.0 dB | 0.0 dB | 0.4 dB | 1.2 dB | 1.2 dB |
            //                       | < 6.0 dB | 0.0 dB | 0.3 dB | 0.9 dB | 0.9 dB |
            //                       | < 8.0 dB | 0.0 dB | 0.2 dB | 0.6 dB | 0.6 dB |
            //                       | <10.0 dB | 0.0 dB | 0.1 dB | 0.3 dB | 0.3 dB |
            //                       | >10.0 dB | 0.0 dB | 0.0 dB | 0.0 dB | 0.0 dB |
            //-----------------------+----------+--------+--------+--------+--------+
            // CTL                   | all      | 0.0 dB | 1.0 dB | 1.0 dB | 1.0 dB |
            //-----------------------+----------+--------+--------+--------+--------+
            // XDSLRTFW-3212 : Upstream performance dip on short loops
            //                 add KPN or BRCM boost on top.
            {
               int16  s_PSD_boost_US1;
               int16  s_PSD_boost_US2_US3;

               s_PSD_boost_US1      =  0;     // 0.3dB boost
               s_PSD_boost_US2_US3  =  0;     // 0.9dB boost => target
               // KPN mode:
               // BRCM mode on short loops / Intel mode on mid to long loops
               if ( us_TxPSD_Mode == TEST_TX_PSD_CONTROL_KPN_MODE )
               {
                  if (gt_Kl0ElectricalLength.s_kl0_estimate < 30)               // <  3dB
                  {
                     s_PSD_boost_US1      =  5;     // 0.5dB boost
                     s_PSD_boost_US2_US3  = 15;     // 1.5dB boost => target
                  }
                  else if (gt_Kl0ElectricalLength.s_kl0_estimate < 40)          // <  4dB
                  {
                     s_PSD_boost_US1      =  4;     // 0.4dB boost
                     s_PSD_boost_US2_US3  = 12;     // 1.2dB boost => target
                  }
                  else if (gt_Kl0ElectricalLength.s_kl0_estimate < 60)          // <  6dB
                  {
                     s_PSD_boost_US1      =  3;     // 0.3dB boost
                     s_PSD_boost_US2_US3  =  9;     // 0.9dB boost => target
                  }
                  else if (gt_Kl0ElectricalLength.s_kl0_estimate < 100)         // < 10dB
                  {
                     s_PSD_boost_US1      =  2;                                 // 0.2dB boost
                     s_PSD_boost_US2_US3  =  6;                                 // 0.6dB boost => target
                  }
               }

               // XDSLRTFW-3742: Adapt PSD boost in US1, US2, US3 if BRCM PSD mode is enabled.
               //                Or if new BRCM_MODE_IN_VECTORING_ONLY mode is enabled and vectoring is enabled.
               //                Both BRCM PSD modes are disabled by default.
               if (( us_TxPSD_Mode == TEST_TX_PSD_CONTROL_BRCM_MODE )     // BRCM mode by default disabled
                     || (( us_TxPSD_Mode == TEST_TX_PSD_CONTROL_BRCM_MODE_IN_VECTORING_ONLY ) && (gft_DSVectoringEnabled == TRUE) ))
               {
                  s_PSD_boost_US1      = 10;    // 1.0 dB boost according recent measurements (was 0.5dB)
                  s_PSD_boost_US2_US3  = 10;    // 1.0 dB boost according recent measurements (was 0.5dB)
               }

               if ( us_TxPSD_Mode == TEST_TX_PSD_CONTROL_CTL_MODE )     // CTL mode by default disabled
               {
                  // XDSLRTFW-3833 CTL qualification tests
                  s_PSD_boost_US1      = 10;                                    // 1.0 dB
                  s_PSD_boost_US2_US3  = 10;                                    // 1.0 dB boost => target
               }

               gt_TxPsdControl.s_US_PSD_Boost_0dB1[0] += 0;
               gt_TxPsdControl.s_US_PSD_Boost_0dB1[1] += s_PSD_boost_US1;
               gt_TxPsdControl.s_US_PSD_Boost_0dB1[2] += s_PSD_boost_US2_US3;


               // XDSLRTFW-3990:  TR115A: sync losses in AA12a and AA17a due to tx path clipping (Start)
               // Find maximum band dependent Tx PSD boost
               s_MaxUsPSDBoost = gt_TxPsdControl.s_US_PSD_Boost_0dB1[0];
               for (s_BandCnt = 1 ; s_BandCnt < 5; s_BandCnt++ ) // only up to 5 band is considered(!) due to array size of s_US_PSD_Boost_0dB1[] variable
               {
                  if (gt_TxPsdControl.s_US_PSD_Boost_0dB1[s_BandCnt] >= s_MaxUsPSDBoost)
                  {
                     s_MaxUsPSDBoost = gt_TxPsdControl.s_US_PSD_Boost_0dB1[s_BandCnt];
                  }
               }

               gt_TxPsdControl.s_US_PSD_Boost_TimeDomain = 0; // no time domain boost

               if(s_MaxUsPSDBoost >=0)
               {
                  // Initialize time domain frequency independent Tx boost
                  gt_TxPsdControl.s_US_PSD_Boost_TimeDomain = (s_MaxUsPSDBoost << 8)/10; // boost in dB in 8.8 dB format (+ve : boost PSD)

                  // Since frequency independent s_MaxUsPSDBoost is introduced in time domain, to make the final PSD unchanged same amount boost is
                  // needed to be removed from frequency domain.
                  for (s_BandCnt = 0 ; s_BandCnt < 5; s_BandCnt++ )
                  {
                     gt_TxPsdControl.s_US_PSD_Boost_0dB1[s_BandCnt] -= s_MaxUsPSDBoost;
                  }
               }
               // XDSLRTFW-3990:  TR115A: sync losses in AA12a and AA17a due to tx path clipping(End)

            }

            // Write the channel discovery value in the global variable, so it gets applied in SetTransmitGains
            gs_DeltaUsPsdCeilValue = gs_DeltaUsPsdCeilValue_ChDis;

            guc_SetTransmitGainsState = TRAINING_IN_PROGRESS;
            AddFunctionToBkgdFifo((PtrToBkgdFunc)SetTransmitGains);

            // Setup for Loading of Transmit Gains
            guc_LoadTransmitGainsState = LTG_LOAD_IRIDIA;

            //go to next substate
            gs_TxSubState = R_P_QUITE1_TX_SET_TXGAINS_INIT;

            // Tx State machine has finished the frame alignment. Now Tx state machine tells Rx state machine to
            // detect the Rx sync symbol marker, i.e. guc_TxRxCommunicator = TX_TIMING_ADVANCE_COMPLETED.
            // Note: Can be done always!
            //if(gft_DSVectoringEnabled == TRUE)
            {
               guc_TxRxCommunicator |= TX_TIMING_ADVANCE_COMPLETED;
            }
         }
         break;

      case R_P_QUITE1_TX_SET_TXGAINS_INIT:

         if ( (guc_SetTransmitGainsState == TRAINING_DONE) &&
               ((gft_DSVectoringEnabled == FALSE) || (guc_TxRxCommunicator & TX_RPVECTOR1_START)) )
         {
            //Ceiling to be encoded for message will be the one determined in SetTransmitGains(),
            //including any impact of the maxnomatp ceiling
            gt_PwrConfigParam.s_Up_InitialPsdCeiling = gt_TxPsdControl.s_PsdCeilingUsed;

            if (guc_LoadTransmitGainsState != LTG_LOAD_DONE)
            {
               LoadTransmitGains();
            }
            else
            {
               // Send the structure of gt_TxPsdControl as debug stream    (size of TxPsdControl = 788bytes considering MAX_NUM_PSD_POINTS] = 64)
               DSH_SendStream(DSH_TX_PSD_CONTROL,36,&gt_TxPsdControl);
               DSH_SendStream(DSH_MAX_PSD_DESC_IN,386,gt_TxPsdControl.pt_MaxPsdDescIn);
               DSH_SendStream(DSH_Max_PSD_DESC_OUT,386,gt_TxPsdControl.pt_ActPsdDescOut);
               if (gft_DSVectoringEnabled == TRUE)
               {
                  // XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts (Start/End)
                  // Switch global state timout after RP-Quiet1
                  // Note:
                  //       1) It is not possible to set the value here directly since
                  //          it is lower than the previous one. So the timeout check
                  //          in RxForeGround would directly kick in.
                  //       2) guc_SwitchGlobalTimeout must be initialized to zero
                  guc_SwitchGlobalTimeout |= SWITCH_GLOBAL_TIMEOUT_TX;

                  // go to a new Tx State
                  gs_TxNextState = VDSL2_R_P_VECTOR1_TX; // start R-P Vector 1 state for vectoring VDSL2

                  // During state change sub state will initialize to zero and the code
                  // will go to R_P_VECTOR1_TX_INIT sub state
               }
               else
               {
                  // XDSLRTFW-2155 (START_END)
                  // setup Iridia QT data path to send R-P_Vector1 signal at Sync frame
                  gus_RefGain = 0x2000;              // 0x2000 = Unity Gain in 2.13 format
                  gft_IT_MISC_RefGainEna = FALSE;    // Coupled GAIN TABLE on Tx data path
                  AddFunctionToFifo(gp_TxLoadingFunctionFifo, IridiaTxGainScaleDataPathControl);

                  // go to a new Tx State
                  gs_TxNextState = R_MSG1_TX;     // start O-P Channel Discovery1 state for non vectoring VDSL2
                  // During state change sub state will initialize to zero and the code
                  // will go to R_MSG1_TX_SET_TXGAINS sub state

               }
            }
         }
         break;
      } //end switch   for R_P_QUITE1_TX state
   } //end if( gs_TxState == R_P_QUITE1_TX)
   else if (gs_TxState == VDSL2_R_P_VECTOR1_TX) // R-P Vector 1 state
   {
//XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START_END)
      gsa_IndirectStat0[4]=VDSL2_R_P_VECTOR1;
      switch (gs_TxSubState)
      {
      case R_P_VECTOR1_TX_INIT:

         // XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts (Start/End)
         // Switch global state timout from OP_VECTOR1_STATE_TIMEOUT_LEN back
         // to normal VECTORING_STATE_TIMEOUT_LEN.
         // Note: This code is only active for Vectoring!
         if(guc_SwitchGlobalTimeout & SWITCH_GLOBAL_TIMEOUT_TX)
         {
            gl_TxStateTimeOutCount = VECTORING_STATE_TIMEOUT_LEN;
            guc_SwitchGlobalTimeout &= (~SWITCH_GLOBAL_TIMEOUT_TX);
         }

         // XDSLRTFW-2155 (START_END)
         gl_RPVector1PhaseSymCnt_start = (gl_TotalTxSymbolCount+1);

         gs_TxSubState = R_P_VECTOR1_TX_SIGNAL;
         break;

      case R_P_VECTOR1_TX_SIGNAL:

         int16 i;
         uint16 us_PilotSeqIndexWithCycShift;
         int32 l_SocTxDataForPrbs;

         if ((gs_MsgReceiveOPSYNCHROV1Flag != TRUE)||( gs_TxFrmCnt == TX_DATA_SYMBOLS_PER_SUPERFRAME))
         {
            // check for Tx Sync frame
            // setup Transmitter to send R-P_Vector1 signal
            // transmit every sync frame
            // Last symbol of R-P Vector1 must not be at sync symbol position!

            // setup Iridia QT data path to send R-P_Vector1 signal at Sync frame
            if (gs_TxFrmCnt == TX_DATA_SYMBOLS_PER_SUPERFRAME - 1)
            {
               if (!(guc_VecDebugSwitchEnable & VEC_DEBUG_SKIP_SIGNAL_RPVEC1))
               {
                  // setup Iridia QT data path to send R-P_Vector1 signal at Sync frame
                  gus_RefGain = 0x2000;             // 0x2000 = Unity Gain in 2.13 format
                  gft_IT_MISC_RefGainEna = FALSE;   // Coupled GAIN TABLE on Tx data path
               }
               else
               {
                  gus_RefGain = 0x0;                // 0 = Zero Gain
                  gft_IT_MISC_RefGainEna = TRUE;    // coupled IT_REF_GAIN register on Tx data path
               }
               AddFunctionToFifo(gp_TxLoadingFunctionFifo, IridiaTxGainScaleDataPathControl);

               // XDSLRTFW-1696 (Start)
               // Get US Sync symbol constellation points using FDPS array and the control parameters and
               // store the result in gl_SocTxDataForPrbs variable

               l_SocTxDataForPrbs = 0;
               gl_SocTxDataForPrbs = 0;

               for (i = 0 ; i < 8; i++)
               {
                  // main array  : gpuca_FDPSActive     size [8][64]
                  // control 1 : gs_VecIndx_PilotSeq
                  // control 2 : guca_IndexIndependentPSActive[8]
                  // control 3 : guca_CyclicalShiftOfSeqActive[8]
                  // control 4 : guc_SignOfSeqActive
                  // modulo parameter : gs_VecNPilotLength_US bit field length

                  //considing Cyclical Shift of the sequence
                  us_PilotSeqIndexWithCycShift  = (uint16)(gs_VecIndx_PilotSeq + guca_CyclicalShiftOfSeqActive[i]*(gs_VecNPilotLength_US >>3));
                  // modulo (s_PilotSeqIndexWithCycShift, gs_VecNPilotLength_US)
                  while (us_PilotSeqIndexWithCycShift > gs_VecNPilotLength_US)
                  {
                     us_PilotSeqIndexWithCycShift -= gs_VecNPilotLength_US;
                  }

                  // get the bit considiring US super frame counter (gs_VecIndx_PilotSeq), cyclical shift and independent pilot sequence
                  s_RevSeg = (*(gpuca_FDPSActive + guca_IndexIndependentPSActive[i]*64  + (us_PilotSeqIndexWithCycShift>>3)) >> (us_PilotSeqIndexWithCycShift & 0x7)) & 0x1; // XDSLRTFW-1321 Feature_US_Vec_VDSL2_USPS (start-end)

                  // considereing sign bit
                  if ((guc_SignOfSeqActive >> i) & 0x1)
                  {
                     s_RevSeg = (~s_RevSeg) & 0x1;
                  }
                  // Preparation of constallation points for tones 0,2,3,4,5,6,8 & 9
                  if (s_RevSeg == 1)
                  {
                     l_SocTxDataForPrbs |= (0x3 << 2*i);
                  }
                  else
                  {
                     l_SocTxDataForPrbs |= (0x0 << 2*i);
                  }
               }

               // add flag tones (tone 10n+1 and 10n+7) constallation to zero
               gl_SocTxDataForPrbs = (l_SocTxDataForPrbs & 0x3)| ((l_SocTxDataForPrbs & 0xFFC) << 2) | ((l_SocTxDataForPrbs & 0xF000) << 4);


               // XDSLRTFW-2451 (start)
               // Special treatment for flag tones ( tone 10n+1 & 10n+7)
               if (gus_VectoringOptionsEnabled & VEC_OPTIONS_FDPS_US_MASK ) // FDPS enabled both at CO and CPE
               {
                   gl_SocTxDataForPrbs |= 0xC00C;   // According to Section 6.2.3/G993.5 (During initialization,
                                                    // the VTU-O(!) may modulate on all flag tones of the downstream sync
                                                    // symbols either the downstream pilot sequence (the same as modulated on the probe tones),
                                                    // or an all ONEs sequence).
                                                    // since in FDPS, it is not posible to modulate flag tones according to pilit sequence bit,
                                                    // here we modultat 'ONEs' on flag tones
               }
               else // non FDPS
               {
                  // Avinax CO requires zero on flag tones so that we don't need to modify the flag tones
                  if (gul_fe_G994VendorID != IFX_VENDOR_ID)
                  {
                     if (gl_SocTxDataForPrbs != 0)
                     {
                          gl_SocTxDataForPrbs = 0xFFFFF; // modulate flag tones according to pilot sequence bit
                     }
                  }
               }
               // XDSLRTFW-2451 (end)

               //if ( gs_PauseControl == 0x3E)
               //   Pause(gs_PauseControl);

               // XDSLRTFW-1696 (end)
            }
            else if (gs_TxFrmCnt == TX_DATA_SYMBOLS_PER_SUPERFRAME)
            {
               if (!(guc_VecDebugSwitchEnable & VEC_DEBUG_SKIP_QUITESIGNAL_RPVEC1))
               {
                  // Setup QUITE signal on Non Sync Frames at R-P Vector1 state
                  gus_RefGain = 0x0;               // 0 = Zero Gain
                  gft_IT_MISC_RefGainEna = TRUE;   // coupled IT_REF_GAIN register on Tx data path
                  AddFunctionToFifo(gp_TxLoadingFunctionFifo, IridiaTxGainScaleDataPathControl);
               }
            }
         }
         else
         {
            // start O-P Channel Discovery1 state for vectoring VDSL2
            gs_TxNextState = R_MSG1_TX;
            gl_DiscoveryPhaseSymCnt_start = (gl_TotalTxSymbolCount+1);

            // XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts (Start/End)
            // Switch global state timout after RP-Quiet1
            // Note:
            //       1) It is not possible to set the value here directly since
            //          it is lower than the previous one. So the timeout check
            //          in TxForeGround would directly kick in.
            //       2) guc_SwitchGlobalTimeout must be initialized to zero
            guc_SwitchGlobalTimeout |= SWITCH_GLOBAL_TIMEOUT_TX;

            // During state change sub state will initialize to zero and the code
            // will go to R_MSG1_TX_SET_TXGAINS sub state
         }
         break;
      } // end switch for VDSL2_R_P_VECTOR1_TX state
   } // end if (gs_TxState == VDSL2_R_P_VECTOR1_TX)
   else if ( gs_TxState == R_MSG1_TX)   //R-P Channel Discovery 1 State
   {
//XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (START)
      gsa_IndirectStat0[4]=VDSL2_R_P_CHANNEL_DISCOVERY1;
      gsa_IndirectStat0[2]=VDSL2_R_MSG1;
//XDSLRTFW-1901: ADSL/VDSL ACTIVATION STATE MACHINE (END)
      switch (gs_TxSubState)
      {
      case R_MSG1_TX_SET_TXGAINS:

         // log start of R-MSG1 idle flags
         gpt_DebugTiming->l_TxRMsg1_IdleStart = gl_TotalTxSymbolCount;

         if (gft_DSVectoringEnabled == TRUE)
         {
            //Coupled Gain Table at Iridia QT at the end of R-P Vector1 State ( default Iridia QT setup)
            gus_RefGain = 0x2000 ;           // 0x2000 = Unity Gain in 2.13 format
            gft_IT_MISC_RefGainEna = FALSE;  // Coupled GAIN TABLE on Tx data path
            AddFunctionToFifo(gp_TxLoadingFunctionFifo, IridiaTxGainScaleDataPathControl);

            // XDSLRTFW-1242: Bugfix_ALL_VDSL2_ALL_CleanupGlobalVectoringTimeouts (Start/End)
            // Switch global state timout from VECTORING_STATE_TIMEOUT_LEN back
            // to normal STATE_TIMEOUT_LEN.
            // Note: This code is only active for Vectoring!
            if(guc_SwitchGlobalTimeout & SWITCH_GLOBAL_TIMEOUT_TX)
            {
               gl_TxStateTimeOutCount = STATE_TIMEOUT_LEN;
               guc_SwitchGlobalTimeout &= (~SWITCH_GLOBAL_TIMEOUT_TX);
            }
         }
         // Set up transmitter to send HDLC flag once O-SIGNATURE is received
         uc_octet = (uint8)HDLC_FLAG;
         ModulateHdlcOctets(&uc_octet, 1);

         //Enable the feature to repetively send HDLC idle flags in the diag mode
         if(gul_OperationModeStatus_VDSL2 & V2_LOOP_DIAG)
         {
            gft_SocRepeatFlag = TRUE;
            gft_TxSocMsgModFlag = TRUE;
         }
         //start transmitting IDLE flags so RX side can train PGA with echo on.
         gs_TxRxMessaging = PGA_RETRAIN_XMITTING;

         //go to next substate
         gs_TxSubState = R_MSG1_TX_WAIT_FOR_PGA_TRAINED;
         break;

      case R_MSG1_TX_WAIT_FOR_PGA_TRAINED:

         // Wait for RX side to retrain PGA and FDQ.
         if (gs_TxRxMessaging == PGA_RETRAIN_FDQDONE)
         {
            //Start a background process to initialize the TX message payload buffer
            //by R-MSG1 message
            gs_FormMsgFlag = TRAINING_IN_PROGRESS;

            AddFunctionToBkgdFifo((PtrToBkgdFunc)FormRMsg1Msg_VDSL2);

            gs_TxSubState = R_MSG1_TX_READY_TO_SEND_MSG;
         }
         break;

      case R_MSG1_TX_READY_TO_SEND_MSG:

         if (gs_FormMsgFlag == TRAINING_DONE)
         {
            gta_StrymonRegDump[20][1]=5;           // update state

            for(int idx=0;idx<SIZE_STRYMON_REG_DUMP-2;idx++)
            {
               ReadCoreReg(gta_StrymonRegDump[idx][0], &gta_StrymonRegDump[idx][1]);
            }
            // Send the Strymon pre-ghs configuration as debug stream
            DSH_SendStream(STRYMON_CONF,2*SIZE_STRYMON_REG_DUMP*sizeof(uint32),&gta_StrymonRegDump[0][0]);
            // go to the bridge state, and be ready to load TRAIN_1_V2 swap page
            gs_TxNextState = VDSL2_R_MSG1_TX_BRIDGE;
            gpF_TxStateFunc = (PtrToFunc)RMsg1TxF_VDSL2_Bridge;

      #ifdef DEBUG_VRX518_AFE  //0x5108
            if(gs_PauseControl == 0x5108)
            Pause(gs_PauseControl);
      #endif
         }
         break;
      }   // end switch for R_MSG1_TX state
   } // end R_MSG1_TX state
}   // end of function RMsg1TxF_VDSL2
